{******************************************************************************}
{ FileName............: DvbRegisters                                           }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  DVB related defines (might be specific to used hardware)                    }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030529 - Initial release                                           }
{******************************************************************************}

unit DvbRegisters;

interface

uses
  Windows;

const
  CDvbPacketSize            = 204 - 16;                    // DVB packet size: parity is stripped by the SAA7146A
  CDvbPacketSyncWord        = $47;                         // DVB sync word
  CDvbPacketHSync           = CDvbPacketSize;              // 'Horizontal' synchronization DD1 of SAA7146A
  CDvbPacketVSync           = 512;                         // 'Vertical'   synchronization DD1 of SAA7146A
  CDvbPacketBufferSize      = CDvbPacketHSync*CDvbPacketVSync;       // All packets in a single buffer

  CTransportPidPAT          = $0000;                       // Program Association Table identifier
  CTransportPidCAT          = $0001;                       // Conditional Access Table identifier
  CTransportPidAssignStart  = $0010;                       // First Pid user assignable
  CTransportPidAssignEnd    = $1FFE;                       // Last  Pid user assignable
  CTransportPidNull         = $1FFF;                       // Null packet

type
  PTransportPackets = ^TTransportpacket;
  TTransportPackets = array[0..CDvbPacketBufferSize-1] of Byte;

  PTransportPacket = ^TTransportpacket;
  TTransportPacket = array[0..187] of Byte;

  TTransportPids   = array[0..$1FFF] of Boolean;

  // Transport stream header individual components
  TTransportStreamHeader = record
    HeaderAndPayload                 : TTransportPacket;
    TransportValid                   : Boolean;            // Indicates no obvious errors found, eg. SyncByte/Error indicator
    SyncByte                         : Byte;               // Should be $47
    TransportErrorIndicator          : Boolean;
    PayloadUnitStartIndicator        : Boolean;
    TransportPriority                : Boolean;
    Pid                              : Word;
    TransportScramblingControl       : Byte;
    AdaptationFieldControl           : Byte;
    ContinuityCounter                : Byte;
    AdaptationFieldLength            : Byte;
    DiscontinuityIndicator           : Boolean;
    RandomAccessIndicator            : Boolean;
    ElementaryStreamPriorityIndicator: Boolean;
    PcrFlag                          : Boolean;
    OpcrFlag                         : Boolean;
    SplicingPointFlag                : Boolean;
    TransportPrivateDataFlag         : Boolean;
    AdaptationFieldExtensionFlag     : Boolean;
    ProgramClockReferenceBase        : Int64;
    PcrReserved                      : Byte;
    ProgramClockReferenceExtension   : Word;
    OriginalProgramClockReferenceBase     : Int64;
    OpcrReserved                          : Byte;
    OriginalProgramClockReferenceExtension: Word;
    SpliceCountdown                  : Byte;
    TransportPrivateDataLength       : Byte;
    PrivateDataByte                  : array[0..255] of Byte;
    AdaptationFieldExtensionLength   : Byte;
    LtwFlag                          : Boolean;
    PiecewiseRateFlag                : Boolean;
    SeamlessSpliceFlag               : Boolean;
    AdaptationExtensionReserved      : Byte;
    LtwValidFlag                     : Boolean;
    LtwOffset                        : Word;
    PiecewiseReserved                : Byte;
    PiecewiseRate                    : Dword;
    SpliceType                       : Byte;
    DtsNextAu                        : Int64;
    DataBytes                        : array[0..183] of Byte;
    DataBytesLength                  : Word;
  end;


  // Program Association Table PID part
  TProgramAssociationPid = record
    ProgramNumber                    : Word;
    Reserved                         : Byte;
    NetworkPid                       : Word;
    ProgramMapPid                    : Word;
  end;


  // Program Association Table
  TProgramAssociationTable = record
    TableId                          : Byte;
    SectionSyntaxIndicator           : Boolean;
    Zero                             : Boolean;
    Reserved                         : Byte;
    SectionLength                    : Word;
    TransportStreamId                : Word;
    Reserved2                        : Byte;
    VersionNumber                    : Byte;
    CurrentNextIndicator             : Boolean;
    SectionNumber                    : Byte;
    LastSectionNumber                : Byte;
    ProgramData                      : array[0..255] of TProgramAssociationPid;
    Crc32                            : Dword;
  end;


implementation

var
  Crc16Table: array[0..255] of Word;
  Crc32Table: array[0..255] of Dword;

const
  // 16 Bit CCITT standard polynomial x^16 + x^12 + x^5 + x^1 + x^0
  CPolynomial16 = $1021;
  CCrcInitial16 = $FFFF;
  // 32 Bit standard polynomial x^32 + x^26 + x^23 + x^22 + x^16 +
  //   x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x^1 + x^0
  CPolynomial32 = $04C11DB7;
  CCrcInitial32 = $FFFFFFFF;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Generate CRC tables
  Notes   :
------------------------------------------------------------------------------}
procedure DvbGenerateCrcTables;
var
  ByteLoop: Byte;
  BitLoop : Byte;
  Crc16   : Word;
  Crc32   : Dword;
begin
  // Go through the whole array
  for ByteLoop := 0 to 255 do
  begin
    Crc16 := ByteLoop shl 8;
    Crc32 := ByteLoop shl 24;
    for BitLoop := 0 to 7 do
    begin
      if (Crc16 and $8000) <> 0 then
        Crc16 := ((Crc16 and $7FFF) shl 1) xor CPolynomial16
      else
        Crc16 := (Crc16 shl 1);
      if (Crc32 and $80000000) <> 0 then
        Crc32 := ((Crc32 and $7FFFFFFF) shl 1) xor CPolynomial32
      else
        Crc32 := (Crc32 shl 1);
      Crc16Table[ByteLoop] := Crc16;
      Crc32Table[ByteLoop] := Crc32;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC16 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial16> as
            starting value
------------------------------------------------------------------------------}
function CalcCrc16(Checksum: Word; Data: Byte): Word;
begin
  Result := ((Checksum and $FF) shl 8) xor Crc16Table[((Checksum shr 8) xor Data) and $FF];
end;


{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC32 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial32> as
            starting value
------------------------------------------------------------------------------}
function CalcCrc32(Checksum: Dword; Data: Byte): Dword;
begin
  Result := ((Checksum and $FFFFFF) shl 8) xor Crc32Table[((Checksum shr 24) xor Data) and $FF];
end;


initialization
  // The CRC tables are generated at startup
  DvbGenerateCrcTables;

finalization


end.

