{******************************************************************************}
{ FileName............: CommonInterfaceTransportLayer                          }
{ Project.............: DVB-S                                                  }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Common interface support (transport layer)                                  }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ The transport layer (on the command interface) creates and disassembles data }
{ to and from the link layer.                                                  }
{                                                                              }
{ The layers involved:                                                         }
{                  APPLICATION                                                 }
{               Application layer                                              }
{                 Session layer                                                }
{          ****  Transport layer                                               }
{                   Link layer                                                 }
{                 Physical layer                                               }
{                    HARDWARE                                                  }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030726 - Initial release                                           }
{******************************************************************************}
unit CommonInterfaceTransportLayer;

interface
uses
  Windows;

type
  TStates = (stUnassigned,             // Not in use
             stAssigned,               // In use
             stIdle,                   // Idle, suspended
             stInCreation,             // Wait for 'C_T_C_Reply'
             stActive,                 // Active, will poll
             stInDeletion);            // Wait for 'D_T_C_Reply'

  function CommonInterfaceTransportLayerGetTransportData
             (TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
  function CommonInterfaceTransportLayerWriteDataToModule
             (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
  function CommonInterfaceTransportLayerReadDataFromModule
             (Handle: THandle; var TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
  function CommonInterfaceTransportLayerCheckModule
             (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
  function CommonInterfaceTransportLayerResetModule
             (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;

  function CommonInterfaceTransportLayerAllocateTransportConnection(Handle: THandle; var ConnectionIdentifier: Byte): Boolean;
  function CommonInterfaceTransportLayerReleaseTransportConnection(ConnectionIdentifier: Byte): Boolean;
  function CommonInterfaceTransportLayerGetStatus(ConnectionIdentifier: Byte; var State: TStates; var ReceiveCounter: Word): Boolean;
  procedure CommonInterfaceTransportLayerSetState(ConnectionIdentifier: Byte; State: TStates);


implementation
uses
  Classes,
  CommonInterfaceLinkLayer,
  Saa7146aDebi,
  SyncObjs,
  SysUtils;

const
  CTagSB        = #$80;
  CTagRcv       = #$81;
  CTagCreateTc  = #$82;
  CTagCtcReply  = #$83;
  CTagDeleteTc  = #$84;
  CTagDtcReply  = #$85;
  CTagRequestTc = #$86;
  CTagNewTc     = #$87;
  CTagTcError   = #$88;
  CTagDataLast  = #$A0;
  CTagDataMore  = #$A1;

type

  TModule = class(TObject)
  private
    { Private declarations }
    FBufferSize        : Word;                   // Size of transfer buffers
  protected
  public
  published
  end;

  TTransportThread = class(TThread)
  private
    FParent: TObject;                                      // Reference to parent
  protected
    procedure Execute; override;
  end;

  TTransportConnection = class(TObject)
  private
    { Private declarations }
    FState   : TStates;                                    // Current state
    FModule  : Byte;                                       // Module socket
    FTransportIdentifier: Byte;                            // Transport identification
    FHandle  : THandle;                                    // Handle to driver
    FThread  : TTransportThread;                           // Thread handling the transports
    FReceived: Word;                                       // (Debug) counter for received answers
    function  GetState: TStates;
    procedure SetState(State: TStates);
    function  GetReceiveCounter: Word;
    procedure SetReceiveCounter(ReceiveCounter: Word);
  protected
  public
    constructor Create(TransportIdentification: Byte);
    destructor  Destroy; override;
  published
    property State: TStates read GetState write SetState;
    property ReceiveCounter: Word read GetReceiveCounter write SetReceiveCounter;
  end;


var
  CommonInterfaceLock : TCriticalSection;        // Prevents multiple operations which interfere
  Modules             : array[0..3] of TModule;  // Modules supported
  TransportConnections: array[1..16] of TTransportConnection;
  Loop                : Integer;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result> Current state

  Descript: Get current state of transport connection
  Notes   :
------------------------------------------------------------------------------}
function TTransportConnection.GetState: TStates;
begin
  Result := FState;
end;


{------------------------------------------------------------------------------
  Params  : <State>  State to set
  Returns : -

  Descript: Set current state of transport connection
  Notes   :
------------------------------------------------------------------------------}
procedure TTransportConnection.SetState(State: TStates);
begin
  FState := State;
  if FThread.Suspended then
    FThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Receive counter

  Descript: Get receive counterof transport connection
  Notes   :
------------------------------------------------------------------------------}
function TTransportConnection.GetReceiveCounter: Word;
begin
  Result := FReceived;
end;


{------------------------------------------------------------------------------
  Params  : <ReceiveCounter>  Value to set
  Returns : -

  Descript: Set receive counter of transport connection
  Notes   :
------------------------------------------------------------------------------}
procedure TTransportConnection.SetReceiveCounter(ReceiveCounter: Word);
begin
  FReceived := ReceiveCounter;
end;


{------------------------------------------------------------------------------
  Params  : <TransportIdentifcation>  Identifier for transport object
  Returns : -

  Descript: Creation for transport connection object
  Notes   :
------------------------------------------------------------------------------}
constructor TTransportConnection.Create(TransportIdentification: Byte);
begin
  inherited Create;
  FState := stUnassigned;                                  // Not is use
  FReceived := 0;
  FModule := 0;
  FTransportIdentifier := TransportIdentification;
  FHandle := INVALID_HANDLE_VALUE;
  FThread := TTransportThread.Create(True);
  FThread.FParent := Self;
  FThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destruction for transport connection object
  Notes   :
------------------------------------------------------------------------------}
destructor TTransportConnection.Destroy;
begin
  if FState <> stUnassigned then
  begin
  end;
  if Assigned(FThread) then
  begin
    FThread.Terminate;
    FThread.Resume;
    FThread.WaitFor;
  end;
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>               Handle of driver
  Returns : <Result>               True if connection identifier available
            <ConnectionIdentifier> Allocated identifier
                                   0 = invalid

  Descript: Allocate a new transport connection.
  Notes   : Does not 'creates' the transport connection. This is done
            by changing the 'state'.
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerAllocateTransportConnection(Handle: THandle; var ConnectionIdentifier: Byte): Boolean;
var
  Loop : Byte;
  Found: Boolean;
begin
  Found := False;
  ConnectionIdentifier := 0;
  Loop := Low(TransportConnections);
  repeat
    if TransportConnections[Loop].FState <> stUnassigned then
      Inc(Loop)
    else
    begin
      Found := True;
      ConnectionIdentifier := Loop;
    end;
  until Found or (Loop > High(TransportConnections));
  if Found then
  begin
    TransportConnections[Loop].FHandle := Handle;
    TransportConnections[Loop].State   := stAssigned;
    TransportConnections[Loop].FThread.Resume;
  end;
  Result := Found;
end;


{------------------------------------------------------------------------------
  Params  : <ConnectionIdentifier>  Connection to check
  Returns : <Result>                True if valid connection
            <State>                 Current state
            <ReceiveCounter>        Number of received replies from module
                                    (in Active state updated only)
  Descript: Get state of connection
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerGetStatus(
           ConnectionIdentifier: Byte;
           var State: TStates;
           var ReceiveCounter: Word): Boolean;
begin
  Result := False;
  ReceiveCounter := 0;
  State := stUnassigned;
  if ConnectionIdentifier < Low(TransportConnections) then
    Exit;
  if ConnectionIdentifier > High(TransportConnections) then
    Exit;
  State := TransportConnections[ConnectionIdentifier].State;
  ReceiveCounter := TransportConnections[ConnectionIdentifier].ReceiveCounter;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <ConnectionIdentifier>  Connection to set
            <State>                 State to set connection to
                                    stInCreation or stInDeletion allowed
  Returns : <Result>                State of connection

  Descript: Set state of connection
  Notes   :
------------------------------------------------------------------------------}
procedure CommonInterfaceTransportLayerSetState(ConnectionIdentifier: Byte; State: TStates);
begin
  if ConnectionIdentifier < Low(TransportConnections) then
    Exit;
  if ConnectionIdentifier > High(TransportConnections) then
    Exit;
  case State of
    stInCreation,
    stInDeletion: TransportConnections[ConnectionIdentifier].State := State;
  end;  
end;


{------------------------------------------------------------------------------
  Params  : <ConnectionIdentifier> Allocated identifier
  Returns : <Result>               True if success

  Descript: Release a transport connection.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerReleaseTransportConnection(ConnectionIdentifier: Byte): Boolean;
begin
  CommonInterfaceTransportLayerSetState(ConnectionIdentifier, stInDeletion);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>       Pointer to data buffer
            <BufferIndex>  Points to the length field data
  Returns : <Result>       True if success
            <BufferIndex>  Points to after the length field data
            <LengthValue>  Retrieved length

  Descript: Get the length from the length field.
  Notes   : 'Limited' support for a 4 bytes length.
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerGetLengthField(Buffer: PChar; var BufferIndex: Word; var LengthValue: Dword): Boolean;
begin
  Result := CommonInterfaceLinkLayerGetLengthField(Buffer, BufferIndex, LengthValue);
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>       Pointer to data buffer
            <BufferIndex>  Points to the length field data
            <LengthValue>  Length to set
  Returns : <Result>       True if success
            <BufferIndex>  Points to after the length field data

  Descript: Set the length field.
  Notes   : 'Limited' support for a 4 bytes length.
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerSetLengthField(Buffer: PChar; var BufferIndex: Word; LengthValue: Dword): Boolean;
var
  LengthBytes: Byte;
  Loop       : Byte;
  ByteValue  : Byte;
begin
  Result := False;
  try
    // If the length is < 128 then it is contained within a single byte
    if (LengthValue < $80) then
    begin
      Buffer[BufferIndex] := Chr(LengthValue and $7F);
      Inc(BufferIndex);
    end
    else
    begin
      // A length > 128 uses multiple bytes. We allow only a limited length.
      // Scan form MS byte to LS byte
      LengthBytes := 0;
      for Loop := SizeOf(LengthValue) - 1 downto 0 do
      begin
        // Byte to check
        ByteValue := (LengthValue shr (8 * Loop)) and $FF;
        if (ByteValue <> 0) or (LengthBytes <> 0) then
        begin
          Inc(LengthBytes);
          Buffer[BufferIndex+LengthBytes] := Chr(ByteValue);
        end;
      end;
      Buffer[BufferIndex] := Chr(LengthBytes or $80);
      Inc(BufferIndex, LengthBytes + 1);
      // We now know how many bytes are required
    end;
    Result := True;
  except
    // If invalid buffer ....
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>               Pointer to target data buffer
            <Tag>                  Tag
            <ConnectionIdentifier> Connection identifier
            <DataBytes>            Points to the optional data bytes (nil if none)
            <DataBytesLength>      Number of bytes in data bytes
  Returns : <Result>               True if success
            <BufferIndex>          Points to after the last valid data

  Descript: Assemble a Command Transport Protocol Data Unit
  Notes   : This one is used when the connection is NOT yet OPEN
            or used in the DataBytes of an open connection
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerAssembleCTPDU(Buffer: PChar; var BufferIndex: Word; Tag: Byte; ConnectionIdentifier: Byte; DataBytes: PChar; DataBytesLength: Word): Boolean;
var
  BufferLength: Dword;
  Loop        : Word;
begin
  Result := False;
  try
    Buffer[0] := Chr(Tag);
    BufferIndex := 1;
    if DataBytes <> nil then
      BufferLength := DataBytesLength + 1
    else
      BufferLength := 1;
    if not CommonInterfaceTransportLayerSetLengthField(Buffer, BufferIndex, BufferLength) then
      Exit;
    Buffer[BufferIndex] := Chr(ConnectionIdentifier);
    Inc(BufferIndex);
    if BufferLength > 1 then
      for Loop := 0 to BufferLength - 1 do
      begin
        Buffer[BufferIndex] := DataBytes[Loop];
        Inc(BufferIndex);
      end;
    Result := True;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>               Pointer to source data buffer
            <Tag>                  Tag
            <ConnectionIdentifier> Connection identifier
            <DataBytes>            Points to the optional data bytes
  Returns : <Result>               True if success
                                   False if any error detected
            <BufferIndex>          Points to after the last valid data
            <DataBytesLength>      Number of bytes in data bytes
            <Status>               Status tag returned by module

  Descript: Disassemble a Response Transport Protocol Data Unit
  Notes   : This one is used when the connection is NOT yet OPEN
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerDisassembleRTPDU(Buffer: PChar; var BufferIndex: Word; var Tag: Byte; var ConnectionIdentifier: Byte; DataBytes: PChar; var DataBytesLength: Word; var Status: Byte): Boolean;
var
  BufferLength: Dword;
  Loop        : Word;
//  TransportId : Byte;
begin
  Result := False;
  try
    Tag := Ord(Buffer[0]);
    BufferIndex := 1;
    DataBytesLength := 0;
    if not CommonInterfaceTransportLayerGetLengthField(Buffer, BufferIndex, BufferLength) then
      Exit;
    ConnectionIdentifier := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    if BufferLength > 1 then
    begin
      if DataBytes = nil then
        Exit;
      for Loop := 0 to BufferLength - 2 do
      begin
        DataBytes[DataBytesLength] := Buffer[BufferIndex];
        Inc(BufferIndex);
        Inc(DataBytesLength);
      end;
    end;
    // Now follows the status information returned
    // There is a special case when we receive a <CTagSb> RTPDU since this type
    // does not carry additional status information (it is present in the
    // databytes in that case)
    if Buffer[BufferIndex] <> CTagSb then
    begin
      // Exit with error if it is one of the others
      if Tag <> Ord(CTagSb) then
        Exit;
      // If this is a status RTPDU get the status from the databyte
      if DataBytesLength <> 1 then
        Exit;
      Status := Ord(DataBytes[0]);
      Result := True;
      Exit;
    end;
    Inc(BufferIndex);
    if not CommonInterfaceTransportLayerGetLengthField(Buffer, BufferIndex, BufferLength) then
      Exit;
    if BufferLength <> 2 then
      Exit;
    // Not sure what to do with this connection identifier ...
    // TransportId := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    Status := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    Result := True;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>               Pointer to target data buffer
            <Tag>                  Tag
                                   Allowed tags are:
                                      CTagDataLast  = $A0;
                                      CTagDataMore  = $A1;
            <ConnectionIdentifier> Connection identifier
            <DataBytes>            Points to the optional data bytes (nil if none)
            <DataBytesLength>      Number of bytes in data bytes
            <Status>               Status
  Returns : <Result>               True if success
            <BufferIndex>          Points to after the last valid data

  Descript: Assemble a Command Transport Protocol Data Unit
  Notes   : This one is used when the connection is OPEN
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerAssembleSendDataCTPDU(Buffer: PChar; var BufferIndex: Word; Tag: Byte; ConnectionIdentifier: Byte; DataBytes: PChar; DataBytesLength: Word; Status: Byte): Boolean;
var
  BufferLength: Dword;
  Loop        : Word;
begin
  Result := False;
  try
    // Only two types of tags allowed here
    if not Tag in [Ord(CTagDataLast), Ord(CTagDataMore)] then
      Exit;
    Buffer[0] := Chr(Tag);
    BufferIndex := 1;
    if DataBytes <> nil then
      BufferLength := DataBytesLength + 1
    else
      BufferLength := 1;
    if not CommonInterfaceTransportLayerSetLengthField(Buffer, BufferIndex, BufferLength) then
      Exit;
    Buffer[BufferIndex] := Chr(ConnectionIdentifier);
    Inc(BufferIndex);
    if BufferLength > 1 then
      for Loop := 0 to BufferLength - 1 do
      begin
        Buffer[BufferIndex] := DataBytes[Loop];
        Inc(BufferIndex);
      end;
    Buffer[BufferIndex] := Chr(Status);
    Inc(BufferIndex);
    Result := True;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>               Pointer to source data buffer
            <DataBytes>            Points to the optional data bytes
  Returns : <Result>               True if success
                                   False if any error detected
            <BufferIndex>          Points to after the last valid data
            <Tag>                  Tag
                                   Allowed tags are:
                                      CTagDataLast  = $A0;
                                      CTagDataMore  = $A1;
            <ConnectionIdentifier> Connection identifier
            <DataBytesLength>      Number of bytes in data bytes
            <Status>               Status tag returned by module

  Descript: Disassemble a Response Transport Protocol Data Unit
  Notes   : This one is used when the connection is OPEN
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerDisassembleReceiveDataRTPDU(Buffer: PChar; var BufferIndex: Word; var Tag: Byte; var ConnectionIdentifier: Byte; DataBytes: PChar; var DataBytesLength: Word; var Status: Byte): Boolean;
var
  BufferLength: Dword;
  Loop        : Word;
begin
  Result := False;
  try
    Tag := Ord(Buffer[0]);
    // Only two types of tags allowed here
    if not Tag in [Ord(CTagDataLast), Ord(CTagDataMore)] then
      Exit;
    BufferIndex := 1;
    DataBytesLength := 0;
    if not CommonInterfaceTransportLayerGetLengthField(Buffer, BufferIndex, BufferLength) then
      Exit;
    ConnectionIdentifier := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    if BufferLength > 1 then
    begin
      if DataBytes = nil then
        Exit;
      for Loop := 0 to BufferLength - 2 do
      begin
        DataBytes[DataBytesLength] := Buffer[BufferIndex];
        Inc(BufferIndex);
        Inc(DataBytesLength);
      end;
    end;
    // Now follows the status information returned
    Status := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    Result := True;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Module> Module to remove
  Returns : -

  Descript: Remove the module (from memory).
  Notes   :
------------------------------------------------------------------------------}
procedure CommonInterfaceTransportLayerRemoveModule(Module: Byte);
begin
  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  if Assigned(Modules[Module]) then
  begin
    try
    finally
      FreeAndNil(Modules[Module]);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Remove all modules.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerRemoveModules: Boolean;
var
  Loop: Byte;
begin
  for Loop := Low(Modules) to High(Modules) do
    CommonInterfaceTransportLayerRemoveModule(Loop);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Module>  Module to add
  Returns : <Result>  True if success (module has been added)

  Descript: Add the module (memory and initialization) if present.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerAddModule(Handle: THandle; Module: Byte): Boolean;
begin
  Result := False;

  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  // Remove the module if already present
  CommonInterfaceTransportLayerRemoveModule(Module);
  // Create the module
  Modules[Module] := TModule.Create;
  Modules[Module].FBufferSize := 0;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <TransportConnectionId>  Identifier to find data for
            <Module>                 Module to 'access'
            <Buffer>                 Pointer to buffer
            <BufferSize>             Available buffer size
  Returns : <Result>                 True if success
                                     False if buffersize smaller than needed
                                     (required buffer size in <BufferSize>
                                     False if module id error
            <BufferSize>             Returned buffer size (or required when
                                     buffer size error)

  Descript: Get data which correspondents with transport connection (and module).
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerGetTransportData
           (TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceLinkLayerGetTransportData(TransportConnectionId, Module, Buffer, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 <Saa7146aCreateFile> handle
            <TransportConnectionId>  Identifier
            <Module>                 Module to write to
            <Buffer>                 Pointer to buffer to write
            <BufferSize>             Size of buffer
  Returns : <Result>                 True if success
                                     False if module not ready

  Descript: Write data to module. Adds header information and splits it up
            in several packages if necessary.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerWriteDataToModule
  (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceLinkLayerWriteDataToModule(Handle, TransportConnectionId, Module, Buffer, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 <Saa7146aCreateFile> handle
            <TransportConnectionId>  Identifier
            <Module>                 Module to check
            <Buffer>                 Pointer to buffer to read to
            <BufferSize>             Size of buffer
  Returns : <Result>                 True if success
                                     False if buffersize smaller than needed (data is lost)
                                     False if module not ready
            <BufferSize>             Data read
                                     0 if no data available

  Descript: Read data from module. Removes header information and combines
            multiple packets if necessary.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerReadDataFromModule
  (Handle: THandle; var TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceLinkLayerReadDataFromModule(Handle, TransportConnectionId, Module, Buffer, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;



{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <ModuleIsPresent> True if module is available
            <ModuleHasData>   True if module has data available

  Descript: Check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerCheckModule
  (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceLinkLayerCheckModule(Handle, Module, ModuleIsPresent, ModuleHasData);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <BufferSize>      Negotiated buffer size

  Descript: Reset and check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceTransportLayerResetModule
  (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    CommonInterfaceTransportLayerRemoveModule(Module);
    if CommonInterfaceTransportLayerAddModule(Handle, Module) then
    begin
      Result := CommonInterfaceLinkLayerResetModule(Handle, Module, Modules[Module].FBufferSize);
      BufferSize := Modules[Module].FBufferSize;
    end
    else
      Result := False;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Remove all transport connections.
  Notes   :
------------------------------------------------------------------------------}
procedure RemoveTransportConnections;
var
  Loop: Byte;
begin
  for Loop := Low(TransportConnections) to High(TransportConnections) do
    TransportConnections[Loop].Free;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Transport thread
  Notes   :
------------------------------------------------------------------------------}
procedure TTransportThread.Execute;
const
  CBufferSize    = 2048;
  CTimeOut       = 300;                // Max allowed timeout is 300 ms
  CPollingPeriod = 10;                 // Max allowed polling period is 100 ms
var
  PreviousState  : TStates;
  NewState       : Boolean;
  Buffer         : PChar;
  DataBytes      : PChar;
  BufferSize     : Word;
  BufferIndex    : Word;
  Tag            : Byte;
  ConnectionId   : Byte;
  DataBytesLength: Word;
  Status         : Byte;
  DataAvailable  : Boolean;            // True when module has data available
  DataAwaiting   : Boolean;            // True when waiting for reply from module
  TimeOut        : Dword;
  PollTimeOut    : Dword;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'idle'
    Notes   : Reset variables and go into suspension
  ------------------------------------------------------------------------------}
  procedure StateIdle;
  begin
    DataAvailable := False;
    DataAwaiting  := False;
    Suspend;
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'in creation'
    Notes   : When this state is entered for the first time a 'Create_T_C' is
              send. We then wait for a correct reply from the module. If we don't
              get a correct reply within a certain time we revert to the idle
              state.
              We expect C_T_C_Reply which includes T_SB
  ------------------------------------------------------------------------------}
  procedure StateInCreation;
  begin
    // If first time state entered: send 'Create_T_C'
    if NewState then
    begin
      // Construct 'Create_T_C' data and send
      if CommonInterfaceTransportLayerAssembleCTPDU(
           Buffer, BufferIndex,
           Ord(CTagCreateTc), TTransportConnection(FParent).FTransportIdentifier,
           nil, 0) then
        CommonInterfaceTransportLayerWriteDataToModule(
          TTransportConnection(FParent).FHandle,
          TTransportConnection(FParent).FTransportIdentifier,
          TTransportConnection(FParent).FModule,
          Buffer, BufferIndex);
        DataAwaiting := True;                    // Not needed but since we ALWAYS wait for a reply afterwards
        TimeOut := GetTickCount;
    end
    else
    begin
      // If not first time then we MUST be waiting for a reply
      if BufferSize > 2 then
      begin
        // Only act on the correct reply
        if Buffer[0] = CTagCtcReply then
        begin
          if CommonInterfaceTransportLayerDisassembleRTPDU(
               Buffer, BufferIndex, Tag, ConnectionId, DataBytes, DataBytesLength, Status) then
          begin
            DataAvailable := (Status and $80) <> 0;          // Module has more data available ?
            DataAwaiting  := False;                          // Required
            TTransportConnection(FParent).FState := stActive;
            TTransportConnection(FParent).FReceived := 0;
            TimeOut := GetTickCount;                         // Make sure no timeout
          end;
        end;
      end;
      // Check for timeout waiting for correct reply
      if Abs(GetTickCount - TimeOut) > CTimeOut then
        TTransportConnection(FParent).FState := stIdle;
    end;
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'active'
    Notes   : We get here the first time after receiving a correct reply from
              the 'Create_T_C' (InCreation). From that point on we have different
              types of data we can expect. Since these are only send to us by
              the module when specifically requested, we need to poll the
              module regularly. We do this for each polling period if we are not already
              expecting data from the module.
              We expect T_Data_More, T_Data_Last, Request_T_C, Delete_T_C, T_SB
  ------------------------------------------------------------------------------}
  procedure StateActive;
  var
    Success  : Boolean;
    NewId    : Byte;
    ErrorCode: Byte;
  begin
    // Check if their is data from the module
    if BufferSize > 2 then
    begin
      // Increase (debug) counter
      if TTransportConnection(FParent).FReceived <> $FFFF then
        Inc(TTransportConnection(FParent).FReceived)
      else
        TTransportConnection(FParent).FReceived := 0;
      DataAwaiting := False;                               // We don't expect data
      // Here we handle <T_Data_More> and <T_Data_Last>, although
      // <T_Data_More> is not supported at this point ....
      // For <T_Data_More> we would typically exit here (but still awaiting data)
      // and we would append new data to previous received data...
      // Try decoding using the open connection
      Success := CommonInterfaceTransportLayerDisassembleReceiveDataRTPDU(
                   Buffer, BufferIndex, Tag, ConnectionId, DataBytes, DataBytesLength, Status);
      // If no success decode using a not yet open connection
      if not Success then
        Success := CommonInterfaceTransportLayerDisassembleRTPDU(
                     Buffer, BufferIndex, Tag, ConnectionId, DataBytes, DataBytesLength, Status);
      if Success then
      begin
        case Char(Tag) of
          // A status report is returned by the module
          // We, the host, don't have to act on it (<Status> is handled elsewhere)
          CTagSB       :;
          // When we receive a request for a new connection we must make sure
          // we send <New_T_C> before a new connection (with <Create_T_C> etc.),
          // is actually made.
          CTagRequestTc:
            begin
              // First find out if we have a new connection available, if not
              // we need to reply with a <T_C_Error>.
              if CommonInterfaceTransportLayerAllocateTransportConnection(
                   TTransportConnection(FParent).FHandle, NewId) then
              begin
                // New connection has been made available, now send the reply
                // Construct 'New_T_C' data and send
                Tag := Ord(CTagDataLast);
                Status := 0;
                DataBytesLength := 1;
                // Construct data in DATABYTES, with NewId as additional byte
                if CommonInterfaceTransportLayerAssembleCTPDU(
                     DataBytes, DataBytesLength, Ord(CTagNewTc),
                     TTransportConnection(FParent).FTransportIdentifier,
                     @NewId, 1) then
                  if CommonInterfaceTransportLayerAssembleSendDataCTPDU(
                       Buffer, BufferIndex,  Tag,
                       TTransportConnection(FParent).FTransportIdentifier, DataBytes,
                       DataBytesLength, Status) then
                    CommonInterfaceTransportLayerWriteDataToModule(
                      TTransportConnection(FParent).FHandle,
                      TTransportConnection(FParent).FTransportIdentifier,
                      TTransportConnection(FParent).FModule,
                      Buffer, BufferIndex);
                // Reply has been send, now make the new connection 'active'
                CommonInterfaceTransportLayerSetState(NewId, stInCreation);
              end
              else
              begin
                // No new transport connections are available, so report an error
                // Construct 'T_C_Error' data and send
                Tag := Ord(CTagDataLast);
                Status := 0;
                DataBytesLength := 1;
                ErrorCode       := 1;
                // Construct data in DATABYTES, with ErrorCode as additional byte
                if CommonInterfaceTransportLayerAssembleCTPDU(
                     DataBytes, DataBytesLength, Ord(CTagTcError),
                     TTransportConnection(FParent).FTransportIdentifier,
                     @ErrorCode, 1) then
                  if CommonInterfaceTransportLayerAssembleSendDataCTPDU(
                       Buffer, BufferIndex,  Tag,
                       TTransportConnection(FParent).FTransportIdentifier, DataBytes,
                       DataBytesLength, Status) then
                    CommonInterfaceTransportLayerWriteDataToModule(
                      TTransportConnection(FParent).FHandle,
                      TTransportConnection(FParent).FTransportIdentifier,
                      TTransportConnection(FParent).FModule,
                      Buffer, BufferIndex);
              end;
            end;
          // When we receive a delete transport connection from the module we
          // respond with a reply and go into idle state
          CTagDeleteTc :
            begin
              // Construct 'D_T_C_Reply' data and send
              Tag := Ord(CTagDataLast);
              Status := 0;
              DataBytesLength := 1;
              // Construct data in DATABYTES
              if CommonInterfaceTransportLayerAssembleCTPDU(
                   DataBytes, DataBytesLength, Ord(CTagDtcReply),
                   TTransportConnection(FParent).FTransportIdentifier,
                   nil, 0) then
                if CommonInterfaceTransportLayerAssembleSendDataCTPDU(
                     Buffer, BufferIndex, Tag,
                     TTransportConnection(FParent).FTransportIdentifier, DataBytes,
                     DataBytesLength, Status) then
                  CommonInterfaceTransportLayerWriteDataToModule(
                    TTransportConnection(FParent).FHandle,
                    TTransportConnection(FParent).FTransportIdentifier,
                    TTransportConnection(FParent).FModule,
                    Buffer, BufferIndex);
              // We are deleted, go into idle state
              TTransportConnection(FParent).FState := stIdle;
            end;
          // All other tags are not for the transport layer, so pass them through
          // to the session layer  
          else ;
        end;
        DataAvailable := (Status and $80) <> 0;            // Module has more data available ?
        TimeOut := GetTickCount;                           // Make sure no timeout
      end;
    end;

    // If it is indicated that the module has data available then
    // request this data from the module by means of a 'poll' request. We
    // also have to 'poll' regularly so we do this also when there is no
    // reply expected.
    if DataAwaiting then
      DataAvailable := False;                              // Sanity check, should never be needed
    if DataAvailable or
       DataAwaiting then
      PollTimeOut := GetTickCount;                         // If no need for poll, reset timer
    if DataAvailable or
       (Abs(GetTickCount - PollTimeOut) >= CPollingPeriod) then
    begin
      // The module reported that it has data available or we manually poll.
      // We have to request this data.
      // We use 'Send data C_TPDU' since we have an open connection
      Tag := Ord(CTagDataLast);
      Status := 0;
      DataBytesLength := 1;
      // Construct data in DATABYTES first
      if CommonInterfaceTransportLayerAssembleCTPDU(
           DataBytes, DataBytesLength, Ord(CTagRcv),
           TTransportConnection(FParent).FTransportIdentifier,
           nil, 0) then
        if CommonInterfaceTransportLayerAssembleSendDataCTPDU(
             Buffer, BufferIndex,  Tag,
             TTransportConnection(FParent).FTransportIdentifier, DataBytes,
             DataBytesLength, Status) then
          CommonInterfaceTransportLayerWriteDataToModule(
            TTransportConnection(FParent).FHandle,
            TTransportConnection(FParent).FTransportIdentifier,
            TTransportConnection(FParent).FModule, Buffer, BufferIndex);
      DataAvailable := False;                              // Module has no data available
      DataAwaiting  := True;                               // We expect a reply
      TimeOut := GetTickCount;                             // Reset timeout
    end
    else
    begin
      // Check for timeout waiting for data
      // A timeout will revert us to the idle state
      if DataAwaiting then
        if Abs(GetTickCount - TimeOut) > CTimeOut then
          TTransportConnection(FParent).FState := stIdle;
    end;
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'in deletion'
    Notes   : We expect D_T_C_Reply which includes T_SB
  ------------------------------------------------------------------------------}
  procedure StateInDeletion;
  begin
    // If we get here the first time then we need to send a 'Delete_T_C'
    if NewState then
    begin
      // Construct 'Delete_T_C' data and send
      Tag := Ord(CTagDataLast);
      Status := 0;
      DataBytesLength := 1;
      if CommonInterfaceTransportLayerAssembleCTPDU(
           DataBytes, DataBytesLength, Ord(CTagDeleteTc),
           TTransportConnection(FParent).FTransportIdentifier,
           nil, 0) then
        if CommonInterfaceTransportLayerAssembleSendDataCTPDU(
             Buffer, BufferIndex,  Tag,
             TTransportConnection(FParent).FTransportIdentifier, DataBytes,
             DataBytesLength, Status) then
          if CommonInterfaceTransportLayerWriteDataToModule(
               TTransportConnection(FParent).FHandle,
               TTransportConnection(FParent).FTransportIdentifier,
               TTransportConnection(FParent).FModule,
               Buffer, BufferIndex) then
            DataAwaiting := True;                            // Not needed but since we ALWAYS wait for a reply afterwards
        TimeOut := GetTickCount;
    end
    else
    begin
      // If not first time then we MUST be waiting for a reply
      if BufferSize > 2 then
      begin
        // Only act on the correct reply
        if Buffer[0] = CTagDtcReply then
        begin
          if CommonInterfaceTransportLayerDisassembleRTPDU(
               Buffer, BufferIndex, Tag, ConnectionId, DataBytes, DataBytesLength, Status) then
            TTransportConnection(FParent).FState := stIdle;
        end;
      end;
      // Check for timeout waiting for correct reply
      if Abs(GetTickCount - TimeOut) > CTimeOut then
        TTransportConnection(FParent).FState := stIdle;
    end;
  end;

begin
  GetMem(Buffer, CBufferSize);
  GetMem(DataBytes, CBufferSize);
  try
    PreviousState := stUnassigned;
    repeat
      // Update state change
      NewState := (TTransportConnection(FParent).FState <> PreviousState);
      PreviousState := TTransportConnection(FParent).FState;
      // Get data in buffer if anything available
      BufferSize := CBufferSize;
      if CommonInterfaceLinkLayerGetTransportData(
           TTransportConnection(FParent).FTransportIdentifier,
           TTransportConnection(FParent).FModule, Buffer, BufferSize) then
      begin
        // We received something
        Sleep(0);
      end
      else
        BufferSize := 0;
      // State machine
      case TTransportConnection(FParent).FState of
        stAssigned  : TTransportConnection(FParent).FState := stIdle;
        stIdle      : StateIdle;
        stInCreation: StateInCreation;
        stActive    : StateActive;
        stInDeletion: StateInDeletion;
        stUnassigned: Suspend;
      end;
      Sleep(1);
    until Terminated;
  finally
    FreeMem(Buffer, CBufferSize);
    FreeMem(DataBytes, CBufferSize);
  end;
end;


initialization
  CommonInterfaceLock   := TCriticalSection.Create;
  for Loop := Low(Modules) to High(Modules) do
    Modules[Loop] := nil;
  for Loop := Low(TransportConnections) to High(TransportConnections) do
    TransportConnections[Loop] := TTransportConnection.Create(Loop);

finalization
  RemoveTransportConnections;
  CommonInterfaceTransportLayerRemoveModules;
  CommonInterfaceLock.Free;
end.

