{******************************************************************************}
{ FileName............: CommonInterfaceLinkLayer                               }
{ Project.............: DVB-S                                                  }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Common interface support (link layer)                                       }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ The link layer (on the command interface) fragments and reassembles data     }
{ to and from the physical layer. Since the physical has a limited buffer size }
{ data is packetized such that they are acceptable for the physical layer.     }
{                                                                              }
{ The layers involved:                                                         }
{                  APPLICATION                                                 }
{               Application layer                                              }
{                 Session layer                                                }
{                Transport layer                                               }
{          ****     Link layer                                                 }
{                 Physical layer                                               }
{                    HARDWARE                                                  }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030726 - Initial release                                           }
{******************************************************************************}
unit CommonInterfaceLinkLayer;

interface
uses
  Windows;

  function CommonInterfaceLinkLayerGetTransportData
             (TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
  function CommonInterfaceLinkLayerReadDataFromModule
             (Handle: THandle; var TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
  function CommonInterfaceLinkLayerWriteDataToModule
             (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
  function CommonInterfaceLinkLayerCheckModule
             (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
  function CommonInterfaceLinkLayerResetModule
             (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;

  function CommonInterfaceLinkLayerGetLengthField
             (Buffer: PChar; var BufferIndex: Word; var LengthValue: Dword): Boolean;

implementation
uses
  Classes,
  CommonInterfacePhysicalLayer,
  Saa7146aDebi,
  SyncObjs,
  SysUtils;

const
  CQueueTimeOut = 300;                           // Timeout allowed before data is removed from queu
  CQueueSize    = 2048;                          // Buffer size of each queue

type
  TModule = class(TObject)
  private
    { Private declarations }
    FBufferSize: Word;                           // Size of negociated transfer buffers
  protected
  public
  published
  end;

  TQueueItem = record
    Buffer               : PChar;                // Pointer to buffer
    BufferSize           : Word;                 // Max size of input buffer
    BufferSizeIn         : Word;                 // Data in buffer
    TransportConnectionId: Byte;                 // Transport connection identifier
    TickCount            : DWord;                // Time of update
  end;

  TLinkThread = class(TThread)
  private
    FQueue   : array[0..0, 0..15] of TQueueItem; // 0..0 == 1 module
                                                 // Per module 16 messages
    FQueueIn : array[0..0] of Word;              // Per module one queue in
    FQueueOut: array[0..0] of Word;              // Per module one queue out
  protected
    procedure Execute; override;
  end;



var
  CommonInterfaceLock: TCriticalSection;         // Prevents multiple operations which interfere
  Modules            : array[0..3] of TModule;   // Modules supported
  Loop               : Integer;
  LinkThread         : TLinkThread;              // Thread handling link layer
  GHandle            : THandle;                  // Handle to driver (this limits the use to a single DVB-S card)


{------------------------------------------------------------------------------
  Params  : <Buffer>       Pointer to data buffer
            <BufferIndex>  Points to the length field data
  Returns : <Result>       True if success
            <BufferIndex>  Points to after the length field data
            <LengthValue>  Retrieved length

  Descript: Get the length from the length field.
  Notes   : 'Limited' support for a 4 bytes length.
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerGetLengthField(Buffer: PChar; var BufferIndex: Word; var LengthValue: Dword): Boolean;
var
  LengthBytes: Byte;
  Loop       : Byte;
begin
  Result := False;
  try
    LengthBytes := Ord(Buffer[BufferIndex]);
    Inc(BufferIndex);
    // If the length is < 128 then it is contained within a single byte
    if (LengthBytes and $80) = 0 then
      LengthValue := LengthBytes
    else
    begin
      // A length > 128 uses multiple bytes. We allow only a limited length.
      LengthBytes := LengthBytes and $7F;
      if LengthBytes > SizeOf(LengthValue) then
        Exit;
      LengthValue := 0;
      for Loop := 0 to LengthBytes - 1 do
      begin
        // MS bytes are followed by the LS bytes
        LengthValue := LengthValue shl 8;
        LengthValue := LengthValue or Ord(Buffer[BufferIndex]);
        Inc(BufferIndex);
      end;
    end;
    Result := True;
  except
    // If invalid buffer ....
    LengthValue := 0;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Link layer thread
  Notes   : Handles reception of data and queueing it. Processes all modules
            and all buffers. Out of date data is automatically removed.
------------------------------------------------------------------------------}
procedure TLinkThread.Execute;
var
  CurrentTime : Dword;
  Module      : Byte;
  Loop        : Byte;
  TransportId : Byte;
  BufferSize  : Word;
  BufferLength: Dword;
  BufferIndex : Word;
begin
  // Setup queue
  for Module := Low(FQueue) to High(FQueue) do
  begin
    FQueueIn[Module]  := Low(FQueue[0]);
    FQueueOut[Module] := FQueueIn[Module];
    for Loop := Low(FQueue[0]) to High(FQueue[0]) do
    begin
      FQueue[Module, Loop].BufferSize := CQueueSize;
      FQueue[Module, Loop].BufferSizeIn := 0;
      GetMem(FQueue[Module, Loop].Buffer, CQueueSize);
    end;
  end;
  try
    Module := Low(FQueue);
    // Wait for end of or first communication with layer
    repeat
      Sleep(10);
    until Terminated or (GHandle <> INVALID_HANDLE_VALUE);
    if not Terminated then
    repeat
      // Data which is at their 'end of life' (not claimed) are removed
      CurrentTime := GetTickCount;
      for Loop := Low(FQueue[0]) to High(FQueue[0]) do
      begin
        if FQueue[Module, Loop].BufferSizeIn > 0 then
          if Abs(CurrentTime - FQueue[Module, Loop].TickCount) > CQueueTimeOut then
            FQueue[Module, Loop].BufferSizeIn := 0;
      end;

      // Read data from module and place in queue
      BufferSize := FQueue[Module, FQueueIn[Module]].BufferSize;
      if CommonInterfaceLinkLayerReadDataFromModule(
           GHandle, TransportId, Module, FQueue[Module, FQueueIn[Module]].Buffer, BufferSize) then
      begin
        if BufferSize > 0 then
        begin
          FQueue[Module, FQueueIn[Module]].TickCount    := GetTickCount;
          FQueue[Module, FQueueIn[Module]].BufferSizeIn := BufferSize;
          // For the transport connection identifier we must 'decode' the data somewhat.
          // We have to skip the tag and length field to get at the transport connection
          // id.
          BufferIndex := 1;
          if CommonInterfaceLinkLayerGetLengthField(FQueue[Module, FQueueIn[Module]].Buffer, BufferIndex, BufferLength) then
            FQueue[Module, FQueueIn[Module]].TransportConnectionId := Ord(FQueue[Module, FQueueIn[Module]].Buffer[BufferIndex])
          else
            FQueue[Module, FQueueIn[Module]].TransportConnectionId := 0;
          Inc(FQueueIn[Module]);
          // Circular buffer
          if FQueueIn[Module] > High(FQueue[0]) then
            FQueueIn[Module] := Low(FQueueIn[Module]);
        end;
      end;
      // Next run handles next module
      Inc(Module);
      if Module > High(FQueue) then
        Module := Low(FQueue);

      Sleep(1);
    until Terminated;
  finally
    for Module := Low(FQueue) to High(FQueue) do
    begin
      for Loop := Low(FQueue[0]) to High(FQueue[0]) do
        FreeMem(FQueue[Module, Loop].Buffer, FQueue[Module, Loop].BufferSize);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <TransportConnectionId>  Identifier to find data for
            <Module>                 Module to 'access'
            <Buffer>                 Pointer to buffer
            <BufferSize>             Available buffer size
  Returns : <Result>                 True if success
                                     False if buffersize smaller than needed
                                     (required buffer size in <BufferSize>
                                     False if module id error
            <BufferSize>             Returned buffer size (or required when
                                     buffer size error)

  Descript: Get data which correspondents with transport connection (and module).
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerGetTransportData
           (TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
var
  Loop     : Byte;
  Found    : Boolean;
  DataIndex: Word;
begin
  Result := False;
  if not Assigned(LinkThread) then
    Exit;
  if not Module in [Low(LinkThread.FQueue)..High(LinkThread.FQueue)] then
    Exit;
  Loop := LinkThread.FQueueOut[Module];
  Found := False;
  repeat
    if LinkThread.FQueue[Module, Loop].BufferSizeIn > 0 then
      if LinkThread.FQueue[Module, Loop].TransportConnectionId = TransportConnectionId then
      begin
        Found := True;
        if LinkThread.FQueue[Module, Loop].BufferSizeIn <= BufferSize then
        begin
          for DataIndex := 0 to LinkThread.FQueue[Module, Loop].BufferSizeIn - 1 do
            Buffer[DataIndex] := LinkThread.FQueue[Module, Loop].Buffer[DataIndex];
          BufferSize := LinkThread.FQueue[Module, Loop].BufferSizeIn;
          // Remove from queue
          LinkThread.FQueue[Module, Loop].BufferSizeIn := 0;
          LinkThread.FQueueOut[Module] := Loop;
        end
        else
        begin
          // Report required buffersize
          BufferSize := LinkThread.FQueue[Module, Loop].BufferSizeIn;
        end;
      end;
    Inc(Loop);
    if Loop > High(LinkThread.FQueue[0]) then
      Loop := Low(LinkThread.FQueue[0]);
  until Found or (Loop = LinkThread.FQueueIn[Module]);
  Result := Found;
end;


{------------------------------------------------------------------------------
  Params  : <Module> Module to remove
  Returns : -

  Descript: Remove the module (from memory).
  Notes   :
------------------------------------------------------------------------------}
procedure CommonInterfaceLinkLayerRemoveModule(Module: Byte);
begin
  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  if Assigned(Modules[Module]) then
  begin
    try
    finally
      FreeAndNil(Modules[Module]);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Remove all modules.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerRemoveModules: Boolean;
var
  Loop: Byte;
begin
  for Loop := Low(Modules) to High(Modules) do
    CommonInterfaceLinkLayerRemoveModule(Loop);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Module>  Module to add
  Returns : <Result>  True if success (module has been added)

  Descript: Add the module (memory and initialization) if present.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerAddModule(Handle: THandle; Module: Byte): Boolean;
begin
  Result := False;

  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  // Remove the module if already present
  CommonInterfaceLinkLayerRemoveModule(Module);
  // Create the module
  Modules[Module] := TModule.Create;
  Modules[Module].FBufferSize := 0;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 <Saa7146aCreateFile> handle
            <TransportConnectionId>  Identifier
            <Module>                 Module to write to
            <Buffer>                 Pointer to buffer to write
            <BufferSize>             Size of buffer
  Returns : <Result>                 True if success
                                     False if module not ready

  Descript: Write data to module. Adds header information and splits it up
            in several packages if necessary.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerWriteDataToModule
  (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
var
  LinkBuffer    : PChar;
  SourceIndex   : Word;
  TargetIndex   : Word;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := False;
    if not Module in [Low(Modules)..High(Modules)] then
      Exit;
    if not Assigned(Modules[Module]) then
      Exit;
    if Modules[Module].FBufferSize < 3 then
      Exit;
    // When writing data we need to add 2 bytes to each packet. If it does not
    // fit in the physical buffer size then we need to split it up in several packets
    // First allocate the buffer we are using
    GetMem(LinkBuffer, Modules[Module].FBufferSize + 2);
    try
      SourceIndex := 0;
      repeat
        TargetIndex := 2;
        // Setup header data
        LinkBuffer[0] := Chr(TransportConnectionId);
        // Fill in data
        repeat
          LinkBuffer[TargetIndex] := Buffer[SourceIndex];
          Inc(SourceIndex);
          Inc(TargetIndex);
        until (SourceIndex = BufferSize) or
              (TargetIndex = Modules[Module].FBufferSize);
        // Setup more/last header data
        if SourceIndex < BufferSize then
          LinkBuffer[1] := #$80
        else
          LinkBuffer[1] := #$00;
        if not CommonInterfacePhysicalLayerWriteDataToModule(Handle, Module, LinkBuffer, TargetIndex) then
          Exit;
      until SourceIndex = BufferSize;

      Result := True;
    finally
      FreeMem(LinkBuffer, Modules[Module].FBufferSize + 2);
    end;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 <Saa7146aCreateFile> handle
            <TransportConnectionId>  Identifier
            <Module>                 Module to check
            <Buffer>                 Pointer to buffer to read to
            <BufferSize>             Size of buffer
  Returns : <Result>                 True if success
                                     False if buffersize smaller than needed (data is lost)
                                     False if module not ready
            <BufferSize>             Data read
                                     0 if no data available

  Descript: Read data from module. Removes header information and combines
            multiple packets if necessary.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerReadDataFromModule
  (Handle: THandle; var TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
var
  LinkBuffer : PChar;
  SourceIndex: Word;
  TargetIndex: Word;
  FirstId    : Boolean;
  DataRead   : Word;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := False;
    if not Module in [Low(Modules)..High(Modules)] then
      Exit;
    if not Assigned(Modules[Module]) then
      Exit;
    if Modules[Module].FBufferSize < 3 then
      Exit;
    // First allocate the buffer we are using
    GetMem(LinkBuffer, Modules[Module].FBufferSize + 2);
    try
      FirstId := True;
      TargetIndex := 0;
      repeat
        DataRead := Modules[Module].FBufferSize;
        if not CommonInterfacePhysicalLayerReadDataFromModule(Handle, Module, LinkBuffer, DataRead) then
          Exit;
        // If no data available
        if DataRead = 0 then
        begin
          // Report no data available in size
          BufferSize := 0;
          Result := True;
          Exit;
        end;
        // Must be at least 3 bytes received
        if DataRead < 3 then
          Exit;
        // Check for identical transport id's
        // Note: This is NOT necessarily the transport id of the transport layer,
        //       but this is the one from the link protocol data unit
        //       The id for the transport layer is contained in the data
        if FirstId then
          TransportConnectionId := Ord(LinkBuffer[0]);
        FirstId := False;
        if TransportConnectionId <> Ord(LinkBuffer[0]) then
          Exit;
        // Skip header
        SourceIndex := 2;
        repeat
          Buffer[TargetIndex] := LinkBuffer[SourceIndex];
          Inc(TargetIndex);
          Inc(SourceIndex);
        until (SourceIndex = DataRead) or
              (TargetIndex = BufferSize);
      until ((Ord(LinkBuffer[1]) and $80) = 0);              // 'Last' indication
      // Check for situation were buffer migth not have enough space
      if TargetIndex = BufferSize then
      begin
        if SourceIndex <> DataRead then
          Exit;
      end;
      // Return size
      BufferSize := TargetIndex;
    finally
      FreeMem(LinkBuffer, Modules[Module].FBufferSize + 2);
    end;
    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;



{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <ModuleIsPresent> True if module is available
            <ModuleHasData>   True if module has data available

  Descript: Check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerCheckModule
  (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfacePhysicalLayerCheckModule(Handle, Module, ModuleIsPresent, ModuleHasData);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <BufferSize>      Negotiated buffer size

  Descript: Reset and check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceLinkLayerResetModule
  (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;
begin
  if GHandle = INVALID_HANDLE_VALUE then
    GHandle := Handle;
  CommonInterfaceLock.Acquire;
  try
    CommonInterfaceLinkLayerRemoveModule(Module);
    if CommonInterfaceLinkLayerAddModule(Handle, Module) then
    begin
      Result := CommonInterfacePhysicalLayerResetModule(Handle, Module, Modules[Module].FBufferSize);
      BufferSize := Modules[Module].FBufferSize;
    end
    else
      Result := False;
  finally
    CommonInterfaceLock.Release;
  end;
end;


initialization
  GHandle := INVALID_HANDLE_VALUE;
  CommonInterfaceLock   := TCriticalSection.Create;
  for Loop := Low(Modules) to High(Modules) do
    Modules[Loop] := nil;
  LinkThread := TLinkThread.Create(False);

finalization
  if Assigned(LinkThread) then
  begin
    LinkThread.Terminate;
    LinkThread.Resume;
    LinkThread.WaitFor;
  end;
  CommonInterfaceLinkLayerRemoveModules;
  CommonInterfaceLock.Free;
end.

