{*********************************************************************************
 *                                                                               *
 * WinSTB, TT-PC line budget card interface (NOVA)                               *
 *                                                                               *
 * Copyright (C) 2003 M.Majoor                                                   *
 *                                                                               *
 *                                                                               *
 * This program is free software; you can redistribute it and/or                 *
 * modify it under the terms of the GNU General Public License                   *
 * as published by the Free Software Foundation; either version 2                *
 * of the License, or (at your option) any later version.                        *
 *                                                                               *
 *                                                                               *
 * This program is distributed in the hope that it will be useful,               *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 * GNU General Public License for more details.                                  *
 *                                                                               *
 *                                                                               *
 * You should have received a copy of the GNU General Public License             *
 * along with this program; if not, write to the Free Software                   *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.    *
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html                *
 *                                                                               *
 *                                                                               *
 * The author can be reached at m.majoor@majority.nl                             *
 * Homepage: http://www.majority.nl                                              *
 *********************************************************************************}

{******************************************************************************}
{ FileName............: UTTPClineBudget                                        }
{ Project.............: WinSTB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.12                                                   }
{------------------------------------------------------------------------------}
{ TechnoTrend PC-line budget card (NOVA) support.                              }
{                                                                              }
{ ALL handling is done here or in the support routines. Nothing is handled by  }
{ the device driver except for acknowledging interrupts.                       }
{------------------------------------------------------------------------------}
{                                                                              }
{ Note: Requires an installed and working SAA7146A.SYS driver!                 }
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030809 - Initial release                                           }
{  1.01   20030810 - Added priority for packet thread (by default higher)      }
{                  - Added critical section for changing/reading <Filters>     }
{  1.02   20030813 - LNB1 settings used for DiSEqC 1.2                         }
{                  - Only 'valid' PIDs are recorded                            }
{                  - Additional buffering added                                }
{  1.03   20030819 - Included lower thread priorities; mainly for testing of   }
{                    buffering                                                 }
{                  - Form now tabbed                                           }
{                  - Added local processing of PIDs                            }
{                  - DiSEqC 1.2 issues a 'stop movement' command for positioner}
{                    at initialization (this typically switches the positioner }
{                    to level 1.2 commands)                                    }
{                  - Specific streams can be included in recording if desired  }
{                  - Video rate is deduced                                     }
{                  - Software CSA handled internally                           }
{  1.04   20030820 - Very slight modification in UCSAEmu (moved 'n=0' check up)}
{                    so speed improved....                                     }
{                  - Added slave support                                       }
{                    . no initialization of SAA7146A, instead DMA setup by     }
{                      master is readout                                       }
{                    . DiSEqC function disabled                                }
{                    . Signal strength function disabled                       }
{                  - Added 'DisplayDuringRecording' setting, since application }
{                    handles this incorrectly                                  }
{  1.05   20030827 - Added 'DisplayOff' setting which disables video/audio     }
{                    Also added them as special remote control commands        }
{                      'DisplayOn' / 'DisplayOff' / 'DisplayToggle'            }
{                  - Manual DiSEqC command now also refresh frequency and such }
{                    (after sending DiSEqC command lock usually gone)          }
{                    Note: instead of refreshing we could also just reset the  }
{                          derotator loop, but this way we use 'standard' calls}
{                  - Exit now takes being a slave into account (prior to this  }
{                    version all usually de-initialization was done, leaving a }
{                    master inoperable)                                        }
{                  - Build data also shown                                     }
{                  - Remote control command also support multiple devices now  }
{                  - Additional check which generated range check in timer     }
{                    event added                                               }
{  1.06   20030903 - 'TSBufferSize' setting added for number of buffers to use }
{                  - Removed single buffer usage                               }
{                  - Detecting master is different: any kind of DMA > 128 kB   }
{                    indicates a master.                                       }
{  1.07   20030905 - Thread destruction uses <CardHandle> for manual           }
{                    notifications -> should prevent deadlock wespecially when }
{                    exiting the application when tuned to a transponder which }
{                    does not contain a data stream.                           }
{                  - Uses seperate unit for buffering mechanism                }
{                    . memory allocation more dynamic -> will use less memory  }
{                      if not able to allocate the requested amount            }
{                  - Using <FileStream> instead of <FileWrite> method          }
{  1.08   20030906 - Range check bug in timer routine corrected (videorate)    }
{                  - Common 1 s timing included in timer                       }
{                  - Both internal and external decrypting possible            }
{  1.09   20030913 - Added log capabilities <LogLevel> (TTPClineBudget.log)    }
{                    (only with local INI file)                                }
{                  - Added <LogClear> option                                   }
{                  - Added local INI file capability                           }
{                  - <StrToInt> changed into <Val>                             }
{                  - Added <HandleMdApi> setting                               }
{                  - Added MDAPI calls so they could be handled by the driver  }
{  1.10   20030920 - Support for SU1278 tuner added                            }
{                  - Using generic routines (no index in very large buffer)    }
{  1.10-063                                                                    }
{         20031116 - Compiled for V0.63 of WinSTB which uses different         }
{                    <UDVBHardware>                                            }
{  1.11   20031123 - The PIDs to be recorded are explicitly added to the       }
{                    filters if they were removed by some plugin.              }
{                  - <DvbSetChannel> PIDS explicitly stored so they can be used}
{                    even if no filters are set for those                      }
{                    Only <DvbSetChannel> now 'resets' these local PIDs        }                                                          
{  1.11-063                                                                    }
{         20031123 - Compiled for V0.63 of WinSTB which uses different         }
{                    <UDVBHardware>                                            }
{  1.12   20040422 - Additional support for other cards                        }
{******************************************************************************}
{$IFDEF ISDLL}
  library TTPClineBudget;
{$ELSE}
  unit UTTPClineBudget;
  interface
{$ENDIF}

uses
  Classes, ComCtrls, Dialogs, ExtCtrls, Forms, Gauges, Graphics, IniFiles,
  Mask, Messages,
  StdCtrls, SyncObjs, SysUtils, Windows,
  UMDAPI, UDVBHardware,
{$IFDEF USESOFTCSA}
  UCSAEmu,
{$ENDIF}
  Dvb, DvbStreamBuffering,
  Saa7146aI2c, Saa7146aInterface, Saa7146aIoControl, Saa7146aRegisters,
  Saa7146aGpio, Stv0299bRegisters,
  DirectShow9, ActiveX, DSUtil,
  Controls,
  Grids;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    tmrUpdate: TTimer;
    pgPageControl: TPageControl;
    tsDebug: TTabSheet;
    grpInformatio: TGroupBox;
    txtDebugStr: TStaticText;
    StaticText17: TStaticText;
    StaticText16: TStaticText;
    StaticText10: TStaticText;
    txtDebug: TStaticText;
    txtFiltersDefined: TStaticText;
    txtFilters: TStaticText;
    txtPackets: TStaticText;
    StaticText19: TStaticText;
    txtRemoteControl: TStaticText;
    StaticText24: TStaticText;
    txtOvertaken: TStaticText;
    txtMs: TStaticText;
    StaticText25: TStaticText;
    mmoDriver: TMemo;
    grpPid: TGroupBox;
    StaticText20: TStaticText;
    StaticText21: TStaticText;
    StaticText22: TStaticText;
    StaticText23: TStaticText;
    txtPidVideo: TStaticText;
    txtPidPmt: TStaticText;
    txtPidAudio: TStaticText;
    txtPidPcr: TStaticText;
    tsDiSEqC: TTabSheet;
    GroupBox2: TGroupBox;
    btnStepWest: TButton;
    btnStepEast: TButton;
    StaticText11: TStaticText;
    btnStopMovement: TButton;
    btnResetPositioner: TButton;
    btnGoEast: TButton;
    btnGoWest: TButton;
    StaticText12: TStaticText;
    btnSetEastLimit: TButton;
    btnSetWestLimit: TButton;
    btnDisableLimits: TButton;
    btnStorePosition: TButton;
    udPosition: TUpDown;
    btnGotoPosition: TButton;
    StaticText13: TStaticText;
    StaticText14: TStaticText;
    StaticText15: TStaticText;
    GroupBox1: TGroupBox;
    btnBurstA: TButton;
    btnBurstB: TButton;
    btnDiseqcAOptionA: TButton;
    btnDiseqcBOptionA: TButton;
    btnDiseqcAOptionB: TButton;
    btnDiseqcBOptionB: TButton;
    StaticText9: TStaticText;
    ggLock: TGauge;
    stDeviationSymbolrate: TStaticText;
    stDeviation: TStaticText;
    StaticText4: TStaticText;
    StaticText7: TStaticText;
    mskSymbolRate: TMaskEdit;
    mskFrequency: TMaskEdit;
    StaticText3: TStaticText;
    StaticText1: TStaticText;
    StaticText2: TStaticText;
    rgPolarity: TRadioGroup;
    StaticText8: TStaticText;
    txtNoise: TStaticText;
    txtStatus: TStaticText;
    StaticText6: TStaticText;
    StaticText5: TStaticText;
    txtPower: TStaticText;
    StaticText35: TStaticText;
    txtVideoRate: TStaticText;
    StaticText36: TStaticText;
    StaticText37: TStaticText;
    StaticText38: TStaticText;
    StaticText39: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnStopMovementClick(Sender: TObject);
    procedure btnResetPositionerClick(Sender: TObject);
    procedure btnGoEastClick(Sender: TObject);
    procedure btnGoWestClick(Sender: TObject);
    procedure btnSetEastLimitClick(Sender: TObject);
    procedure btnSetWestLimitClick(Sender: TObject);
    procedure btnDisableLimitsClick(Sender: TObject);
    procedure btnStorePositionClick(Sender: TObject);
    procedure btnStepEastClick(Sender: TObject);
    procedure btnStepWestClick(Sender: TObject);
    procedure btnGotoPositionClick(Sender: TObject);
    procedure udPositionChangingEx(Sender: TObject;
      var AllowChange: Boolean; NewValue: Smallint;
      Direction: TUpDownDirection);
    procedure btnDiseqcAOptionAClick(Sender: TObject);
    procedure btnDiseqcBOptionAClick(Sender: TObject);
    procedure btnDiseqcAOptionBClick(Sender: TObject);
    procedure btnDiseqcBOptionBClick(Sender: TObject);
    procedure btnBurstAClick(Sender: TObject);
    procedure btnBurstBClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;
{$IFDEF ISDLL}
  {$R UTTPClineBudget.DFM}
{$ELSE}
  procedure TtPcLineBudgetFillDVBHardwareInfo(DVBHardware: PByte); stdcall;
  implementation
  {$R *.DFM}
{$ENDIF}

const
  MAX_FILTER_COUNT = 255;
  CInfraredAddress = $4000;
  Version          = $0112;
  Build            = $20040422;

type
  TRecordingStates = set of (rsIdle, rsStartRecording, rsRecording, rsStopRecording);

  // Thread used for acquiring data packerts
  TDataThread = class(TThread)
  private
    { Private declarations }
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
    BufferId      : Dword;             // Buffer identifier for DMA data
    PacketBuffers : Word;              // Number of buffers used
  protected
    procedure Execute; override;
  end;

  // Thread used for remote control
  TRcThread = class(TThread)
  private
    { Private declarations }
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
  protected
    procedure Execute; override;
  end;

  TMDAPIFilterProc = procedure( hFilter: DWORD; Len: Cardinal; Buf: PByteArray); cdecl;

  // PID filter definition
  TFilter = record
    Name            : string;                    // Name of filter
    Active          : Boolean;                   // Filter is (to be) set active
    InRecording     : Boolean;                   // True if used in recording
    Pid             : Word;                      // Only used for cross reference
    CallBackFunction: TMDAPIFilterProc;          // Function to call
  end;
{$IFDEF USESOFTCSA}
  // SoftCSA
  TSetKeyProc     = procedure(Key: PByteArray; Keystruct: PByteArray); cdecl;
  TCSADecryptProc = procedure(Keystruct: PByteArray; Encrypted: PByteArray); cdecl;
{$ENDIF}

var
{$IFDEF ISDLL}
  ExitSave    : Pointer;
{$ENDIF}

  TTPClineBudgetForm: TfrmMain;

  // Settings
  DisplayDuringRecording: Boolean = True;
  DisplayOff            : Boolean = False;

  // Driver
  CardHandle  : THandle;                         // Handle to driver
  CardInUse   : Integer;                         // Number of the card
  IsSlave     : Boolean;                         // True if act as slave
                                                 // A 'slave' disables some functions

  DvbCard           : string;                    // DVB card identifier used
  DvbCardPreset     : Boolean;                   // True if correct DvbCard identifier used

  TunerType         : Byte;                      // Tuner type
  SynthesizerAddress: Byte;                      // Synthesizer address tuner
  TunerReset        : Byte;                      // GPIO line for reset ($FF if not to be used)
  LNBPolarity       : Byte;                      // OPx line to use for polarity ($FF if not to be used)
  LNBOnOff          : Byte;                      // OPx line to use for on/off ($FF if not to be used)

  IniFile     : TMemIniFile;                     // Ini file

  // Packet data
  ThreadHandle     : THandle;                    // Handle to driver in thread
  PacketThread     : TDataThread;                // Thread handling packets
  StreamDataBuffer : TDvbTransportPackets;       // Received data
  RecordingState   : TRecordingStates;           // Recording states
  VideoInRecording : Boolean;                    // True if to be included in recordings
  AudioInRecording : Boolean;                    // True if to be included in recordings
  PmtInRecording   : Boolean;                    // True if to be included in recordings
  PcrInRecording   : Boolean;                    // True if to be included in recordings
  EcmInRecording   : Boolean;                    // True if to be included in recordings
  SidInRecording   : Boolean;                    // True if to be included in recordings
  Ac3InRecording   : Boolean;                    // True if to be included in recordings
  TeletextInRecording: Boolean;                  // True if to be included in recordings
  FileStream       : TFileStream;                // Filestream for recording
  LogStream        : TFileStream;                // Filestream for log
  LogLevel         : Byte;                       // LogLevel (00 = no log)
                                                 //  $x0 = No log
                                                 //  $x1 = Lowest level
                                                 //  $x2 = Command errors
                                                 //  $x3 = Procedure/Function entries
                                                 //  $x4 = Procedure/Function specifics
                                                 //  $x5 = Generic procedure/Function entries
  HandleMdApi      : Boolean;                    // True will handle more MDAPI commands
                                                 // than only the DVB_COMMAND

  // Remote control
  RcThread  : TRcThread;                         // Thread handling remote control
  RcRPS0    : TSaa7146aDmaBuffer;                // DMA buffer for RPS0 program
  RcRPS0Data: TSaa7146aDmaBuffer;                // DMA buffer for RPS data
  RcKey1    : Byte;                              // First remote control key detected
  RcKey2    : Byte;                              // Second remote control key detected
  RcRepeat  : Boolean;                           // Repeat information remote control
  RcDevice  : Byte;                              // Device identification remote control

  // DiSEqC
  DiSEqCRepeats: Byte;                           // Number of repeats for DiSEqC commands
  DISEQCType   : string;                         // DiSEqC type to use
  SetDiSEqCSource       : string;                // Last used settings <SetDiSEqc> (for refresh purposes)
  SetDiSEqCLNB          : string;
  SetDiSEqCPolarization : Char;
  SetDiSEqCFrequency    : Dword;
  SetDiSEqCSymbolRate   : Dword;

  //
  PDvbHardwareAPI     : PDvbHardware;            // Pointer to API data
  Filters             : array[1..MAX_FILTER_COUNT] of TFilter; // Pid filters
  FilterCrossReference: array[0..$1FFF] of Word; // Pid index references to <Filters>
  FiltersDefined      : Word;                    // Number of active filters
  FiltersDefinedPids  : string;                  // All active filter Pids as ASCII
  FilterLock          : TCriticalSection;        // Prevents changing filters while they are being in use


  ChannelSet     : Boolean;                      // Used for startup

  // Debug
  Debug              : Dword;                    // Generic debug
  DebugStr           : string;                   // Generic debug
  PacketBufferCount  : Word;                     // Counts received buffers (multiple packets)
  PacketPreviousCount: Word;                     // Counts received buffers previous check
  UpdatePreviousTimer: Dword;                    // Previous tick count checked
  VideoPacketCount        : Word;                // Counts received video packets
  VideoPacketPreviousCount: Word;                // Counts received buffers previous check
  FilterCount      : Word;                       // Counts filter procedures called
  PidVideo         : Word;                       // Video Pid
  PidAudio         : Word;                       // Audio Pid
  PidPmt           : Word;                       // PMT Pid
  PidPcr           : Word;                       // PCR Pid
  PidEcm           : Word;                       // ECM Pid
  PidSid           : Word;                       // Sid Pid
  PidAc3           : Word;                       // AC3 Pid
  PidTeletext      : Word;                       // Teletext Pid
  Overtaken        : Dword;                      // Number of packets which would have been lost without extra buffering
                                                 // Since we DO use extra buffering we (hopefully) have not lost anything


{$IFDEF USESOFTCSA}
  // SoftCSA
  InternalCSA      : Boolean;                  // True indicates internal mechanism to use
  DLLID            : Integer;
  SetKeyProc       : TSetKeyProc;
  CSADecryptProc   : TCSADecryptProc;
  KeyStruct        : array[0..800] of Byte;
  Key              : array[0..15] of Byte;
  KeysOdd          : TBlock;
  KeysEven         : TBlock;
  KeysPreviousOdd  : TBlock;
  KeysPreviousEven : TBlock;
  OddKeyFound      : Boolean = False;
  EvenKeyFound     : Boolean = False;
  KeyFound         : Boolean = False;
  SAA_COMMAND      : array [0..13] of Byte = ($10,4,5,0,0,0,0,0,0,0,0,0,0,0);
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : <LogString>  Data to log
            <Level>      Log level: if this number if <= <LogLevel> then
                         it is logged
                         $8x indicates that the string is intended for
                             displaying on form too.
  Returns : -

  Descript: Write data to log. A timestamp is added.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ToLog(LogString: string; Level: Byte);
{$IFDEF USELOG}
var
  NewLog: string;
{$ENDIF}
begin
  if ((Level and $80) <> 0) and
     Assigned(TTPClineBudgetForm) then
    TTPClineBudgetForm.mmoDriver.Lines.Add(LogString);
{$IFDEF USELOG}
  // Only log if the level is high enough
  if (Level and $0F) > LogLevel then
    Exit;
  // The check on <LogStream> is necessary because the application might
  // call this prior to completing initialization
  if not Assigned(LogStream) then
    Exit;
  NewLog := FormatDateTime('YYYYMMDD"T"HHMMSS"  "', Now) + LogString + #13#10;
  LogStream.Write(NewLog[1], Length(NewLog));
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : <Section>    Section ta access in INI file
            <Paremeter>  Paremeter in INI file
            <Default>    Default value (eg. if not present in INI file)
  Returns : <Result>     String value for parameter

  Descript: Get parameter from INI file.
  Notes   : Either the local INI file is used or the global INI file.
            Comments (;) are stripped out (' and " indicates strings and are also
            removed from the result -> either ' or " is used as string
            indentifier, meaning the other one is handled as a normal character)
            If in the local ini no reference is found the application ini
            file is tried too.
 ------------------------------------------------------------------------------}
function GetParameter(Section: string; Parameter: string; Default: string): string;
var
  FromIni : string;
  Position: Integer;
  TempStr : string;
begin
{$IFDEF USELOG}
//  ToLog('GetParameter', $05);
{$ENDIF}
  // If we don't have a local INI file then use the applications INI file
  if not Assigned(IniFile) then
  begin
    if not Assigned(PDVBHardwareAPI) then
    begin
      Result := '';
      Exit;
    end;
{$IFDEF USELOG}
    ToLog('GetParameter [application]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
    Result := PDVBHardwareAPI^.GetParameter(@Section[1], @Parameter[1], @Default[1]);
    Exit;
  end;
  // Try to find it locally (no default so we get an empty string if nothing is found)
  FromIni := IniFile.ReadString(Section, Parameter, '');
  // If nothing found, try application
  if (FromIni = '') and
     Assigned(PDVBHardwareAPI) then
  begin
{$IFDEF USELOG}
    ToLog('GetParameter [application]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
    FromIni := PDVBHardwareAPI^.GetParameter(@Section[1], @Parameter[1], '');
  end
  else
{$IFDEF USELOG}
    ToLog('GetParameter [local]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
  // Nothing found at all then use default
  if FromIni = '' then
    FromIni := Default;
  // Strip out leading/trailing spaces
  FromIni := Trim(FromIni);
  // Check for starting string parameter
  Position := Pos('''', FromIni);
  if Position = 1 then
  begin
    TempStr  := Copy(FromIni, Position+1, 255);
    // Find ending of string
    Position := Pos('''', TempStr);
    if Position <> 0 then
    begin
      // End of string found, which will be the result
      Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
      ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
      Exit;
    end;
  end
  else
  begin
    Position := Pos('"', FromIni);
    if Position = 1 then
    begin
      TempStr := Copy(FromIni, Position+1, 255);
      // Find ending of string
      Position := Pos('"', TempStr);
      if Position <> 0 then
      begin
        // End of string found, which will be the result
        Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
        ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
        Exit;
      end;
    end;
  end;
  // We know we don't have a string type so handle the whole string as
  // normal text. We could have a comment in it so check this too
  Position := Pos(';', FromIni);
  if Position <> 0 then
  begin
    TempStr := Copy(FromIni, 1, Position-1);
    // Strip out leading/trailing spaces
    Result := Trim(TempStr);
  end
  else
    Result := FromIni;
{$IFDEF USELOG}
  ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
end;


{******************************************************************************}
{                               REMOTE CONTROL                                 }
{******************************************************************************}
{------------------------------------------------------------------------------}
{ The infrared interface uses a dedicated uP and is connected to the GPIO3     }
{ input. When an infrared command is detected by this uP it activates this     }
{ line after which the code can be read through the DEBI interface.            }
{ Here we use a RPS program which runs continuously. This RPS waits            }
{ for a GPIO3 activation and then reads the infrared data.                     }
{ No direct GPIO interrupts are involved because of the continuous             }
{ 'polling' by means of the RPS program.                                       }
{ Note that the infrared uP activates GPIO3 three times for the total          }
{ information of the infrared data.                                            }
{ Two of the data words (low byte) contain the key codes itself, but with the  }
{ high nibbles changed: eg. $C4 and $44                                        }
{                           $Cx <> $4x                                         }
{                           $Dx <> $5x                                         }
{                           $Ex <> $6x                                         }
{                           $Fx <> $7x                                         }
{ These two key codes are the last detected keys. Typically these are the same }
{ except when a key is pressed very shortly.                                   }
{ The key code starts at $40 (command 0) and ends at $7F (command 63)          }
{ Extended commands (64..127) are not supported, meaning that command 64 will  }
{ be interpreted as command 0, and command 127 as 63.                          }
{ The third data byte contains the toggle code and the device (low nibble)     }
{ eg. $x3 indicates device '3'                                                 }
{ The toggle adds $20 to the device code.                                      }
{ In short:                                                                    }
{   Data < $40 == device/repeat                                                }
{   Data >=$40 == key code                                                     }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Remote control check
  Notes   :
 ------------------------------------------------------------------------------}
procedure ProcessRemoteControl;
var
  TransferBuffer : array[0..2] of Dword;
  InfraredData   : array[0..2] of Byte;
  Transfer       : TSaa7146aTransferBuffer;
  Key1           : Byte;
  Key2           : Byte;
  RepeatCode     : Boolean;
  Device         : Byte;
  Loop           : Integer;
  FirstKey       : Boolean;
  SecondKey      : Boolean;
  HasDevice      : Boolean;
  Valid          : Boolean;
  UseDevice      : Integer;
  UseCommand     : string;
  LUseCommand    : string;
  SpecialCommand : Byte;
  Error          : Integer;
  Parameter      : string;
begin
  // Setup transfer data and transfer it
  Transfer.Identifier := RcRPS0Data.Identifier;
  Transfer.SourceIndex  := 0;
  Transfer.TargetIndex  := 0;
  Transfer.TransferAddress := @TransferBuffer[0];
  Transfer.TransferLength := Sizeof(TransferBuffer);      // Transfer data
  Saa7146aReadFromDma(CardHandle, Transfer);
  // We have received three data items from the remote. Now seperate them.
  RepeatCode := False;
  Key1 := 0;
  Key2 := 0;
  Device := 0;
  FirstKey  := False;
  SecondKey := False;
  HasDevice := False;
  Valid     := True;
  InfraredData[0] := Swap(TransferBuffer[0]) and $FF;
  InfraredData[1] := Swap(TransferBuffer[1]) and $FF;
  InfraredData[2] := Swap(TransferBuffer[2]) and $FF;
  // Scan the data
  for Loop := 0 to 2 do
  begin
    if InfraredData[Loop] < $40 then
    begin
      if HasDevice then                                    // Check for more than 1 entry == invalid
        Valid := False;
      HasDevice := True;
      RepeatCode := (InfraredData[Loop] and $20) <> 0;
      if RepeatCode then
        Device := InfraredData[Loop] - $20
      else
        Device := InfraredData[Loop];
    end
    else
      if InfraredData[Loop] < $80 then
      begin
        if FirstKey then                                   // Check for more than 1 entry == invalid
          Valid := False;
        FirstKey := True;
        Key1 := InfraredData[Loop] - $40;
        Key1 := Key1 and $7F;
      end
      else
        begin
          if SecondKey then                                // Check for more than 1 entry == invalid
            Valid := False;
          SecondKey := True;
          Key2 := InfraredData[Loop] - $40;
          Key2 := Key2 and $7F;
        end;
  end;
  // We identify a valid key when both keycodes are the same
  if (Key1 = Key2) and
      Valid then
  begin
    // If the repeat code has changed or the keycode has changed then a key is pressed
    if (RcRepeat <> RepeatCode) or
       (RcDevice <> Device) then
    begin
      // New key (or repeat)
      RcKey1 := Key1;
      RcKey2 := Key2;
      RcDevice := Device;
      RcRepeat := RepeatCode;
      if Assigned(TTPClineBudgetForm) then
      begin
        TTPClineBudgetForm.txtRemoteControl.Caption := format('%2.2d %2.2d', [RcDevice, RcKey1]);
        if RcRepeat then
          TTPClineBudgetForm.txtRemoteControl.Caption := TTPClineBudgetForm.txtRemoteControl.Caption + '   /'
        else
          TTPClineBudgetForm.txtRemoteControl.Caption := TTPClineBudgetForm.txtRemoteControl.Caption + '   \';
      end;
      // Find command to send from INI file
      // First use the extended (alternative) method
      UseCommand := format('Key%2.2d%2.2d', [RcDevice, RcKey1]);
      UseCommand := GetParameter('Remote', UseCommand, '');
      // Nothing found yet, then use the original method
      if UseCommand = '' then
      begin
        // Get device to check from INI file
        Parameter := GetParameter('Remote', 'Device', '3');
        Val(Parameter, UseDevice, Error);
        if Error <> 0 then
          UseDevice := 3;
        if UseDevice = RcDevice then
        begin
          UseCommand := format('Key%d', [RcKey1]);
          UseCommand := GetParameter('Remote', UseCommand, '');
        end;
      end;
      // If found something
      if UseCommand <> '' then
      begin
        LUseCommand := LowerCase(UseCommand);
        // No special command
        SpecialCommand := 0;
        // Check for special commands
        if LUseCommand = 'displayon' then
          SpecialCommand := 1
        else
          if LUseCommand = 'displayoff' then
            SpecialCommand := 2
          else
            if LUseCommand = 'displaytoggle' then
              SpecialCommand := 3;
{$IFDEF USELOG}
        ToLog('ProcessRemoteControl: [' + UseCommand + ']', $04);
{$ENDIF}
        if Assigned(TTPClineBudgetForm) and (UseCommand <> '') then
          TTPClineBudgetForm.txtRemoteControl.Caption := TTPClineBudgetForm.txtRemoteControl.Caption + ' "' + UseCommand + '"';
        case SpecialCommand of
          0: if UseCommand <> '' then
               PDVBHardwareAPI^.ProcessKeyByName(@UseCommand[1]);
          1: DisplayOff := False;
          2: DisplayOff := True;
          3: DisplayOff := not(DisplayOff);
        end;
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Activate>  True if to activate infared
  Returns : <Result>    True if success

  Descript: (De)activate infrared
  Notes   :
 ------------------------------------------------------------------------------}
function RemoteControlActivate(Activate: Boolean): Boolean;
var
  Status        : TSaa7146aDmaBuffer;
  TransferBuffer: array[0..50] of Dword;
  Index         : Integer;
  Transfer      : TSaa7146aTransferBuffer;
  Address       : Dword;
  LAddress      : Word;
begin
{$IFDEF USELOG}
  if Activate then
    ToLog('Activate remote control', $03)
  else
    ToLog('Deactivate remote control', $03);
{$ENDIF}
  Result := False;
  // If the remote control thread is active: close it
  if Assigned(RcThread) and
    (not RcThread.HasStopped) then                         // If already running
  begin
    RcThread.ProgrammedStop := True;
    RcThread.Terminate;                                    // Mark it for termination
    if RcThread.Suspended then
      RcThread.Resume;
    RcThread.WaitFor;
    FreeAndNil(RcThread);
  end;
  if Activate then
  begin
    LAddress := CInfraredAddress;

    // Allocate some DMA buffers which will hold the RPS program
    if not (Saa7146aAllocateDma(CardHandle, DMASIZE * 1, Status)) then
      Exit;
    // Save info of allocated memory
    RcRPS0.Identifier      := Status.Identifier;
    RcRPS0.VirtualAddress  := Status.VirtualAddress;
    RcRPS0.PhysicalAddress := Status.PhysicalAddress;
    RcRPS0.Size            := Status.Size;
    // Allocate one DMA buffer for returned data
    if not (Saa7146aAllocateDma(CardHandle, DMASIZE * 1, Status)) then
      Exit;
    // Save info of allocated memory
    RcRPS0Data.Identifier      := Status.Identifier;
    RcRPS0Data.VirtualAddress  := Status.VirtualAddress;
    RcRPS0Data.PhysicalAddress := Status.PhysicalAddress;
    RcRPS0Data.Size            := Status.Size;

    // Now setup the RPS program. We first construct it in local memory and
    // then transport it to the SAA7146A
    Index    := 0;
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 0;                                                                // Target index
    TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 4;                                                                // Target index
    TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 8;                                                                // Target index
    TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsJump;                                   // Restart program
    Inc(Index);
    TransferBuffer[Index] := DWord(RcRPS0.PhysicalAddress.LowPart);              // Address of start program
    Inc(Index);

    // Setup transfer data and transfer it
    Transfer.Identifier   := RcRPS0.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Index * Sizeof(DWord);
    Saa7146aWriteToDma(CardHandle, Transfer);              // Transfer data
    // Now make sure the RPS program is started
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsAddr0,
      DWord(DWord(RcRPS0.PhysicalAddress.LowPart)));       // Set physical start of RPS program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsPage0,
      0);                                                  // RPS program performs no explicit mem writes
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsTov0 ,
      0);                                                  // Disable RPS timeouts
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps0On);                                 // Start RPS0 program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aDebiConfig,
      CSaa7146aDebiCfg16);                                 // Setup DEBI transfer type (MOTOROLA/INTEL does not care)
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aDebiPage,
      CSaa7146aDebiPageDisable);                           // No page check
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1DebiEnable);                             // Enable DEBI transfers
    RcThread := TRcThread.Create(False);                   // Activate thread handling received data
    Result := True;
  end
  else
  begin
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps0Off);                                // Stop RPS0 program
    Result := True;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles the remote control.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TRcThread.Execute;
begin
{$IFDEF USELOG}
  ToLog('TRcThread.Execute', $03);
{$ENDIF}
  HasStopped     := False;
  ProgrammedStop := False;
  try
    repeat
      ProcessRemoteControl;
      Sleep(100);
    until Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
    begin
      ToLog('Unexpected termination of remote control handler.', $81);
      if not Assigned(TTPClineBudgetForm) then
        ShowMessage('Unexpected termination of remote control handler.');
    end
    else
      ToLog('Normal termination of remote control handler.', $81);
  end;
end;


{******************************************************************************}
{                            REMOTE CONTROL END                                }
{******************************************************************************}



{******************************************************************************}
{                                     DISEQC                                   }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Set horizontal polarity
  Notes   :
------------------------------------------------------------------------------}
function DiseqcSetHorizontalPolarity: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcSetHorizontalPolarity', $05);
{$ENDIF}
  Result := Saa7146aHorizontalPolarityLNB(CardHandle, (LNBPolarity = 0));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Set vertical polarity
  Notes   :
------------------------------------------------------------------------------}
function DiseqcSetVerticalPolarity: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcSetVerticalPolarity', $05);
{$ENDIF}
  Result := Saa7146aVerticalPolarityLNB(CardHandle, (LNBPolarity = 0));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Enable LNB
  Notes   :
------------------------------------------------------------------------------}
function DiseqcEnableLnb: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcEnableLnb', $05);
{$ENDIF}
  Result := Saa7146aEnableLNB(CardHandle, (LNBOnOff = 1));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Disable LNB
  Notes   :
------------------------------------------------------------------------------}
function DiseqcDisableLnb: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcDisableLnb', $05);
{$ENDIF}
  Result := Saa7146aDisableLNB(CardHandle, (LNBOnOff = 1));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Simple burst only, selects LNB A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcBurstLnbA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcBurstLnbA', $05);
{$ENDIF}
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCBurstA, 0, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Simple burst only, selects LNB B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcBurstLnbB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcBurstLnbB', $05);
{$ENDIF}
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCBurstB, 0, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A, option A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAOptionA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAOptionA', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F0;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A, option B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAOptionB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAOptionB', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F8;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B, option A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBOptionA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBOptionA', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F4;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B, option B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBOptionB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBOptionB', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $FC;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A including simple burst mechanism
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAPlusBurst: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAPlusBurst', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F0;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand or CDiSEqCBurstA, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B including simple burst mechanism
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBPlusBurst: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBPlusBurst', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F4;                                    // Switch ID in low nibble
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand or CDiSEqCBurstB, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner reset command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerReset: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerReset', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $00;                                    // Reset
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner stop movement command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStopMovement: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStopMovement', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $60;                                    // Stop positioner movement
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner disable limits command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerDisableLimits: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerDisableLimits', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $63;                                    // Disable limits
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner set east limit command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerSetEastLimit: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerSetEastLimit', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $66;                                    // Set East limit
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner set west limit command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerSetWestLimit: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerSetWestLimit', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $67;                                    // Set West limit
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner move east command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGoEast: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGoEast', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner step east command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStepEast: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStepEast', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner move west command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGoWest: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGoWest', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner step west command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStepWest: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStepWest', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : <Position> Satellite position to store
  Returns : <Result>   False if error

  Descript: Positioner store current position command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStorePosition(Position: Byte): Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStorePosition', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6A;                                    // Store position xx
  DiSEqCData[3] := Position;                               // Position xx
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : <Position> Satellite position to store
  Returns : <Result>   False if error

  Descript: Positioner goto position command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGotoPosition(Position: Byte): Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGotoPosition', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6B;                                    // Goto position xx
  DiSEqCData[3] := Position;                               // Position xx
  Result := Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;
{******************************************************************************}
{                                  DISEQC END                                  }
{******************************************************************************}


{------------------------------------------------------------------------------
  Params  : <Source>  Satellite name
  Returns : <Result>  LNB number for setup to use
                      Empty if error occured

  Descript: Set LNB
  Notes   :
------------------------------------------------------------------------------}
function SetLNB(Source: PChar): string;
var
  IdSearch      : AnsiString;
  CheckSatellite: Byte;
  SatelliteFound: Boolean;

  function CmpSrc(s1, s2: string): Boolean;
  begin
    Result := s1 = (Copy(s2, 1, Length(s1)-1)+'S');
  end;
begin
{$IFDEF USELOG}
  ToLog('SetLNB', $05);
{$ENDIF}  
  Result := '';
  if Lowercase(DiSEqCType) = 'none' then
  begin
    Result := '1';
  end
  else
    if LowerCase(DiSEqCType) = 'mini-diseqc' then
    begin
      if CmpSrc(Source, GetParameter('DiSEqC', 'LNB1Source', '')) then
      begin
        if not DiseqcBurstLnbA then
          Exit;
        Result := '1';
      end
      else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB2Source', '')) then
      begin
        if not DiseqcBurstLnbB then
          Exit;
        Result := '2';
      end
    end
    else
      if DiSEqCType = '1.0' then
      begin
        if CmpSrc(Source, GetParameter('DiSEqC', 'LNB1Source', '')) then
        begin
          if not DiseqcLnbAOptionA then
            Exit;
          Result := '1';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB2Source', '')) then
        begin
          if not DiseqcLnbBOptionA then
            Exit;
          Result := '2';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB3Source', '')) then
        begin
          if not DiseqcLnbAOptionB then
            Exit;
          Result := '3';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB4Source', '')) then
        begin
          if not DiseqcLnbBOptionB then
            Exit;
          Result := '4';
        end;
      end
      else
        // If positioner support
        if DiSEqCType = '1.2' then
        begin
          // Find number for satellite
          // We search for 'LNB0Source' .. 'LNB99Source'
          CheckSatellite := 0;
          repeat
            IdSearch := format('LNB%dSource', [CheckSatellite]);
            SatelliteFound := CmpSrc(Source, GetParameter('DiSEqC', IdSearch, ''));
            if not(SatelliteFound) then
              Inc(CheckSatellite);
          until SatelliteFound or (CheckSatellite > 100);
          if SatelliteFound then
          begin
            // We found the satellite, now control the positioner
            if not DiseqcPositionerGotoPosition(CheckSatellite) then
              Exit;
            Result := '1';                              // Use LNB1 for settings
          end;
        end;
end;


{------------------------------------------------------------------------------
  Params  : <LNB>           LNB identification (for high/low band frequency)
            <Polarization>  Polarization ('H' or 'V')
            <Frequency>     Frequency in kHz
            <Symbolrate>    Symbolrate in kHz
  Returns : <Result>        True if no error

  Descript: Set frequency, polarization, symbolrate
  Notes   :
------------------------------------------------------------------------------}
function SetFrequency(LNBNumber: string; Polarization: Char; Frequency, SymbolRate: Dword): Boolean;
var
  LowBand   : Dword;
  HighBand  : Dword;
  Parameter : string;
  Error     : Integer;
begin
{$IFDEF USELOG}
  ToLog('SetFrequency', $05);
{$ENDIF}  
  Result := False;
  // If we have a valid LNB then set frequency and such
  if LNBNumber <> '' then
  begin
    if GetParameter('LNB', 'LNB'+LNBNumber+'Type', '') = 'Circular' then
      Frequency := Frequency - 150000;
    Parameter := GetParameter('LNB', 'LNB'+LNBNumber+'LOF1', '9750000' );
    Val(Parameter, LowBand, Error);
    if Error <> 0 then
      LowBand := 9750000;
    Parameter := GetParameter('LNB', 'LNB'+LNBNumber+'LOF2', '10600000');
    Val(Parameter, HighBand, Error);
    if Error <> 0 then
      HighBand := 10600000;
    // Check if for some reason we are a factor 1000 too small
    if LowBand < 100000 then
      LowBand := LowBand * 1000;
    if HighBand < 100000 then
      HighBand := HighBand * 1000;
    // Check if we are for some reason a factor 1000 too high
    if SymbolRate > 1000000 then
      SymbolRate := SymbolRate div 1000;
    if Assigned(TTPClineBudgetForm) then
    begin
      // Reflect in form
      TTPClineBudgetForm.mskFrequency.Text  := format('%8.8d', [Frequency]);
      TTPClineBudgetForm.mskSymbolRate.Text := format('%5.5d', [SymbolRate]);
      if Polarization = 'V' then
        TTPClineBudgetForm.rgPolarity.ItemIndex := 1
      else
        TTPClineBudgetForm.rgPolarity.ItemIndex := 0;
    end;
    // Set polarity/frequency/symbolrate
    if Polarization = 'V' then
    begin
      if not Saa7146aVerticalPolarityLNB(CardHandle, (LNBPolarity = 0)) then
        Exit;
    end
    else
      if not Saa7146aHorizontalPolarityLNB(CardHandle, (LNBPolarity = 0)) then
        Exit;
    if not Saa7146aFrequencyLNB(CardHandle, Frequency, LowBand, HighBand, TunerType, SynthesizerAddress) then
      Exit;
    if not Saa7146aQpskSetSymbolRate(CardHandle, SymbolRate) then
      Exit;
    Result := True;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Source>        Satellite name
            <Polarization>  Polarization ('H' or 'V')
            <Frequency>     Frequency in kHz
            <Symbolrate>    Symbolrate in kHz
  Returns : <Result>        0 = no error

  Descript: Set frequency, polarization, symbolrate and satellite
  Notes   :
------------------------------------------------------------------------------}
function SetDiSEqC(Source, Polarisation: PChar; Frequency, SymbolRate: Dword): Dword; stdcall;
var
  LNBNumber: string;
begin
{$IFDEF USELOG}
  ToLog(format('Set DiSEqC: %s, %s, %d, %d', [Source, Polarisation, Frequency, SymbolRate]), $03);
{$ENDIF}  
  Result := 0;
  if IsSlave then
    Exit;

  // Save copy if we want to use it locally
  SetDiSEqCSource       := Source;
  SetDiSEqCPolarization := Polarisation^;
  SetDiSEqCFrequency    := Frequency;
  SetDiSEqCSymbolRate   := Symbolrate;

  LNBNumber := SetLNB(Source);
  SetDiSEqCLNB := LNBNumber;
  if SetFrequency(SetDiSEqCLNB, SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolRate) then
    Result := 0
  else
    result := 1;  
end;


{******************************************************************************}
{                                     PACKETS                                  }
{******************************************************************************}
{$IFDEF USESOFTCSA}
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: Sets keys for SoftCSA decoding
    Notes   : Command is layout as follows
               0 1 2 3 4 5 6 7 8 9 10 11 12 13
                       x                          0 = even keys in [6..13]
                       x                          1 = odd  keys in [6..13]
                           1 2 3 4 5  6  7  8     Typically the keys need endian swapping!
   ------------------------------------------------------------------------------}
  procedure SetCSAKeys(Command: array of Byte);
  var
    i: Integer;
    j: Integer;
    Change: Boolean;
{$IFDEF USELOG}
    KeyString: string;
{$ENDIF}
  begin
{$IFDEF USELOG}
    ToLog('SetCSAKeys', $05);
{$ENDIF}    
    if Command[4] = 1 then
      OddKeyFound := True
    else
      EvenKeyFound := True;
    for i := 0 to 3 do
    begin
      // Do endian swapping
      Key[(Command[4])*8+i*2+0] := Command[6+i*2+1];
      Key[(Command[4])*8+i*2+1] := Command[6+i*2+0];
      // If we are using the internal decrytor then prepare keys
      if InternalCSA then
      begin
        if Command[4] = 0 then
          for j := 0 to 3 do
          begin
            KeysEven[j*2+0] := Command[6+j*2+1];
            KeysEven[j*2+1] := Command[6+j*2+0];
          end
        else
          for j := 0 to 3 do
          begin
            KeysOdd[j*2+0] := Command[6+j*2+1];
            KeysOdd[j*2+1] := Command[6+j*2+0];
          end;
      end;
    end;
    if OddKeyFound and EvenKeyFound then
    begin
      KeyFound := True;
      // Internal handling checkcs for a change here
      if InternalCSA then
      begin
        Change := False;
        for j := 0 to 7 do
        begin
          if KeysOdd[j] <> KeysPreviousOdd[j] then
            Change := True;
          if KeysEven[j] <> KeysPreviousEven[j] then
            Change := True;
        end;
        if Change then
        begin
{$IFDEF USELOG}
          if LogLevel >= 3 then
          begin
            KeyString := 'Set CSA keys:  Even= ';
            for j := 0 to 7 do
              KeyString := KeyString + format('$%2.2x ', [KeysEven[j]]);
            KeyString := KeyString + '  Odd= ';
            for j := 0 to 7 do
              KeyString := KeyString + format('$%2.2x ', [KeysOdd[j]]);
            ToLog(KeyString, $03);
          end;
{$ENDIF}
          CSASetKeys(KeysOdd, KeysEven);
          for j := 0 to 7 do
          begin
            KeysPreviousOdd[j]  := KeysOdd[j];
            KeysPreviousEven[j] := KeysEven[j];
          end;
        end;
      end
      else
      begin
{$IFDEF USELOG}
        if LogLevel >= 3 then
        begin
          KeyString := 'Set CSA keys:  Even= ';
          for j := 0 to 7 do
            KeyString := KeyString + format('$%2.2x ', [Key[j]]);
          KeyString := KeyString + '  Odd= ';
          for j := 0 to 7 do
            KeyString := KeyString + format('$%2.2x ', [Key[j+8]]);
          ToLog(KeyString, $03);
        end;
{$ENDIF}
        SetKeyProc(@Key[0], @KeyStruct[0]);
      end;
    end;
  end;
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Decode transport stream. Uses <StreamDataBuffer> which holds
            multiple stream packets.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ProcessTS;
var
  StreamPacket: Word;
  Pid         : Word;
  DoRecord    : Boolean;
begin
  FilterLock.Acquire;
  try
    for StreamPacket := 0 to CDvbPacketVSync-1 do
    begin
      // Get Pid
      Pid := ((StreamDataBuffer[StreamPacket][1] and $1F) shl 8) or StreamDataBuffer[StreamPacket][2];
      // If cross reference exists then its active
      if FilterCrossReference[Pid] <> $FFFF then
      begin
{$IFDEF USESOFTCSA}
        // SoftCSA, check for scrambled data
        if ((StreamDataBuffer[StreamPacket][3] and $80) <> 0) and
             KeyFound then
          if InternalCSA then
            CSADecrypt(@StreamDataBuffer[StreamPacket])
          else
            CSADecryptProc(@KeyStruct[0], @StreamDataBuffer[StreamPacket]);
{$ENDIF}
        if Assigned(@Filters[FilterCrossReference[Pid]].CallBackFunction) then
        begin
          Filters[FilterCrossReference[Pid]].CallBackFunction(FilterCrossReference[Pid], 184, @StreamDataBuffer[StreamPacket][4]);
          if FilterCount = $FFFF then
            FilterCount := 0
          else
            Inc(FilterCount);
        end;
      end;

      // Debug
      if (Pid <> 0) and (Pid = PidVideo) then
      begin
        if VideoPacketCount = $FFFF then
          VideoPacketCount := 0
        else
          Inc(VideoPacketCount);
      end;

      // If recording is started, synchronize on start indicator
      if (RecordingState = [rsStartRecording]) and
         ((StreamDataBuffer[StreamPacket][1] and $40) <> 0) then
        RecordingState := [rsRecording];
      // If recording is being stopped, synchronize on start indicator
      if (RecordingState = [rsStopRecording]) and
         ((StreamDataBuffer[StreamPacket][1] and $40) <> 0) then
      begin
        RecordingState := [rsIdle];
        FreeAndNil(FileStream);
      end;
      // Write packet to file if requested
      // Only if a recording has been started or if we are stopping
      // Strip out PID=0 (if only audio PID exists for radio channels)
      if ((RecordingState = [rsRecording]) or
          (RecordingState = [rsStopRecording])) and
          (Pid <> 0) then
      begin
        // We no longer check the filters for recording but use the local settings
        // so we can record these independend of the filter settings
        // Assume no recording
        DoRecord := False;
        if (Pid = PidVideo) and VideoInRecording then
          DoRecord := True
        else
          if (Pid = PidAudio) and AudioInRecording then
            DoRecord := True
          else
            if (Pid = PidPmt) and PmtInRecording then
              DoRecord := True
            else
              if (Pid = PidPcr) and PcrInRecording then
                DoRecord := True
              else
                if (Pid = PidEcm) and EcmInRecording then
                  DoRecord := True
                else
                  if (Pid = PidSid) and SidInRecording then
                    DoRecord := True
                  else
                    if (Pid = PidAc3) and Ac3InRecording then
                      DoRecord := True
                    else
                      if (Pid = PidTeletext) and TeletextInRecording then
                        DoRecord := True;
        if DoRecord then
          FileStream.Write(StreamDataBuffer[StreamPacket], CDvbPacketSize);
      end;
    end;
  finally
    FilterLock.Release;
  end;
  // Pass all packets to DirectShow
  // Note: Only sending the typical Pids which have been setup for filtering will not
  //       generate an image, except when all Pid's in in the <TProgramm82> record
  //       have been activated.
  if not DisplayOff then
    if DisplayDuringRecording or
       (RecordingState = [rsIdle]) then
      PDVBHardwareAPI^.SendDataToDirectShow(@StreamDataBuffer, CDvbPacketBufferSize);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrupt at both edges of the field indicator,
            meaning that we get notified when a switch is made from
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Data       : Dword;
  LastBuffer : Dword;
  Buffer     : TSaa7146aTransferBuffer;
begin
  // Debug
{$IFDEF USELOG}
  ToLog('TDataThread.Execute', $05);
{$ENDIF}
  PacketBufferCount :=0;
  VideoPacketCount  :=0;
  FilterCount       :=0;

  HasStopped     := False;
  ProgrammedStop := False;
  // The driver must be running and must be in use
  if Saa7146aGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't then only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls eg. 'generate manual notification'
  // which might be needed in a deadlock situation.
  ThreadHandle := Saa7146aCreateFile(CardInUse);
  Buffer.Identifier      := BufferId;
  Buffer.TransferAddress := @StreamDataBuffer;
  Buffer.TargetIndex     := 0;
  Buffer.TransferLength  := Sizeof(StreamDataBuffer);
  // Wait for application to have setup stuff
  // This is necessary because if we send data with <SendDataToDirectShow>
  // too early (without it being set up) then this generates and error and
  // this thread will terminate.
  ChannelSet := False;
  repeat
   Sleep(100);
  until ChannelSet or Terminated;
  LastBuffer := 0;
  try
    if not Terminated then
    repeat
      if ChannelSet then
      begin
{$IFDEF USELOG}
        ToLog('TDataThread [channel set]', $05);
{$ENDIF}
        Overtaken := 0;
        ChannelSet := False;
      end;
      if PacketBufferCount = $FFFF then
        PacketBufferCount := 0
      else
        Inc(PacketBufferCount);
      // First check if there are already buffers filled
      Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
      // If no new data, wait (or if first time)
      if Data = LastBuffer then
      begin
        if IsSlave then
        repeat
          Sleep(1);
          Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
        until Terminated or (Data <> LastBuffer)
        else
          Saa7146aWaitForNotification(ThreadHandle);
      end
      else
      begin
        if Overtaken = $FFFF then
          Overtaken := 0
        else
          Inc(Overtaken);
      end;    
      Buffer.SourceIndex := CDvbPacketBufferSize * LastBuffer;
      // Next buffer to wait for
      Inc(LastBuffer);
      // Circular buffer
      if LastBuffer = (PacketBuffers*2) then
        LastBuffer := 0;
      // Get data in local buffer
      Saa7146aReadFromDma(ThreadHandle, Buffer);
      // Process packet data
      ProcessTS;
    until Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
    begin
      ToLog('Unexpected termination of packets handler.', $81);
      if not Assigned(TTPClineBudgetForm) then
        ShowMessage('Unexpected termination of packets handler.');
    end
    else
      ToLog('Normal termination of packets handler.', $81);
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Start DVB card
  Notes   :
 ------------------------------------------------------------------------------}
function StartDVB: Boolean;
var
  Priority  : string;
  BufferTime: Word;
  BufferId  : Dword;
  Parameter : string;
  Error     : Integer;
begin
{$IFDEF USELOG}
  ToLog('StartDVB', $03);
{$ENDIF}
  Result        := False;
  PacketThread  := nil;
  CardHandle    := INVALID_HANDLE_VALUE;
  CardInUse     := -1;
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    CardHandle := Saa7146aCreateFile(CardInUse);
    CardInUse  := Saa7146aGetCardOfHandle(CardHandle);
  end
  else
    Exit;


  // Get tuner type
  Parameter := LowerCase(GetParameter('Hardware', 'Tuner', 'BSRU6'));
  if LowerCase(Parameter) = 'su1278' then
    TunerType := CTunerSU1278
  else
    TunerType := CTunerBSRU6;

  // Get synthesizer address
  Parameter := LowerCase(GetParameter('Hardware', 'SynthesizerAddress', '1'));
  Val(Parameter, SynthesizerAddress, Error);
  if Error <> 0 then
    SynthesizerAddress := 1;
  if SynthesizerAddress > 3 then
    SynthesizerAddress := 255;

  // Get tuner reset line
  Parameter := LowerCase(GetParameter('Hardware', 'TunerReset', '2'));
  Val(Parameter, TunerReset, Error);
  if Error <> 0 then
    TunerReset := 2;
  if TunerReset > 3 then
    TunerReset := 255;

  // Get LNB polarity line
  Parameter := LowerCase(GetParameter('Hardware', 'LNBPolarity', '0'));
  Val(Parameter, LNBPolarity, Error);
  if Error <> 0 then
    LNBPolarity := 0;
  if LNBPolarity > 1 then
    LNBPolarity := 255;

  // Get LNB on/off line
  Parameter := LowerCase(GetParameter('Hardware', 'LNBOnOff', '1'));
  Val(Parameter, LNBOnOff, Error);
  if Error <> 0 then
    LNBOnOff := 0;
  if LNBOnOff > 1 then
    LNBOnOff := 255;
  if LNBOnOff = LNBPolarity then
    LNBOnOff := 255;

  // Get card type (overrules other settings)
  DvbCardPreset := False;
  DvbCard := GetParameter('Hardware', 'Card', '');
  if UpperCase(DvbCard) = 'TTPCLINEBUDGET' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerBSRU6;
    SynthesizerAddress := 1;
    TunerReset         := 2;
    LNBPolarity        := 0;
    LNBOnOff           := 1;
  end;
  if UpperCase(DvbCard) = 'NOVA' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerBSRU6;
    SynthesizerAddress := 1;
    TunerReset         := 2;
    LNBPolarity        := 0;
    LNBOnOff           := 1;
  end;
  if UpperCase(DvbCard) = 'TERRATEC' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerSU1278;
    SynthesizerAddress := 1;
    TunerReset         := 255;
    LNBPolarity        := 1;
    LNBOnOff           := 255;
  end;
  if UpperCase(DvbCard) = 'KNC' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerSU1278;
    SynthesizerAddress := 1;
    TunerReset         := 255;
    LNBPolarity        := 1;
    LNBOnOff           := 255;
  end;
  if UpperCase(DvbCard) = 'TYPHOON' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerSU1278;
    SynthesizerAddress := 1;
    TunerReset         := 255;
    LNBPolarity        := 1;
    LNBOnOff           := 255;
  end;
  if UpperCase(DvbCard) = 'ANUBIS' then
  begin
    DvbCardPreset      := True;
    TunerType          := CTunerSU1278;
    SynthesizerAddress := 1;
    TunerReset         := 255;
    LNBPolarity        := 1;
    LNBOnOff           := 255;
  end;




  // Get number of buffers to use: we readout the TIME in ms here
  Parameter := GetParameter('Interface', 'TSBufferSize', '400');
  Val(Parameter, BufferTime, Error);
  if Error <> 0 then
    BufferTime := 400;
  // Setup data transfer from tuner to memory
  if not DvbSetupBuffering(CardHandle, True, TunerType, TunerReset, IsSlave, BufferTime, BufferId) then
    Exit;

  // Start the thread which receives notifications
  PacketThread := TDataThread.Create(True);
  PacketThread.BufferId      := BufferId;
  PacketThread.PacketBuffers := BufferTime;
  // Check the priority to use
  Priority  := GetParameter('Hardware', 'ThreadPriority', 'Default');
  if LowerCase(Priority) = 'lowest' then
    PacketThread.Priority := tpLowest;
  if LowerCase(Priority) = 'low' then
    PacketThread.Priority := tpLower;
  if LowerCase(Priority) = 'normal' then
    PacketThread.Priority := tpNormal;
  if LowerCase(Priority) = 'default' then
    PacketThread.Priority := tpHigher;
  if LowerCase(Priority) = 'high' then
    PacketThread.Priority := tpHigher;
  if LowerCase(Priority) = 'highest' then
    PacketThread.Priority := tpHighest;
  if LowerCase(Priority) = 'realtime' then
    PacketThread.Priority := tpTimeCritical;
  PacketThread.Resume;

  // Activate remote control
  // MUST be after other setup because that might have released all DMA memory
  if not IsSlave then
    Result := RemoteControlActivate(True)
  else
    Result := True;

  if IsSlave then
    ToLog('Set as slave.', $81)
  else
    ToLog('Set as master.', $81);

  // Inform about settings
  if DvbCardPreset then
    ToLog('Using presets for ''' + DvbCard + ''' card.', $81);
  case TunerType of
    CTunerBSRU6:  ToLog('Tuner: BSRU6', $81);
    CTunerSU1278: ToLog('Tuner: SU1278', $81);
    else ToLog('Tuner: Unknown', $81);
  end;
  if TunerReset = 255 then
    ToLog('Reset: -', $81)
  else
    ToLog(format('Reset: GPIO%d', [TunerReset]), $81);
  if LnbPolarity = 255 then
    ToLog('Polarity: -', $81)
  else
    ToLog(format('Polarity: OP%d', [LnbPolarity]), $81);
  if LnbOnOff = 255 then
    ToLog('LNB on/off: -', $81)
  else
    ToLog(format('LNB on/off: OP%d', [LnbOnOff]), $81);
  ToLog(format('Synthesizer address: %d', [SynthesizerAddress]), $81);

  if not IsSlave then
    ToLog(format('Using %d buffers (appr. %d ms).', [PacketThread.PacketBuffers, PacketThread.PacketBuffers * 20]), $81);
{$IFDEF USESOFTCSA}
  if InternalCSA then
    ToLog('Using internal CSA mechanism.', $81)
  else
    ToLog('Using external CSA mechanism.', $81);
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Stop DVB card
  Notes   :
 ------------------------------------------------------------------------------}
procedure StopDVB;
var
  Channel: Dword;
begin
{$IFDEF USELOG}
  ToLog('StopDVB', $03);
{$ENDIF}
  // Remove form
  FreeAndNil(TTPClineBudgetForm);
  // Stop remote control program
  if not IsSlave then
    if CardHandle <> INVALID_HANDLE_VALUE then
      RemoteControlActivate(False);
  // Stop the packet thread if it running
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    // Note the use of <CardHandle> and NOT <ThreadHandle>. This is because
    // <ThreadHandle> might be 'busy' with a <Saa7146aWaitForNotification>!
    Saa7146aGenerateManualNotification(CardHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;
  // Software reset. We migth be capturing data, so stop ALL
  if not IsSlave then
  begin
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1, CSaa7146aMc1SoftReset);
    // Release all allocated memory by the driver
    Channel := 0;
    while Saa7146aReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    Saa7146aDisableLNB(CardHandle, (LNBOnOff = 1));
    Saa7146aCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;
end;
{******************************************************************************}
{                                PACKETS END                                   }
{******************************************************************************}


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Update cross reference table for filters.
  Notes   :
------------------------------------------------------------------------------}
procedure FilterUpdateCrossReference;
var
  FilterIndex: Integer;
begin
{$IFDEF USELOG}
  ToLog('FilterUpdateCrossReference', $05);
{$ENDIF}  
  // Clear cross references
  FiltersDefined := 0;
  FiltersDefinedPids := '';
  for FilterIndex := Low(FilterCrossReference) to High(FilterCrossReference) do
    FilterCrossReference[FilterIndex] := $FFFF;
  // Renew cross reference
  for FilterIndex := Low(Filters) to High(Filters) do
    if Filters[FilterIndex].Active then
    begin
      FilterCrossReference[Filters[FilterIndex].Pid] := FilterIndex;
      Inc(FiltersDefined);
      if FiltersDefinedPids = '' then
        FiltersDefinedPids := IntToStr(Filters[FilterIndex].Pid)
      else
        FiltersDefinedPids := FiltersDefinedPids + ' ' + IntToStr(Filters[FilterIndex].Pid);
    end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  0 = no error

  Descript: Allocate DVB card
  Notes   :
------------------------------------------------------------------------------}
function DvbInit: Dword; stdcall;
var
  FilterIndex: Integer;
  ShowForm   : string;
  Error      : Boolean;
  CheckString: string;
  Parameter  : string;
  ValError   : Integer;
begin
{$IFDEF USELOG}
  ToLog('DvbInit', $03);
{$ENDIF}  
{$IFDEF USESOFTCSA}
  InternalCSA      := True;
{$ENDIF}
  Overtaken        := 0;
  RecordingState   := [rsIdle];
  VideoInRecording := False;
  AudioInRecording := False;
  PmtInRecording   := False;
  PcrInRecording   := False;
  EcmInRecording   := False;
  SidInRecording   := False;
  Ac3InRecording   := False;
  TeletextInRecording := False;

  // Initialize filters
  for FilterIndex := Low(Filters) to High(Filters) do
    Filters[FilterIndex].Active := False;
  FilterUpdateCrossReference;
  // Get DiSEqC repeats and type
  Parameter := GetParameter('DiSEqC', 'RepeatCommand', '1');
  Val(Parameter, DiSEqCRepeats, ValError);
  if ValError <> 0 then
    DiSEqCRepeats := 1;
  DISEQCType     := GetParameter('DiSEqC','DiSEqCType','');

  // Get slave setting
  CheckString := LowerCase(GetParameter('Debug', 'TTPClineBudgetSlave', 'Yes'));
  if CheckString = 'yes' then
    IsSlave := True
  else
    IsSlave := False;

  // Get display setting
  CheckString := LowerCase(GetParameter('Interface', 'DisplayDuringRecording', 'Yes'));
  if CheckString = 'yes' then
    DisplayDuringRecording := True
  else
    DisplayDuringRecording := False;
  CheckString := LowerCase(GetParameter('Interface', 'DisplayOff', ''));
  if CheckString = 'yes' then
    DisplayOff := True
  else
    DisplayOff := False;

  // Get settings what is included in recordings
  CheckString := LowerCase(GetParameter('Recording', 'IncludeInTSRecording', 'Video, Audio, PMT, PCR'));
  if Pos('video', CheckString) <> 0 then
    VideoInRecording := True;
  if Pos('audio', CheckString) <> 0 then
    AudioInRecording := True;
  if Pos('pmt', CheckString) <> 0 then
    PmtInRecording := True;
  if Pos('pcr', CheckString) <> 0 then
    PcrInRecording := True;
  if Pos('ecm', CheckString) <> 0 then
    EcmInRecording := True;
  if Pos('sid', CheckString) <> 0 then
    SidInRecording := True;
  if Pos('ac3', CheckString) <> 0 then
    Ac3InRecording := True;
  if Pos('teletext', CheckString) <> 0 then
    TeletextInRecording := True;

  ShowForm  := GetParameter('Debug', 'TTPClineBudgetForm', 'No');
  if LowerCase(ShowForm) = 'yes' then
  begin
    TTPClineBudgetForm := TfrmMain.Create(nil);
    TTPClineBudgetForm.Caption := TTPClineBudgetForm.Caption + format(' - V%1.1x.%2.2x, build %x', [Version div $100, Version mod $100, Build]);
    TTPClineBudgetForm.pgPageControl.ActivePage := TTPClineBudgetForm.tsDebug;
    TTPClineBudgetForm.Show;
    TTPClineBudgetForm.Refresh;
  end;
{$IFDEF USESOFTCSA}
    // SoftCSA
    CheckString := LowerCase(GetParameter('Interface', 'CSA', ''));
    if CheckString = '' then
      InternalCSA := True
    else
    begin
      InternalCSA := False;
      DLLID       := LoadLibrary(PChar(CheckString));
      // If we failed using the library use the internal mechanism
      if DLLID <> 0 then
      begin
        SetKeyProc     := GetProcAddress(DLLID, 'set_cws');
        CSADecryptProc := GetProcAddress(DLLID, 'decrypt');
        if (@SetKeyProc = nil) or (@CSADecryptProc = nil) then
        begin
          ToLog('Could not use external CSA mechanism: expected procedures not present.', $81);
          FreeLibrary(DLLID);
          InternalCSA := True;
        end;
      end
      else
      begin
        ToLog('Could not load external CSA mechanism [' + CheckString + '].', $81);
        InternalCSA := True;
      end;
    end;
{$ENDIF}

  // Initialize card
  Error := not StartDVB;
  if Error then
  begin
    ToLog('Error initializing SAA7146A driver of TT PC-line budget card.', $81);
    ShowMessage('Error initializing SAA7146A driver of TT PC-line budget card.'#13#13+
                'Possible reason:'#13+
                ' - Device driver not installed or not running (SAA7146A.SYS)'#13+
                ' - Requested buffer size too large - decrease ''TSBufferSize'''#13);
  end;
  if Error then
    Result := 1                                  // What to return? Application does not check it anyhow .
  else
    Result := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  0 = no error

  Descript: Release DVB card
  Notes   :
------------------------------------------------------------------------------}
function DvbExit: Dword; stdcall;
begin
{$IFDEF USELOG}
  ToLog('DvbExit', $03);
{$ENDIF}  
  Result := 0;
  StopDVB;
  PDVBHardwareAPI^.StopDirectshow;
{$IFDEF USESOFTCSA}
  if not InternalCSA then
    FreeLibrary(DLLID);
{$ENDIF}
  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    // Disable LNB (power off)
    DiseqcDisableLnb;
    Saa7146aCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;
end;


{------------------------------------------------------------------------------
  Params  : <Locked>        Locked (1) or not locked (0)
            <Signal>        Signal level
            <Quality>       Signal quality
            <AGC>           Automatic gain setting
            <SNR_SQE>       Carrier noise ratio
  Returns : <Result>  Filter index

  Descript: Get signal status
  Notes   :
------------------------------------------------------------------------------}
function DvbSignalQuality(Locked: PByte; Signal: PByte; Quality: PByte; AGC: PByte; SNR_SQE: PByte): Dword; stdcall;
var
  LockState : Byte;
  Noise     : Double;
  StrengthDecibel   : Double;
  StrengthPercentage: Double;
begin
  Result := 0;
  if IsSlave then
  begin
    Locked^  := 50;
    Quality^ := 50;
    Signal^  := 50;
    SNR_SQE^ := 50;
    AGC^     := 50;
    Locked^  := 1;
    Exit;
  end;
  // Signal power
  Saa7146aQpskGetSignalPower(CardHandle, StrengthPercentage, StrengthDecibel);
  Signal^ := Trunc(StrengthPercentage);
  // Lock/sync
  if Saa7146aReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
  begin
    if (LockState and $98) = $98 then
      Locked^ := 1
    else
      Locked^ := 0;
  end;
  // Carrier noise (dB)
  Saa7146aQpskGetNoiseIndicator(CardHandle, Noise);
  // We have to give a percentage back ... (?)
  // Range in dB = -92.4 .. 19
  // We translate 0..19 dB to 0..100%
  // Note: This is NOT the correct calculation!! dB = log!!
  if Noise < 0 then
    Noise := 0;
  Noise := Noise * 5.26;
  Quality^ := Trunc(Noise);
  SNR_SQE^ := Trunc(Noise);
  AGC^     := 100 - Trunc(StrengthPercentage);
  // Mainly for debugging; set everything to zero if thread not running
  if not Assigned(PacketThread) or
    (PacketThread.HasStopped) then                         // If stopped running
  begin
    Locked^  := 0;
    Quality^ := 0;
    Signal^  := 0;
    SNR_SQE^ := 0;
    AGC^     := 0;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Pid>         Pid for filter to set
            <FilterProc>  Callback function for Pid
            <Name>        Name for filter
  Returns : <Result>      Filter index

  Descript: Set filter data
  Notes   :
------------------------------------------------------------------------------}
function DvbSetFilter(Pid: Word; FilterProc: TMDAPIFilterProc; Name: PChar): Dword; stdcall;
var
  FilterIndex: Integer;
  Valid      : Boolean;
begin
{$IFDEF USELOG}
  ToLog(format('Set filter PID: %d', [Pid]), $03);
{$ENDIF}  
  FilterLock.Acquire;
  try
    Result := 0;
    if Pid > High(FilterCrossReference) then
      Exit;
    Valid := False;
    // First try to find existing active filter
    if FilterCrossReference[Pid] <> $FFFF then
    begin
      FilterIndex := FilterCrossReference[Pid];
      Valid       := True;
    end
    else
    begin
      // No existing filter found for Pid, find first available one
      FilterIndex := Low(Filters);
      repeat
        if Filters[FilterIndex].Active then
          Inc(FilterIndex)
        else
          Valid := True;
      until Valid or (FilterIndex > High(Filters));
    end;
    if Valid then
    begin
      // Now set filter data and
      Filters[FilterIndex].Pid := Pid;
      Filters[FilterIndex].CallBackFunction := FilterProc;
      Filters[FilterIndex].Name := Name;
      Filters[FilterIndex].InRecording := False;
      if LowerCase(Name) = 'video' then
      begin
        PidVideo := Pid;
        if VideoInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'audio' then
      begin
        PidAudio := Pid;
        if AudioInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'pmt' then
      begin
        PidPmt := Pid;
        if PmtInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'pcr' then
      begin
        PidPcr := Pid;
        if PcrInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'ecm' then
      begin
        PidEcm := Pid;
        if EcmInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'sid' then
      begin
        PidSid := Pid;
        if SidInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'ac3' then
      begin
        PidAc3 := Pid;
        if Ac3InRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'teletext' then
      begin
        PidTeletext := Pid;
        if TeletextInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      Filters[FilterIndex].Active := True;
      FilterUpdateCrossReference;
      Result := FilterIndex;
    end;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FilterIndex> Filter index to clear
  Returns : <Result>      0 if no error

  Descript: Clear filter data
  Notes   :
------------------------------------------------------------------------------}
function DvbStopFilter(FilterIndex: Dword): Dword; stdcall;
begin
  FilterLock.Acquire;
  try
    Result := 0;
    if FilterIndex < Low(Filters) then
      Exit;
    if FilterIndex > High(Filters) then
      Exit;
{$IFDEF USELOG}
    ToLog(format('Stop filter PID: %d', [Filters[FilterIndex].Pid]), $03);
{$ENDIF}    
    Filters[FilterIndex].Active := False;
    FilterUpdateCrossReference;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Name>  Name to search for
  Returns : -

  Descript: Remove filters which have the indicated name
  Notes   :
------------------------------------------------------------------------------}
procedure FilterRemoveByName(Name: string);
var
  FilterIndex: Integer;
begin
{$IFDEF USELOG}
//  ToLog('FilterRemoveByName', $05);
{$ENDIF}
  FilterLock.Acquire;
  try
    for FilterIndex := Low(Filters) to High(Filters) do
    begin
      Name := LowerCase(Name);
(* LOCAL DEFINITIONS ARE NOT REMOVED BECAUSE OF RECORDING
   SOME PLUGINS SPECIFICLY DELETE SOME FILTERS, LIKE THE PMT
      if Name = 'video' then
        PidVideo := 0;
      if Name = 'audio' then
        PidAudio := 0;
      if Name = 'pmt' then
        PidPmt := 0;
      if Name = 'pcr' then
        PidPcr := 0;
      if Name = 'ecm' then
        PidEcm := 0;
      if Name = 'sid' then
        PidSid := 0;
      if Name = 'ac3' then
        PidAc3 := 0;
      if Name = 'teletext' then
        PidTeletext := 0;
*)
      if LowerCase(Filters[FilterIndex].Name) = Name then
      begin
        Filters[FilterIndex].Active := False;
        FilterUpdateCrossReference;
      end;
    end;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Prg>  Program settings
  Returns : -

  Descript: Set channel PIDs
  Notes   :
------------------------------------------------------------------------------}
function DvbSetChannel(Prg: PProgramm82): Dword; stdcall;
begin
{$IFDEF USELOG}
  if Loglevel = 1 then
    ToLog('New channel.', $01)
  else
    ToLog(format('New channel [%d/%d/%d/%d].', [Prg.Video_pid, Prg.Audio_pid, Prg.PMT_pid, Prg.PCR_pid]), $02);
{$ENDIF}    
{$IFDEF USECSA}
  OddKeyFound  := False;
  EvenKeyFound := False;
  KeyFound     := False;
{$ENDIF}
  Result := 0;
  // We have to replace any existing audio/video/.. Pids, so first remove
  // old ones
  PidVideo    := 0;
  PidAudio    := 0;
  PidPmt      := 0;
  PidPcr      := 0;
  PidEcm      := 0;
  PidSid      := 0;
  PidAc3      := 0;
  PidTeletext := 0;

  FilterRemoveByName('Video');
  FilterRemoveByName('Audio');
  FilterRemoveByName('PMT');
  FilterRemoveByName('PCR');

//  FilterRemoveByName('ECM');
//  FilterRemoveByName('SID');
//  FilterRemoveByName('AC3');
//  FilterRemoveByName('Teletext');
  // Store PIDs explicitly (some will be written over by defined filters)
  PidVideo    := Prg.Video_pid;
  PidAudio    := Prg.Audio_pid;
  PidPmt      := Prg.PMT_pid;
  PidPcr      := Prg.PCR_pid;
  PidEcm      := Prg.ECM_PID;
  PidSid      := Prg.SID_pid;
  PidAc3      := Prg.AC3_pid;
  PidTeletext := Prg.TeleText_pid;

  DVBSetFilter(Prg.Video_pid, nil, 'Video');
  DVBSetFilter(Prg.Audio_pid, nil, 'Audio');
  DVBSetFilter(Prg.PMT_pid,   nil, 'PMT');
  DVBSetFilter(Prg.PCR_pid,   nil, 'PCR');

//  DVBSetFilter(Prg.ECM_pid,   nil, 'ECM');
//  DVBSetFilter(Prg.SID_pid,   nil, 'SID');
//  DVBSetFilter(Prg.AC3_pid,   nil, 'AC3');
//  DVBSetFilter(Prg.TeleText_pid,   nil, 'Teletext');

  ChannelSet := True;                                      // Used for startup
end;


{------------------------------------------------------------------------------
  Params  : <FileName>  File name
  Returns : <Result>    0 if no error

  Descript: Start recording data
  Notes   :
------------------------------------------------------------------------------}
function DVBStartRecording(FileName: PChar): Dword; stdcall;
var
  s: string;
begin
{$IFDEF USELOG}
  ToLog('Starting recording: [' + FileName + ']', $01);
{$ENDIF}  
  Result := 1;
  if RecordingState <> [rsIdle] then
    Exit;
  if Assigned(FileStream) then
    FreeAndNil(FileStream);
  s := GetParameter('Recording', 'RecordDirectory', ExtractFilePath(Application.ExeName));
  s := s + '\' + FileName + '.ts';
  try
    FileStream := TFileStream.Create(s, fmCreate);
  except
    FreeAndNil(FileStream);
    Exit;
  end;
  RecordingState := [rsStartRecording];
  PDvbHardwareAPI^.RecordStarted := 1;
  Result := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    0 if no error

  Descript: Stop recording data
  Notes   :
------------------------------------------------------------------------------}
function DVBStopRecording: Dword; stdcall;
begin
{$IFDEF USELOG}
  ToLog('Stopping recording', $01);
{$ENDIF}
  Result := 0;
  if PDVBHardwareAPI^.RecordStarted = 0 then
    Exit;
  if RecordingState = [rsRecording] then
    RecordingState := [rsStopRecording]
  else
    RecordingState := [rsIdle];
  PDVBHardwareAPI^.RecordStarted := 0;
end;


{------------------------------------------------------------------------------
  Params  : <Msg>       Windows message (=MDAPI call)
                        WParam = MDAPI call
                        LParam = pointer to data or data itself
  Returns : <Result>    1 means handled

  Descript: MDAPI calls
  Notes   :
------------------------------------------------------------------------------}
function ProcessMDAPI(var Msg: TMessage): DWORD; stdcall;
{$IFDEF USESOFTCSA}
var
  i               : Integer;
{$ENDIF}
var
  StartFilterParam: PStartFilterParam;
//  Locked          : Byte;
//  Level           : Byte;
//  Quality         : Byte;
//  AGC             : Byte;
//  SNR_SQE         : Byte;
begin
{$IFDEF USELOG}
  ToLog(format('MDAPI command: $%x', [Msg.WParam]), $05);
{$ENDIF}
  Result := 0;
  Msg.Result := result;
  case Msg.WParam of
    // MDAPI calls which might by handled by the application ....
    // Note: setting the result to '1' indicates that the call has been handled
    //       and therefore the application does not handle it anymore.
//    MDAPI_GET_TRANSPONDER: ;
//    MDAPI_SET_TRANSPONDER: ;
//    MDAPI_GET_PROGRAMM   : ;
//    MDAPI_SET_PROGRAMM   : ;
//    MDAPI_RESCAN_PROGRAMM: ;
//    MDAPI_SAVE_PROGRAMM  : ;
//    MDAPI_GET_PROGRAMM_NUMMER: ;
//    MDAPI_SET_PROGRAMM_NUMMER: ;
    MDAPI_START_FILTER:
      begin
        if not HandleMdAPi then
          Exit;
        StartFilterParam := PStartFilterParam(Msg.LParam);
        StartFilterParam^.Running_ID := DvbSetFilter(StartFilterParam^.Pid, TMDAPIFilterProc(StartFilterParam^.Irq_Call_Adresse),PChar(@(StartFilterParam^.Name[0])));
        Result := 1;
      end;
    MDAPI_STOP_FILTER:
      begin
        if not HandleMdAPi then
          Exit;
        DvbStopFilter(Msg.LParam);
        Result := 1;
      end;
//    MDAPI_SCAN_CURRENT_TP : ;
//    MDAPI_SCAN_CURRENT_CAT: ;
//    MDAPI_START_OSD       : ;
//    MDAPI_OSD_DRAWBLOCK   : ;
//    MDAPI_OSD_SETFONT     : ;
//    MDAPI_OSD_TEXT        : ;
//    MDAPI_SEND_OSD_KEY    : ;
//    MDAPI_STOP_OSD        : ;
//    MDAPI_DVB_COMMAND     : ;
//    MDAPI_GET_VERSION     :
//      begin
//        StrPCopy(PChar(Msg.LParam), 'MD-API Version 01.02 Root 254376');
//        Result := 1;
//      end;
//    PROGAPI_GET_SIGNAL_LEVEL:
//      begin
//        if not HandleMdAPi then
//          Exit;
//        DvbSignalQuality(@Locked, @Level, @Quality, @AGC, @SNR_SQE);
//        PTSIGNAL_INFO(Msg.LParam)^.SignalLevel := SNR_SQE;
//        PTSIGNAL_INFO(Msg.LParam)^.SignalError := AGC;
//        Result := 1;
//      end;
//    PROGAPI_SEND_DISEQC     : ;
//    PROGAPI_GET_CHANNEL_NAME: ;
//    PROGAPI_SET_CHANNEL     : ;
//    PROGAPI_SET_TRANSPONDER : ;
    // Command always handled by the driver
    MDAPI_DVB_COMMAND:
      begin
{$IFDEF USESOFTCSA}
        // SoftCSA
        if PDVB_COMMAND(msg.LParam)^.Cmd_laenge = 7 then
          for i:= 4 to 13 do
            SAA_COMMAND[i] := PDVB_COMMAND(msg.LParam)^.Cmd_Buffer[i];
        SetCSAKeys(SAA_COMMAND);
        Result := 1;
{$ENDIF}
      end;
{$IFDEF USELOG}
    else
      ToLog(format('MDAPI command $%x unprocessed', [Msg.WParam]), $03);
{$ENDIF}
  end;
  Msg.Result := Result;
end;


{------------------------------------------------------------------------------
  Params  : <DVBHardware> Pointer for returned data
  Returns : <DVBHardware> Pointer for returned data

  Descript: Fill in the information fields
  Notes   :
------------------------------------------------------------------------------}
procedure TtPcLineBudgetFillDVBHardwareInfo(DVBHardware: PByte); stdcall;
begin
  PDVBHardwareAPI := PDVBHardware(DVBHardware);
  PDVBHardwareAPI^.DVBPacketStyle         := WINSTB_PACKET_STYLE_RAW;
  PDVBHardwareAPI^.DVBPacketLegth         := CDvbPacketSize;         // Support for full packet
  PDVBHardwareAPI^.PacketCountInCallback  := 1;                      // 1 packet per callback
  PDVBHardwareAPI^.HardwareTVOutType      := WINSTB_HARDWARE_TVOUT_NOT_PRESENT; // No TV out on the DVB card itself
  PDVBHardwareAPI^.DVBInit                := @DvbInit;                // Initialization function
  PDVBHardwareAPI^.DVBExit                := @DvbExit;                // Finalization function
  PDVBHardwareAPI^.AddFilter              := @DvbSetFilter;           // Adding filter handling function
  PDVBHardwareAPI^.DelFilter              := @DvbStopFilter;          // Removal filter handling function
  PDVBHardwareAPI^.StartRecording         := @DvbStartRecording;      // Start recording handling function
  PDVBHardwareAPI^.StopRecording          := @DvbStopRecording;       // Stop recording handling function
  PDVBHardwareAPI^.MDPluginProcessMessage := @ProcessMDAPI;           // MD API handling function
  PDVBHardwareAPI^.SetDISEQC              := @SetDISEQC;              // DiSEqC handling function
  PDVBHardwareAPI^.SetChannel             := @DvbSetChannel;          // Channel setting function
  PDVBHardwareAPI^.GetSignalLevelAndQuality := @DvbSignalQuality;     // Signal quality and level
  PDVBHardwareAPI^.InternalDSHandling     := WINSTB_DS_HANDLED_BY_WINSTB;
  PDVBHardwareAPI^.SkipDS                 := WINSTB_PROCESS_DIRECTSHOW_OUTPUT;
end;





{******************************************************************************}
{                            FORM (MAINLY DEBUG)                               }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Close;
end;


procedure TfrmMain.rgPolarityClick(Sender: TObject);
begin
  case rgPolarity.ItemIndex of
    0: if not Saa7146aHorizontalPolarityLNB(CardHandle, (LNBPolarity = 0)) then
         ToLog('Could not set horizontal polarity.', $82);
    1: if not Saa7146aVerticalPolarityLNB(CardHandle, (LNBPolarity = 0)) then
         ToLog('Could not set vertical polarity.', $82);
  end;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    ToLog('Could not convert symbolrate.', $82);
    Exit;
  end;
  if not Saa7146aQpskSetSymbolRate(CardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  PowerPercentage: Double;
  PowerDecibel   : Double;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  DeltaTime : Dword;
  DeltaCount: Dword;
  Ticks     : Dword;
  Count     : Word;
  Second    : Boolean;
begin
  Ticks := GetTickCount;
  DeltaTime := Abs(Ticks - UpdatePreviousTimer);
  if DeltaTime >= 1000 then
  begin
    Second := True;
    UpdatePreviousTimer := Ticks;
  end
  else
    Second := False;

  if pgPageControl.ActivePage = tsDebug then
  begin
    if Second then
    begin
      if PacketBufferCount <> PacketPreviousCount then
      begin
        // Additional check because of circular buffer
        if PacketBufferCount > PacketPreviousCount then
        begin
          DeltaCount := PacketBufferCount - PacketPreviousCount;
          txtMs.Caption := format('%d', [DeltaTime div DeltaCount]);
        end;
        PacketPreviousCount := PacketBufferCount;
      end;
    end;

    if Second then
    begin
      Count := VideoPacketCount;
      if Count <> VideoPacketPreviousCount then
      begin
        if Count > VideoPacketPreviousCount then
        begin
          DeltaCount := Count - VideoPacketPreviousCount;
          // Only for a valid difference (eg. buffer restart at 0)
          if DeltaCount < 10000 then
          begin
            // Note: We use <CDvbPacketSize> but the actual data contents
            //       is less...  Since the rate varies this 'error' does not matter.
            DeltaCount := DeltaCount * CDvbPacketSize * 8;
            DeltaCount := DeltaCount div DeltaTime;
            if DeltaCount < 10 then
              txtVideoRate.Caption := '-'
            else
              txtVideoRate.Caption := IntToStr(DeltaCount);
          end;
        end;
        VideoPacketPreviousCount := Count;
      end;
    end;

//    txtDebug.Caption    := IntToStr(Debug);
//    txtDebugStr.Caption := DebugStr;
    txtPackets.Caption := IntToStr(PacketBufferCount);
    txtFilters.Caption := IntToStr(FilterCount);
    txtFiltersDefined.Caption := IntToStr(FiltersDefined) + ' (' + FiltersDefinedPids + ')';
    txtPidVideo.Caption := IntToStr(PidVideo);
    txtPidAudio.Caption := IntToStr(PidAudio);
    txtPidPmt.Caption   := IntToStr(PidPmt);
    txtPidPcr.Caption   := IntToStr(PidPcr);
    txtOvertaken.Caption := IntToStr(Overtaken);
  end;

  if pgPageControl.ActivePage = tsDiSEqC then
  begin
    if Saa7146aQpskGetLockDetectorPercentage(CardHandle, Percentage) then
      ggLock.Progress := Percentage;
    if Saa7146aQpskGetCarrierDeviation(CardHandle, Deviation) then
      stDeviation.Caption := format('%d', [Deviation]);
    if Saa7146aQpskGetSymbolRateDeviation(CardHandle, Deviation) then
      stDeviationSymbolRate.Caption := format('%d', [Deviation]);
    if Saa7146aQpskGetSignalPower(CardHandle, PowerPercentage, PowerDecibel) then
    begin
      txtPower.Caption := format('%f%% (%f dB)', [PowerPercentage, PowerDecibel]);
    end;
    if Saa7146aReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
    begin
      if (LockState and $80) <> 0 then
        StatusStr := 'CARRIER'
      else
        StatusStr := 'NO CARRIER';
      if (LockState and $10) <> 0 then
        StatusStr := StatusStr + ' - VITERBI'
      else
        StatusStr := StatusStr + ' - NO VITERBI';
      if (LockState and $08) <> 0 then
        StatusStr := StatusStr + ' - SYNC'
      else
        StatusStr := StatusStr + ' - NO SYNC';
      if (LockState and $98) = $98 then
        StatusStr := StatusStr + ' - LOCK'
      else
        StatusStr := StatusStr + ' - NO LOCK';
      txtStatus.Caption := format('$%2.2x : %s', [LockState, StatusStr]);
    end;
    if Saa7146aQpskGetNoiseIndicator(CardHandle, Noise) then
      txtNoise.Caption := format('%f dB', [Noise]);
  end;    
end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error    : Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    ToLog('Could not convert frequency.', $82);
    Exit;
  end;
  if not Saa7146aFrequencyLNB(CardHandle, Frequency, 9750000, 10600000, TunerType, SynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
end;


procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;


procedure TfrmMain.btnBurstAClick(Sender: TObject);
begin
  if not DiseqcBurstLnbA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('1', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnBurstBClick(Sender: TObject);
begin
  if not DiseqcBurstLnbB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('2', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcAOptionAClick(Sender: TObject);
begin
  if not DiseqcLnbAOptionA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('1', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;


procedure TfrmMain.btnDiseqcBOptionAClick(Sender: TObject);
begin
  if not DiseqcLnbBOptionA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('2', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcAOptionBClick(Sender: TObject);
begin
  if not DiseqcLnbAOptionB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('3', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcBOptionBClick(Sender: TObject);
begin
  if not DiseqcLnbBOptionB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('4', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;




procedure TfrmMain.btnResetPositionerClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $00;                                    // Reset
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStopMovementClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $60;                                    // Stop positioner movement
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDisableLimitsClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $63;                                    // Disable limits
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnSetEastLimitClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $66;                                    // Set East limit
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnSetWestLimitClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $67;                                    // Set West limit
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnGoEastClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStepEastClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnGoWestClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStepWestClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStorePositionClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6A;                                    // Store position xx
  DiSEqCData[3] := UdPosition.Position;                    // Position xx
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
  udPosition.Position := udPosition.Position + 1;
end;

procedure TfrmMain.btnGotoPositionClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6B;                                    // Goto position xx
  DiSEqCData[3] := UdPosition.Position;                    // Position xx
  if not Saa7146aDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;


procedure TfrmMain.udPositionChangingEx(Sender: TObject;
  var AllowChange: Boolean; NewValue: Smallint;
  Direction: TUpDownDirection);
begin
  Newvalue := udPosition.Position;
  if Direction = updUp then
  begin
    Newvalue := NewValue + 1;
    if NewValue > udPosition.Max then
      NewValue := udPosition.Max;
  end
  else
    if Direction = updDown then
    begin
      Newvalue := NewValue - 1;
      if NewValue < udPosition.Min then
        NewValue := udPosition.Min;
    end;
  btnStorePosition.Caption := format('Store satellite %3.3d', [NewValue]);
  btnGotoPosition.Caption := format('Goto satellite %3.3d', [NewValue]);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
{$IFDEF USELOG}
var
  Parameter: string;
  Error    : Integer;
  LogClear : Boolean;
{$ENDIF}
begin
  CardHandle   := INVALID_HANDLE_VALUE;
  ThreadHandle := INVALID_HANDLE_VALUE;
  FilterLock   := TCriticalSection.Create;
  IniFile      := nil;
  LogStream    := nil;
  PDVBHardwareAPI := nil;
  LogLevel     := 0;
  Debug        := 0;
  DebugStr     := 'Debug:';
  HandleMdAPi  := False;
  TunerType    := CTunerBSRU6;
{$IFDEF USELOG}
  LogClear     := True;
{$ENDIF}
  if FileExists('drivers\ttpclinebudget\TTPClineBudget.ini') then
  begin
    IniFile := TMemIniFile.Create('drivers\ttpclinebudget\TTPClineBudget.ini');
    if LowerCase(GetParameter('Interface', 'HandleMdAPi', 'No')) = 'yes' then
      HandleMdApi := True
    else
      HandleMdApi := False;  
{$IFDEF USELOG}
    // Only with a local INI file we allow logging
    Parameter := GetParameter('Debug', 'LogLevel', '0');
    Val(Parameter, LogLevel, Error);
    if Error <> 0 then
      LogLevel := 0;
    Parameter := GetParameter('Debug', 'LogClear', 'Yes');
    if LowerCase(Parameter) = 'no' then
      LogClear := False
    else
      LogClear := True;
{$ENDIF}
  end;
{$IFDEF USELOG}
  if LogLevel in [1..5] then
  begin
    try
      if (not LogClear) and
         FileExists('drivers\ttpclinebudget\TTPClineBudget.log') then
      begin
        LogStream := TFileStream.Create('drivers\ttpclinebudget\TTPClineBudget.log', fmOpenReadWrite);
        // Go to end of file (we will be appending)
        LogStream.Seek(0, soFromEnd);
      end
      else
        LogStream := TFileStream.Create('drivers\ttpclinebudget\TTPClineBudget.log', fmCreate);
      ToLog('-------------------------------------------------------------', $00);
      ToLog(format('Log level: %d', [LogLevel]), $00);
    except
      FreeAndNil(LogStream);
    end;
  end;
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  StopDVB;
  FilterLock.Free;
  if Assigned(LogStream) then
    FreeAndNil(LogStream);
  if Assigned(IniFile) then
    FreeAndNil(IniFile);
end;


{$IFDEF ISDLL}
  procedure LibExit(Reason: Integer);
  begin
    ExitProc := ExitSave;
    Finalize;
  end;

  exports   TtPcLineBudgetFillDVBHardwareInfo name 'FillDVBHardwareInfo';

  begin
    Initialize;
    ExitSave := ExitProc;
    ExitProc := @LibExit;

{$ELSE}

initialization

  finalization
    Finalize;
{$ENDIF}

end.

