{*********************************************************************************
 *                                                                               *
 * WinSTB, FlexCop card interface (SkyStar2)                                     *
 *                                                                               *
 * Copyright (C) 2005 M.Majoor                                                   *
 *                                                                               *
 *                                                                               *
 * This program is free software; you can redistribute it and/or                 *
 * modify it under the terms of the GNU General Public License                   *
 * as published by the Free Software Foundation; either version 2                *
 * of the License, or (at your option) any later version.                        *
 *                                                                               *
 *                                                                               *
 * This program is distributed in the hope that it will be useful,               *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
 * GNU General Public License for more details.                                  *
 *                                                                               *
 *                                                                               *
 * You should have received a copy of the GNU General Public License             *
 * along with this program; if not, write to the Free Software                   *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.    *
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html                *
 *                                                                               *
 *                                                                               *
 * The author can be reached at m.majoor@majority.nl                             *
 * Homepage: http://www.majority.nl                                              *
 *********************************************************************************}

{******************************************************************************}
{ FileName............: UFlexCop                                               }
{ Project.............: WinSTB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.01                                                   }
{------------------------------------------------------------------------------}
{ FlexCop (SkyStar2) card support                                              }
{                                                                              }
{ ALL handling is done here or in the support routines. Nothing is handled by  }
{ the device driver except for acknowledging interrupts.                       }
{------------------------------------------------------------------------------}
{                                                                              }
{ Note: Requires an installed and working FLEXCOP.SYS driver!                  }
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050410 - Initial release                                           }
{  1.01   20050417 - Synchronization added (FlexCop not always captures        }
{                    with packet starting at zero offset)                      }
{******************************************************************************}
{$IFDEF ISDLL}
  library UFlexCop;
{$ELSE}
  unit UFlexCop;
  interface
{$ENDIF}

uses
  Classes, ComCtrls, Dialogs, ExtCtrls, Forms, Gauges, Graphics, IniFiles,
  Mask, Messages,
  StdCtrls, SyncObjs, SysUtils, Windows,
  UMDAPI, UDVBHardware, Controls,
{$IFDEF USESOFTCSA}
  UCSAEmu,
{$ENDIF}
  FlexCopGeneral, FlexCopI2c, FlexCopInterface, FlexCopIoControl, FlexCopRegisters,
  Stv0299bRegisters,
  DirectShow9, ActiveX, DSUtil,
  Grids;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    tmrUpdate: TTimer;
    pgPageControl: TPageControl;
    tsDebug: TTabSheet;
    grpInformatio: TGroupBox;
    txtDebugStr: TStaticText;
    StaticText17: TStaticText;
    StaticText16: TStaticText;
    StaticText10: TStaticText;
    txtDebug: TStaticText;
    txtFiltersDefined: TStaticText;
    txtFilters: TStaticText;
    txtPackets: TStaticText;
    StaticText24: TStaticText;
    txtOvertaken: TStaticText;
    txtMs: TStaticText;
    StaticText25: TStaticText;
    mmoDriver: TMemo;
    grpPid: TGroupBox;
    StaticText20: TStaticText;
    StaticText21: TStaticText;
    StaticText22: TStaticText;
    StaticText23: TStaticText;
    txtPidVideo: TStaticText;
    txtPidPmt: TStaticText;
    txtPidAudio: TStaticText;
    txtPidPcr: TStaticText;
    tsDiSEqC: TTabSheet;
    GroupBox2: TGroupBox;
    btnStepWest: TButton;
    btnStepEast: TButton;
    StaticText11: TStaticText;
    btnStopMovement: TButton;
    btnResetPositioner: TButton;
    btnGoEast: TButton;
    btnGoWest: TButton;
    StaticText12: TStaticText;
    btnSetEastLimit: TButton;
    btnSetWestLimit: TButton;
    btnDisableLimits: TButton;
    btnStorePosition: TButton;
    udPosition: TUpDown;
    btnGotoPosition: TButton;
    StaticText13: TStaticText;
    StaticText14: TStaticText;
    StaticText15: TStaticText;
    GroupBox1: TGroupBox;
    btnBurstA: TButton;
    btnBurstB: TButton;
    btnDiseqcAOptionA: TButton;
    btnDiseqcBOptionA: TButton;
    btnDiseqcAOptionB: TButton;
    btnDiseqcBOptionB: TButton;
    StaticText9: TStaticText;
    ggLock: TGauge;
    stDeviationSymbolrate: TStaticText;
    stDeviation: TStaticText;
    StaticText4: TStaticText;
    StaticText7: TStaticText;
    mskSymbolRate: TMaskEdit;
    mskFrequency: TMaskEdit;
    StaticText3: TStaticText;
    StaticText1: TStaticText;
    StaticText2: TStaticText;
    rgPolarity: TRadioGroup;
    StaticText8: TStaticText;
    txtNoise: TStaticText;
    txtStatus: TStaticText;
    StaticText6: TStaticText;
    StaticText5: TStaticText;
    txtPower: TStaticText;
    StaticText35: TStaticText;
    txtVideoRate: TStaticText;
    StaticText36: TStaticText;
    StaticText37: TStaticText;
    StaticText38: TStaticText;
    StaticText39: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnStopMovementClick(Sender: TObject);
    procedure btnResetPositionerClick(Sender: TObject);
    procedure btnGoEastClick(Sender: TObject);
    procedure btnGoWestClick(Sender: TObject);
    procedure btnSetEastLimitClick(Sender: TObject);
    procedure btnSetWestLimitClick(Sender: TObject);
    procedure btnDisableLimitsClick(Sender: TObject);
    procedure btnStorePositionClick(Sender: TObject);
    procedure btnStepEastClick(Sender: TObject);
    procedure btnStepWestClick(Sender: TObject);
    procedure btnGotoPositionClick(Sender: TObject);
    procedure udPositionChangingEx(Sender: TObject;
      var AllowChange: Boolean; NewValue: Smallint;
      Direction: TUpDownDirection);
    procedure btnDiseqcAOptionAClick(Sender: TObject);
    procedure btnDiseqcBOptionAClick(Sender: TObject);
    procedure btnDiseqcAOptionBClick(Sender: TObject);
    procedure btnDiseqcBOptionBClick(Sender: TObject);
    procedure btnBurstAClick(Sender: TObject);
    procedure btnBurstBClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;
{$IFDEF ISDLL}
  {$R UFlexCop.DFM}
{$ELSE}
  procedure FlexCopFillDVBHardwareInfo(DVBHardware: PByte); stdcall;
  implementation
  {$R *.DFM}
{$ENDIF}

const
  MAX_FILTER_COUNT = 255;
  CInfraredAddress = $4000;
  Version          = $0101;
  Build            = $20050417;
  CPacketBuffers   = 20;
  CBufferPackets   = 512;
  CPacketSize      = 188;
  CBufferSize      = CBufferPackets * CPacketSize;

type
  PDvbTransportPacket = ^TDvbTransportPacket;
  TDvbTransportPacket = array[0..CPacketSize - 1] of Byte;

  PDvbTransportPackets = ^TDvbTransportPackets;
  TDvbTransportPackets = array[0..CBufferPackets - 1] of TDvbTransportPacket;

  // 'One too many' for synchronization
  PDvbTransportPackets2 = ^TDvbTransportPackets2;
  TDvbTransportPackets2 = array[0..CBufferPackets - 1] of TDvbTransportPacket;

  TRecordingStates = set of (rsIdle, rsStartRecording, rsRecording, rsStopRecording);

  // Thread used for acquiring data packerts
  TDataThread = class(TThread)
  private
    { Private declarations }
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
  protected
    procedure Execute; override;
  end;

  TMDAPIFilterProc = procedure( hFilter: DWORD; Len: Cardinal; Buf: PByteArray); cdecl;

  // PID filter definition
  TFilter = record
    Name            : string;                    // Name of filter
    Active          : Boolean;                   // Filter is (to be) set active
    InRecording     : Boolean;                   // True if used in recording
    Pid             : Word;                      // Only used for cross reference
    CallBackFunction: TMDAPIFilterProc;          // Function to call
  end;
{$IFDEF USESOFTCSA}
  // SoftCSA
  TSetKeyProc     = procedure(Key: PByteArray; Keystruct: PByteArray); cdecl;
  TCSADecryptProc = procedure(Keystruct: PByteArray; Encrypted: PByteArray); cdecl;
{$ENDIF}

var
{$IFDEF ISDLL}
  ExitSave    : Pointer;
{$ENDIF}

  FlexCopForm: TfrmMain;

  // Settings
  DisplayDuringRecording: Boolean = True;
  DisplayOff            : Boolean = False;

  // Driver
  CardHandle  : THandle;                         // Handle to driver
  CardInUse   : Integer;                         // Number of the card

  IniFile     : TMemIniFile;                     // Ini file

  // Packet data
  ThreadHandle     : THandle;                    // Handle to driver in thread
  PacketThread     : TDataThread;                // Thread handling packets
  StreamDataBuffer : TDvbTransportPackets2;      // Received data
  RecordingState   : TRecordingStates;           // Recording states
  VideoInRecording : Boolean;                    // True if to be included in recordings
  AudioInRecording : Boolean;                    // True if to be included in recordings
  PmtInRecording   : Boolean;                    // True if to be included in recordings
  PcrInRecording   : Boolean;                    // True if to be included in recordings
  EcmInRecording   : Boolean;                    // True if to be included in recordings
  SidInRecording   : Boolean;                    // True if to be included in recordings
  Ac3InRecording   : Boolean;                    // True if to be included in recordings
  TeletextInRecording: Boolean;                  // True if to be included in recordings
  FileStream       : TFileStream;                // Filestream for recording
  LogStream        : TFileStream;                // Filestream for log
  LogLevel         : Byte;                       // LogLevel (00 = no log)
                                                 //  $x0 = No log
                                                 //  $x1 = Lowest level
                                                 //  $x2 = Command errors
                                                 //  $x3 = Procedure/Function entries
                                                 //  $x4 = Procedure/Function specifics
                                                 //  $x5 = Generic procedure/Function entries
  HandleMdApi      : Boolean;                    // True will handle more MDAPI commands
                                                 // than only the DVB_COMMAND

  // DiSEqC
  DiSEqCRepeats: Byte;                           // Number of repeats for DiSEqC commands
  DISEQCType   : string;                         // DiSEqC type to use
  SetDiSEqCSource       : string;                // Last used settings <SetDiSEqc> (for refresh purposes)
  SetDiSEqCLNB          : string;
  SetDiSEqCPolarization : Char;
  SetDiSEqCFrequency    : Dword;
  SetDiSEqCSymbolRate   : Dword;

  //
  PDvbHardwareAPI     : PDvbHardware;            // Pointer to API data
  Filters             : array[1..MAX_FILTER_COUNT] of TFilter; // Pid filters
  FilterCrossReference: array[0..$1FFF] of Word; // Pid index references to <Filters>
  FiltersDefined      : Word;                    // Number of active filters
  FiltersDefinedPids  : string;                  // All active filter Pids as ASCII
  FilterLock          : TCriticalSection;        // Prevents changing filters while they are being in use


  ChannelSet     : Boolean;                      // Used for startup

  // Debug
  Debug              : Dword;                    // Generic debug
  DebugStr           : string;                   // Generic debug
  PacketBufferCount  : Word;                     // Counts received buffers (multiple packets)
  PacketPreviousCount: Word;                     // Counts received buffers previous check
  UpdatePreviousTimer: Dword;                    // Previous tick count checked
  VideoPacketCount        : Word;                // Counts received video packets
  VideoPacketPreviousCount: Word;                // Counts received buffers previous check
  FilterCount      : Word;                       // Counts filter procedures called
  PidVideo         : Word;                       // Video Pid
  PidAudio         : Word;                       // Audio Pid
  PidPmt           : Word;                       // PMT Pid
  PidPcr           : Word;                       // PCR Pid
  PidEcm           : Word;                       // ECM Pid
  PidSid           : Word;                       // Sid Pid
  PidAc3           : Word;                       // AC3 Pid
  PidTeletext      : Word;                       // Teletext Pid

  FifoBuffer  : TFlexCopFifoTransferBuffer;      // Data buffering
  IrqHandling0: TFlexCopIrqTransferBuffer;       // Subbuffer 0 interrupts
  IrqHandling1: TFlexCopIrqTransferBuffer;       // Subbuffer 1 interrupts
  Buffer1     : TFlexCopDmaBuffer;
  Buffer2     : TFlexCopDmaBuffer;
  BufferSize  : Integer;                         // Actual buffer size


{$IFDEF USESOFTCSA}
  // SoftCSA
  InternalCSA      : Boolean;                  // True indicates internal mechanism to use
  DLLID            : Integer;
  SetKeyProc       : TSetKeyProc;
  CSADecryptProc   : TCSADecryptProc;
  KeyStruct        : array[0..800] of Byte;
  Key              : array[0..15] of Byte;
  KeysOdd          : TBlock;
  KeysEven         : TBlock;
  KeysPreviousOdd  : TBlock;
  KeysPreviousEven : TBlock;
  OddKeyFound      : Boolean = False;
  EvenKeyFound     : Boolean = False;
  KeyFound         : Boolean = False;
  SAA_COMMAND      : array [0..13] of Byte = ($10,4,5,0,0,0,0,0,0,0,0,0,0,0);
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : <LogString>  Data to log
            <Level>      Log level: if this number if <= <LogLevel> then
                         it is logged
                         $8x indicates that the string is intended for
                             displaying on form too.
  Returns : -

  Descript: Write data to log. A timestamp is added.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ToLog(LogString: string; Level: Byte);
{$IFDEF USELOG}
var
  NewLog: string;
{$ENDIF}
begin
  if ((Level and $80) <> 0) and
     Assigned(FlexCopForm) then
    FlexCopForm.mmoDriver.Lines.Add(LogString);
{$IFDEF USELOG}
  // Only log if the level is high enough
  if (Level and $0F) > LogLevel then
    Exit;
  // The check on <LogStream> is necessary because the application might
  // call this prior to completing initialization
  if not Assigned(LogStream) then
    Exit;
  NewLog := FormatDateTime('YYYYMMDD"T"HHMMSS"  "', Now) + LogString + #13#10;
  LogStream.Write(NewLog[1], Length(NewLog));
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : <Section>    Section ta access in INI file
            <Paremeter>  Paremeter in INI file
            <Default>    Default value (eg. if not present in INI file)
  Returns : <Result>     String value for parameter

  Descript: Get parameter from INI file.
  Notes   : Either the local INI file is used or the global INI file.
            Comments (;) are stripped out (' and " indicates strings and are also
            removed from the result -> either ' or " is used as string
            indentifier, meaning the other one is handled as a normal character)
            If in the local ini no reference is found the application ini
            file is tried too.
 ------------------------------------------------------------------------------}
function GetParameter(Section: string; Parameter: string; Default: string): string;
var
  FromIni : string;
  Position: Integer;
  TempStr : string;
begin
{$IFDEF USELOG}
//  ToLog('GetParameter', $05);
{$ENDIF}
  // If we don't have a local INI file then use the applications INI file
  if not Assigned(IniFile) then
  begin
    if not Assigned(PDVBHardwareAPI) then
    begin
      Result := '';
      Exit;
    end;
{$IFDEF USELOG}
    ToLog('GetParameter [application]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
    Result := PDVBHardwareAPI^.GetParameter(@Section[1], @Parameter[1], @Default[1]);
    Exit;
  end;
  // Try to find it locally (no default so we get an empty string if nothing is found)
  FromIni := IniFile.ReadString(Section, Parameter, '');
  // If nothing found, try application
  if (FromIni = '') and
     Assigned(PDVBHardwareAPI) then
  begin
{$IFDEF USELOG}
    ToLog('GetParameter [application]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
    FromIni := PDVBHardwareAPI^.GetParameter(@Section[1], @Parameter[1], '');
  end
  else
{$IFDEF USELOG}
    ToLog('GetParameter [local]: [' + Section + '][' + Parameter + '][' + Default + ']', $04);
{$ENDIF}
  // Nothing found at all then use default
  if FromIni = '' then
    FromIni := Default;
  // Strip out leading/trailing spaces
  FromIni := Trim(FromIni);
  // Check for starting string parameter
  Position := Pos('''', FromIni);
  if Position = 1 then
  begin
    TempStr  := Copy(FromIni, Position+1, 255);
    // Find ending of string
    Position := Pos('''', TempStr);
    if Position <> 0 then
    begin
      // End of string found, which will be the result
      Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
      ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
      Exit;
    end;
  end
  else
  begin
    Position := Pos('"', FromIni);
    if Position = 1 then
    begin
      TempStr := Copy(FromIni, Position+1, 255);
      // Find ending of string
      Position := Pos('"', TempStr);
      if Position <> 0 then
      begin
        // End of string found, which will be the result
        Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
        ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
        Exit;
      end;
    end;
  end;
  // We know we don't have a string type so handle the whole string as
  // normal text. We could have a comment in it so check this too
  Position := Pos(';', FromIni);
  if Position <> 0 then
  begin
    TempStr := Copy(FromIni, 1, Position-1);
    // Strip out leading/trailing spaces
    Result := Trim(TempStr);
  end
  else
    Result := FromIni;
{$IFDEF USELOG}
  ToLog('GetParameter result: [' + Result + ']', $05);
{$ENDIF}
end;


{******************************************************************************}
{                                     DISEQC                                   }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Set horizontal polarity
  Notes   :
------------------------------------------------------------------------------}
function DiseqcSetHorizontalPolarity: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcSetHorizontalPolarity', $05);
{$ENDIF}
  Result := FlexCopHorizontalPolarityLNB(CardHandle);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Set vertical polarity
  Notes   :
------------------------------------------------------------------------------}
function DiseqcSetVerticalPolarity: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcSetVerticalPolarity', $05);
{$ENDIF}
  Result := FlexCopVerticalPolarityLNB(CardHandle);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Enable LNB
  Notes   :
------------------------------------------------------------------------------}
function DiseqcEnableLnb: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcEnableLnb', $05);
{$ENDIF}
  Result := FlexCopEnableLNB(CardHandle);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Disable LNB
  Notes   :
------------------------------------------------------------------------------}
function DiseqcDisableLnb: Boolean;
begin
{$IFDEF USELOG}
  ToLog('DiseqcDisableLnb', $05);
{$ENDIF}
  Result := FlexCopDisableLNB(CardHandle);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Simple burst only, selects LNB A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcBurstLnbA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcBurstLnbA', $05);
{$ENDIF}
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCBurstA, 0, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Simple burst only, selects LNB B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcBurstLnbB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcBurstLnbB', $05);
{$ENDIF}
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCBurstB, 0, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A, option A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAOptionA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAOptionA', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F0;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A, option B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAOptionB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAOptionB', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F8;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B, option A
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBOptionA: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBOptionA', $05);
{$ENDIF}
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F4;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B, option B
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBOptionB: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBOptionB', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $FC;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB A including simple burst mechanism
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbAPlusBurst: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbAPlusBurst', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F0;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand or CDiSEqCBurstA, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Select LNB B including simple burst mechanism
  Notes   :
------------------------------------------------------------------------------}
function DiseqcLnbBPlusBurst: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcLnbBPlusBurst', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $10;                                    // Any LNB/switcher
  DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
  DiSEqCData[3] := $F4;                                    // Switch ID in low nibble
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand or CDiSEqCBurstB, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner reset command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerReset: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerReset', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $00;                                    // Reset
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner stop movement command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStopMovement: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStopMovement', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $60;                                    // Stop positioner movement
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner disable limits command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerDisableLimits: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerDisableLimits', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $63;                                    // Disable limits
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner set east limit command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerSetEastLimit: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerSetEastLimit', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $66;                                    // Set East limit
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner set west limit command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerSetWestLimit: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerSetWestLimit', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $67;                                    // Set West limit
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner move east command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGoEast: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGoEast', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner step east command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStepEast: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStepEast', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner move west command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGoWest: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGoWest', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  False if error

  Descript: Positioner step west command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStepWest: Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStepWest', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : <Position> Satellite position to store
  Returns : <Result>   False if error

  Descript: Positioner store current position command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerStorePosition(Position: Byte): Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerStorePosition', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6A;                                    // Store position xx
  DiSEqCData[3] := Position;                               // Position xx
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;


{------------------------------------------------------------------------------
  Params  : <Position> Satellite position to store
  Returns : <Result>   False if error

  Descript: Positioner goto position command
  Notes   :
------------------------------------------------------------------------------}
function DiseqcPositionerGotoPosition(Position: Byte): Boolean;
var
  DiSEqcData: TDiSEqCData;
begin
{$IFDEF USELOG}
  ToLog('DiseqcPositionerGotoPosition', $05);
{$ENDIF}  
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6B;                                    // Goto position xx
  DiSEqCData[3] := Position;                               // Position xx
  Result := FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData);
end;
{******************************************************************************}
{                                  DISEQC END                                  }
{******************************************************************************}


{------------------------------------------------------------------------------
  Params  : <Source>  Satellite name
  Returns : <Result>  LNB number for setup to use
                      Empty if error occured

  Descript: Set LNB
  Notes   :
------------------------------------------------------------------------------}
function SetLNB(Source: PChar): string;
var
  IdSearch      : AnsiString;
  CheckSatellite: Byte;
  SatelliteFound: Boolean;

  function CmpSrc(s1, s2: string): Boolean;
  begin
    Result := s1 = (Copy(s2, 1, Length(s1)-1)+'S');
  end;
begin
{$IFDEF USELOG}
  ToLog('SetLNB', $05);
{$ENDIF}  
  Result := '';
  if Lowercase(DiSEqCType) = 'none' then
  begin
    Result := '1';
  end
  else
    if LowerCase(DiSEqCType) = 'mini-diseqc' then
    begin
      if CmpSrc(Source, GetParameter('DiSEqC', 'LNB1Source', '')) then
      begin
        if not DiseqcBurstLnbA then
          Exit;
        Result := '1';
      end
      else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB2Source', '')) then
      begin
        if not DiseqcBurstLnbB then
          Exit;
        Result := '2';
      end
    end
    else
      if DiSEqCType = '1.0' then
      begin
        if CmpSrc(Source, GetParameter('DiSEqC', 'LNB1Source', '')) then
        begin
          if not DiseqcLnbAOptionA then
            Exit;
          Result := '1';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB2Source', '')) then
        begin
          if not DiseqcLnbBOptionA then
            Exit;
          Result := '2';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB3Source', '')) then
        begin
          if not DiseqcLnbAOptionB then
            Exit;
          Result := '3';
        end
        else if CmpSrc(Source, GetParameter('DiSEqC', 'LNB4Source', '')) then
        begin
          if not DiseqcLnbBOptionB then
            Exit;
          Result := '4';
        end;
      end
      else
        // If positioner support
        if DiSEqCType = '1.2' then
        begin
          // Find number for satellite
          // We search for 'LNB0Source' .. 'LNB99Source'
          CheckSatellite := 0;
          repeat
            IdSearch := format('LNB%dSource', [CheckSatellite]);
            SatelliteFound := CmpSrc(Source, GetParameter('DiSEqC', IdSearch, ''));
            if not(SatelliteFound) then
              Inc(CheckSatellite);
          until SatelliteFound or (CheckSatellite > 100);
          if SatelliteFound then
          begin
            // We found the satellite, now control the positioner
            if not DiseqcPositionerGotoPosition(CheckSatellite) then
              Exit;
            Result := '1';                              // Use LNB1 for settings
          end;
        end;
end;


{------------------------------------------------------------------------------
  Params  : <LNB>           LNB identification (for high/low band frequency)
            <Polarization>  Polarization ('H' or 'V')
            <Frequency>     Frequency in kHz
            <Symbolrate>    Symbolrate in kHz
  Returns : <Result>        True if no error

  Descript: Set frequency, polarization, symbolrate
  Notes   :
------------------------------------------------------------------------------}
function SetFrequency(LNBNumber: string; Polarization: Char; Frequency, SymbolRate: Dword): Boolean;
var
  LowBand   : Dword;
  HighBand  : Dword;
  Parameter : string;
  Error     : Integer;
begin
{$IFDEF USELOG}
  ToLog('SetFrequency', $05);
{$ENDIF}  
  Result := False;
  // If we have a valid LNB then set frequency and such
  if LNBNumber <> '' then
  begin
    if GetParameter('LNB', 'LNB'+LNBNumber+'Type', '') = 'Circular' then
      Frequency := Frequency - 150000;
    Parameter := GetParameter('LNB', 'LNB'+LNBNumber+'LOF1', '9750000' );
    Val(Parameter, LowBand, Error);
    if Error <> 0 then
      LowBand := 9750000;
    Parameter := GetParameter('LNB', 'LNB'+LNBNumber+'LOF2', '10600000');
    Val(Parameter, HighBand, Error);
    if Error <> 0 then
      HighBand := 10600000;
    // Check if for some reason we are a factor 1000 too small
    if LowBand < 100000 then
      LowBand := LowBand * 1000;
    if HighBand < 100000 then
      HighBand := HighBand * 1000;
    // Check if we are for some reason a factor 1000 too high
    if SymbolRate > 1000000 then
      SymbolRate := SymbolRate div 1000;
    if Assigned(FlexCopForm) then
    begin
      // Reflect in form
      FlexCopForm.mskFrequency.Text  := format('%8.8d', [Frequency]);
      FlexCopForm.mskSymbolRate.Text := format('%5.5d', [SymbolRate]);
      if Polarization = 'V' then
        FlexCopForm.rgPolarity.ItemIndex := 1
      else
        FlexCopForm.rgPolarity.ItemIndex := 0;
    end;
    // Set polarity/frequency/symbolrate
    if Polarization = 'V' then
    begin
      if not FlexCopVerticalPolarityLNB(CardHandle) then
        Exit;
    end
    else
      if not FlexCopHorizontalPolarityLNB(CardHandle) then
        Exit;
    if not FlexCopFrequencyLNB(CardHandle, Frequency, LowBand, HighBand, 0, 1) then
      Exit;
    if not FlexCopQpskSetSymbolRate(CardHandle, SymbolRate) then
      Exit;
    Result := True;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Source>        Satellite name
            <Polarization>  Polarization ('H' or 'V')
            <Frequency>     Frequency in kHz
            <Symbolrate>    Symbolrate in kHz
  Returns : <Result>        0 = no error

  Descript: Set frequency, polarization, symbolrate and satellite
  Notes   :
------------------------------------------------------------------------------}
function SetDiSEqC(Source, Polarisation: PChar; Frequency, SymbolRate: Dword): Dword; stdcall;
var
  LNBNumber: string;
begin
{$IFDEF USELOG}
  ToLog(format('Set DiSEqC: %s, %s, %d, %d', [Source, Polarisation, Frequency, SymbolRate]), $03);
{$ENDIF}  
  // Save copy if we want to use it locally
  SetDiSEqCSource       := Source;
  SetDiSEqCPolarization := Polarisation^;
  SetDiSEqCFrequency    := Frequency;
  SetDiSEqCSymbolRate   := Symbolrate;

  LNBNumber := SetLNB(Source);
  SetDiSEqCLNB := LNBNumber;
  if SetFrequency(SetDiSEqCLNB, SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolRate) then
    Result := 0
  else
    result := 1;  
end;


{******************************************************************************}
{                                     PACKETS                                  }
{******************************************************************************}
{$IFDEF USESOFTCSA}
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: Sets keys for SoftCSA decoding
    Notes   : Command is layout as follows
               0 1 2 3 4 5 6 7 8 9 10 11 12 13
                       x                          0 = even keys in [6..13]
                       x                          1 = odd  keys in [6..13]
                           1 2 3 4 5  6  7  8     Typically the keys need endian swapping!
   ------------------------------------------------------------------------------}
  procedure SetCSAKeys(Command: array of Byte);
  var
    i: Integer;
    j: Integer;
    Change: Boolean;
{$IFDEF USELOG}
    KeyString: string;
{$ENDIF}
  begin
{$IFDEF USELOG}
    ToLog('SetCSAKeys', $05);
{$ENDIF}    
    if Command[4] = 1 then
      OddKeyFound := True
    else
      EvenKeyFound := True;
    for i := 0 to 3 do
    begin
      // Do endian swapping
      Key[(Command[4])*8+i*2+0] := Command[6+i*2+1];
      Key[(Command[4])*8+i*2+1] := Command[6+i*2+0];
      // If we are using the internal decrytor then prepare keys
      if InternalCSA then
      begin
        if Command[4] = 0 then
          for j := 0 to 3 do
          begin
            KeysEven[j*2+0] := Command[6+j*2+1];
            KeysEven[j*2+1] := Command[6+j*2+0];
          end
        else
          for j := 0 to 3 do
          begin
            KeysOdd[j*2+0] := Command[6+j*2+1];
            KeysOdd[j*2+1] := Command[6+j*2+0];
          end;
      end;
    end;
    if OddKeyFound and EvenKeyFound then
    begin
      KeyFound := True;
      // Internal handling checkcs for a change here
      if InternalCSA then
      begin
        Change := False;
        for j := 0 to 7 do
        begin
          if KeysOdd[j] <> KeysPreviousOdd[j] then
            Change := True;
          if KeysEven[j] <> KeysPreviousEven[j] then
            Change := True;
        end;
        if Change then
        begin
{$IFDEF USELOG}
          if LogLevel >= 3 then
          begin
            KeyString := 'Set CSA keys:  Even= ';
            for j := 0 to 7 do
              KeyString := KeyString + format('$%2.2x ', [KeysEven[j]]);
            KeyString := KeyString + '  Odd= ';
            for j := 0 to 7 do
              KeyString := KeyString + format('$%2.2x ', [KeysOdd[j]]);
            ToLog(KeyString, $03);
          end;
{$ENDIF}
          CSASetKeys(KeysOdd, KeysEven);
          for j := 0 to 7 do
          begin
            KeysPreviousOdd[j]  := KeysOdd[j];
            KeysPreviousEven[j] := KeysEven[j];
          end;
        end;
      end
      else
      begin
{$IFDEF USELOG}
        if LogLevel >= 3 then
        begin
          KeyString := 'Set CSA keys:  Even= ';
          for j := 0 to 7 do
            KeyString := KeyString + format('$%2.2x ', [Key[j]]);
          KeyString := KeyString + '  Odd= ';
          for j := 0 to 7 do
            KeyString := KeyString + format('$%2.2x ', [Key[j+8]]);
          ToLog(KeyString, $03);
        end;
{$ENDIF}
        SetKeyProc(@Key[0], @KeyStruct[0]);
      end;
    end;
  end;
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Decode transport stream. Uses <StreamDataBuffer> which holds
            multiple stream packets.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ProcessTS;
var
  StreamPacket: Word;
  Pid         : Word;
  DoRecord    : Boolean;
begin
  try
    FilterLock.Acquire;
    try
      for StreamPacket := 0 to CBufferPackets-1 do
      begin
        // Get Pid
        Pid := ((StreamDataBuffer[StreamPacket][1] and $1F) shl 8) or StreamDataBuffer[StreamPacket][2];
        // If cross reference exists then its active
        if FilterCrossReference[Pid] <> $FFFF then
        begin
  {$IFDEF USESOFTCSA}
          // SoftCSA, check for scrambled data
          if ((StreamDataBuffer[StreamPacket][3] and $80) <> 0) and
               KeyFound then
            if InternalCSA then
              CSADecrypt(@StreamDataBuffer[StreamPacket])
            else
              CSADecryptProc(@KeyStruct[0], @StreamDataBuffer[StreamPacket]);
  {$ENDIF}
          if Assigned(@Filters[FilterCrossReference[Pid]].CallBackFunction) then
          begin
            Filters[FilterCrossReference[Pid]].CallBackFunction(FilterCrossReference[Pid], 184, @StreamDataBuffer[StreamPacket][4]);
            if FilterCount = $FFFF then
              FilterCount := 0
            else
              Inc(FilterCount);
          end;
        end;

        // Debug
        if (Pid <> 0) and (Pid = PidVideo) then
        begin
          if VideoPacketCount = $FFFF then
            VideoPacketCount := 0
          else
            Inc(VideoPacketCount);
        end;

        // If recording is started, synchronize on start indicator
        if (RecordingState = [rsStartRecording]) and
           ((StreamDataBuffer[StreamPacket][1] and $40) <> 0) then
          RecordingState := [rsRecording];
        // If recording is being stopped, synchronize on start indicator
        if (RecordingState = [rsStopRecording]) and
           ((StreamDataBuffer[StreamPacket][1] and $40) <> 0) then
        begin
          RecordingState := [rsIdle];
          FreeAndNil(FileStream);
        end;
        // Write packet to file if requested
        // Only if a recording has been started or if we are stopping
        // Strip out PID=0 (if only audio PID exists for radio channels)
        if ((RecordingState = [rsRecording]) or
            (RecordingState = [rsStopRecording])) and
            (Pid <> 0) then
        begin
          // We no longer check the filters for recording but use the local settings
          // so we can record these independend of the filter settings
          // Assume no recording
          DoRecord := False;
          if (Pid = PidVideo) and VideoInRecording then
            DoRecord := True
          else
            if (Pid = PidAudio) and AudioInRecording then
              DoRecord := True
            else
              if (Pid = PidPmt) and PmtInRecording then
                DoRecord := True
              else
                if (Pid = PidPcr) and PcrInRecording then
                  DoRecord := True
                else
                  if (Pid = PidEcm) and EcmInRecording then
                    DoRecord := True
                  else
                    if (Pid = PidSid) and SidInRecording then
                      DoRecord := True
                    else
                      if (Pid = PidAc3) and Ac3InRecording then
                        DoRecord := True
                      else
                        if (Pid = PidTeletext) and TeletextInRecording then
                          DoRecord := True;
          if DoRecord then
            FileStream.Write(StreamDataBuffer[StreamPacket], CPacketSize);
        end;
      end;
    finally
      FilterLock.Release;
    end;
    // Pass all packets to DirectShow
    // Note: Only sending the typical Pids which have been setup for filtering will not
    //       generate an image, except when all Pid's in in the <TProgramm82> record
    //       have been activated.
    if not DisplayOff then
      if DisplayDuringRecording or
         (RecordingState = [rsIdle]) then
        PDVBHardwareAPI^.SendDataToDirectShow(@StreamDataBuffer, CBufferSize);
  except
  end;      
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrupt at both edges of the field indicator,
            meaning that we get notified when a switch is made from
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Buffer      : TFlexCopFifoTransferBuffer;
  LastRead    : Dword;
  SyncRequired: Integer;                         // Counts invalid sync at start buffer
  SyncOffset  : Integer;                         // Offset for correction
  SyncData    : TDvbTransportPacket;             // Remainder data previous data block
begin
  // Debug
{$IFDEF USELOG}
  ToLog('TDataThread.Execute', $05);
{$ENDIF}
  PacketBufferCount :=0;
  VideoPacketCount  :=0;
  FilterCount       :=0;

  LastRead       := $FFFF;
  HasStopped     := False;
  ProgrammedStop := False;
  // The driver must be running and must be in use
  if FlexCopGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't then only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls eg. 'generate manual notification'
  // which might be needed in a deadlock situation.
  ThreadHandle := FlexCopCreateFile(CardInUse);
  SyncRequired := 0;
  SyncOffset := 0;
  Buffer.TransferAddress[0] := @StreamDataBuffer[0][SyncOffset];
  // Wait for application to have setup stuff
  // This is necessary because if we send data with <SendDataToDirectShow>
  // too early (without it being set up) then this generates and error and
  // this thread will terminate.
  ChannelSet := False;
  repeat
   Sleep(100);
  until ChannelSet or Terminated;
  try
    if not Terminated then
    repeat
      if ChannelSet then
      begin
{$IFDEF USELOG}
        ToLog('TDataThread [channel set]', $05);
{$ENDIF}
        ChannelSet := False;
      end;
      FlexCopReadFromIrqHandling(ThreadHandle, IrqHandling0);
      while (not Terminated) and
            (IrqHandling0.Information.FifoBufferPreviousIndex = LastRead) do
      begin
        Sleep(1);
        FlexCopReadFromIrqHandling(ThreadHandle, IrqHandling0);
      end;
      if PacketBufferCount = $FFFF then
        PacketBufferCount := 0
      else
        Inc(PacketBufferCount);
      Inc(LastRead);
      if LastRead > IrqHandling0.Information.FifoBufferLastIndex then
        LastRead := IrqHandling0.Information.FifoBufferFirstIndex;

      Buffer.TransferLength := CBufferSize;
      Buffer.Identifier := LastRead;
      FlexCopReadFromFifo(ThreadHandle, Buffer);

      // Check first packet (the offset should have made the second packet correct)
      if (StreamDataBuffer[1][0] <> $47)  then
        Inc(SyncRequired)
      else
        if SyncRequired > 0 then
          Dec(SyncRequired);
      if SyncRequired > 50 then
      begin
        // Search for sync
        SyncOffset := 0;
        while (SyncOffset < CPacketSize) and (StreamDataBuffer[0][SyncOffset] <> $47) do
          Inc(SyncOffset);
        // Not found, default to start
        // Otherwise the correction is related to the packet size because
        // the correct data is to be placed in the second packet of
        // StreamDataBuffer
        if (SyncOffset = 0) or (SyncOffset = CPacketSize) then
          SyncOffset := 0
        else
          SyncOffset := CPacketSize - SyncOffset;
        Buffer.TransferAddress[0] := @StreamDataBuffer[0][SyncOffset];
        SyncRequired := 0;
      end
      else
      begin
        // We have to store the data at the end and use it for the
        // next block of data
        if SyncOffset <> 0 then
        begin
          // Copy remainder of previous block
          CopyMemory(@StreamDataBuffer, @SyncData, SyncOffset);
          // Copy leftover of current block ('one too many' packet)
          CopyMemory(@SyncData, @StreamDataBuffer[High(TDvbTransportPackets2)], SyncOffset);
        end;
      end;
      if not Terminated then
        ProcessTS;
    until Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
    begin
      ToLog('Unexpected termination of packets handler.', $81);
      if not Assigned(FlexCopForm) then
        ShowMessage('Unexpected termination of packets handler.');
    end
    else
      ToLog('Normal termination of packets handler.', $81);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>    Handle to driver
            <DmaChannel2>   False for DMA1
                            True  for DMA2
            <ActivateDma>   True if DMA also to be activated (in address)
            <Packets>       Number of packets to use for size
            <SubBuffer0>    Pointer to physical address DMA subbuffer 0
            <SubBuffer1>    Pointer to physical address DMA subbuffer 0
            <PacketIrqs>    True if packet interrupts are to be enabled
  Returns : <Result>        True if success

  Descript: Setup FlexCop registers
  Notes   :
 ------------------------------------------------------------------------------}
function SetupFlexCopAddress(CardHandle: THandle; DmaChannel2: Boolean; ActivateDma: Boolean;
           Packets: Word; SubBuffer0: Cardinal; SubBuffer1: Cardinal; PacketIrqs: Boolean): Boolean;
var
  Dma        : Dword;
  Address    : Dword;
  PacketSized: Integer;
  SetSize    : Dword;
begin
  Result      := False;
  PacketSized := Packets * CPacketSize;

  if ActivateDma then
    Dma := CFlexCopSubBufferEnableIrq
  else
    Dma := 0;
  // Fortunately our allocated memory is always dword aligned ....
  // Buffer 1 address
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer0Address
  else
    Address := CFlexCopDma1SubBuffer0Address;
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer0 or Dma) then
    Exit;

  // Sub buffer 1 is either enabled or disabled (nothing to do with actual DMA enabling)  
  if ActivateDma then
    Dma := CFlexCopSubBufferEnableSubBuffer1
  else
    Dma := 0;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer1Address
  else
    Address := CFlexCopDma1SubBuffer1Address;
  // Buffer 2 address
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer1 or Dma) then
    Exit;

  // Buffer size
  // We have allocated memory which is a multiple of the DMA block size.
  // For convenience we adjust this to the even packet size.
  SetSize := PacketSized;
  SetSize := SetSize shr 2;                              // Size if 4-byte chunks
  SetSize := SetSize shl 8;                              // Shift size
  if not PacketIrqs then
    SetSize := SetSize or CFlexCopSubBufferDisablePacketIrq;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBufferSize
  else
    Address := CFlexCopDma1SubBufferSize;
  if not FlexCopWriteToFlexCopRegister(CardHandle, Address, SetSize) then
    Exit;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>     Handle to driver
            <IsSlave>        True if setup as slave
            <PacketBuffers>  Number of buffers to allocate
            <PacketIrqs>     True if packet interrupts to be enabled
  Returns : <Result>         True if success
            <PacketBuffers>  Actual buffers allocated
            <BufferId>       Buffer identifier (DMA) for data stored into

  Descript: Setup buffering
  Notes   : We don't remove allocated DMA/FIFO's and such since
            it is implicitly done at the start here and in the
            termination of the application
 ------------------------------------------------------------------------------}
function BufferingSetup(CardHandle: THandle; IsSlave: Boolean;
  var PacketBuffers: Word; var BufferId: Dword; PacketIrqs: Boolean): Boolean;
var
  Channel    : Dword;
  Buffer3    : TFlexCopDmaBuffer;
  Buffer4    : TFlexCopDmaBuffer;
begin
  Result     := False;
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;

  // Setup DMA
  //
  // We need to setup several buffers.
  // Buffer 1:  FlexCop target buffer for satellite data (subbuffer 0) first DMA
  // Buffer 2:  FlexCop target buffer for satellite data (subbuffer 1) first DMA
  // Buffer 3:  Dummy buffer for second DMA
  // Buffer 4:  Dummy buffer for second DMA
  //
  // The FlexCop issues an interrupt when one of it's buffers is filled (in our
  // case this buffer is buffer 1 or 2). The driver (if properly set up) then
  // copies this data to somewere in buffer 3.
  // Typically the FlexCop issues two interrupts. One when it's first subbuffer
  // has been filled, and another when it's second subbuffer has been filled.
  // Using the 'sharing' mechanism of the FIFO we can make sure that all data
  // is placed on our common buffer 3.
  //
  // We are dealing with DMA 1 only here (DMA2 is not used). To recap, the
  // following registers/settings are to be set up:
  // DMA1 - generates two distinguishes interrupts: One for subbuffer 0, and
  //        another for subbuffer 1
  //        CFlexCopIrqDma10                : DMA1, subbuffer 0 interrupt
  //        CFlexCopIrqDma11                : DMA1, subbuffer 1 interrupt
  //        CFlexCopEnableDma10             : DMA1, subbuffer 0 interrupt enable
  //        CFlexCopEnableDma11             : DMA1, subbuffer 1 interrupt enable
  //        CFlexCopDma1SubBuffer0Address   : DMA1, subbuffer 0 target address
  //        CFlexCopDma1SubBufferSize       : DMA1, subbuffer size
  //        CFlexCopDma1SubBuffer1Address   : DMA1, subbuffer 1 target address

  // First disable all possible interrupts and such
  if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0) then
    Exit;

  // Release any memory already allocated
  Channel := 0;
  while FlexCopReleaseDma(CardHandle, Channel) do
    Inc(Channel);
  // Release all FIFO's
  FifoBuffer.Identifier := -1;
  FlexCopReleaseFifo(CardHandle, FifoBuffer);

  // Allocate buffer 1
  if not FlexCopAllocateDma(CardHandle, (((CBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer1) then
    Exit;
  // Allocate buffer 2
  if not FlexCopAllocateDma(CardHandle, (((CBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer2) then
    Exit;
  // Dummy buffers so we have at least a valid buffer, which we use for DMA2
  // Allocate buffer 3
  if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer3) then
    Exit;
  // Allocate buffer 4
  if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer4) then
    Exit;

  // Setup initial addresses/size
  BufferSize := CBufferSize;
  if not SetupFlexCopAddress(CardHandle, False, False, CBufferPackets,
    Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;
  // Also do this for DMA2 although we don't use DMA2
  // (just to make sure DMA addresses are valid)
  if not SetupFlexCopAddress(CardHandle, True, False, 1,
    Buffer3.PhysicalAddress.LowPart, Buffer4.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;

  // Setup Fifo. This allocates the number of buffers we want.
  with FifoBuffer do
  begin
    NumberOfBuffers := PacketBuffers;
    // Make sure all unused items are reset
    for Channel := Low(TransferAddress) to High(TransferAddress) do
      TransferAddress[Channel] := nil;
    // Note: The VIRTUAL address !
    // Fill in for subbuffer 0 and 1
    TransferAddress[CFlexCopIrqDma10Irq] := Pointer(Buffer1.VirtualAddress);
    TransferAddress[CFlexCopIrqDma11Irq] := Pointer(Buffer2.VirtualAddress);
    TransferLength := BufferSize;
  end;
  if not FlexCopAllocateFifo(CardHandle, FifoBuffer) then
    Exit;
    
  // At this point all buffers has been setup (allocated):
  //  Buffer 1 and 2 (DMA mechanism, so contiguous memory)

  // Enable interrupt mechanism of the driver
  // When this is set the actual buffering of data (to the FIFO) is activated
  // Setup for subbuffer 0 (DMA1 - subbuffer 0)
  with IrqHandling0.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := False;
    UseSignaling         := False;
    UseFifo              := True;
    FifoBufferFirstIndex := FifoBuffer.Identifier;
    FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
  end;
  IrqHandling0.Identifier := CFlexCopIrqDma10Irq;
  FlexCopWriteToIrqHandling(CardHandle, IrqHandling0);

  // Setup for subbuffer 1 (DMA1 - subbuffer 1)
  // Subbuffer 1 uses the same fifo buffers as subbuffer 0
  with IrqHandling1.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := False;
    UseSignaling         := False;
    UseFifo              := True;
    FifoBufferFirstIndex := FifoBuffer.Identifier;
    FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
  end;
  IrqHandling1.Identifier := CFlexCopIrqDma11Irq;
  FlexCopWriteToIrqHandling(CardHandle, IrqHandling1);

  // Now we have to enable DMA of the FlexCop. We already have setup the address
  // and such.

  // Enable subbuffer 0/1 DMA
  // Now we have to enable the reception of data
  // First the DMA bits in the address of the subbuffers
  if not SetupFlexCopAddress(CardHandle, False, True, CBufferPackets,
    Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, PacketIrqs) then
    Exit;

  // Enable reception of data
  FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable,
    CFlexCopEnableDma10 or CFlexCopEnableDma11 or CFlexCopEnableReceive);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Start DVB card
  Notes   :
 ------------------------------------------------------------------------------}
function StartDVB: Boolean;
var
  Priority   : string;
  Dummy      : Dword;
  Packets    : Word;
  MajorVersion       : Word;
  MinorVersion       : Word;
  Build              : Dword;
  DeviceName         : string;
  Data               : Dword;
  MacAddress         : Int64;
  TunerType          : Longword;
  ByteArray          : array of Byte;
  NewStr             : string;
  Loop               : Integer;
  Identification     : Word;
  AByte              : Byte;
  SynthesizerAddress : Byte;
  Size               : Word;
  CheckByte          : Byte;
  Error              : Boolean;
begin
{$IFDEF USELOG}
  ToLog('StartDVB', $03);
{$ENDIF}
  Result        := False;
  PacketThread  := nil;
  CardHandle    := INVALID_HANDLE_VALUE;
  CardInUse     := -1;
  if FlexCopGetNumberOfCards <> 0 then
  begin
    CardHandle := FlexCopCreateFile(CardInUse);
    CardInUse  := FlexCopGetCardOfHandle(CardHandle);
  end
  else
    Exit;

  // Clear pending interrupt(s)
  FlexCopReadFromFlexCopRegister(CardHandle, CFlexCopEnable, Dummy);
  // Disable all interrupts
  FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0);

  // Now with the handle we can actually access the driver.
  // Start with getting the version information of the driver.
  if FlexCopGetDriverVersion(CardHandle, MajorVersion, MinorVersion, Build, DeviceName) then
    ToLog(format('Driver "%s", Version %d.%2.2d, Build %8.8x.', [DeviceName, MajorVersion, MinorVersion, Build]), $81);

  // Pseudo reset
  FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopSpecials, CFlexCopSpecialReset);
  Sleep(1);
  FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, CFlexCopBoardReset);
  Sleep(1);
  FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, 0);
  Sleep(1);

  FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn);
  if FlexCopReadFromFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, Data) then
  begin
    ToLog(format('FlexCop power/revision register: $%8.8x', [Data]), $81);
    if (Data and CFlexCopTunerPowerMask) = CFlexCopTunerPowerOn then
      ToLog('Tuner/LNB power is on', $81)
    else
      ToLog('Tuner/LNB power is off', $81);
    if (Data and CFlexCopSystemPowerMask) = CFlexCopSystemPowerOn then
      ToLog('System power is on', $81)
    else
      ToLog('System power is off', $81);
    if (Data and CFlexCopLnbPolarityMask) = CFlexCopLnbPolarityHorizontal then
      ToLog('LNB horizontal polarity set', $81)
    else
      ToLog('LNB horizontal vertical set', $81);
    if (Data and CFlexCopWakeUpMask) = CFlexCopWakeUpOn then
      ToLog('Wake up is on', $81)
    else
      ToLog('Wake up is off', $81);
    Data := (Data shr CFlexCopRevisionShift) and CFlexCopRevisionAnd;
    case Data of
      $40: ToLog('FlexCop revision 064', $81);
      $82: ToLog('FlexCop revision 130 (FLEXCOP II)', $81);
      $C0: ToLog('FlexCop revision 192 (FLEXCOP III)', $81);
      $C2: ToLog('FlexCop revision 194', $81);
      $C3: ToLog('FlexCop revision 195 (FLEXCOP IIB)', $81);
      else ToLog(format('Unknown FlexCop revision %3.3d', [Data]), $81);
    end;
  end
  else
    ToLog('Error accessing driver', $81);

  // Get B2C2 identication
  SetLength(ByteArray, CFlexCopI2cEepromB2c2AddressBytes);
  if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromB2c2Address, ByteArray) then
  begin
    Identification := ByteArray[0];
    Identification := (Identification shl 8) or ByteArray[1];
    ToLog(format('EEPROM: Identification: %4.4x', [Identification]), $81)
  end
  else
    ToLog('Error reading MAC address from EEPROM', $81);

  // RAM
  Size := FlexCopRamDetectSize(CardHandle);
  if Size = $FFFF then
    ToLog('RAM size could not be determined', $81)
  else
  begin
    Error := False;
    ToLog(format('RAM size: %d kB', [Size]), $81);
    for Loop := 0 to (1024 * Size)-1 do
    begin
      AByte := (Loop and $FF) xor ((Loop shr 8) and $FF) xor  ((Loop shr 16) and $FF);
      if not FlexCopRamWrite(CardHandle, Loop, AByte) then
      begin
        ToLog(format('RAM writing interupted at: $%x.', [Loop]), $81);
        Error := True;
        Break;
      end;
    end;
    for Loop := 0 to (1024 * Size)-1 do
    begin
      if not FlexCopRamRead(CardHandle, Loop, AByte) then
      begin
        ToLog(format('RAM reading interupted at: $%x.', [Loop]), $81);
        Break;
      end;
      CheckByte := ((Loop and $FF) xor ((Loop shr 8) and $FF) xor ((Loop shr 16) and $FF));
      if CheckByte <> AByte then
      begin
        ToLog(format('RAM check error at address $%5.5x (read $%2.2x but was expecting $%2.2x).', [Loop, AByte, CheckByte]), $81);
        Error := True;
        Break;
      end;
    end;
    if not Error then
      ToLog('RAM check done. No errors found.', $81);
  end;

  // Get MAC address
  SetLength(ByteArray, CFlexCopI2cEepromMacAddressBytes);
  if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromMacAddress, ByteArray) then
  begin
    MacAddress := ByteArray[0];
    MacAddress := (MacAddress shl 8) or ByteArray[1];
    MacAddress := (MacAddress shl 8) or ByteArray[2];
    MacAddress := (MacAddress shl 8) or ByteArray[3];
    MacAddress := (MacAddress shl 8) or ByteArray[4];
    MacAddress := (MacAddress shl 8) or ByteArray[5];
    ToLog(format('EEPROM: Mac address: $%12.12x, checksum $%2.2x', [MacAddress, FlexCopEepromChecksum(ByteArray)]), $81)
  end
  else
    ToLog('Error reading MAC address from EEPROM', $81);

  // Get tuner type (if any)
  SetLength(ByteArray, CFlexCopI2cEepromTunerAddressBytes);
  if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromTunerAddress, ByteArray) then
  begin
    TunerType := ByteArray[0];
    TunerType := (TunerType shl 8) or ByteArray[1];
    TunerType := (TunerType shl 8) or ByteArray[2];
    TunerType := (TunerType shl 8) or ByteArray[3];
    ToLog(format('EEPROM: Tuner type: $%8.8x, checksum $%2.2x', [TunerType, FlexCopEepromChecksum(ByteArray)]), $81);
  end
  else
    ToLog('Error reading tuner type from EEPROM', $81);

  // Get keys
  SetLength(ByteArray, CFlexCopI2cEepromKeyAddressBytes);
  if FlexCopReadEeprom(CardHandle, CFlexCopI2cEepromTunerAddress, ByteArray) then
  begin
    NewStr := '';
    for Loop := 0 to CFlexCopI2cEepromKeyAddressBytes-2 do
      NewStr := NewStr + format('%2.2x', [ByteArray[Loop]]);
    ToLog(format('EEPROM: keys: $%s, checksum $%2.2x', [NewStr, FlexCopEepromChecksum(ByteArray)]), $81);
  end
  else
    ToLog('Error reading keys from EEPROM', $81);

  if not FlexCopWriteEeprom(CardHandle, $300, ByteArray) then
    ToLog('Error writing data to EEPROM', $81);

  if FlexCopCheckQpskResetValues(CardHandle) then
    ToLog('QPSK values comply with reset values', $81)
  else
    ToLog('QPSK values do not comply with reset values (tuner might not have been switched off long enough for a reset)', $81);

  if not FlexCopWriteDefaultsToQpsk(CardHandle, 0) then
    ToLog('Error writing default values to QPSK', $81)
  else
    ToLog('Default values set for QPSK', $81);

  if FlexCopCheckQpskDefaultValues(CardHandle, 0) then
    ToLog('QPSK values comply with default values', $81)
  else
    ToLog('QPSK values do not comply with default values (migth be normal)', $81);

  SynthesizerAddress := FlexCopDetectSynthesizerAddress(CardHandle);
  if SynthesizerAddress = 255 then
    ToLog('Could not detect the synthesizer', $81)
  else
  begin
    ToLog(format('Synthesizer detected at index %d', [SynthesizerAddress]), $81);
    if not FlexCopReadFromSynthesizer(CardHandle, SynthesizerAddress, AByte) then
      ToLog('Could not read from synthesizer', $81)
    else
      ToLog(format('SL1935 synthesizer status: $%2.2x', [AByte]), $81);
  end;

  FlexCopSelect(CardHandle, CFlexCopMemorySelectMedia, CFlexCopMemorySelect1);     // Required

  Packets := CPacketBuffers;
  if not BufferingSetup(CardHandle, False, Packets, Dummy, False) then
    Exit;

  // Start the thread which receives notifications
  PacketThread := TDataThread.Create(True);

  // Check the priority to use
  Priority  := GetParameter('Hardware', 'ThreadPriority', 'Default');
  if LowerCase(Priority) = 'lowest' then
    PacketThread.Priority := tpLowest;
  if LowerCase(Priority) = 'low' then
    PacketThread.Priority := tpLower;
  if LowerCase(Priority) = 'normal' then
    PacketThread.Priority := tpNormal;
  if LowerCase(Priority) = 'default' then
    PacketThread.Priority := tpHigher;
  if LowerCase(Priority) = 'high' then
    PacketThread.Priority := tpHigher;
  if LowerCase(Priority) = 'highest' then
    PacketThread.Priority := tpHighest;
  if LowerCase(Priority) = 'realtime' then
    PacketThread.Priority := tpTimeCritical;
  PacketThread.Resume;

  Result := True;

  ToLog(format('Using %d buffers (appr. %d ms).', [Packets, Packets * 20]), $81);
{$IFDEF USESOFTCSA}
  if InternalCSA then
    ToLog('Using internal CSA mechanism.', $81)
  else
    ToLog('Using external CSA mechanism.', $81);
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Stop DVB card
  Notes   :
 ------------------------------------------------------------------------------}
procedure StopDVB;
var
  Channel: Dword;
  Dummy  : Dword;
begin
{$IFDEF USELOG}
  ToLog('StopDVB', $03);
{$ENDIF}
  // Remove form
  FreeAndNil(FlexCopForm);
  // Stop the packet thread if it running
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    // Note the use of <CardHandle> and NOT <ThreadHandle>. This is because
    // <ThreadHandle> might be 'busy' with a <FlexCopWaitForNotification>!
    FlexCopGenerateManualNotification(CardHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;
  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    // Disable all interrupts
    FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0);
    // Clear pending interrupt(s)
    FlexCopReadFromFlexCopRegister(CardHandle, CFlexCopEnable, Dummy);
    // Remove DMA addresses
    SetupFlexCopAddress(CardHandle, False, False, 0, 0, 0, False);
    SetupFlexCopAddress(CardHandle, True,  False, 0, 0, 0, False);
    // Disable power to LNB
    FlexCopDisableLNB(CardHandle);
    // Pseudo reset
    FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopSpecials, CFlexCopSpecialReset);
    // Release FIFO allocated memory
    FifoBuffer.Identifier := -1;
    FlexCopReleaseFifo(CardHandle, FifoBuffer);
    // Release DMA allocated memory
    Channel := 0;
    while FlexCopReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    FlexCopCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;
end;
{******************************************************************************}
{                                PACKETS END                                   }
{******************************************************************************}


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Update cross reference table for filters.
  Notes   :
------------------------------------------------------------------------------}
procedure FilterUpdateCrossReference;
var
  FilterIndex: Integer;
begin
{$IFDEF USELOG}
  ToLog('FilterUpdateCrossReference', $05);
{$ENDIF}
  // Clear cross references
  FiltersDefined := 0;
  FiltersDefinedPids := '';
  for FilterIndex := Low(FilterCrossReference) to High(FilterCrossReference) do
    FilterCrossReference[FilterIndex] := $FFFF;
  // Renew cross reference
  for FilterIndex := Low(Filters) to High(Filters) do
    if Filters[FilterIndex].Active then
    begin
      FilterCrossReference[Filters[FilterIndex].Pid] := FilterIndex;
      Inc(FiltersDefined);
      if FiltersDefinedPids = '' then
        FiltersDefinedPids := IntToStr(Filters[FilterIndex].Pid)
      else
        FiltersDefinedPids := FiltersDefinedPids + ' ' + IntToStr(Filters[FilterIndex].Pid);
    end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  0 = no error

  Descript: Allocate DVB card
  Notes   :
------------------------------------------------------------------------------}
function DvbInit: Dword; stdcall;
var
  FilterIndex: Integer;
  ShowForm   : string;
  Error      : Boolean;
  CheckString: string;
  Parameter  : string;
  ValError   : Integer;
begin
{$IFDEF USELOG}
  ToLog('DvbInit', $03);
{$ENDIF}  
{$IFDEF USESOFTCSA}
  InternalCSA      := True;
{$ENDIF}
  RecordingState   := [rsIdle];
  VideoInRecording := False;
  AudioInRecording := False;
  PmtInRecording   := False;
  PcrInRecording   := False;
  EcmInRecording   := False;
  SidInRecording   := False;
  Ac3InRecording   := False;
  TeletextInRecording := False;

  // Initialize filters
  for FilterIndex := Low(Filters) to High(Filters) do
    Filters[FilterIndex].Active := False;
  FilterUpdateCrossReference;
  // Get DiSEqC repeats and type
  Parameter := GetParameter('DiSEqC', 'RepeatCommand', '1');
  Val(Parameter, DiSEqCRepeats, ValError);
  if ValError <> 0 then
    DiSEqCRepeats := 1;
  DISEQCType     := GetParameter('DiSEqC','DiSEqCType','');

  // Get display setting
  CheckString := LowerCase(GetParameter('Interface', 'DisplayDuringRecording', 'Yes'));
  if CheckString = 'yes' then
    DisplayDuringRecording := True
  else
    DisplayDuringRecording := False;
  CheckString := LowerCase(GetParameter('Interface', 'DisplayOff', ''));
  if CheckString = 'yes' then
    DisplayOff := True
  else
    DisplayOff := False;

  // Get settings what is included in recordings
  CheckString := LowerCase(GetParameter('Recording', 'IncludeInTSRecording', 'Video, Audio, PMT, PCR'));
  if Pos('video', CheckString) <> 0 then
    VideoInRecording := True;
  if Pos('audio', CheckString) <> 0 then
    AudioInRecording := True;
  if Pos('pmt', CheckString) <> 0 then
    PmtInRecording := True;
  if Pos('pcr', CheckString) <> 0 then
    PcrInRecording := True;
  if Pos('ecm', CheckString) <> 0 then
    EcmInRecording := True;
  if Pos('sid', CheckString) <> 0 then
    SidInRecording := True;
  if Pos('ac3', CheckString) <> 0 then
    Ac3InRecording := True;
  if Pos('teletext', CheckString) <> 0 then
    TeletextInRecording := True;

  ShowForm  := GetParameter('Debug', 'FlexCopForm', 'No');
  if LowerCase(ShowForm) = 'yes' then
  begin
    FlexCopForm := TfrmMain.Create(nil);
    FlexCopForm.Caption := FlexCopForm.Caption + format(' - V%1.1x.%2.2x, build %x', [Version div $100, Version mod $100, Build]);
    FlexCopForm.pgPageControl.ActivePage := FlexCopForm.tsDebug;
    FlexCopForm.Show;
    FlexCopForm.Refresh;
  end;
{$IFDEF USESOFTCSA}
    // SoftCSA
    CheckString := LowerCase(GetParameter('Interface', 'CSA', ''));
    if CheckString = '' then
      InternalCSA := True
    else
    begin
      InternalCSA := False;
      DLLID       := LoadLibrary(PChar(CheckString));
      // If we failed using the library use the internal mechanism
      if DLLID <> 0 then
      begin
        SetKeyProc     := GetProcAddress(DLLID, 'set_cws');
        CSADecryptProc := GetProcAddress(DLLID, 'decrypt');
        if (@SetKeyProc = nil) or (@CSADecryptProc = nil) then
        begin
          ToLog('Could not use external CSA mechanism: expected procedures not present.', $81);
          FreeLibrary(DLLID);
          InternalCSA := True;
        end;
      end
      else
      begin
        ToLog('Could not load external CSA mechanism [' + CheckString + '].', $81);
        InternalCSA := True;
      end;
    end;
{$ENDIF}

  // Initialize card
  Error := not StartDVB;
  if Error then
  begin
    ToLog('Error initializing FlexCop driver of FlexCop card.', $81);
    ShowMessage('Error initializing FlexCop driver of FlexCop card.'#13#13+
                'Possible reason:'#13+
                ' - Device driver not installed or not running (FLEXCOP.SYS)'#13);
  end;
  if Error then
    Result := 1                                  // What to return? Application does not check it anyhow .
  else
    Result := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  0 = no error

  Descript: Release DVB card
  Notes   :
------------------------------------------------------------------------------}
function DvbExit: Dword; stdcall;
begin
{$IFDEF USELOG}
  ToLog('DvbExit', $03);
{$ENDIF}  
  Result := 0;
  StopDVB;
  PDVBHardwareAPI^.StopDirectshow;
{$IFDEF USESOFTCSA}
  if not InternalCSA then
    FreeLibrary(DLLID);
{$ENDIF}
  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    // Disable LNB (power off)
    DiseqcDisableLnb;
    FlexCopCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;
end;


{------------------------------------------------------------------------------
  Params  : <Locked>        Locked (1) or not locked (0)
            <Signal>        Signal level
            <Quality>       Signal quality
            <AGC>           Automatic gain setting
            <SNR_SQE>       Carrier noise ratio
  Returns : <Result>  Filter index

  Descript: Get signal status
  Notes   :
------------------------------------------------------------------------------}
function DvbSignalQuality(Locked: PByte; Signal: PByte; Quality: PByte; AGC: PByte; SNR_SQE: PByte): Dword; stdcall;
var
  LockState : Byte;
  Noise     : Double;
  StrengthDecibel   : Double;
  StrengthPercentage: Double;
begin
  Result := 0;
  // Signal power
  try
    FlexCopQpskGetSignalPower(CardHandle, StrengthPercentage, StrengthDecibel);
    Signal^ := Trunc(StrengthPercentage);
    // Lock/sync
    if FlexCopReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
    begin
      if (LockState and $98) = $98 then
        Locked^ := 1
      else
        Locked^ := 0;
    end;
    // Carrier noise (dB)
    FlexCopQpskGetNoiseIndicator(CardHandle, Noise);
    // We have to give a percentage back ... (?)
    // Range in dB = -92.4 .. 19
    // We translate 0..19 dB to 0..100%
    // Note: This is NOT the correct calculation!! dB = log!!
    if Noise < 0 then
      Noise := 0;
    Noise := Noise * 5.26;
    Quality^ := Trunc(Noise);
    SNR_SQE^ := Trunc(Noise);
    AGC^     := 100 - Trunc(StrengthPercentage);
    // Mainly for debugging; set everything to zero if thread not running
    if not Assigned(PacketThread) or
      (PacketThread.HasStopped) then                         // If stopped running
    begin
      Locked^  := 0;
      Quality^ := 0;
      Signal^  := 0;
      SNR_SQE^ := 0;
      AGC^     := 0;
    end;
  except
  end;  
end;


{------------------------------------------------------------------------------
  Params  : <Pid>         Pid for filter to set
            <FilterProc>  Callback function for Pid
            <Name>        Name for filter
  Returns : <Result>      Filter index

  Descript: Set filter data
  Notes   :
------------------------------------------------------------------------------}
function DvbSetFilter(Pid: Word; FilterProc: TMDAPIFilterProc; Name: PChar): Dword; stdcall;
var
  FilterIndex: Integer;
  Valid      : Boolean;
begin
{$IFDEF USELOG}
  ToLog(format('Set filter PID: %d', [Pid]), $03);
{$ENDIF}
  FilterLock.Acquire;
  try
    Result := 0;
    if Pid > High(FilterCrossReference) then
      Exit;
    Valid := False;
    // First try to find existing active filter
    if FilterCrossReference[Pid] <> $FFFF then
    begin
      FilterIndex := FilterCrossReference[Pid];
      Valid       := True;
    end
    else
    begin
      // No existing filter found for Pid, find first available one
      FilterIndex := Low(Filters);
      repeat
        if Filters[FilterIndex].Active then
          Inc(FilterIndex)
        else
          Valid := True;
      until Valid or (FilterIndex > High(Filters));
    end;
    if Valid then
    begin
      // Now set filter data and
      Filters[FilterIndex].Pid := Pid;
      Filters[FilterIndex].CallBackFunction := FilterProc;
      Filters[FilterIndex].Name := Name;
      Filters[FilterIndex].InRecording := False;
      if LowerCase(Name) = 'video' then
      begin
        PidVideo := Pid;
        if VideoInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'audio' then
      begin
        PidAudio := Pid;
        if AudioInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'pmt' then
      begin
        PidPmt := Pid;
        if PmtInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'pcr' then
      begin
        PidPcr := Pid;
        if PcrInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'ecm' then
      begin
        PidEcm := Pid;
        if EcmInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'sid' then
      begin
        PidSid := Pid;
        if SidInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'ac3' then
      begin
        PidAc3 := Pid;
        if Ac3InRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      if LowerCase(Name) = 'teletext' then
      begin
        PidTeletext := Pid;
        if TeletextInRecording and (Pid <> 0) then
          Filters[FilterIndex].InRecording := True;
      end;
      Filters[FilterIndex].Active := True;
      FilterUpdateCrossReference;
      Result := FilterIndex;
    end;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FilterIndex> Filter index to clear
  Returns : <Result>      0 if no error

  Descript: Clear filter data
  Notes   :
------------------------------------------------------------------------------}
function DvbStopFilter(FilterIndex: Dword): Dword; stdcall;
begin
  FilterLock.Acquire;
  try
    Result := 0;
    if FilterIndex < Low(Filters) then
      Exit;
    if FilterIndex > High(Filters) then
      Exit;
{$IFDEF USELOG}
    ToLog(format('Stop filter PID: %d', [Filters[FilterIndex].Pid]), $03);
{$ENDIF}    
    Filters[FilterIndex].Active := False;
    FilterUpdateCrossReference;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Name>  Name to search for
  Returns : -

  Descript: Remove filters which have the indicated name
  Notes   :
------------------------------------------------------------------------------}
procedure FilterRemoveByName(Name: string);
var
  FilterIndex: Integer;
begin
{$IFDEF USELOG}
//  ToLog('FilterRemoveByName', $05);
{$ENDIF}
  FilterLock.Acquire;
  try
    for FilterIndex := Low(Filters) to High(Filters) do
    begin
      Name := LowerCase(Name);
(* LOCAL DEFINITIONS ARE NOT REMOVED BECAUSE OF RECORDING
   SOME PLUGINS SPECIFICLY DELETE SOME FILTERS, LIKE THE PMT
      if Name = 'video' then
        PidVideo := 0;
      if Name = 'audio' then
        PidAudio := 0;
      if Name = 'pmt' then
        PidPmt := 0;
      if Name = 'pcr' then
        PidPcr := 0;
      if Name = 'ecm' then
        PidEcm := 0;
      if Name = 'sid' then
        PidSid := 0;
      if Name = 'ac3' then
        PidAc3 := 0;
      if Name = 'teletext' then
        PidTeletext := 0;
*)
      if LowerCase(Filters[FilterIndex].Name) = Name then
      begin
        Filters[FilterIndex].Active := False;
        FilterUpdateCrossReference;
      end;
    end;
  finally
    FilterLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Prg>  Program settings
  Returns : -

  Descript: Set channel PIDs
  Notes   :
------------------------------------------------------------------------------}
function DvbSetChannel(Prg: PProgramm82): Dword; stdcall;
begin
{$IFDEF USELOG}
  if Loglevel = 1 then
    ToLog('New channel.', $01)
  else
    ToLog(format('New channel [%d/%d/%d/%d].', [Prg.Video_pid, Prg.Audio_pid, Prg.PMT_pid, Prg.PCR_pid]), $02);
{$ENDIF}    
{$IFDEF USECSA}
  OddKeyFound  := False;
  EvenKeyFound := False;
  KeyFound     := False;
{$ENDIF}
  Result := 0;
  // We have to replace any existing audio/video/.. Pids, so first remove
  // old ones
  PidVideo    := 0;
  PidAudio    := 0;
  PidPmt      := 0;
  PidPcr      := 0;
  PidEcm      := 0;
  PidSid      := 0;
  PidAc3      := 0;
  PidTeletext := 0;

  FilterRemoveByName('Video');
  FilterRemoveByName('Audio');
  FilterRemoveByName('PMT');
  FilterRemoveByName('PCR');

//  FilterRemoveByName('ECM');
//  FilterRemoveByName('SID');
//  FilterRemoveByName('AC3');
//  FilterRemoveByName('Teletext');
  // Store PIDs explicitly (some will be written over by defined filters)
  PidVideo    := Prg.Video_pid;
  PidAudio    := Prg.Audio_pid;
  PidPmt      := Prg.PMT_pid;
  PidPcr      := Prg.PCR_pid;
  PidEcm      := Prg.ECM_PID;
  PidSid      := Prg.SID_pid;
  PidAc3      := Prg.AC3_pid;
  PidTeletext := Prg.TeleText_pid;

  DVBSetFilter(Prg.Video_pid, nil, 'Video');
  DVBSetFilter(Prg.Audio_pid, nil, 'Audio');
  DVBSetFilter(Prg.PMT_pid,   nil, 'PMT');
  DVBSetFilter(Prg.PCR_pid,   nil, 'PCR');

//  DVBSetFilter(Prg.ECM_pid,   nil, 'ECM');
//  DVBSetFilter(Prg.SID_pid,   nil, 'SID');
//  DVBSetFilter(Prg.AC3_pid,   nil, 'AC3');
//  DVBSetFilter(Prg.TeleText_pid,   nil, 'Teletext');

  ChannelSet := True;                                      // Used for startup
end;


{------------------------------------------------------------------------------
  Params  : <FileName>  File name
  Returns : <Result>    0 if no error

  Descript: Start recording data
  Notes   :
------------------------------------------------------------------------------}
function DVBStartRecording(FileName: PChar): Dword; stdcall;
var
  s: string;
begin
{$IFDEF USELOG}
  ToLog('Starting recording: [' + FileName + ']', $01);
{$ENDIF}  
  Result := 1;
  if RecordingState <> [rsIdle] then
    Exit;
  if Assigned(FileStream) then
    FreeAndNil(FileStream);
  s := GetParameter('Recording', 'RecordDirectory', ExtractFilePath(Application.ExeName));
  s := s + '\' + FileName + '.ts';
  try
    FileStream := TFileStream.Create(s, fmCreate);
  except
    FreeAndNil(FileStream);
    Exit;
  end;
  RecordingState := [rsStartRecording];
  PDvbHardwareAPI^.RecordStarted := 1;
  Result := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    0 if no error

  Descript: Stop recording data
  Notes   :
------------------------------------------------------------------------------}
function DVBStopRecording: Dword; stdcall;
begin
{$IFDEF USELOG}
  ToLog('Stopping recording', $01);
{$ENDIF}
  Result := 0;
  if PDVBHardwareAPI^.RecordStarted = 0 then
    Exit;
  if RecordingState = [rsRecording] then
    RecordingState := [rsStopRecording]
  else
    RecordingState := [rsIdle];
  PDVBHardwareAPI^.RecordStarted := 0;
end;


{------------------------------------------------------------------------------
  Params  : <Msg>       Windows message (=MDAPI call)
                        WParam = MDAPI call
                        LParam = pointer to data or data itself
  Returns : <Result>    1 means handled

  Descript: MDAPI calls
  Notes   :
------------------------------------------------------------------------------}
function ProcessMDAPI(var Msg: TMessage): DWORD; stdcall;
{$IFDEF USESOFTCSA}
var
  i               : Integer;
{$ENDIF}
var
  StartFilterParam: PStartFilterParam;
//  Locked          : Byte;
//  Level           : Byte;
//  Quality         : Byte;
//  AGC             : Byte;
//  SNR_SQE         : Byte;
begin
{$IFDEF USELOG}
  ToLog(format('MDAPI command: $%x', [Msg.WParam]), $05);
{$ENDIF}
  Result := 0;
  Msg.Result := result;
  case Msg.WParam of
    // MDAPI calls which might by handled by the application ....
    // Note: setting the result to '1' indicates that the call has been handled
    //       and therefore the application does not handle it anymore.
//    MDAPI_GET_TRANSPONDER: ;
//    MDAPI_SET_TRANSPONDER: ;
//    MDAPI_GET_PROGRAMM   : ;
//    MDAPI_SET_PROGRAMM   : ;
//    MDAPI_RESCAN_PROGRAMM: ;
//    MDAPI_SAVE_PROGRAMM  : ;
//    MDAPI_GET_PROGRAMM_NUMMER: ;
//    MDAPI_SET_PROGRAMM_NUMMER: ;
    MDAPI_START_FILTER:
      begin
        if not HandleMdAPi then
          Exit;
        StartFilterParam := PStartFilterParam(Msg.LParam);
        StartFilterParam^.Running_ID := DvbSetFilter(StartFilterParam^.Pid, TMDAPIFilterProc(StartFilterParam^.Irq_Call_Adresse),PChar(@(StartFilterParam^.Name[0])));
        Result := 1;
      end;
    MDAPI_STOP_FILTER:
      begin
        if not HandleMdAPi then
          Exit;
        DvbStopFilter(Msg.LParam);
        Result := 1;
      end;
//    MDAPI_SCAN_CURRENT_TP : ;
//    MDAPI_SCAN_CURRENT_CAT: ;
//    MDAPI_START_OSD       : ;
//    MDAPI_OSD_DRAWBLOCK   : ;
//    MDAPI_OSD_SETFONT     : ;
//    MDAPI_OSD_TEXT        : ;
//    MDAPI_SEND_OSD_KEY    : ;
//    MDAPI_STOP_OSD        : ;
//    MDAPI_DVB_COMMAND     : ;
//    MDAPI_GET_VERSION     :
//      begin
//        StrPCopy(PChar(Msg.LParam), 'MD-API Version 01.02 Root 254376');
//        Result := 1;
//      end;
//    PROGAPI_GET_SIGNAL_LEVEL:
//      begin
//        if not HandleMdAPi then
//          Exit;
//        DvbSignalQuality(@Locked, @Level, @Quality, @AGC, @SNR_SQE);
//        PTSIGNAL_INFO(Msg.LParam)^.SignalLevel := SNR_SQE;
//        PTSIGNAL_INFO(Msg.LParam)^.SignalError := AGC;
//        Result := 1;
//      end;
//    PROGAPI_SEND_DISEQC     : ;
//    PROGAPI_GET_CHANNEL_NAME: ;
//    PROGAPI_SET_CHANNEL     : ;
//    PROGAPI_SET_TRANSPONDER : ;
    // Command always handled by the driver
    MDAPI_DVB_COMMAND:
      begin
{$IFDEF USESOFTCSA}
        // SoftCSA
        if PDVB_COMMAND(msg.LParam)^.Cmd_laenge = 7 then
          for i:= 4 to 13 do
            SAA_COMMAND[i] := PDVB_COMMAND(msg.LParam)^.Cmd_Buffer[i];
        SetCSAKeys(SAA_COMMAND);
        Result := 1;
{$ENDIF}
      end;
{$IFDEF USELOG}
    else
      ToLog(format('MDAPI command $%x unprocessed', [Msg.WParam]), $03);
{$ENDIF}
  end;
  Msg.Result := Result;
end;


{------------------------------------------------------------------------------
  Params  : <DVBHardware> Pointer for returned data
  Returns : <DVBHardware> Pointer for returned data

  Descript: Fill in the information fields
  Notes   :
------------------------------------------------------------------------------}
procedure FlexCopFillDVBHardwareInfo(DVBHardware: PByte); stdcall;
begin
  PDVBHardwareAPI := PDVBHardware(DVBHardware);
  PDVBHardwareAPI^.DVBPacketStyle         := WINSTB_PACKET_STYLE_RAW;
  PDVBHardwareAPI^.DVBPacketLegth         := CPacketSize;            // Support for full packet
  PDVBHardwareAPI^.PacketCountInCallback  := 1;                      // 1 packet per callback
  PDVBHardwareAPI^.HardwareTVOutType      := WINSTB_HARDWARE_TVOUT_NOT_PRESENT; // No TV out on the DVB card itself
  PDVBHardwareAPI^.DVBInit                := @DvbInit;                // Initialization function
  PDVBHardwareAPI^.DVBExit                := @DvbExit;                // Finalization function
  PDVBHardwareAPI^.AddFilter              := @DvbSetFilter;           // Adding filter handling function
  PDVBHardwareAPI^.DelFilter              := @DvbStopFilter;          // Removal filter handling function
  PDVBHardwareAPI^.StartRecording         := @DvbStartRecording;      // Start recording handling function
  PDVBHardwareAPI^.StopRecording          := @DvbStopRecording;       // Stop recording handling function
  PDVBHardwareAPI^.MDPluginProcessMessage := @ProcessMDAPI;           // MD API handling function
  PDVBHardwareAPI^.SetDISEQC              := @SetDISEQC;              // DiSEqC handling function
  PDVBHardwareAPI^.SetChannel             := @DvbSetChannel;          // Channel setting function
  PDVBHardwareAPI^.GetSignalLevelAndQuality := @DvbSignalQuality;     // Signal quality and level
  PDVBHardwareAPI^.InternalDSHandling     := WINSTB_DS_HANDLED_BY_WINSTB;
  PDVBHardwareAPI^.SkipDS                 := WINSTB_PROCESS_DIRECTSHOW_OUTPUT;
end;





{******************************************************************************}
{                            FORM (MAINLY DEBUG)                               }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Close;
end;


procedure TfrmMain.rgPolarityClick(Sender: TObject);
begin
  case rgPolarity.ItemIndex of
    0: if not FlexCopHorizontalPolarityLNB(CardHandle) then
         ToLog('Could not set horizontal polarity.', $82);
    1: if not FlexCopVerticalPolarityLNB(CardHandle) then
         ToLog('Could not set vertical polarity.', $82);
  end;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    ToLog('Could not convert symbolrate.', $82);
    Exit;
  end;
  if not FlexCopQpskSetSymbolRate(CardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  PowerPercentage: Double;
  PowerDecibel   : Double;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  DeltaTime : Dword;
  DeltaCount: Dword;
  Ticks     : Dword;
  Count     : Word;
  Second    : Boolean;
begin
  Ticks := GetTickCount;
  DeltaTime := Abs(Ticks - UpdatePreviousTimer);
  if DeltaTime >= 1000 then
  begin
    Second := True;
    UpdatePreviousTimer := Ticks;
  end
  else
    Second := False;

  if pgPageControl.ActivePage = tsDebug then
  begin
    if Second then
    begin
      if PacketBufferCount <> PacketPreviousCount then
      begin
        // Additional check because of circular buffer
        if PacketBufferCount > PacketPreviousCount then
        begin
          DeltaCount := PacketBufferCount - PacketPreviousCount;
          if DeltaCount <> 0 then
            txtMs.Caption := format('%d', [DeltaTime div DeltaCount]);
        end;
        PacketPreviousCount := PacketBufferCount;
      end;
    end;

    if Second then
    begin
      Count := VideoPacketCount;
      if Count <> VideoPacketPreviousCount then
      begin
        if Count > VideoPacketPreviousCount then
        begin
          DeltaCount := Count - VideoPacketPreviousCount;
          // Only for a valid difference (eg. buffer restart at 0)
          if DeltaCount < 10000 then
          begin
            // Note: We use <CDvbPacketSize> but the actual data contents
            //       is less...  Since the rate varies this 'error' does not matter.
            DeltaCount := DeltaCount * CPacketSize * 8;
            if DeltaTime <> 0 then
              DeltaCount := DeltaCount div DeltaTime
            else
              DeltaCount := 0;  
            if DeltaCount < 10 then
              txtVideoRate.Caption := '-'
            else
              txtVideoRate.Caption := IntToStr(DeltaCount);
          end;
        end;
        VideoPacketPreviousCount := Count;
      end;
    end;

//    txtDebug.Caption    := IntToStr(Debug);
//    txtDebugStr.Caption := DebugStr;
    txtPackets.Caption := IntToStr(PacketBufferCount);
    txtFilters.Caption := IntToStr(FilterCount);
    txtFiltersDefined.Caption := IntToStr(FiltersDefined) + ' (' + FiltersDefinedPids + ')';
    txtPidVideo.Caption := IntToStr(PidVideo);
    txtPidAudio.Caption := IntToStr(PidAudio);
    txtPidPmt.Caption   := IntToStr(PidPmt);
    txtPidPcr.Caption   := IntToStr(PidPcr);
// Done by thread    FlexCopReadFromIrqHandling(CardHandle, IrqHandling0);
    txtOvertaken.Caption   := IntToStr(IrqHandling0.Information.FifoOverflows);
  end;

  if pgPageControl.ActivePage = tsDiSEqC then
  begin
    if FlexCopQpskGetLockDetectorPercentage(CardHandle, Percentage) then
      ggLock.Progress := Percentage;
    if FlexCopQpskGetCarrierDeviation(CardHandle, Deviation) then
      stDeviation.Caption := format('%d', [Deviation]);
    if FlexCopQpskGetSymbolRateDeviation(CardHandle, Deviation) then
      stDeviationSymbolRate.Caption := format('%d', [Deviation]);
    if FlexCopQpskGetSignalPower(CardHandle, PowerPercentage, PowerDecibel) then
    begin
      txtPower.Caption := format('%f%% (%f dB)', [PowerPercentage, PowerDecibel]);
    end;
    if FlexCopReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
    begin
      if (LockState and $80) <> 0 then
        StatusStr := 'CARRIER'
      else
        StatusStr := 'NO CARRIER';
      if (LockState and $10) <> 0 then
        StatusStr := StatusStr + ' - VITERBI'
      else
        StatusStr := StatusStr + ' - NO VITERBI';
      if (LockState and $08) <> 0 then
        StatusStr := StatusStr + ' - SYNC'
      else
        StatusStr := StatusStr + ' - NO SYNC';
      if (LockState and $98) = $98 then
        StatusStr := StatusStr + ' - LOCK'
      else
        StatusStr := StatusStr + ' - NO LOCK';
      txtStatus.Caption := format('$%2.2x : %s', [LockState, StatusStr]);
    end;
    if FlexCopQpskGetNoiseIndicator(CardHandle, Noise) then
      txtNoise.Caption := format('%f dB', [Noise]);
  end;    
end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error    : Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    ToLog('Could not convert frequency.', $82);
    Exit;
  end;
  if not FlexCopFrequencyLNB(CardHandle, Frequency, 9750000, 10600000, 0, 1) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
end;


procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;


procedure TfrmMain.btnBurstAClick(Sender: TObject);
begin
  if not DiseqcBurstLnbA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('1', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnBurstBClick(Sender: TObject);
begin
  if not DiseqcBurstLnbB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('2', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcAOptionAClick(Sender: TObject);
begin
  if not DiseqcLnbAOptionA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('1', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;


procedure TfrmMain.btnDiseqcBOptionAClick(Sender: TObject);
begin
  if not DiseqcLnbBOptionA then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('2', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcAOptionBClick(Sender: TObject);
begin
  if not DiseqcLnbAOptionB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('3', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDiseqcBOptionBClick(Sender: TObject);
begin
  if not DiseqcLnbBOptionB then
    ToLog('DiSEqC error.', $82)
  else
    if not SetFrequency('4', SetDiSEqCPolarization, SetDiSEqCFrequency, SetDiSEqCSymbolrate) then
      ToLog('DiSEqC error.', $82);
end;




procedure TfrmMain.btnResetPositionerClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $00;                                    // Reset
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStopMovementClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $60;                                    // Stop positioner movement
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnDisableLimitsClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $63;                                    // Disable limits
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnSetEastLimitClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $66;                                    // Set East limit
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnSetWestLimitClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $67;                                    // Set West limit
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 3, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnGoEastClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStepEastClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $68;                                    // Drive East
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnGoWestClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $00;                                    // Timeout (no timeout)
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStepWestClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $69;                                    // Drive West
  DiSEqCData[3] := $FF;                                    // $00 - steps, $FF = smallest
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;

procedure TfrmMain.btnStorePositionClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6A;                                    // Store position xx
  DiSEqCData[3] := UdPosition.Position;                    // Position xx
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
  udPosition.Position := udPosition.Position + 1;
end;

procedure TfrmMain.btnGotoPositionClick(Sender: TObject);
var
  DiSEqcData: TDiSEqCData;
begin
  DiSEqCData[0] := 0;                                      // Will be written over
  DiSEqCData[1] := $30;                                    // Any positioner
  DiSEqCData[2] := $6B;                                    // Goto position xx
  DiSEqCData[3] := UdPosition.Position;                    // Position xx
  if not FlexCopDiSEqCCommand(CardHandle, DiSEqCRepeats, CDiSEqCCommand, 4, DiSEqCData) then
    ToLog('DiSEqC error.', $82);
end;


procedure TfrmMain.udPositionChangingEx(Sender: TObject;
  var AllowChange: Boolean; NewValue: Smallint;
  Direction: TUpDownDirection);
begin
  Newvalue := udPosition.Position;
  if Direction = updUp then
  begin
    Newvalue := NewValue + 1;
    if NewValue > udPosition.Max then
      NewValue := udPosition.Max;
  end
  else
    if Direction = updDown then
    begin
      Newvalue := NewValue - 1;
      if NewValue < udPosition.Min then
        NewValue := udPosition.Min;
    end;
  btnStorePosition.Caption := format('Store satellite %3.3d', [NewValue]);
  btnGotoPosition.Caption := format('Goto satellite %3.3d', [NewValue]);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
{$IFDEF USELOG}
var
  Parameter: string;
  Error    : Integer;
  LogClear : Boolean;
{$ENDIF}
begin
  CardHandle   := INVALID_HANDLE_VALUE;
  ThreadHandle := INVALID_HANDLE_VALUE;
  FilterLock   := TCriticalSection.Create;
  IniFile      := nil;
  LogStream    := nil;
  PDVBHardwareAPI := nil;
  LogLevel     := 0;
  Debug        := 0;
  DebugStr     := 'Debug:';
  HandleMdAPi  := False;
{$IFDEF USELOG}
  LogClear     := True;
{$ENDIF}
  if FileExists('drivers\flexcop\FlexCop.ini') then
  begin
    IniFile := TMemIniFile.Create('drivers\flexcop\FlexCop.ini');
    if LowerCase(GetParameter('Interface', 'HandleMdAPi', 'No')) = 'yes' then
      HandleMdApi := True
    else
      HandleMdApi := False;  
{$IFDEF USELOG}
    // Only with a local INI file we allow logging
    Parameter := GetParameter('Debug', 'LogLevel', '0');
    Val(Parameter, LogLevel, Error);
    if Error <> 0 then
      LogLevel := 0;
    Parameter := GetParameter('Debug', 'LogClear', 'Yes');
    if LowerCase(Parameter) = 'no' then
      LogClear := False
    else
      LogClear := True;
{$ENDIF}
  end;
{$IFDEF USELOG}
  if LogLevel in [1..5] then
  begin
    try
      if (not LogClear) and
         FileExists('drivers\flexcop\FlexCop.log') then
      begin
        LogStream := TFileStream.Create('drivers\flexcop\FlexCop.log', fmOpenReadWrite);
        // Go to end of file (we will be appending)
        LogStream.Seek(0, soFromEnd);
      end
      else
        LogStream := TFileStream.Create('drivers\flexcop\FlexCop.log', fmCreate);
      ToLog('-------------------------------------------------------------', $00);
      ToLog(format('Log level: %d', [LogLevel]), $00);
    except
      FreeAndNil(LogStream);
    end;
  end;
{$ENDIF}
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  StopDVB;
  FilterLock.Free;
  if Assigned(LogStream) then
    FreeAndNil(LogStream);
  if Assigned(IniFile) then
    FreeAndNil(IniFile);
end;


{$IFDEF ISDLL}
  procedure LibExit(Reason: Integer);
  begin
    ExitProc := ExitSave;
    Finalize;
  end;

  exports   FlexCopFillDVBHardwareInfo name 'FillDVBHardwareInfo';

  begin
    Initialize;
    ExitSave := ExitProc;
    ExitProc := @LibExit;

{$ELSE}

initialization

  finalization
    Finalize;
{$ENDIF}

end.

