{******************************************************************************}
{ FileName............: FlexCopGeneral                                         }
{ Project.............: FLEXCOP                                                }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  General support functions                                                   }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040810 - Initial release                                           }
{ **************************************************************************** }

unit FlexCopGeneral;

interface
uses
  FlexCopInterface,
  FlexCopRegisters,
  Windows;

  // LNB control
  function FlexCopEnableLNB              (Handle: THandle): Boolean;
  function FlexCopDisableLNB             (Handle: THandle): Boolean;
  function FlexCopVerticalPolarityLNB    (Handle: THandle): Boolean;
  function FlexCopHorizontalPolarityLNB  (Handle: THandle): Boolean;
  function FlexCopRamDetectSize          (Handle: THandle): Word;
  function FlexCopRamWrite               (Handle: THandle; Address: Dword; Data: array of Byte): Boolean;
  function FlexCopRamRead                (Handle: THandle; Address: Dword; var Data: array of Byte): Boolean;
  function FlexCopSelect                 (Handle: THandle; Select: Dword; Destination: Byte): Boolean;

implementation


var
  // Note: We only support 1 card ....
  FlexCopMemorySize: Word;                       // Detected size of memory (0 / 32 / 64 / 128 / $FFFF=error)


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Enable the LNB
  Notes   : The LNB (power) is controlled throug the <CFlexCopPowerAndRevision>
            register.
------------------------------------------------------------------------------}
function FlexCopEnableLNB(Handle: THandle): Boolean;
begin
  Result := FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopPowerAndRevision,
              CFlexCopTunerPowerMask, CFlexCopTunerPowerOn);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Disable the LNB
  Notes   : The LNB (power) is controlled throug the <CFlexCopPowerAndRevision>
            register.
            It takes about 2 ms to rise from 13 -> 18V (unloaded output).
------------------------------------------------------------------------------}
function FlexCopDisableLNB(Handle: THandle): Boolean;
begin
  Result := FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopPowerAndRevision,
              CFlexCopTunerPowerMask, CFlexCopTunerPowerOff);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set vertical polarity of LNB (13V)
  Notes   : The polarity (13V/18V) is controlled throug the <CFlexCopPowerAndRevision>
            register.
------------------------------------------------------------------------------}
function FlexCopVerticalPolarityLNB(Handle: THandle): Boolean;
begin
  Result := FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopPowerAndRevision,
              CFlexCopLnbPolarityMask, CFlexCopLnbPolarityVertical);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set horizontal polarity of LNB (18V)
  Notes   : The polarity (13V/18V) is controlled throug the <CFlexCopPowerAndRevision>
            register.
------------------------------------------------------------------------------}
function FlexCopHorizontalPolarityLNB(Handle: THandle): Boolean;
begin
  Result := FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopPowerAndRevision,
              CFlexCopLnbPolarityMask, CFlexCopLnbPolarityHorizontal);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          Size (32 / 64 / 128 / $FFFF=error)

  Descript: Get/initialize RAM size
  Notes   : Only used when RAM size aready setup!
------------------------------------------------------------------------------}
function FlexCopRamGetSize(Handle: THandle): Word;
var
  OrgRamSize : Dword;
  RamInternal: Dword;
begin
  Result := $FFFF;
  // Keep original contents
  if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemorySize, OrgRamSize) then
    Exit;
  // Setup for detection/initialization
  if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySize, CFlexCopMemoryInternalCheck) then
    Exit;
  // Detect internal/external RAM
  if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemorySize, RamInternal) then
    Exit;

  // Now check if what the size returned is correct (only with external RAM)
  if RamInternal <> CFlexCopMemoryInternal then
  begin
    // Restore original contents
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySize, OrgRamSize) then
      Exit;
    OrgRamSize := OrgRamSize and CFlexCopMemorySizeMask;
    if OrgRamSize = CFlexCopMemorySize32k then
      Result := 32;
    if OrgRamSize = CFlexCopMemorySize64k then
      Result := 64;
    if OrgRamSize = CFlexCopMemorySize128k then
      Result := 128;
  end
  else
    Result := 32;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>   <FlexCopCreateFile> handle
            <RamSize>  Size to check for (32 / 64 / 128)
  Returns : <Result>   True if valid

  Descript: Check for valid RAM addresses
  Notes   :
------------------------------------------------------------------------------}
function FlexCopRamCheckSize(Handle: THandle; RamSize: Word): Boolean;
var
  TestData1   : Byte;
  TestData2   : Byte;
  CheckData   : Byte;
  CheckAddress: Dword;
begin
  Result  := False;
  try
    // Set size (we must also set the global flag otherwise we will not
    // be able to read/write)
    FlexCopMemorySize := RamSize;
    case RamSize of
      32 : begin
             if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                      CFlexCopMemorySizeMask, CFlexCopMemorySize32k) then
               Exit;
             CheckAddress := $00000;
           end;
      64 : begin
             if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                      CFlexCopMemorySizeMask, CFlexCopMemorySize64k) then
               Exit;
             CheckAddress := $10000;
           end;
      128: begin
             if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                      CFlexCopMemorySizeMask, CFlexCopMemorySize128k) then
               Exit;
             CheckAddress := $18000;
           end;
      else Exit;
    end;
    // Initialize and check it
    if FlexCopRamGetSize(Handle) <> RamSize then
      Exit;
    // Now put some data and read them back
    TestData1 := $A5;
    if not FlexCopRamWrite(Handle, CheckAddress, TestData1) then
      Exit;
    TestData2 := $4F;
    if not FlexCopRamWrite(Handle, CheckAddress + 4, TestData2) then
      Exit;
    if not FlexCopRamRead(Handle, CheckAddress, CheckData) then
      Exit;
    // If data checks out then check with inverted data
    if CheckData = TestData1 then
    begin
      TestData1 := $5A;
      if not FlexCopRamWrite(Handle, CheckAddress, TestData1) then
        Exit;
      TestData2 := $F4;
      if not FlexCopRamWrite(Handle, CheckAddress + 4, TestData2) then
        Exit;
      if not FlexCopRamRead(Handle, CheckAddress, CheckData) then
        Exit;
    end;
    if CheckData = TestData1 then
      Result := True;
  finally
    FlexCopMemorySize := $FFFF;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>   <FlexCopCreateFile> handle
  Returns : <Result>   Ram size (32 / 64 / 128 or $FFFF if error)

  Descript: Detect RAM size
  Notes   :
------------------------------------------------------------------------------}
function FlexCopRamDetectSize(Handle: THandle): Word;
var
  OrgIrqEnable: Dword;
  OrgRamSize  : Dword;
  RamInternal : Dword;
begin
  FlexCopMemorySize := $FFFF;
  Result := $FFFF;

  // Setup for initialization ????
  if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopPidHardwarePidEmmEcm, $FFFFFFFF, $4000) then
    Exit;

  // Disable interrupts
  if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopEnable, OrgIrqEnable) then
    Exit;
  try
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopEnable, 0) then
      Exit;

    // Keep original contents
    if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemorySize, OrgRamSize) then
      Exit;
    // Setup for detection/initialization
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySize, CFlexCopMemoryInternalCheck) then
      Exit;
    // Detect
    if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemorySize, RamInternal) then
      Exit;
    // Restore original contents
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySize, OrgRamSize) then
      Exit;
    // Now check if what the size returned is correct
    if RamInternal = CFlexCopMemoryInternal then
    begin
      // Set 32kB size
      if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
              CFlexCopMemorySizeMask, CFlexCopMemorySize32k) then
        Exit;
      if FlexCopRamGetSize(Handle) <> 32 then
        Exit;
      FlexCopMemorySize := 32;
    end
    else
    begin
      // We have to specificly check for the size now
      // Check for 128K
      if FlexCopRamCheckSize(Handle, 128) then
      begin
        if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                CFlexCopMemorySizeMask, CFlexCopMemorySize128k) then
          Exit;
        if FlexCopRamGetSize(Handle) <> 128 then
          Exit;
        FlexCopMemorySize := 128;
        Exit;
      end;
      // Check for 64K
      if FlexCopRamCheckSize(Handle, 64) then
      begin
        if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                CFlexCopMemorySizeMask, CFlexCopMemorySize64k) then
          Exit;
        if FlexCopRamGetSize(Handle) <> 64 then
          Exit;
        FlexCopMemorySize := 64;
        Exit;
      end;
      // Check for 32K
      if FlexCopRamCheckSize(Handle, 32) then
      begin
        if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
                CFlexCopMemorySizeMask, CFlexCopMemorySize32k) then
          Exit;
        if FlexCopRamGetSize(Handle) <> 32 then
          Exit;
        FlexCopMemorySize := 32;
        Exit;
      end;
      // Default 32K although report invalid size!
      if not FlexCopWriteWithMaskToFlexCopRegister(Handle, CFlexCopMemorySize,
              CFlexCopMemorySizeMask, CFlexCopMemorySize32k) then
        Exit;
      FlexCopRamGetSize(Handle);
      FlexCopMemorySize := $FFFF;
    end;
  finally
    // Restore original contents
    FlexCopWriteToFlexCopRegister(Handle, CFlexCopEnable, OrgIrqEnable);
    Result := FlexCopMemorySize;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Address>         Memory address
            <Data>            Bytes to store
  Returns : <Result>          True for success
                              false if error or timeout

  Descript: Write to FlexCop RAM memory
  Notes   : The FlexCop RAM is divided into 32K banks. Depending on the address
            a different bank has to be selected.
            Note that the banks selected are not as logical as you might expect.
            If we have 64K of total memory then bank 1 and 2 are used (not
            bank 0 and 1 as you might expect).
------------------------------------------------------------------------------}
function FlexCopRamWrite(Handle: THandle; Address: Dword; Data: array of Byte): Boolean;
var
  Loop   : Integer;
  Bank   : Dword;
  Control: Dword;
  Status : Dword;
  Busy   : Boolean;
  Retry  : Integer;
begin
  Result := False;
  // If RAM 'unavailable' then don't read it
  if (FlexCopMemorySize = $FFFF) or (FlexCopMemorySize = 0) then
    Exit;
  for Loop := 0 to High(Data) do
  begin
    // Each bank is 32K
    Bank := (Address and $18000) shr 15;
    // Banks used for 64K is slightly different
    if FlexCopMemorySize = 64 then
      Inc(Bank);
    Bank := Bank and $3;                                   // Juts in case
    // Construct according to type / bank / data / address / R/W)
    Control := CFlexCopMemoryControlWrite;
    Control := Control or (Bank shl CFlexCopMemoryBankShift);
    Control := Control or (Data[Loop] shl CFlexCopMemoryByteWriteShift);
    Control := Control or (Address and $7FFF);
    Control := Control or CFlexCopMemoryWrite;
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemoryControl, Control) then
      Exit;
    // Now wait until data has been written
    Retry := 2;
    repeat
      if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemoryControl, Status) then
        Exit;
      Busy := ((Status and CFlexCopMemoryBusy) <> 0);
      if Busy then
      begin
        Sleep(1);
        Dec(Retry);
      end;
    until ((not Busy) or (Retry = 0));
    if Busy then
      Exit;
    Inc(Address);
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Address>         Memory address
  Returns : <Result>          True for success
            <Data>            Bytes read

  Descript: Read from FlexCop RAM memory
  Notes   : The FlexCop RAM is divided into 32K banks. Depending on the address
            a different bank has to be selected.
            Note that the banks selected are not as logical as you might expect.
            If we have 64K of total memory then bank 1 and 2 are used (not
            bank 0 and 1 as you might expect).
------------------------------------------------------------------------------}
function FlexCopRamRead(Handle: THandle; Address: Dword; var Data: array of Byte): Boolean;
var
  Loop   : Integer;
  Bank   : Dword;
  Control: Dword;
  Status : Dword;
  Busy   : Boolean;
  Retry  : Integer;
begin
  Result := False;
  // If RAM 'unavailable' then don't read it
  if (FlexCopMemorySize = $FFFF) or (FlexCopMemorySize = 0) then
    Exit;
  for Loop := 0 to High(Data) do
  begin
    // Each bank is 32K
    Bank := (Address and $18000) shr 15;
    // Banks used for 64K is slightly different
    if FlexCopMemorySize = 64 then
      Inc(Bank);
    Bank := Bank and $3;                                   // Juts in case
    // Construct according to type / bank / data / address / R/W)
    Control := CFlexCopMemoryControlWrite;
    Control := Control or (Bank shl CFlexCopMemoryBankShift);
    Control := Control or (Address and $7FFF);
    Control := Control or CFlexCopMemoryRead;
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemoryControl, Control) then
      Exit;
    // Now wait until data has been written
    Retry := 10;
    repeat
      if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemoryControl, Status) then
        Exit;
      Busy := ((Status and CFlexCopMemoryBusy) <> 0);
      if Busy then
      begin
        Sleep(1);
        Dec(Retry);
      end;
    until ((not Busy) or (Retry = 0));
    if Busy then
      Exit;
    Data[Loop] := (Status shr CFlexCopMemoryByteWriteShift) and $FF;
    Inc(Address);
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Select>          Selection (NET/MEDIA/CA IN/CA OUT
            <Destination>     1/2
  Returns : <Result>          True for success

  Descript: Set destination NET/MEDIA/CA in/out
  Notes   : Functionality ??
------------------------------------------------------------------------------}
function FlexCopSelect(Handle: THandle; Select: Dword; Destination: Byte): Boolean;
var
  OrgData: Dword;
  Data   : Dword;
  Mask   : Dword;
  NewData: Dword;
begin
  Result := False;
  if Destination > 3 then
    Exit;
  Sleep(1);

  // Read original settings
  case Select of
    CFlexCopMemorySelectNet   : begin
                                  Data := Destination shl CFlexCopMemorySelectNetShift;
                                  Mask := CFlexCopMemorySelectNet xor $FFFFFFFF;
                                end;
    CFlexCopMemorySelectCaIn  : begin
                                  Data := Destination shl CFlexCopMemorySelectCaInShift;
                                  Mask := CFlexCopMemorySelectCaIn xor $FFFFFFFF;
                                end;
    CFlexCopMemorySelectCaOut : begin
                                  Data := Destination shl CFlexCopMemorySelectCaOutShift;
                                  Mask := CFlexCopMemorySelectCaOut xor $FFFFFFFF;
                                end;
    CFlexCopMemorySelectMedia : begin
                                  Data := Destination shl CFlexCopMemorySelectMediaShift;
                                  Mask := CFlexCopMemorySelectMedia xor $FFFFFFFF;
                                end;
    else                        Exit;
  end;
  if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopMemorySelect, OrgData) then
    Exit;
  Sleep(2);
  NewData := (OrgData and Mask) or Data;
  if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySelect, NewData) then
    Exit;
  if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopMemorySelect, NewData) then
    Exit;
  Sleep(1);
  Result := True;
end;


initialization
  FlexCopmemorySize := 0;

finalization


end.

