{******************************************************************************}
{ FileName............: DvbFilter                                              }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.05                                                   }
{------------------------------------------------------------------------------}
{  Filtering of data streams                                                   }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Note: EPG not correctly implemented .(??)...                                 }
{                                                                              }
{ There are different PID's. PID's which indicate information (PSI) or data    }
{ (PES). We first have to receive PSI data before we can identify the PES data.}
{ The global structure is as follows:                                          }
{   Program Association Table (PAT) = PID $0000                                }
{      -> PAT indicates other PID's for the Program Map Table (PMT)            }
{          -> PMT indicates other PID's for the individual streams (PES)       }
{ Thus at first we have no information at all. Then, after receiving the PAT   }
{ we can identify the PID's which carry PMT data. Then we can identify         }
{ PID's used for stream data (PES).                                            }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040222 - Initial release                                           }
{  1.01   20040507 - AC3 audio is indicated by capital language (or '***'      }
{                    instead of '---' if no language specified)                }
{  1.02   20040801 - Real program number (service identification) added        }
{  1.03   20040811 - CaDescriptors made available                              }
{  1.04   20040828 - Additional info send with PostMessage                     }
{                  - CA has now version number and callback                    }
{                  - PAT has callback                                          }
{                  - <DvbFilterGetEventInfo> also using service id             }
{  1.05   20041229 - <DvbGetPayloadOffset> detected offset now checks on       }
{                    value                                                     }
{******************************************************************************}
unit DvbFilter;

interface

uses
  Classes,
  SyncObjs,
  SysUtils,
  Windows;

const
  CSignalEvent = $0001;
  CSignalPmt = $0002;
  CSignalPid = $0004;
  CSignalCat = $0008;
  CSignalPat = $0010;

  // SI table PIDs (mandatory)
  CPidPAT = $0000; // Program Association Table
  CPidCAT = $0001; // Conditional Access Table
  CPidNIT = $0010; // Network Information Table
  CPidSDT = $0011; // Service Description Table
  CPidEIT = $0012; // Event Information Table
  CPidRST = $0013; // Running Status Table
  CPidTDT = $0014; // Time and Date Table

  CDvbPacketSize = 204 - 16;
    // DVB packet size: parity is stripped by the SAA7146A
  CDvbPacketHSync = CDvbPacketSize;
    // 'Horizontal' synchronization DD1 of SAA7146A
  CDvbPacketVSync = 512; // 'Vertical'   synchronization DD1 of SAA7146A
  CDvbPacketBufferSize = CDvbPacketHSync * CDvbPacketVSync;
    // All packets in a single buffer

  // The event information returned consists of different types of information which is present in the
  // first character of the returned strings
  CDvbEventTypeUndefined = $00;
  CDvbEventTypeShortName = $01;
  CDvbEventTypeShortText = $02;
  CDvbEventTypeExtendedItemName = $03;
  CDvbEventTypeExtendedItemText = $04;
  CDvbEventTypeExtendedText = $05;

type
  PDvbTransportPacket = ^TDvbTransportPacket;
  TDvbTransportPacket = array[0..CDvbPacketSize - 1] of Byte;

  PDvbTransportPackets = ^TDvbTransportPackets;
  TDvbTransportPackets = array[0..CDvbPacketVSync - 1] of TDvbTransportPacket;

  TDvbEventSection = class(TObject)
    FStartTime: TDateTime; // Start time/date
    FDuration: TDateTime; // Duration
    FEventText: TList; // Event info
  public
    constructor Create;
    destructor Destroy; override;
    property StartTime: TDateTime read FStartTime write FStartTime;
    property Duration: TDateTime read FDuration write FDuration;
    property EventText: TList read FEventText write FEventText;
  end;

  TDvbEventSections = class(TObject)
    FDvbEventSections: array[0..$FF] of TDvbEventSection;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetStartTime(Index: Word): TDateTime;
    procedure SetStartTime(Index: Word; Value: TDateTime);
    function GetDuration(Index: Word): TDateTime;
    procedure SetDuration(Index: Word; Value: TDateTime);
    function GetEventText(Index: Word): TList;
    procedure SetEventText(Index: Word; Value: TList);
  public
    constructor Create;
    destructor Destroy; override;
    property StartTime[Index: Word]: TDateTime read GetStartTime write
      SetStartTime;
    property Duration[Index: Word]: TDateTime read GetDuration write
      SetDuration;
    property EventText[Index: Word]: TList read GetEventText write SetEventText;
  end;

  PDvbSection = ^TDvbSection;
  TDvbSection = array[0..1023] of Byte; // Max number of bytes in a section

  // For single packet filtering
  TDvbPacketFilter = function(StreamPacketData: PDvbTransportPacket): Boolean;
    stdcall;
  // For PSI section filtering
  TDvbSectionFilter = function(SectionData: PDvbSection; Pid: Word): Boolean;
    stdcall;
  // For message callback
  PDvbCallback = ^TDvbCallback;
  TDvbCallback = record
    Handle: HWND;
    Msg: UINT;
  end;

  TDvbPids = array[0..99] of Word; // Multiple PIDs array
  TDvbLanguages = array[0..99] of string[3]; // Multiple identifiers array

function DvbFilterGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word):
  Boolean;
function DvbFilterSetSignallingCallback(Pid: Word; hWnd: HWND; Msg: UINT):
  Boolean;
function DvbFilterSetSignallingCallbackPat(hWnd: HWND; Msg: UINT): Boolean;
function DvbFilterSetSignallingCallbackCat(hWnd: HWND; Msg: UINT): Boolean;
function DvbFilterSetSignallingCallbackPmt(hWnd: HWND; Msg: UINT): Boolean;
function DvbFilterSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
function DvbFilterSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
function DvbFilterGetPidFilter(Pid: Word): TDvbPacketFilter;
function DvbFilterSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
function DvbFilterGetPsiFilter(Psi: Byte): TDvbSectionFilter;
function DvbFilterGetProgramInfo(var ProgramNumber: Byte;
  var ServiceId: Word;
  var PmtPid: Word;
  var PcrPid: Word;
  var VideoPids: TDvbPids;
  var AudioPids: TDvbPids;
  var TeletextPids: TDvbPids;
  var SubtitlePids: TDvbPids;
  var AudioLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids: TDvbPids;
  var CaIds: TDvbPids;
  var ProgramName: string): Boolean;
function DvbFilterGetEmm(Index: Integer; var CaSystemId: Word; var CaPid: Word):
  Boolean;
function DvbFilterGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present:
  Boolean; Events: TDvbEventSections): Boolean;
function DvbFilterGetNumberOfPrograms: Byte;
function DvbFilterGetErrors: Word;
function DvbFilterGetPacketSyncErrors: Word;
function DvbFilterGetDebug: Word;
procedure DvbFilterResetErrors;
procedure DvbFilterCreateTables;

implementation

const
  CDvbProgramNumberUndefined = $FFFF;
  CDvbProgramIndexUndefined = $FFFF;
  CDvbPidUndefined = $FFFF;

  CDvbStreamTypeVideo = $02;
  CDvbStreamTypeAudio = $03;
  CDvbStreamTypeAudio2 = $04;
  // $06 indicates a private stream and can be used for different purposes
  CDvbStreamTypeVideotext = $06; // Descriptor needs to be <CDvbTagTeletext>
  CDvbStreamTypeAc3 = $06; // Descriptor needs to be <CDvbTagAc3Descriptor>
  CDvbStreamTypeSubtitle = $06;
    // Descriptor needs to be <CDvbTagSubtitleDescriptor>
  CDvbStreamTypeUndefined = $FF;

  // PSI Table Id's
  CDvbPsiTableIdPat = $00; // PAT Program Association Table
  CDvbPsiTableIdCat = $01; // CAT Conditional Access Table
  CDvbPsiTableIdPmt = $02; // PMT Program Map Table
  CDvbPsiTableIdTsdt = $03; // TSDT Transport Stream Decription Table
  CDvbPsiTableIdNitA = $40; // NIT Network Information Table - actual network
  CDvbPsiTableIdNitO = $41; // NIT Network Information Table - other  network
  CDvbPsiTableIdSdtA = $42;
    // SDT Service Description Table - actual transport stream
  CDvbPsiTableIdSdtO = $46;
    // SDT Service Description Table - other  transport stream
  CDvbPsiTableIdBat = $4A; // BAT Bouquet Association Table
  CDvbPsiTableIdEitA = $4E;
    // EIT Event Information Table - actual transport stream - present/following
  CDvbPsiTableIdEitO = $4F;
    // EIT Event Information Table - other  transport stream - present/following
  CDvbPsiTableIdEitAS = $50;
    // EIT Event Information Table - actual transport stream - schedule (runs form $50..$5F)
  CDvbPsiTableIdEitOS = $60;
    // EIT Event Information Table - other  transport stream - schedule (runs form $60..$6F)
  CDvbPsiTableIdTdt = $70; // TDT Time Date Table
  CDvbPsiTableIdRst = $71; // RST Running Status Table
  CDvbPsiTableIdSt = $72; // ST  Stuffing Table
  CDvbPsiTableIdTot = $73; // TOT Time Offset Table
  CDvbPsiTableIdDit = $7E; // DIT Discontinuity Information Table
  CDvbPsiTableIdSit = $7F; // SIT Selection Information Table
  // Above $7F are used for other PID identification and are not table id's
  CDvbPsiTableIdPcr = $80; // PCR Program Clock Reference
  CDvbPsiTableIdUndefined = $FF; // Undefined

  // Descriptors
  CDvbTagVideo = $02;
  CDvbTagAudio = $03;
  CDvbTagCa = $09;
  CDvbTagLanguage = $0A;
  CDvbTagSatelliteDeliverySystem = $43;
  CDvbTagService = $48;
  CDvbTagShortEventDescriptor = $4D;
  CDvbTagExtendedEventDescriptor = $4E;
  CDvbTagStreamIdentifier = $52;
  CDvbTagTeletext = $56;
  CDvbTagSubtitleDescriptor = $59;
  CDvbTagAc3Descriptor = $6A;

  CDvbPacketSyncWord = $47; // DVB sync word

  // StreamId assignments
  CDvbProgramStreamMap = $BC;
  CDvbPrivateStream1 = $BD;
  CDvbPaddingStream = $BE;
  CDvbPrivateStream2 = $BF;
  CDvbEcmStream = $F0;
  CDvbEmmStream = $F1;
  CDvbDsmccStream = $F2;
  CDvbTypeEStream = $F8;
  CDvbProgramStreamDirectory = $FF;

  // 16 Bit CCITT standard polynomial x^16 + x^12 + x^5 + x^1 + x^0
  CDvbPolynomial16 = $1021;
  CDvbCrcInitial16 = $FFFF;
  // 32 Bit standard polynomial x^32 + x^26 + x^23 + x^22 + x^16 +
  //   x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x^1 + x^0
  CDvbPolynomial32 = $04C11DB7;
  CDvbCrcInitial32 = $FFFFFFFF;

  // Next table is for reference only. This is the generated data and should
  // be equal to <Crc32Table>.
  Crc32Table2: array[0..255] of Dword =
  ($00000000, $04C11DB7, $09823B6E, $0D4326D9, $130476DC, $17C56B6B,
    $1A864DB2, $1E475005, $2608EDB8, $22C9F00F, $2F8AD6D6, $2B4BCB61,
    $350C9B64, $31CD86D3, $3C8EA00A, $384FBDBD, $4C11DB70, $48D0C6C7,
    $4593E01E, $4152FDA9, $5F15ADAC, $5BD4B01B, $569796C2, $52568B75,
    $6A1936C8, $6ED82B7F, $639B0DA6, $675A1011, $791D4014, $7DDC5DA3,
    $709F7B7A, $745E66CD, $9823B6E0, $9CE2AB57, $91A18D8E, $95609039,
    $8B27C03C, $8FE6DD8B, $82A5FB52, $8664E6E5, $BE2B5B58, $BAEA46EF,
    $B7A96036, $B3687D81, $AD2F2D84, $A9EE3033, $A4AD16EA, $A06C0B5D,
    $D4326D90, $D0F37027, $DDB056FE, $D9714B49, $C7361B4C, $C3F706FB,
    $CEB42022, $CA753D95, $F23A8028, $F6FB9D9F, $FBB8BB46, $FF79A6F1,
    $E13EF6F4, $E5FFEB43, $E8BCCD9A, $EC7DD02D, $34867077, $30476DC0,
    $3D044B19, $39C556AE, $278206AB, $23431B1C, $2E003DC5, $2AC12072,
    $128E9DCF, $164F8078, $1B0CA6A1, $1FCDBB16, $018AEB13, $054BF6A4,
    $0808D07D, $0CC9CDCA, $7897AB07, $7C56B6B0, $71159069, $75D48DDE,
    $6B93DDDB, $6F52C06C, $6211E6B5, $66D0FB02, $5E9F46BF, $5A5E5B08,
    $571D7DD1, $53DC6066, $4D9B3063, $495A2DD4, $44190B0D, $40D816BA,
    $ACA5C697, $A864DB20, $A527FDF9, $A1E6E04E, $BFA1B04B, $BB60ADFC,
    $B6238B25, $B2E29692, $8AAD2B2F, $8E6C3698, $832F1041, $87EE0DF6,
    $99A95DF3, $9D684044, $902B669D, $94EA7B2A, $E0B41DE7, $E4750050,
    $E9362689, $EDF73B3E, $F3B06B3B, $F771768C, $FA325055, $FEF34DE2,
    $C6BCF05F, $C27DEDE8, $CF3ECB31, $CBFFD686, $D5B88683, $D1799B34,
    $DC3ABDED, $D8FBA05A, $690CE0EE, $6DCDFD59, $608EDB80, $644FC637,
    $7A089632, $7EC98B85, $738AAD5C, $774BB0EB, $4F040D56, $4BC510E1,
    $46863638, $42472B8F, $5C007B8A, $58C1663D, $558240E4, $51435D53,
    $251D3B9E, $21DC2629, $2C9F00F0, $285E1D47, $36194D42, $32D850F5,
    $3F9B762C, $3B5A6B9B, $0315D626, $07D4CB91, $0A97ED48, $0E56F0FF,
    $1011A0FA, $14D0BD4D, $19939B94, $1D528623, $F12F560E, $F5EE4BB9,
    $F8AD6D60, $FC6C70D7, $E22B20D2, $E6EA3D65, $EBA91BBC, $EF68060B,
    $D727BBB6, $D3E6A601, $DEA580D8, $DA649D6F, $C423CD6A, $C0E2D0DD,
    $CDA1F604, $C960EBB3, $BD3E8D7E, $B9FF90C9, $B4BCB610, $B07DABA7,
    $AE3AFBA2, $AAFBE615, $A7B8C0CC, $A379DD7B, $9B3660C6, $9FF77D71,
    $92B45BA8, $9675461F, $8832161A, $8CF30BAD, $81B02D74, $857130C3,
    $5D8A9099, $594B8D2E, $5408ABF7, $50C9B640, $4E8EE645, $4A4FFBF2,
    $470CDD2B, $43CDC09C, $7B827D21, $7F436096, $7200464F, $76C15BF8,
    $68860BFD, $6C47164A, $61043093, $65C52D24, $119B4BE9, $155A565E,
    $18197087, $1CD86D30, $029F3D35, $065E2082, $0B1D065B, $0FDC1BEC,
    $3793A651, $3352BBE6, $3E119D3F, $3AD08088, $2497D08D, $2056CD3A,
    $2D15EBE3, $29D4F654, $C5A92679, $C1683BCE, $CC2B1D17, $C8EA00A0,
    $D6AD50A5, $D26C4D12, $DF2F6BCB, $DBEE767C, $E3A1CBC1, $E760D676,
    $EA23F0AF, $EEE2ED18, $F0A5BD1D, $F464A0AA, $F9278673, $FDE69BC4,
    $89B8FD09, $8D79E0BE, $803AC667, $84FBDBD0, $9ABC8BD5, $9E7D9662,
    $933EB0BB, $97FFAD0C, $AFB010B1, $AB710D06, $A6322BDF, $A2F33668,
    $BCB4666D, $B8757BDA, $B5365D03, $B1F740B4);

type
  TDvbTransportPids = array[0..$1FFF] of Boolean;
  TDvbMaxPesPacket = array[0..$FFFF] of Byte;

  // Section: Gathered over multiple packets
  TDvbReceivingSection = class(TObject)
    FValid: Boolean; // True when valid (first part received)
    FPointerField: Word; // Bytes to discard (PointerField)
    FStreamData: PDvbSection; // Data of multiple packets
    FDataIndex: Word; // Index in data
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property Valid: Boolean read FValid write FValid;
    property PointerField: Word read FPointerField write FPointerField;
    property StreamData: PDvbSection read FStreamData write FStreamData;
    property DataIndex: Word read FDataIndex write FDataIndex;
  end;

  // Section: Gathered over multiple packets
  TDvbReceivingSections = class(TObject)
    FDvbReceivingSections: array[0..$1FFF] of TDvbReceivingSection;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetValid(Index: Word): Boolean;
    procedure SetValid(Index: Word; Value: Boolean);
    function GetPointerField(Index: Word): Word;
    procedure SetPointerField(Index: Word; Value: Word);
    function GetStreamData(Index: Word): PDvbSection;
    procedure SetStreamData(Index: Word; Value: PDvbSection);
    function GetDataIndex(Index: Word): Word;
    procedure SetDataIndex(Index: Word; Value: Word);
  public
    constructor Create;
    destructor Destroy; override;
    property Valid[Index: Word]: Boolean read GetValid write SetValid;
    property PointerField[Index: Word]: Word read GetPointerField write
      SetPointerField;
    property StreamData[Index: Word]: PDvbSection read GetStreamData write
      SetStreamData;
    property DataIndex[Index: Word]: Word read GetDataIndex write SetDataIndex;
  end;

  TDvbTypePid = class(TObject)
  private
    FPidType: Byte; // Type of Pids (typically table id setting)
    FProgramNumber: Word; // Associated program number
    FProgramIndex: Word; // Index in PMT table array
    FStreamType: Byte; // Associated stream type
    FEsLength: Word; // Associated length of descriptor
    FEsDescriptor: PDvbSection; // Associated descriptor
    FPacketFilter: TDvbPacketFilter; // Function which processes the packet
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property PidType: Byte read FPidType write FPidType;
    property ProgramNumber: Word read FProgramNumber write FProgramNumber;
    property ProgramIndex: Word read FProgramIndex write FProgramIndex;
    property StreamType: Byte read FStreamType write FStreamType;
    property EsLength: Word read FEsLength write FEsLength;
    property EsDescriptor: PDvbSection read FEsDescriptor write FEsDescriptor;
    property PacketFilter: TDvbPacketFilter read FPacketFilter write
      FPacketFilter;
  end;

  TDvbTypePids = class(TObject)
  private
    FDvbTypePids: array[0..$1FFF] of TDvbTypePid;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetPidType(Index: Word): Byte;
    procedure SetPidType(Index: Word; Value: Byte);
    function GetProgramNumber(Index: Word): Word;
    procedure SetProgramNumber(Index: Word; Value: Word);
    function GetProgramIndex(Index: Word): Word;
    procedure SetProgramIndex(Index: Word; Value: Word);
    function GetStreamType(Index: Word): Byte;
    procedure SetStreamType(Index: Word; Value: Byte);
    function GetEsLength(Index: Word): Word;
    procedure SetEsLength(Index: Word; Value: Word);
    function GetEsDescriptor(Index: Word): PDvbSection;
    procedure SetEsDescriptor(Index: Word; Value: PDvbSection);
    function GetPacketFilter(Index: Word): TDvbPacketFilter;
    procedure SetPacketFilter(Index: Word; Value: TDvbPacketFilter);
  public
    constructor Create;
    destructor Destroy; override;
    property PidType[Index: Word]: Byte read GetPidType write SetPidType;
    property ProgramNumber[Index: Word]: Word read GetProgramNumber write
      SetProgramNumber;
    property ProgramIndex[Index: Word]: Word read GetProgramIndex write
      SetProgramIndex;
    property StreamType[Index: Word]: Byte read GetStreamType write
      SetStreamType;
    property EsLength[Index: Word]: Word read GetEsLength write SetEsLength;
    property EsDescriptor[Index: Word]: PDvbSection read GetEsDescriptor write
      SetEsDescriptor;
    property PacketFilter[Index: Word]: TDvbPacketFilter read GetPacketFilter
      write SetPacketFilter;
  end;

  TDvbEitEvent = class(TObject)
  private
    FStartDate: Word; // Start date (MJD format)
    FStartTime: Dword; // Start time (BCD)
    FDuration: Dword; // Duration (BCD)
    FRunningStatus: Byte; // Status
    FFreeCaMode: Boolean; // True indicates non-scrambled
    FDescriptorLength: Word; // Length of descriptors
    FStreamDescriptor: PDvbSection; // Descriptors
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property StartDate: Word read FStartDate write FStartDate;
    property StartTime: DWord read FStartTime write FStartTime;
    property Duration: DWord read FDuration write FDuration;
    property RunningStatus: Byte read FRunningStatus write FRunningStatus;
    property FreeCaMode: Boolean read FFreeCaMode write FFreeCaMode;
    property DescriptorLength: Word read FDescriptorLength write
      FDescriptorLength;
    property StreamDescriptor: PDvbSection read FStreamDescriptor write
      FStreamDescriptor;
  end;

  TDvbEitEvents = class(TObject)
  private
    FVersionNumber: Byte; // Version number
    FLastSection: Byte; // Last section
    FLastTableId: Byte; // Last table id
    FEvent: Byte; // Event index
    FDvbEitEvents: array[0..$FF] of TDvbEitEvent; // Individual events
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetStartDate(Index: Word): Word;
    procedure SetStartDate(Index: Word; Value: Word);
    function GetStartTime(Index: Word): DWord;
    procedure SetStartTime(Index: Word; Value: DWord);
    function GetDuration(Index: Word): DWord;
    procedure SetDuration(Index: Word; Value: DWord);
    function GetRunningStatus(Index: Word): Byte;
    procedure SetRunningStatus(Index: Word; Value: Byte);
    function GetFreeCaMode(Index: Word): Boolean;
    procedure SetFreeCaMode(Index: Word; Value: Boolean);
    function GetDescriptorLength(Index: Word): Word;
    procedure SetDescriptorLength(Index: Word; Value: Word);
    function GetStreamDescriptor(Index: Word): PDvbSection;
    procedure SetStreamDescriptor(Index: Word; Value: PDvbSection);
  public
    constructor Create;
    destructor Destroy; override;
    property VersionNumber: Byte read FVersionNumber write FVersionNumber;
    property LastSection: Byte read FLastSection write FLastSection;
    property LastTableId: Byte read FLastTableId write FLastTableId;
    property Event: Byte read FEvent write FEvent;
    property StartDate[Index: Word]: Word read GetStartDate write SetStartDate;
    property StartTime[Index: Word]: DWord read GetStartTime write SetStartTime;
    property Duration[Index: Word]: DWord read GetDuration write SetDuration;
    property RunningStatus[Index: Word]: Byte read GetRunningStatus write
      SetRunningStatus;
    property FreeCaMode[Index: Word]: Boolean read GetFreeCaMode write
      SetFreeCaMode;
    property DescriptorLength[Index: Word]: Word read GetDescriptorLength write
      SetDescriptorLength;
    property StreamDescriptor[Index: Word]: PDvbSection read GetStreamDescriptor
      write SetStreamDescriptor;
  end;

  // Program Map table streams
  TDvbProgramMapStream = class(TObject)
  private
    FStreamType: Byte; // Type of program element
    FElementaryPid: Word; // PID for this program element
    FStreamInfoLength: Word; // Length of descriptor
    FStreamDescriptor: PDvbSection; // Descriptor for program element
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property StreamType: Byte read FStreamType write FStreamType;
    property ElementaryPid: Word read FElementaryPid write FElementaryPid;
    property StreamInfoLength: Word read FStreamInfoLength write
      FStreamInfoLength;
    property StreamDescriptor: PDvbSection read FStreamDescriptor write
      FStreamDescriptor;
  end;

  TDvbProgramMapStreams = class(TObject)
  private
    FDvbProgramMapStreams: array[0..199] of TDvbProgramMapStream;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetStreamType(Index: Word): Byte;
    procedure SetStreamType(Index: Word; Value: Byte);
    function GetElementaryPid(Index: Word): Word;
    procedure SetElementaryPid(Index: Word; Value: Word);
    function GetStreamInfoLength(Index: Word): Word;
    procedure SetStreamInfoLength(Index: Word; Value: Word);
    function GetStreamDescriptor(Index: Word): PDvbSection;
    procedure SetStreamDescriptor(Index: Word; Value: PDvbSection);
  public
    constructor Create;
    destructor Destroy; override;
    property StreamType[Index: Word]: Byte read GetStreamType write
      SetStreamType;
    property ElementaryPid[Index: Word]: Word read GetElementaryPid write
      SetElementaryPid;
    property StreamInfoLength[Index: Word]: Word read GetStreamInfoLength write
      SetStreamInfoLength;
    property StreamDescriptor[Index: Word]: PDvbSection read GetStreamDescriptor
      write SetStreamDescriptor;
  end;

  // Program Map table streams
  TDvbProgramMapProgram = class(TObject)
  private
    FPid: Word; // Associated PID
    FProgramNumber: Word; // Program number identifier
    FVersionNumber: Byte; // Version number
    FPcrPid: Word; // PCR PID
    FInfoLength: Word; // Length of <InfoDescriptor>
    FInfoDescriptor: PDvbSection; // Descriptor of program
    FSdtInfoLength: Word; // Length of <SdtDescriptor>
    FSdtDescriptor: PDvbSection; // Descriptor of service
    FNumberOfStreams: Word; // Number of streams
    FStreams: TDvbProgramMapStreams; // Program map streams
    FEvents: array[CDvbPsiTableIdEitA..CDvbPsiTableIdEitAS + $0F] of
      TDvbEitEvents; // Event information
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetEvents(Index: Word): TDvbEitEvents;
    procedure SetEvents(Index: Word; Value: TDvbEitEvents);
  public
    constructor Create;
    destructor Destroy; override;
    property Pid: Word read FPid write FPid;
    property ProgramNumber: Word read FProgramNumber write FProgramNumber;
    property VersionNumber: Byte read FVersionNumber write FVersionNumber;
    property PcrPid: Word read FPcrPid write FPcrPid;
    property InfoLength: Word read FInfoLength write FInfoLength;
    property InfoDescriptor: PDvbSection read FInfoDescriptor write
      FInfoDescriptor;
    property SdtInfoLength: Word read FSdtInfoLength write FSdtInfoLength;
    property SdtDescriptor: PDvbSection read FSdtDescriptor write
      FSdtDescriptor;
    property NumberOfStreams: Word read FNumberOfStreams write FNumberOfStreams;
    property Streams: TDvbProgramMapStreams read FStreams write FStreams;
    property Events[Index: Word]: TDvbEitEvents read GetEvents write SetEvents;
  end;

  // Program Map table streams
  TDvbProgramMapPrograms = class(TObject)
  private
    FDvbProgramMapPrograms: array[0..$FF] of TDvbProgramMapProgram;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetPid(Index: Word): Word;
    procedure SetPid(Index: Word; Value: Word);
    function GetProgramNumber(Index: Word): Word;
    procedure SetProgramNumber(Index: Word; Value: Word);
    function GetVersionNumber(Index: Word): Byte;
    procedure SetVersionNumber(Index: Word; Value: Byte);
    function GetPcrPid(Index: Word): Word;
    procedure SetPcrPid(Index: Word; Value: Word);
    function GetInfoLength(Index: Word): Word;
    procedure SetInfoLength(Index: Word; Value: Word);
    function GetInfoDescriptor(Index: Word): PDvbSection;
    procedure SetInfoDescriptor(Index: Word; Value: PDvbSection);
    function GetSdtInfoLength(Index: Word): Word;
    procedure SetSdtInfoLength(Index: Word; Value: Word);
    function GetSdtDescriptor(Index: Word): PDvbSection;
    procedure SetSdtDescriptor(Index: Word; Value: PDvbSection);
    function GetNumberOfStreams(Index: Word): Word;
    procedure SetNumberOfStreams(Index: Word; Value: Word);
    function GetStreams(Index: Word): TDvbProgramMapStreams;
    procedure SetStreams(Index: Word; Value: TDvbProgramMapStreams);
    function GetEvents(Index: Word; Index2: Word): TDvbEitEvents;
    procedure SetEvents(Index: Word; Index2: Word; Value: TDvbEitEvents);
  public
    constructor Create;
    destructor Destroy; override;
    property Pid[Index: Word]: Word read GetPid write SetPid;
    property ProgramNumber[Index: Word]: Word read GetProgramNumber write
      SetProgramNumber;
    property VersionNumber[Index: Word]: Byte read GetVersionNumber write
      SetVersionNumber;
    property PcrPid[Index: Word]: Word read GetPcrPid write SetPcrPid;
    property InfoLength[Index: Word]: Word read GetInfoLength write
      SetInfoLength;
    property InfoDescriptor[Index: Word]: PDvbSection read GetInfoDescriptor
      write SetInfoDescriptor;
    property SdtInfoLength[Index: Word]: Word read GetSdtInfoLength write
      SetSdtInfoLength;
    property SdtDescriptor[Index: Word]: PDvbSection read GetSdtDescriptor write
      SetSdtDescriptor;
    property NumberOfStreams[Index: Word]: Word read GetNumberOfStreams write
      SetNumberOfStreams;
    property Streams[Index: Word]: TDvbProgramMapStreams read GetStreams write
      SetStreams;
    property Events[Index: Word; Index2: Word]: TDvbEitEvents read GetEvents
      write SetEvents;
  end;

  // Conditional Access descriptor
  TDvbCaDescriptor = class(TObject)
  private
    FPid: Word; // Associated PID
    FCaSystemId: Word; // System identifier
    FCaPid: Word; // Ca pid
    FPrivateDataLength: Word; // Length of private data
    FPrivateData: PDvbSection;
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property Pid: Word read FPid write FPid;
    property CaSystemId: Word read FCaSystemId write FCaSystemId;
    property CaPid: Word read FCaPid write FCaPid;
    property PrivateDataLength: Word read FPrivateDataLength write
      FPrivateDataLength;
    property PrivateData: PDvbSection read FPrivateData write FPrivateData;
  end;

  TDvbCaDescriptors = class(TObject)
  private
    FDvbCaDescriptors: array[0..$FF] of TDvbCaDescriptor;
    FVersionNumber: Byte; // Version
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetPid(Index: Word): Word;
    procedure SetPid(Index: Word; Value: Word);
    function GetCaSystemId(Index: Word): Word;
    procedure SetCaSystemId(Index: Word; Value: Word);
    function GetCaPid(Index: Word): Word;
    procedure SetCaPid(Index: Word; Value: Word);
    function GetPrivateDataLength(Index: Word): Word;
    procedure SetPrivateDataLength(Index: Word; Value: Word);
    function GetPrivateData(Index: Word): PDvbSection;
    procedure SetPrivateData(Index: Word; Value: PDvbSection);
  public
    constructor Create;
    destructor Destroy; override;
    property VersionNumber: Byte read FVersionNumber write FVersionNumber;
    property Pid[Index: Word]: Word read GetPid write SetPid;
    property CaSystemId[Index: Word]: Word read GetCaSystemId write
      SetCaSystemId;
    property CaPid[Index: Word]: Word read getCaPid write SetCaPid;
    property PrivateDataLength[Index: Word]: Word read GetPrivateDataLength write
      SetPrivateDataLength;
    property PrivateData[Index: Word]: PDvbSection read GetPrivateData write
      SetPrivateData;
  end;

  // Network streams
  TDvbNetworkStream = class(TObject)
  private
    FStreamInfoLength: Word; // Length of descriptor
    FStreamDescriptor: PDvbSection; // Descriptor for program element
  protected
  public
    constructor Create;
    destructor Destroy; override;
    property StreamInfoLength: Word read FStreamInfoLength write
      FStreamInfoLength;
    property StreamDescriptor: PDvbSection read FStreamDescriptor write
      FStreamDescriptor;
  end;

  TDvbNetworkStreams = class(TObject)
  private
    FVersionNumber: Byte; // Version
    FNetworkId: Word; // Network identifier
    FInfoLength: Word; // Length of <InfoDescriptor>
    FInfoDescriptor: PDvbSection; // Descriptor of program
    FNumberOfStreams: Word; // Number of streams
    FDvbNetworkStreams: array[0..199] of TDvbNetworkStream;
    procedure CreateIfNotAssigned(Index: Word);
  protected
    function GetStreamInfoLength(Index: Word): Word;
    procedure SetStreamInfoLength(Index: Word; Value: Word);
    function GetStreamDescriptor(Index: Word): PDvbSection;
    procedure SetStreamDescriptor(Index: Word; Value: PDvbSection);
  public
    constructor Create;
    destructor Destroy; override;
    property VersionNumber: Byte read FVersionNumber write FVersionNumber;
    property NetworkId: Word read FNetworkId write FNetworkId;
    property InfoLength: Word read FInfoLength write FInfoLength;
    property InfoDescriptor: PDvbSection read FInfoDescriptor write
      FInfoDescriptor;
    property NumberOfStreams: Word read FNumberOfStreams write FNumberOfStreams;
    property StreamInfoLength[Index: Word]: Word read GetStreamInfoLength write
      SetStreamInfoLength;
    property StreamDescriptor[Index: Word]: PDvbSection read GetStreamDescriptor
      write SetStreamDescriptor;
  end;

function DvbResetValueCrc16: Word; forward;
function DvbResetValueCrc32: Dword; forward;
function DvbNewValueCrc16(Checksum: Word; Data: Byte): Word; forward;
function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword; forward;

function DvbDecodeProgramSpecificInformationData(StreamPacketData:
  PDvbTransportPacket): Boolean; stdcall; forward;

var
  Debug: Word; // Generic use for debug purposes
  PacketError: Word;
  PacketSyncErrors: Word;
  ResetLock: TCriticalSection; // Prevents processing during 'reset' condition
  PidList: TDvbTypePids; // Type of PIDs

  PatPrograms: Word; // Number of programs detected in PAT
  PatVersion: Byte; // Current version number of PAT

  Crc16Table: array[0..255] of Word;
  Crc32Table: array[0..255] of Dword;

  TransportPids: TDvbTransportPids;
  TransportPidsCounter: Word; // Count the number of different transport PID's

  SectionReceiving: TDvbReceivingSections;
    // Section build up from several packets

  ProgramToTableReference: array[0..$FFFF] of Word; // Cross reference
  PmtPrograms: TDvbProgramMapPrograms; // PMT
  PmtProgramIndex: Word; // Number of programs defined
  PatCallback: TDvbCallback; // Callback for new PAT
  CatCallback: TDvbCallback; // Callback for new CAT
  PmtCallback: TDvbCallback; // Callback for new PMT
  EventCallback: TDvbCallback; // Callback for new event
  PidListCallback: array[0..$1FFF] of TDvbCallback; // Callbacks for PIDs
  // These are NOT part of PidList because the callback usually stays active throughout the program
  // whereas the PidList is 'cleared' when a trasnponder change takes place
  CaDescriptors: TDvbCaDescriptors; // CA  (EMM) system wide
  CaDescriptorQuantity: Word; // Number of descriptors

  PsiFilters: array[0..$FF] of TDvbSectionFilter; // Filters for PSI sections
  // We use $FF because we don't have to do any checks on
  // a valid index this way
  PsiTableId: array[0..$FF] of Boolean;
    // True if specific table identification used

  Network: TDvbNetworkStreams; // Network information

  TimeInfo: string;
  Year: Word;
  Month: Word;
  Day: Word;

  {------------------------------------------------------------------------------
                                   TDvbEventSection
   ------------------------------------------------------------------------------}
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: Constructor
    Notes   :
   ------------------------------------------------------------------------------}

constructor TDvbEventSection.Create;
begin
  inherited Create;
  FStartTime := 0;
  FDuration := 0;
  FEventText := TList.Create;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbEventSection.Destroy;
var
  Index: Word;
  ARecord: PAnsiString;
begin
  if FEventText.Count > 0 then
    for Index := 0 to FEventText.Count - 1 do
    begin
      // Note: We must make sure we use the same type as when it was allocated
      // otherwise not all memory will be released
      ARecord := FEventText.Items[Index];
      Dispose(ARecord);
    end;
  FreeAndNil(FEventText);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbEventSection
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbEventSections
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbEventSections.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbEventSections) to High(FDvbEventSections) do
    FDvbEventSections[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbEventSections.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbEventSections) to High(FDvbEventSections) do
    if Assigned(FDvbEventSections[Loop]) then
      FreeAndNil(FDvbEventSections[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEventSections.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbEventSections) then
    Exit;
  if not Assigned(FDvbEventSections[Index]) then
    FDvbEventSections[Index] := TDvbEventSection.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
  Returns : <Result>  Start time

  Descript: Get start time
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEventSections.GetStartTime(Index: Word): TDateTime;
begin
  Result := 0;
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbEventSections[Index]) then
    Result := FDvbEventSections[Index].FStartTime;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
            <Value>   Start time
  Returns : -

  Descript: Set start time
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEventSections.SetStartTime(Index: Word; Value: TDateTime);
begin
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEventSections[Index].FStartTime := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
  Returns : <Result>  Duration

  Descript: Get duration
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEventSections.GetDuration(Index: Word): TDateTime;
begin
  Result := 0;
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbEventSections[Index]) then
    Result := FDvbEventSections[Index].FDuration;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
            <Value>   Duration
  Returns : -

  Descript: Set duration
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEventSections.SetDuration(Index: Word; Value: TDateTime);
begin
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEventSections[Index].FDuration := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
  Returns : <Result>  Event text list

  Descript: Get event text list
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEventSections.GetEventText(Index: Word): TList;
begin
  Result := nil;
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbEventSections[Index]) then
    Result := FDvbEventSections[Index].FEventText;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Event id
            <Value>   Event text list
  Returns : -

  Descript: Set event text list
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEventSections.SetEventText(Index: Word; Value: TList);
begin
  if Index > High(FDvbEventSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEventSections[Index].FEventText := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbEventSections
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbTypePid
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbTypePid.Create;
begin
  inherited Create;
  FPidType := CDvbPsiTableIdUndefined;
  FProgramNumber := CDvbProgramNumberUndefined;
  FProgramIndex := CDvbProgramIndexUndefined;
  FStreamType := CDvbStreamTypeUndefined;
  FEsLength := 0;
  try
    New(FEsDescriptor);
  except
    FEsDescriptor := nil;
  end;
  FPacketFilter := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbTypePid.Destroy;
begin
  if Assigned(FEsDescriptor) then
    Dispose(FEsDescriptor);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbTypePid
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbTypePids
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbTypePids.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbTypePids) to High(FDvbTypePids) do
    FDvbTypePids[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbTypePids.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbTypePids) to High(FDvbTypePids) do
    if Assigned(FDvbTypePids[Loop]) then
      FreeAndNil(FDvbTypePids[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  if not Assigned(FDvbTypePids[Index]) then
    FDvbTypePids[Index] := TDvbTypePid.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Type of PID
                      <CDvbPsiTableIdUndefined> if undefined

  Descript: Get PID type
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetPidType(Index: Word): Byte;
begin
  Result := CDvbPsiTableIdUndefined;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FPidType;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Type to set
  Returns : -

  Descript: Set PID type
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetPidType(Index: Word; Value: Byte);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FPidType := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Program number
                      <CDvbProgramNumberUndefined> if undefined

  Descript: Get program number
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetProgramNumber(Index: Word): Word;
begin
  Result := CDvbProgramNumberUndefined;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FProgramNumber;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Program number to set
  Returns : -

  Descript: Set program number
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetProgramNumber(Index: Word; Value: Word);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FProgramNumber := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Program index
                      <CDvbProgramIndexUndefined> if undefined

  Descript: Get program index
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetProgramIndex(Index: Word): Word;
begin
  Result := CDvbProgramIndexUndefined;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FProgramIndex;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Program index to set
  Returns : -

  Descript: Set program index
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetProgramIndex(Index: Word; Value: Word);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FProgramIndex := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Stream type
                      <CDvbStreamTypeUndefined> if undefined

  Descript: Get stream type
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetStreamType(Index: Word): Byte;
begin
  Result := CDvbStreamTypeUndefined;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FStreamType;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Stream type to set
  Returns : -

  Descript: Set stream type
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetStreamType(Index: Word; Value: Byte);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FStreamType := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Length

  Descript: Get elementary stream descriptor length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetEsLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FEsLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Length to set
  Returns : -

  Descript: Set elementary stream descriptor length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetEsLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FEsLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Descriptor pointer

  Descript: Get elementary stream descriptor
  Notes   : Aslo creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetEsDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FEsDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to descriptor
  Returns : -

  Descript: Set elementary stream descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetEsDescriptor(Index: Word; Value: PDvbSection);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FEsDescriptor := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Callback pointer

  Descript: Get packet filter callback
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbTypePids.GetPacketFilter(Index: Word): TDvbPacketFilter;
begin
  Result := nil;
  if Index > High(FDvbTypePids) then
    Exit;
  if Assigned(FDvbTypePids[Index]) then
    Result := FDvbTypePids[Index].FPacketFilter;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to callback function
  Returns : -

  Descript: Set packet filter callback
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbTypePids.SetPacketFilter(Index: Word; Value: TDvbPacketFilter);
begin
  if Index > High(FDvbTypePids) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbTypePids[Index].FPacketFilter := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbTypePids
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbReceivingSection
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbReceivingSection.Create;
begin
  inherited Create;
  FValid := False;
  FPointerField := 0;
  try
    New(FStreamData);
  except
    FStreamData := nil;
  end;
  FDataIndex := 0;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbReceivingSection.Destroy;
begin
  if Assigned(FStreamData) then
    Dispose(FStreamData);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbReceivingSection
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbReceivingSections
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbReceivingSections.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbReceivingSections) to High(FDvbReceivingSections) do
    FDvbReceivingSections[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbReceivingSections.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbReceivingSections) to High(FDvbReceivingSections) do
    if Assigned(FDvbReceivingSections[Loop]) then
      FreeAndNil(FDvbReceivingSections[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  if not Assigned(FDvbReceivingSections[Index]) then
    FDvbReceivingSections[Index] := TDvbReceivingSection.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Valid flag
                      <FALSE> if invalid PID id

  Descript: Get valid flag
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetValid(Index: Word): Boolean;
begin
  Result := False;
  if Index > High(FDvbReceivingSections) then
    Exit;
  if Assigned(FDvbReceivingSections[Index]) then
    Result := FDvbReceivingSections[Index].FValid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Flag to set
  Returns : -

  Descript: Set valid flag
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetValid(Index: Word; Value: Boolean);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbReceivingSections[Index].FValid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer field

  Descript: Get pointer field
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetPointerField(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbReceivingSections) then
    Exit;
  if Assigned(FDvbReceivingSections[Index]) then
    Result := FDvbReceivingSections[Index].FPointerField;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Type to set
  Returns : -

  Descript: Set pointer field
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetPointerField(Index: Word; Value: Word);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbReceivingSections[Index].FPointerField := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer to stream data
                      <nil> if invalid PID id

  Descript: Get stream data
  Notes   : Also creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetStreamData(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbReceivingSections) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbReceivingSections[Index]) then
    Result := FDvbReceivingSections[Index].FStreamData;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to stream data
  Returns : -

  Descript: Set stream data
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetStreamData(Index: Word; Value: PDvbSection);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbReceivingSections[Index].FStreamData := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer to stream data

  Descript: Get data index
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetDataIndex(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbReceivingSections) then
    Exit;
  if Assigned(FDvbReceivingSections[Index]) then
    Result := FDvbReceivingSections[Index].FDataIndex;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to stream data
  Returns : -

  Descript: Set data index
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetDataIndex(Index: Word; Value: Word);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbReceivingSections[Index].FDataIndex := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbReceivingSections
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbEitEvent
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbEitEvent.Create;
begin
  inherited Create;
  FStartDate := 0;
  FStartTime := 0;
  FDuration := 0;
  FRunningStatus := 0;
  FFreeCaMode := False;
  FDescriptorLength := 0;
  try
    New(FStreamDescriptor);
  except
    FStreamDescriptor := nil;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbEitEvent.Destroy;
begin
  if Assigned(FStreamDescriptor) then
    Dispose(FStreamDescriptor);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbEitEvent
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbEitEvents
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbEitEvents.Create;
var
  Loop: Integer;
begin
  inherited Create;
  FVersionNumber := $FF;
  FLastSection := $FF;
  FLastTableId := $FF;
  FEvent := 0;
  for Loop := Low(FDvbEitEvents) to High(FDvbEitEvents) do
    FDvbEitEvents[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbEitEvents.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbEitEvents) to High(FDvbEitEvents) do
    if Assigned(FDvbEitEvents[Loop]) then
      FreeAndNil(FDvbEitEvents[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  if not Assigned(FDvbEitEvents[Index]) then
    FDvbEitEvents[Index] := TDvbEitEvent.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Start date

  Descript: Get start date
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetStartDate(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FStartDate;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Start date
  Returns : -

  Descript: Set start date
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetStartDate(Index: Word; Value: Word);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FStartDate := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Start time

  Descript: Get start time
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetStartTime(Index: Word): DWord;
begin
  Result := 0;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FStartTime;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Start time
  Returns : -

  Descript: Set start time
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetStartTime(Index: Word; Value: DWord);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FStartTime := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Duration

  Descript: Get duration
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetDuration(Index: Word): DWord;
begin
  Result := 0;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FDuration;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Duration
  Returns : -

  Descript: Set duration
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetDuration(Index: Word; Value: DWord);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FDuration := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Running status

  Descript: Get running status
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetRunningStatus(Index: Word): Byte;
begin
  Result := 0;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FRunningStatus;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Running status
  Returns : -

  Descript: Set running status
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetRunningStatus(Index: Word; Value: Byte);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FRunningStatus := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Free CA mode

  Descript: Get free CA mode
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetFreeCaMode(Index: Word): Boolean;
begin
  Result := False;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FFreeCaMode;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Free CA mode
  Returns : -

  Descript: Set free CA mode
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetFreeCaMode(Index: Word; Value: Boolean);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FFreeCaMode := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Descriptor length

  Descript: Get descriptor length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetDescriptorLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbEitEvents) then
    Exit;
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FDescriptorLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Descriptor length
  Returns : -

  Descript: Set descriptor length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetDescriptorLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FDescriptorLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
  Returns : <Result>  Stream descriptor pointer

  Descript: Get stream descriptor
  Notes   : Also creates it if not assigned
 ------------------------------------------------------------------------------}

function TDvbEitEvents.GetStreamDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbEitEvents[Index]) then
    Result := FDvbEitEvents[Index].FStreamDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Eit id
            <Value>   Stream descriptor pointer
  Returns : -

  Descript: Set stream descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbEitEvents.SetStreamDescriptor(Index: Word; Value: PDvbSection);
begin
  if Index > High(FDvbEitEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbEitEvents[Index].FStreamDescriptor := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbEitEvents
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbProgramMapStream
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbProgramMapStream.Create;
begin
  inherited Create;
  FStreamType := CDvbStreamTypeUndefined;
  FElementaryPid := CDvbPidUndefined;
  FStreamInfoLength := 0;
  try
    New(FStreamDescriptor);
  except
    FStreamDescriptor := nil;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbProgramMapStream.Destroy;
begin
  if Assigned(FStreamDescriptor) then
    Dispose(FStreamDescriptor);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbProgramMapStream
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbProgramMapStreams
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbProgramMapStreams.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbProgramMapStreams) to High(FDvbProgramMapStreams) do
    FDvbProgramMapStreams[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbProgramMapStreams.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbProgramMapStreams) to High(FDvbProgramMapStreams) do
    if Assigned(FDvbProgramMapStreams[Loop]) then
      FreeAndNil(FDvbProgramMapStreams[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapStreams.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  if not Assigned(FDvbProgramMapStreams[Index]) then
    FDvbProgramMapStreams[Index] := TDvbProgramMapStream.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Stream type

  Descript: Get stream type
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapStreams.GetStreamType(Index: Word): Byte;
begin
  Result := 0;
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  if Assigned(FDvbProgramMapStreams[Index]) then
    Result := FDvbProgramMapStreams[Index].FStreamType;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Stream type
  Returns : -

  Descript: Set stream type
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapStreams.SetStreamType(Index: Word; Value: Byte);
begin
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapStreams[Index].FStreamType := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Elementary PID

  Descript: Get elementary PID
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapStreams.GetElementaryPid(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  if Assigned(FDvbProgramMapStreams[Index]) then
    Result := FDvbProgramMapStreams[Index].FElementaryPid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Elementary PID
  Returns : -

  Descript: Set elementary PID
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapStreams.SetElementaryPid(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapStreams[Index].FElementaryPid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Stream info length

  Descript: Get stream info length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapStreams.GetStreamInfoLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  if Assigned(FDvbProgramMapStreams[Index]) then
    Result := FDvbProgramMapStreams[Index].FStreamInfoLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Stream info length
  Returns : -

  Descript: Set stream info length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapStreams.SetStreamInfoLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapStreams[Index].FStreamInfoLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Stream descriptor pointer

  Descript: Get stream descriptor
  Notes   : Also create sit if not assigned
 ------------------------------------------------------------------------------}

function TDvbProgramMapStreams.GetStreamDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbProgramMapStreams[Index]) then
    Result := FDvbProgramMapStreams[Index].FStreamDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Stream descriptor pointer
  Returns : -

  Descript: Set stream descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapStreams.SetStreamDescriptor(Index: Word; Value:
  PDvbSection);
begin
  if Index > High(FDvbProgramMapStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapStreams[Index].FStreamDescriptor := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbProgramMapStreams
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbProgramMapProgram
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbProgramMapProgram.Create;
var
  Loop: Integer;
begin
  inherited Create;
  FPid := CDvbPidUndefined;
  FProgramNumber := CDvbProgramNumberUndefined;
  FVersionNumber := $FF;
  FPcrPid := CDvbPidUndefined;
  FInfoLength := 0;
  try
    New(FInfoDescriptor);
  except
    FInfoDescriptor := nil;
  end;
  FSdtInfoLength := 0;
  try
    New(FSdtDescriptor);
  except
    FSdtDescriptor := nil;
  end;
  FNumberOfStreams := 0;
  FStreams := TDvbProgramMapStreams.Create;
  for Loop := Low(FEvents) to High(FEvents) do
    FEvents[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbProgramMapProgram.Destroy;
var
  Loop: Integer;
begin
  if Assigned(FInfoDescriptor) then
    Dispose(FInfoDescriptor);
  if Assigned(FSdtDescriptor) then
    Dispose(FSdtDescriptor);
  FreeAndNil(FStreams);
  for Loop := Low(FEvents) to High(FEvents) do
    if Assigned(FEvents[Loop]) then
      FreeAndNil(FEvents[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapProgram.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FEvents) then
    Exit;
  if not Assigned(FEvents[Index]) then
    FEvents[Index] := TDvbEitEvents.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Events pointer

  Descript: Get events
  Notes   : Also creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbProgramMapProgram.GetEvents(Index: Word): TDvbEitEvents;
begin
  Result := nil;
  if Index > High(FEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FEvents[Index]) then
    Result := FEvents[Index];
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Events pointer
  Returns : -

  Descript: Set events
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapProgram.SetEvents(Index: Word; Value: TDvbEitEvents);
begin
  if Index > High(FEvents) then
    Exit;
  CreateIfNotAssigned(Index);
  FEvents[Index] := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbProgramMapProgram
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbProgramMapPrograms
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbProgramMapPrograms.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbProgramMapPrograms) to High(FDvbProgramMapPrograms) do
    FDvbProgramMapPrograms[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbProgramMapPrograms.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbProgramMapPrograms) to High(FDvbProgramMapPrograms) do
    if Assigned(FDvbProgramMapPrograms[Loop]) then
      FreeAndNil(FDvbProgramMapPrograms[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if not Assigned(FDvbProgramMapPrograms[Index]) then
    FDvbProgramMapPrograms[Index] := TDvbProgramMapProgram.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Pid id

  Descript: Get PID
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetPid(Index: Word): Word;
begin
  Result := CDvbPidUndefined;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FPid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Pid id
  Returns : -

  Descript: Set PID
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetPid(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FPid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Program number

  Descript: Get program number
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetProgramNumber(Index: Word): Word;
begin
  Result := CDvbProgramNumberUndefined;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FProgramNumber;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Program number
  Returns : -

  Descript: Set program number
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetProgramNumber(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FProgramNumber := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Version number

  Descript: Get version number
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetVersionNumber(Index: Word): Byte;
begin
  Result := $FF;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FVersionNumber;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Version number
  Returns : -

  Descript: Set version number
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetVersionNumber(Index: Word; Value: Byte);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FVersionNumber := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Pcr pid id

  Descript: Get PCR PID
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetPcrPid(Index: Word): Word;
begin
  Result := CDvbPidUndefined;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FPcrPid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Pcr pid id
  Returns : -

  Descript: Set PCR PID
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetPcrPid(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FPcrPid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Info length

  Descript: Get info length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetInfoLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FInfoLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Info length
  Returns : -

  Descript: Set info length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetInfoLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FInfoLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Info descriptor pointer

  Descript: Get info descriptor
  Notes   : Also creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetInfoDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FInfoDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Info descriptor pointer
  Returns : -

  Descript: Set info descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetInfoDescriptor(Index: Word; Value:
  PDvbSection);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FInfoDescriptor := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Sdt length

  Descript: Get SDT length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetSdtInfoLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FSdtInfoLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Sdt length
  Returns : -

  Descript: Set SDT length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetSdtInfoLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FSdtInfoLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Sdt descriptor pointer

  Descript: Get SDT descriptor
  Notes   : Also creates it if not assigned
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetSdtDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FSdtDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Sdt descriptor pointer
  Returns : -

  Descript: Set SDT descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetSdtDescriptor(Index: Word; Value:
  PDvbSection);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FSdtDescriptor := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Number of streams

  Descript: Get number of streams
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetNumberOfStreams(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FNumberOfStreams;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Number of streams
  Returns : -

  Descript: Set number of streams
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetNumberOfStreams(Index: Word; Value: Word);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FNumberOfStreams := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Streams pointer

  Descript: Get streams
  Notes   : Also creates it if not yet available!
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetStreams(Index: Word): TDvbProgramMapStreams;
begin
  Result := nil;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].FStreams;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Streams pointer
  Returns : -

  Descript: Set streams
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetStreams(Index: Word; Value:
  TDvbProgramMapStreams);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].FStreams := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Index2>  Event id
  Returns : <Result>  Event stream pointer

  Descript: Get event stream
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbProgramMapPrograms.GetEvents(Index: Word; Index2: Word):
  TDvbEitEvents;
begin
  Result := nil;
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbProgramMapPrograms[Index]) then
    Result := FDvbProgramMapPrograms[Index].Events[Index2];
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Index2>  Event id
            <Value>   Event stream
  Returns : -

  Descript: Set event stream
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbProgramMapPrograms.SetEvents(Index: Word; Index2: Word; Value:
  TDvbEitEvents);
begin
  if Index > High(FDvbProgramMapPrograms) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbProgramMapPrograms[Index].Events[Index2] := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbProgramMapPrograms
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbCaDescriptor
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbCaDescriptor.Create;
begin
  inherited Create;
  FPid := CDvbPidUndefined;
  FCaSystemId := CDvbPidUndefined;
  FCaPid := CDvbPidUndefined;
  FPrivateDataLength := 0;
  try
    New(FPrivateData);
  except
    FPrivateData := nil;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbCaDescriptor.Destroy;
begin
  if Assigned(FPrivateData) then
    Dispose(FPrivateData);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbCaDescriptor
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbCaDescriptors
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbCaDescriptors.Create;
var
  Loop: Integer;
begin
  inherited Create;
  FVersionNumber := $FF;
  for Loop := Low(FDvbCaDescriptors) to High(FDvbCaDescriptors) do
    FDvbCaDescriptors[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbCaDescriptors.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbCaDescriptors) to High(FDvbCaDescriptors) do
    if Assigned(FDvbCaDescriptors[Loop]) then
      FreeAndNil(FDvbCaDescriptors[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  if not Assigned(FDvbCaDescriptors[Index]) then
    FDvbCaDescriptors[Index] := TDvbCaDescriptor.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Pid id

  Descript: Get PID
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbCaDescriptors.GetPid(Index: Word): Word;
begin
  Result := CDvbPidUndefined;
  if Index > High(FDvbCaDescriptors) then
    Exit;
  if Assigned(FDvbCaDescriptors[Index]) then
    Result := FDvbCaDescriptors[Index].FPid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Pid id
  Returns : -

  Descript: Set PID
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.SetPid(Index: Word; Value: Word);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbCaDescriptors[Index].FPid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  CA system id

  Descript: Get CA system id
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbCaDescriptors.GetCaSystemId(Index: Word): Word;
begin
  Result := CDvbPidUndefined;
  if Index > High(FDvbCaDescriptors) then
    Exit;
  if Assigned(FDvbCaDescriptors[Index]) then
    Result := FDvbCaDescriptors[Index].FCaSystemId;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   CA system id
  Returns : -

  Descript: Set CA system id
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.SetCaSystemId(Index: Word; Value: Word);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbCaDescriptors[Index].FCaSystemId := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  CA pid id

  Descript: Get CA PID
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbCaDescriptors.GetCaPid(Index: Word): Word;
begin
  Result := CDvbPidUndefined;
  if Index > High(FDvbCaDescriptors) then
    Exit;
  if Assigned(FDvbCaDescriptors[Index]) then
    Result := FDvbCaDescriptors[Index].FCaPid;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   CA pid id
  Returns : -

  Descript: Set CA PID
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.SetCaPid(Index: Word; Value: Word);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbCaDescriptors[Index].FCaPid := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Private data length

  Descript: Get private data length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbCaDescriptors.GetPrivateDataLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbCaDescriptors) then
    Exit;
  if Assigned(FDvbCaDescriptors[Index]) then
    Result := FDvbCaDescriptors[Index].FPrivateDataLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Private data length
  Returns : -

  Descript: Set private data length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.SetPrivateDataLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbCaDescriptors[Index].FPrivateDataLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Private data descriptor pointer

  Descript: Get private data descriptor
  Notes   : Also creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbCaDescriptors.GetPrivateData(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbCaDescriptors[Index]) then
    Result := FDvbCaDescriptors[Index].FPrivateData;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Private data descriptor pointer
  Returns : -

  Descript: Set private data descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbCaDescriptors.SetPrivateData(Index: Word; Value: PDvbSection);
begin
  if Index > High(FDvbCaDescriptors) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbCaDescriptors[Index].FPrivateData := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbCaDescriptors
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbNetworkStream
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbNetworkStream.Create;
begin
  inherited Create;
  FStreamInfoLength := 0;
  try
    New(FStreamDescriptor);
  except
    FStreamDescriptor := nil;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbNetworkStream.Destroy;
begin
  if Assigned(FStreamDescriptor) then
    Dispose(FStreamDescriptor);
  inherited Destroy;
end;
{------------------------------------------------------------------------------
                                 TDvbNetworkStream
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbNetworkStreams
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbNetworkStreams.Create;
var
  Loop: Integer;
begin
  inherited Create;
  FVersionNumber := $FF;
  FNetworkId := CDvbPidUndefined;
  FInfoLength := 0;
  try
    New(FInfoDescriptor);
  except
    FInfoDescriptor := nil;
  end;
  FNumberOfStreams := 0;
  for Loop := Low(FDvbNetworkStreams) to High(FDvbNetworkStreams) do
    FDvbNetworkStreams[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbNetworkStreams.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbNetworkStreams) to High(FDvbNetworkStreams) do
    if Assigned(FDvbNetworkStreams[Loop]) then
      FreeAndNil(FDvbNetworkStreams[Loop]);
  if Assigned(FInfoDescriptor) then
    Dispose(FInfoDescriptor);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : -

  Descript: Create object if not assigned
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbNetworkStreams.CreateIfNotAssigned(Index: Word);
begin
  if Index > High(FDvbNetworkStreams) then
    Exit;
  if not Assigned(FDvbNetworkStreams[Index]) then
    FDvbNetworkStreams[Index] := TDvbNetworkStream.Create;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Stream info length

  Descript: Get stream info length
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbNetworkStreams.GetStreamInfoLength(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbNetworkStreams) then
    Exit;
  if Assigned(FDvbNetworkStreams[Index]) then
    Result := FDvbNetworkStreams[Index].FStreamInfoLength;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Pid Stream info length
  Returns : -

  Descript: Set stream info length
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbNetworkStreams.SetStreamInfoLength(Index: Word; Value: Word);
begin
  if Index > High(FDvbNetworkStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbNetworkStreams[Index].FStreamInfoLength := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
  Returns : <Result>  Stream descriptor pointer

  Descript: Get stream descriptor
  Notes   : Also creates it if unassigned
 ------------------------------------------------------------------------------}

function TDvbNetworkStreams.GetStreamDescriptor(Index: Word): PDvbSection;
begin
  Result := nil;
  if Index > High(FDvbNetworkStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  if Assigned(FDvbNetworkStreams[Index]) then
    Result := FDvbNetworkStreams[Index].FStreamDescriptor;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Program map id
            <Value>   Stream descriptor pointer
  Returns : -

  Descript: Set stream descriptor
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbNetworkStreams.SetStreamDescriptor(Index: Word; Value:
  PDvbSection);
begin
  if Index > High(FDvbNetworkStreams) then
    Exit;
  CreateIfNotAssigned(Index);
  FDvbNetworkStreams[Index].FStreamDescriptor := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbNetworkStreams
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 Implementation
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Packet data buffer
  Returns : <Result>            True if valid Pid (no other errors detected)
            <Pid>               Pid value

  Descript: Get PID from transport stream.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word):
  Boolean;
begin
  Result := False;
  // One of the few checks at this point is the synchronization byte which must be correct
  if StreamPacketData[0] = CDvbPacketSyncWord then
    Result := True
  else if PacketSyncErrors = $FFFF then
    PacketSyncErrors := 0
  else
    Inc(PacketSyncErrors);
  Pid := ((StreamPacketData[1] and $1F) shl 8) or StreamPacketData[2];
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Number of programs

  Descript: Get number of programs in stream
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetNumberOfPrograms: Byte;
begin
  Result := PatPrograms;
end;

{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Packet data buffer
  Returns : <Result>            True if valid offset
                                False if error or no payload
            <Offset>            Offset into first data byte of payload

  Descript: Get offset where payload starts.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbGetPayloadOffset(StreamPacketData: PDvbTransportPacket; var Offset:
  Word): Boolean;
var
  AdaptationFieldControl: Byte;
begin
  Result := False;
  Offset := 0;
  // The start of the payload depends on the adaptation field. If there is
  // an adaptation field then the payload starts after this. If there is no
  // adaptation field then the payload start immediately.
  // Also there is the posibility that there is no payload at all.
  AdaptationFieldControl := StreamPacketData[3] and $30;
  case AdaptationFieldControl of
    $00: Exit; // Reserved
    $10: Offset := 4; // Payload only
    $20: Exit; // Adaptation field only, no payload
    $30: begin
           Offset := StreamPacketData[4] + 5; // Adaptation field with payload
           // Check for excessive value
           if Offset > CDvbPacketSize then
           begin
             Offset := 0;
             Exit;
           end;
         end;    
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
  Returns : <Result>          True if success
            <SatelliteInfo>   Info of satellite

  Descript: Get satelite delivery descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbGetSatelliteDeliverySystem(DescriptorData: PDvbSection; var
  SatelliteInfo: string): Boolean;
begin
  Result := False;
  SatelliteInfo := '';
  if DescriptorData[0] <> CDvbTagSatelliteDeliverySystem then
    Exit;
  if DescriptorData[1] <> 11 then
    Exit;
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[2] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[2] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[3] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + '.';
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[3] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[4] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[4] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[5] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[5] and $0F));
  SatelliteInfo := SatelliteInfo + ' GHz, ';
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[6] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[6] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[7] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + '.';
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[7] and $0F));
  SatelliteInfo := SatelliteInfo + ' degrees ';
  if (DescriptorData[8] and $80) <> 0 then
    SatelliteInfo := SatelliteInfo + ' West, '
  else
    SatelliteInfo := SatelliteInfo + ' East, ';
  case (DescriptorData[8] and $60) of
    $00: SatelliteInfo := SatelliteInfo + ' Horizontal, ';
    $20: SatelliteInfo := SatelliteInfo + ' Vertical, ';
    $40: SatelliteInfo := SatelliteInfo + ' Circular left, ';
    $60: SatelliteInfo := SatelliteInfo + ' Circular right, ';
  end;
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[09] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[09] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[10] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + '.';
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[10] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[11] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + (DescriptorData[11] and $0F));
  SatelliteInfo := SatelliteInfo + Chr(Ord('0') + ((DescriptorData[12] and $F0)
    shr 4));
  SatelliteInfo := SatelliteInfo + ' Msymbols/s, ';
  case (DescriptorData[12] and $0F) of
    $0: SatelliteInfo := SatelliteInfo + ' Modulation undefined';
    $1: SatelliteInfo := SatelliteInfo + ' QPSK';
    $2: SatelliteInfo := SatelliteInfo + ' 8PSK';
    $3: SatelliteInfo := SatelliteInfo + ' 16-QAM';
  else
    SatelliteInfo := SatelliteInfo + ' Reserved';
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
  Returns : <Result>          True if success
            <ServiceInfo>     Info on service
            <ServiceName>     Service name (part of <ServiceInfo>)

  Descript: Get service descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbGetService(DescriptorData: PDvbSection; var ServiceInfo: string; var
  ServiceName: string): Boolean;
var
  Loop: Word;
  Index: Word;
begin
  Result := False;
  if DescriptorData[0] <> CDvbTagService then
    Exit;
  ServiceInfo := '';
  case DescriptorData[2] of
    $01: ServiceInfo := ServiceInfo + 'Digital television service';
    $02: ServiceInfo := ServiceInfo + 'Digital radio sound service';
    $03: ServiceInfo := ServiceInfo + 'Teletext service';
    $04: ServiceInfo := ServiceInfo + 'NVOD reference service';
    $05: ServiceInfo := ServiceInfo + 'NVOD time-shifted service';
    $06: ServiceInfo := ServiceInfo + 'Mosaic service';
    $07: ServiceInfo := ServiceInfo + 'PAL coded signal';
    $08: ServiceInfo := ServiceInfo + 'SECAM coded signal';
    $09: ServiceInfo := ServiceInfo + 'D/D2-MAC';
    $0A: ServiceInfo := ServiceInfo + 'FM Radio';
    $0B: ServiceInfo := ServiceInfo + 'NTSC coded signal';
    $0C: ServiceInfo := ServiceInfo + 'Data broadcast service';
    $0D: ServiceInfo := ServiceInfo + 'Common Interface';
    $0E: ServiceInfo := ServiceInfo + 'RCS map';
    $0F: ServiceInfo := ServiceInfo + 'RCS FLS';
    $10: ServiceInfo := ServiceInfo + 'DVB MHP service';
  else
    ServiceInfo := ServiceInfo + 'Reserved';
  end;
  ServiceInfo := ServiceInfo + ', ';
  if DescriptorData[3] = 0 then
    ServiceInfo := serviceInfo + '-, '
  else
  begin
    for Loop := 0 to DescriptorData[3] - 1 do
      ServiceInfo := ServiceInfo + Chr(DescriptorData[4 + Loop]);
    ServiceInfo := serviceInfo + ', ';
  end;
  Index := DescriptorData[3] + 4;
  ServiceName := '';
  if DescriptorData[Index] = 0 then
    ServiceInfo := serviceInfo + '-'
  else
  begin
    for Loop := 0 to DescriptorData[Index] - 1 do
      ServiceInfo := ServiceInfo + Chr(DescriptorData[Index + Loop + 1]);
    for Loop := 0 to DescriptorData[Index] - 1 do
      ServiceName := ServiceName + Chr(DescriptorData[Index + Loop + 1]);
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode present/following and schedule event information table data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeEventInformationTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
var
  Version: Byte;
  Index: Word;
  Loop: Word;
  LastIndex: Word;
  ServiceId: Word;
  ProgramIndex: Word;
  Events: TDvbEitEvents;
begin
  Result := False;
  if not (SectionData[0] in [CDvbPsiTableIdEitAS..CDvbPsiTableIdEitAS + $F]) and
    (SectionData[0] <> CDvbPsiTableIdEitA) then
    Exit;
  // If not yet applicable, skip it
  if (SectionData[5] and $01) = 0 then
    Exit;
  // Get associated program number
  ServiceId := (SectionData[3] shl 8) or SectionData[4];
  ProgramIndex := ProgramToTableReference[ServiceId];
  // If no program exists for this information, exit
  if ProgramIndex = $FFFF then
    Exit;
  // Only continue if new version of event information
  Version := SectionData[5] and $3E;
  if PmtPrograms.Events[ProgramIndex, SectionData[0]].VersionNumber = Version
    then
    Exit;
  // Choose the correct event information table
  Events := PmtPrograms.Events[ProgramIndex, SectionData[0]];
  begin
    // Save new version
    if SectionData[6] = 0 then
      Events.Event := 0;
    Events.LastSection := SectionData[7];
    Events.LastTableId := SectionData[13];
    LastIndex := ((SectionData[1] and $0F) shl 8) or SectionData[2] + 2 - 4;
    if LastIndex > (13 + 12) then
    begin
      // Start of first block
      Index := 14;
      repeat
        Events.StartDate[Events.Event] := (SectionData[Index + 2] shl 8) or
          sectionData[Index + 3];
        Events.StartTime[Events.Event] := (SectionData[Index + 4] shl 16) or
          (SectionData[Index + 5] shl 8) or
          SectionData[Index + 6];
        Events.Duration[Events.Event] := (SectionData[Index + 7] shl 16) or
          (SectionData[Index + 8] shl 8) or
          SectionData[Index + 9];
        Events.RunningStatus[Events.Event] := (SectionData[Index + 10] and $E0)
          shr 5;
        if (SectionData[Index + 10] and $10) <> 0 then
          Events.FreeCaMode[Events.Event] := False
        else
          Events.FreeCaMode[Events.Event] := True;
        Events.DescriptorLength[Events.Event] := ((SectionData[Index + 10] and
          $0F) shl 8) or SectionData[Index + 11];
        if Events.DescriptorLength[Events.Event] > 0 then
          for Loop := 0 to Events.DescriptorLength[Events.Event] - 1 do
            Events.StreamDescriptor[Events.Event][Loop] := SectionData[Index + 12
              + Loop];

        Inc(Index, Events.DescriptorLength[Events.Event] + 12);
        if Events.Event <> 255 then
          Events.Event := Events.Event + 1;
      until (Index >= LastIndex) or
        (Events.Event = 255);
    end;
    // Inform that a (one) new event is available if the whole section
    // has been received.
    if (Events.LastTableId = SectionData[0]) and
      (SectionData[6] = Events.LastSection) then
    begin
      // Make sure we receive a completely new one once again
      if Events.VersionNumber = $FF then
        Events.VersionNumber := $FE
      else
      begin
        Events.VersionNumber := Version;
        if EventCallback.Handle <> INVALID_HANDLE_VALUE then
          PostMessage(EventCallback.Handle, EventCallback.Msg, CSignalEvent,
            ServiceId);
      end;
    end;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode running status table data.
  Notes   : Not yet implemented
 ------------------------------------------------------------------------------}

function DvbDecodeRunningStatusTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
begin
  Result := False;
  if SectionData[0] <> CDvbPsiTableIdRst then
    Exit;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode time offset table data.
  Notes   : Not yet implemented
 ------------------------------------------------------------------------------}

function DvbDecodeTimeOffsetTable(SectionData: PDvbSection; Pid: Word): Boolean;
  stdcall;
begin
  Result := False;
  if SectionData[0] <> CDvbPsiTableIdTot then
    Exit;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode time and date table data.
  Notes   : Not tested
 ------------------------------------------------------------------------------}

function DvbDecodeTimeAndDateTable(SectionData: PDvbSection; Pid: Word): Boolean;
  stdcall;
var
  SectionLength: Word;
  Calc: Real;
  YCalc: Real;
  MCalc: Real;
  YInt: Word;
  MInt: Word;
  Date: Word;
begin
  Result := False;
  if SectionData[0] <> CDvbPsiTableIdTdt then
    Exit;
  SectionLength := ((SectionData[1] and $0F) shl 8) or SectionData[2];
  if SectionLength <> 5 then
    Exit;
  Date := (SectionData[3] shl 8) or SectionData[4];
  Calc := (Date - 15078.2) / 365.25;
  Year := Trunc(Calc);
  YCalc := Year * 365.25;
  YInt := Trunc(YCalc);
  Calc := Date - 14956.1 - YInt;
  Calc := Calc / 30.6001;
  Month := Trunc(Calc);
  MCalc := Month * 30.6001;
  MInt := Trunc(MCalc);
  Calc := Date - 14956 - YInt - MInt;
  Day := Trunc(Calc);
  if (Month = 14) or (Month = 15) then
  begin
    Inc(Year);
    Month := Month - 13;
  end
  else
    Dec(Month);
  Year := Year + 1900;

  TimeInfo := '';
  TimeInfo := TimeInfo + Chr(Ord('0') + ((SectionData[5] and $F0) shr 4));
  TimeInfo := TimeInfo + Chr(Ord('0') + (SectionData[5] and $0F));
  TimeInfo := TimeInfo + ':';
  TimeInfo := TimeInfo + Chr(Ord('0') + ((SectionData[6] and $F0) shr 4));
  TimeInfo := TimeInfo + Chr(Ord('0') + (SectionData[6] and $0F));
  TimeInfo := TimeInfo + ':';
  TimeInfo := TimeInfo + Chr(Ord('0') + ((SectionData[7] and $F0) shr 4));
  TimeInfo := TimeInfo + Chr(Ord('0') + (SectionData[7] and $0F));
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode network information table data (actual network).
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeNetworkInformationTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
var
  SectionLength: Word;
  Index: Word;
  Loop: Word;
begin
  Result := False;
  if (SectionData[0] <> CDvbPsiTableIdNitA) and
    (SectionData[0] <> CDvbPsiTableIdNitO) then
    Exit;
  // If no new version then skip it
  if Network.VersionNumber = SectionData[5] then
    Exit;

  Network.NetworkId := (SectionData[3] shl 8) or SectionData[4];
  Network.InfoLength := ((SectionData[8] and $0F) shl 8) or SectionData[9];
  if Network.InfoLength > 0 then
    for Loop := 0 to (Network.InfoLength - 1) do
      Network.InfoDescriptor[Loop] := SectionData[Loop + 10];
  Index := Network.InfoLength + 10;

  Network.NumberOfStreams := 0;
  SectionLength := ((SectionData[Index] and $0F) shl 8) or SectionData[Index +
    1];
  Inc(Index, 2);
  if SectionLength > 0 then
  begin
    // Set end marker
    Inc(SectionLength, Index);
    repeat
      Network.StreamInfoLength[Network.NumberOfStreams] := ((SectionData[Index +
        4] and $0F) shl 8) or SectionData[Index + 5];
      if Network.StreamInfoLength[Network.NumberOfStreams] > 0 then
      begin
        for Loop := 0 to Network.StreamInfoLength[Network.NumberOfStreams] - 1
          do
          Network.StreamDescriptor[Network.NumberOfStreams][Loop] :=
            SectionData[Loop + Index + 6];
      end;
      Inc(Index, Network.StreamInfoLength[Network.NumberOfStreams] + 6);
      Network.NumberOfStreams := Network.NumberOfStreams + 1;
    until Index >= SectionLength;
  end;
  Network.VersionNumber := SectionData[5];
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode service description table data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeServiceDescriptionTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
var
  Loop: Word;
  LastIndex: Word;
  DescriptorLength: Word;
  ProgramNr: Word;
  ProgramIndex: Word;
  Index: Word;
begin
  Result := False;
  if (SectionData[0] <> CDvbPsiTableIdSdtA) and
    (SectionData[0] <> CDvbPsiTableIdSdtO) then
    Exit;
  LastIndex := ((SectionData[1] and $0F) shl 8) or SectionData[2] + 2 - 4;
  // Start of first block
  Index := 11;
  repeat
    ProgramNr := (SectionData[Index] shl 8) or SectionData[Index + 1];
    // Get the table reference for our program number
    ProgramIndex := ProgramToTableReference[ProgramNr];
    DescriptorLength := ((SectionData[Index + 3] and $0F) shl 8) or
      SectionData[Index + 4];
    Inc(Index, 5);
    // Only store data if valid
    if ProgramIndex <> $FFFF then
    begin
      PmtPrograms.SdtInfoLength[ProgramIndex] := DescriptorLength;
      if DescriptorLength > 0 then
        for Loop := 0 to DescriptorLength - 1 do
          PmtPrograms.SdtDescriptor[ProgramIndex][Loop] := SectionData[Index +
            Loop];
    end;
    Inc(Index, DescriptorLength);
  until Index >= LastIndex;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode program map table data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeProgramMapTable(SectionData: PDvbSection; Pid: Word): Boolean;
  stdcall;
var
  Loop: Word;
  LastIndex: Word;
  Index: Word;
  Reference: Word;
  Version: Byte;
  ProgramId: Word;
  NumberOfStreams: Word;
  Streams: TDvbProgramMapStreams;
begin
  Result := False;
  if (SectionData[0] <> CDvbPsiTableIdPmt) then
    Exit;
  // Don't bother if table not (yet) applicable
  if (SectionData[5] and $01) = 0 then
    Exit;
  ProgramId := (SectionData[3] shl 8) or Sectiondata[4];
  Reference := ProgramToTableReference[ProgramId];
  // If no valid reference then skip it
  if Reference = $FFFF then
    Exit;
  Version := SectionData[5] and $3E;
  begin
    // If same 'version' then don't do anything. Only for changes we will be active.
    if Version = PmtPrograms.VersionNumber[Reference] then
      Exit;
    PmtPrograms.VersionNumber[Reference] := Version;
    PmtPrograms.PcrPid[Reference] := ((SectionData[8] and $1F) shl 8) or
      Sectiondata[9];
    PmtPrograms.InfoLength[Reference] := ((SectionData[10] and $0F) shl 8) or
      SectionData[11];
    if PmtPrograms.InfoLength[Reference] > 0 then
      for Loop := 0 to PmtPrograms.InfoLength[Reference] - 1 do
        PmtPrograms.InfoDescriptor[Reference][Loop] := SectionData[Loop + 12];
    // Calculate last possible index
    LastIndex := ((SectionData[1] and $0F) shl 8) or SectionData[2] + 2 - 4;
    // Point into first stream data
    Index := 12 + PmtPrograms.InfoLength[Reference];
    PmtPrograms.NumberOfStreams[Reference] := 0;
    while (Index < LastIndex) do
    begin
      NumberOfStreams := PmtPrograms.NumberOfStreams[Reference];
      Streams := PmtPrograms.Streams[NumberOfStreams];
      Streams.StreamType[Reference] := SectionData[Index];
      Inc(Index);
      Streams.ElementaryPid[Reference] := ((SectionData[Index] and $1F) shl 8) or
        SectionData[Index + 1];
      Inc(Index, 2);
      Streams.StreamInfoLength[Reference] := ((SectionData[Index] and $0F) shl 8)
        or SectionData[Index + 1];
      Inc(Index, 2);
      if Streams.StreamInfoLength[Reference] > 0 then
        for Loop := 0 to Streams.StreamInfoLength[Reference] - 1 do
        begin
          Streams.StreamDescriptor[Reference][Loop] := SectionData[Index];
          Inc(Index);
        end;
      PmtPrograms.NumberOfStreams[Reference] :=
        PmtPrograms.NumberOfStreams[Reference] + 1;
    end;
  end;
  // Inform that a (one) new PMT is available
  if PmtCallback.Handle <> INVALID_HANDLE_VALUE then
    PostMessage(PmtCallback.Handle, PmtCallback.Msg, CSignalPmt, ProgramId);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode conditional access table data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeConditionalAccessTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
var
  Loop: Word;
  LastIndex: Word;
  Descriptors: Word;
  Index: Word;
  Version: Byte;
begin
  Result := False;
  if (SectionData[0] <> CDvbPsiTableIdCat) then
    Exit;
  LastIndex := (((SectionData[1] and $0F) shl 8) or SectionData[2]) + 2 - 4;
  // Only continue if new version of event information
  Version := SectionData[5] and $3E;
  if CaDescriptors.VersionNumber = Version then
    Exit;
  Descriptors := 0;
  Index := 8;
  CaDescriptors.VersionNumber := Version;
  repeat
    // Check for descriptor tag
    // A CDvbTagCA in the CA section indicates EMM data (system wide data)
    if SectionData[Index] = CDvbTagCa then
    begin
      CaDescriptors.Pid[Descriptors] := Pid;
      CaDescriptors.PrivateDataLength[Descriptors] := SectionData[Index + 1] -
        4;
      CaDescriptors.CaSystemId[Descriptors] := (SectionData[Index + 2] shl 8) or
        SectionData[Index + 3];
      CaDescriptors.CaPid[Descriptors] := ((SectionData[Index + 4] and $1F) shl
        8) or SectionData[Index + 5];
      if CaDescriptors.PrivateDataLength[Descriptors] > 0 then
        for Loop := 0 to CaDescriptors.PrivateDataLength[Descriptors] - 1 do
          CaDescriptors.PrivateData[Descriptors][Loop] := SectionData[Index + 6
            + Loop];
      Inc(Index, SectionData[Index + 1] + 2);
      Inc(Descriptors);
    end
    else
      Inc(Index, SectionData[Index + 1] + 2);
  until (Index >= LastIndex) or (Descriptors >
    High(CaDescriptors.FDvbCaDescriptors));
  CaDescriptorQuantity := Descriptors;
  // Inform that a (one) new PMT is available
  if CatCallback.Handle <> INVALID_HANDLE_VALUE then
    PostMessage(CatCallback.Handle, CatCallback.Msg, CSignalCat, Version);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Full section data
            <Pid>          Pid
  Returns : <Result>       True if no error

  Descript: Decode program association table data.
  Notes   : This only builds up the initial list of <PmtPrograms>. Sometimes
            the PmtPrograms.Pid> contains multiple program numbers, and not
            only the indicated <ProgramNr> here. This is handled in
            <DvbDecodeProgramMapTable>.
 ------------------------------------------------------------------------------}

function DvbDecodeProgramAssociationTable(SectionData: PDvbSection; Pid: Word):
  Boolean; stdcall;
var
  Loop: Word;
  SectionLength: Word;
  ProgramPid: Word;
  ProgramNr: Word;
  Programs: Word;
begin
  Result := False;
  if (SectionData[0] <> CDvbPsiTableIdPat) then
    Exit;
  // Don't bother if table not (yet) applicable
  if (SectionData[5] and $01) = 0 then
    Exit;
  // If same 'version' then don't do anything. Only for changes we will be active.
  if (SectionData[5] and $3E) = PatVersion then
    Exit;
  // Clear program list
  FreeAndNil(PmtPrograms);
  PmtPrograms := TDvbProgramMapPrograms.Create;
  // Clear reference list
  for Loop := Low(ProgramToTableReference) to High(ProgramToTableReference) do
    ProgramToTableReference[Loop] := $FFFF;

  SectionLength := ((SectionData[1] and $0F) shl 8) or SectionData[2];
  Programs := (SectionLength - 9) div 4;
    // Programs in section (9 = CRC + part header)
  // Sanity check
  if Programs > High(PmtPrograms.FDvbProgramMapPrograms) then
    Exit;
  PmtProgramIndex := 0;
  if Programs > 0 then
    for Loop := 0 to Programs - 1 do
    begin
      // Check program number: <>0 is valid otherwise it is a network PID
      // Potential problem: since we are updating the active list, while
      // the data we receive might be not 'active' we can get into problems..
      ProgramNr := (SectionData[Loop * 4 + 8] shl 8) or SectionData[Loop * 4 +
        9];
      if ProgramNr <> 0 then
      begin
        ProgramPid := ((SectionData[Loop * 4 + 10] and $1F) shl 8) or
          SectionData[Loop * 4 + 11];
        PidList.PidType[ProgramPid] := CDvbPsiTableIdPmt;
        PidList.ProgramNumber[ProgramPid] := ProgramNr;
        PidList.ProgramIndex[ProgramPid] := PmtProgramIndex;
        ProgramToTableReference[ProgramNr] := PmtProgramIndex;
        PmtPrograms.Pid[PmtProgramIndex] := ProgramPid;
        PmtPrograms.ProgramNumber[PmtProgramIndex] := ProgramNr;
        PidList.PacketFilter[ProgramPid] :=
          @DvbDecodeProgramSpecificInformationData;
        Inc(PmtProgramIndex);
      end;
    end;
  PatPrograms := PmtProgramIndex;
  PatVersion := (SectionData[5] and $3E);
  // Inform that a (one) new PMT is available
  if PatCallback.Handle <> INVALID_HANDLE_VALUE then
    PostMessage(PatCallback.Handle, PatCallback.Msg, CSignalPat, PatVersion);

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Stream packet data
  Returns : <Result>            True if no error

  Descript: Decode program specific information data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeProgramSpecificInformationData(StreamPacketData:
  PDvbTransportPacket): Boolean; stdcall;
var
  LoopStart: Word;
  InitialStart: Word;
  Loop: Word;
  Checksum32: Dword;
  Pid: Word;
  PayloadBytes: Word;
  SectionLength: Word;
begin
  Result := False;
  ResetLock.Acquire;
  try
    // Sanity checks
    if StreamPacketData[0] <> CDvbPacketSyncWord then
      Exit;
    if not DvbFilterGetPid(StreamPacketData, Pid) then
      Exit;
    try
      begin
        // Because the actual data can be spread out over multiple packets we have
        // to acquire them first. The payload unit start indicator gives us the
        // indication if we are dealing with a new block of data or that data is
        // being 'continued'.
        if (StreamPacketData[1] and $40) <> 0 then
        begin
          // Start of a new packet resets the indexes
          SectionReceiving.DataIndex[Pid] := 0;
          // All so-called PSI tables have a first byte which is the <PointerField>.
          // This indicates at what offset the actual data starts. We know that
          // the offset points into this very first packet...
          // First add the pointer field to the start of the payload data
          if not DvbGetPayloadOffset(StreamPacketData, InitialStart) then
          begin
            // No payload then exit without an error
            Result := True;
            Exit;
          end;
          // Add pointer field value to current offset
          SectionReceiving.PointerField[Pid] := StreamPacketData[InitialStart];
          LoopStart := InitialStart + SectionReceiving.PointerField[Pid] + 1;
          // Some providers use an offset in the <PointerField> which is
          // larger than the packet size. This does not comply with the standard
          // since the payload unit start indicator should not be set in such
          // circumstances. However, to be as flexible as possible we have to
          // allow for that .... We need to throw away data in the next packet(s)
          // as well when the <PointerField> is more than this packet ...
          // CORRECTION HAS BEEN REMOVED (NOT TESTED SO JUST SEE IT AS AN ERROR)
          if LoopStart > CDvbPacketSize then
          begin
            // Adjust pointer field value with currently discarded values and
            // exit without an error
  //          Dec(PointerField, CDvbPacketSize - InitialStart - 1);
  //          Result := True;
            Exit;
          end;
          SectionReceiving.PointerField[Pid] := 0;
          // Calculate additional number of bytes in payload
          PayloadBytes := CDvbPacketSize - LoopStart;
          sectionReceiving.Valid[Pid] := True;
        end
        else
        begin
          if not SectionReceiving.Valid[Pid] then
            Exit;
          // If this is the continuing data then we only have to add the
          // complete payload to our data collected sofar
          if not DvbGetPayloadOffset(StreamPacketData, LoopStart) then
          begin
            // No payload then exit without an error
            Result := True;
            Exit;
          end;
          // Calculate number of bytes in payload
          // CORRECTION HAS BEEN REMOVED (NOT TESTED SO JUST SEE IT AS AN ERROR)
  //        if PointerField > 0 then
  //        begin
  //          // If PointerField stretches over multiple packets (non-confirming to the
  //          // standard) then we have to discard additional data in the next
  //          // packet(s)
  //          if (LoopStart + PointerField) > CDvbPacketSize then
  //          begin
  //            // If still no data in this packet
  //            Dec(PointerField, CDvbPacketSize - LoopStart - 1);
  //            PayloadBytes := 0;
  //          end
  //          else
  //          begin
  //            // If data starts in this packet make the necessary correction
  //            PayLoadBytes := CDvbPacketSize - LoopStart - PointerField;
  //            PointerField := 0;
  //          end;
  //        end
  //        else
          PayloadBytes := CDvbPacketSize - LoopStart;
        end;

        if LoopStart < PayloadBytes then
        begin
          // Copy payload data to stream data buffer
          for Loop := LoopStart to (LoopStart + PayloadBytes - 1) do
          begin
            if SectionReceiving.DataIndex[Pid] >=
              SizeOf(SectionReceiving.StreamData[Pid]^) then
              Break;
            SectionReceiving.StreamData[Pid][SectionReceiving.DataIndex[Pid]] :=
              StreamPacketData[Loop];
            SectionReceiving.DataIndex[Pid] := SectionReceiving.DataIndex[Pid] +
              1;
          end;
          SectionLength := ((SectionReceiving.StreamData[Pid][1] and $0F) shl 8)
            or SectionReceiving.StreamData[Pid][2];
          if SectionLength < 6 then
          begin
            Result := True;
            Exit;
          end;
          if (SectionReceiving.DataIndex[Pid] > (SectionLength + 2)) then
            // If we have it all
          begin
            // Before we do any further processing at all we check for a correct checksum
            Checksum32 := DvbResetValueCrc32;
            for Loop := 0 to SectionLength + 2 do
              Checksum32 := DvbNewValueCrc32(Checksum32,
                SectionReceiving.StreamData[Pid][Loop]);
            // Check for correct checksum
            if Checksum32 <> 0 then
            begin
              if PacketError = $FFFF then
                PacketError := 0
              else
                Inc(PacketError);
              Exit;
            end;
            // We have a valid section at this point. Now depending on the type of
            // section (table id) we are dealing with, we have to process it differently
            // The <PsiFilters> array contains the procedure to call
            PsiTableId[SectionReceiving.StreamData[Pid][0]] := True;
            if Assigned(@PsiFilters[SectionReceiving.StreamData[Pid][0]]) then
              if not
                PsiFilters[SectionReceiving.StreamData[Pid][0]](SectionReceiving.StreamData[Pid], Pid) then
                Exit;
            // If a signalling exists, inform it
            if not (Pid > High(PidListCallback)) then
              if PidListCallback[Pid].Handle <> INVALID_HANDLE_VALUE then
                PostMessage(PidListCallback[Pid].Handle,
                  PidListCallback[Pid].Msg, CSignalPid, Pid);
          end;
        end;
        Result := True;
      end;
    except
      if PacketError = $FFFF then
        PacketError := 0
      else
        Inc(PacketError);
    end;
  finally
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Reset error count.
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbFilterResetErrors;
begin
  PacketError := 0;
  PacketSyncErrors := 0;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Errors counted

  Descript: Get error count.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetErrors: Word;
begin
  Result := PacketError;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Errors counted

  Descript: Get error count.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetPacketSyncErrors: Word;
begin
  Result := PacketSyncErrors;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  debug value

  Descript: Get debug value.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetDebug: Word;
begin
  Result := Debug;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Set all table references to unassigned status.
  Notes   : Creates all objects too.
 ------------------------------------------------------------------------------}

procedure DvbNilTables;
var
  Loop: Word;
begin
  for Loop := Low(TransportPids) to High(TransportPids) do
    TransportPids[Loop] := False;

  FreeAndNil(PidList);
  PidList := TDvbTypePids.Create;

  FreeAndNil(SectionReceiving);
  SectionReceiving := TDvbReceivingSections.Create;

  FreeAndNil(PmtPrograms);
  PmtPrograms := TDvbProgramMapPrograms.Create;

  FreeAndNil(CaDescriptors);
  CaDescriptors := TDvbCaDescriptors.Create;

  FreeAndNil(Network);
  Network := TDvbNetworkStreams.Create;

  PatCallback.Handle := INVALID_HANDLE_VALUE;
  CatCallback.Handle := INVALID_HANDLE_VALUE;
  PmtCallback.Handle := INVALID_HANDLE_VALUE;
  EventCallback.Handle := INVALID_HANDLE_VALUE;
  for Loop := Low(PidListCallback) to High(PidListCallback) do
    PidListCallback[Loop].Handle := INVALID_HANDLE_VALUE;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Clear allocated memory and such.
  Notes   : Typically used when switching channels.
            Should be synchronized with packet handler!
 ------------------------------------------------------------------------------}

procedure DvbDestroyTables;
var
  Loop: Word;
begin
  for Loop := Low(TransportPids) to High(TransportPids) do
    TransportPids[Loop] := False;

  FreeAndNil(PidList);
  FreeAndNil(SectionReceiving);
  FreeAndNil(PmtPrograms);
  FreeAndNil(CaDescriptors);
  FreeAndNil(Network);

  for Loop := Low(ProgramToTableReference) to High(ProgramToTableReference) do
    ProgramToTableReference[Loop] := $FFFF;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Reset tables and such.
  Notes   : All variables are set to their intial values.
            Should be synchronized with packet handler!
 ------------------------------------------------------------------------------}

procedure DvbFilterCreateTables;
var
  Loop: Word;
begin
  ResetLock.Acquire;
  try
    DvbDestroyTables;
    // Create base objects
    PidList := TDvbTypePids.Create;
    SectionReceiving := TDvbReceivingSections.Create;
    PmtPrograms := TDvbProgramMapPrograms.Create;
    CaDescriptors := TDvbCaDescriptors.Create;
    Network := TDvbNetworkStreams.Create;

    // Setup defaults so new data is handled, eg. VersionNumber
    PatVersion := $FF;
    PatPrograms := 0;
    TransportPidsCounter := 0;

    Network.NumberOfStreams := 0;
    Network.VersionNumber := $FF;

    // Setup automatically decoding of PAT
    // This is the only PID recognized at this point
    PidList.PacketFilter[CPidPAT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidPAT] := CDvbPsiTableIdPat;

    PidList.PacketFilter[CPidCAT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidCAT] := CDvbPsiTableIdCat;

    // The DVB standard also defines other reserved PIDs. These
    // are actually 'predefined' PSI tables.
    // The PID type as such is not important here, just that it is
    // not 'undefined'.
    PidList.PacketFilter[CPidNIT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidNIT] := CDvbPsiTableIdNitA;
    PidList.PacketFilter[CPidSDT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidSDT] := CDvbPsiTableIdSdtA;
    PidList.PacketFilter[CPidEIT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidEIT] := CDvbPsiTableIdEitA;
    PidList.PacketFilter[CPidRST] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidRST] := CDvbPsiTableIdRst;
    PidList.PacketFilter[CPidTDT] := @DvbDecodeProgramSpecificInformationData;
    PidList.PidType[CPidTDT] := CDvbPsiTableIdTdt;

    // Clear indication
    for Loop := Low(PsiTableId) to High(PsiTableId) do
      PsiTableId[Loop] := False;

    // Setup the default list of filtering of PSI data
    for Loop := Low(PsiFilters) to High(PsiFilters) do
    begin
      // By default not supported
      PsiFilters[Loop] := nil;
      // Now fill in the processing we (already) support
      if Loop = CDvbPsiTableIdPat then
        PsiFilters[Loop] := @DvbDecodeProgramAssociationTable;
      if Loop = CDvbPsiTableIdCat then
        PsiFilters[Loop] := @DvbDecodeConditionalAccessTable;
      if Loop = CDvbPsiTableIdPmt then
        PsiFilters[Loop] := @DvbDecodeProgramMapTable;
      if Loop = CDvbPsiTableIdSdtA then
        PsiFilters[Loop] := @DvbDecodeServiceDescriptionTable;
      if Loop = CDvbPsiTableIdNitA then
        PsiFilters[Loop] := @DvbDecodeNetworkInformationTable;
      if Loop = CDvbPsiTableIdTdt then
        PsiFilters[Loop] := @DvbDecodeTimeAndDateTable;
      if Loop = CDvbPsiTableIdTot then
        PsiFilters[Loop] := @DvbDecodeTimeOffsetTable;
      if Loop = CDvbPsiTableIdRst then
        PsiFilters[Loop] := @DvbDecodeRunningStatusTable;
      if Loop = CDvbPsiTableIdEitA then
        PsiFilters[Loop] := @DvbDecodeEventInformationTable;
      if Loop in [CDvbPsiTableIdEitAS..CDvbPsiTableIdEitAS + $F] then
        PsiFilters[Loop] := @DvbDecodeEventInformationTable;
    end;
  finally
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Pid>        Pid identifier
            <hWnd>       Window handle to send message to
            <Msg>        Message to send
  Returns : <Result>     True if success

  Descript: Set message 'callback' for Pid
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetSignallingCallback(Pid: Word; hWnd: HWND; Msg: UINT):
  Boolean;
begin
  Result := False;
  if Pid > High(PidListCallback) then
    Exit;
  PidListCallback[Pid].Msg := Msg;
  PidListCallback[Pid].Handle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <hWnd>       Window handle to send message to
            <Msg>        Message to send
  Returns : <Result>     True if success

  Descript: Set message 'callback' for PAT
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetSignallingCallbackPat(hWnd: HWND; Msg: UINT): Boolean;
begin
  PatCallback.Msg := Msg;
  PatCallback.Handle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <hWnd>       Window handle to send message to
            <Msg>        Message to send
  Returns : <Result>     True if success

  Descript: Set message 'callback' for CAT
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetSignallingCallbackCat(hWnd: HWND; Msg: UINT): Boolean;
begin
  CatCallback.Msg := Msg;
  CatCallback.Handle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <hWnd>       Window handle to send message to
            <Msg>        Message to send
  Returns : <Result>     True if success

  Descript: Set message 'callback' for PMT
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetSignallingCallbackPmt(hWnd: HWND; Msg: UINT): Boolean;
begin
  PmtCallback.Msg := Msg;
  PmtCallback.Handle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <hWnd>       Window handle to send message to
            <Msg>        Message to send
  Returns : <Result>     True if success

  Descript: Set message 'callback' for event
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
begin
  EventCallback.Msg := Msg;
  EventCallback.Handle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Pid>        Pid identifier
            <Filter>     Filter function which will process the data
  Returns : <Result>     True if success

  Descript: Set filter for Pid
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  PidList.PacketFilter[Pid] := Filter;
  if Assigned(@PidList.PacketFilter[Pid]) then
    Result := True
  else
    Result := False;
end;

{------------------------------------------------------------------------------
  Params  : <Pid>        Pid identifier
  Returns : <Result>     Assigned if valid Pid
                         Nil otherwise

  Descript: Get filter for Pid (only if valid Pid)
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := nil;
  if PidList.PidType[Pid] <> CDvbPsiTableIdUndefined then
    Result := PidList.PacketFilter[Pid];
end;

{------------------------------------------------------------------------------
  Params  : <Psi>        Psi identifier
            <Filter>     Filter function which will process the data
  Returns : <Result>     True if success

  Descript: Set filter for Psi
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  PsiFilters[Psi] := Filter;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Psi>        Psi identifier
  Returns : <Result>     Assigned if valid Psi
                         Nil otherwise

  Descript: Get filter for Psi (only if valid Psi)
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := @PsiFilters[Psi];
end;

{------------------------------------------------------------------------------
  Params  : <ProgramNumber>  Program number
                             Index in list of available programs
                             A setting of $FF wil use <ServiceId> for search instead
            <ServiceId>      Program number (service identifier)
                             Real identifier
  Returns : <Result>         True if valid
            <ServiceNumber>  Program number (service identifier)
                             Real identifier
            <PmtPid>         PMT PID
            <PcrPid>         PCR PID
            <VideoPids>      Video PIDs
            <AudioPids>      Audio PIDs
            <TeletextPids>   Teletext PIDs
            <AudioLanguages> Language identifiers audio (capitals if AC3)
            <EcmPids>        ECM PIDs
            <CaIds>          CA System Ids
            <ProgramName>    Name of program

  Descript: Get PIDs of program
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetProgramInfo(var ProgramNumber: Byte;
  var ServiceId: Word;
  var PmtPid: Word;
  var PcrPid: Word;
  var VideoPids: TDvbPids;
  var AudioPids: TDvbPids;
  var TeletextPids: TDvbPids;
  var SubtitlePids: TDvbPids;
  var AudioLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids: TDvbPids;
  var CaIds: TDvbPids;
  var ProgramName: string): Boolean;
type
  Descriptors = record
    DescriptorType: Byte;
    DescriptorTag: Byte;
  end;
var
  ProcessProgram: Word;
  StreamsLoop: Word;
  Loop: Word;
  FullInfo: string;
  VideoIndex: Word;
  AudioIndex: Word;
  TeletextIndex: Word;
  EcmIndex: Word;
  CaIndex: Word;
  SubtitleIndex: Word;
  StreamType: Word;
  StreamDescriptorTag: Word;

  {------------------------------------------------------------------------------
    Params  : <StreamType>    Stream type
              <StreamSection> Pointer to stream data
              <StreamLength>  Length of stream sectionPointer to stream data
    Returns : -

    Descript: Check and save stream type/descriptor
    Notes   : Since a single stream might contain several tags we go through the
              whole lot.
   ------------------------------------------------------------------------------}
  procedure CheckStreamTypeAndDescriptor(StreamType: Word; StreamSection:
    PDvbSection; StreamLength: Word);
  var
    Index: Word;
    Length: Byte;
  begin
    if StreamLength = 0 then
      StreamSection := nil;
    if StreamSection = nil then
      StreamLength := 0;
    // The section migth contain multiple tag segments ...
    Index := 0;
    repeat
      if StreamSection = nil then
        StreamDescriptorTag := 0
      else
        StreamDescriptorTag := StreamSection[Index + 0];
      if StreamType = CDvbStreamTypeVideo then
      begin
        // Video found
        if (VideoIndex = 0) or
          (VideoPids[VideoIndex - 1] <>
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram]) then
        begin
          VideoPids[VideoIndex] :=
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram];
          Inc(VideoIndex);
        end;
      end;

      // If subtitle
      if ((StreamType = CDvbStreamTypeSubtitle) and
        (StreamDescriptorTag = CDvbTagSubtitleDescriptor)) then
      begin
        // One subtitle can contain different languages
        // At this point, although there is more information about the subtitle,
        // we don't do anything with it other than getting the language
        // Get length
        Length := StreamSection[Index + 1];
        // Point to language
        Inc(Index, 2);
        repeat
          SubtitlePids[SubtitleIndex] :=
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram];
          SubtitleLanguages[SubtitleIndex] := Chr(StreamSection[Index + 0]) +
            Chr(StreamSection[Index + 1]) +
            Chr(StreamSection[Index + 2]);
          // Index + 3   -> subtitling type
          // Index + 4/5 -> composition page id
          // Index + 6/7 -> ancilary page id
          Inc(SubtitleIndex);
          Inc(Index, 8);
        until (Index >= Length);
      end;

      // If AC3 sound or 'normal' sound
      if ((StreamType = CDvbStreamTypeAc3) and
        (StreamDescriptorTag = CDvbTagAc3Descriptor)) or
        ((StreamType = CDvbStreamTypeAudio) or
        (StreamType = CDvbStreamTypeAudio2)) then
      begin
        // Audio found
        if (AudioIndex = 0) or
          (AudioPids[AudioIndex - 1] <>
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram]) then
        begin
          AudioPids[AudioIndex] :=
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram];
          // Check for Audio language
          if StreamDescriptorTag = CDvbTagLanguage then
            AudioLanguages[AudioIndex] := Chr(StreamSection[Index + 2]) +
              Chr(StreamSection[Index + 3]) +
              Chr(StreamSection[Index + 4])
          else
            AudioLanguages[AudioIndex] := '---';
          // AC3 sound is indicated by capitals or '***' when no language was specified
          if ((StreamType = CDvbStreamTypeAc3) and
            (StreamDescriptorTag = CDvbTagAc3Descriptor)) then
          begin
            AudioLanguages[AudioIndex] := UpperCase(AudioLanguages[AudioIndex]);
            if AudioLanguages[AudioIndex] = '---' then
              AudioLanguages[AudioIndex] := '***';
          end
          else
            AudioLanguages[AudioIndex] := LowerCase(AudioLanguages[AudioIndex]);
          Inc(AudioIndex);
        end;
      end;

      if (StreamType = CDvbStreamTypeVideotext) and
        (StreamDescriptorTag = CDvbTagTeletext) then
      begin
        // Teletext found
        // Only informational, there is more data which could be returned
        if (TeletextIndex = 0) or
          (TeletextPids[TeletextIndex - 1] <>
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram]) then
        begin
          TeletextPids[TeletextIndex] :=
            PmtPrograms.Streams[Loop].ElementaryPid[ProcessProgram];
          Inc(TeletextIndex);
        end;
      end;

      if StreamDescriptorTag = CDvbTagCa then
      begin
        // Conditional access found
        // A CDvbTagCA in the PMT indicates ECM data
        if (EcmIndex = 0) or
          (EcmPids[EcmIndex - 1] <> ((StreamSection[Index + 4] and $1F) shl 8)
            or
          StreamSection[Index + 5]) then
        begin
          EcmPids[EcmIndex] := ((StreamSection[Index + 4] and $1F) shl 8) or
            StreamSection[Index + 5];
          Inc(EcmIndex);
          CaIds[CaIndex] := (StreamSection[Index + 2] shl 8) or
            StreamSection[Index + 3];
          Inc(CaIndex);
        end;
      end;
      // Increase by tag length
      if StreamSection = nil then
        Index := StreamLength
      else
        Inc(Index, StreamSection[Index + 1] + 2);
    until Index >= StreamLength;
  end;
begin
  try
    ResetLock.Acquire;
    try
      Result := False;
      ProcessProgram := ProgramNumber;
      VideoIndex := 0;
      AudioIndex := 0;
      TeletextIndex := 0;
      SubtitleIndex := 0;
      EcmIndex := 0;
      CaIndex := 0;
      PmtPid := 0;
      PcrPid := 0;
      // If we want to use <ServiceId> as source
      if ProcessProgram = $FF then
      begin
        ProcessProgram := 0;
        while (PmtPrograms.ProgramNumber[ProcessProgram] <> ServiceId) and
          (ProcessProgram < PatPrograms) do
          Inc(ProcessProgram);
      end;

      if ProcessProgram >= PatPrograms then
        Exit;
      // Return found program number for service if requested
      if ProgramNumber = $FF then
        ProgramNumber := ProcessProgram;
      PmtPid := PmtPrograms.Pid[ProcessProgram];
      PcrPid := PmtPrograms.PcrPid[ProcessProgram];
      ServiceId := PmtPrograms.ProgramNumber[ProcessProgram];

      StreamsLoop := PmtPrograms.NumberOfStreams[ProcessProgram];
      if StreamsLoop > 0 then
        for Loop := 0 to StreamsLoop - 1 do
        begin
          // Get stream type
          StreamType := PmtPrograms.Streams[Loop].StreamType[ProcessProgram];
          CheckStreamTypeAndDescriptor(StreamType,
            PmtPrograms.Streams[Loop].StreamDescriptor[ProcessProgram],
            PmtPrograms.Streams[Loop].StreamInfoLength[ProcessProgram])
        end;
      ProgramName := '';
      if PmtPrograms.SdtInfoLength[ProcessProgram] <> 0 then
        DvbGetService(PmtPrograms.SdtDescriptor[ProcessProgram], FullInfo,
          ProgramName);

      // Sometimes information is contained in the header of the PMT itself, and not
      // in the stream sections
      if PmtPrograms.InfoLength[ProcessProgram] > 0 then
      begin
        if PmtPrograms.InfoDescriptor[ProcessProgram][0] =
          CDvbTagStreamIdentifier then
        begin
          if ProgramName <> '' then
            ProgramName := '--Component tag--';
        end;
        CheckStreamTypeAndDescriptor(0,
          PmtPrograms.InfoDescriptor[ProcessProgram],
          PmtPrograms.InfoLength[ProcessProgram]);
      end;
      // End marker
      VideoPids[VideoIndex] := 0;
      AudioPids[AudioIndex] := 0;
      TeletextPids[TeletextIndex] := 0;
      SubtitlePids[SubtitleIndex] := 0;
      AudioLanguages[AudioIndex] := '   ';
      SubtitleLanguages[SubtitleIndex] := '   ';
      EcmPids[EcmIndex] := 0;
      CaIds[CaIndex] := 0;
      Result := True;
    except
      Result := False;
      //      MessageBox(null, 'Fatal error detected getting program information', 'DvbGetProgramInfo', MB_OK);
      MessageBox(0, 'Fatal error detected getting program information',
        'DvbGetProgramInfo', MB_OK);
    end;
  finally
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <ProgramNumber> Program number ($FF if using ServiceId)
            <ServiceId>     Service identifier
            <Present>       True if present information only
                            False if all information is requested
  Returns : <Result>        True if valid
            <Events>        Event information

  Descript: Get event information of program
  Notes   : We assume we have enough room for each single event
 ------------------------------------------------------------------------------}

function DvbFilterGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present:
  Boolean; Events: TDvbEventSections): Boolean;
var
  EventIndex: Word;
  Year: Word;
  Month: Word;
  Day: Word;
  Calc: Real;
  YCalc: Real;
  MCalc: Real;
  YInt: Word;
  MInt: Word;
  EitEventIndex: Word;
  Hours: Word;
  Minutes: Word;
  Seconds: Word;
  Index: Word;
  LocalIndex: Word;
  LocalLoop: Word;
  LocalEnd: Word;
  TableId: Word;
  LEvents: TDvbEitEvents;
  Descriptor: PDvbSection;
  LText: PAnsiString;
begin
  try
    ResetLock.Acquire;
    try
      Result := False;
      if not Assigned(Events) then
        Exit;
      EventIndex := 0;
      // If we want to use <ServiceId> as source
      if ProgramNumber = $FF then
      begin
        ProgramNumber := 0;
        while (PmtPrograms.ProgramNumber[ProgramNumber] <> ServiceId) and
          (ProgramNumber < PatPrograms) do
          Inc(ProgramNumber);
      end;
      if ProgramNumber >= PatPrograms then
        Exit;
      for TableId := CDvbPsiTableIdEitA to CDvbPsiTableIdEitAS + $0F do
        if Assigned(PmtPrograms.Events[ProgramNumber, TableId]) then
        begin
          LEvents := PmtPrograms.Events[ProgramNumber, TableId];
          if (LEvents.VersionNumber < $F0) and
            (LEvents.Event > 0) then
            for EitEventIndex := 0 to LEvents.Event - 1 do
            begin
              // 'Decode' the event information
              if LEvents.StartDate[EitEventIndex] = 0 then
                DecodeDate(Now, Year, Month, Day)
              else
              begin
                Calc := (LEvents.StartDate[EitEventIndex] - 15078.2) / 365.25;
                Year := Trunc(Calc);
                YCalc := Year * 365.25;
                YInt := Trunc(YCalc);
                Calc := LEvents.StartDate[EitEventIndex] - 14956.1 - YInt;
                Calc := Calc / 30.6001;
                Month := Trunc(Calc);
                MCalc := Month * 30.6001;
                MInt := Trunc(MCalc);
                Calc := LEvents.StartDate[EitEventIndex] - 14956 - YInt - MInt;
                Day := Trunc(Calc);
                if (Month = 14) or (Month = 15) then
                begin
                  Inc(Year);
                  Month := Month - 13;
                end
                else
                  Dec(Month);
                Year := Year + 1900;
              end;
              Hours := (((LEvents.StartTime[EitEventIndex] and $F00000) shr 20)
                * 10) +
                ((LEvents.StartTime[EitEventIndex] and $0F0000) shr 16);
              Minutes := (((LEvents.StartTime[EitEventIndex] and $00F000) shr 12)
                * 10) +
                ((LEvents.StartTime[EitEventIndex] and $000F00) shr 8);
              Seconds := (((LEvents.StartTime[EitEventIndex] and $0000F0) shr 4)
                * 10) +
                (LEvents.StartTime[EitEventIndex] and $00000F);
              // Sanity check
              if (Hours > 23) or (Minutes > 59) or (Seconds > 59) then
              begin
                Hours := 0;
                Minutes := 0;
                Seconds := 0;
              end;
              Events.StartTime[EventIndex] := EncodeDate(Year, Month, Day) +
                EncodeTime(Hours, Minutes, Seconds, 0);
              Hours := (((LEvents.Duration[EitEventIndex] and $F00000) shr 20) *
                10) +
                ((LEvents.Duration[EitEventIndex] and $0F0000) shr 16);
              Minutes := (((LEvents.Duration[EitEventIndex] and $00F000) shr 12)
                * 10) +
                ((LEvents.Duration[EitEventIndex] and $000F00) shr 8);
              Seconds := (((LEvents.Duration[EitEventIndex] and $0000F0) shr 4)
                * 10) +
                (LEvents.Duration[EitEventIndex] and $00000F);
              // Sanity check
              if (Hours > 23) or (Minutes > 59) or (Seconds > 59) then
              begin
                Hours := 0;
                Minutes := 0;
                Seconds := 0;
              end;
              Events.Duration[EventIndex] := EncodeTime(Hours, Minutes, Seconds,
                0);
              // Reset text count and make sure first one is marked as end
              Index := 0;
              if LEvents.DescriptorLength[EitEventIndex] > 0 then
                repeat
                  // Check type of descriptor (short or extended event)
                  Descriptor := LEvents.StreamDescriptor[EitEventIndex];
                  case Descriptor[Index + 0] of
                    CDvbTagShortEventDescriptor:
                      if Descriptor[Index + 1] > 4 then
                      begin
                        LocalIndex := Index + 5;
                        if Descriptor[LocalIndex] > 0 then
                        begin
                          New(LText);
                          LText^ := Chr(CDvbEventTypeShortName);
                          for LocalLoop := 1 to Descriptor[LocalIndex] do
                            LText^ := LText^ + Chr(Descriptor[LocalIndex +
                              LocalLoop]);
                          Events.EventText[EventIndex].Add(LText);
                        end;
                        LocalIndex := LocalIndex + Descriptor[LocalIndex] + 1;
                        if Descriptor[LocalIndex] > 0 then
                        begin
                          New(LText);
                          LText^ := Chr(CDvbEventTypeShortText);
                          for LocalLoop := 1 to Descriptor[LocalIndex] do
                            LText^ := LText^ + Chr(Descriptor[LocalIndex +
                              LocalLoop]);
                          Events.EventText[EventIndex].Add(LText);
                        end;
                      end;
                    CDvbTagExtendedEventDescriptor:
                      if Descriptor[Index + 1] > 4 then
                      begin
                        // First the individual items
                        LocalIndex := Index + 6;
                        LocalEnd := LocalIndex + Descriptor[LocalIndex];
                        // Only items if there are items (duh)
                        if Descriptor[LocalIndex] > 0 then
                        begin
                          Inc(LocalIndex);
                          repeat
                            if Descriptor[LocalIndex] > 0 then
                            begin
                              New(LText);
                              LText^ := Chr(CDvbEventTypeExtendedItemName);
                              for LocalLoop := 1 to Descriptor[LocalIndex] do
                                LText^ := LText^ + Chr(Descriptor[LocalIndex +
                                  LocalLoop]);
                              Events.EventText[EventIndex].Add(LText);
                            end;
                            LocalIndex := LocalIndex + Descriptor[LocalIndex] +
                              1;
                            if Descriptor[LocalIndex] > 0 then
                            begin
                              New(LText);
                              LText^ := Chr(CDvbEventTypeExtendedItemText);
                              for LocalLoop := 1 to Descriptor[LocalIndex] do
                                LText^ := LText^ + Chr(Descriptor[LocalIndex +
                                  LocalLoop]);
                              Events.EventText[EventIndex].Add(LText);
                            end;
                            LocalIndex := LocalIndex + Descriptor[LocalIndex] +
                              1;
                          until LocalIndex >= LocalEnd;
                        end;
                        // Now the individual text
                        New(LText);
                        LText^ := Chr(CDvbEventTypeExtendedText);
                        LocalIndex := LocalIndex + Descriptor[LocalIndex] + 1;
                        for LocalLoop := 1 to Descriptor[LocalIndex] do
                          LText^ := LText^ + Chr(Descriptor[LocalIndex +
                            LocalLoop]);
                        Events.EventText[EventIndex].Add(LText);
                      end;
                  end;
                  // Increase by descriptor length
                  Inc(Index, Descriptor[Index + 1] + 2);
                until (Index >= LEvents.DescriptorLength[EitEventIndex]);
              // Adjust event counter
              Inc(EventIndex);
            end;
          // If we only want present information (not the whole lot) then exit after first
          // table id run, which is the present information
          if Present then
            Break;
        end;
      Result := True;
    except
      Result := False;
      //      MessageBox(null, 'Fatal error detected getting event information', 'DvbGetEventInfo', MB_OK);
      MessageBox(0, 'Fatal error detected getting event information',
        'DvbGetEventInfo', MB_OK);
    end;
  finally
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Index>        Descriptor index to get
  Returns : <Result>       True if valid
            <SystemId>     System identifications
            <CaPid>        Ca PID (EMM system wide, data comes from PMT)

  Descript: Get EMM data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterGetEmm(Index: Integer; var CaSystemId: Word; var CaPid: Word):
  Boolean;
begin
  Result := False;
  if Index >= CaDescriptorQuantity then
    Exit;
  CaSystemId := CaDescriptors.CaSystemId[Index];
  CaPid := CaDescriptors.CaPid[Index];
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Generate CRC tables
  Notes   :
------------------------------------------------------------------------------}

procedure DvbGenerateCrcTables;
var
  ByteLoop: Byte;
  BitLoop: Byte;
  Crc16: Word;
  Crc32: Dword;
begin
  // Go through the whole array
  for ByteLoop := 0 to 255 do
  begin
    Crc16 := ByteLoop shl 8;
    Crc32 := ByteLoop shl 24;
    for BitLoop := 0 to 7 do
    begin
      if (Crc16 and $8000) <> 0 then
        Crc16 := ((Crc16 and $7FFF) shl 1) xor CDvbPolynomial16
      else
        Crc16 := (Crc16 shl 1);
      if (Crc32 and $80000000) <> 0 then
        Crc32 := ((Crc32 and $7FFFFFFF) shl 1) xor CDvbPolynomial32
      else
        Crc32 := (Crc32 shl 1);
      Crc16Table[ByteLoop] := Crc16;
      Crc32Table[ByteLoop] := Crc32;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc16: Word;
begin
  Result := CDvbCrcInitial16;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc32: Dword;
begin
  Result := CDvbCrcInitial32;
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC16 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial16> as
            starting value
------------------------------------------------------------------------------}

function DvbNewvalueCrc16(Checksum: Word; Data: Byte): Word;
begin
  Result := ((Checksum and $FF) shl 8) xor Crc16Table[((Checksum shr 8) xor Data)
    and $FF];
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC32 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial32> as
            starting value
------------------------------------------------------------------------------}

function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword;
begin
  Result := ((Checksum and $FFFFFF) shl 8) xor Crc32Table[((Checksum shr 24) xor
    Data) and $FF];
end;

initialization
  // The CRC tables are generated at startup
  DvbGenerateCrcTables;
  DvbNilTables;
  DvbFilterResetErrors;
  Debug := 0;
  ResetLock := TCriticalSection.Create;

finalization
  DvbDestroyTables;
  ResetLock.Free;

end.

