{******************************************************************************}
{ FileName............: DvbDirectShow                                          }
{ Project.............: DVB-SB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.10                                                   }
{------------------------------------------------------------------------------}
{  DirectShow interface                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Graph is also added to the Running Object Table so it can be connected to    }
{ when running (typically for debugging purposes)                              }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  1.00   20031113 - Initial release                                           }
{  1.01   20040213 - Added 'NoRender' parameter                                }
{  1.02   20040218 - Cleanup: might have introduced spurious runtime error at  }
{                    application ending                                        }
{                  - External access to graph (AddToRot) made variable         }
{                  - LogFile handle reset prior to destruction                 }
{  1.05   20040320 - Added <MajorDvbPsi> DirectShow filter support             }
{                    (V2.01/V2.02)                                             }
{                  - All <DvbFilter> calls are through this unit               }
{                  - Added windowless mode                                     }
{                  - If no .GRF file given the graph is created manually       }
{  1.06   20040605 - Additional Try/Except added                               }
{  1.07   20040815 - ServiceId in <..GetProgramInfo> added                     }
{                  - Added <DirectShowResize>                                  }
{  1.08   20040828 - Added CAT callback                                        }
{                  - Added PAT callback                                        }
{                  - Removed support for PsiFilter (too much differences)      }
{  1.09   20040920 - VMR9 instead of windowless mode plus some additional VRM  }
{                    stuff (eg. 'OSD')                                         }
{  1.10   20041003 - Retrieval if used filters added                           }
{                  - Added Property page detection/showing                     }                     
{******************************************************************************}
unit DvbDirectShow;

interface
uses
  Controls,
  DirectShow9,
  DvbFilter,
  ExtCtrls,
  Graphics,
  SysUtils,
  Windows;


{------------------------------------------------------------------------------
  Descript: DirectShow basic functions
 ------------------------------------------------------------------------------}
  function  DirectShowStart(Owner           : HWND;
                            GraphicsFile    : string;
                            UseVRM          : Boolean;
                            DoNotRender     : Boolean;
                            AllowAddToRot   : Boolean;
                            var ErrorMessage: string;
                            LogFile         : THandle): Boolean;
  procedure DirectShowStop;
  procedure DirectShowFullScreenMode(FullScreen: Boolean);
  procedure DirectShowSetVideo(Video: HWND);
  procedure DirectShowResize;
  function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
  function  DirectShowGetFilterName(Index: Integer): string;
  function  DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;

{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer/universal source filter functions
 ------------------------------------------------------------------------------}
  procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
  procedure DirectShowUniversalSourceSendData     (Buffer: PChar; BufferLength: Integer);

{------------------------------------------------------------------------------
  Descript: DvbFilter functions
  Note    : These functions are passed onto <DvbFilter>
 ------------------------------------------------------------------------------}
  function  DvbGetPid                    (StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
  function  DvbSetSignallingCallback     (Pid: Word; hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackPat  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackCat  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackPmt  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetPidFilter              (Pid: Word; Filter: TDvbPacketFilter): Boolean;
  function  DvbGetPidFilter              (Pid: Word): TDvbPacketFilter;
  function  DvbSetPsiFilter              (Psi: Byte; Filter: TDvbSectionFilter): Boolean;
  function  DvbGetPsiFilter              (Psi: Byte): TDvbSectionFilter;
  function  DvbGetProgramInfo(ProgramNumber: Byte;
                              var ServiceId: Word;
                              var PmtPid: Word;
                              var PcrPid: Word;
                              var VideoPids: TDvbPids;
                              var AudioPids: TDvbPids;
                              var TeletextPids: TDvbPids;
                              var SubtitlePids: TDvbPids;
                              var AudioLanguages: TDvbLanguages;
                              var SubtitleLanguages: TDvbLanguages;
                              var EcmPids: TDvbPids;
                              var CaIds: TDvbPids;
                              var ProgramName: string): Boolean;
  function  DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
  function  DvbGetNumberOfPrograms: Byte;
  function  DvbGetErrors: Word;
  function  DvbGetPacketSyncErrors: Word;
  procedure DvbResetErrors;
  procedure DvbCreateTables;


implementation
uses
  ActiveX,
  Dialogs,
  Forms;

const
{------------------------------------------------------------------------------
  Descript: DirectShow identifiers
 ------------------------------------------------------------------------------}
  IID_IUSRCSet                      = '{AE1A2884-540E-4077-B1AB-67A34A72299F}';
  T_IID_IUSRCSet            : TGUID = IID_IUSRCSet;
  CLSID_IUSRCSet            : TGUID = '{FD501041-8888-1111-9153-00AB00577DA2}';
  IID_IPersistStream        : TGUID = '{00000109-0000-0000-C000-000000000046}';

  CLSID_MPEG2Demultiplexer  : TGUID = (D1:$afb6c280;D2:$2c41;D3:$11d3;D4:($8a,$60,$00,$00,$f8,$1e,$0e,$4a));

  IID_ISpecifyPropertyPages : TGUID = '{B196B28B-BAB4-101A-B69C-00AA00341D07}';  // See DsUtil


type
{------------------------------------------------------------------------------
  Descript: Universal Source filter interface
 ------------------------------------------------------------------------------}
  IUSRCSet = interface(IUnknown)
  [IID_IUSRCSet]
    function WSetMediaType(Pmt   : PAMMEDIATYPE): HRESULT; stdcall;
    function WSetBufSize  (Size  : Dword): HRESULT; stdcall;
    function WSendSample  (Buffer: PChar; BufferLength: Dword): HRESULT; stdcall;
  end;

{------------------------------------------------------------------------------
  Descript: The DirectShow object
 ------------------------------------------------------------------------------}
  TDirectShowGraph = class(TObject)
    FGraph           : IFilterGraph2;
    FUseVRM          : Boolean;
    FOwner           : HWND;
    FVMR9Filter      : IBaseFilter;
    FAlphaBitmap     : VMR9AlphaBitmap;
    FMPEG2Demux      : IBaseFilter;
    FUniversalSource : IBaseFilter;
    FVideoPin        : IPin;
    FAudioPin        : IPin;
    FActiveAudioPid  : Integer;
    FActiveVideoPid  : Integer;
    FRotEntry        : LongInt;
    constructor Create(Owner: HWND);
    destructor  Destroy; override;
  private
    { Private declarations }
    function BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
  public
    { Public declarations }
  end;

var
  DirectShowGraph: TDirectShowGraph;


{------------------------------------------------------------------------------
  Params  : <UnknownGraph>  Graph to add to ROT
  Returns : <Result>        Result (S_OK if success)
            <RotEntry>      Assigned number in ROT

  Descript: Add graph to Running Object Table so we can connect to it when
            it is running.
  Notes   :
 ------------------------------------------------------------------------------}
function AddToRot(UnknownGraph: IUnknown; var RotEntry: LongInt): HRESULT;
var
  Moniker: IMoniker;
  Rot    : IRunningObjectTable;
  Wsz    : WideString;
  Wsz2   : WideString;
  Hr     : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, ROT);
    if Failed(Hr) then
    begin
      Result := E_FAIL;
      Exit;
    end;
    Wsz  := format('FilterGraph %p pid %8.8x', [Pointer(UnknownGraph), GetCurrentProcessId]);
    Wsz2 := '!';
    Hr := CreateItemMoniker(@Wsz2[1], @Wsz[1], Moniker);
    if Succeeded(Hr) then
    begin
      Hr := ROT.Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, UnknownGraph,
              Moniker, RotEntry);
    end;
    Result := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <RotEntry>      Assigned number in ROT
  Returns : -

  Descript: Remove graph from Running Object Table
  Notes   :
 ------------------------------------------------------------------------------}
procedure RemoveFromRot(RotEntry: LongInt);
var
  Rot: IRunningObjectTable ;
  Hr : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, Rot);
    if Succeeded(Hr) then
      Rot.Revoke(RotEntry);
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <pGraph>    Pointer to graph builder object
            <FileName>  Name of graph file
  Returns : -

  Descript: Load a DirectShow graph (.GRF) file
  Notes   :
 ------------------------------------------------------------------------------}
function LoadGraphFile(pGraph: IFilterGraph2; FileName: string): HRESULT;
var
  PStorage      : IStorage;
  PPersistStream: IPersistStream;
  PStream       : IStream;
  WideName      : array[0..1023] of WideChar;
  Hr            : HRESULT;
begin
  try
    PStorage       := nil;
    PPersistStream := nil;
    PStream        := nil;
    StringToWideChar(FileName, WideName, SizeOf(WideName));
    hr := StgIsStorageFile(@WideName);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    hr := StgOpenStorage(@WideName, nil, STGM_TRANSACTED or STGM_READ or STGM_SHARE_DENY_WRITE,
            nil, 0, PStorage);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    Hr := pGraph.QueryInterface(IID_IPersistStream, PPersistStream);
    if not Failed(Hr) then
    begin
      Hr := PStorage.OpenStream('ActiveMovieGraph', nil,
              STGM_READ or STGM_SHARE_EXCLUSIVE, 0, PStream);
      if not Failed(Hr) then
      begin
        Hr := PPersistStream.Load(PStream);
        PStream := nil;
      end;
      PPersistStream := nil;
    end;
    PStorage := nil;
    Result  := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Owner>           Handle to window for video
            <GraphicsFile>    .GRF file to use
                              If empty the graph is manually created
            <UseVRM>          Will use VRM9 if TRUE                  
            <DoNotRender>     TRUE will not render audio/video pins: the graph is
                              not automatically 'completed'
            <AllowAddToRot>   TRUE will add graph toi table so it can be remotely
                              connected to
            <LogFile>         Handle for log file
  Returns : <Result>          TRUE if success, FALSE if failure
            <ErrorMessage>    Error message (if any)

  Descript: Setup direct show graph
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowStart(Owner           : HWND;
                          GraphicsFile    : string;
                          UseVRM          : Boolean;
                          DoNotRender     : Boolean;
                          AllowAddToRot   : Boolean;
                          var ErrorMessage: string;
                          LogFile         : THandle): Boolean;
var
  Hr                          : HRESULT;
  Pins                        : IPin;
  PinsIn                      : IPin;
  PinsOut                     : IPin;
  PinsEnum                    : IEnumPins;
  PinsInfo                    : TPinInfo;
  MediaType                   : AM_MEDIA_TYPE;
  MPEG2Filter                 : IBaseFilter;
  USRCFilter                  : IBaseFilter;
  MPEG2DemultiplexerInterface : IMPEG2Demultiplexer;
  DestinationRect             : TRect;

  {------------------------------------------------------------------------------
    Params  : <Graph>           Graph to render pin in
              <FilterName>      Filter which has the pin
              <PinName>         Name of the pin to render
    Returns : <Result>          TRUE if success, FALSE if failure

    Descript: Render a pin
    Notes   :
   ------------------------------------------------------------------------------}
  function RenderPin(Graph: IFilterGraph2; FilterName: WideString; PinName: WideString): Boolean;
  var
    Hr       : HRESULT;
    Filter   : IBaseFilter;
    EnumPins : IEnumPins;
    Pin      : IPin;
    pInfo    : TPinInfo;
  begin
    Result := False;
    Hr := Graph.FindFilterByName(PWideChar(FilterName), Filter);
    if not Failed(Hr) then
    begin
      Filter.EnumPins(EnumPins);
      // Go through all pins until we find it
      while not Failed(EnumPins.Next(1, Pin, nil)) do
      begin
        Pin.QueryPinInfo(pInfo);
        if pInfo.achName = PinName then
        begin
          // Try using RenderEx if possible (only succeeds for video pins ....)
          Hr := Graph.RenderEx(Pin, AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
          if Failed(Hr) then
            Hr := Graph.Render(Pin);
          // Failing of rendering can be normal since it might be already connected ...
          // Reconnecting it shoudl succeed though
          if Failed(Hr) then
            Hr := Graph.ReconnectEx(Pin, nil);
          if Failed(Hr) then
            Result := False
          else
            Result := True;
          Exit;
        end;
      end;
    end;
  end;

begin
  try
    // Make sure a previously started DirectShow interface is stopped and released
    DirectShowStop;
    ErrorMessage := '';
    Result := False;
    DirectShowGraph := TDirectShowGraph.Create(Owner);
    try
      DirectShowGraph.FUseVRM := UseVRM;
      Hr := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER, IID_IFilterGraph2, DirectShowGraph.FGraph);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter graph creation error.';
        Exit;
      end;
      if UseVRM then
      begin
        Hr := CoCreateInstance(CLSID_VideoMixingRenderer9, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, DirectShowGraph.FVMR9Filter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Video mixing renderer 9 error.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraph.AddFilter(DirectShowGraph.FVMR9Filter, 'VMR9');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Adding Video Mixing Renderer filter failed.';
          Exit;
        end;
      end;
      // Add to ROT if requested
      if AllowAddToRot then
      begin
        Hr := AddToRot(DirectShowGraph.FGraph, DirectShowGraph.FRotEntry);
        if Failed(Hr) then
        begin
          DirectShowGraph.FRotEntry := -1;
        end;
      end;
      if UseVRM then
      begin
        (DirectShowGraph.FVMR9Filter as IVMRFilterConfig9).SetRenderingMode(VMR9Mode_Windowless);
        (DirectShowGraph.FVMR9Filter as IVMRWindowlessControl9).SetVideoClippingWindow(DirectShowGraph.FOwner);
      end;

      // Load the file in the filter graph and
      if FileExists(GraphicsFile) then
      begin
        {------------------------------------------------------------------------------
          Descript: Loading a graph file
         ------------------------------------------------------------------------------}
        Hr := LoadGraphFile(DirectShowGraph.FGraph, GraphicsFile);
        if Failed(Hr) then
        begin
          ErrorMessage := GraphicsFile + ' file not loaded (missing/invalid/incorrect).';
          Exit;
        end;
      end
      else
      begin
        {------------------------------------------------------------------------------
          Descript: Building a graph manually
         ------------------------------------------------------------------------------}
        // No graph filter file: create the graph manually (at least the major parts)
        // First the universal source
        Hr := CoCreateInstance(CLSID_IUSRCSet, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, USRCFilter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not use the Universal Source filter.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraph.AddFilter(USRCFilter, 'Universal Source');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not add Universal Source filter.';
          Exit;
        end;
        // Then the MPEG-2 demultiplexer
        Hr := CoCreateInstance(CLSID_MPEG2Demultiplexer, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, MPEG2Filter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not use the MPEG-2 Demultiplexer filter.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraph.AddFilter(MPEG2Filter, 'MPEG-2 Demultiplexer');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not add MPEG-2 Demultiplexer filter.';
          Exit;
        end;
        // Now we must create the output pins on the MPEG-2 demultiplexer
        // The video pin
        Hr := MPEG2Filter.QueryInterface(IID_IMpeg2Demultiplexer, MPEG2DemultiplexerInterface);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter MPEG-2 Demultiplexer interface not found.';
          Exit;
        end;
        MediaType.MajorType := MEDIATYPE_VIDEO;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_VIDEO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Video', DirectShowGraph.FVideoPin);
        if Failed(Hr) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the video output pin for the MPEG-2 Demultiplexer.';
          Exit;
        end;
        // The audio pin
        MediaType.MajorType := MEDIATYPE_AUDIO;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_AUDIO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Audio', DirectShowGraph.FAudioPin);
        if Failed(Hr) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the audio output pin for the MPEG-2 Demultiplexer.';
          Exit;
        end;
        MPEG2DemultiplexerInterface := nil;
        PinsIn  := nil;
        PinsOut := nil;


        // Enumerate the pins
        Hr := USRCFilter.EnumPins(PinsEnum);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not enumerate the Universal Source pins.';
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          if UpperCase(PinsInfo.achName) = 'OUTPUT' then
            PinsOut := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        Hr := MPEG2Filter.EnumPins(PinsEnum);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not enumerate the MPEG-2 Demultiplexer pins.';
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          if UpperCase(PinsInfo.achName) = 'MPEG-2 STREAM' then
            PinsIn := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        // Connect the pins
        if not Assigned(PinsIn) or not Assigned(PinsOut) then
        begin
          PinsIn  := nil;
          PinsOut := nil;
          ErrorMessage := 'Could not find correct pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraph.Connect(PinsOut, PinsIn);
        PinsIn  := nil;
        PinsOut := nil;
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not connect pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
        MPEG2Filter := nil;
        USRCFilter  := nil;



        {------------------------------------------------------------------------------
          Descript: Building a graph manually end
         ------------------------------------------------------------------------------}
      end;
      // Check for required components
      Hr := DirectShowGraph.FGraph.FindFilterByName('MPEG-2 Demultiplexer', DirectShowGraph.FMPEG2Demux);
      if Failed(Hr) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer filter not found.';
        Exit;
      end;
      Hr := DirectShowGraph.FMPEG2Demux.FindPin('Video', DirectShowGraph.FVideoPin);
      if Failed(Hr) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer video pin not found.';
        Exit;
      end;
      Hr := DirectShowGraph.FMPEG2Demux.FindPin('Audio', DirectShowGraph.FAudioPin);
      if Failed(Hr) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer audio pin not found.';
        Exit;
      end;
      if not DoNotRender then
      begin
        if not RenderPin(DirectShowGraph.FGraph, 'MPEG-2 Demultiplexer', 'Audio') then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer audio pin rendering error.';
          Exit;
        end;
        if not RenderPin(DirectShowGraph.FGraph, 'MPEG-2 Demultiplexer', 'Video') then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer video pin rendering error.';
          Exit;
        end;
      end;
      Hr := DirectShowGraph.FGraph.FindFilterByName('Universal Source', DirectShowGraph.FUniversalSource);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Universal Source filter not found.';
        Exit;
      end;

      DirectShowGraph.FMPEG2Demux.SetSyncSource(nil);
      DirectShowGraph.FGraph.SetDefaultSyncSource;


      if not UseVRM then
      begin
        Hr := (DirectShowGraph.FGraph as IVideoWindow).put_Owner(DirectShowGraph.FOwner);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set video handle.';
          Exit;
        end;
        Hr := (DirectShowGraph.FGraph as IVideoWindow).put_MessageDrain(DirectShowGraph.FOwner);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set messages handle.';
          Exit;
        end;
        Hr := (DirectShowGraph.FGraph as IVideoWindow).put_WindowStyle(WS_CHILD or WS_CLIPSIBLINGS or WS_CLIPCHILDREN);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set window style.';
          Exit;
        end;

        GetClientRect(DirectShowGraph.FOwner, DestinationRect);
        Hr := (DirectShowGraph.FGraph as IVideoWindow).SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set window size.';
          Exit;
        end;
      end;

      // Indicate success
      Result := True;
      (DirectShowGraph.FGraph as IMediaControl).Run;
    finally
      // Make sure that at a failure all resources are freed
      if not Result then
        DirectShowStop;
    end;
  except
    Result := False;
    DirectShowStop;
    ShowMessage('DirectShow Start error');
  end;
end;


function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
begin
  Result := False;
  if not Assigned(DirectShowGraph) then
    Exit;
  if not DirectShowGraph.FUseVRM then
    Exit;
  Result := DirectShowGraph.BlendApplicationImage(Bitmap, Src, Blend, ColorKey);
end;


{------------------------------------------------------------------------------
  Params  : <Owner>  Handle to window for video
  Returns : -

  Descript: Constructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TDirectShowGraph.Create(Owner: HWND);
begin
  inherited Create;
  FOwner           := Owner;
  FRotEntry        := -1;
  FGraph           := nil;
  FVMR9Filter      := nil;
  FMPEG2Demux      := nil;
  FUniversalSource := nil;
  FVideoPin        := nil;
  FAudioPin        := nil;
  FActiveAudioPid  := -1;
  FActiveVideoPid  := -1;
  ZeroMemory(@FAlphaBitmap, sizeof(FAlphaBitmap));
  FAlphaBitmap.Hdc := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
destructor TDirectShowGraph.Destroy;
begin
  if Assigned(FGraph) then
  begin
    try
      (FGraph as IMediaControl).Stop;
      FGraph.Abort;
      FGraph.SetLogFile(0);
      if FRotEntry <> -1 then
        RemoveFromRot(FRotEntry);
      // Important to stop the message drain, otherwise errors occur

      FRotEntry        := -1;
      FGraph           := nil;
      FVMR9Filter      := nil;
      FMPEG2Demux      := nil;
      FUniversalSource := nil;
      FVideoPin        := nil;
      FAudioPin        := nil;
      FActiveAudioPid  := -1;
      FActiveVideoPid  := -1;
    except
    end;
  end;
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : <Bitmap>    Bitmap to blend in
            <Dest>      Source position/size
            <Blend>     Blending value (0.0...1.0 == transparent..opaque)
            <ColorKey>  Transparancy color
  Returns : <Result>  True if success

  Descript: Blend in application image
  Notes   :
 ------------------------------------------------------------------------------}
function TDirectShowGraph.BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
begin
  Result := True;
  with DirectShowGraph.FAlphaBitmap do
  begin
    Hdc          := Bitmap;
    rSrc         := Src;
    rDest.Left   := 0;
    rDest.Top    := 0;
    rDest.Right  := 1;
    rDest.Bottom := 1;
    dwFlags   := VMRBITMAP_SRCCOLORKEY or VMRBITMAP_HDC;
    clrSrcKey := ColorKey;
    fAlpha    := Blend;
  end;
  (DirectShowGraph.FVMR9Filter as IVMRMixerBitmap9).SetAlphaBitmap(@DirectShowGraph.FAlphaBitmap);
end;


{------------------------------------------------------------------------------
  Params  : <Index>  Filter index (0 = first)
  Returns : <Result> Name of filter (empty if no filter at index)

  Descript: Get filter name
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowGetFilterName(Index: Integer): string;
var
  EnumFilters: IEnumFilters;
  BaseFilter : IBaseFilter;
  FilterInfo : TFilterInfo;
  FilterIndex: Integer;
begin
  Result := '';
  if not Assigned(DirectShowGraph) then
    Exit;
  try
    DirectShowGraph.FGraph.EnumFilters(EnumFilters);
    FilterIndex := -1;
    while (EnumFilters.Next(1, BaseFilter, nil) = S_OK) and
          (FilterIndex < Index) do
    begin
      BaseFilter.QueryFilterInfo(FilterInfo);
      Result := FilterInfo.achName;
      Inc(FilterIndex);
    end;
    // Make sure an incorrect index returns nothing
    if FilterIndex <> Index then
      Result := '';
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Parent>      Handle of parent to create property page in
            <FilterName>  Name of filter
            <QueryOnly>   True if only to check for property page (not to show)
  Returns : <Result>      True if property page available

  Descript: Show property page of filter
  Notes   :
 ------------------------------------------------------------------------------}
function DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;
var
  BaseFilter          : IBaseFilter;
  SpecifyPropertyPages: ISpecifyPropertyPages;
  CAGUID              : TCAGUID;
  FilterInfo          : TFilterInfo;
  Hr                  : HRESULT;
begin
  Result := False;
  if not Assigned(DirectShowGraph) then
    Exit;
  try
    // Get filter using the name
    Hr := DirectShowGraph.FGraph.FindFilterByName(PWideChar(FilterName), BaseFilter);
    if not Failed(Hr) then
    begin
  //    ZeroMemory(@FilterInfo, SizeOf(TFilterInfo));
      // Get property pages
      Hr := BaseFilter.QueryInterface(IID_ISpecifyPropertyPages, SpecifyPropertyPages);
      if not Failed(Hr) then
      begin
        // Get GUID
        Hr := SpecifyPropertyPages.GetPages(CAGUID);
        if not Failed(Hr) then
        begin
          // Check number of pages available
          if CAGUID.cElems < 1 then
            Exit;
          if not QueryOnly then
          begin
            // Get info
            Hr := BaseFilter.QueryFilterInfo(FilterInfo);
            if not Failed(Hr) then
            begin
              Hr := OleCreatePropertyFrame(Parent, 0, 0, FilterInfo.achName, 1, @BaseFilter, CAGUID.cElems, CAGUID.pElems, 0, 0, nil);
              if not Failed(Hr) then
                Result := True;
            end;
          end
          else
            Result := True;    
        end;
      end;
    end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Stop DirectShow interface
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowStop;
begin
  if Assigned(DirectShowGraph) then
    FreeAndNil(DirectShowGraph);
end;


{------------------------------------------------------------------------------
  Params  : <FullScreen>  True to set full screen mode
                          False to restore original mode
  Returns : -

  Descript: Set full screen mode
  Notes   : Only for non-VMR9 modes
 ------------------------------------------------------------------------------}
procedure DirectShowFullScreenMode(FullScreen: Boolean);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    // Only for non-VMR9 mode
    (DirectShowGraph.FGraph as IVideoWindow).put_FullScreenMode(FullScreen);
  except
    DirectShowStop;
    ShowMessage('DirectShow FullScreenMode error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Video>  Video window
  Returns : -

  Descript: Set video window
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowSetVideo(Video: HWND);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    if not DirectShowGraph.FUseVRM then
      Exit;
    DirectShowGraph.FOwner := Video;
    (DirectShowGraph.FVMR9Filter as IVMRWindowlessControl9).SetVideoClippingWindow(DirectShowGraph.FOwner);
  except
    DirectShowStop;
    ShowMessage('DirectShow SetVideo error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Resize to full client area
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowResize;
var
  DestinationRect: TRect;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    GetWindowRect(DirectShowGraph.FOwner, DestinationRect);
    DestinationRect.Right  := DestinationRect.Right  - DestinationRect.Left;
    DestinationRect.Bottom := DestinationRect.Bottom - DestinationRect.Top;
    DestinationRect.Top := 0;
    DestinationRect.Left := 0;
    if DirectShowGraph.FUseVRM then
    begin
      if not Assigned(DirectShowGraph.FVMR9Filter) then
        Exit;
      (DirectShowGraph.FVMR9Filter as IVMRWindowlessControl9).SetVideoPosition(nil, @DestinationRect);
    end
    else
      (DirectShowGraph.FGraph as IVideoWindow).SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
  except
    DirectShowStop;
    ShowMessage('DirectShow Resize error');
  end;
end;



{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <VideoPid>  Video PID
            <AudioPid>  Audio PID
  Returns : -

  Descript: Set new PIDs to use for MPEG2 demultiplexer
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
var
  MapPid: Cardinal;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;

    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      (DirectShowGraph.FVideoPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      (DirectShowGraph.FAudioPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;

    // Set new active PIDs
    DirectShowGraph.FActiveVideoPid := VideoPid;
    DirectShowGraph.FActiveAudioPid := AudioPid;

    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      (DirectShowGraph.FVideoPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      (DirectShowGraph.FAudioPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;

  except
    DirectShowStop;
    ShowMessage('DirectShow Mpeg2DemultiplexerSetNewPids error');
  end;
end;


{------------------------------------------------------------------------------
  Descript: DirectShow Universal Source filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <Buffer>        Pointer to data
            <BufferLength>  Length of valid data in buffer
  Returns : -

  Descript: Send data to Universal Source filter
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowUniversalSourceSendData(Buffer: PChar; BufferLength: Integer);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    (DirectShowGraph.FUniversalSource as IUSRCSet).WSendSample(Buffer, BufferLength);
  except
    DirectShowStop;
    ShowMessage('DirectShow UniversalSourceSendData');
  end;
end;


{------------------------------------------------------------------------------
  Descript: <DvbFilter> functions: Either interanl function called or
            <DvbMajorPsi> DirectShow filter ios called
  Note    : For descriptions of functions/procedures see <DvbFilter>
 ------------------------------------------------------------------------------}
function DvbGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
begin
  Result := DvbFilter.DvbFilterGetPid(StreamPacketData, Pid);
end;


function  DvbSetSignallingCallback     (Pid: Word;  hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallback(Pid, hWnd, Msg);
end;


function  DvbSetSignallingCallbackPat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackCat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackCat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackPmt  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPmt  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackEvent( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackEvent(hWnd, Msg);
end;


function  DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPidFilter(Pid, Filter);
end;


function  DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := DvbFilter.DvbFilterGetPidFilter(Pid);
end;


function  DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPsiFilter(Psi, Filter);
end;


function  DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := DvbFilter.DvbFilterGetPsiFilter(Psi);
end;


function  DvbGetProgramInfo(ProgramNumber: Byte;
                            var ServiceId: Word;
                            var PmtPid: Word;
                            var PcrPid: Word;
                            var VideoPids: TDvbPids;
                            var AudioPids: TDvbPids;
                            var TeletextPids: TDvbPids;
                            var SubtitlePids: TDvbPids;
                            var AudioLanguages: TDvbLanguages;
                            var SubtitleLanguages: TDvbLanguages;
                            var EcmPids: TDvbPids;
                            var CaIds: TDvbPids;
                            var ProgramName: string): Boolean;
begin
  Result := DvbFilter.DvbFilterGetProgramInfo(ProgramNumber, ServiceId,
              PmtPid, PcrPid, VideoPids, AudioPids,
              TeletextPids, SubtitlePids, AudioLanguages,
              SubtitleLanguages, EcmPids, CaIds, ProgramName);
end;


function  DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
begin
  Result := DvbFilter.DvbFilterGetEventInfo(ProgramNumber, ServiceId, Present, Events);
end;


function  DvbGetNumberOfPrograms: Byte;
begin
  Result := DvbFilter.DvbFilterGetNumberOfPrograms;
end;


function  DvbGetErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetErrors;
end;


function  DvbGetPacketSyncErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetPacketSyncErrors;
end;


procedure DvbResetErrors;
begin
  DvbFilter.DvbFilterResetErrors;
end;


procedure DvbCreateTables;
begin
  DvbFilter.DvbFilterCreateTables;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}
procedure InitializeUnit;
begin
  DirectShowGraph := nil;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization
  Notes   :
 ------------------------------------------------------------------------------}
procedure FinalizeUnit;
begin
  DirectShowStop;
end;


initialization
  InitializeUnit;

finalization
  FinalizeUnit;
end.
