{******************************************************************************}
{ FileName............: PidProcessingDataUnit001                               }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Buffering / Pid processing / Showing streams.                               }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ The default settings should provide you with a video/audio of the german ARD }
{ program being transmitted (select program 01 for this first). It is assumed  }
{ the dish is facing Astra 19.2E and that the PIDs have not changed....        }                                           
{ If no video/audio is shown (without an error being reported) then check the  }
{ also the graphics setup (see note below).                                    }
{                                                                              }
{ Note: Master / Slave operation for DirectShow currently generates an error   }
{       at the second instance (slave)                                         }
{       Use 'master' as command line parameter to force being a master,        }
{       otherwise this is detected automaticly                                 }
{       Use 'slave' as command line parameter to force being a slave,          }
{       otherwise this is detected automaticly                                 }
{ Note: For addressing the SU1278 tuner instead of the BSRU6 use '1' as second }
{       parameter                                                              }
{ Note: Only one DirectShow per process can be run ....                        }
{ Note: Debug not always possible because of DirectShow stuff ...              }
{ Note: Stream decoding not active ...                                         }
{ Note: The supplied VIDEO.GRF has no connected output pins. This is not       }
{       absolute necessary because these are automatically connected to the    }
{       'correct' video/audio filters.                                         }
{       These video/audio filters are typically part of a DVD viewer (like     }
{       InterVideo / PowerDVD etc. You can also connect these pins as required }
{       so they are forced to a specific connection. You need the FilterGraph  }
{       from DirectX 9 for this.                                               }
{                                                                              }
{ Note: Some DirectShow Filters DO NO ALLOW DEBUGGING. You typically get an    }
{       error '$FC' when DirectShow is being initialized. If you set debugging }
{       off then you don't have a problem.                                     }
{       Some audio decoders do not correctly decode. You get either a runtime  }
{       error or a Mickey Mouse sound on some audio channels....               }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030905 - Initial release                                           }
{******************************************************************************}
unit PidProcessingDataUnit001;

interface
uses
  Forms, ExtCtrls, StdCtrls, Mask, Controls, Classes, Gauges, ComCtrls,
  Messages, Buttons, Graphics;

const
  WM_SIGNALLING = WM_USER+10;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    tmrUpdate: TTimer;
    pnlVideo: TPanel;
    tmrNewChannel: TTimer;
    imgLeft: TImage;
    imgRight: TImage;
    imgDown: TImage;
    imgUp: TImage;
    imgEnter: TImage;
    pgInfo: TPageControl;
    tsTuning: TTabSheet;
    rgPolarity: TRadioGroup;
    mskFrequency: TMaskEdit;
    mskSymbolRate: TMaskEdit;
    tsProgram: TTabSheet;
    tsMessages: TTabSheet;
    mmoDriver: TMemo;
    tsDebug: TTabSheet;
    Label2: TLabel;
    Label3: TLabel;
    StaticText5: TStaticText;
    txtSignalPower: TStaticText;
    StaticText6: TStaticText;
    txtSignalState: TStaticText;
    StaticText8: TStaticText;
    txtNoiseIndicator: TStaticText;
    StaticText29: TStaticText;
    StaticText30: TStaticText;
    txtOvertaken: TStaticText;
    txtPacketCount: TStaticText;
    StaticText28: TStaticText;
    txtMs: TStaticText;
    StaticText27: TStaticText;
    mskVideoPid: TMaskEdit;
    mskAudioPid: TMaskEdit;
    chkVideo: TCheckBox;
    chkPids: TCheckBox;
    Label5: TLabel;
    Label6: TLabel;
    cmbTransponder: TComboBox;
    Label7: TLabel;
    mskProgram: TMaskEdit;
    txtMaxPrograms: TLabel;
    Label8: TLabel;
    imgRecord: TImage;
    imgRecord2: TImage;
    txtVideoRate: TLabel;
    Label1: TLabel;
    Label4: TLabel;
    btnDebug: TButton;
    lblDebug: TLabel;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure mskPidChange(Sender: TObject);
    procedure mskPidKeyPress(Sender: TObject; var Key: Char);
    procedure FormDestroy(Sender: TObject);
    procedure chkVideoClick(Sender: TObject);
    procedure udProgramClick(Sender: TObject; Button: TUDBtnType);
    procedure chkPidsClick(Sender: TObject);
    procedure mskProgramChange(Sender: TObject);
    procedure mskProgramKeyPress(Sender: TObject; var Key: Char);
    procedure cmbTransponderChange(Sender: TObject);
    procedure udTransponderClick(Sender: TObject; Button: TUDBtnType);
    procedure tmrNewChannelTimer(Sender: TObject);
    procedure imgUpClick(Sender: TObject);
    procedure imgDownClick(Sender: TObject);
    procedure imgLeftClick(Sender: TObject);
    procedure imgRightClick(Sender: TObject);
    procedure imgRecordClick(Sender: TObject);
    procedure btnDebugClick(Sender: TObject);
  private
    { Private declarations }
    procedure WMSignalling(var Msg: TMessage  ); message WM_SIGNALLING;
  public
    { Public declarations }
  end;


var
  frmMain: TfrmMain;

implementation

{$R *.DFM}

uses
  SyncObjs,
  Windows, SysUtils, Dialogs,
  DvbDirectShow2,
  DvbStreamBuffering, DvbFilter,
  Saa7146AInterface,
  Saa7146AI2c,
  Saa7146aIoControl,
  Saa7146aRegisters,
  Stv0299bRegisters,
  Tsa5059Registers;

type
  TDataThread = class(TThread)
  private
    { Private declarations }
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
    BufferId      : Dword;             // Buffer identifier for DMA data
    PacketBuffers : Word;              // Number of buffers used
  protected
    procedure Execute; override;
  end;

var
  Timing : Dword;
  Counts : Dword;

  IsSlave     : Boolean;                         // True if act as slave
  ThreadHandle: THandle;                         // Handle to driver in thread
  PacketThread: TDataThread;                     // Thread handling packets

  CardHandle  : THandle;
  CardNoError : Boolean;
  SynthesizerAddress: Byte;

  CardInUse   : Integer;
  TunerType   : Byte;

  // Debug
  PacketBufferCount       : Word;                // Number of packet buffers received
  PacketPreviousCount     : Word;                // <PacketBufferCount> previous run
  PacketPreviousTimer     : Dword;               // Time of previous <PacketBufferCount>
  VideoPacketCount        : Word;                // Counts received video packets
  VideoPacketPreviousCount: Word;                // Counts received buffers previous check
  Overtaken               : Word;                // Number of times we run 'behind', eg. NOT waiting for data

  StreamDataBuffer    : TDvbTransportPackets;

  NewChannelSet       : Boolean;                 // Set to True when a new channel has been set
                                                 // Used for clearing DVB tables

  ShowVideo           : Boolean;                           // True shows video
  ProcessPids         : Boolean;                           // True processes PIDs

  FileStream          : TFileStream;                       // Recording stream
  Recording           : Boolean;                           // True if recording
  ActiveProgram       : Word;                              // Current program number
  PidVideo            : Word;                              // Current PID for video
  PidAudio            : Word;                              // Current PID for audio
  PidPMT              : Word;                              // Current PID for PMT

  UpdateProgram       : Boolean;                           // True if new program information available

{------------------------------------------------------------------------------
  Params  : <Msg>  Windows message
                   WParam has type of signalling index
                   LParam has additional signalling data (eg. PID if CSignalPid)
  Returns : -

  Descript: Handle messages from PID filter signalling
  Notes   :
------------------------------------------------------------------------------}
procedure TfrmMain.WMSignalling(var Msg: TMessage);
begin
  case Msg.WParam of
//    CSignalEvent  : UpdateEvent := True;
//    CSignalPmt    : UpdatePmt := True;
    CSignalPid    : case Msg.LParam of
                      CPidNIT: UpdateProgram := True;
                    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <StreamData>  Stream packets (multiple packets of 188 bytes)
  Returns : -

  Descript: Decode transport stream. We simply call the filter procedures
            which have been set.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeTransportStreamData(StreamData: PDvbTransportPackets);
var
  StreamPacket: Word;
  Pid         : Word;
  Filter      : TDvbPacketFilter;
begin
  for StreamPacket := 0 to CDvbPacketVSync-1 do
  begin
    // The Pid is the identifier of what type of information we are dealing with
    if DvbFilterGetPid(@StreamData[StreamPacket], Pid) then
    begin
      // Check PID
      // There are different PID's. PID's which indicate information (PSI) or data (PES).
      // We first have to receive PSI data before we can identify the PES data.
      // The structure is as follows:
      //   Program Association Table (PAT)
      //      -> PAT indicates other PID's for the Program Map Table (PMT)
      //          -> PMT indicates other PID's for the individual streams (PES)
      // Thus at first we have no information at all. Then, after receiving the PAT
      // we can identify the PID's which carry PMT data. Then we can identify
      // PID's used for stream data (PES).

      // Get function for PID for processing
      Filter := DvbFilterGetPidFilter(Pid);
      if Assigned(Filter) then
        Filter(@StreamData[StreamPacket]);

      if (Pid <> 0) and (Pid = PidVideo) then
      begin
        if VideoPacketCount = $FFFF then
          VideoPacketCount := 0
        else
          Inc(VideoPacketCount);
      end;

      // Write packet to file if requested
      if Recording and (Pid <> 0) then
      begin
        if (Pid = PidAudio) or
           (Pid = PidVideo) or
           (Pid = PidPmt)   then
          FileStream.Write(StreamData[StreamPacket], CDvbPacketSize);
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrupt at both edges of the field indicator,
            meaning that we get notified when a switch is made from
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Data       : Dword;
  LastBuffer : Dword;
  Buffer     : TSaa7146aTransferBuffer;
  SkipBuffer : Byte;                             // Forces buffers to skip after first setup
begin
  // Debug
  PacketBufferCount :=0;

  HasStopped     := False;
  ProgrammedStop := False;
  // The driver must be running and must be in use
  if Saa7146aGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't then only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls eg. 'generate manual notification'
  // which might be needed in a deadlock situation.
  ThreadHandle := Saa7146aCreateFile(CardInUse);
  Buffer.Identifier      := BufferId;
  Buffer.TransferAddress := @StreamDataBuffer;
  Buffer.TargetIndex     := 0;
  Buffer.TransferLength  := Sizeof(StreamDataBuffer);
  
  // Wait for application to have started and such
  repeat
   Sleep(100);
  until NewChannelSet or Terminated;

  LastBuffer := 0;
  Data       := $FFFF;
  SkipBuffer := 0;
  try
    if not Terminated then
    repeat
      // Some special handling required when a new channel has been set
      if NewChannelSet then
      begin
        DvbFilterResetErrors;
        Overtaken  := 0;
        DvbFilterCreateTables;
        if Assigned(frmMain) then
          frmMain.tmrNewChannel.Enabled := True;
        // Skip first xx buffers
        // We use more buffers when first startup (not REALLY necessary), since
        // only data is processed when correctly tuned ....
        if Data = $FFFF then
          SkipBuffer := PacketBuffers
        else
          SkipBuffer := 1;
        NewChannelSet := False;
      end;
      if PacketBufferCount = $FFFF then
        PacketBufferCount := 0
      else
        Inc(PacketBufferCount);
      // First check if there are already buffers filled
      Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
      // In case we have never received anything then the returned data is
      // not valid so check for this
      if Data >= (PacketBuffers*2) then
        Data := 0;
      // If no new data, wait (or if first time)
      if Data = LastBuffer then
      begin
        if IsSlave then
        repeat
          Sleep(1);
          Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
        until Terminated or (Data <> LastBuffer)
        else
          Saa7146aWaitForNotification(ThreadHandle);
      end
      else
      begin
        if OverTaken = $FFFF then
          OverTaken := 0
        else
          Inc(Overtaken);
      end;
      Buffer.SourceIndex := CDvbPacketBufferSize * LastBuffer;
      // Next buffer to wait for
      Inc(LastBuffer);
      // Circular buffer
      if LastBuffer = (PacketBuffers*2) then
        LastBuffer := 0;
      if SkipBuffer = 0 then
      begin
        // Get data in local buffer
        Saa7146aReadFromDma(ThreadHandle, Buffer);
        // Process packet data
        if ProcessPids then
          DecodeTransportStreamData(@StreamDataBuffer);
        // Send to DirectShow
        if Assigned(frmMain) and
           ShowVideo then
          DirectShowUniversalSourceSendData(@StreamDataBuffer, CDvbPacketBufferSize);
      end
      else
        Dec(SkipBuffer);
    until Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
      ShowMessage('Unexpected termination of packets handler.');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  FileStream: TFileStream;
  ErrorMessage: string;
begin
  // Check if the driver is present and/or multiple cards are present
  case Saa7146aGetNumberOfCards of
    0: mmoDriver.Lines.Add('No SAA7146A driver and/or card detected.');
  end;
  if IsSlave then
    mmoDriver.Lines.Add('Set to operate as slave.')
  else
    mmoDriver.Lines.Add('Set to operate as master.');
  case TunerType of
    CTunerBSRU6:   mmoDriver.Lines.Add('Controlling BSRU6 tuner.');
    CTunerSU1278:  mmoDriver.Lines.Add('Controlling SU1278 tuner.');
    else mmoDriver.Lines.Add('Controlling unknown tuner.');
  end;

  mmoDriver.Lines.Add(format('Using %d buffers (appr. %d ms).', [PacketThread.PacketBuffers, PacketThread.PacketBuffers * 20]));
  // We use the tag for the returned identifier
  if not DirectShowStart(frmMain.pnlVideo.Handle, 'VIDEO.GRF', False, False, False, ErrorMessage, INVALID_HANDLE_VALUE) then
    frmMain.mmoDriver.Lines.Add('DirectShow error')
  else
  begin
    frmMain.MskPidChange(nil);  // Send PID info to DirectShow
    ShowVideo := True;
  end;

  FileStream := TFileStream.Create('TRANSPONDER.LST', fmOpenRead);
  cmbTransponder.Items.LoadFromStream(FileStream);
  FileStream.Free;
  // Make sure first one is selected
  cmbTransponder.ItemIndex := 0;

  rgPolarityClick(nil);
  MskFrequencyChange(nil);
  MskSymbolRateChange(nil);
  Sleep(50);
  NewChannelSet := True;
  DvbFilterSetSignallingCallback(CPidNIT, frmMain.Handle, WM_SIGNALLING);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Form destruction.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  DvbFilterSetSignallingCallback(CPidNIT, INVALID_HANDLE_VALUE, WM_SIGNALLING);
  // Note: We MUST make sure that <DSSendDataDirect> is never called when DirectShow
  //       has stopped, therefore we inform the thread
  ProcessPids := False;
  ShowVideo   := False;
  Sleep(1);

  DirectShowSTop;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.chkVideoClick(Sender: TObject);
begin
  ShowVideo := chkVideo.Checked;
end;

procedure TfrmMain.chkPidsClick(Sender: TObject);
begin
  ProcessPids := chkPids.Checked;
end;



procedure TfrmMain.rgPolarityClick(Sender: TObject);
begin
  case rgPolarity.ItemIndex of
    0: if not Saa7146aHorizontalPolarityLNB(CardHandle, (TunerType = CTunerBSRU6)) then
         mmoDriver.Lines.Add('Could not set horizontal polarity.');
    1: if not Saa7146aVerticalPolarityLNB(CardHandle, (TunerType = CTunerBSRU6)) then
         mmoDriver.Lines.Add('Could not set vertical polarity.');
  end;
  NewChannelSet := True;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  if not Saa7146aQpskSetSymbolRate(CardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
  NewChannelSet := True;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  StrengthDecibel   : Double;
  StrengthPercentage: Double;
  DeltaTime : Dword;
  DeltaCount: Dword;
  Ticks     : Dword;
  Programs  : Byte;
  Second    : Boolean;
  Count     : Word;
  PidPcr    : Word;
  ServiceNumber: Word;
  PidsVideo : TDvbPids;
  PidsAudio : TDvbPids;
  PidsTeletext: TDvbPids;
  PidsSubtitle: TDvbPids;
  PidsEcm     : TDvbPids;
  CaIds       : TDvbPids;
  AudioLanguages: TDvbLanguages;
  SubtitleLanguages: TDvbLanguages;
  ProgramNumber: Byte;
begin
  if UpdateProgram then
  begin
    ProgramNumber := ActiveProgram;
    if DvbFilterGetProgramInfo(ProgramNumber, ServiceNumber, PidPmt, PidPcr, PidsVideo, PidsAudio, PidsTeletext, PidsSubtitle, AudioLanguages, SubtitleLanguages, PidsEcm, CaIds, StatusStr) then
      Caption := StatusSTr;
    PidVideo := PidsVideo[0];
    PidAudio := PidsAudio[0];
    UpdateProgram := False;
  end;
  Programs := DvbFilterGetNumberOfPrograms;
  if Programs > 0 then
    Dec(Programs);
  txtMaxPrograms.Caption := format('(00..%2.2d)', [Programs]);

  Ticks := GetTickCount;
  DeltaTime := Abs(Ticks - PacketPreviousTimer);
  if DeltaTime >= 1000 then
  begin
    Second := True;
    PacketPreviousTimer := Ticks;
  end
  else
    Second := False;

  if Second and Recording then
  begin
    imgRecord.Visible  := not(imgRecord.Visible);
    imgRecord2.Visible := not(imgRecord2.Visible);
  end;

  if (pgInfo.ActivePage = tsProgram) and Second then
  begin
    Count := VideoPacketCount;
    if Count <> VideoPacketPreviousCount then
    begin
      if Count > VideoPacketPreviousCount then
      begin
        DeltaCount := Count - VideoPacketPreviousCount;
        // Only for a valid difference (eg. buffer restart at 0)
        if DeltaCount < 10000 then
        begin
          // Note: We use <CDvbPacketSize> but the actual data contents
          //       is less...  Since the rate varies this 'error' does not matter.
          DeltaCount := DeltaCount * CDvbPacketSize * 8;
          DeltaCount := DeltaCount div DeltaTime;
          if DeltaCount < 10 then
            txtVideoRate.Caption := '-'
          else
            txtVideoRate.Caption := IntToStr(DeltaCount);
        end;
      end;  
      VideoPacketPreviousCount := Count;
    end;
  end;

  if pgInfo.ActivePage = tsDebug then
  begin
    // Only update every second
    if Second then
    begin
      if PacketBufferCount <> PacketPreviousCount then
      begin
        // Additional check because of circular buffer
        if PacketBufferCount > PacketPreviousCount then
        begin
          DeltaCount := PacketBufferCount - PacketPreviousCount;
          txtMs.Caption := format('%d', [DeltaTime div DeltaCount]);
        end;
        PacketPreviousCount := PacketBufferCount;
      end;
    end;
    txtPacketCount.Caption := format('%d', [PacketBufferCount]);
    txtOvertaken.Caption   := format('%d', [Overtaken]);
  end;

  // Don't handle updating other information if we are not the master
  if IsSlave then
    Exit;

  if pgInfo.ActivePage = tsTuning then
  begin
    if Saa7146aQpskGetSignalPower(CardHandle, StrengthPercentage, StrengthDecibel) then
    begin
      txtSignalPower.Caption := format('%f%% (%f dB)', [StrengthPercentage, StrengthDecibel]);
    end;
    if Saa7146aReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
    begin
      if (LockState and $80) <> 0 then
        StatusStr := 'CARRIER';
      if (LockState and $10) <> 0 then
        StatusStr := StatusStr + ' - VITERBI';
      if (LockState and $08) <> 0 then
        StatusStr := StatusStr + ' - SYNC';
      if (LockState and $98) = $98 then
        StatusStr := StatusStr + ' - LOCK';
      txtSignalState.Caption := StatusStr;
    end;
    if Saa7146aQpskGetNoiseIndicator(CardHandle, Noise) then
    begin
      txtNoiseIndicator.Caption := format('%f dB', [Noise]);
    end;
  end;
end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error:     Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert frequency.');
    Exit;
  end;

  if not Saa7146aFrequencyLNB(CardHandle, Frequency, 9750000, 10600000, TunerType, SynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
  NewChannelSet := True;
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;

procedure TfrmMain.mskPidKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskPidChange(nil);
end;

procedure TfrmMain.mskProgramKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskProgramChange(nil);
end;


procedure TfrmMain.mskPidChange(Sender: TObject);
var
  Error: Integer;
begin
  Val(mskVideoPid.EditText, PidVideo, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Video PID.');
    Exit;
  end;
  Val(mskAudioPid.EditText, PidAudio, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Audio PID.');
    Exit;
  end;

  DirectShowMpeg2DemultiplexerSetNewPids(PidVideo, PidAudio);
end;


procedure TfrmMain.mskProgramChange(Sender: TObject);
var
  Error        : Integer;
  ProgramNumber: Byte;
  Invalid      : Boolean;
  Programs     : Byte;
  OnChange     : TNotifyEvent;
  ProgramName  : string;
  ServiceNumber: Word;
  PidPcr    : Word;
  PidsVideo : TDvbPids;
  PidsAudio : TDvbPids;
  PidsTeletext: TDvbPids;
  PidsSubtitle: TDvbPids;
  PidsEcm     : TDvbPids;
  CaIds       : TDvbPids;
  AudioLanguages: TDvbLanguages;
  SubtitleLanguages: TDvbLanguages;
begin
  Val(mskProgram.EditText, ProgramNumber, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert program number.');
    Exit;
  end;
  Invalid := True;
  // First we want to check the program number
  Programs := DvbFilterGetNumberOfPrograms;
  if Programs <> 0 then
    if ProgramNumber >= Programs then
    begin
      ProgramNumber := Programs - 1;
      mskProgram.EditText :=  format('%2.2d', [ProgramNumber]);
    end;
  if DvbFilterGetProgramInfo(ProgramNumber, ServiceNumber, PidPmt, PidPcr, PidsVideo, PidsAudio, PidsTeletext, PidsSubtitle, AudioLanguages, SubtitleLanguages, PidsEcm, CaIds, ProgramName) then
  begin
    PidVideo := PidsVideo[0];
    PidAudio := PidsAudio[0];
    Caption := ProgramName;
    if ((PidAudio <> 0) or (PidVideo <> 0)) then
    begin
      // Note: Make sure no events generated by this, otherwise too many
      // are generated (for video and audio both - we only need a single one)
      OnChange := mskVideoPid.OnChange;
      mskVideoPid.OnChange := nil;
      mskVideoPid.EditText := format('%4.4d', [PidVideo]);
      mskVideoPid.OnChange := OnChange;
      OnChange := mskAudioPid.OnChange;
      mskAudioPid.OnChange := nil;
      mskAudioPid.EditText := format('%4.4d', [PidAudio]);
      mskAudioPid.OnChange := OnChange;
      mskPidChange(nil);
      Invalid := False;
    end;
  end;
  if Invalid then
  begin
    mskProgram.Color  := clRed;
    mskVideoPid.EditText := '0000';
    mskAudioPid.EditText := '0000';
    mskVideoPid.Color := clRed;
    mskAudioPid.Color := clRed;
  end
  else
  begin
    mskProgram.Color  := clLime;
    mskVideoPid.Color := clLime;
    mskAudioPid.Color := clLime;
  end;
  ActiveProgram := ProgramNumber;
end;


procedure TfrmMain.udProgramClick(Sender: TObject; Button: TUDBtnType);
var
  Error        : Integer;
  ProgramNumber: Integer;
  Programs     : Byte;
begin
  Val(mskProgram.EditText, ProgramNumber, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert program number.');
    Exit;
  end;
  if Button = btNext then
    Inc(ProgramNumber);
  if Button = btPrev then
    Dec(ProgramNumber);
  if ProgramNumber < 0 then
    ProgramNumber := 0;
  Programs := DvbFilterGetNumberOfPrograms;
  if Programs <> 0 then
    if ProgramNumber >= Programs then
      ProgramNumber := Programs - 1;
  mskProgram.EditText :=  format('%2.2d', [ProgramNumber]);
end;


procedure TfrmMain.tmrNewChannelTimer(Sender: TObject);
begin
  tmrNewChannel.Enabled := False;
  mskProgramChange(nil);
end;



procedure TfrmMain.cmbTransponderChange(Sender: TObject);
var
  Translate : string;
  Position  : Integer;
  PartStr   : string;
  Value     : Integer;
  Error     : Integer;
  DiSEqcData: TDiSEqCData;
begin
  Translate := cmbTransponder.Text;
  // First part is satellite
  Position := Pos(',', Translate);
  // Get value and set satellite (level 1.0)
  PartStr := Trim(Copy(Translate, 1, Position-1));
  Delete(Translate, 1, Position);
  Val(PartStr, Value, Error);
  if Error = 0 then
  begin
    case Value of
      1: DiSEqCData[3] := $F0;
      2: DiSEqCData[3] := $F4;
      3: DiSEqCData[3] := $F8;
      4: DiSEqCData[3] := $FC;
      else DiSEqCData[3] := $F0;
    end;
    DiSEqCData[0] := 0;                                      // Will be written over
    DiSEqCData[1] := $10;                                    // Any LNB/switcher
    DiSEqCData[2] := $38;                                    // Committed sitches (DiSEqc level 1.0)
    Saa7146aDiSEqCCommand(CardHandle, 2, CDiSEqCCommand, 4, DiSEqCData);
  end;
  // Second part is frequency
  Position := Pos(',', Translate);
  PartStr := Trim(Copy(Translate, 1, Position-1));
  Delete(Translate, 1, Position);
  Val(PartStr, Value, Error);
  if Error = 0 then
    mskFrequency.EditText := format('%5.5d000', [Value]);

  // Third part is polarity
  Position := Pos(',', Translate);
  PartStr := Trim(Copy(Translate, 1, Position-1));
  Delete(Translate, 1, Position);
  if LowerCase(PartStr) = 'v' then
    rgPolarity.ItemIndex := 1
  else
    rgPolarity.ItemIndex := 0;
  rgPolarityClick(nil);

  // Fourth part is symbolrate
  PartStr := Trim(Translate);
  Delete(Translate, 1, Position);
  Val(PartStr, Value, Error);
  if Error = 0 then
    mskSymbolrate.EditText := format('%5.5d', [Value]);
end;


procedure TfrmMain.udTransponderClick(Sender: TObject; Button: TUDBtnType);
var
  Transponder: Integer;
begin
  Transponder := cmbTransponder.ItemIndex;
  if Button = btNext then
    Inc(Transponder)
  else
    Dec(Transponder);
  cmbTransponder.ItemIndex := Transponder;
  cmbTransponderChange(nil);
end;


procedure TfrmMain.imgUpClick(Sender: TObject);
begin
  udProgramClick(nil, btNext);
end;


procedure TfrmMain.imgDownClick(Sender: TObject);
begin
  udProgramClick(nil, btPrev);
end;


procedure TfrmMain.imgLeftClick(Sender: TObject);
begin
  udTransponderClick(nil, btPrev);
end;

procedure TfrmMain.imgRightClick(Sender: TObject);
begin
  udTransponderClick(nil, btNext);
end;


procedure TfrmMain.imgRecordClick(Sender: TObject);
var
  FileName : string;
begin
  if Assigned(FileStream) then
  begin
    FreeAndNil(FileStream);
    Recording := False;
    imgRecord2.Visible := False;
    imgRecord.Visible  := True;
    mmoDriver.Lines.Add('Recording stopped.');
  end
  else
  begin
    try
      FileName := FormatDateTime('YYYYMMDD"T"HHMMSS', Now)+ '.TS';
      FileStream := TFileStream.Create(FileName, fmCreate);
    except
      FreeAndNil(FileStream);
      mmoDriver.Lines.Add('Error starting recording ''' + FileName + '''.');
      Exit;
    end;
    Recording := True;
    mmoDriver.Lines.Add('Started recording ''' + FileName + '''.');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  BufferId  : Dword;
  BufferTime: Word;
begin
  FileStream    := nil;
  UpdateProgram := False;
  Recording     := False;
  VideoPacketCount := 0;
  NewChannelSet := False;
  ShowVideo     := False;
  ProcessPids   := True;
  Timing        := 0;
  Counts        := 0;
  IsSlave       := False;
  PacketThread  := nil;
  CardHandle    := INVALID_HANDLE_VALUE;
  CardNoError   := False;
  CardInUse     := -1;
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    CardHandle := Saa7146aCreateFile(CardInUse);
    CardInUse  := Saa7146aGetCardOfHandle(CardHandle);
  end;

  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    IsSlave := True;
    // Check for parameter for forcing it to master/slave mode
    if (ParamCount <> 0) and (LowerCase(ParamStr(1)) = 'master') then
      IsSlave := False;
    if (ParamCount <> 0) and (LowerCase(ParamStr(1)) = 'slave') then
      IsSlave := True;

    if (ParamCount > 1) and (ParamStr(2) = '1') then
      TunerType := CTunerSU1278
    else
      TunerType := CTunerBSRU6;
    // Setup data transfer from tuner to memory
    // We use a buffer of 1000 ms here
    BufferTime := 1000;
    if not DvbSetupBuffering(CardHandle, True, TunerType, 2, @IsSlave, @BufferTime, @BufferId, -1) then
      Exit;
    // Synthesizer address after I2C initialization ....
    SynthesizerAddress := Saa7146aDetectSynthesizerAddress(CardHandle);
    // Start the thread which receives notifications
    PacketThread := TDataThread.Create(True);
    PacketThread.BufferId      := BufferId;
    PacketThread.PacketBuffers := BufferTime;
    PacketThread.Priority      := tpNormal;
    PacketThread.Resume;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
var
  Channel: Dword;
begin
  if Assigned(FileStream) then
    FreeAndNil(FileStream);
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    // Note the use of <CardHandle> and NOT <ThreadHandle>. This is because
    // <ThreadHandle> might be 'busy' with a <Saa7146aWaitForNotification>!
    Saa7146aGenerateManualNotification(CardHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;
  // Software reset. We migth be capturing data, so stop ALL
  if not IsSlave then
  begin
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1, CSaa7146aMc1SoftReset);
    // Release all allocated memory by the driver
    Channel := 0;
    while Saa7146aReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    Saa7146aDisableLNB(CardHandle, (TunerType = CTunerBSRU6));
    Saa7146aCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;
end;




procedure TfrmMain.btnDebugClick(Sender: TObject);
begin
//  lbldebug.caption := dvbdebug(ActiveProgram);
end;


initialization
  Initialize;


finalization
  Finalize;
end.
