{******************************************************************************}
{ FileName............: ShowingStreamDataUnit001                               }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Buffering / Showing streams.                                                }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Includes master / slave principle for multiple instances which will use the  }
{ same packet data.                                                            }
{                                                                              }
{ The default settings should provide you with a video/audio of the german ARD }
{ program being transmitted. It is assumed the dish is facing Astra 19.2E and  }
{ that the PIDs have not changed....                                           }
{ If no video/audio is shown (without an error being reported) then check the  }
{ also the graphics setup (see note below).                                    }
{                                                                              }
{ Note: Master / Slave operation for DirectShow currently generates an error   }
{       at the second instance (slave)                                         }
{ Note: Only one DirectShow per process can be run ....                        }
{ Note: Debug not always possible because of DirectShow stuff ...              }
{ Note: Stream decoding not active ...                                         }
{ Note: The supplied VIDEO.GRF has no connected output pins. This is not       }
{       absolute necessary because these are automatically connected to the    }
{       'correct' video/audio filters.                                         }
{       These video/audio filters are typically part of a DVD viewer (like     }
{       InterVideo / PowerDVD etc. You can also connect these pins as required }
{       so they are forced to a specific connection. You need the FilterGraph  }
{       from DirectX 9 for this.                                               }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030824 - Initial release                                           }
{******************************************************************************}
unit ShowingStreamDataUnit001;

interface
uses
  Forms, ExtCtrls, StdCtrls, Mask, Controls, Classes, Gauges;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    rgPolarity: TRadioGroup;
    mskSymbolRate: TMaskEdit;
    StaticText1: TStaticText;
    tmrUpdate: TTimer;
    ggLock: TGauge;
    StaticText2: TStaticText;
    StaticText3: TStaticText;
    mskFrequency: TMaskEdit;
    StaticText4: TStaticText;
    stDeviation: TStaticText;
    txtSignalPower: TStaticText;
    txtSignalState: TStaticText;
    StaticText7: TStaticText;
    stDeviationSymbolrate: TStaticText;
    txtNoiseIndicator: TStaticText;
    StaticText27: TStaticText;
    txtMs: TStaticText;
    txtPacketCount: TStaticText;
    StaticText28: TStaticText;
    StaticText29: TStaticText;
    StaticText30: TStaticText;
    txtOvertaken: TStaticText;
    StaticText5: TStaticText;
    StaticText6: TStaticText;
    StaticText8: TStaticText;
    mskVideoPid: TMaskEdit;
    mskAudioPid: TMaskEdit;
    StaticText9: TStaticText;
    StaticText10: TStaticText;
    pnlVideo: TPanel;
    StaticText11: TStaticText;
    StaticText12: TStaticText;
    StaticText13: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnClearClick(Sender: TObject);
    procedure mskPidChange(Sender: TObject);
    procedure mskPidKeyPress(Sender: TObject; var Key: Char);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;


var
  frmMain: TfrmMain;

implementation

{$R *.DFM}

uses
  SyncObjs,
  Windows, Messages, SysUtils, Graphics, Dialogs,
  DvbDirectShow2,
  Dvb,
  Saa7146AInterface, Saa7146aGpio, Saa7146AI2c, Stv0299bRegisters,
  Tsa5059Registers,
  Saa7146aIoControl,
  Saa7146aRegisters;


const
  PacketBuffers = 20;

type
  TDataThread = class(TThread)
  private
    { Private declarations }
    HasStopped    : Boolean;           // Flag indicating thread not running
    ProgrammedStop: Boolean;           // If true indicates programmed termination
  protected
    procedure Execute; override;
  end;

var
  Timing : Dword;
  Counts : Dword;
  SynthesizerAddress: Byte;

  IsSlave     : Boolean;                         // True if act as slave
  ThreadHandle: THandle;                         // Handle to driver in thread
  PacketThread: TDataThread;                     // Thread handling packets

  CardHandle  : THandle;
  CardNoError : Boolean;

  BufferRps1  : TSaa7146aDmaBuffer;              // DMA buffer for RPS1 program

  Fifo3       : TSaa7146aDmaBuffer;
  CardInUse   : Integer;
  TunerType   : Byte;                            // Tuner type

  // Debug
  PacketBufferCount  : Word;                     // Number of packet buffers received
  PacketPreviousCount: Word;                     // <PacketBufferCount> previous run
  PacketPreviousTimer: Dword;                    // Time of previous <PacketBufferCount>
  Overtaken          : Word;                     // Number of times we run 'behind', eg. NOT waiting for data

  StreamDataBuffer : TDvbTransportPackets;
  StreamBuffer     : array[0..CDvbPacketVSync-1] of TDvbTransportStreamHeader;     // Decoded data (one buffer)

  NewChannelSet       : Boolean;                 // Set to True when a new channle has been set
                                                 // Used for clearing DVB tables

  TransportPids       : TDvbTransportPids;
  TransportPidsCounter: Word;                              // Count the number of different trasnport PID's
  PacketError         : Word;

  PatReceiving        : TDvbProgramAssociationTable;       // Current Program Association Table receiving
  PatActive           : TDvbProgramAssociationTable;       // Current active Program Association Table
  Pats                : Word;                              // Counter for (new) number of PATs

  CatReceiving        : TDvbConditionalAccessTable;        // Current Conditional Access Table receiving
  CatActive           : TDvbConditionalAccessTable;        // Current active Conditional Access Table
  Cats                : Word;                              // Counter for (new) number of CATs

  PmtReceiving        : TDvbProgramMapTables;              // Current Program Map Tables receiving
  PmtActive           : TDvbProgramMapTables;              // Current active Program Map Tables
  Pmts                : Word;                              // Counter for (new) number of PMTs

  PesReceiving        : TDvbProgramElementaryStreams;      // Current Program Elementary Streams receiving
  PesActive           : TDvbProgramElementaryStreams;      // Current active Program Elemntary Streams
  Pess                : Word;                              // Counter for (new) number of PESs
  PessCreated         : Word;                              // Counter for (new) number of created PESs

  PidList             : TDvbTypeOfPids;                    // Type of PIDs
  ProgramList         : TDvbProgramNumbers;                // Program numbers
  Programs            : Word;                              // Number of programs;

  ShowVideo           : Boolean;                           // True shows video

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Set all table references to unassigned status.
  Notes   :
 ------------------------------------------------------------------------------}
procedure NilDvbTables;
var
  Loop: Word;
begin
  for Loop := Low(PesActive) to High(PesActive) do
    PesActive[Loop] := nil;
  for Loop := Low(PesReceiving) to High(PesReceiving) do
    PesReceiving[Loop] := nil;
  for Loop := Low(PmtActive) to High(PmtActive) do
    PmtActive[Loop] := nil;
  for Loop := Low(PmtReceiving) to High(PmtReceiving) do
    PmtReceiving[Loop] := nil;

  for Loop := Low(TransportPids) to High(TransportPids) do
    TransportPids[Loop] := False;

  for Loop := Low(PidList) to High(PidList) do
    PidList[Loop].PidType := CDvbTypePidUndefined;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Clear allocated memory and such.
  Notes   : Typically used when switching channels.
            Should be synchronized with packet handler!
 ------------------------------------------------------------------------------}
procedure DestroyDvbTables;
var
  Loop: Word;
begin
  for Loop := Low(PesActive) to High(PesActive) do
    if Assigned(PesActive[Loop]) then
    begin
      FreeMem(PesActive[Loop]);
      PesActive[Loop] := nil;
    end;
  for Loop := Low(PesReceiving) to High(PesReceiving) do
    if Assigned(PesReceiving[Loop]) then
    begin
      FreeMem(PesReceiving[Loop]);
      PesReceiving[Loop] := nil;
    end;
  for Loop := Low(PmtActive) to High(PmtActive) do
    if Assigned(PmtActive[Loop]) then
    begin
      FreeMem(PmtActive[Loop]);
      PmtActive[Loop] := nil;
    end;
  for Loop := Low(PmtReceiving) to High(PmtReceiving) do
    if Assigned(PmtReceiving[Loop]) then
    begin
      FreeMem(PmtReceiving[Loop]);
      PmtReceiving[Loop] := nil;
    end;

  for Loop := Low(TransportPids) to High(TransportPids) do
    TransportPids[Loop] := False;

  for Loop := Low(PidList) to High(PidList) do
    PidList[Loop].PidType := CDvbTypePidUndefined;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Reset tables and such.
  Notes   : All variables are set to their intial values.
            Should be synchronized with packet handler!
 ------------------------------------------------------------------------------}
procedure CreateDvbTables;
begin
  DestroyDvbTables;

  // Setup defaults so new data is handled, eg. VersionNumber
  PatReceiving.Sync       := False;
  PatReceiving.TableValid := False;
  CatReceiving.Sync       := False;
  CatReceiving.TableValid := False;
  PatActive.TableValid    := False;
  PatActive.VersionNumber := $FF;
  CatActive.TableValid    := False;
  CatActive.VersionNumber := $FF;

  TransportPidsCounter := 0;
  Programs             := 0;
  Pats                 := 0;
  Cats                 := 0;
  Pmts                 := 0;
  Pess                 := 0;
  PessCreated          := 0;
end;


{------------------------------------------------------------------------------
  Params  : <StreamPacket>  Stream packet index
  Returns : -

  Descript: Decode program elementary stream (PES) data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeProgramElementaryStreamData(StreamPacket: Word);
var
  LoopStart    : Word;
  Loop         : Word;
  Checksum32   : Dword;
  Index        : Word;
begin
  try
    with StreamBuffer[StreamPacket] do
    begin
      // If this is the first time this PID is used, allocate the memory for it
      if not Assigned(PesReceiving[Pid]) then
      begin
        GetMem(PesReceiving[Pid], SizeOf(TDvbProgramElementaryStream));
        PesReceiving[Pid]^.RawIndex        := 0;
        PesReceiving[Pid]^.TableValid      := False;
        PesReceiving[Pid]^.Sync            := False;

        GetMem(PesActive[Pid], SizeOf(TDvbProgramElementaryStream));
        PesActive[Pid]^.RawIndex        := 0;
        PesActive[Pid]^.TableValid      := False;
        PesActive[Pid]^.Sync            := False;

        Inc(PessCreated);
      end;
      with PesReceiving[Pid]^ do
      begin
        if PayloadUnitStartIndicator then
        begin
          // Start of a new packet resets the indexes
          RawIndex        := 0;
          TableValid      := False;
          Sync            := True;
          LoopStart       := 0;
        end
        else
        begin
          if not Sync then
            Exit;
          LoopStart := 0;                                      // Not first packet
        end;

        // We first have to put all data without processing into a buffer
        // We only check the section length if we have received all data
        // for the whole section. If so we will process it
        // We copy all data bytes although this might be more than the actual section
        if LoopStart < DataBytesLength then
        begin
          for Loop := LoopStart to DataBytesLength - 1 do
            if RawIndex < High(TDvbProgramMapRaw) then
            begin
              RawData[RawIndex] := DataBytes[Loop];
              Inc(RawIndex);
            end;
          // We can now make a check on if we received the whole section
          // We assume we always have received at least 3 bytes ....
          PesPacketLength := (RawData[4] shl 8) or RawData[5];
          if ((PesPacketLength+4) < RawIndex) then           // If we have it all
          begin
            PacketStartCodePrefix := (RawData[0] shl 16) or
                                     (RawData[1] shl 8)  or
                                      RawData[2];
            StreamId              := RawData[3];
            if (StreamId <> CDvbProgramStreamMap)       and
               (StreamId <> CDvbPaddingStream)          and
               (StreamId <> CDvbPrivateStream2)         and
               (StreamId <> CDvbEcmStream)              and
               (StreamId <> CDvbEmmStream)              and
               (StreamId <> CDvbProgramStreamDirectory) and
               (StreamId <> CDvbDsmccStream)            and
               (StreamId <> CDvbTypeEStream) then
            begin
              Pes10                  := (RawData[6] and $C0) shr 6;
              PesScramblingControl   := (RawData[6] and $30) shr 4;
              PesPriority            := (RawData[6] and $08) <> 0;
              DataAlignmentIndicator := (RawData[6] and $04) <> 0;
              Copyright              := (RawData[6] and $02) <> 0;
              OriginalOrCopy         := (RawData[6] and $01) <> 0;
              PtsDtsFlags            := (RawData[7] and $C0) shr 6;
              EscrFlag               := (RawData[7] and $20) <> 0;
              EsRateFlag             := (RawData[7] and $10) <> 0;
              DsmTrickModeFlag       := (RawData[7] and $08) <> 0;
              AdditionalCopyInfoFlag := (RawData[7] and $04) <> 0;
              PesCrcFlag             := (RawData[7] and $02) <> 0;
              PesExtensionFlag       := (RawData[7] and $01) <> 0;
              PesHeaderDataLength    := RawData[8];
              // From this point on difference will occur
              Index := 9;
              if PtsDtsFlags = $02 then
              begin
                Pes0010           := (RawData[Index] and $F0) shr 4;
                Pts00103230       := (RawData[Index] and $0E) shr 1;
                MarkerBit00103230 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00102915       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00102915 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00101400       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00101400 := (RawData[Index] and $01) <> 0;
                Inc(Index);
              end;
              if PtsDtsFlags = $03 then
              begin
                Pes0011           := (RawData[Index] and $F0) shr 4;
                Pts00113230       := (RawData[Index] and $0E) shr 1;
                MarkerBit00113230 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00112915       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00112915 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00111400       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00111400 := (RawData[Index] and $01) <> 0;
                Inc(Index);

                Pes0001           := (RawData[Index] and $F0) shr 4;
                Pts00013230       := (RawData[Index] and $0E) shr 1;
                MarkerBit00013230 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00012915       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00012915 := (RawData[Index] and $01) <> 0;
                Inc(Index);
                Pts00011400       := (RawData[Index] shl 7) or
                                     (RawData[Index+1] and $7F);
                Inc(Index);
                MarkerBit00011400 := (RawData[Index] and $01) <> 0;
                Inc(Index);
              end;
              if EscrFlag then
              begin
                Reserved           := (RawData[Index] and $C0) shr 6;
                EscrBase3230       := (RawData[Index] and $38) shr 3;
                MarkerBit3230      := (RawData[Index] and $04) <> 0;
                EscrBase2915       := ((RawData[Index] and $03) shl 12) or
                                      (RawData[Index+1] shl 5) or
                                      ((RawData[Index+2] and $F8) shr 3);
                Inc(Index, 2);
                MarkerBit2915      := (RawData[Index] and $04) <> 0;
                EscrBase1400       := ((RawData[Index] and $03) shl 12) or
                                      (RawData[Index+1] shl 5) or
                                      ((RawData[Index+2] and $F8) shr 3);
                Inc(Index, 2);
                MarkerBit1400      := (RawData[Index] and $04) <> 0;
                EscrExtension      := ((RawData[Index] and $03) shl 7) or
                                      ((RawData[Index+1] and $FE) shr 1);
                Inc(Index);
                MarkerBitExtension := (RawData[Index] and $01) <> 0;
                Inc(Index);
              end;
              if EsRateFlag then
              begin
                MarkerBitEsRateFlag := (RawData[Index] and $80) <> 0;
                EsRate              := ((RawData[Index] and $7F) shl 15) or
                                        (RawData[Index+1] shl 7) or
                                       ((RawData[Index+2] and $FE) shr 1);
                Inc(Index, 2);
                MarkerBitEsRate     := (RawData[Index] and $01) <> 0;
                Inc(Index);
              end;
              if DsmTrickModeFlag then
              begin
                TrickModeControl    := (RawData[Index] and $E0) shr 5;
                if TrickModeControl = CDvbTrickModeFastForward then
                begin
                  FieldIdFf             := (RawData[Index] and $18) shr 3;
                  IntraSliceRefreshFf   := (RawData[Index] and $04) <> 0;
                  FrequencyTruncationFf := (RawData[Index] and $03);
                  Inc(Index);
                end
                else
                  if TrickModeControl = CDvbTrickModeSlowMotion then
                  begin
                    RepCntrlSm := (RawData[Index] and $1F);
                    Inc(Index);
                  end
                  else
                    if TrickModeControl = CDvbTrickModeFreezeFrame then
                    begin
                      FieldIdFrF  := (RawData[Index] and $18) shr 3;
                      ReservedFrF := (RawData[Index] and $07);
                      Inc(Index);
                    end
                    else
                      if TrickModeControl = CDvbTrickModeFastReverse then
                      begin
                        FieldIdFr             := (RawData[Index] and $18) shr 3;
                        IntraSliceRefreshFr   := (RawData[Index] and $04) <> 0;
                        FrequencyTruncationFr := (RawData[Index] and $03);
                        Inc(Index);
                      end
                      else
                        if TrickModeControl = CDvbTrickModeSlowReverse then
                        begin
                          RepCntrlSr := (RawData[Index] and $1F);
                          Inc(Index);
                        end
                        else
                          begin
                            ReservedTrickMode := (RawData[Index] and $1F);
                            Inc(Index);
                          end;
              end;
              if AdditionalCopyInfoFlag then
              begin
                MarkerBitAdditionalCopyFlag := (RawData[Index] and $80) <> 0;
                AdditionalCopyInfo          := RawData[Index] and $7F;
                Inc(Index);
              end;
              if PesCrcFlag then
              begin
                PreviousPesPacketCrc := (RawData[Index] shl 8) or (RawData[Index+1]);
                Inc(Index, 2);
              end;
              if PesExtensionFlag then
              begin
                PesPrivateDataFlag               := (RawData[Index] and $80) <> 0;
                PackHeaderFieldFlag              := (RawData[Index] and $40) <> 0;
                ProgramPacketSequenceCounterFlag := (RawData[Index] and $20) <> 0;
                PstdBufferFlag                   := (RawData[Index] and $10) <> 0;
                PstdBufferFlag                   := (RawData[Index] and $10) <> 0;
                ReservedPesExtension             := (RawData[Index] and $0E) shr 1;
                PesExtensionFlag2                := (RawData[Index] and $01) <> 0;
                Inc(Index);
                if PesPrivateDataFlag then
                  for Loop := 0 to 15 do
                  begin
                    PesPrivateData[Loop] := RawData[Index];
                    Inc(Index);
                  end;
                if PackHeaderFieldFlag then
                begin
                  PackFieldLength := RawData[Index];
                  Inc(Index);
                  if PackFieldLength > 0 then
                    for Loop := 0 to PackFieldLength-1 do
                    begin
                      PackHeader[Loop] := RawData[Index];
                      // The pack header needs to be decoded further
                      Inc(Index);
                    end;
                end;
                if ProgramPacketSequenceCounterFlag then
                begin
                  MarkerBitPpsc                := (RawData[Index] and $80) <> 0;
                  ProgramPacketSequenceCounter := (RawData[Index] and $7F);
                  Inc(Index);
                  MarkerBitPpsc2               := (RawData[Index] and $80) <> 0;
                  Mpeg1Mpeg2Identifier         := (RawData[Index] and $40) <> 0;
                  OriginalStuffLength          := (RawData[Index] and $3F);
                  Inc(Index);
                end;
                if PstdBufferFlag then
                begin
                  Pstd01                       := (RawData[Index] and $C0) shr 6;
                  PstdBufferScale              := (RawData[Index] and $20) <> 0;
                  PstdBufferSize               := ((RawData[Index] and $1F) shl 8) or
                                                   RawData[Index+1];
                  Inc(Index, 2);
                end;
                if PesExtensionFlag2 then
                begin
                  MarkerBitPef                 := (RawData[Index] and $80) <> 0;
                  PesExtensionFieldLength      := (RawData[Index] and $7F);
                  Inc(Index);
                  if PesExtensionFieldLength > 0 then
                    for Loop := 0 to PesExtensionFieldLength-1 do
                    begin
                      ReservedPesExtensionField[Loop] := RawData[Index];
                      Inc(Index);
                    end;
                end;
              end;
            end
            else
              if (StreamId = CDvbProgramStreamMap)       or
                 (StreamId = CDvbPrivateStream2)         or
                 (StreamId = CDvbEcmStream)              or
                 (StreamId = CDvbEmmStream)              or
                 (StreamId = CDvbProgramStreamDirectory) or
                 (StreamId = CDvbDsmccStream)            or
                 (StreamId = CDvbTypeEStream) then
               begin
                 for Index := 0 to PesPacketLength-1 do
                   PesPacketDataByte[Index] := RawData[Index+6];
               end
               else
                 if (StreamId = CDvbPaddingStream) then ;    // With padding do nothing

             if (PacketStartCodePrefix = $000001) then
               TableValid := True;
              // If it indicates a currently active PES, update the active PES
              if TableValid then
              begin
                PesActive[Pid]^ := PesReceiving[Pid]^;
                if Pess = $FFFF then
                  Pess := 0
                else
                  Inc(Pess);
              end;
          end;
        end;
      end;
    end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <StreamPacket>  Stream packet index
  Returns : -

  Descript: Decode program map table data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeProgramMapTableData(StreamPacket: Word);
var
  LoopStart    : Word;
  Loop         : Word;
  Checksum32   : Dword;
  Index        : Word;
begin
  try
    with StreamBuffer[StreamPacket] do
    begin
      // If this is the first time this PID is used, allocate the memory for it
      if not Assigned(PmtReceiving[Pid]) then
      begin
        GetMem(PmtReceiving[Pid], SizeOf(TDvbProgramMapTable));
        PmtReceiving[Pid]^.RawIndex        := 0;
        PmtReceiving[Pid]^.StreamIndex     := 0;
        PmtReceiving[Pid]^.TableValid      := False;
        PmtReceiving[Pid]^.Sync            := False;

        GetMem(PmtActive[Pid], SizeOf(TDvbProgramMapTable));
        PmtActive[Pid]^.RawIndex        := 0;
        PmtActive[Pid]^.StreamIndex     := 0;
        PmtActive[Pid]^.TableValid      := False;
        PmtActive[Pid]^.Sync            := False;
        PmtActive[Pid]^.VersionNumber   := $FF;
      end;
      with PmtReceiving[Pid]^ do
      begin
        if PayloadUnitStartIndicator then
        begin
          // Start of a new packet resets the indexes
          RawIndex        := 0;
          StreamIndex     := 0;
          TableValid      := False;
          Sync            := True;

          // All so-called PSI tables have a first byte which is the <PointerField>.
          // This indicates at what offset the actual data starts. We know that
          // the offset points into this very first packet...
          LoopStart := DataBytes[0]+1;
          if LoopStart > DataBytesLength then                  // If incorrect value
            LoopStart := DataBytesLength;
        end
        else
        begin
          if not Sync then
            Exit;
          LoopStart := 0;                                      // Not first packet
        end;

        // We first have to put all data without processing into a buffer
        // We only check the section length if we have received all data
        // for the whole section. If so we will process it
        // We copy all data bytes although this might be more than the actual section
        if LoopStart < DataBytesLength then
        begin
          for Loop := LoopStart to DataBytesLength - 1 do
            if RawIndex < High(TDvbProgramMapRaw) then
            begin
              RawData[RawIndex] := DataBytes[Loop];
              Inc(RawIndex);
            end;
          // We can now make a check on if we received the whole section
          // We assume we always have received at least 3 bytes ....
          SectionLength := ((RawData[1] and $0F) shl 8) or
                             RawData[2];
          if (SectionLength < 1022) and                        // Sanity check
             ((SectionLength+2) < RawIndex) then               // If we have it all
          begin
            // Before we do any further processing at all we migth check for a correct checksum
            Checksum32 := DvbResetValueCrc32;
            for Loop := 0 to SectionLength+2 do
              Checksum32 := DvbNewValueCrc32(Checksum32, RawData[Loop]);
            // Check for correct checksum
            if Checksum32 = 0 then
            begin
              TableId                := RawData[0];
              SectionSyntaxIndicator := (RawData[1] and $80) <> 0;
              Zero                   := (RawData[1] and $40) <> 0;
              Reserved               := (RawData[1] and $30) shr 4;
              // SectionLength       := ((RawData[1] and $0F) shl 8) or
              //                        RawData[2];
              ProgramNumber          := (RawData[3] shl 8) or
                                        RawData[4];
              Reserved2              := (RawData[5] and $C0) shr 6;
              VersionNumber          := (RawData[5] and $37) shr 1;
              CurrentNextIndicator   := (RawData[5] and $01) <> 0;
              SectionNumber          := RawData[6];
              LastSectionNumber      := RawData[7];
              Reserved3              := (RawData[8] and $E0) shr 5;
              PcrPid                 := ((RawData[8] and $1F) shl 8) or
                                        RawData[9];
              Reserved4              := (RawData[10] and $F0) shr 4;
              ProgramInfoLength      := ((RawData[10] and $0F) shl 8) or
                                        RawData[11];
              if ProgramInfoLength <> 0 then
                for Loop := 0 to ProgramInfoLength-1 do
                  DescriptorData[Loop] := RawData[Loop+12];
              // Because there is now ay to determine how much stream data blocks
              // there are by forehand we have to go through it manually
              StreamIndex := 0;
              Index := ProgramInfoLength + 12;               // First stream block
              while Index < (SectionLength+2-4-5) do         // Minimum margin
              begin
                StreamData[StreamIndex].StreamType := RawData[Index];
                Inc(Index);
                StreamData[StreamIndex].Reserved := (RawData[Index] and $E0) shr 5;
                StreamData[StreamIndex].ElementaryPid := ((RawData[Index] and $1F) shr 8) or
                                                         RawData[Index+1];
                Inc(Index, 2);
                StreamData[StreamIndex].Reserved2     := (RawData[Index] and $F0) shr 4;
                StreamData[StreamIndex].EsInfoLength  := ((RawData[Index] and $0F) shr 8) or
                                                         RawData[Index+1];
                Inc(Index, 2);
                if StreamData[StreamIndex].EsInfoLength > 0 then
                  for Loop := 0 to StreamData[StreamIndex].EsInfoLength-1 do
                  begin
                    StreamData[StreamIndex].Descriptor[Loop] := RawData[Index];
                    Inc(Index);
                  end;
                Inc(StreamIndex);
              end;
              Crc32 := (RawData[SectionLength+2-3] shl 24) or
                       (RawData[SectionLength+2-2] shl 16) or
                       (RawData[SectionLength+2-1] shl 8) or
                        RawData[SectionLength+2];
              if (TableId = CDvbTransportPidPMT) and
                 (Index = (SectionLength+2-3)) then          // Sanity checks
                TableValid := True;
              // If it indicates a currently active PMT, update the active PMT
              if TableValid and
                 CurrentNextIndicator and
                 (PmtActive[Pid]^.VersionNumber <> VersionNumber) then
              begin
                PmtActive[Pid]^ := PmtReceiving[Pid]^;

                // What we must do here is to update the PID list with references we
                // acquired in the PMT. The PMT now has references to the PID which
                // carry the Program Elementary Streams (PES).
                if PcrPid <> CDvbTransportPidNull then
                  PidList[PcrPid].PidType := CDvbTypePidPcr;
                if StreamIndex > 0 then
                  for Loop := 0 to StreamIndex-1 do
                  begin
                    PidList[StreamData[Loop].ElementaryPid].PidType      := CDvbTypePidPes;
                    PidList[StreamData[Loop].ElementaryPid].ProgramNumber:= PmtActive[Pid]^.ProgramNumber;
                    PidList[StreamData[Loop].ElementaryPid].StreamType   := PmtActive[Pid]^.StreamData[Loop].StreamType;
                    PidList[StreamData[Loop].ElementaryPid].EsInfoLength := PmtActive[Pid]^.StreamData[Loop].EsInfoLength;
                    PidList[StreamData[Loop].ElementaryPid].Descriptor   := PmtActive[Pid]^.StreamData[Loop].Descriptor;
                    // Try to find if we already have the program number in our list.
                    // If not add it to the list
                    Index := 0;
                    while (Index < Programs) and (ProgramList[Index] <> PmtActive[Pid]^.ProgramNumber) do
                      Inc(Index);
                    ProgramList[Index] := PmtActive[Pid]^.ProgramNumber;
                    if Index = Programs then
                      Inc(Programs);
  case PidList[StreamData[Loop].ElementaryPid].StreamType of
    2: frmMain.mmoDriver.Lines.Add(format('Program %d, $02: ISO/IEC 13818-2 Video', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber]));
    3: frmMain.mmoDriver.Lines.Add(format('Program %d, $03: ISO/IEC 11172 Audio', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber]));
    4: frmMain.mmoDriver.Lines.Add(format('Program %d, $04: ISO/IEC 13818-3 Audio', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber]));
    5: frmMain.mmoDriver.Lines.Add(format('Program %d, $05: ISO/IEC 13818-1 Private section', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber]));
    6: frmMain.mmoDriver.Lines.Add(format('Program %d, $06: ISO/IEC 13818-1 PES Private data', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber]));
    else frmMain.mmoDriver.Lines.Add(format('Program %d, Stream id $%2.2x added', [PidList[StreamData[Loop].ElementaryPid].ProgramNumber, PidList[StreamData[Loop].ElementaryPid].StreamType]));
  end;
                  end;
                if Pmts = $FFFF then
                  Pmts := 0
                else
                  Inc(Pmts);
              end;
            end
            else
              begin
                Inc(PacketError);
                if PacketError = $FFFF then
                  PacketError := 0;
              end;
          end;
        end;
      end;
    end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <StreamPacket>  Stream packet index
  Returns : -

  Descript: Decode conditional access table data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeConditionalAccessTableData(StreamPacket: Word);
var
  LoopStart    : Word;
  Loop         : Word;
  Checksum32   : Dword;
  Descriptors  : Word;
begin
  try
    with StreamBuffer[StreamPacket] do
      with CatReceiving do
      begin
        if PayloadUnitStartIndicator then
        begin
          // Start of a new packet resets the indexes
          RawIndex := 0;
          TableValid   := False;
          Sync         := True;
          // All so-called PSI tables have a first byte which is the <PointerField>.
          // This indicates at what offset the actual data starts. We know that
          // the offset points into this very first packet...
          LoopStart := DataBytes[0]+1;
          if LoopStart > DataBytesLength then                  // If incorrect value
            LoopStart := DataBytesLength;
        end
        else
        begin
          if not Sync then
            Exit;
          LoopStart := 0;                                      // Not first packet
        end;

        // We first have to put all data without processing into a buffer
        // We only check the section length if we have received all data
        // for the whole section. If so we will process it
        // We copy all data bytes although this might be more than the actual section
        if LoopStart < DataBytesLength then
        begin
          for Loop := LoopStart to DataBytesLength - 1 do
            if RawIndex < High(TDvbConditionalAccessRaw) then
            begin
              RawData[RawIndex] := DataBytes[Loop];
              Inc(RawIndex);
            end;
          // We can now make a check on if we received the whole section
          // We assume we always have received at least 3 bytes ....
          SectionLength := ((RawData[1] and $0F) shl 8) or
                             RawData[2];
          if (SectionLength < 1022) and                        // Sanity check
             ((SectionLength+2) < RawIndex) then               // If we have it all
          begin
            // Before we do any further processing at all we migth check for a correct checksum
            Checksum32 := DvbResetValueCrc32;
            for Loop := 0 to SectionLength+2 do
              Checksum32 := DvbNewValueCrc32(Checksum32, RawData[Loop]);
            // Check for correct checksum
            if Checksum32 = 0 then
            begin
              TableId                := RawData[0];
              SectionSyntaxIndicator := (RawData[1] and $80) <> 0;
              Zero                   := (RawData[1] and $40) <> 0;
              Reserved               := (RawData[1] and $30) shr 4;
              // SectionLength       := ((RawData[1] and $0F) shl 8) or
              //                        RawData[2];
              Reserved3              := (RawData[3] shl 8) or
                                        RawData[4];
              Reserved2              := (RawData[5] and $C0) shr 6;
              VersionNumber          := (RawData[5] and $37) shr 1;
              CurrentNextIndicator   := (RawData[5] and $01) <> 0;
              SectionNumber          := RawData[6];
              LastSectionNumber      := RawData[7];

              Descriptors := (SectionLength - 9);            // Descriptor bytes in section (9 = CRC + part header)
              if Descriptors > 0 then
                for Loop := 0 to Descriptors-1 do
                  DescriptorData[Loop] := RawData[Loop+8];
              Crc32 := (RawData[SectionLength+2-3] shl 24) or
                       (RawData[SectionLength+2-2] shl 16) or
                       (RawData[SectionLength+2-1] shl 8) or
                        RawData[SectionLength+2];
              if TableId = CDvbTransportPidCAT then          // Sanity check
                TableValid := True;
              // If it indicates a currently active CAT and it is new to us, update the active CAT
              if TableValid and
                 CurrentNextIndicator and
                 (CatActive.VersionNumber <> VersionNumber) then
              begin
                CatActive := CatReceiving;
                if Cats = $FFFF then
                  Cats := 0
                else
                  Inc(Cats);
              end;
            end
            else
              begin
                Inc(PacketError);
                if PacketError = $FFFF then
                  PacketError := 0;
              end;
          end;
        end;
      end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <StreamPacket>  Stream packet index
  Returns : -

  Descript: Decode program association table data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeProgramAssociationTableData(StreamPacket: Word);
var
  LoopStart    : Word;
  Loop         : Word;
  Checksum32   : Dword;
  Programs     : Byte;
begin
  try
    with StreamBuffer[StreamPacket] do
      with PatReceiving do
      begin
        if PayloadUnitStartIndicator then
        begin
          // Start of a new packet resets the indexes
          RawIndex := 0;
          ProgramIndex := 0;
          TableValid   := False;
          Sync         := True;
          // All so-called PSI tables have a first byte which is the <PointerField>.
          // This indicates at what offset the actual data starts. We know that
          // the offset points into this very first packet...
          LoopStart := DataBytes[0]+1;
          if LoopStart > DataBytesLength then                  // If incorrect value
            LoopStart := DataBytesLength;
        end
        else
        begin
          if not Sync then
            Exit;
          LoopStart := 0;                                      // Not first packet
        end;

        // We first have to put all data without processing into a buffer
        // We only check the section length if we have received all data
        // for the whole section. If so we will process it
        // We copy all data bytes although this might be more than the actual section
        if LoopStart < DataBytesLength then
        begin
          for Loop := LoopStart to DataBytesLength - 1 do
            if RawIndex < High(TDvbProgramAssociationRaw) then
            begin
              RawData[RawIndex] := DataBytes[Loop];
              Inc(RawIndex);
            end;
          // We can now make a check on if we received the whole section
          // We assume we always have received at least 3 bytes ....
          SectionLength := ((RawData[1] and $0F) shl 8) or
                             RawData[2];
          if (SectionLength < 1022) and                        // Sanity check
             ((SectionLength+2) < RawIndex) then               // If we have it all
          begin
            // Before we do any further processing at all we migth check for a correct checksum
            Checksum32 := DvbResetValueCrc32;
            for Loop := 0 to SectionLength+2 do
              Checksum32 := DvbNewValueCrc32(Checksum32, RawData[Loop]);
            // Check for correct checksum
            if Checksum32 = 0 then
            begin
              TableId                := RawData[0];
              SectionSyntaxIndicator := (RawData[1] and $80) <> 0;
              Zero                   := (RawData[1] and $40) <> 0;
              Reserved               := (RawData[1] and $30) shr 4;
              // SectionLength       := ((RawData[1] and $0F) shl 8) or
              //                        RawData[2];
              TransportStreamId      := (RawData[3] shl 8) or
                                        RawData[4];
              Reserved2              := (RawData[5] and $C0) shr 6;
              VersionNumber          := (RawData[5] and $37) shr 1;
              CurrentNextIndicator   := (RawData[5] and $01) <> 0;
              SectionNumber          := RawData[6];
              LastSectionNumber      := RawData[7];

              Programs := (SectionLength - 9) div 4;         // Programs in section (9 = CRC + part header)
              if Programs > 0 then
                for Loop := 0 to Programs-1 do
                begin
                  ProgramData[Loop+ProgramIndex].ProgramNumber   := (RawData[Loop*4+8] shl 8) or
                                                                  RawData[Loop*4+9];
                  ProgramData[Loop+ProgramIndex].Reserved        := (RawData[Loop*4+10] and $E0) shr 5;
                  if ProgramData[Loop+ProgramIndex].ProgramNumber = 0 then
                    ProgramData[Loop+ProgramIndex].NetworkPid    := ((RawData[Loop*4+10] and $1F) shl 8) or
                                                                    RawData[Loop*4+11]
                  else
                    ProgramData[Loop+ProgramIndex].ProgramMapPid := ((RawData[Loop*4+10] and $1F) shl 8) or
                                                                    RawData[Loop*4+11];
                  Inc(ProgramIndex);
                end;
              Crc32 := (RawData[SectionLength+2-3] shl 24) or
                       (RawData[SectionLength+2-2] shl 16) or
                       (RawData[SectionLength+2-1] shl 8) or
                        RawData[SectionLength+2];
              if TableId = CDvbTransportPidPAT then          // Sanity check
                TableValid := True;
              // If it indicates a currently active PAT and it is new to us, update the active PAT
              if TableValid and
                 CurrentNextIndicator and
                 (PatActive.VersionNumber <> VersionNumber) then
              begin
                PatActive := PatReceiving;
                // We could remove all PID references at this point but this will
                // mean we will loose data until all references are updated.
                // Instead we keep the old list(s) intact and overwrite it.
                // This can only lead to 'problems' when switching channels since
                // the old references are wrong by default...

                // What we must do here is to update the PID list with references we
                // acquired in the PAT. The PAT now has references to the PID which
                // carry the Program Map Tables (PMT).
                if ProgramIndex > 0 then
                 for Loop := 0 to ProgramIndex-1 do
                 begin
                   if ProgramData[Loop].ProgramNumber <> 0 then    // If not a network ID
                     PidList[ProgramData[Loop].ProgramMapPid].PidType := CDvbTypePidPmt;
                 end;
                if Pats = $FFFF then
                  Pats := 0
                else
                  Inc(Pats);
              end;
            end
            else
              begin
                Inc(PacketError);
                if PacketError = $FFFF then
                  PacketError := 0;
              end;
          end;
        end;
      end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <StreamData>    Original packet data
  Returns : <Result>        True if payload

  Descript: Decode packet data.
  Notes   :
 ------------------------------------------------------------------------------}
function DecodePacketData(StreamPacketData: TDvbTransportPacket; StreamPacket: Word): Boolean;
var
  DataIndex    : Word;
  Loop         : Word;
  Mark1        : Word;
  Mark2        : Word;
begin
  Result := False;
  DataIndex := 0;
  try
    with StreamBuffer[StreamPacket] do
    begin
      // Copy buffer to record data
      HeaderAndPayload := StreamPacketData;
      SyncByte := HeaderAndPayload[DataIndex];
      Inc(DataIndex);
      TransportErrorIndicator    := (HeaderAndPayload[DataIndex] and $80) <> 0;
      PayloadUnitStartIndicator  := (HeaderAndPayload[DataIndex] and $40) <> 0;
      TransportPriority          := (HeaderAndPayload[DataIndex] and $20) <> 0;
      Pid                        := ((HeaderAndPayload[DataIndex] and $1F) shl 8) or
                                    HeaderAndPayload[DataIndex+1];
      // If new PID mark/count it
      if not TransportPids[Pid] then
      begin
        Inc(TransportPidsCounter);
        TransportPids[Pid] := True;
      end;
      Inc(DataIndex, 2);
      TransportScramblingControl := (HeaderAndPayload[DataIndex] and $C0) shr 6;
      AdaptationFieldControl     := (HeaderAndPayload[DataIndex] and $30) shr 4;

      ContinuityCounter          := (HeaderAndPayload[DataIndex] and $0F);
      Inc(DataIndex);
      if (AdaptationFieldControl = 2) or
         (AdaptationFieldControl = 3) then
      begin
        AdaptationFieldLength    := HeaderAndPayload[DataIndex];
        Inc(DataIndex);
        Mark1 := DataIndex;                                  // Mark index for stuffing
        if AdaptationFieldLength > 0 then
        begin
          DiscontinuityIndicator             := (HeaderAndPayload[DataIndex] and $80) <> 0;
          RandomAccessIndicator              := (HeaderAndPayload[DataIndex] and $40) <> 0;
          ElementaryStreamPriorityIndicator  := (HeaderAndPayload[DataIndex] and $20) <> 0;
          PcrFlag                            := (HeaderAndPayload[DataIndex] and $10) <> 0;
          OpcrFlag                           := (HeaderAndPayload[DataIndex] and $08) <> 0;
          SplicingPointFlag                  := (HeaderAndPayload[DataIndex] and $04) <> 0;
          TransportPrivateDataFlag           := (HeaderAndPayload[DataIndex] and $02) <> 0;
          AdaptationFieldExtensionFlag       := (HeaderAndPayload[DataIndex] and $01) <> 0;
          Inc(DataIndex);
          if PcrFlag then
          begin
            // Since we can not shift > 32 bits we have to handle a 33 bit shift somewhat differently
            ProgramClockReferenceBase      := (HeaderAndPayload[DataIndex]   shl 32) or
                                              (HeaderAndPayload[DataIndex+1] shl 16) or
                                              (HeaderAndPayload[DataIndex+2] shl 8)  or
                                              (HeaderAndPayload[DataIndex+3]);
            ProgramClockReferenceBase      := (ProgramClockReferenceBase * 2) or
                                              ((HeaderAndPayload[DataIndex+4] shr 7) and $1);
            Inc(DataIndex, 4);
            PcrReserved                    := (HeaderAndPayload[DataIndex] shr 1) and $3F;
            ProgramClockReferenceExtension := ((HeaderAndPayload[DataIndex] and 1) shl 8) or
                                              HeaderAndPayload[DataIndex+1];
            Inc(DataIndex, 2);
          end;
          if OpcrFlag then
          begin
            // Since we can not shift > 32 bits we have to handle a 33 bit shift somewhat differently
            OriginalProgramClockReferenceBase      := (HeaderAndPayload[DataIndex]   shl 32) or
                                                      (HeaderAndPayload[DataIndex+1] shl 16) or
                                                      (HeaderAndPayload[DataIndex+2] shl 8)  or
                                                      (HeaderAndPayload[DataIndex+3]);
            OriginalProgramClockReferenceBase      := (OriginalProgramClockReferenceBase * 2) or
                                                      ((HeaderAndPayload[DataIndex+4] shr 7) and $1);
            Inc(DataIndex, 4);
            OpcrReserved                           := (HeaderAndPayload[DataIndex] shr 1) and $3F;
            OriginalProgramClockReferenceExtension := ((HeaderAndPayload[DataIndex] and 1) shl 8) or
                                                      HeaderAndPayload[DataIndex+1];
            Inc(DataIndex, 2);
          end;
          if SplicingPointFlag then
          begin
            SpliceCountdown := HeaderAndPayload[DataIndex];
            Inc(DataIndex);
          end;
          if TransportPrivateDataFlag then
          begin
            TransportPrivateDataLength := HeaderAndPayload[DataIndex];
            Inc(DataIndex);
            if (TransportPrivateDataLength > 0) and
               (TransportPrivateDataLength < High(PrivateDataByte)) then
              for Loop := 0 to TransportPrivateDataLength-1 do
              begin
                PrivateDataByte[Loop] := HeaderAndPayload[DataIndex];
                Inc(DataIndex);
                // Sanity check 
                if DataIndex > High(HeaderAndPayload) then
                  Exit;
              end;
          end;
          if AdaptationFieldExtensionFlag then
          begin
            AdaptationFieldExtensionLength := HeaderAndPayload[DataIndex];
            Inc(DataIndex);
            Mark2 := DataIndex;                              // Mark index for stuffing
            LtwFlag                     := (HeaderAndPayload[DataIndex] and $80) <> 0;
            PiecewiseRateFlag           := (HeaderAndPayload[DataIndex] and $40) <> 0;
            SeamlessSpliceFlag          := (HeaderAndPayload[DataIndex] and $20) <> 0;
            AdaptationExtensionReserved := (HeaderAndPayload[DataIndex] and $1F);
            Inc(DataIndex);
            if LtwFlag then
            begin
              LtwValidFlag := (HeaderAndPayload[DataIndex] and $80) <> 0;
              LtwOffset    := ((HeaderAndPayload[DataIndex] and $7F) shl 8) or
                              HeaderAndPayload[DataIndex+1];
              Inc(DataIndex, 2);
            end;
            if PiecewiseRateFlag then
            begin
              PiecewiseReserved := (HeaderAndPayload[DataIndex] shr 6) and $3;
              PiecewiseRate     := ((HeaderAndPayload[DataIndex] and $3F) shl 16) or
                                   (HeaderAndPayload[DataIndex+1] shl 8) or
                                   HeaderAndPayload[DataIndex+2];
              Inc(DataIndex, 3);
            end;
            if SeamlessSpliceFlag then
            begin
              SpliceType := (HeaderAndPayload[DataIndex] shr 4) and $0F;
              DtsNextAu := (HeaderAndPayload[DataIndex] shr 1) and $07;
              Inc(DataIndex);
              DtsNextAu := (DtsNextAu * 256) or
                           HeaderAndPayload[DataIndex];
              DtsNextAu := (DtsNextAu * 128) or
                           ((HeaderAndPayload[DataIndex+1] shr 1) and $7F);
              Inc(DataIndex, 2);
              DtsNextAu := (DtsNextAu * 256) or
                           HeaderAndPayload[DataIndex];
              DtsNextAu := (DtsNextAu * 128) or
                           ((HeaderAndPayload[DataIndex+1] shr 1) and $7F);
              Inc(DataIndex, 2);
            end;
            while (DataIndex - Mark2) < AdaptationFieldExtensionLength do
              Inc(DataIndex);
          end;
        end;
        while (DataIndex - Mark1) < AdaptationFieldLength do
          Inc(DataIndex);
      end
      else
        AdaptationFieldLength := 0;
      DataBytesLength := 0;
      if (AdaptationFieldControl = 1) or
         (AdaptationFieldControl = 3) then
      begin
        Result := True;                                      // Indicate payload included
        while DataIndex < Sizeof(TDvbTransportPacket) do
        begin
          DataBytes[DataBytesLength] := HeaderAndPayload[DataIndex];
          Inc(DataBytesLength);
          Inc(DataIndex);
        end;
      end;

      // Note: the SyncByte **might** be complemented every 8 packets ...
      if ((SyncByte = CDvbPacketSyncWord) or
          (SyncByte = (CDvbPacketSyncWord xor $FF))) and (not TransportErrorIndicator) then
        TransportValid := True
      else
        TransportValid := False;

      // Check indexes for debugging purposes
      if DataIndex <> CDvbPacketSize then
      begin
        Inc(PacketError);
        if PacketError = $FFFF then
          PacketError := 0;
      end;
    end;
  except
    // We might be getting here if the packet data is incorrect and the
    // <DataIndex> points outside the valid range
  end
end;


{------------------------------------------------------------------------------
  Params  : <StreamData>  Stream packets
  Returns : -

  Descript: Decode transport stream. Uses <StreamDataBuffer> which holds
            multiple stream packets.
  Notes   :
 ------------------------------------------------------------------------------}
procedure DecodeTransportStreamData(StreamData: PDvbTransportPackets);
var
  StreamPacket: Word;
begin
  for StreamPacket := 0 to CDvbPacketVSync-1 do
  begin
    // Decode packet
    if DecodePacketData(StreamData[StreamPacket], StreamPacket) then
    begin
      // If payload in packet
      // Check PID
      // There are different PID's. PID's which indicate information (PSI) or data (PES).
      // We first have to receive PSI data before we can identify the PES data.
      // The structure is as follows:
      //   Program Association Table (PAT)
      //      -> PAT indicates other PID's for the Program Map Table (PMT)
      //          -> PMT indicates other PID's for the individual streams (PES)
      // Thus at first we have no information at all. Then, after receiving the PAT
      // we can identify the PID's which carry PMT data. Then we can identify
      // PID's used for stream data (PES).
      case StreamBuffer[StreamPacket].Pid of
        // If we have a Program Association Table packet
        CDvbTransportPidPat: DecodeProgramAssociationTableData(StreamPacket);
        // If we have a Conditional Access Table packet
        CDvbTransportPidCat: DecodeConditionalAccessTableData(StreamPacket);
        // A null packet does not require any handling
        CDvbTransportPidNull: ;
        // All other PID's are user assignable and are only available after receiving
        // the PAT.
        else
        begin
          case PidList[StreamBuffer[StreamPacket].Pid].PidType of
            CDvbTypePidPmt: DecodeProgramMapTableData(StreamPacket);
            CDvbTypePidPes: DecodeProgramElementaryStreamData(StreamPacket);
            CDvbTypePidPcr: ;
          end;
        end;
      end;
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrupt at both edges of the field indicator,
            meaning that we get notified when a switch is made from
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Data       : Dword;
  LastBuffer : Dword;
  Buffer     : TSaa7146aTransferBuffer;
  SkipBuffer : Byte;                             // Forces buffers to skip after first setup
begin
  // Debug
  PacketBufferCount :=0;

  HasStopped     := False;
  ProgrammedStop := False;
  // The driver must be running and must be in use
  if Saa7146aGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't then only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls eg. 'generate manual notification'
  // which might be needed in a deadlock situation.
  ThreadHandle := Saa7146aCreateFile(CardInUse);
  Buffer.Identifier      := Fifo3.Identifier;
  Buffer.TransferAddress := @StreamDataBuffer;
  Buffer.TargetIndex     := 0;
  Buffer.TransferLength  := Sizeof(StreamDataBuffer);
  
  // Wait for application to have started and such
  repeat
   Sleep(100);
  until NewChannelSet or Terminated;

  LastBuffer := 0;
  Data       := $FFFF;
  SkipBuffer := 0;
  try
    if not Terminated then
    repeat
      // Some special handling required when a new channel has been set
      if NewChannelSet then
      begin
        PacketError  := 0;
        Overtaken    := 0;
        CreateDvbTables;
        // Skip first xx buffers
        // We use more buffers when first startup (not REALLY necessary), since
        // only data is processed when correctly tuned ....
        if Data = $FFFF then
          SkipBuffer := PacketBuffers
        else
          SkipBuffer := 1;
        NewChannelSet := False;
      end;
      if not Terminated then
      begin
        if PacketBufferCount = $FFFF then
          PacketBufferCount := 0
        else
          Inc(PacketBufferCount);
        // Depending if additional buffering is used determine the address
        if PacketBuffers <= 1 then
        begin
          Saa7146aWaitForNotification(ThreadHandle);
          // No additional buffering (all handled without aid)
          // We need to know which buffer we need to read (odd or even)
          Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aPsr, Data);
          if Data and (1 shl CSaa7146aIrqFidB) = 0 then
            Buffer.SourceIndex := 0
          else
            Buffer.SourceIndex := CDvbPacketBufferSize;
        end
        else
        begin
          // First check if there are already buffers filled
          Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
          // Just in case invalid Data
          if Data >= (PacketBuffers*2) then
            Data := 0;
          // If no new data, wait (or if first time)
          if Data = LastBuffer then
          begin
            if IsSlave then
            repeat
              Sleep(1);
              Saa7146aReadFromSaa7146aRegister(ThreadHandle, CSaa7146aEcT1R, Data);
            until Terminated or (Data <> LastBuffer)
            else
              Saa7146aWaitForNotification(ThreadHandle);
          end
          else
          begin
            if OverTaken = $FFFF then
              OverTaken := 0
            else  
            Inc(Overtaken);
          end;
          Buffer.SourceIndex := CDvbPacketBufferSize * LastBuffer;
          // Next buffer to wait for
          Inc(LastBuffer);
          // Circular buffer
          if LastBuffer = (PacketBuffers*2) then
            LastBuffer := 0;
        end;
        if SkipBuffer = 0 then
        begin
          // Get data in local buffer
          Saa7146aReadFromDma(ThreadHandle, Buffer);
          // Process packet data
//          DecodeTransportStreamData(@StreamDataBuffer);
          if Assigned(frmMain) and
             ShowVideo then
            DirectShowUniversalSourceSendData(@StreamDataBuffer, CDvbPacketBufferSize);
        end
        else
          Dec(SkipBuffer);
      end;
    until Terminated;
  finally
    HasStopped := True;
    if not ProgrammedStop then
      ShowMessage('Unexpected termination of packets handler.');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    $FFFF if no master

  Descript: Detect a POSSIBLE master (fails if for some reason DMA memory
            was not released, eg. because of a crash).
  Notes   :
 ------------------------------------------------------------------------------}
function GetMasterChannel: Dword;
var
  Info          : TSaa7146aGetDmaStatus;
  Channel       : Dword;
  ChannelFound  : Boolean;
begin
  // As a slave we only need the identifier of the DMA buffer which was
  // used by the master.
  // The only way we can do is to check the size of such a buffer..
  Channel := 0;
  ChannelFound := False;
  while (Saa7146aGetDmaStatus(CardHandle, Channel, Info) and (not ChannelFound)) and (Channel < 256) do
  begin
    if Info.Size = (PacketBuffers * ((CDvbPacketBufferSize * 2) div DMASIZE) + 1) * DMASIZE then
      ChannelFound := True
    else
      Inc(Channel);
  end;
  if ChannelFound then
    Result := Channel
  else
    result := $FFFF;
end;


{------------------------------------------------------------------------------
  Params  : <Activate>  True if to activate infared
  Returns : <Result>    True if success

  Descript: Setup buffering of DMA3 data.
  Notes   :
 ------------------------------------------------------------------------------}
procedure BufferingSetup;
var
  Status        : TSaa7146aDmaBuffer;
  Info          : TSaa7146aGetDmaStatus;
  TransferBuffer: array[0..PacketBuffers*30] of Dword;
  Index         : Integer;
  Transfer      : TSaa7146aTransferBuffer;
  Buffers       : Byte;
  Channel       : Dword;
begin
  // Before we can activate FIFO3 transfer we have to set the DMA settings.
  // One of those settings is the target memory data will be saved to. For
  // this we use the DMA memory feature of the driver, since the memory it
  // allocates is always valid. Since the driver has allocated continuous
  // physical memory we don't have to use the MMU.

  // Note: If we are acting as a slave then we don't setup anything, but
  //       instead readout the DMA information

  if IsSlave then
  begin
    // As a slave we only need the identifier of the DMA buffer which was
    // used by the master.
    // The only way we can do is to check the size of such a buffer..
    Channel := GetMasterChannel;
    if Channel <> $FFFF then
    begin
      Saa7146aGetDmaStatus(CardHandle, Channel, Info);
      Fifo3.Identifier      := Channel;
      Fifo3.VirtualAddress  := Info.VirtualAddress;
      Fifo3.PhysicalAddress := Info.PhysicalAddress;
      Fifo3.Size            := Info.Size;
    end
    else
      ShowMessage('The driver is used as slave but could not detect an active master.');
  end
  else
  begin
    // First release all already allocated memory
    Channel := 0;
    while Saa7146aReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    // Disable all interrupts
    Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIer, 0);

    // Allocate a DMA buffer which will hold the FIFO 3 data
    // No matter how we will be doing thing, we can expect a maximum of
    // 2 * 512 (VS) * 188 (HS) = 192512 == 96256 words == 48128 dwords
    // The VS changes every 512 packets. The HS is activated for every packet.
    // Every packet is 188 bytes. The factor '2' is for the 'odd' and 'even' fields
    // We can have more than 1 buffer here.
    if not (Saa7146aAllocateDma(CardHandle, (PacketBuffers * ((CDvbPacketBufferSize * 2) div DMASIZE) + 1) * DMASIZE, Status)) then
      Exit;
    // Save info of allocated memory
    Fifo3.Identifier      := Status.Identifier;
    Fifo3.VirtualAddress  := Status.VirtualAddress;
    Fifo3.PhysicalAddress := Status.PhysicalAddress;
    Fifo3.Size            := Status.Size;

    // Now we have allocated the memory we can setup the DMA3 for it.
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aBaseOdd3,                              // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart)) then      // Odd lines to here
      Exit;
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aBaseEven3,                             // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart + CDvbPacketBufferSize)) then     // Even lines to here
      Exit;

    // We must make sure we don't get a protection error
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aProtAddr3,                             // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart + (PacketBuffers * ((CDvbPacketBufferSize * 2) div DMASIZE) + 1) * DMASIZE)) then
      Exit;

    // If we are using 1 buffer only then no special handling is necessary.
    if PacketBuffers <= 1 then
      Exit;

    // If we want more buffering then we use one of the RPS programs
    // of the SAA7146A to handli it. What this program does is nothing
    // more then changing the buffer addresses when it's necessary,
    // and therby extending out data buffer.

    // Allocate some DMA buffers which will hold the RPS program
    if not (Saa7146aAllocateDma(CardHandle, DMASIZE * 1, Status)) then
      Exit;
    // Save info of allocated memory
    BufferRPS1.Identifier      := Status.Identifier;
    BufferRPS1.VirtualAddress  := Status.VirtualAddress;
    BufferRPS1.PhysicalAddress := Status.PhysicalAddress;
    BufferRPS1.Size            := Status.Size;

    // Now setup the RPS program. We first construct it in local memory and
    // then transport it to the SAA7146A
    Index := 0;
    for Buffers := 0 to PacketBuffers-1 do
    begin
      // Wait for even fields starting to be processed
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsEFidB;
      Inc(Index);
      // Now the even fields are being processed, change the odd field buffer address
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aBaseOdd3 shr 2);  // Write DMA3 base even field in shadow RAM
      Inc(Index);
      TransferBuffer[Index] := Fifo3.PhysicalAddress.LowPart + (CDvbPacketBufferSize * (Buffers*2));   // New address
      Inc(Index);
      // Needed to clear upload flag ??
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsVtd3;          // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsVtd3;         // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsVtd3;          // Wait for shadow upload to finish
      Inc(Index);
      // For feedback we use the event counter threshold register
      // We indicate which buffer is ready to be read, which is one before
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aEcT1R shr 2);    // Set indicator
      Inc(Index);
      if Buffers = 0 then
        TransferBuffer[Index] := (PacketBuffers*2) - 2
      else
        TransferBuffer[Index] := (Buffers*2) - 2;
      Inc(Index);

      TransferBuffer[Index] := CSaa7146aRpsInterrupt or CSaa7146aRpsLogicalOr or CSaa7146aRpsVtd3;    // Issue interrupt
      Inc(Index);

      // Wait for odd fields starting to be processed
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsOFidB;
      Inc(Index);
      // Now the odd fields are being processed, change the even field buffer address
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aBaseEven3 shr 2);  // Write DMA3 base even field in shadow RAM
      Inc(Index);
      TransferBuffer[Index] := Fifo3.PhysicalAddress.LowPart + (CDvbPacketBufferSize * ((Buffers * 2)+1)); // New address
      Inc(Index);
      // Needed to clear upload flag ??
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsVtd3;      // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);

      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsVtd3;         // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsVtd3;          // Wait for shadow upload to finish
      Inc(Index);
      // For feedback we use the event counter threshold register
      // We indicate which buffer is ready to be read, which is one before
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aEcT1R shr 2);    // Set indicator
      Inc(Index);
      if Buffers = 0 then
        TransferBuffer[Index] := (PacketBuffers*2)-1
      else
        TransferBuffer[Index] := (Buffers*2)-1;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsInterrupt or CSaa7146aRpsLogicalOr or CSaa7146aRpsVtd3;    // Issue interrupt
      inc(Index);
    end;
    // Jump to start program
    TransferBuffer[Index] := CSaa7146aRpsJump;                                 // Restart program
    Inc(Index);
    TransferBuffer[Index] := DWord(BufferRPS1.PhysicalAddress.LowPart);        // Address of start program
    Inc(Index);
    // Setup transfer data and transfer it
    Transfer.Identifier   := BufferRPS1.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Index * Sizeof(DWord);
    Saa7146aWriteToDma(CardHandle, Transfer);              // Transfer data
    // Now make sure the RPS program is started
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsAddr1,
      DWord(DWord(BufferRPS1.PhysicalAddress.LowPart)));   // Set physical start of RPS program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsPage1,
      0);                                                  // RPS program performs no explicit mem writes
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsTov1 ,
      0);                                                  // Disable RPS timeouts
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps1On);                                 // Start RPS1 program
  end;    
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  ErrorMessage: string;
begin
  // Check if the driver is present and/or multiple cards are present
  case Saa7146aGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No SAA7146A driver and/or card detected.');
  end;
  if IsSlave then
    mmoDriver.Lines.Add('Interface set to operate as slave.')
  else
    mmoDriver.Lines.Add('Interface set to operate as master.');

  // We use the tag for the returned identifier
  if not DirectShowStart(frmMain.pnlVideo.Handle, 'VIDEO.GRF', False, False, False, ErrorMessage, INVALID_HANDLE_VALUE) then
    frmMain.mmoDriver.Lines.Add('DirectShow error')
  else
  begin
    frmMain.MskPidChange(nil);  // Send PID info to DirectShow
    ShowVideo := True;
  end;

  NewChannelSet := True;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Form destruction.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  // Note: We MUST make sure that <DSSendDataDirect> is never called when DirectShow
  //       has stopped, therefore we inform the thread
  ShowVideo := False;
  Sleep(1);

  DirectShowStop;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.rgPolarityClick(Sender: TObject);
begin
  case rgPolarity.ItemIndex of
    0: if not Saa7146aHorizontalPolarityLNB(CardHandle, (TunerType = CTunerBSRU6)) then
         mmoDriver.Lines.Add('Could not set horizontal polarity.');
    1: if not Saa7146aVerticalPolarityLNB(CardHandle, (TunerType = CTunerBSRU6)) then
         mmoDriver.Lines.Add('Could not set vertical polarity.');
  end;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  if not Saa7146aQpskSetSymbolRate(CardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  StrengthDecibel   : Double;
  StrengthPercentage: Double;
  DeltaTime : Dword;
  DeltaCount: Dword;
  Ticks     : Dword;
begin
  // Calculate time for each packetbuffer
  Ticks := GetTickCount;
  DeltaTime := Abs(Ticks - PacketPreviousTimer);
  PacketPreviousTimer := Ticks;
  if PacketBufferCount <> PacketPreviousCount then
  begin
    DeltaCount := PacketBufferCount - PacketPreviousCount;
    txtMs.Caption := IntToStr(DeltaTime div DeltaCount);
    PacketPreviousCount := PacketBufferCount;
  end;
  txtPacketCount.Caption := IntToStr(PacketBufferCount);
  txtOvertaken.Caption   := IntToStr(Overtaken);

  // Don't handle updating other information if we are not the master
  if IsSlave then
    Exit;

  if Saa7146aQpskGetLockDetectorPercentage(CardHandle, Percentage) then
    ggLock.Progress := Percentage;
  if Saa7146aQpskGetCarrierDeviation(CardHandle, Deviation) then
    stDeviation.Caption := format('%d', [Deviation]);
  if Saa7146aQpskGetSymbolRateDeviation(CardHandle, Deviation) then
    stDeviationSymbolRate.Caption := format('%d', [Deviation]);
  if Saa7146aQpskGetSignalPower(CardHandle, StrengthPercentage, StrengthDecibel) then
  begin
    txtSignalPower.Caption := format('%f%% (%f dB)', [StrengthPercentage, StrengthDecibel]);
  end;
  if Saa7146aReadFromQpsk(CardHandle, CStv0299bVStatus, LockState) then
  begin
    if (LockState and $80) <> 0 then
      StatusStr := 'CARRIER'
    else
      StatusStr := 'NO CARRIER';
    if (LockState and $10) <> 0 then
      StatusStr := StatusStr + ' - VITERBI'
    else
      StatusStr := StatusStr + ' - NO VITERBI';
    if (LockState and $08) <> 0 then
      StatusStr := StatusStr + ' - SYNC'
    else
      StatusStr := StatusStr + ' - NO SYNC';
    if (LockState and $98) = $98 then
      StatusStr := StatusStr + ' - LOCK'
    else
      StatusStr := StatusStr + ' - NO LOCK';
    txtSignalState.Caption := StatusStr;
  end;
  if Saa7146aQpskGetNoiseIndicator(CardHandle, Noise) then
  begin
    txtNoiseIndicator.Caption := format('%f dB', [Noise]);
  end;

end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error:     Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert frequency.');
    Exit;
  end;
  // Befi

  if not Saa7146aFrequencyLNB(CardHandle, Frequency, 9750000, 10600000, TunerType, SynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;

procedure TfrmMain.mskPidKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskPidChange(nil);
end;

procedure TfrmMain.btnClearClick(Sender: TObject);
begin
  NewChannelSet := True;
  mmoDriver.Clear;
end;

procedure TfrmMain.mskPidChange(Sender: TObject);
var
  Error    : Integer;
  PidVideo : Integer;
  PidAudio : Integer;
begin
  Val(mskVideoPid.EditText, PidVideo, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Video PID.');
    Exit;
  end;
  Val(mskAudioPid.EditText, PidAudio, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert Audio PID.');
    Exit;
  end;

  DirectShowMpeg2demultiplexerSetNewPids(PidVideo, PidAudio);
end;



{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Setup data throughput of driver (packet data)
  Notes   : This will setup the SAA7146A so data from the tuner is passed through
            the DD1 port. The DD1 port has the unprocessed data.
 ------------------------------------------------------------------------------}
function SetupDd1: Boolean;
var
  IrqHandling: TSaa7146aIrqTransferBuffer;
  Ie         : Dword;
begin
  Result := False;
  // If the thread handling data gathering is active: close it
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    Saa7146aGenerateManualNotification(ThreadHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;

  // Only setup the interface when we are the master
  if not IsSlave then
  begin
    // Set initial settings of dual D1 interface
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aDd1Init,                               // Register to write
             CSaa7146aDd1LlcInput or                         // LLC input
             CSaa7146aDd1SioInput or                         // HS/VS inputs
             CSaa7146aDd1PvoDirect or                        // Direct polarity VS
             CSaa7146aDd1PhoDirect or                        // Direct polarity HS
             CSaa7146aDd1Sync6 or                            // Sync edge selection, HS rising, VS both frame, Field direct
             CSaa7146aDd1FidesEdgesIrq) then                 // Field both edges interrupt
      Exit;
    // Above registers need an upload to activate the setting
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc2,                                   // Register to write
             CSaa7146aMc2UpldDd1Initial) then                // Upload initial settings
      Exit;
    // Set video stream handling of port
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aDd1Data,                               // Register to write
             CSaa7146aDd1VidInput or                         // Port and PXQ inputs
             CSaa7146aDd1Y8cNoSavEav or                      // No SAV/EAV data
             CSaa7146aDd1PfidDirect) then                    // Direct polarity change
      Exit;
    // Above registers need an upload to activate the setting
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc2,                                   // Register to write
             CSaa7146aMc2UpldDd1A) then                      // Upload port A settings
      Exit;
    // Above registers need an upload to activate the setting
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc2,                                   // Register to write
             CSaa7146aMc2UpldDd1B) then                      // Upload port B settings
      Exit;
    // Set BRS control (we use the data from the D1 ports directly (unprocessed) so we need
    // to use the BRS and consequently FIFO3/DMA3
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aBrsCtrl,                               // Register to write
             CSaa7146aDd1BrsStreamInB or                     // Video data stream from port B
             CSaa7146aDd1BrsSyncInB or                       // Sync signals from port B
             (0 shl CSaa7146aDd1BrsVOffsetShift) or          // Vertical offset
             CSaa7146aDd1BrsVScale1 or                       // Vertical scale 1:1
             (0 shl CSaa7146aDd1BrsHOffsetShift) or          // Horizontal offset
             CSaa7146aDd1BrsHScale1 or                       // Horizontal scale 1:1
             CSaa7146aDd1BrsFormatYuv) then                  // YUV 4:2:2 format
      Exit;
    // Above registers need an upload to activate the setting
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc2,                                   // Register to write
             CSaa7146aMc2UpldBrs) then                       // Upload BRS registers
      Exit;

    // We have setup the basic stuff.
    // When the DD1 is activated (at this point it is NOT), data will be
    // acquired and stored in FIFO3. At some point the data has to be transferred
    // from this FIFO to the computers memory. This is done with DMA3.
    // If we activate the DD1 port now, we should 'see' FIFO3 filling and
    // generating (error) signals when it has been filled (without it being
    // able to transfer it any further).

    // Enable the DD1 hardware.
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc1,                                   // Register to write
             CSaa7146aMc1Dd1Enable) then                     // Enable real time video ports (DD1)
      Exit;
  end;

  // Setup buffering/addresses
  BufferingSetup;

  // Only continue to setup the interface when we are the master
  if not IsSlave then
  begin
    // Setup distance between start addresses for consecutive lines of a field
    // If '0' data stays in first page (4096) bytes of selected memory
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aPitch3,                                // Register to write
             CDvbPacketHSync) then
      Exit;

    // Setup field/lines/bytes
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aNumLines3,                             // Register to write
             (CDvbPacketVSync shl 16) or CDvbPacketHSync) then
      Exit;

    // For more effective transfers we can extend the burst/threshold at which data is
    // transferred.
    if not Saa7146aWriteWithMaskToSaa7146aRegister(CardHandle,
             CSaa7146aPciBtV,
             $001F0000,
             $00170000) then                                 // Burst 32 dwords and 32 dwords threshold
      Exit;

    // Above registers need an upload to activate the setting
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc2,                                   // Register to write
             CSaa7146aMc2UpldDma3) then                      // Upload DMA3 settings
      Exit;

    // Enable FIFO3 transfers.
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aMc1,                                   // Register to write
             CSaa7146aMc1TrVideo3Enable) then                // Enable transfers video 3
      Exit;
  end;

  // Start the thread which receives notifications we setup next
  PacketThread := TDataThread.Create(True);
  PacketThread.Priority := tpTimeCritical;
  PacketThread.Resume;

  // Only continue to setup the interface when we are the master
  if not IsSlave then
  begin
    // For recording purposes or additional buffering we setup an interrupt.
    // For now just record interrupts, nothing more.
    with IrqHandling.Information do
    begin
      Irqs                 := 0;
      IrqsWhenActive       := 0;
      IrqBufferingIsActive := True;
      UseNotificationEvent := True;
      UseSignaling         := False;
      UseFifo              := False;
      IrqAutoDisable       := False;
    end;
    // Depending on if the RPS program is assisting in buffering setup the correct
    // interrupt source
    if PacketBuffers <= 1 then
      IrqHandling.Identifier := CSaa7146aIrqFidB
    else
      IrqHandling.Identifier := CSaa7146aIrqRpsI1;
    Saa7146aWriteToIrqHandling(CardHandle, IrqHandling);
    // Enable interrupt
    if not Saa7146aReadFromSaa7146aRegister(CardHandle, CSaa7146aIer, Ie) then
      Exit;
    if PacketBuffers <= 1 then
      Ie := Ie or (1 shl CSaa7146aIrqFidB)
    else
      Ie := Ie or (1 shl CSaa7146aIrqRpsI1);
    if not Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIer, Ie) then
      Exit;
    // Clear all pending interrupts
    if not Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIsr, $FFFFFFFF) then
      Exit;
  end;    
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Do a (re)initialization of the card
  Notes   :
 ------------------------------------------------------------------------------}
function InitializeCard: Boolean;
var
  ReturnedByte : Byte;
begin
  Result := False;
  // The RESET of the tuner is connected to a GPIO line of the SAA7146A, so we have to configure this
  if not Saa7146aInitializeGpio(CardHandle, (TunerType = CTunerBSRU6)) then
    Exit;
  // Now issue a reset signal
  if (TunerType = CTunerBSRU6) then
  begin
    if not Saa7146aIssueReset(CardHandle, 2) then
      Exit;
  end;
  // Initialize the I2C
  if not Saa7146aInitializeI2c(CardHandle) then
    Exit;

  SynthesizerAddress := Saa7146aDetectSynthesizerAddress(CardHandle);

  // Check QPSK
  if not Saa7146aReadFromQpsk(CardHandle, CStv0299bId, ReturnedByte) then
    Exit;
  // Check identifier of QPSK
  if not ReturnedByte = $A1 then
    Exit;
  // Set default QPSK values
  if not Saa7146aWriteDefaultsToQpsk(CardHandle, TunerType) then
    Exit;
  // Set inversion on
  if not Saa7146aQpskSetInversion(CardHandle, True) then
    Exit;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
begin
  NilDvbTables;

  NewChannelSet := False;
  ShowVideo     := False;
  Timing        := 0;
  Counts        := 0;
  IsSlave       := False;
  PacketThread  := nil;
  Fifo3.Size    := 0;
  CardHandle    := INVALID_HANDLE_VALUE;
  CardNoError   := False;
  CardInUse     := -1;
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    CardHandle := Saa7146aCreateFile(CardInUse);
    CardInUse  := Saa7146aGetCardOfHandle(CardHandle);
  end;

  if CardHandle <> INVALID_HANDLE_VALUE then
  begin
    // If we are set to be allowed a slave then try detecting a master
    // If there is no master detected then be a master ourselves
    if GetMasterChannel = $FFFF then
      IsSlave := False
    else
      IsSlave := True;

// IsSlave := False;  // Necessary when we did not terminate properly
    TunerType := CTunerBSRU6;
    if ParamCount <> 0 then
      if ParamStr(1) = '1' then
        TunerType := CTunerSU1278;

    // Initialize card
    if not IsSlave then
      InitializeCard;
    // Setup data transfer from tuner to memory
    SetupDd1;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
var
  Channel: Dword;
begin
  if Assigned(PacketThread) and
    (not PacketThread.HasStopped) then                     // If already running
  begin
    PacketThread.ProgrammedStop := True;
    PacketThread.Terminate;                                // Mark it for termination
    if PacketThread.Suspended then
      PacketThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    Saa7146aGenerateManualNotification(ThreadHandle);
    PacketThread.WaitFor;
    FreeAndNil(PacketThread);
  end;
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;
  // Software reset. We migth be capturing data, so stop ALL
  if not IsSlave then
  begin
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1, CSaa7146aMc1SoftReset);
    // Release all allocated memory by the driver
    Channel := 0;
    while Saa7146aReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    Saa7146aDisableLNB(CardHandle, (TunerType = CTunerBSRU6));
    Saa7146aCloseHandle(CardHandle);
  end;
  CardHandle := INVALID_HANDLE_VALUE;

  DestroyDvbTables;
end;


initialization
  Initialize;


finalization
  Finalize;
end.
