{******************************************************************************}
{ FileName............: GettingPacketDataUnit001                               }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Getting packet data.                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030529 - Initial WDM release                                       }
{******************************************************************************}
unit GettingPacketDataUnit001;

interface

uses
  SyncObjs,
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Saa7146AInterface, Saa7146aGpio, Saa7146AI2c, Stv0299bRegisters,
  Tsa5059Registers, ExtCtrls, Mask, Gauges,
  DvbRegisters,
  Saa7146aIoControl,
  Saa7146aRegisters;

type
  TDataThread = class(TThread)
  private
    { Private declarations }
  protected
    constructor Create;
    procedure Execute; override;
  end;

  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    rgPolarity: TRadioGroup;
    mskSymbolRate: TMaskEdit;
    StaticText1: TStaticText;
    tmrUpdate: TTimer;
    ggLock: TGauge;
    StaticText2: TStaticText;
    StaticText3: TStaticText;
    mskFrequency: TMaskEdit;
    StaticText4: TStaticText;
    stDeviation: TStaticText;
    StaticText5: TStaticText;
    StaticText6: TStaticText;
    StaticText7: TStaticText;
    stDeviationSymbolrate: TStaticText;
    btnBeforeLock: TButton;
    btnAfterLock: TButton;
    rgLNB: TRadioGroup;
    btnReset: TButton;
    StaticText8: TStaticText;
    rgInversion: TRadioGroup;
    btnSetupDd1: TButton;
    btnResetSAA7146: TButton;
    StaticText9: TStaticText;
    StaticText10: TStaticText;
    StaticText11: TStaticText;
    StaticText12: TStaticText;
    StaticText13: TStaticText;
    StaticText14: TStaticText;
    StaticText17: TStaticText;
    StaticText18: TStaticText;
    StaticText19: TStaticText;
    StaticText15: TStaticText;
    rgTuner: TRadioGroup;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure btnBeforeLockClick(Sender: TObject);
    procedure btnAfterLockClick(Sender: TObject);
    procedure rgLNBClick(Sender: TObject);
    procedure btnResetClick(Sender: TObject);
    procedure rgInversionClick(Sender: TObject);
    procedure btnSetupDd1Click(Sender: TObject);
    procedure btnResetSAA7146Click(Sender: TObject);
    procedure rgTunerClick(Sender: TObject);
  private
    { Private declarations }
    FCardHandle : THandle;
    FCardNoError: Boolean;
    FFrequencySearch: Boolean;
    FFrequencySearchCountDown: Byte;
    FSynthesizerAddress: Byte;
    FThread: TDataThread;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;
  Timing : Dword;
  Counts : Dword;
  ThreadCounts: Dword;
  OddCounts   : Dword;
  EvenCounts  : Dword;
  Fifo3       :  TSaa7146aDmaBuffer;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrup at the both edges of the field indicator,
            meaning that we get notified when a switch is made form
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  LCardHandle: THandle;                                    // Handle to driver
  Data       : Dword;
  Buffer     : TSaa7146aTransferBuffer;
  DvbData    : array[0..CDvbPacketVSync * CDvbPacketHSync-1] of Byte;
  OutFile    : TFileStream;
  AutoEnd    : Dword;
begin
  // The driver must be running and must be in use
  if Saa7146aGetNumberOfCards = 0 then
    Exit;
  // We MUST use a new file handle. If we don't only one call per handle is
  // allowed and when this happens to be the 'wait for notification'
  // we can not send any other calls (eg. 'generate manual notification'
  // which migth be needed in a deadlock situation.
  LCardHandle := Saa7146aCreateFile(-1);
  ThreadCounts := 0;
  OddCounts    := 0;
  EvenCounts   := 0;
  Buffer.Identifier := Fifo3.Identifier;
  Buffer.TransferAddress := @DvbData[0];
  Buffer.TargetIndex := 0;
  Buffer.TransferLength := Sizeof(DvbData);
  AutoEnd := 100;
  OutFile := TFileStream.Create('TransportStream.bin', fmCreate);
  try
    repeat
      // In this thread we wait for notifications and act on it
      Sleep(0);                                              // Essential!!
      Saa7146aWaitForNotification(LCardhandle);
      // We need to know which buffer we need to read (odd or even)
      if not Saa7146aReadFromSaa7146aRegister(LCardHandle,
               CSaa7146aPsr,
               Data) then
        Terminate;
      if Data and (1 shl CSaa7146aIrqFidB) = 0 then
      begin
        Inc(EvenCounts);
        Buffer.SourceIndex := 0;
      end
      else
      begin
        Inc(OddCounts);
        Buffer.SourceIndex := CDvbPacketVSync * CDvbPacketHSync;
      end;
      // Get data in local buffer
      if not Saa7146aReadFromDma(LCardHandle, Buffer) then
        Terminate;
      OutFile.Write(DvbData, SizeOf(DvbData));
      Inc(ThreadCounts);
      Dec(AutoEnd);
      if AutoEnd = 0 then Terminate;
    until Terminated;
  finally
    OutFile.Free;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Creation of thread.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TDataThread.Create;
begin
  // Create and run (do not suspend)
  inherited Create(False);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
begin
  Timing := 0;
  Counts := 0;
  FThread := nil;
  Fifo3.Size := 0;
  FCardHandle  := INVALID_HANDLE_VALUE;
  FCardNoError := False;
  FFrequencySearch := False;
  FFrequencySearchCountDown := 0;
  // Check if the driver is present and/or multiple cards are present
  case Saa7146aGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No SAA7146A driver and/or card detected.');
  end;
  // If any card detected then display additional information
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    FCardHandle := Saa7146aCreateFile(-1);
  end;
  btnResetClick(nil);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
var
  Channel: Dword;
begin
  if Assigned(FThread) then
  begin
    FThread.Terminate;                                     // Mark it for termination
    if FThread.Suspended then
      FThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    Saa7146aGenerateManualNotification(FCardHandle);
    FThread.WaitFor;
    FreeAndNil(FThread);
  end;
  // Software reset. We migth be capturing data, so stop ALL
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc1,                                   // Register to write
           CSaa7146aMc1SoftReset) then                     // Software reset
    Exit;
  // Release all allocated memory by the driver
  Channel := 0;
  while Saa7146aReleaseDma(FCardHandle, Channel) do
    Inc(Channel);

  if FCardHandle <> INVALID_HANDLE_VALUE then
  begin
    Saa7146aDisableLNB(FCardHandle, (rgTuner.ItemIndex = 0));
    Saa7146aCloseHandle(FCardHandle);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.rgPolarityClick(Sender: TObject);
begin
  case rgPolarity.ItemIndex of
    0: if not Saa7146aHorizontalPolarityLNB(FCardHandle, (rgTuner.ItemIndex = 0)) then
         mmoDriver.Lines.Add('Could not set horizontal polarity.');
    1: if not Saa7146aVerticalPolarityLNB(FCardHandle, (rgTuner.ItemIndex = 0)) then
         mmoDriver.Lines.Add('Could not set vertical polarity.');
  end;
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
var
  SymbolRate: Dword;
  Error:      Integer;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  if not Saa7146aQpskSetSymbolRate(FCardHandle, SymbolRate) then
    mskSymbolRate.Color := clRed
  else
    mskSymbolrate.Color := clLime;
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Percentage: Byte;
  Deviation : Integer;
  Frequency : Dword;
  Error     : Integer;
  PowerPercentage: Double;
  PowerDecibel   : Double;
  LockState : Byte;
  Noise     : Double;
  StatusStr : string;
  Data      : Dword;
  IrqHandling: TSaa7146aIrqTransferBuffer;
  Calc      : Extended;
begin
FFrequencySearch := False;
  if Saa7146aQpskGetLockDetectorPercentage(FCardHandle, Percentage) then
    ggLock.Progress := Percentage;
  if Saa7146aQpskGetCarrierDeviation(FCardHandle, Deviation) then
    stDeviation.Caption := format('%d', [Deviation]);
  if Saa7146aQpskGetSymbolRateDeviation(FCardHandle, Deviation) then
    stDeviationSymbolRate.Caption := format('%d', [Deviation]);
  if FFrequencySearch then
  begin
    Val(mskFrequency.EditText, Frequency, Error);
    if Error = 0 then
    begin
      Frequency := Frequency + Dword(Deviation);
      mskFrequency.Text := format('%d', [Frequency]);
      mskFrequencyChange(nil);
    end;
    if FFrequencySearchCountDown <> 0 then
      Dec(FFrequencySearchCountDown);
    if FFrequencySearchCountDown = 0 then
      FFrequencySearch := False;
  end;

  if Saa7146aQpskGetSignalPower(FCardHandle, PowerPercentage, PowerDecibel) then
  begin
    statictext5.caption := format('Power %f%% (%f dB)', [PowerPercentage, PowerDecibel]);
  end;
  if Saa7146aReadFromQpsk(FCardHandle, CStv0299bVStatus, LockState) then
  begin
    if (LockState and $80) <> 0 then
      StatusStr := 'CARRIER'
    else
      StatusStr := 'NO CARRIER';
    if (LockState and $10) <> 0 then
      StatusStr := StatusStr + ' - VITERBI'
    else
      StatusStr := StatusStr + ' - NO VITERBI';
    if (LockState and $08) <> 0 then
      StatusStr := StatusStr + ' - SYNC'
    else
      StatusStr := StatusStr + ' - NO SYNC';
    if (LockState and $98) = $98 then
      StatusStr := StatusStr + ' - LOCK'
    else
      StatusStr := StatusStr + ' - NO LOCK';
    statictext6.caption := format('Status $%2.2x : %s', [LockState, StatusStr]);
  end;
  if Saa7146aQpskGetNoiseIndicator(FCardHandle, Noise) then
  begin
    statictext8.caption := format('Noise %f dB', [Noise]);
  end;



  if Saa7146aReadFromSaa7146aRegister(FCardHandle,
       CSaa7146aPciVdp3,
       Data) then
    statictext9.caption := format('FIFO3: $%8.8x%', [Data]);

  if Saa7146aReadFromSaa7146aRegister(FCardHandle,
       CSaa7146aMc1,
       Data) then
  begin
    if Data and CSaa7146aMc1TrVideo3Mask <> 0 then
      statictext11.caption := 'ENABLED'
    else
      statictext11.caption := 'DISABLED';

  end;
  if Saa7146aReadFromSaa7146aRegister(FCardHandle,
       CSaa7146aPsr,
       Data) then
      statictext12.caption := format('PSR1: $%8.8x', [Data]);
  if Saa7146aReadFromSaa7146aRegister(FCardHandle,
       CSaa7146aSsr,
       Data) then
      statictext14.caption := format('PSR2: $%8.8x', [Data]);

  if Saa7146aReadFromSaa7146aRegister(FCardHandle,
     CSaa7146aIsr,
       Data) then
      statictext17.caption := format('ISR: $%8.8x', [Data]);

  IrqHandling.Identifier := CSaa7146aIrqFidB;
  if Saa7146aReadFromIrqHandling(FCardHandle, IrqHandling) then
    statictext18.caption := format('Field IRQs: %d', [IrqHandling.Information.Irqs]);
  if GetTickCount - Timing > 1000 then
  begin
    Calc := IrqHandling.Information.Irqs;
    Calc := (Calc - Counts) * 1000;
    Calc := Calc / (GetTickCount - Timing);
    statictext19.caption := format('IRQS/s: %f', [Calc]);
    Timing := GetTickCount;
    Counts := IrqHandling.Information.Irqs;
  end;
  statictext15.caption := format('Threads counts: %d %d %d', [ThreadCounts, EvenCounts, OddCounts]);
end;

procedure TfrmMain.mskFrequencyChange(Sender: TObject);
var
  Frequency: Dword;
  Error:     Integer;
begin
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert frequency.');
    Exit;
  end;
  // Befi

  if not Saa7146aFrequencyLNB(FCardHandle, Frequency, 9750000, 10600000, rgTuner.ItemIndex, FSynthesizerAddress) then
    mskFrequency.Color := clRed
  else
    mskFrequency.Color := clLime;
  if FFrequencySearchCountDown = 0 then
  begin
//    Saa7146aWriteToQpsk(FCardHandle, CStv0299bBclc, LockState) then
    FFrequencySearchCountDown := 10;
  end;
  FFrequencySearch := True;
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;

procedure TfrmMain.btnBeforeLockClick(Sender: TObject);
var
  Data: Byte;
begin
  if not Saa7146aReadFromQpsk(FCardHandle, CStv0299bCfd, Data) then
    Exit;
  Data := Data or  $80;
  Saa7146aWriteToQpsk(FCardHandle, CStv0299bCfd, Data);
end;

procedure TfrmMain.btnAfterLockClick(Sender: TObject);
var
  Data: Byte;
begin
  if not Saa7146aReadFromQpsk(FCardHandle, CStv0299bCfd, Data) then
    Exit;
  Data := Data and $7F;
  Saa7146aWriteToQpsk(FCardHandle, CStv0299bCfd, Data);
end;

procedure TfrmMain.rgLNBClick(Sender: TObject);
begin
  case rgLNB.ItemIndex of
    0: if not Saa7146aEnableLNB(FCardHandle, (rgTuner.ItemIndex =0)) then
         mmoDriver.Lines.Add('Could not enable LNB.');
    1: if not Saa7146aDisableLNB(FCardHandle, (rgTuner.ItemIndex = 0)) then
         mmoDriver.Lines.Add('Could not disable LNB.');
  end;
end;

procedure TfrmMain.btnResetClick(Sender: TObject);
var
  ReturnedByte : Byte;
  Address      : Integer;
  Success      : Boolean;
begin
  FCardNoError := False;
  // The RESET of the tuner is connected to a GPIO line of the SAA7146A, so we have to configure this
  if not Saa7146aInitializeGpio(FCardHandle, (rgTuner.ItemIndex = 0)) then
  begin
    mmoDriver.Lines.Add('Could not initialize GPIO.');
    Exit;
  end;
  // Now issue a reset signal
  if rgTuner.ItemIndex = 0 then
  begin
    if not Saa7146aIssueReset(FCardHandle, 2) then
    begin
      mmoDriver.Lines.Add('Could not issue a reset.');
      Exit;
    end;
  end;
  if not Saa7146aInitializeI2c(FCardHandle) then
  begin
    mmoDriver.Lines.Add('Error initializing I2C');
    Exit;
  end;

  FSynthesizerAddress := Saa7146aDetectSynthesizerAddress(FCardHandle);

  if not Saa7146aReadFromQpsk(FCardHandle, CStv0299bId, ReturnedByte) then
  begin
    mmoDriver.Lines.Add('Could not read identification of QPSK demodulator.');
    Exit;
  end;
  if not ReturnedByte = $A1 then
  begin
    mmoDriver.Lines.Add(format('QPSK demodulator identification NOT correct ($%x).', [ReturnedByte]));
    Exit;
  end;
  if not Saa7146aWriteDefaultsToQpsk(FCardHandle, rgTuner.ItemIndex) then
  begin
    mmoDriver.Lines.Add('QPSK registers could not be set to default values.');
    Exit;
  end;
  FCardNoError := True;
  rgPolarityClick(nil);
  rgLNBClick(nil);
  rgInversionClick(nil);
  mskFrequencyChange(nil);
  mskSymbolrateChange(nil);
  if FCardNoError then
    tmrUpdate.Enabled := True;
end;

procedure TfrmMain.rgInversionClick(Sender: TObject);
begin
  case rgInversion.ItemIndex of
    0: if not Saa7146aQpskSetInversion(FCardHandle, True) then
         mmoDriver.Lines.Add('Could not set inversion.');
    1: if not Saa7146aQpskSetInversion(FCardHandle, False) then
         mmoDriver.Lines.Add('Could not reset inversion.');
  end;
end;

procedure TfrmMain.btnSetupDd1Click(Sender: TObject);
var
  Channel    : Dword;
  Status     : TSaa7146aDmaBuffer;
  IrqHandling: TSaa7146aIrqTransferBuffer;
  Ie         : Dword;
begin
  // If the thread handling data gathering is active: close it
  if Assigned(FThread) then                                // If already running
  begin
    FThread.Terminate;                                     // Mark it for termination
    if FThread.Suspended then
      FThread.Resume;
    // If we are waiting for a notification then generate (end) it manually.
    Saa7146aGenerateManualNotification(FCardHandle);
    FThread.WaitFor;
    FreeAndNil(FThread);
  end;

  // Set initial settings of dual D1 interface
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aDd1Init,                               // Register to write
           CSaa7146aDd1LlcInput or                         // LLC input
           CSaa7146aDd1SioInput or                         // HS/VS inputs
           CSaa7146aDd1PvoDirect or                        // Direct polarity VS
           CSaa7146aDd1PhoDirect or                        // Direct polarity HS
           CSaa7146aDd1Sync6 or                            // Sync edge selection, HS rising, VS both frame, Field direct
           CSaa7146aDd1FidesEdgesIrq) then                 // Field both edges interrupt
    Exit;
  // Above registers need an upload to activate the setting
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc2,                                   // Register to write
           CSaa7146aMc2UpldDd1Initial) then                // Upload initial settings
    Exit;
  // Set video stream handling of port
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aDd1Data,                               // Register to write
           CSaa7146aDd1VidInput or                         // Port and PXQ inputs
           CSaa7146aDd1Y8cNoSavEav or                      // No SAV/EAV data
           CSaa7146aDd1PfidDirect) then                    // Direct polarity change
    Exit;
  // Above registers need an upload to activate the setting
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc2,                                   // Register to write
           CSaa7146aMc2UpldDd1A) then                      // Upload port A settings
    Exit;
  // Above registers need an upload to activate the setting
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc2,                                   // Register to write
           CSaa7146aMc2UpldDd1B) then                      // Upload port B settings
    Exit;
  // Set BRS control (we use the data from the D1 ports directly (unprocessed) so we need
  // to use the BRS and consequently FIFO3/DMA3
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aBrsCtrl,                               // Register to write
           CSaa7146aDd1BrsStreamInB or                     // Video data stream from port B
           CSaa7146aDd1BrsSyncInB or                       // Sync signals from port B
           (0 shl CSaa7146aDd1BrsVOffsetShift) or          // Vertical offset
           CSaa7146aDd1BrsVScale1 or                       // Vertical scale 1:1
           (0 shl CSaa7146aDd1BrsHOffsetShift) or          // Horizontal offset
           CSaa7146aDd1BrsHScale1 or                       // Horizontal scale 1:1
           CSaa7146aDd1BrsFormatYuv) then                  // YUV 4:2:2 format
    Exit;
  // Above registers need an upload to activate the setting
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc2,                                   // Register to write
           CSaa7146aMc2UpldBrs) then                       // Upload BRS registers
    Exit;

  // We have setup the basic stuff.
  // When the DD1 is activated (at this point it is NOT), data will be
  // acquired and stored in FIFO3. At some point the data has to be trasnferred
  // from this FIFO to the computers memory. This is done with DMA3.
  // If we activate the DD1 port now, we should 'see' FIFO3 filling and
  // generating (error) signals when it has been filled (without it being
  // able to transfer it any further).

  // Enable the DD1 hardware.
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc1,                                   // Register to write
           CSaa7146aMc1Dd1Enable) then                     // Enable real time video ports (DD1)
    Exit;

  // Before we can activate FIFO3 transfer we have to set the DMA settings.
  // One of those settings is the target memory data will be saved to. For
  // this we use the DMA memory feature of the driver, since the memory it
  // allocates is always valid. Since the driver has allocated continuous
  // physical memory we don't have to use the MMU.

  // First release all already allocated memory
  Channel := 0;
  while Saa7146aReleaseDma(FCardHandle, Channel) do
    Inc(Channel);
  // Disable all interrupts
  Saa7146aWriteToSaa7146aRegister(FCardHandle, CSaa7146aIer, 0);

  // Allocate a DMA buffer which will hold the FIFO 3 data
  // No matter how we will be doing thing, we can expect a maximum of
  // 2 * 512 (VS) * 188 (HS) = 192512 == 96256 words == 48128 dwords
  // The VS changes every 512 packets. The HS is activated for every packet.
  // Every packet is 188 bytes. The 2 is for 'even'/'odd' fields detection
  if not (Saa7146aAllocateDma(FCardHandle, DMASIZE * 50, Status)) then
    Exit;
  // Save info of allocated memory
  Fifo3.Identifier      := Status.Identifier;
  Fifo3.VirtualAddress  := Status.VirtualAddress;
  Fifo3.PhysicalAddress := Status.PhysicalAddress;
  Fifo3.Size            := Status.Size;

  // Now we have allocated the memory we can setup the DMA3 for it.
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aBaseOdd3,                              // Register to write
           DWord(Fifo3.PhysicalAddress.LowPart)) then     // Odd lines to here
    Exit;
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aBaseEven3,                              // Register to write
           DWord(Fifo3.PhysicalAddress.LowPart + (CDvbPacketVSync * CDvbPacketHSync))) then     // Even lines to here
    Exit;

  // We must make sure we don't get a protection error
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aProtAddr3,                             // Register to write
           DWord(Fifo3.PhysicalAddress.LowPart + DMASIZE * 50)) then
    Exit;

  // Setup distance between start addresses for consecutive lines of a field
  // If '0' data stays in first page (4095) bytes of selected memory
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aPitch3,                             // Register to write
           CDvbPacketHSync) then
    Exit;

  // Setup field/lines/bytes
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aNumLines3,                             // Register to write
           (CDvbPacketVSync shl 16) or CDvbPacketHSync) then
    Exit;

  // For more effective transfers we can extend the burst/threshold at which data is
  // transferred.
  if not Saa7146aWriteWithMaskToSaa7146aRegister(FCardHandle,
           CSaa7146aPciBtV,
           $001F0000,
           $00170000) then                                 // Burst 32 dwords and 32 dwords threshold
    Exit;

  // Above registers need an upload to activate the setting
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc2,                                   // Register to write
           CSaa7146aMc2UpldDma3) then                      // Upload DMA3 settings
    Exit;

  // Enable FIFO3 transfers.
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc1,                                   // Register to write
           CSaa7146aMc1TrVideo3Enable) then                // Enable transfers video 3
    Exit;


  // Start the thread which receives notifications we setup next
  FThread := TDataThread.Create;

  // For recording purposes or additional buffering we setup an interrupt
  // Setup interrupt behaviour driver. For now just record interrupts, nothing more.
  with IrqHandling.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := True;
    UseSignaling         := False;
    UseFifo              := False;
    IrqAutoDisable       := False;
  end;
  IrqHandling.Identifier := CSaa7146aIrqFidB;
  Saa7146aWriteToIrqHandling(FCardHandle, IrqHandling);

  if not Saa7146aReadFromSaa7146aRegister(FCardHandle, CSaa7146aIer, Ie) then
    Exit;
  Ie := Ie or (1 shl CSaa7146aIrqFidB);
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle, CSaa7146aIer, Ie) then
    Exit;

  if not Saa7146aWriteToSaa7146aRegister(FCardHandle, CSaa7146aIsr, $FFFFFFFF) then
    Exit;

end;


procedure TfrmMain.btnResetSAA7146Click(Sender: TObject);
begin
  // Software reset
  if not Saa7146aWriteToSaa7146aRegister(FCardHandle,
           CSaa7146aMc1,                                   // Register to write
           CSaa7146aMc1SoftReset) then                     // Software reset
    Exit;
end;

procedure TfrmMain.rgTunerClick(Sender: TObject);
begin
  btnResetClick(nil);
end;

end.
