{******************************************************************************}
{ FileName............: Saa7146aI2c                                            }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.08                                                   }
{------------------------------------------------------------------------------}
{  I2C support                                                                 }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030619 - Initial release                                           }
{  1.01   20040214 - <Saa7146aFrequencyLNB> auto 'invert'                      }
{  1.02   20040301 - <Saa7146aFrequencyLNB> possible integer overflow resolved }
{  1.03   20040418 - Added tuner address so tuners using a different address   }
{                    can also be controlled                                    }
{                  - Charge pump for SU1278 used instead of selectable,        }
{                    because SU1278 might lock if incorrectly set              }
{                    Note: If SU1278 'locked' then only power down will revert }
{                          from this situation (a reset will not do)           }
{                  - Procedures controlling OP0/OP1 of tuner (on/off/polarity) }
{                    have additional parameter indicating which OP to control  }
{  1.04   20040523 - Fixed ratio of 125000 kHz                                 }
{  1.05   20040616 - <Saa7146aDiSEqCCommand2> added which has no repeat and    }
{                    needs correct framing byte                                }
{  1.06   20040804 - Symbolrate SU1278 as per datasheet adjusted               }
{                  - Most optimum ratio re-introduced so we get the optimum    }
{                    phase noise performance                                   }
{  1.07   20040808 - Corrected bug in DiSEqC functions which did not restore   }
{                    the 22 kHz tone correctly                                 }
{                  - Added separate function for band detection so it can also }
{                    be used by other functions                                }
{  1.08   20040905 - Added BSBE1 tuner                                         }
{                  - Added inut level indication                               }
{  1.09   20041006 - Fixed possible range check error                          }
{                    (Saa7146aQpskGetInputLevel)                               }
{  1.10   20041112 - Added VDR type of DiSEqC commands including support       }
{                    routines for it                                           }
{  1.11   20041207 - Bug corrected (for Loop := 0 to CommandLength-1 do)       }
{                    instead of (forr Loop := 0 to CommandLength do) which     }
{                    always would send one byte too many.                      }
{                  - Fixed delay of 16 ms optionally in <Saa7146aSendDiSEqCMsg>}
{******************************************************************************}
unit Saa7146aI2c;

interface
uses
  Saa7146aInterface,
  Saa7146aRegisters,
  Stv0299bRegisters,
  Tsa5059Registers,
  Windows;

const
  CDiSEqCBurstA  = $01;
  CDiSEqCBurstB  = $02;
  CDiSEqCCommand = $04;
  CTunerBSRU6         = $00;
  CTunerSU1278        = $01;
  CTunerBSBE1         = $02;
  CCardTTPClineBudget = $00;

type
  TDiSEqCData = array[0..5] of Byte;

  function ExtractSpacedPart              (var Extracted: ShortString): ShortString;
  function ExtractFromDiSEqCSetting       (Setting      : ShortString;
                                           var Satellite: ShortString;
                                           var SLOF     : Integer;
                                           var Polarity : Char;
                                           var LOF      : Integer;
                                           var DiSEqC   : ShortString): Boolean;
  function CreateDiSEqCSetting            (Satellite    : ShortString;
                                           SLOF         : Integer;
                                           Polarity     : Char;
                                           LOF          : Integer;
                                           DiSEqC       : ShortString): ShortString;

  function Saa7146aUsDelay                          (UsDelay: Dword): Boolean;
  function Saa7146aGetLastI2cError                  : string;
  function Saa7146aInitializeI2c                    (Handle: THandle): Boolean;
  function Saa7146aReadFromQpsk                     (Handle: THandle; AccessRegister: Byte; var Data: Byte): Boolean;
  function Saa7146aWriteToQpsk                      (Handle: THandle; AccessRegister: Byte; Data: Byte): Boolean;
  function Saa7146aWriteDefaultsToQpsk              (Handle: THandle; TunerType: Byte): Boolean;
  function Saa7146aCheckQpskResetValues             (Handle: THandle): Boolean;
  function Saa7146aCheckQpskDefaultValues           (Handle: THandle; TunerType: Byte): Boolean;

  function Saa7146aQpskSetSymbolRate                (Handle: THandle; SymbolRateKS: Dword): Boolean;
  function Saa7146aQpskSetAGCInversion              (Handle: THandle; Inversion: Boolean): Boolean;
  function Saa7146aQpskSetInversion                 (Handle: THandle; Inversion: Boolean): Boolean;
  function Saa7146aQpskGetLockDetectorPercentage    (Handle: THandle; var Percentage: Byte): Boolean;
  function Saa7146aQpskGetCarrierDeviation          (Handle: THandle; var DeviationKHz: Integer): Boolean;
  function Saa7146aQpskGetSymbolRateDeviation       (Handle: THandle; var DeviationKHz: Integer): Boolean;
  function Saa7146aQpskGetInputLevel(Handle: THandle; var InputLevelDbm: Integer; var InputLevelPercentage: Byte): Boolean;
  function Saa7146aQpskGetSignalPower(Handle: THandle;
    var StrengthPercentage: Double;
    var StrengthDecibel   : Double): Boolean;
  function Saa7146aQpskGetNoiseIndicator            (Handle: THandle; var Noise: Double): Boolean;

  // LNB control
  function Saa7146aEnableLNB              (Handle: THandle; Op1: Boolean): Boolean;
  function Saa7146aDisableLNB             (Handle: THandle; Op1: Boolean): Boolean;
  function Saa7146aLowBandLNB             (Handle: THandle): Boolean;
  function Saa7146aHighBandLNB            (Handle: THandle): Boolean;
  function Saa7146aVerticalPolarityLNB    (Handle: THandle; Op0: Boolean): Boolean;
  function Saa7146aHorizontalPolarityLNB  (Handle: THandle; Op0: Boolean): Boolean;
  function Saa7146aDiSEqCCommand          (Handle: THandle; Repeats: Byte; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
  function Saa7146aDiSEqCCommand2         (Handle: THandle; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
  function Saa7146aDiSEqCCommandVdr       (Handle: THandle; CommandData: ShortString): Boolean;

  function Saa7146aWriteToSynthesizer     (Handle: THandle; AddressIndex: Byte; Data: Dword): Boolean;
  function Saa7146aReadFromSynthesizer    (Handle: THandle; AddressIndex: Byte; var Data: Byte): Boolean;
  function Saa7146aDetectSynthesizerAddress(Handle: THandle): Byte;
  function Saa7146aFrequencyLNB(Handle: THandle; FrequencyKHz: Dword;
    LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword; TunerType: Byte; AddressIndex: Byte): Boolean;

  function Saa7146aFrequencyTunerLNB(Handle: THandle; FrequencyKHz: Dword; TunerType: Byte; AddressIndex: Byte): Boolean;

  function Saa7146aUseLowBand             (Handle: THandle; FrequencyKHz: Dword;
    LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword; TunerType: Byte): Boolean;

implementation
uses
  Math,
  SyncObjs,
  SysUtils;

const
  CI2cClock = CSaa7146aI2cClkSel275kHz;

var
  LastI2cError            : Dword;                         // Last I2c error (debugging, I2C status register)
  HighPerformanceAvailable: Boolean;                       // Indicates available high performance counter
  HighPerformanceFrequency: TLargeInteger;                 // Frequency of high performance counter
  I2cLock                 : TCriticalSection;              // Prevents multiple I2C operations which interfere
  LastSymbolRate          : Dword;                         // Last symbolrate kS (eg. 'disables' true multiple card support)
  LastSynthesizerData     : Dword;                         // Last written synthesizer data (eg. 'disables' true multiple card support)
  LastSynthesizerIndex    : Byte;                          // Last address index synthesizer (eg. 'disables' true multiple card support)
  LastTunerType           : Byte;                          // Last tuner type (eg. 'disables' true multiple card support)
  LastEnableOp            : Boolean;                       // Last used OP1 setting enable/disable LNB
  LastPolarityOp          : Boolean;                       // Last used OP0 setting polarity

  
{------------------------------------------------------------------------------
  Params  : <UsDelay> Delay in us
  Returns : <Result>  True is success
                      False if no high performance counter (should not be possible)

  Descript: Delay for a defined time
  Notes   : If no high performace counter is available then the delay is
            still implemented but with a worse resolution.
            During the delay no processes are handled...
------------------------------------------------------------------------------}
function Saa7146aUsDelay(UsDelay: Dword): Boolean;
var
  StartTiming: TLargeInteger;
  Timing     : TLargeInteger;
  DiffTiming : TLargeInteger;
  DeltaTiming: TLargeInteger;
  StartTime  : Dword;
  MsDelay    : DWord;
begin
  if HighPerformanceAvailable then
  begin
    Result := True;
    QueryPerformanceCounter(StartTiming);
    DeltaTiming := UsDelay * HighPerformanceFrequency;
    DeltaTiming := DeltaTiming div 1000000;
    repeat
      QueryPerformanceCounter(Timing);
      if Timing > StartTiming then
        DiffTiming := Timing - StartTiming
      else
        DiffTiming := StartTiming - Timing;
    until DiffTiming >= DeltaTiming;
  end
  else
  begin
    Result := False;
    MsDelay   := UsDelay div 1000;
    StartTime := GetTickCount;
    // Now wait for the delay
    repeat
      repeat
      until StartTime <> GetTickCount;
    until Dword(Abs(GetTickCount - StartTime)) > MsDelay;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Explanatory string

  Descript: Get last I2C error (as string)
  Notes   : For debugging only. The error is cleared afterwards.
------------------------------------------------------------------------------}
function Saa7146aGetLastI2CError: string;
begin
  if (LastI2cError and CSaa7146aI2cInvalidStartStop) <> 0 then
    Result := 'I2C error: Bus error due to invalid start/stop condition';
  if (LastI2cError and CSaa7146aI2cNackAddressPhase) <> 0 then
    Result := 'I2C NACK: Error in address phase';
  if (LastI2cError and CSaa7146aI2cNackTransmission) <> 0 then
    Result := 'I2C NACK: Error in data transmission';
  if (LastI2cError and CSaa7146aI2cNackReception) <> 0 then
    Result := 'I2C NACK: Error when receiving data';
  if (LastI2cError and CSaa7146aI2cArbitrationLost) <> 0 then
    Result := 'I2C error: Arbitration lost';
  if LastI2cError = $FFFF then
    Result := 'I2C error: Timeout';
  if (LastI2cError and CSaa7146aI2cErr) = 0 then
    Result := 'No I2C error';
  LastI2cError := $0000;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Initialize I2C interface.
  Notes   : Send an abort command and sets the clock used for I2C transfers
------------------------------------------------------------------------------}
function Saa7146aInitializeI2c(Handle: THandle): Boolean;
var
  Loop     : Word;
  Mc2      : Dword;
  StartTime: Dword;
begin
  Result := False;                                         // Assume we fail

  // Note that the I2C registers <CSaa7146aI2cSta> and <CSaa7146aI2cTrf> need
  // to be 'uploaded'
  // First setup the abort command (which clears the busy flag) and while we
  // at at set the communication speed
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cSta, CSaa7146aI2cAbort) then
    Exit;
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2   , CSaa7146aMc2UpldI2c) then
    Exit;
  // The data is being uploaded and we need to wait for it to be uploaded
  // To prevent an endless wait we have a simple check on an excessive timeout
  StartTime := GetTickCount;
  repeat
    if GetTickCount <> StartTime then
    begin
      if Abs(GetTickCount - StartTime) > 1 then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
      Exit;
  until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);

  // Per SAA7146 data sheet, write to STATUS register twice to reset all I2C error flags
  for Loop := 0 to 1 do
  begin
    // Reset errors, set clock and upload it
    if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cSta, CI2cClock) then
      Exit;
    if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2   , CSaa7146aMc2UpldI2c) then
      Exit;
    // The data is being uploaded and we need to wait for it to be uploaded
    // To prevent an endless wait we have a simple check on an excessive timeout
    StartTime := GetTickCount;
    repeat
      if GetTickCount <> StartTime then
      begin
        if Abs(GetTickCount - StartTime) > 1 then
          Exit;
      end;
      if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
        Exit;
    until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Clear errors and BUSY flag of I2C interface (if any).
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aClearErrorsI2c(Handle: THandle): Boolean;
var
  I2cSta   : Dword;
begin
  Result := False;                                         // Assume we fail

  // Determine if error or busy
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cSta, I2cSta) then
    Exit;
  // If no errors/busy then no need to reset them
  if (I2cSta and (CSaa7146aI2cBusy or CSaa7146aI2cErr) = 0) then
  begin
    Result := True;
    Exit;
  end;
   Result := Saa7146aInitializeI2c(Handle);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Value>   Value to transfer
            <TimeOut> Allowed timeout in ms
  Returns : <Result>  True for success

  Descript: Write I2C transfer control register and wait for transfer ready.
  Notes   : -
------------------------------------------------------------------------------}
function Saa7146aI2cHandshake(Handle: THandle; Value: Dword; TimeOut: Dword): Boolean;
var
  Mc2      : Dword;
  StartTime: Dword;
begin
  Result := False;
  LastI2cError := $FFFF;                                   // Assume timeout error

  // Set up the value to write
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cTrf, Value) then
    Exit;
  // Then we need to upload it
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2, CSaa7146aMc2UpldI2c) then
    Exit;
  // And wait for the upload to finish
  StartTime := GetTickCount;
  repeat
    if GetTickCount <> StartTime then
    begin
      if Abs(GetTickCount - StartTime) > 1 then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
      Exit;
  until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);

  // Now wait until I2C bus transfer is finished or an error occurs
  StartTime := GetTickCount;
  repeat
    LastI2cError := $FFFF;                                 // Assume timeout error
    if GetTickCount <> StartTime then
    begin
      if Dword(Abs(GetTickCount - StartTime)) > TimeOut then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cSta, LastI2cError) then
      Exit;
    if (LastI2cError and CSaa7146aI2cErr) <> 0 then
      Exit;
  until ((LastI2cError and CSaa7146aI2cBusy) = 0);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessRegister>  Register to read from
  Returns : <Result>          True for success
            <Data>            Data read from register

  Descript: Read data from register of QPSK demodulator.
  Notes   : Reading from the demodulator requires a dummy write (without data)
            to the register we want to read from. Then a read cycle will read
            the targetted data from the register.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}
function Saa7146aReadFromQpsk(Handle: THandle; AccessRegister: Byte; var Data: Byte): Boolean;
var
  I2cTrf     : Dword;
  Retry      : Byte;
  RetryWhole : Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // For a read we first have to write the register we like to read afterwards
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart             shl 6)  or      // First byte send
          (CStv0299bDemodulatorWriteAddress  shl 24) or      // START - device address WRITE
          (CSaa7146aI2cAttrStop              shl 4)  or      // Second byte send
          (AccessRegister                    shl 16) or      // register - STOP
          (CSaa7146aI2cAttrNop               shl 2)  or      // Third byte not sent
          (0                                 shl 8),
          20);                                               // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if NeedNoRetry then
      begin
        // Execute the read
        Retry := 1;
        repeat
          NeedNoRetry := Saa7146aI2cHandshake(Handle,
            (CSaa7146aI2cAttrStart             shl 6)  or    // First byte send
            (CStv0299bDemodulatorReadAddress   shl 24) or    // START - device address READ
            (CSaa7146aI2cAttrStop              shl 4)  or    // Second byte send
            (0                                 shl 16) or    // dummy (filled in by targetted device) - STOP
            (CSaa7146aI2cAttrNop               shl 2)  or    // Third byte not sent
            (0                                 shl 8),
            20);                                             // Timeout essential otherwise sometimes error
          if not NeedNoRetry then
            Dec(Retry);
        until NeedNoRetry or (Retry = 0);
      end;
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;
    // Return copy of read value
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cTrf, I2cTrf) then
      Exit;
    Data := (I2cTrf shr 16) and $FF;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessRegister>  Register to write to
            <Data>            Data to write to register
  Returns : <Result>          True for success

  Descript: Write data to register of QPSK demodulator
  Notes   : We always write a single register only, although it is possible
            to write consequtive registers.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}
function Saa7146aWriteToQpsk(Handle: THandle; AccessRegister: Byte; Data: Byte): Boolean;
var
  RetryWhole : Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      NeedNoRetry := Saa7146aI2cHandshake(Handle,
        (CSaa7146aI2cAttrStart             shl 6)  or      // First byte is
        (CStv0299bDemodulatorWriteAddress  shl 24) or      //   START - device address WRITE
        (CSaa7146aI2cAttrCont              shl 4)  or      // Second byte is
        (AccessRegister                    shl 16) or      //   register
        (CSaa7146aI2cAttrStop              shl 2)  or      // Third byte is
        (Data                              shl 8),         //   data - STOP
        20);                                               // Timeout essential otherwise sometimes error
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <TunerType>       Type of tuner
  Returns : <Result>          True if success

  Descript: Write all default values of the QSPK demodulator
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aWriteDefaultsToQpsk(Handle: THandle; TunerType: Byte): Boolean;
var
  Loop: Byte;
begin
  Result := False;
  LastTunerType := TunerType;
  case TunerType of
    CTunerBSRU6:
      begin
        for Loop := Low(CStv0299bDefaultsBSRU6) to High(CStv0299bDefaultsBSRU6) do
        begin
          // Write value to register
          if CStv0299bDefaultsBSRU6[Loop, 0] <> $FF then
            if not Saa7146aWriteToQpsk(Handle, CStv0299bDefaultsBSRU6[Loop, 0], CStv0299bDefaultsBSRU6[Loop, 1]) then
              Exit;
        end;
      end;
    CTunerBSBE1:
      begin
        for Loop := Low(CStv0299bDefaultsBSBE1) to High(CStv0299bDefaultsBSBE1) do
        begin
          // Write value to register
          if CStv0299bDefaultsBSBE1[Loop, 0] <> $FF then
            if not Saa7146aWriteToQpsk(Handle, CStv0299bDefaultsBSBE1[Loop, 0], CStv0299bDefaultsBSBE1[Loop, 1]) then
              Exit;
        end;
      end;
    CTunerSU1278:
      begin
        for Loop := Low(CStv0299bDefaultsSU1278) to High(CStv0299bDefaultsSU1278) do
        begin
          // Write value to register
          if CStv0299bDefaultsSU1278[Loop, 0] <> $FF then
            if not Saa7146aWriteToQpsk(Handle, CStv0299bDefaultsSU1278[Loop, 0], CStv0299bDefaultsSU1278[Loop, 1]) then
              Exit;
        end;
      end;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <TunerType>       Type of tuner
  Returns : <Result>          True if values comply with default values

  Descript: Check values of the QSPK demodulator with there default values
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aCheckQpskDefaultValues(Handle: THandle; TunerType: Byte): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;
  LastTunerType := TunerType;
  case TunerType of
    CTunerBSRU6:
      begin
        for Loop := Low(CStv0299bDefaultsBSRU6) to High(CStv0299bDefaultsBSRU6) do
        begin
          if CStv0299bDefaultsBSRU6[Loop, 0] <> $FF then
          begin
            // Read value from register
            if not Saa7146aReadFromQpsk(Handle, CStv0299bDefaultsBSRU6[Loop, 0], Data) then
              Exit;
            // Check it with expected value
            if Data <> CStv0299bDefaultsBSRU6[Loop, 1] then
              Exit;
          end;
        end;
      end;
    CTunerBSBE1:
      begin
        for Loop := Low(CStv0299bDefaultsBSBE1) to High(CStv0299bDefaultsBSBE1) do
        begin
          if CStv0299bDefaultsBSBE1[Loop, 0] <> $FF then
          begin
            // Read value from register
            if not Saa7146aReadFromQpsk(Handle, CStv0299bDefaultsBSBE1[Loop, 0], Data) then
              Exit;
            // Check it with expected value
            if Data <> CStv0299bDefaultsBSBE1[Loop, 1] then
              Exit;
          end;
        end;
      end;
    CTunerSU1278:
      begin
        for Loop := Low(CStv0299bDefaultsSU1278) to High(CStv0299bDefaultsSU1278) do
        begin
          if CStv0299bDefaultsSU1278[Loop, 0] <> $FF then
          begin
            // Read value from register
            if not Saa7146aReadFromQpsk(Handle, CStv0299bDefaultsSU1278[Loop, 0], Data) then
              Exit;
            // Check it with expected value
            if Data <> CStv0299bDefaultsSU1278[Loop, 1] then
              Exit;
          end;
        end;
      end;
    end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True if values comply with reset values

  Descript: Check values of the QSPK demodulator with there reset values
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aCheckQpskResetValues(Handle: THandle): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;

  for Loop := Low(CStv0299bResets) to High(CStv0299bResets) do
  begin
    // Read value from register
    if CStv0299bResets[Loop, 0] <> $FF then
    begin
      if not Saa7146aReadFromQpsk(Handle, CStv0299bResets[Loop, 0], Data) then
        Exit;
      // Check it with expected value
      if Data <> CStv0299bResets[Loop, 1] then
        Exit;
    end;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <SymbolRateKS>    Symbolrate to set (KSymbols/sec)
  Returns : <Result>          True for success

  Descript: Set symbol rate
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskSetSymbolRate(Handle: THandle; SymbolRateKS: Dword): Boolean;
var
  Data   : Dword;
  AClc   : Byte;
  BClc   : Byte;
  Agc1R  : Byte;
  NewData: Dword;
begin
  Result := False;

  LastSymbolRate := SymbolRateKS;

  // Check for excessive values
  if (SymbolRateKS * 1000) > CStv0299bMasterClock then
    Exit;

  // The symbolrate also requires additional settings
  AClc := $04;                                             // Alpha_car
  BClc := $11;                                             // Beta_car
  if SymbolRateKS < 30000 then
  begin
    AClc := $06;                                           // Alpha_car
    BClc := $13;                                           // Beta_car
  end;
  if SymbolRateKS < 14000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $13;                                           // Beta_car
  end;
  if SymbolRateKS < 7000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $0F;                                           // Beta_car
  end;
  if SymbolRateKS < 3000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $0B;                                           // Beta_car
  end;
  if SymbolRateKS < 1500 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $07;                                           // Beta_car
  end;
  AClc := AClc or $B0;                                     // Derotator on, 256 k symbols
  BClc := BClc or $40;                                     // QPSK algorithm 1

  if LastTunerType = CTunerSU1278 then
  begin
    // The SU1278 uses specific settings for different symbolrates (as per datasheet)
    AClc := $B5;
    BClc := $95;
    if SymbolRateKS < 15000 then
      BClc := $8F;
    if SymbolRateKS < 5000 then
      BClc := $89;
    if SymbolRateKS < 2000 then
      BClc := $86;
    // Special case for SU1278 tuner for low symbolrates
    // Check if we need to adjust the synthesizer
    // Reset the bit responsible for low symbolrates
    NewData := LastSynthesizerData and (not (CTsa5059SetPort0 shl 24));
    Agc1R := $94;
    if SymbolRateKS < 4000 then
    begin
      Agc1R   := $90;
      NewData := NewData or (CTsa5059SetPort0 shl 24);
    end;
    if (LastSynthesizerData <> NewData) then
    begin
      if not Saa7146aWriteToSynthesizer(Handle, LastSynthesizerIndex, NewData) then
        Exit;
      if not Saa7146aWriteToQpsk(Handle, CStv0299bAgc1R, Agc1R) then
        Exit;
    end;
  end;

  if not Saa7146aWriteToQpsk(Handle, CStv0299bAclc, AClc) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bBclc, BClc) then
    Exit;

  // Note: we need to write the MSB register before we write the Middle bit register
  Data := Round((SymbolRateKS * 1000) / CStv0299bSetSymbolrateUnits);
  // Write the values -> they are spread out over 5 nibbles (lowest byte has LSnibble in high part)
  Data := Data shl 4;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrH, (Data shr 16) and $FF) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrM, (Data shr 8) and $FF) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrL, Data and $F0) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <Percentage>      Lock percentage (0-100)

  Descript: Get lock percentage
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskGetLockDetectorPercentage(Handle: THandle; var Percentage: Byte): Boolean;
var
  Data: Byte;
  Temp: Integer;
begin
  Result := False;

  // Read lock value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCldi, Data) then
    Exit;
  // Since the lock value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  // And to percentage
  Temp := Temp + 128;
  Temp := (Temp * 100) div 255;
  Percentage := Temp;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>               <Saa7146aCreateFile> handle
  Returns : <Result>               True if valid
            <InputLevelDbm>        Dbm level
            <InputLevelPercentage> Level in percentage

  Descript: Get signal level.
  Notes   : Approximation only. Could correct for different tuners but this
            is not done. Eg. BSRU6 tuner is more or less linearly running
            from -70 dBm (-100 AGC1I) to -20 dBm (+50 AGC1I). A BSBE1
            is not so linear and runs from -70 dBm (-100 AGC1I) to
            -20 dBm (+65 AGC1I).
------------------------------------------------------------------------------}
function Saa7146aQpskGetInputLevel(Handle: THandle; var InputLevelDbm: Integer; var InputLevelPercentage: Byte): Boolean;
var
  Data   : Byte;
  NewData: Integer;
  Calc   : Integer;
begin
  Result := False;

  // Read values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc1I, Data) then
    Exit;
  // Data is actually signed
  if Data > 128 then
    NewData := Data - 256
  else
    NewData := Data;
  // Convert to dBm range
  // -128..+128 -> -178..78
  // -100..50 to -70..-20 -> 150 range to 50 range
  // First -150..0
  NewData := NewData - 50;
  // -178..0
  if NewData > 0 then
    NewData := 0;
  Calc := 100 - Abs((NewData * 2) div 3);
  if Calc < 0 then
    Calc := 0;
  InputLevelPercentage := Calc;
  // Then factor 3: -50..0
  NewData := NewData div 3;
  // Then the -20
  NewData := NewData - 20;
  InputLevelDbm := NewData;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>             <Saa7146aCreateFile> handle
  Returns : <Result>             True for success
            <StrengthPercentage> Signal strength in %
            <StrengthDecibel>    Signal strength in dB

  Descript: Get signal power
  Notes   : We read the value of the signal AGC. A low AGC setting indicates
            a low amplification and therefore a high input signal. A small
            input signal has to be amplifier more than a high input signal,
            resulting in a higher gain (= higher AGC value).
            Typically an amplification is given in dB which is based on a
            logarithmic scale. 'dB' itself indicates the relation of the
            output with reference to it's input. 'dB' by itself has no 'fixed'
            reference, as does for example 'dBmV' has ('dBmV' indicate that
            0 dB equals 1 mV).
            We have to determine our own reference, meaning what the input
            signal (range) is at the minimum AGC and at the maximum AGC.
            It turns out this is about 4. This means that when the AGC goes
            from it's minimum setting to it's maximum, it amplifies the
            input signal by 4.
            In dB this factor 4 is equivalent to 12 dB (a factor 1 equals 0 dB).
            So, if the AGC is at it's minimum value it means that the signal is
            12 dB stronger than when the AGC is at it's maximum
            (smaller AGC == less amplification == stronger input signal).
------------------------------------------------------------------------------}
function Saa7146aQpskGetSignalPower(Handle: THandle;
  var StrengthPercentage: Double;
  var StrengthDecibel   : Double): Boolean;
var
  DataMSB    : Byte;
  DataLSB    : Byte;
  Data       : Integer;
begin
  Result := False;

  // Read values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc2I1, DataMSB) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc2I2, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;

  // The lower the AGC, the stronger the signal, so '100% - AGC%'
  StrengthPercentage := 100 - ((Data * 100) / $FFFF);
  if StrengthPercentage <> 0 then
    StrengthDecibel := (Log10(4) * 20) + Log10(StrengthPercentage/100) * 20
  else
    StrengthDecibel := 0;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <Noise>           Noise indicator

  Descript: Get noise indicator
  Notes   : The C/N is calculated with the following formula:
              C/N (dB) = -0.0017 * Nir + 19.02
            To check out:
            The sensitivity is dependent on the averaging period (SN) of the
            rate error detection and the pucture rate.
            It is also dependent on the AGC1 reference setting.
            Note that the result is only valid with a carrier
            and puncture rate found and known.
------------------------------------------------------------------------------}
function Saa7146aQpskGetNoiseIndicator(Handle: THandle; var Noise: Double): Boolean;
var
  DataMSB: Byte;
  DataLSB: Byte;
  Data   : Word;
begin
  Result := False;

  // Read values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bNirH, DataMSB) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bNirL, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;

  Noise := (-0.0017 * Data) + 19.02;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Inversion>       True when inversion requested
  Returns : <Result>          True for success

  Descript: Set inversion of AGC output
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskSetAGCInversion(Handle: THandle; Inversion: Boolean): Boolean;
var
  OriginalData: Byte;
  Data        : Byte;
begin
  Result := False;

  // Read original data
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc1R, OriginalData) then
    Exit;
  Data := OriginalData and not CStv0299bAGCInversion;
  if Inversion then
    Data := Data or CStv0299bAGCInversion;
  if Data <> OriginalData then
    if not Saa7146aWriteToQpsk(Handle, CStv0299bAgc1R, Data) then
      Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Inversion>       True when inversion requested
  Returns : <Result>          True for success

  Descript: Set inversion of I/Q inputs
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskSetInversion(Handle: THandle; Inversion: Boolean): Boolean;
var
  OriginalData: Byte;
  Data        : Byte;
begin
  Result := False;

  // Read original data
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, OriginalData) then
    Exit;
  Data := OriginalData and not CStv0299bInversion;
  if Inversion then
    Data := Data or CStv0299bInversion;
  if Data <> OriginalData then
    if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
      Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Frequency deviation (in kHz)

  Descript: Get deviation from carrier
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskGetCarrierDeviation(Handle: THandle; var DeviationKHz: Integer): Boolean;
var
  DataM: Byte;
  DataL: Byte;
  Temp: Int64;
begin
  Result := False;

  // Read derotator values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrM, DataM) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrL, DataL) then
    Exit;
  Temp := (DataM shl 8) or DataL;
  // Since its a signed number we convert it
  if (DataM and $80) <> 0 then
    Temp := Temp - 65536;
  Temp := Temp * (CStv0299bMasterClock div 1000);
  // And to percentage
  DeviationKHz := (Temp div 65536);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Symbol rate deviation in kHz

  Descript: Get deviation
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aQpskGetSymbolRateDeviation(Handle: THandle; var DeviationKHz: Integer): Boolean;
var
  Data: Byte;
  Temp: Integer;
  Calc: Extended;
begin
  Result := False;

  // Read timing frequency (deviation)
  if not Saa7146aReadFromQpsk(Handle, CStv0299bRtf, Data) then
    Exit;

  // Since the value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  Calc := Temp;
  Calc := Calc * CStv0299bGetSymbolrateUnits;
  Calc := Calc / 1000;
  // Additional factor so what we return can be used directly for the symbol rate
  Calc := Calc / 2;
  DeviationKHz := Round(Calc);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Op1>             True if OP1 to use (otherwise OP0 is used)
                              Nova/budget card: True
  Returns : <Result>          True for success

  Descript: Enable the LNB
  Notes   : The LNB (power) is controlled throug the OPx output of the QPSK
            demodulator.
            Typical time for the enable is 1.5 ms
------------------------------------------------------------------------------}
function Saa7146aEnableLNB(Handle: THandle; Op1: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  LastEnableOp := Op1;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  if Op1 then
  begin
    Data := Data and not CStv0299bOP1Mask;
    Data := Data or  CStv0299bOP1Value;
  end
  else
  begin
    Data := Data and not CStv0299bOP0Mask;
    Data := Data or  CStv0299bOP0Value;
  end;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Op1>             True if OP1 to use (otherwise OP1 is used)
                              Nova/budget card: True
  Returns : <Result>          True for success

  Descript: Disable the LNB
  Notes   : The LNB (power) is controlled throug the OPx output of the QPSK
            demodulator.
            Disabling is almost directly.
------------------------------------------------------------------------------}
function Saa7146aDisableLNB(Handle: THandle; Op1: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  LastEnableOp := Op1;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  if Op1 then
    Data := Data and not CStv0299bOP1Mask
  else
    Data := Data and not CStv0299bOP0Mask;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set low band of LNB
  Notes   : The LNB band is controlled by the absence of a 22kHz signal.
------------------------------------------------------------------------------}
function Saa7146aLowBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulation0;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set high band of LNB
  Notes   : The LNB band is controlled by the presence of a 22kHz signal.
------------------------------------------------------------------------------}
function Saa7146aHighBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulationContinuous;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Op0>             True if OP0 to use (otherwise OP1 is used)
                              Nova/budget card: True
  Returns : <Result>          True for success

  Descript: Set vertical polarity of LNB
  Notes   : The polarity is controlled by the 13V signal on the LNB, which
            is controlled by OP0.
------------------------------------------------------------------------------}
function Saa7146aVerticalPolarityLNB(Handle: THandle; Op0: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  LastPolarityOp := Op0;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  if Op0 then
    Data := Data and not CStv0299bOP0Mask
  else
    Data := Data and not CStv0299bOP1Mask;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Op0>             True if OP0 to use (otherwise OP1 is used)
                              Nova/budget card: True
  Returns : <Result>          True for success

  Descript: Set horizontal polarity of LNB
  Notes   : The polarity is controlled by the 18V signal on the LNB, which
            is controlled by OP0.
------------------------------------------------------------------------------}
function Saa7146aHorizontalPolarityLNB(Handle: THandle; Op0: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  LastPolarityOp := Op0;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  if Op0 then
  begin
    Data := Data and not CStv0299bOP0Mask;
    Data := Data or CStv0299bOP0Value;
  end
  else
  begin
    Data := Data and not CStv0299bOP1Mask;
    Data := Data or CStv0299bOP1Value;
  end;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Wait for empty Fifo
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aDiSEqCWaitIdle(Handle: THandle): Boolean;
var
  Data : Byte;
  Empty: Boolean;
  Retry: Integer;
begin
  Result := False;
  // Data is send as 9-bits; each bit is 33 periods of the 22 kHz signal
  // 33 * 9 == 300 periods == 14 ms per byte to transfer
  // A transfer is typically 54 ms long and will have a total length of 6 bytes
  // although 3 bytes is more typical
  Retry  := 100;
  repeat
    if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqCStatus, Data) then
      Exit;
    Empty := ((Data and CStv0299bFifoMask) = CStv0299bFifoEmpty);
    if not Empty then
    begin
      Saa7146aUsDelay(1000);
      Dec(Retry);
    end;
  until Empty or (Retry = 0);
  if Retry = 0 then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Bit>             Bit to transmit
  Returns : <Result>          True for success

  Descript: Wait for fifo not full (can be written to)
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aDiSEqCWaitFree(Handle: THandle): Boolean;
var
  Data : Byte;
  Full : Boolean;
  Retry: Integer;
begin
  Result := False;
  // Data is send as 9-bits; each bit is 33 periods of the 22 kHz signal
  // 33 * 9 == 300 periods == 14 ms per byte to transfer
  // A transfer is typically 54 ms long and will have a total length of 6 bytes
  // although 3 bytes is more typical
  Retry  := 20;
  repeat
    if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqCStatus, Data) then
      Exit;
    Full := ((Data and CStv0299bFifoFull) = CStv0299bFifoFull);
    if Full then
    begin
      Saa7146aUsDelay(1000);
      Dec(Retry);
    end;
  until (not Full) or (Retry = 0);
  if Retry = 0 then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <data>            Byte to transmit
  Returns : <Result>          True for success

  Descript: Send DiSEqC byte
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aDiSEqCSendByte(Handle: THandle; Data: Byte): Boolean;
begin
  Result := False;

  // Wait for space in fifo
  if not Saa7146aDiSEqCWaitFree(Handle) then
    Exit;
  // Write data to fifo
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqCFifo, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <CommandType>     Type of command
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              Note: Only one bit should be activated here!
            <CommandLength>   Number of commands
            <Commands>        Pointer to commands
            <Delay>           Use implicit delay
  Returns : <Result>          True for success

  Descript: Send single DiSEqC message or burst (no repeat feature,
            no reset of original signals). The data is send as is.
  Notes   : This function is typically called from <Saa7146aDiSEqCCommand>.
------------------------------------------------------------------------------}
function Saa7146aSendDiSEqCMsg(Handle: THandle; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData; Delay: Boolean): Boolean;
var
  Loop: Byte;
begin
  Result := False;

  // Check for double bits (not allowed since we do either a burst or DiSEqc command)
  // or 'idle' commands (nothing to do).
  // Double bits return an error.
  case (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB or CDiSEqCCommand)) of
    $00:  begin
            Result := True;
            Exit;
          end;
    CDiSEqCBurstA: ;
    CDiSEqCBurstB: ;
    CDiSEqCCommand: if CommandLength = 0 then
         begin
           Result := True;
           Exit;
         end;
    else Exit;
  end;

  // Wait for empty DiSEqc fifo
  if not Saa7146aDiSEqCWaitIdle(Handle) then
    Exit;
  // Setup for DiSEqC
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, CStv0299bModulationDiSEqC) then
    Exit;
  // After removing the continuous tone or change of voltage a delay of >15ms
  // is required
  if Delay then
    Saa7146aUsDelay(16000);

  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) then
  begin
    for Loop := 0 to CommandLength-1 do
    begin
      if not Saa7146aDiSEqCSendByte(Handle, CommandData[Loop]) then
        Exit;
    end;
    // Wait for empty fifo
    if not Saa7146aDiSEqCWaitIdle(Handle) then
      Exit;
  end;
  // Do burst
  if ((CommandType and (CDiSEqCBurstA or CDiSEqcBurstB)) <> 0) then
  begin
    if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, CStv0299bModulationUnmodulated) then
      Exit;
    if (CommandType and CDiSEqCBurstA) <> 0 then
      Saa7146aDiseqcSendByte(Handle, $00)
    else
      Saa7146aDiseqcSendByte(Handle, $FF);
    // Wait for empty fifo
    if not Saa7146aDiSEqCWaitIdle(Handle) then
      Exit;
  end;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Repeats>         Repeats for DiSEqC command
                              Typically the same as the number of cascaded DiSEqc
                              devices
            <CommandType>     Type of command (bit-wise selection)
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              It is allowed to enable both burst and
                              DiSEqc, although only one type of burst can be selected
                              (A has priority)
            <CommandLength>   Number of <CommandData> bytes
            <CommandData>     Data bytes to transmit through DiSEqC
                              First byte is not used but instead replaced by
                              locally generated framing byte.
  Returns : <Result>          Error number

  Descript: Send DiSEqc command and/or simple burst mechanism (or both)
            Can send it repeatedly. The data is expanded with the
            framing byte.
  Notes   : This call is used to send a sequence of DiSEqC and/or V-SEC
            commands.
            The 'burst' is used for simple switchers and can be used together
            with a 'normal' DiSEqC command.
            DiSEqC basically consists of the following:
            . framing byte
              $E0 == Command from master, no reply required, first transmission
              $E1 == Command from master, no reply required, repeated transmission
              $E2..$E7 require a reply from a slave which is not supported
            . address byte
              $X_ == family
              $_X == sub type
              $00 == Any device
              $10 == Any LNB, switcher or SMATV
              $11 == LNB
              $12 == LNB with loop through switching
              $14 == Switcher (DC blocking)
              $15 == Switcher with DC loop through
              $18 == SMATV
              $20 == Any polariser
              $21 == Linear polariser (skwe) controller
              $30 == Any positioner
              $31 == Polar/azimuth positioner
              $32 == Elevation positioner
              $40 == Any installer aid
              $41 == Signal strength analog value
              $6x == Reserved for address re-allocations
              $70 == Any intelligent slave interface
              $71 == Interface for subscriber controlled headends
              $Fx == Reserved for OEM extensions
            . command byte (only mandatory commands are shown)
              $00 == Reset DiSEqC microcontroller
              $38 == Write to port group 0 (committed switches), 1 additional byte for switch id
                     Additional data byte:
                     %1111Oxxx = Option (A/B)
                     %1111xPxx = Satellite Position (A/B)
                     %1111xxHx = Polarization (Vertical/Horizontal)
                     %1111xxxB = Band (Low/High)
              $39 == Write to port group 1 (uncommitted switches), 1 additional byte for switch id
                     Additional data byte:
                     %11114xxx = Switch 4 (A/B)
                     %1111x3xx = Switch 3 (A/B)
                     %1111xx2x = Switch 2 (A/B)
                     %1111xxx1 = Switch 1 (A/B)
              $58 == Write channel frequency (BCD), 2/3 additional bytes for frequency
              $60 == Stop positioner movement
              $63 == Disable limits
              $66 == Set east limit
              $67 == Set west limit
              $68 == Drive motor east, 1 additional byte with timeout/steps
              $69 == Drive motor west, 1 additional byte with timeout/steps
              $6A == Store satellite position, 1 additional byte for satellite id
              $6B == Drive motor to satellite position,  1 additional byte for satellite id
            . additional data bytes
------------------------------------------------------------------------------}
function Saa7146aDiSEqCCommand(Handle: THandle; Repeats: Byte; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
var
  DiSEqCOriginal: Byte;
  RepeatCommands: Byte;
begin
  Result := False;

  // Because DiSEqC distorts the 22 kHz signal we need to restore it afterwards
  // For this we need the current status
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, DiSEqCOriginal) then
    Exit;

  // First we have to handle any DiSEqc command
  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) and (CommandLength > 0) then
  begin
    CommandData[0] := $E0;
    if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength, CommandData, True) then
      Exit;
    RepeatCommands := Repeats;
    while RepeatCommands > 0 do
    begin
      // Before a repeated transmission is send a pause of 100 ms is
      // required to have the device being initialized.
      Saa7146aUsDelay(100000);
      // Command from master, no reply required, repeated transmission
      CommandData[0] := $E1;
      if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength, CommandData, True) then
        Exit;
      Dec(RepeatCommands);
    end;
  end;
  // If only a burst is requested
  if (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB)) <> 0 then
  begin
    if (CommandType and CDiSEqCBurstA) <> 0 then
    begin
      if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, CommandData, True) then
        Exit;
    end
    else
      if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, CommandData, True) then
        Exit;
  end;

  // Restore 22kHz tone, we need a minimum of 15 ms delay first
  Saa7146aUsDelay(16000);
  if (DiSEqCOriginal and CStv0299bModulationModeMask) = CStv0299bModulationContinuous then
    Saa7146aHighBandLNB(Handle)
  else
    Saa7146aLowBandLNB(Handle);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <CommandType>     Type of command (bit-wise selection)
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              It is allowed to enable both burst and
                              DiSEqc, although only one type of burst can be selected
                              (A has priority)
            <CommandLength>   Number of <CommandData> bytes
            <CommandData>     Data bytes to transmit through DiSEqC
                              First byte is not used but instead replaced by
                              locally generated framing byte.
  Returns : <Result>          Error number

  Descript: Send DiSEqc command and/or simple burst mechanism (or both)
            Data is used -as is-.
  Notes   : This call is used to send a sequence of DiSEqC and/or V-SEC
            commands.
            The 'burst' is used for simple switchers and can be used together
            with a 'normal' DiSEqC command.
            DiSEqC basically consists of the following:
            . framing byte
              $E0 == Command from master, no reply required, first transmission
              $E1 == Command from master, no reply required, repeated transmission
              $E2..$E7 require a reply from a slave which is not supported
            . address byte
              $X_ == family
              $_X == sub type
              $00 == Any device
              $10 == Any LNB, switcher or SMATV
              $11 == LNB
              $12 == LNB with loop through switching
              $14 == Switcher (DC blocking)
              $15 == Switcher with DC loop through
              $18 == SMATV
              $20 == Any polariser
              $21 == Linear polariser (skwe) controller
              $30 == Any positioner
              $31 == Polar/azimuth positioner
              $32 == Elevation positioner
              $40 == Any installer aid
              $41 == Signal strength analog value
              $6x == Reserved for address re-allocations
              $70 == Any intelligent slave interface
              $71 == Interface for subscriber controlled headends
              $Fx == Reserved for OEM extensions
            . command byte (only mandatory commands are shown)
              $00 == Reset DiSEqC microcontroller
              $38 == Write to port group 0 (committed switches), 1 additional byte for switch id
                     Additional data byte:
                     %1111Oxxx = Option (A/B)
                     %1111xPxx = Satellite Position (A/B)
                     %1111xxHx = Polarization (Vertical/Horizontal)
                     %1111xxxB = Band (Low/High)
              $39 == Write to port group 1 (uncommitted switches), 1 additional byte for switch id
                     Additional data byte:
                     %11114xxx = Switch 4 (A/B)
                     %1111x3xx = Switch 3 (A/B)
                     %1111xx2x = Switch 2 (A/B)
                     %1111xxx1 = Switch 1 (A/B)
              $58 == Write channel frequency (BCD), 2/3 additional bytes for frequency
              $60 == Stop positioner movement
              $63 == Disable limits
              $66 == Set east limit
              $67 == Set west limit
              $68 == Drive motor east, 1 additional byte with timeout/steps
              $69 == Drive motor west, 1 additional byte with timeout/steps
              $6A == Store satellite position, 1 additional byte for satellite id
              $6B == Drive motor to satellite position,  1 additional byte for satellite id
            . additional data bytes
------------------------------------------------------------------------------}
function Saa7146aDiSEqCCommand2(Handle: THandle; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
var
  DiSEqCOriginal: Byte;
begin
  Result := False;

  // Because DiSEqC distorts the 22 kHz signal we need to restore it afterwards
  // For this we need the current status
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, DiSEqCOriginal) then
    Exit;

  // First we have to handle any DiSEqc command
  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) and (CommandLength > 0) then
    if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength, CommandData, True) then
      Exit;
  // If only a burst is requested
  if (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB)) <> 0 then
  begin
    if (CommandType and CDiSEqCBurstA) <> 0 then
    begin
      if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, CommandData, True) then
        Exit;
    end
    else
      if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, CommandData, True) then
        Exit;
  end;

  // Restore 22kHz tone, we need a minimum of 15 ms delay first
  Saa7146aUsDelay(16000);
  if (DiSEqCOriginal and CStv0299bModulationModeMask) = CStv0299bModulationContinuous then
    Saa7146aHighBandLNB(Handle)
  else
    Saa7146aLowBandLNB(Handle);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Extracted>  String to process with as result the extracted part
  Returns : <Result>     Original string without extracted part

  Descript: Extract part from string (space separated)
  Notes   :
------------------------------------------------------------------------------}
function ExtractSpacedPart(var Extracted: ShortString): ShortString;
var
  SpacePos: Integer;
begin
  Result := '';
  Extracted := Trim(Extracted);
  SpacePos := Pos(' ', Extracted);
  if SpacePos <> 0 then
  begin
    Result := Copy(Extracted, 1, SpacePos-1);
    Delete(Extracted, 1, SpacePos);
  end
  else
  begin
    Result := Extracted;
    Extracted := '';
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Setting>         Linux VDR type of string
  Returns : <Result>          True for success
                              False if error detected (DiSEqC part is not checked)
            <Satellite>       Satellite
            <SLOF>            Switch LOF
                              0 if error
            <Polarity>        Polarity
                              ' ' if error
            <LOF>             LOF
                              0 if error
            <DiSEqC>          DiSEqC command

  Descript: Extract individual information from setting string
  Notes   : DiSEqC configuration for VDR
              Format:   satellite slof polarization lof command
                            |       |        |       |      |
                            |       |        |       |     'DiSEqC' command (optional)
                            |       |        |      Local Oscillator Frequency
                            |       |       H/V polarization
                            |      Switch frequency
                           Satellite identifier
            S19.2E  11700 V  9750  t v W15 [E0 10 38 F0] W15 A W15 t
------------------------------------------------------------------------------}
function ExtractFromDiSEqCSetting(
           Setting      : ShortString;
           var Satellite: ShortString;
           var SLOF     : Integer;
           var Polarity : Char;
           var LOF      : Integer;
           var DiSEqC   : ShortString): Boolean;
var
  ProcessString: ShortString;
  Error        : Integer;
begin
  Result    := True;
  Satellite := '';
  SLOF      := 0;
  Polarity  := ' ';
  LOF       := 0;
  DiSEqC    := '';

  Satellite     := ExtractSpacedPart(Setting);             // Satellite
  ProcessString := ExtractSpacedPart(Setting);             // SLOF
  Val(ProcessString, SLOF, Error);
  if (ProcessString = '') or (Error <> 0) then
  begin
    SLOF := 0;
    Result := False;
  end;
  ProcessString := ExtractSpacedPart(Setting);             // Polarity
  if Length(ProcessString) <> 1 then
  begin
    Polarity := ' ';
    Result := False;
  end
  else
  begin
    Polarity := ProcessString[1];
    if not ((Polarity) in ['V', 'v', 'H', 'h']) then
      Result := False;
  end;
  ProcessString := ExtractSpacedPart(Setting);             // LOF
  Val(ProcessString, LOF, Error);
  if (ProcessString = '') or (Error <> 0) then
  begin
    LOF := 0;
    Result := False;
  end;
  DiSEqC := Setting;                                       // DiSEqC
end;


{------------------------------------------------------------------------------
  Params  : <Setting>         Linux VDR type of string
  Returns : <Result>          True for success
                              False if error detected (DiSEqC part is not checked)
            <Satellite>       Satellite
            <SLOF>            Switch LOF
                              0 if error
            <Polarity>        Polarity
                              ' ' if error
            <LOF>             LOF
                              0 if error
            <DiSEqC>          DiSEqC command

  Descript: Create setting string froms ettings
  Notes   :
------------------------------------------------------------------------------}
function CreateDiSEqCSetting(
           Satellite: ShortString;
           SLOF     : Integer;
           Polarity : Char;
           LOF      : Integer;
           DiSEqC   : ShortString): ShortString;
begin
  Result := format('%s %5.5d %s %5.5d %s', [Satellite, SLOF, Polarity, LOF, DiSEqC]);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <CommandData>     Linux VDR type of string
  Returns : <Result>          True for success

  Descript: Send DiSEqC sequence. Equivalent with Linux (VDR) definition.
  Notes   : Commands are processed until no more commands or an error is detected,
            which means that incorrect command sequences are carried out until
            the incorrect part is processed.
            DiSEqC configuration for VDR
              Format:   satellite slof polarization lof [command]
                            |       |        |       |      |
                            |       |        |       |     'DiSEqC' command (optional)
                            |       |        |      Local Oscillator Frequency
                            |       |       H/V polarization
                            |      Switch frequency
                           Satellite identifier
              The optional [command] is made up of the following
                t         Tone off
                T         Tone on
                v         Voltage low (13V)
                V         Voltage high (18V)
                A         Mini A (burst A)
                B         Mini B (burst B)
                Wxx       Wait xx milliseconds
                [xx ..]   Hex code sequence (max 6), eg [E0 10 38 F0]
            Example:
            S19.2E  11700 V  9750  t v W15 [E0 10 38 F0] W15 A W15 t
------------------------------------------------------------------------------}
function Saa7146aDiSEqCCommandVdr(Handle: THandle; CommandData: ShortString): Boolean;
var
  Satellite    : ShortString;
  Slof         : Integer;
  Polarity     : Char;
  Lof          : Integer;
  DiSEqCCommand: ShortString;
  ProcessString: ShortString;
  Error        : Integer;
  Value        : Integer;
  DiSEqCData   : TDiSEqCData;
  DiSEqCIndex  : Integer;
  EndDetected  : Boolean;
begin
  Result := True;

  if ExtractFromDiSEqCSetting(CommandData, Satellite, Slof, Polarity, Lof, DiSEqCCommand) then
    while (DiSEqCCommand <> '') and (Result = True) do
    begin
      ProcessString := ExtractSpacedPart(DiSEqCCommand);
      if ProcessString <> '' then
      begin
        case Ord(ProcessString[1]) of
          Ord('t'):
            begin
              // Tone off (22 kHz)
              if not Saa7146aLowBandLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('T'):
            begin
              // Tone on (22 kHz)
              if not Saa7146aHighBandLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('v'):
            begin
              // 13V
              if not Saa7146aVerticalPolarityLNB(Handle, LastPolarityOp) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('V'):
            begin
              // 18V
              if not Saa7146aHorizontalPolarityLNB(Handle, LastPolarityOp) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('A'), Ord('a'):
            begin
              // Burst A
              if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, DiSEqCData, False) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('B'), Ord('b'):
            begin
              // Burst B
              if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, DiSEqCData, False) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('W'), Ord('w'):
            begin
              // Wait
              Delete(ProcessString, 1, 1);
              Val(ProcessString, Value, Error);
              if (Error = 0) and (Value > 0) then
                Saa7146aUsDelay(Value * 1000);
            end;
          Ord('['):
            begin
              // DiSEqC command sequence
              Delete(ProcessString, 1, 1);
              // In case a '[' without trailing data was used read next data
              if ProcessString = '' then
                ProcessString := ExtractSpacedPart(DiSEqCCommand);
              DiSEqCIndex := Low(DiSEqCData);
              EndDetected := False;
              repeat
                // Check on exceeding length
                if DiSEqCIndex > High(DiSEqCData) then
                begin
                  Result := False;
                  Exit;
                end;
                // Check on two byte value
                // Since a single byte command just don't exists it does not
                // matter if the command was something like '[E0]' which
                // should be allowed but is detected as en error here (which
                // is good BTW).
                if Length(ProcessString) <> 2 then
                begin
                  Result := False;
                  Exit;
                end;
                // Convert as hexadecimal
                Val('$' + ProcessString, DiSEqCData[DiSEqCIndex], Error);
                // Check on conversion error
                if Error <> 0 then
                begin
                  Result := False;
                  Exit;
                end;
                Inc(DiSEqCIndex);
                if not EndDetected then
                begin
                  // Get next data 'byte'. If we detect the ']' marker then
                  // indicate this is only a final pass is done
                  ProcessString := ExtractSpacedPart(DiSEqCCommand);
                  if (Length(ProcessString) = 1) and (ProcessString[1] = ']') then
                  begin
                    // If we are dealing with a separate ']' we are done too
                    ProcessString := '';
                  end;
                  if Length(ProcessString) > 2 then
                  begin
                    if ProcessString[3] <> ']' then
                    begin
                      Result := False;
                      Exit;
                    end;
                    Delete(ProcessString, 3, 1);
                    EndDetected := True;
                  end;
                end
                else
                  ProcessString := '';
              until (ProcessString = '');
              // Send DiSEqCData
              if not Saa7146aSendDiSEqCMsg(Handle, CDiSEqCCommand, DiSEqCIndex, DiSEqCData, False) then
              begin
                Result := False;
                Exit;
              end;
            end;
          else
            Result := False;  
        end;
      end;
    end
  else
    Result := False;
    end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AddressIndex>    Address of TSA5059 chip (0..3)
            <Data>            Data to write (control2, control1, divider2, divider1)
  Returns : <Result>          True for success

  Descript: Write data to synthesizer
  Notes   : Since the synthesizer is NOT connected to the SAA7146A, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}
function Saa7146aWriteToSynthesizer(Handle: THandle; AddressIndex: Byte; Data: Dword): Boolean;
var
  Retry      : Byte;
  RetryWhole : Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;
  // Save copies for when we need to update the synthesizer elsewhere
  LastSynthesizerData  := Data;
  LastSynthesizerIndex := AddressIndex;
  if AddressIndex > High(CTsa5059SynthesizerWriteAddress) then
    Exit;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // First we need to enable the repeater of the QSPK demodulator
      // BSRU6 or SU1278 does not matter here
      if not Saa7146aWriteToQpsk(Handle, CStv0299bI2cRpt,
        CStv0299bDefaultsBSRU6[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn) then
        Exit;
      // Execute the write (must be 2 stages because we can only send 3 bytes at a time
      // and we need to send 5 bytes in total
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart             shl 6)  or      // First byte
          (CTsa5059SynthesizerWriteAddress[AddressIndex]   shl 24) or      //   START - device address WRITE
          (CSaa7146aI2cAttrCont              shl 4)  or      // Second byte
          (Data and $FF                      shl 16) or      //   programmable divider 1
          (CSaa7146aI2cAttrCont              shl 2)  or      // Third byte
          (((Data shr 8) and $FF)            shl 8),         //   programmable divider 2
          20);                                               // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if NeedNoRetry then
      begin
        Retry := 1;
        repeat
          NeedNoRetry := Saa7146aI2cHandshake(Handle,
            (CSaa7146aI2cAttrCont              shl 6)  or    // Fourth byte
            (((Data shr 16) and $FF)           shl 24) or    //   control 1
            (CSaa7146aI2cAttrStop              shl 4)  or    // Fifth byte
            (((Data shr 24) and $FF)           shl 16) or    //   control 2 - STOP
            (CSaa7146aI2cAttrNop               shl 2)  or    // Sixth byte not sent
            (0                                 shl 8),
            20);                                             // Timeout essential otherwise sometimes error
          if not NeedNoRetry then
            Dec(Retry);
        until NeedNoRetry or (Retry = 0);
      end;
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AddressIndex>    Address of TSA5059 chip (0..3)
  Returns : <Result>          True for success
            <Data>            Data read (status register)

  Descript: Read data from synthesizer
  Notes   : Since the synthesizer is NOT connected to the SAA7146A, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}
function Saa7146aReadFromSynthesizer(Handle: THandle; AddressIndex: Byte; var Data: Byte): Boolean;
var
  I2cTrf     : Dword;
  Retry      : Byte;
  RetryWhole : Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  if AddressIndex > High(CTsa5059SynthesizerWriteAddress) then
    Exit;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // First we need to enable the repeater of the QSPK demodulator
      // BSRU6 or SU1278 does not matter here
      if not Saa7146aWriteToQpsk(Handle, CStv0299bI2cRpt,
        CStv0299bDefaultsBSRU6[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn) then
        Exit;
      // Execute the read
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart             shl 6)  or    // First byte send
          (CTsa5059SynthesizerReadAddress[AddressIndex]    shl 24) or    // START - device address READ
          (CSaa7146aI2cAttrStop              shl 4)  or    // Second byte send
          (0                                 shl 16) or    // dummy (filled in by targetted device) - STOP
          (CSaa7146aI2cAttrNop               shl 2)  or    // Third byte not sent
          (0                                 shl 8),
          20);                                             // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;
    // Return copy of read value
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cTrf, I2cTrf) then
      Exit;
    Data := (I2cTrf shr 16) and $FF;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          Address of synthesizer (0..3) or 255 if failed

  Descript: Detect address to be used for synthesizer
  Notes   : Must be used AFTER I2C initialization ....
------------------------------------------------------------------------------}
function Saa7146aDetectSynthesizerAddress(Handle: THandle): Byte;
var
  Address     : Byte;
  Success     : Boolean;
  ReturnedByte: Byte;
begin
  // Scan for correct synthesizer address
  // This is needed because different cards can set this address differently
  // (by hardware)
  Address := 0;
  repeat
    Success := Saa7146aReadFromSynthesizer(Handle, Address, ReturnedByte);
    if not Success then
      Inc(Address);
  until Success or (Address > 3);
  if Success then
    Result := Address
  else
    Result := 255;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                     <Saa7146aCreateFile> handle
            <Frequency>                  Frequency to set (demodulated, in kHz)
            <LowBandLocalOscillatorKHz>  Local oscillator frequency of low band
            <HighBandLocalOscillatorKHz> Local oscillator frequency of high band
            <TunerType>                  Type of tuner
  Returns : <Result>                     True  for low  band
                                         False for high band

  Descript: Check if low band to use
------------------------------------------------------------------------------}
function Saa7146aUseLowBand(Handle: THandle; FrequencyKHz: Dword;
    LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword; TunerType: Byte): Boolean;
const
  LowestTunerFrequency  =  950000;                      // Lowest  frequency tuner kHz (for all tuners sofar)
  HighestTunerFrequency = 2150000;                      // Highest frequency tuner kHz
var
  ActualFrequency: Integer;
begin
  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  Result := True;
  // Note: The <Abs> allows for negative results (eg. C-band)
  ActualFrequency := FrequencyKHz;
  ActualFrequency := Abs(Integer(ActualFrequency) - Integer(LowBandLocalOscillatorKHz));
  if ActualFrequency > HighestTunerFrequency then
    Result := False;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                     <Saa7146aCreateFile> handle
            <Frequency>                  Frequency to set (demodulated, in kHz)
            <LowBandLocalOscillatorKHz>  Local oscillator frequency of low band
            <HighBandLocalOscillatorKHz> Local oscillator frequency of high band
            <TunerType>                  Type of tuner
            <AddressIndex>               Address index of synthesizer (0..3)
                                         BSRU6 (NOVA) : 1
                                         SU1278       : 0
                                         SU1278 (1200): 1
  Returns : <Result>                     True for success

  Descript: Set frequency of LNB
  Notes   : The frequency is controlled by the TSA5059 synthesizer which
            is controlled by the I2C (on the repeater of the QPSK demodulator!).
            The <LocalOscillator> is used to calculate the actual frequency for
            the synthesizer.
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}
function Saa7146aFrequencyLNB(Handle: THandle; FrequencyKHz: Dword;
           LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword;
           TunerType: Byte; AddressIndex: Byte): Boolean;
const
  Reference = 4000000;
  Ratios : array[0..15] of Dword =
    (Reference div 2,  Reference div 4,   Reference div 8,   Reference div 16,
     Reference div 32, Reference div 64,  Reference div 128, Reference div 256,
     0,
     Reference div 5,  Reference div 10,  Reference div 20,  Reference div 40,
     Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency  =  950000000;                      // Lowest frequency tuner
  HighestTunerFrequency = 2150000000;                      // Highest frequency tuner
var
  Loop                 : Byte;
  ActualFrequency      : Int64;
  SmallestError        : Dword;
  CalcError            : Int64;
  RatioForSmallestError: Word;
  CalcDivider          : Int64;
  ProgrammableDivider1 : Byte;
  ProgrammableDivider2 : Byte;
  ControlData1         : Byte;
  ControlData2         : Byte;
  SynthesizerData      : Dword;
  LowBand              : Boolean;
begin
  Result := False;
  LastTunerType := TunerType;

  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  LowBand := Saa7146aUseLowBand(Handle, FrequencyKHz, LowBandLocalOscillatorKHz, HighBandLocalOscillatorKHz, TunerType);
  // Note: The <Abs> allows for negative results (eg. C-band)
  ActualFrequency := FrequencyKHz;
  if LowBand then
    ActualFrequency := Abs(ActualFrequency - LowBandLocalOscillatorKHz)
  else
    ActualFrequency := Abs(ActualFrequency - HighBandLocalOscillatorKHz);
  ActualFrequency := ActualFrequency * 1000;
  if (ActualFrequency > HighestTunerFrequency)  or
     (ActualFrequency < LowestTunerFrequency) then
    Exit;
  // Now the band is know we have to set this.
  if LowBand then
  begin
    if not Saa7146aLowBandLNB(Handle) then
      Exit;
  end
  else
    if not Saa7146aHighBandLNB(Handle) then
      Exit;
  // To set the correct frequency we have to set two different values: the
  // 'divider' and the 'ratio'. The actual calculations involved are:
  // FREQUENCY = DIVIDER * (4 MHz / RATIO)
  // Since we have to calculate the DIVIDER and RATIO from the FREQUENCY we have to
  // do some calculation, mostly to get an as accurate result.
  // RATIO    = 2,       4,       8,      16,      32,     64,    128,   256,
  // 4M/RATIO = 2000000, 1000000, 500000, 250000,  125000, 62500, 31250, 15625
  // RATIO    = 5,       10,      20,     40,      80,     160,   320
  // 4M/RATIO = 800000,  400000,  200000, 1000000, 50000,  25000, 12500
  // DIVIDER = 0..$1FFFF (0..131071)
  // At this point we could do some calculations and such to get the best result,
  // but the simplest method is just to check all possibilities and choose the
  // one with the smallest error. Alternatively we could choose a fixed ratio which
  // will limit the step size. For now we will use the one with the smallest error.

  RatioForSmallestError := High(Ratios);
  SmallestError         := ActualFrequency;
  for Loop := Low(Ratios) to High(Ratios) do
  begin
    if Ratios[Loop] <> 0 then
    begin
      CalcDivider := ActualFrequency div Ratios[Loop];
      // Must fit in 17 bits
      if CalcDivider <= $1FFFF then
      begin
        CalcError   := ActualFrequency - (CalcDivider * Ratios[Loop]);
        if CalcError < SmallestError then
        begin
          SmallestError := CalcError;
          RatioForSmallestError := Loop;
        end;
      end;
    end;
    // If the error is zero then there is no need to continu checking....
    if SmallestError = 0 then
      Break;
  end;

  // With the divider information available we can construct the 4 bytes we
  // need to send to the synthesizer. The DIVIDER is divided over 3 bytes.
  CalcDivider := ActualFrequency div Ratios[RatioForSmallestError];
  ProgrammableDivider2 := CalcDivider and $FF;
  CalcDivider := CalcDivider shr 8;
  ProgrammableDivider1 := CalcDivider and $7F;
  CalcDivider := (CalcDivider shr 2) and $60;
  ControlData1 := $80 or CalcDivider or RatioForSmallestError;
  case TunerType of
    CTunerBSRU6 : ControlData2 := CTsa5059DefaultControlRegisterBSRU6;
    CTunerBSBE1 : ControlData2 := CTsa5059DefaultControlRegisterBSBE1;
    CTunerSU1278:
      begin
        ControlData2 := CTsa5059DefaultControlRegisterSU1278;
        // Correction for low symbolrate
        if LastSymbolRate < 4000 then
          Controldata2 := Controldata2 or CTsa5059SetPort0;
        // Depending on the frequnecy range we need to set the charge pump correct
        // 950  - 1250  -  1550  -  2050   -   2150
        //     00       01       10        11
        if ActualFrequency <= 1250000000 then
          ControlData2 := ControlData2 or $00
        else
          if ActualFrequency <= 1550000000 then
            ControlData2 := ControlData2 or $40
          else
            if ActualFrequency <= 2050000000 then
              ControlData2 := ControlData2 or $80
            else
              ControlData2 := ControlData2 or $C0;
      end;
    else
      Exit;
  end;
  SynthesizerData := ProgrammableDivider1 or
    (ProgrammableDivider2 shl 8) or
    (ControlData1 shl 16) or
    (ControlData2 shl 24);
  // Now we can send the data to the synthesizer
  if not Saa7146aWriteToSynthesizer(Handle, AddressIndex, SynthesizerData) then
    Exit;
  // Because the STV0299 might have to lock to this (new) frequency we have
  // to reset the de-rotator loop
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrM, $00) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrL, $00) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrL, Loop) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfd, $B9) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                     <Saa7146aCreateFile> handle
            <Frequency>                  Frequency to set (modulated, in kHz)
            <TunerType>                  Type of tuner
            <AddressIndex>               Address index of synthesizer (0..3)
                                         BSRU6 (NOVA) : 1
                                         SU1278       : 0
                                         SU1278 (1200): 1
  Returns : <Result>                     True for success

  Descript: Set frequency of LNB. Frequency is already in tuner format.
            Assumes low/high band is hanlded elsewhere.
  Notes   : The frequency is controlled by the TSA5059 synthesizer which
            is controlled by the I2C (on the repeater of the QPSK demodulator!).
            The <LocalOscillator> is used to calculate the actual frequency for
            the synthesizer.
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}
function Saa7146aFrequencyTunerLNB(Handle: THandle; FrequencyKHz: Dword;
           TunerType: Byte; AddressIndex: Byte): Boolean;
const
  Reference = 4000000;
  Ratios : array[0..15] of Dword =
    (Reference div 2,  Reference div 4,   Reference div 8,   Reference div 16,
     Reference div 32, Reference div 64,  Reference div 128, Reference div 256,
     0,
     Reference div 5,  Reference div 10,  Reference div 20,  Reference div 40,
     Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency  =  950000000;                      // Lowest frequency tuner
  HighestTunerFrequency = 2150000000;                      // Highest frequency tuner
var
  Loop                 : Byte;
  ActualFrequency      : Int64;
  SmallestError        : Dword;
  CalcError            : Int64;
  RatioForSmallestError: Word;
  CalcDivider          : Int64;
  ProgrammableDivider1 : Byte;
  ProgrammableDivider2 : Byte;
  ControlData1         : Byte;
  ControlData2         : Byte;
  SynthesizerData      : Dword;
begin
  Result := False;
  LastTunerType := TunerType;

  ActualFrequency := FrequencyKHz;
  ActualFrequency := ActualFrequency * 1000;
  // To set the correct frequency we have to set two different values: the
  // 'divider' and the 'ratio'. The actual calculations involved are:
  // FREQUENCY = DIVIDER * (4 MHz / RATIO)
  // Since we have to calculate the DIVIDER and RATIO from the FREQUENCY we have to
  // do some calculation, mostly to get an as accurate result.
  // RATIO    = 2,       4,       8,      16,      32,     64,    128,   256,
  // 4M/RATIO = 2000000, 1000000, 500000, 250000,  125000, 62500, 31250, 15625
  // RATIO    = 5,       10,      20,     40,      80,     160,   320
  // 4M/RATIO = 800000,  400000,  200000, 1000000, 50000,  25000, 12500
  // DIVIDER = 0..$1FFFF (0..131071)
  // At this point we could do some calculations and such to get the best result,
  // but the simplest method is just to check all possibilities and choose the
  // one with the smallest error. Alternatively we could choose a fixed ratio which
  // will limit the step size. For now we will use the one with the smallest error.

  RatioForSmallestError := High(Ratios);
  SmallestError         := ActualFrequency;
  for Loop := Low(Ratios) to High(Ratios) do
  begin
    if Ratios[Loop] <> 0 then
    begin
      CalcDivider := ActualFrequency div Ratios[Loop];
      // Must fit in 17 bits
      if CalcDivider <= $1FFFF then
      begin
        CalcError   := ActualFrequency - (CalcDivider * Ratios[Loop]);
        if CalcError < SmallestError then
        begin
          SmallestError := CalcError;
          RatioForSmallestError := Loop;
        end;
      end;
    end;
    // If the error is zero then there is no need to continu checking....
    if SmallestError = 0 then
      Break;
  end;

  // With the divider information available we can construct the 4 bytes we
  // need to send to the synthesizer. The DIVIDER is divided over 3 bytes.
  CalcDivider := ActualFrequency div Ratios[RatioForSmallestError];
  ProgrammableDivider2 := CalcDivider and $FF;
  CalcDivider := CalcDivider shr 8;
  ProgrammableDivider1 := CalcDivider and $7F;
  CalcDivider := (CalcDivider shr 2) and $60;
  ControlData1 := $80 or CalcDivider or RatioForSmallestError;
  case TunerType of
    CTunerBSRU6 : ControlData2 := CTsa5059DefaultControlRegisterBSRU6;
    CTunerBSBE1 : ControlData2 := CTsa5059DefaultControlRegisterBSBE1;
    CTunerSU1278:
      begin
        ControlData2 := CTsa5059DefaultControlRegisterSU1278;
        // Correction for low symbolrate
        if LastSymbolRate < 4000 then
          Controldata2 := Controldata2 or CTsa5059SetPort0;
        // Depending on the frequnecy range we need to set the charge pump correct
        // 950  - 1250  -  1550  -  2050   -   2150
        //     00       01       10        11
        if ActualFrequency <= 1250000000 then
          ControlData2 := ControlData2 or $00
        else
          if ActualFrequency <= 1550000000 then
            ControlData2 := ControlData2 or $40
          else
            if ActualFrequency <= 2050000000 then
              ControlData2 := ControlData2 or $80
            else
              ControlData2 := ControlData2 or $C0;
      end;
    else
      Exit;
  end;
  SynthesizerData := ProgrammableDivider1 or
    (ProgrammableDivider2 shl 8) or
    (ControlData1 shl 16) or
    (ControlData2 shl 24);
  // Now we can send the data to the synthesizer
  if not Saa7146aWriteToSynthesizer(Handle, AddressIndex, SynthesizerData) then
    Exit;
  // Because the STV0299 might have to lock to this (new) frequency we have
  // to reset the de-rotator loop
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrM, $00) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrL, $00) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrL, Loop) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfd, $B9) then
    Exit;

  Result := True;
end;


initialization
  LastI2cError := $0000;
  HighPerformanceAvailable := QueryPerformanceFrequency(HighPerformanceFrequency);
  I2cLock := TCriticalSection.Create;
  LastSymbolRate := 27500;
  LastEnableOp   := True;
  LastPolarityOp := True;

finalization
  I2cLock.Free;
end.

