{******************************************************************************}
{ FileName............: DvbStreamBuffering2                                    }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.04                                                   }
{------------------------------------------------------------------------------}
{   Buffering of stream data.                                                  }
{   Infrared control.                                                          }
{                                                                              }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Notes: Instead of acquiring a single block of contiguous memory we could     }
{        use multiple blocks instead. We need to switch off the protection     }
{        address of the SAA7146A then.                                         }
{                                                                              }
{ Includes master / slave principle for multiple instances which will use the  }
{ same packet data.                                                            }
{                                                                              }
{------------------------------------------------------------------------------}
{ The infrared interface uses a dedicated uP and is connected to the GPIO3     }
{ input. When an infrared command is detected by this uP it activates this     }
{ line after which the code can be read through the DEBI interface.            }
{ Here we use a RPS program which runs continuously. This RPS waits            }
{ for a GPIO3 activation and then reads the infrared data.                     }
{ No direct GPIO interrupts are involved because of the continuous             }
{ 'polling' by means of the RPS program.                                       }
{ Note that the infrared uP activates GPIO3 three times for the total          }
{ information of the infrared data.                                            }
{ Two of the data words (low byte) contain the key codes itself, but with the  }
{ high nibbles changed: eg. $C4 and $44                                        }
{                           $Cx <> $4x                                         }
{                           $Dx <> $5x                                         }
{                           $Ex <> $6x                                         }
{                           $Fx <> $7x                                         }
{ These two key codes are the last detected keys. Typically these are the same }
{ except when a key is pressed very shortly.                                   }
{ The key code starts at $40 (command 0) and ends at $7F (command 63)          }
{ Extended commands (64..127) are not supported, meaning that command 64 will  }
{ be interpreted as command 0, and command 127 as 63.                          }
{ The third data byte contains the toggle code and the device (low nibble)     }
{ eg. $x3 indicates device '3'                                                 }
{ The toggle adds $20 to the device code.                                      }
{ In short:                                                                    }
{   Data < $40 == device/repeat                                                }
{   Data >=$40 == key code                                                     }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20031025 - Initial release                                           }
{  1.01   20040420 - Allow for different card/tuner                            }
{  1.02   20040619 - Corrected a small bug which did not correct the BufferTime}
{                    so it was actuall 20 times too large                      }
{  1.03   20040627 - Calls changed to pointer types so there use as DLL is more}
{                    obvious                                                   }
{                  - Added <try..except> for external calls                    }
{  1.04   20041003 - Added different RPS0 tasks switch (plus VSYNC emulation)  }
{******************************************************************************}
unit DvbStreamBuffering;

interface
uses
  Windows;

type
  PBoolean = ^Boolean;

  function DvbSetupBuffering(const CardHandle: THandle; const Initialize: Boolean; const TunerType: Byte; const TunerReset: Byte;
                             IsSlave: PBoolean; BufferTime: PWord; BufferId: PDword; Rps0Program: Integer): Boolean; stdcall;
  function DvbGetRemoteControl(const CardHandle: THandle; DeviceId: PByte; Command: PByte)                : Boolean; stdcall;


implementation
uses
  Dvb,
  Saa7146aGpio,
  Saa7146AI2c,
  Saa7146AInterface,
  Saa7146aIoControl,
  Saa7146aRegisters,
  Stv0299bRegisters;

const
  MaxPacketBuffers = 250;                        // Absolute maximum number of buffers
                                                 // Larger values are obtainable but time needed
                                                 // by operating system becomes longer (needs to be
                                                 // consecutive piece of memory!)
                                                 // A value of 340 is about 64 Mbytes!!

  CInfraredAddress = $4000;

var
  RcRPS0       : TSaa7146aDmaBuffer;             // DMA buffer for RPS0 program
  RcRPS0Data   : TSaa7146aDmaBuffer;             // DMA buffer for RPS data
  RcKey1       : Byte;                           // Previous results
  RcKey2       : Byte;
  RcRepeat     : Boolean;
  RcDevice     : Byte;


{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
            <TunerType>   Tuner to address
  Returns : <Result>      True if success

  Descript: Do a (re)initialization of the card
  Notes   :
 ------------------------------------------------------------------------------}
function InitializeCard(CardHandle: THandle; TunerType: Byte; TunerReset: Byte): Boolean;
var
  ReturnedByte : Byte;
begin
  Result := False;
  // The RESET of the tuner is connected to a GPIO line of the SAA7146A, so we have to configure this
  if not Saa7146aInitializeGpio(CardHandle, (TunerType = CTunerBSRU6)) then
    Exit;
  // Now issue a reset signal
  if TunerReset <> 255 then
    if not Saa7146aIssueReset(CardHandle, TunerReset) then
      Exit;
  // Initialize the I2C
  if not Saa7146aInitializeI2c(CardHandle) then
    Exit;
  // Check QPSK
  if not Saa7146aReadFromQpsk(CardHandle, CStv0299bId, ReturnedByte) then
    Exit;
  // Check identifier of QPSK
  if not ReturnedByte = $A1 then
    Exit;
  // Set default QPSK values
  if not Saa7146aWriteDefaultsToQpsk(CardHandle, TunerType) then
    Exit;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
  Returns : <Result>      $FFFF if no master

  Descript: Detect a POSSIBLE master (fails if for some reason DMA memory
            was not released, eg. because of a crash).
  Notes   :
 ------------------------------------------------------------------------------}
function GetMasterChannel(CardHandle: THandle): Dword;
var
  Info          : TSaa7146aGetDmaStatus;
  Channel       : Dword;
  ChannelFound  : Boolean;
begin
  // As a slave we only need the identifier of the DMA buffer which was
  // used by the master.
  // The only way we can do is to check the size of such a buffer..
  Channel := 0;
  ChannelFound := False;
  // Any size > 128kB will be identified as a master being active ..
  while (Saa7146aGetDmaStatus(CardHandle, Channel, Info) and (not ChannelFound)) and (Channel < 256) do
  begin
    if Info.Size > 128000 then
      ChannelFound := True
    else
      Inc(Channel);
  end;
  if ChannelFound then
    Result := Channel
  else
    Result := $FFFF;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>     Handle to driver
            <IsSlave>        True if setup as slave
            <PacketBuffers>  Number of buffers to allocate
            <Rps0Program>    Indicates what RPS0 program to run
                             -1  : remote control
                             0..3: VSYNC on GPIOx
  Returns : <Result>         True if success
            <PacketBuffers>  Actual buffers allocated
            <BufferId>       Buffer identifier (DMA) for data stored into

  Descript: Setup buffering of DMA3 data.
  Notes   :
 ------------------------------------------------------------------------------}
function BufferingSetup(CardHandle: THandle; IsSlave: Boolean; var PacketBuffers: Word; var BufferId: Dword; Rps0Program: Integer): Boolean;
var
  Info          : TSaa7146aGetDmaStatus;
  TransferBuffer: array[0..MaxPacketBuffers*30] of Dword;
  Index         : Integer;
  Transfer      : TSaa7146aTransferBuffer;
  Channel       : Dword;
  BufferRps1    : TSaa7146aDmaBuffer;            // DMA buffer for RPS1 program
  Fifo3         : TSaa7146aDmaBuffer;
  Succeeded     : Boolean;

  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: Setup DD1 program
    Notes   :
   ------------------------------------------------------------------------------}
  procedure SetupDD1;
  var
    Buffers: Word;
  begin
    // Now we have allocated the memory we can setup the DMA3 for it.
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aBaseOdd3,                              // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart)) then      // Odd lines to here
      Exit;
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aBaseEven3,                             // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart + CDvbPacketBufferSize)) then     // Even lines to here
      Exit;

    // We must make sure we don't get a protection error
    if not Saa7146aWriteToSaa7146aRegister(CardHandle,
             CSaa7146aProtAddr3,                             // Register to write
             DWord(Fifo3.PhysicalAddress.LowPart + ((((PacketBuffers * CDvbPacketBufferSize * 2) div DMASIZE) + 1) * DMASIZE))) then
      Exit;

    // If we want more buffering then we use one of the RPS programs
    // of the SAA7146A to handli it. What this program does is nothing
    // more then changing the buffer addresses when it's necessary,
    // and thereby extending out data buffer.

    // Allocate some DMA buffers which will hold the RPS program
    // About 30 instruction of 4 bytes per buffer needed
    if not (Saa7146aAllocateDma(CardHandle, (((PacketBuffers * 30 * 4) div DMASIZE) + 1) * DMASIZE, BufferRPS1)) then
      Exit;

    // Now setup the RPS program. We first construct it in local memory and
    // then transport it to the SAA7146A
    Index := 0;
    for Buffers := 0 to PacketBuffers-1 do
    begin
      // Wait for even fields starting to be processed
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsEFidB;
      Inc(Index);
      // Now the even fields are being processed, change the odd field buffer address
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aBaseOdd3 shr 2);  // Write DMA3 base even field in shadow RAM
      Inc(Index);
      TransferBuffer[Index] := Fifo3.PhysicalAddress.LowPart + (CDvbPacketBufferSize * (Buffers*2));   // New address
      Inc(Index);
      // Needed to clear upload flag ??
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsVtd3;          // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsVtd3;         // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsVtd3;          // Wait for shadow upload to finish
      Inc(Index);
      // For feedback we use the event counter threshold register
      // We indicate which buffer is ready to be read, which is one before
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aEcT1R shr 2);    // Set indicator
      Inc(Index);
      if Buffers = 0 then
        TransferBuffer[Index] := (PacketBuffers*2) - 2
      else
        TransferBuffer[Index] := (Buffers*2) - 2;
      Inc(Index);

      TransferBuffer[Index] := CSaa7146aRpsInterrupt or CSaa7146aRpsLogicalOr or CSaa7146aRpsVtd3;    // Issue interrupt
      Inc(Index);

      // Wait for odd fields starting to be processed
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsOFidB;
      Inc(Index);
      // Now the odd fields are being processed, change the even field buffer address
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aBaseEven3 shr 2);  // Write DMA3 base even field in shadow RAM
      Inc(Index);
      TransferBuffer[Index] := Fifo3.PhysicalAddress.LowPart + (CDvbPacketBufferSize * ((Buffers * 2)+1)); // New address
      Inc(Index);
      // Needed to clear upload flag ??
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsVtd3;      // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);

      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsVtd3;         // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsVtd3;          // Wait for shadow upload to finish
      Inc(Index);
      // For feedback we use the event counter threshold register
      // We indicate which buffer is ready to be read, which is one before
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aEcT1R shr 2);    // Set indicator
      Inc(Index);
      if Buffers = 0 then
        TransferBuffer[Index] := (PacketBuffers*2)-1
      else
        TransferBuffer[Index] := (Buffers*2)-1;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsInterrupt or CSaa7146aRpsLogicalOr or CSaa7146aRpsVtd3;    // Issue interrupt
      inc(Index);
    end;
    // Jump to start program
    TransferBuffer[Index] := CSaa7146aRpsJump;                                 // Restart program
    Inc(Index);
    TransferBuffer[Index] := DWord(BufferRPS1.PhysicalAddress.LowPart);        // Address of start program
    Inc(Index);
    // Setup transfer data and transfer it
    Transfer.Identifier   := BufferRPS1.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Index * Sizeof(DWord);
    Saa7146aWriteToDma(CardHandle, Transfer);              // Transfer data
    // Now make sure the RPS program is started
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsAddr1,
      DWord(DWord(BufferRPS1.PhysicalAddress.LowPart)));   // Set physical start of RPS program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsPage1,
      0);                                                  // RPS program performs no explicit mem writes
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsTov1 ,
      0);                                                  // Disable RPS timeouts
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps1On);                                 // Start RPS1 program
  end;


  {------------------------------------------------------------------------------
    Params  : <Rps0Program>  Indicates not a remote control program to use
                             but to output a simulated VSYNC signal on the
                             specified GPIO line
                             -1   will generate the remote control program
                             0..3 will generate the VSYNC program for GPIOx
    Returns : -

    Descript: Setup remote control program
    Notes   :
   ------------------------------------------------------------------------------}
  procedure SetupRemoteControl(Rps0Program: Integer);
  var
    Address : Dword;
    GpioMask: Dword;
  begin
    if (Rps0Program < -1) or (Rps0Program > 3) then
      Exit;
    // Now setup the RPS program. We first construct it in local memory and
    // then transport it to the SAA7146A

    if Rps0Program in [0..3] then
    begin
      // VSYNC output
      // We want a change of VSYNC every 512 lines of our own generated signal
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aRpsThresh0,                          // Register to write
               CDvbPacketVSync or $00001000) then            // Source line counter threshold, every 512 lines (BRS related)
        Exit;
      case Rps0Program of
        0:   GpioMask := CSaa7146aGpio0Mask;
        1:   GpioMask := CSaa7146aGpio1Mask;
        2:   GpioMask := CSaa7146aGpio2Mask;
        3:   GpioMask := CSaa7146aGpio3Mask;
        else GpioMask := 0;
      end;
      Index    := 0;
      // Wait for release of line count threshold
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsHs;
      Inc(Index);
      // And for activation
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsHs;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsMaskLoad or (CSaa7146aGpioCtrl shr 2);
      Inc(Index);
      TransferBuffer[Index] := GpioMask;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aGpioOutputHigh;
      Inc(Index);

      // Wait for release of line count threshold
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsHs;
      Inc(Index);
      // And for next activation
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsHs;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsMaskLoad or (CSaa7146aGpioCtrl shr 2);
      Inc(Index);
      TransferBuffer[Index] := GpioMask;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aGpioOutputLow;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsJump;                                   // Restart program
      Inc(Index);
      TransferBuffer[Index] := DWord(RcRPS0.PhysicalAddress.LowPart);              // Address of start program
      Inc(Index);
    end;  
    if Rps0Program = -1 then
    begin
      // Remote control
      Index    := 0;
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
      Inc(Index);

      // Do a DEBI transfer
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or CInfraredAddress;       // Read, Address, blocklength=2
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
      Inc(Index);
      Address := 0;                                                                // Target index
      TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
      Inc(Index);
      // Do a DEBI transfer
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or CInfraredAddress;       // Read, Address, blocklength=2
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
      Inc(Index);
      Address := 4;                                                                // Target index
      TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
      Inc(Index);
      // Do a DEBI transfer
      TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or CInfraredAddress;       // Read, Address, blocklength=2
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsNop;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
      Inc(Index);
      Address := 8;                                                                // Target index
      TransferBuffer[Index] := DWord(RcRPS0Data.PhysicalAddress.LowPart) + Address;
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
      Inc(Index);
      TransferBuffer[Index] := CSaa7146aRpsJump;                                   // Restart program
      Inc(Index);
      TransferBuffer[Index] := DWord(RcRPS0.PhysicalAddress.LowPart);              // Address of start program
      Inc(Index);
    end;
    // Setup transfer data and transfer it
    Transfer.Identifier   := RcRPS0.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Index * Sizeof(DWord);
    Saa7146aWriteToDma(CardHandle, Transfer);              // Transfer data
    // Now make sure the RPS program is started
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsAddr0,
      DWord(DWord(RcRPS0.PhysicalAddress.LowPart)));       // Set physical start of RPS program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsPage0,
      0);                                                  // RPS program performs no explicit mem writes
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aRpsTov0 ,
      0);                                                  // Disable RPS timeouts
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps0On);                               // Start RPS0 program
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aDebiConfig,
      CSaa7146aDebiCfg16);                                 // Setup DEBI transfer type (MOTOROLA/INTEL does not care)
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aDebiPage,
      CSaa7146aDebiPageDisable);                           // No page check
    Saa7146aWriteToSaa7146aRegister(CardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1DebiEnable);                             // Enable DEBI transfers

  end;

begin
  Result := False;
  // Before we can activate FIFO3 transfer we have to set the DMA settings.
  // One of those settings is the target memory data will be saved to. For
  // this we use the DMA memory feature of the driver, since the memory it
  // allocates is always valid. Since the driver has allocated continuous
  // physical memory we don't have to use the MMU.

  // Note: If we are acting as a slave then we don't setup anything, but
  //       instead readout the DMA information

  if IsSlave then
  begin
    // As a slave we only need the identifier of the DMA buffer which was
    // used by the master.
    // The only way we can do is to check the size of such a buffer..
    Channel := GetMasterChannel(CardHandle);;
    if Channel <> $FFFF then
    begin
      Saa7146aGetDmaStatus(CardHandle, Channel, Info);
      BufferId := Channel;
      // Remote control always follows the master channel
      RcRPS0.Identifier     := BufferId + 1;
      RcRPS0Data.Identifier := BufferId + 2;
    end
    else
      Exit;        // The driver is used as slave but could not detect an active master
  end
  else
  begin
    // First release all already allocated memory
    Channel := 0;
    while Saa7146aReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    // Disable all interrupts
    Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIer, 0);

    // Allocate a DMA buffer which will hold the FIFO 3 data
    // No matter how we will be doing thing, we can expect a maximum of
    // 2 * 512 (VS) * 188 (HS) = 192512 == 96256 words == 48128 dwords
    // The VS changes every 512 packets. The HS is activated for every packet.
    // Every packet is 188 bytes. The factor '2' is for the 'odd' and 'even' fields
    // We can have more than 1 buffer here.
    repeat
      Succeeded := Saa7146aAllocateDma(CardHandle, (((PacketBuffers * CDvbPacketBufferSize * 2) div DMASIZE) + 1) * DMASIZE, Fifo3);
      // If we did not succeeded, so try using less memory. Strip down by 0.5/0.1 second at a time.
      if not(Succeeded) then
      begin
        if PacketBuffers > 49 then
          Dec(PacketBuffers, 25)
        else
          if PacketBuffers > 19 then
            Dec(PacketBuffers, 5);
      end;
    until Succeeded or (PacketBuffers < 20);
    if not Succeeded then
      Exit;
    BufferId := Fifo3.Identifier;
    // Allocate memory for remote control. It must follow the 'master' identifier
    // immediately
    if not Saa7146aAllocateDma(CardHandle, DMASIZE, RcRPS0) then
      Exit;
    if not Saa7146aAllocateDma(CardHandle, DMASIZE, RcRPS0Data) then
      Exit;
    if Dword(RcRPS0.Identifier) <> (BufferId + 1) then
      Exit;
    if Dword(RcRPS0Data.Identifier) <> (BufferId + 2) then
      Exit;

    // Setup DD1 (video) buffeing
    SetupDD1;
    // Setup remote control
    SetupRemoteControl(Rps0Program);
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
            <Initialize>  True if initialization allowed
            <TunerType>   Type of tuner to address
            <TunerReset>  OPx to use for reset
            <IsSlave>     False if forcing master mode
                          True for auto detection of master/slave
            <BufferTime>  Time (in ms) for buffer
            <Rps0Program> RPS0 program to use
                          -1   = remote control
                          0..3 = VSYNC output on GPIOx
  Returns : <Result>      True if success
                          False if not: most likely incorrect tuner type
            <IsSlave>     True indicates slave mode has been set
                          False indicates master mode has been set
            <BufferTime>  Number of buffers actually used
            <BufferId>    Buffer identifier (DMA) where data is stored

  Descript: Setup data throughput of driver (packet data)
  Notes   : This will setup the SAA7146A so data from the tuner is passed through
            the DD1 port. The DD1 port has the unprocessed data.
            Includes detection of master/slave.
 ------------------------------------------------------------------------------}
function DvbSetupBuffering(const CardHandle: THandle; const Initialize: Boolean; const TunerType: Byte; const TunerReset: Byte;
                           IsSlave: PBoolean; BufferTime: PWord; BufferId: PDword; Rps0Program: Integer): Boolean; stdcall;
var
  IrqHandling  : TSaa7146aIrqTransferBuffer;
  Ie           : Dword;
begin
  try
    Result := False;
    if not Assigned(IsSlave) then
      Exit;
    if not Assigned(BufferTime) then
      Exit;
    if not Assigned(BufferId) then
      Exit;

    if IsSlave^ then
      if GetMasterChannel(CardHandle) = $FFFF then
        IsSlave^ := False
      else
        IsSlave^ := True;

    if not IsSlave^ then
      if Initialize then
        if not InitializeCard(CardHandle, TunerType, TunerReset) then
          Exit;

    // Number of buffers is approximated using 20 ms per buffer as reference
    BufferTime^ := BufferTime^ div 20;
    if BufferTime^ > MaxPacketBuffers then
      BufferTime^ := MaxPacketBuffers;
    if BufferTime^ < 2 then
      BufferTime^ := 2;

    // Only setup the interface when we are the master
    if not IsSlave^ then
    begin
      // Disable all interrupts
      if not Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIer, 0) then
        Exit;
      // Set initial settings of dual D1 interface
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aDd1Init,                             // Register to write
               CSaa7146aDd1LlcInput or                       // LLC input
               CSaa7146aDd1SioInput or                       // HS/VS inputs
               CSaa7146aDd1PvoDirect or                      // Direct polarity VS
               CSaa7146aDd1PhoDirect or                      // Direct polarity HS
               CSaa7146aDd1Sync6 or                          // Sync edge selection, HS rising, VS both frame, Field direct
               CSaa7146aDd1FidesEdgesIrq) then               // Field both edges interrupt
        Exit;
      // Above registers need an upload to activate the setting
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc2,                                 // Register to write
               CSaa7146aMc2UpldDd1Initial) then              // Upload initial settings
        Exit;
      // Set video stream handling of port
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aDd1Data,                             // Register to write
               CSaa7146aDd1VidInput or                       // Port and PXQ inputs
               CSaa7146aDd1Y8cNoSavEav or                    // No SAV/EAV data
               CSaa7146aDd1PfidDirect) then                  // Direct polarity change
        Exit;
      // Above registers need an upload to activate the setting
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc2,                                 // Register to write
               CSaa7146aMc2UpldDd1A) then                    // Upload port A settings
        Exit;
      // Above registers need an upload to activate the setting
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc2,                                 // Register to write
               CSaa7146aMc2UpldDd1B) then                    // Upload port B settings
        Exit;
      // Set BRS control (we use the data from the D1 ports directly (unprocessed) so we need
      // to use the BRS and consequently FIFO3/DMA3
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aBrsCtrl,                             // Register to write
               CSaa7146aDd1BrsStreamInB or                   // Video data stream from port B
               CSaa7146aDd1BrsSyncInB or                     // Sync signals from port B
               (0 shl CSaa7146aDd1BrsVOffsetShift) or        // Vertical offset
               CSaa7146aDd1BrsVScale1 or                     // Vertical scale 1:1
               (0 shl CSaa7146aDd1BrsHOffsetShift) or        // Horizontal offset
               CSaa7146aDd1BrsHScale1 or                     // Horizontal scale 1:1
               CSaa7146aDd1BrsFormatYuv) then                // YUV 4:2:2 format
        Exit;

      // Above registers need an upload to activate the setting
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc2,                                 // Register to write
               CSaa7146aMc2UpldBrs) then                     // Upload BRS registers
        Exit;

      // We have setup the basic stuff.
      // When the DD1 is activated (at this point it is NOT), data will be
      // acquired and stored in FIFO3. At some point the data has to be transferred
      // from this FIFO to the computers memory. This is done with DMA3.
      // If we activate the DD1 port now, we should 'see' FIFO3 filling and
      // generating (error) signals when it has been filled (without it being
      // able to transfer it any further).

      // Enable the DD1 hardware.
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc1,                                 // Register to write
               CSaa7146aMc1Dd1Enable) then                   // Enable real time video ports (DD1)
        Exit;
    end;

    // Setup buffering/addresses
    if not BufferingSetup(CardHandle, IsSlave^, BufferTime^, BufferId^, Rps0Program) then
      Exit;

    // Only continue to setup the interface when we are the master
    if not IsSlave^ then
    begin
      // Setup distance between start addresses for consecutive lines of a field
      // If '0' data stays in first page (4096) bytes of selected memory
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aPitch3,                              // Register to write
               CDvbPacketHSync) then
        Exit;

      // Setup field/lines/bytes
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aNumLines3,                           // Register to write
               (CDvbPacketVSync shl 16) or CDvbPacketHSync) then
        Exit;

      // For more effective transfers we can extend the burst/threshold at which data is
      // transferred.
      if not Saa7146aWriteWithMaskToSaa7146aRegister(CardHandle,
               CSaa7146aPciBtV,
               $001F0000,
  //             $00130000) then                               // Burst 16 dwords and 32 dwords threshold
               $00170000) then                               // Burst 32 dwords and 32 dwords threshold
        Exit;

      // Above registers need an upload to activate the setting
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc2,                                 // Register to write
               CSaa7146aMc2UpldDma3) then                    // Upload DMA3 settings
        Exit;

      // Enable FIFO3 transfers.
      if not Saa7146aWriteToSaa7146aRegister(CardHandle,
               CSaa7146aMc1,                                 // Register to write
               CSaa7146aMc1TrVideo3Enable) then              // Enable transfers video 3
        Exit;
    end;

    // Only continue to setup the interface when we are the master
    if not IsSlave^ then
    begin
      // For recording purposes or additional buffering we setup an interrupt.
      // For now just record interrupts, nothing more.
      with IrqHandling.Information do
      begin
        Irqs                 := 0;
        IrqsWhenActive       := 0;
        IrqBufferingIsActive := True;
        UseNotificationEvent := True;
        UseSignaling         := False;
        UseFifo              := False;
        IrqAutoDisable       := False;
      end;
      // Depending on if the RPS program is assisting in buffering setup the correct
      // interrupt source
      IrqHandling.Identifier := CSaa7146aIrqRpsI1;
      Saa7146aWriteToIrqHandling(CardHandle, IrqHandling);
      // Enable interrupt
      if not Saa7146aReadFromSaa7146aRegister(CardHandle, CSaa7146aIer, Ie) then
        Exit;
      Ie := Ie or (1 shl CSaa7146aIrqRpsI1);
      if not Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIer, Ie) then
        Exit;
      // Clear all pending interrupts
      if not Saa7146aWriteToSaa7146aRegister(CardHandle, CSaa7146aIsr, $FFFFFFFF) then
        Exit;
    end;
    Result := True;
  except
    Result := False;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
  Returns : <Result>      True if new remote control data available
            <DeviceId>    Device
            <Command>     Command

  Descript: Get remote control data
  Notes   :
 ------------------------------------------------------------------------------}
function DvbGetRemoteControl(const CardHandle: THandle; DeviceId: PByte; Command: PByte): Boolean; stdcall;
var
  TransferBuffer : array[0..2] of Dword;
  InfraredData   : array[0..2] of Byte;
  Transfer       : TSaa7146aTransferBuffer;
  Key1           : Byte;
  Key2           : Byte;
  RepeatCode     : Boolean;
  Device         : Byte;
  Loop           : Integer;
  FirstKey       : Boolean;
  SecondKey      : Boolean;
  HasDevice      : Boolean;
  Valid          : Boolean;
begin
  try
    Result := False;
    if not Assigned(DeviceId) then
      Exit;
    if not Assigned(Command) then
      Exit;
    // Setup transfer data and transfer it
    Transfer.Identifier := RcRPS0Data.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Sizeof(TransferBuffer);      // Transfer data
    Saa7146aReadFromDma(CardHandle, Transfer);
    // We have received three data items from the remote. Now seperate them.
    RepeatCode := False;
    Key1 := 0;
    Key2 := 0;
    Device := 0;
    FirstKey  := False;
    SecondKey := False;
    HasDevice := False;
    Valid     := True;
    InfraredData[0] := Swap(TransferBuffer[0]) and $FF;
    InfraredData[1] := Swap(TransferBuffer[1]) and $FF;
    InfraredData[2] := Swap(TransferBuffer[2]) and $FF;
    // Scan the data
    for Loop := 0 to 2 do
    begin
      if InfraredData[Loop] < $40 then
      begin
        if HasDevice then                                    // Check for more than 1 entry == invalid
          Valid := False;
        HasDevice := True;
        RepeatCode := (InfraredData[Loop] and $20) <> 0;
        if RepeatCode then
          Device := InfraredData[Loop] - $20
        else
          Device := InfraredData[Loop];
      end
      else
        if InfraredData[Loop] < $80 then
        begin
          if FirstKey then                                   // Check for more than 1 entry == invalid
            Valid := False;
          FirstKey := True;
          Key1 := InfraredData[Loop] - $40;
          Key1 := Key1 and $7F;
        end
        else
          begin
            if SecondKey then                                // Check for more than 1 entry == invalid
              Valid := False;
            SecondKey := True;
            Key2 := InfraredData[Loop] - $40;
            Key2 := Key2 and $7F;
          end;
    end;
    // We identify a valid key when both keycodes are the same
    if (Key1 = Key2) and
        Valid then
    begin
      // If the repeat code has changed or the keycode has changed then a key is pressed
      if (RcRepeat <> RepeatCode) or
         (RcDevice <> Device) then
      begin
        // New key (or repeat)
        RcKey1 := Key1;
        RcKey2 := Key2;
        RcDevice := Device;
        RcRepeat := RepeatCode;
        // Return detected data
        Command^ := RcKey1;
        DeviceId^  := RcDevice;
        Result  := True;
      end
      else
        Result := False;
    end
    else
      Result := False;
  except
    Result := false;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
begin
  //
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  //
end;



initialization
  Initialize;


finalization
  Finalize;
end.
