{******************************************************************************}
{ FileName............: DvbDirectShow2                                         }
{ Project.............: DVB-SB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.08                                                   }
{------------------------------------------------------------------------------}
{  DirectShow interface                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Graph is also added to the Running Object Table so it can be connected to    }
{ when running (typically for debugging purposes)                              }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  1.00   20031113 - Initial release                                           }
{  1.01   20040213 - Added 'NoRender' parameter                                }
{  1.02   20040218 - Cleanup: might have introduced spurious runtime error at  }
{                    application ending                                        }
{                  - External access to graph (AddToRot) made variable         }
{                  - LogFile handle reset prior to destruction                 }
{  1.05   20040320 - Added <MajorDvbPsi> DirectShow filter support             }
{                    (V2.01/V2.02)                                             }
{                  - All <DvbFilter> calls are through this unit               }
{                  - Added windowless mode                                     }
{                  - If no .GRF file given the graph is created manually       }
{  1.06   20040605 - Additional Try/Except added                               }
{  1.07   20040815 - ServiceId in <..GetProgramInfo> added                     }
{                  - Added <DirectShowResize>                                  }
{  1.08   20040828 - Added CAT callback                                        }
{                  - Added PAT callback                                        }
{                  - Removed support for PsiFilter (too much differences)      }                                           
{                                                                              }
{ Note: Windowless mode (SetPosition...) not correct                           ]
{******************************************************************************}
unit DvbDirectShow2;

interface
uses
  Controls,
  DirectShow9,
  DvbFilter,
  ExtCtrls,
  SysUtils,
  Windows;

const
// From newer
  // Note that not all combinations are possible (eg. VRM with CaptureGraph)
  // Miscellaneous
  CDirectShowMethodSpecials                = $FF000000;    // Special options
  CDirectShowMethodDoNotRender             = $80000000;    // Do not render graph
  CDirectShowMethodDoNotAddToRot           = $40000000;    // Do not add to ROT
  CDirectShowMethodRenderAllPins           = $20000000;    // Render all pins
  CDirectShowMethodRenderAllAudioVideo     = $10000000;    // Render all audio/video pins
  CDirectShowMethodNativeVideoSize         = $01000000;    // Keep native video size
  CDirectShowMethodLogToFile               = $02000000;    // Use log file
  CDirectShowMethodRenderEx                = $04000000;    // Use RenderEx first
  // VMR
  CDirectShowMethodVmr                     = $0000F000;    // VMR options
  CDirectShowMethodVideoRenderer           = $00000000;    // Do not use VMR
  CDirectShowMethodVideoRenderer7          = $00007000;    // Use VMR7
  CDirectShowMethodVideoRenderer9          = $00009000;    // Use VMR9
  // VMR specials
  CDirectShowMethodVmrSpecials             = $00000F00;    // VMR special options
  CDirectShowMethodVideoRendererWindowless = $00000100;    // Windowless mode
  CDirectShowMethodVideoRendererFlipX      = $00000200;    // Swap/Flip/Mirror X
  CDirectShowMethodVideoRendererFlipY      = $00000400;    // Swap/Flip/Mirror Y

  // Graph (builder)
  CDirectShowMethodGraph                   = $000000F0;    // Graph
  CDirectShowMethodFilterGraph2            = $00000000;    // Use FilterGraph2
  CDirectShowMethodGraphBuilder            = $00000010;    // Use GraphBuilder
  CDirectShowMethodFilterGraph             = $00000020;    // Use FilterGraph
  CDirectShowMethodDvdGraphBuilder         = $00000030;    // Use DvdGraphBuilder
  // Additional builders
  CDirectShowMethodBuilder                 = $0000000F;    // Builder
  CDirectShowMethodNoBuilders              = $00000000;    // No additional builders
  CDirectShowMethodCaptureGraphBuilder     = $00000001;    // Use CaptureGraphBuilder
  CDirectShowMethodCaptureGraphBuilder2    = $00000002;    // Use CaptureGraphBuilder2

{------------------------------------------------------------------------------
  Descript: DirectShow basic functions
 ------------------------------------------------------------------------------}
  function  DirectShowStart(Owner           : THandle;
                            GraphicsFile    : string;
                            DoNotRender     : Boolean;
                            AllowAddToRot   : Boolean;
                            WindowlessMode  : Boolean;
                            var ErrorMessage: string;
                            LogFile         : THandle): Boolean;
  procedure DirectShowStop;
  procedure DirectShowFullScreenMode(FullScreen: Boolean);
  procedure DirectShowResize;


// From newer
  procedure DirectShowSetVideo(Video: HWND);
  procedure DirectShowGetSize(var Width: Integer; var Height: Integer);
  function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
  function  DirectShowGetFilterName(Index: Integer): string;
  function  DirectShowWindowless: Boolean;
  function  DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;


{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer/universal source filter functions
 ------------------------------------------------------------------------------}
  procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
  procedure DirectShowUniversalSourceSendData     (Buffer: PChar; BufferLength: Integer);

implementation
uses
  ActiveX, Dialogs;

const
{------------------------------------------------------------------------------
  Descript: DirectShow identifiers
 ------------------------------------------------------------------------------}
  IID_IUSRCSet                  = '{AE1A2884-540E-4077-B1AB-67A34A72299F}';
  T_IID_IUSRCSet        : TGUID = IID_IUSRCSet;
  CLSID_IUSRCSet        : TGUID = '{FD501041-8888-1111-9153-00AB00577DA2}';
  IID_IPersistStream    : TGUID = '{00000109-0000-0000-C000-000000000046}';

  CLSID_MPEG2Demultiplexer: TGUID = (D1:$afb6c280;D2:$2c41;D3:$11d3;D4:($8a,$60,$00,$00,$f8,$1e,$0e,$4a));

type
{------------------------------------------------------------------------------
  Descript: Universal Source filter interface
 ------------------------------------------------------------------------------}
  IUSRCSet = interface(IUnknown)
  [IID_IUSRCSet]
    function WSetMediaType(Pmt   : PAMMEDIATYPE): HRESULT; stdcall;
    function WSetBufSize  (Size  : Dword): HRESULT; stdcall;
    function WSendSample  (Buffer: PChar; BufferLength: Dword): HRESULT; stdcall;
  end;

{------------------------------------------------------------------------------
  Descript: The DirectShow object
 ------------------------------------------------------------------------------}
  TDirectShowGraph = class(TObject)
    FOwner                       : THandle;
    FRotEntry                    : LongInt;
    FGraphBuilder                : IGraphbuilder;
    FUniversalSourceInterface    : IUSRCSet;
    FUniversalSourceFilter       : IBaseFilter;
    FMpeg2DeMultiplexerFilter    : IBaseFilter;
    FMajorDvbPsiFilter           : IBaseFilter;
    FVideoMixingRendererControl  : IVMRWindowlessControl9; // IVMRWindowlessControl
    FMediaControl                : IMediaControl;
    FVideoWindow                 : IVideoWindow;
    FBasicVideo                  : IBasicVideo;
    FVideoPin                    : IPin;
    FAudioPin                    : IPin;
    FVideoStreamMap              : IMPEG2PIDMap;
    FAudioStreamMap              : IMPEG2PIDMap;
    FActiveAudioPid              : Integer;
    FActiveVideoPid              : Integer;
    FUseMajorDvbPsi              : Boolean;
    constructor Create(Owner: THandle);
    destructor  Destroy; override;
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  DirectShowGraph: TDirectShowGraph;


{------------------------------------------------------------------------------
  Params  : <UnknownGraph>  Graph to add to ROT
  Returns : <Result>        Result (S_OK if success)
            <RotEntry>      Assigned number in ROT

  Descript: Add graph to Running Object Table so we can connect to it when
            it is running.
  Notes   :
 ------------------------------------------------------------------------------}
function AddToRot(UnknownGraph: IUnknown; var RotEntry: LongInt): HRESULT;
var
  Moniker: IMoniker;
  Rot    : IRunningObjectTable;
  Wsz    : WideString;
  Wsz2   : WideString;
  Hr     : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, ROT);
    if Failed(Hr) then
    begin
      Result := E_FAIL;
      Exit;
    end;
    Wsz  := format('FilterGraph %p pid %8.8x', [Pointer(UnknownGraph), GetCurrentProcessId]);
    Wsz2 := '!';
    Hr := CreateItemMoniker(@Wsz2[1], @Wsz[1], Moniker);
    if Succeeded(Hr) then
    begin
      Hr := ROT.Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, UnknownGraph,
              Moniker, RotEntry);
    end;
    Result := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <RotEntry>      Assigned number in ROT
  Returns : -

  Descript: Remove graph from Running Object Table
  Notes   :
 ------------------------------------------------------------------------------}
procedure RemoveFromRot(RotEntry: LongInt);
var
  Rot: IRunningObjectTable ;
  Hr : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, Rot);
    if Succeeded(Hr) then
      Rot.Revoke(RotEntry);
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <pGraph>    Pointer to graph builder object
            <FileName>  Name of graph file
  Returns : -

  Descript: Load a DirectShow graph (.GRF) file
  Notes   :
 ------------------------------------------------------------------------------}
function LoadGraphFile(pGraph: IGraphBuilder; FileName: string): HRESULT;
var
  PStorage      : IStorage;
  PPersistStream: IPersistStream;
  PStream       : IStream;
  WideName      : array[0..1023] of WideChar;
  Hr            : HRESULT;
begin
  try
    PStorage       := nil;
    PPersistStream := nil;
    PStream        := nil;
    StringToWideChar(FileName, WideName, SizeOf(WideName));
    hr := StgIsStorageFile(@WideName);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    hr := StgOpenStorage(@WideName, nil, STGM_TRANSACTED or STGM_READ or STGM_SHARE_DENY_WRITE,
            nil, 0, PStorage);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    Hr := pGraph.QueryInterface(IID_IPersistStream, PPersistStream);
    if not Failed(Hr) then
    begin
      Hr := PStorage.OpenStream('ActiveMovieGraph', nil,
              STGM_READ or STGM_SHARE_EXCLUSIVE, 0, PStream);
      if not Failed(Hr) then
      begin
        Hr := PPersistStream.Load(PStream);
        PStream := nil;
      end;
      PPersistStream := nil;
    end;
    PStorage := nil;
    Result  := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Owner>           Handle to window for video
            <GraphicsFile>    .GRF file to use
                              If empty the graph is manually created
            <DoNotRender>     TRUE will not render audio/video pins: the graph is
                              not automatically 'completed'
            <AllowAddToRot>   TRUE will add graph toi table so it can be remotely
                              connected to
            <WindowlessMode>  TREU enables windowless mode
            <LogFile>         Handle for log file
  Returns : <Result>          TRUE if success, FALSE if failure
            <ErrorMessage>    Error message (if any)

  Descript: Setup direct show graph
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowStart(Owner           : THandle;
                          GraphicsFile    : string;
                          DoNotRender     : Boolean;
                          AllowAddToRot   : Boolean;
                          WindowlessMode  : Boolean;
                          var ErrorMessage: string;
                          LogFile         : THandle): Boolean;
var
  Hr                          : HRESULT;
  Pins                        : IPin;
  PinsIn                      : IPin;
  PinsOut                     : IPin;
  PinsEnum                    : IEnumPins;
  PinsInfo                    : TPinInfo;
  FilterEnum                  : IEnumFilters;
  FilterOut                   : IBaseFilter;
  FilterInfo                  : TFilterInfo;
  VideoMixingRendererInterface: IVMRFilterConfig9;
  VideoMixingRendererFilter   : IBaseFilter;
  MPEG2DemultiplexerInterface : IMPEG2Demultiplexer;
  MediaType                   : AM_MEDIA_TYPE;
  Width                       : LongInt;
  Height                      : LongInt;
  Dummy                       : LongInt;
  SourceRect                  : TRect;
  DestinationRect             : TRect;
begin
  try
    // Make sure a previously started DirectShow interface is stopped and released
    DirectShowStop;
    ErrorMessage := '';
    Result := False;
    DirectShowGraph := TDirectShowGraph.Create(Owner);
    try
      // Create the filter graph
      Hr := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC, IID_IGraphBuilder, DirectShowGraph.FGraphBuilder);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter graph creation error.';
        Exit;
      end;
      // Set the log file if requested
      // If this failrs we continu normal operation ...
      DirectShowGraph.FGraphBuilder.SetLogFile(LogFile);

      // Add to ROT if requested
      if AllowAddToRot then
      begin
        Hr := AddToRot(DirectShowGraph.FGraphBuilder, DirectShowGraph.FRotEntry);
        if Failed(Hr) then
        begin
          DirectShowGraph.FRotEntry := -1;
  //        ErrorMessage := 'Filter graph could not be added to the ROT.';
  //        Exit;
        end;
      end;

      if WindowlessMode then
      begin
        {------------------------------------------------------------------------------
          Descript: Windowless mode
         ------------------------------------------------------------------------------}
        // For windowless mode we use the Video Mixing Renderer
        Hr := CoCreateInstance(CLSID_VideoMixingRenderer9, nil, CLSCTX_INPROC, IID_IBaseFilter, VideoMixingRendererFilter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not use the Video Mixing Renderer.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraphBuilder.AddFilter(VideoMixingRendererFilter, 'Video Mixing Renderer');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not add "Video Mixing Renderer" filter.';
          Exit;
        end;
        // Now get the interfaces to the filter
        Hr := VideoMixingRendererFilter.QueryInterface(IID_IVMRFilterConfig9, VideoMixingRendererInterface);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter "Video Mixing Renderer" interface not found.';
          Exit;
        end;
        // Set windowless mode
        Hr := VideoMixingRendererInterface.SetRenderingMode(VMRMode_Windowless);
        VideoMixingRendererInterface := nil;
        if not Failed(Hr) then
        begin
          // Get the control and set the window
          Hr := VideoMixingRendererFilter.QueryInterface(IID_IVMRWindowlessControl9, DirectShowGraph.FVideoMixingRendererControl);
          if not Failed(Hr) then
          begin
            Hr := DirectShowGraph.FVideoMixingRendererControl.SetVideoClippingWindow(Owner);
            if Failed(hr) then
              DirectShowGraph.FVideoMixingRendererControl := nil;
          end;
        end;
        VideoMixingRendererFilter := nil;
      end;

      // Load the file in the filter graph
      if FileExists(GraphicsFile) then
      begin
        {------------------------------------------------------------------------------
          Descript: Loading a graph file
         ------------------------------------------------------------------------------}
        Hr := LoadGraphFile(DirectShowGraph.FGraphBuilder, GraphicsFile);
        if Failed(Hr) then
        begin
          ErrorMessage := GraphicsFile + ' file not loaded (missing/invalid/incorrect).';
          Exit;
        end;

        // Enumerate the filters in the filter graph
        Hr := DirectShowGraph.FGraphBuilder.EnumFilters(FilterEnum);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter graph could not enumerate the filters.';
          Exit;
        end;

        // Go through the filters in the graph, identify them and record the
        // filters we require
        while FilterEnum.Next(1, FilterOut, nil) = S_OK do
        begin
          FilterOut.QueryFilterInfo(FilterInfo);
          if UpperCase(FilterInfo.achName) = 'MPEG-2 DEMULTIPLEXER' then
            DirectShowGraph.FMpeg2DemultiplexerFilter := FilterOut;
          if UpperCase(FilterInfo.achName) = 'UNIVERSAL SOURCE' then
            DirectShowGraph.FUniversalSourceFilter := FilterOut;
          if (UpperCase(FilterInfo.achName) = 'MAJORDVB PSI PARSER') then
            DirectShowGraph.FMajorDvbPsiFilter := FilterOut;
        end;
        FilterOut  := nil;
        FilterEnum := nil;
      end
      else
      begin
        {------------------------------------------------------------------------------
          Descript: Building a graph manually
         ------------------------------------------------------------------------------}
        // No graph filter file: create the graph manually (at least the major parts)
        // First the universal source
        Hr := CoCreateInstance(CLSID_IUSRCSet, nil, CLSCTX_INPROC, IID_IBaseFilter, DirectShowGraph.FUniversalSourceFilter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not use the Universal Source filter.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraphBuilder.AddFilter(DirectShowGraph.FUniversalSourceFilter, 'Universal Source');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not add "Universal Source" filter.';
          Exit;
        end;

        // Then the MPEG-2 demultiplexer
        Hr := CoCreateInstance(CLSID_MPEG2Demultiplexer, nil, CLSCTX_INPROC, IID_IBaseFilter, DirectShowGraph.FMPeg2DemultiplexerFilter);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not use the MPEG-2 Demultiplexer filter.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraphBuilder.AddFilter(DirectShowGraph.FMPeg2DemultiplexerFilter, 'MPEG-2 Demultiplexer');
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not add "MPEG-2 Demultiplexer" filter.';
          Exit;
        end;

        // Now we must create the output pins on the MPEG-2 demultiplexer
        // The video pin
        Hr := DirectShowGraph.FMPeg2DemultiplexerFilter.QueryInterface(IID_IMpeg2Demultiplexer, MPEG2DemultiplexerInterface);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter "MPEG-2 Demultiplexer" interface not found.';
          Exit;
        end;
        MediaType.MajorType := MEDIATYPE_VIDEO;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_VIDEO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Video', DirectShowGraph.FVideoPin);
        if Failed(Hr) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the video output pin for the MPEG-2 Demultiplexer.';
          Exit;
        end;

        // The audio pin
        MediaType.MajorType := MEDIATYPE_AUDIO;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_AUDIO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Audio', DirectShowGraph.FAudioPin);
        if Failed(Hr) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the audio output pin for the MPEG-2 Demultiplexer.';
          Exit;
        end;
        MPEG2DemultiplexerInterface := nil;


        // Connect the pins between Universal Source and the MPEG-2 demultiplexer
        PinsIn  := nil;
        PinsOut := nil;
        // Enumerate the pins
        Hr := DirectShowGraph.FUniversalSourceFilter.EnumPins(PinsEnum);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not enumerate the Universal Source pins.';
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          if UpperCase(PinsInfo.achName) = 'OUTPUT' then
            PinsOut := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        Hr := DirectShowGraph.FMPeg2DemultiplexerFilter.EnumPins(PinsEnum);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not enumerate the MPEG-2 Demultiplexer pins.';
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          if UpperCase(PinsInfo.achName) = 'MPEG-2 STREAM' then
            PinsIn := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        // Connect the pins
        if not Assigned(PinsIn) or not Assigned(PinsOut) then
        begin
          PinsIn  := nil;
          PinsOut := nil;
          ErrorMessage := 'Could not find correct pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
        Hr := DirectShowGraph.FGraphBuilder.Connect(PinsOut, PinsIn);
        PinsIn  := nil;
        PinsOut := nil;
        if Failed(Hr) then
        begin
          ErrorMessage := 'Could not connect pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
      end;

      // Check the essential filter parts
      if not Assigned(DirectShowGraph.FMpeg2DemultiplexerFilter) then
      begin
        ErrorMessage := 'Filter "MPEG-2 Demultiplexer" not found.';
        Exit;
      end;
      if not Assigned(DirectShowGraph.FUniversalSourceFilter) then
      begin
        ErrorMessage := 'Filter "Universal Source" not found.';
        Exit;
      end;

      // Now get the interfaces to the filters we need to call from our application
      Hr := DirectShowGraph.FUniversalSourceFilter.QueryInterface(T_IID_IUSRCSet, DirectShowGraph.FUniversalSourceInterface);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter "Universal Source" interface not found.';
        Exit;
      end;

      // Media controller filter
      Hr := DirectShowGraph.FGraphBuilder.QueryInterface(IID_IMediaControl, DirectShowGraph.FMediaControl);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter media control interface not found.';
        Exit;
      end;

      // Video window filter
      Hr := DirectShowGraph.FGraphBuilder.QueryInterface(IID_IVideoWindow, DirectShowGraph.FVideoWindow);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter video interface not found.';
        Exit;
      end;

      // Basic video window filter
      // Allows to set video properties
      Hr := DirectShowGraph.FGraphBuilder.QueryInterface(IID_IBasicVideo, DirectShowGraph.FBasicVideo);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter basic video interface not found.';
        Exit;
      end;

      if not Assigned(DirectShowGraph.FVideoPin) then
      begin
        Hr := DirectShowGraph.FMpeg2DemultiplexerFilter.FindPin('Video', DirectShowGraph.FVideoPin);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter demultiplexer video pin not found.';
          Exit;
        end;
      end;

      if not Assigned(DirectShowGraph.FAudioPin) then
      begin
        Hr := DirectShowGraph.FMpeg2DemultiplexerFilter.FindPin('Audio', DirectShowGraph.FAudioPin);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter demultiplexer audio pin not found.';
          Exit;
        end;
      end;


      // Render the filter if requested
      if not DoNotRender then
      begin
        DirectShowGraph.FGraphBuilder.Render(DirectShowGraph.FVideoPin);
        DirectShowGraph.FGraphBuilder.Render(DirectShowGraph.FAudioPin);
      end;

      if Assigned(DirectShowGraph.FMpeg2DemultiplexerFilter) then
      begin
        DirectShowGraph.FMpeg2DemultiplexerFilter.SetSyncSource(nil);
        DirectShowGraph.FGraphBuilder.SetDefaultSyncSource;
      end;

      Hr := DirectShowGraph.FVideoPin.QueryInterface(IID_IMPEG2PIDMap, DirectShowGraph.FVideoStreamMap);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter video stream interface not found.';
        Exit;
      end;

      Hr := DirectShowGraph.FAudioPin.QueryInterface(IID_IMPEG2PIDMap, DirectShowGraph.FAudioStreamMap);
      if Failed(Hr) then
      begin
        ErrorMessage := 'Filter audio stream interface not found.';
        Exit;
      end;

      if WindowlessMode then
      begin
        // Basic video settings Windowless mode
        Hr := DirectShowGraph.FVideoMixingRendererControl.GetNativeVideoSize(Width, Height, Dummy, Dummy);
        if not Failed(Hr) then
        begin
          // Get the window client area
          GetClientRect(Owner, DestinationRect);
          // Set the source rectangle
          SetRect(SourceRect, 0, 0, Width, Height);
          // Set the video position.
          DirectShowGraph.FVideoMixingRendererControl.SetVideoPosition(@SourceRect, @DestinationRect);
        end;
      end
      else
      begin
        Hr := DirectShowGraph.FVideoWindow.put_Owner(DirectShowGraph.FOwner);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set video handle.';
          Exit;
        end;
        Hr := DirectShowGraph.FVideoWindow.put_MessageDrain(DirectShowGraph.FOwner);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set messages handle.';
          Exit;
        end;
        Hr := DirectShowGraph.FVideoWindow.put_WindowStyle(WS_CHILD or WS_CLIPSIBLINGS or WS_CLIPCHILDREN);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set window style.';
          Exit;
        end;

        GetClientRect(Owner, DestinationRect);
        Hr := DirectShowGraph.FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        if Failed(Hr) then
        begin
          ErrorMessage := 'Filter could not set window size.';
          Exit;
        end;
      end;

      // Indicate success
      Result := True;
    finally
      // Make sure that at a failure all resources are freed
      if not Result then
        DirectShowStop;
    end;
  except
    Result := False;
    DirectShowStop;
    ShowMessage('DirectShow Start error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Owner>  Handle to window for video
  Returns : -

  Descript: Constructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TDirectShowGraph.Create(Owner: THandle);
begin
  inherited Create;
  FOwner                      := Owner;
  FRotEntry                   := -1;
  FGraphBuilder               := nil;
  FUniversalSourceInterface   := nil;
  FUniversalSourceFilter      := nil;
  FMpeg2DeMultiplexerFilter   := nil;
  FMajorDvbPsiFilter          := nil;
  FVideoMixingRendererControl := nil;
  FMediaControl               := nil;
  FVideoWindow                := nil;
  FBasicVideo                 := nil;
  FVideoPin                   := nil;
  FAudioPin                   := nil;
  FVideoStreamMap             := nil;
  FAudioStreamMap             := nil;
  FActiveAudioPid             := -1;
  FActiveVideoPid             := -1;
  FUseMajorDvbPsi             := False;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
destructor TDirectShowGraph.Destroy;
begin
  if Assigned(FGraphBuilder) then
  begin
    try
      FGraphBuilder.SetLogFile(0);
      if FRotEntry <> -1 then
        RemoveFromRot(FRotEntry);
      if Assigned(FMediaControl) then
        FMediaControl.Stop;
      // Important to stop the message drain, otherwise errors occur
      if Assigned(FVideoWindow) then
      begin
        FVideoWindow.put_Owner(0);
        FVideoWindow.put_MessageDrain(0);
      end;

      FGraphBuilder               := nil;
      FUniversalSourceInterface   := nil;
      FUniversalSourceFilter      := nil;
      FMpeg2DeMultiplexerFilter   := nil;
      FMajorDvbPsiFilter          := nil;
      FVideoMixingRendererControl := nil;
      FMediaControl               := nil;
      FVideoWindow                := nil;
      FBasicVideo                 := nil;
      FVideoPin                   := nil;
      FAudioPin                   := nil;
      FVideoStreamMap             := nil;
      FAudioStreamMap             := nil;
    except
    end;
  end;
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Stop DirectShow interface
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowStop;
begin
  if Assigned(DirectShowGraph) then
    FreeAndNil(DirectShowGraph);
end;


{------------------------------------------------------------------------------
  Params  : <FullScreen>  True to set full screen mode
                          False to restore original mode
  Returns : -

  Descript: Set full screen mode
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowFullScreenMode(FullScreen: Boolean);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    if DirectShowGraph.FVideoWindow <> nil then
      DirectShowGraph.FVideoWindow.put_FullScreenMode(FullScreen);
  except
    DirectShowStop;
    ShowMessage('DirectShow FullScreenMode error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Resize to full client area
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowResize;
var
  DestinationRect: TRect;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    if DirectShowGraph.FVideoWindow <> nil then
    begin
      GetClientRect(DirectShowGraph.FOwner, DestinationRect);
      DirectShowGraph.FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
    end;
  except
    DirectShowStop;
    ShowMessage('DirectShow Resize error');
  end;
end;



{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <VideoPid>  Video PID
            <AudioPid>  Audio PID
  Returns : -

  Descript: Set new PIDs to use for MPEG2 demultiplexer
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
var
  MapPid: Cardinal;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    // Stop media control
    DirectShowGraph.FMediaControl.Stop;

    // Remove current active PIDs
    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      DirectShowGraph.FVideoStreamMap.UnmapPID(1, @MapPid);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      DirectShowGraph.FAudioStreamMap.UnmapPID(1, @MapPid);
    end;

    // set new active PIDs
    DirectShowGraph.FActiveVideoPid := VideoPid;
    DirectShowGraph.FActiveAudioPid := AudioPid;

    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      DirectShowGraph.FVideoStreamMap.MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      DirectShowGraph.FAudioStreamMap.MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;

    // Start media control
    DirectShowGraph.FMediaControl.Run;
  except
    DirectShowStop;
    ShowMessage('DirectShow Mpeg2DemultiplexerSetNewPids error');
  end;
end;


{------------------------------------------------------------------------------
  Descript: DirectShow Universal Source filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <Buffer>        Pointer to data
            <BufferLength>  Length of valid data in buffer
  Returns : -

  Descript: Send data to Universal Source filter
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowUniversalSourceSendData(Buffer: PChar; BufferLength: Integer);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    DirectShowGraph.FUniversalSourceInterface.WSendSample(Buffer, BufferLength);
  except
    DirectShowStop;
    ShowMessage('DirectShow UniversalSourceSendData');
  end;
end;


{------------------------------------------------------------------------------
  Descript: <DvbFilter> functions: Either interanl function called or
            <DvbMajorPsi> DirectShow filter ios called
  Note    : For descriptions of functions/procedures see <DvbFilter>
 ------------------------------------------------------------------------------}
function DvbGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
begin
  Result := DvbFilter.DvbFilterGetPid(StreamPacketData, Pid);
end;


function  DvbSetSignallingCallback     (Pid: Word;  hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallback(Pid, hWnd, Msg);
end;


function  DvbSetSignallingCallbackPat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackCat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackCat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackPmt  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPmt  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackEvent( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackEvent(hWnd, Msg);
end;


function  DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPidFilter(Pid, Filter);
end;


function  DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := DvbFilter.DvbFilterGetPidFilter(Pid);
end;


function  DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPsiFilter(Psi, Filter);
end;


function  DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := DvbFilter.DvbFilterGetPsiFilter(Psi);
end;


function  DvbGetProgramInfo(ProgramNumber: Byte;
                            var ServiceId: Word;
                            var PmtPid: Word;
                            var PcrPid: Word;
                            var VideoPids: TDvbPids;
                            var AudioPids: TDvbPids;
                            var TeletextPids: TDvbPids;
                            var SubtitlePids: TDvbPids;
                            var AudioLanguages: TDvbLanguages;
                            var SubtitleLanguages: TDvbLanguages;
                            var EcmPids: TDvbPids;
                            var CaIds: TDvbPids;
                            var ProgramName: string): Boolean;
begin
  Result := DvbFilter.DvbFilterGetProgramInfo(ProgramNumber, ServiceId,
              PmtPid, PcrPid, VideoPids, AudioPids,
              TeletextPids, SubtitlePids, AudioLanguages,
              SubtitleLanguages, EcmPids, CaIds, ProgramName);
end;


function  DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
begin
  Result := DvbFilter.DvbFilterGetEventInfo(ProgramNumber, ServiceId, Present, Events);
end;


function  DvbGetNumberOfPrograms: Byte;
begin
  Result := DvbFilter.DvbFilterGetNumberOfPrograms;
end;


function  DvbGetErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetErrors;
end;


function  DvbGetPacketSyncErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetPacketSyncErrors;
end;


procedure DvbResetErrors;
begin
  DvbFilter.DvbFilterResetErrors;
end;


procedure DvbCreateTables;
begin
  DvbFilter.DvbFilterCreateTables;
end;



{------------------------------------------------------------------------------
  Params  : -
  Returns : <Width>  Width of native video
            <Height> Height of native video

  Descript: Get native video size
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowGetSize(var Width: Integer; var Height: Integer);
begin
  //
end;

function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
begin
  Result := False;
end;


{------------------------------------------------------------------------------
  Params  : <Index>  Filter index (0 = first)
  Returns : <Result> Name of filter (empty if no filter at index)

  Descript: Get filter name
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowGetFilterName(Index: Integer): string;
begin
  Result := '';
end;


{------------------------------------------------------------------------------
  Params  : <Parent>      Handle of parent to create property page in
            <FilterName>  Name of filter
            <QueryOnly>   True if only to check for property page (not to show)
  Returns : <Result>      True if property page available

  Descript: Show property page of filter
  Notes   :
 ------------------------------------------------------------------------------}
function DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;
begin
  Result := False;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if 'windowless' mode

  Descript: Get windowless mode (VMR)
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowWindowless: Boolean;
begin
  Result := False;
end;


{------------------------------------------------------------------------------
  Params  : <Video>  Video window
  Returns : -

  Descript: Set video window
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowSetVideo(Video: HWND);
begin
end;






{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}
procedure InitializeUnit;
begin
  DirectShowGraph := nil;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization
  Notes   :
 ------------------------------------------------------------------------------}
procedure FinalizeUnit;
begin
  DirectShowStop;
end;


initialization
  InitializeUnit;

finalization
  FinalizeUnit;
end.
