{******************************************************************************}
{ FileName............: DvbDirectShow                                          }
{ Project.............: DVB-SB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.11                                                   }
{------------------------------------------------------------------------------}
{  DirectShow interface                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Graph is also added to the Running Object Table so it can be connected to    }
{ when running (typically for debugging purposes)                              }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  1.00   20031113 - Initial release                                           }
{  1.01   20040213 - Added 'NoRender' parameter                                }
{  1.02   20040218 - Cleanup: might have introduced spurious runtime error at  }
{                    application ending                                        }
{                  - External access to graph (AddToRot) made variable         }
{                  - LogFile handle reset prior to destruction                 }
{  1.05   20040320 - Added <MajorDvbPsi> DirectShow filter support             }
{                    (V2.01/V2.02)                                             }
{                  - All <DvbFilter> calls are through this unit               }
{                  - Added windowless mode                                     }
{                  - If no .GRF file given the graph is created manually       }
{  1.06   20040605 - Additional Try/Except added                               }
{  1.07   20040815 - ServiceId in <..GetProgramInfo> added                     }
{                  - Added <DirectShowResize>                                  }
{  1.08   20040828 - Added CAT callback                                        }
{                  - Added PAT callback                                        }
{                  - Removed support for PsiFilter (too much differences)      }
{  1.09   20040920 - VMR9 instead of windowless mode plus some additional VRM  }
{                    stuff (eg. 'OSD')                                         }
{  1.10   20041003 - Retrieval if used filters added                           }
{                  - Added Property page detection/showing                     }
{  1.11   20041021 - Switchable method (for debugging mainly)                  }
{                    Complete rebuild                                          }
{******************************************************************************}
unit DvbDirectShow;

interface
uses
  Controls,
  DirectShow9,
//  DxErr9,
  DvbFilter,
  ExtCtrls,
  Graphics,
  SysUtils,
  Windows;


const
  // Note that not all combinations are possible (eg. VRM with CaptureGraph)
  // Miscellaneous
  CDirectShowMethodSpecials                = $FF000000;    // Special options
  CDirectShowMethodDoNotRender             = $80000000;    // Do not render graph
  CDirectShowMethodDoNotAddToRot           = $40000000;    // Do not add to ROT
  CDirectShowMethodRenderAllPins           = $20000000;    // Render all pins
  CDirectShowMethodRenderAllAudioVideo     = $10000000;    // Render all audio/video pins
  CDirectShowMethodNativeVideoSize         = $01000000;    // Keep native video size
  CDirectShowMethodLogToFile               = $02000000;    // Use log file
  CDirectShowMethodRenderEx                = $04000000;    // Use RenderEx first
  CDirectShowMethodAlternative             = $08000000;    // Use alternative (= older) DirectShow methods
  // VMR
  CDirectShowMethodVmr                     = $0000F000;    // VMR options
  CDirectShowMethodVideoRenderer           = $00000000;    // Do not use VMR
  CDirectShowMethodVideoRenderer7          = $00007000;    // Use VMR7
  CDirectShowMethodVideoRenderer9          = $00009000;    // Use VMR9
  // VMR specials
  CDirectShowMethodVmrSpecials             = $00000F00;    // VMR special options
  CDirectShowMethodVideoRendererWindowless = $00000100;    // Windowless mode
  CDirectShowMethodVideoRendererFlipX      = $00000200;    // Swap/Flip/Mirror X
  CDirectShowMethodVideoRendererFlipY      = $00000400;    // Swap/Flip/Mirror Y

  // Graph (builder)
  CDirectShowMethodGraph                   = $000000F0;    // Graph
  CDirectShowMethodFilterGraph2            = $00000000;    // Use FilterGraph2
  CDirectShowMethodGraphBuilder            = $00000010;    // Use GraphBuilder
  CDirectShowMethodFilterGraph             = $00000020;    // Use FilterGraph
  CDirectShowMethodDvdGraphBuilder         = $00000030;    // Use DvdGraphBuilder
  // Additional builders
  CDirectShowMethodBuilder                 = $0000000F;    // Builder
  CDirectShowMethodNoBuilders              = $00000000;    // No additional builders
  CDirectShowMethodCaptureGraphBuilder     = $00000001;    // Use CaptureGraphBuilder
  CDirectShowMethodCaptureGraphBuilder2    = $00000002;    // Use CaptureGraphBuilder2


{------------------------------------------------------------------------------
  Descript: DirectShow basic functions
 ------------------------------------------------------------------------------}
  function  DirectShowStart(Owner           : HWND;
                            GraphicsFile    : string;
                            Options         : Dword;       // Bitwise options
                            var ErrorMessage: string): Boolean;
  procedure DirectShowStop;
  procedure DirectShowFullScreenMode(FullScreen: Boolean);
  procedure DirectShowSetVideo(Video: HWND);
  procedure DirectShowResize;
  procedure DirectShowGetSize(var Width: Integer; var Height: Integer);
  function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
  function  DirectShowGetFilterName(Index: Integer): string;
  function  DirectShowWindowless: Boolean;
  function  DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;

{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer/universal source filter functions
 ------------------------------------------------------------------------------}
  procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
  procedure DirectShowUniversalSourceSendData     (Buffer: PChar; BufferLength: Integer);

{------------------------------------------------------------------------------
  Descript: DvbFilter functions
  Note    : These functions are passed onto <DvbFilter>
 ------------------------------------------------------------------------------}
  function  DvbGetPid                    (StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
  function  DvbSetSignallingCallback     (Pid: Word; hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackPat  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackCat  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackPmt  (hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
  function  DvbSetPidFilter              (Pid: Word; Filter: TDvbPacketFilter): Boolean;
  function  DvbGetPidFilter              (Pid: Word): TDvbPacketFilter;
  function  DvbSetPsiFilter              (Psi: Byte; Filter: TDvbSectionFilter): Boolean;
  function  DvbGetPsiFilter              (Psi: Byte): TDvbSectionFilter;
  function  DvbGetProgramInfo(ProgramNumber: Byte;
                              var ServiceId: Word;
                              var PmtPid: Word;
                              var PcrPid: Word;
                              var VideoPids: TDvbPids;
                              var AudioPids: TDvbPids;
                              var TeletextPids: TDvbPids;
                              var SubtitlePids: TDvbPids;
                              var AudioLanguages: TDvbLanguages;
                              var SubtitleLanguages: TDvbLanguages;
                              var EcmPids: TDvbPids;
                              var CaIds: TDvbPids;
                              var ProgramName: string): Boolean;
  function  DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
  function  DvbGetNumberOfPrograms: Byte;
  function  DvbGetErrors: Word;
  function  DvbGetPacketSyncErrors: Word;
  procedure DvbResetErrors;
  procedure DvbCreateTables;


implementation
uses
  ActiveX,
  Dialogs,
  Forms;

const
{------------------------------------------------------------------------------
  Descript: DirectShow identifiers
 ------------------------------------------------------------------------------}
  IID_IUSRCSet                      = '{AE1A2884-540E-4077-B1AB-67A34A72299F}';
  T_IID_IUSRCSet            : TGUID = IID_IUSRCSet;
  CLSID_IUSRCSet            : TGUID = '{FD501041-8888-1111-9153-00AB00577DA2}';
  IID_IPersistStream        : TGUID = '{00000109-0000-0000-C000-000000000046}';

  CLSID_MPEG2Demultiplexer  : TGUID = (D1:$afb6c280;D2:$2c41;D3:$11d3;D4:($8a,$60,$00,$00,$f8,$1e,$0e,$4a));

  IID_ISpecifyPropertyPages : TGUID = '{B196B28B-BAB4-101A-B69C-00AA00341D07}';  // See DsUtil

type
{------------------------------------------------------------------------------
  Descript: Universal Source filter interface
 ------------------------------------------------------------------------------}
  IUSRCSet = interface(IUnknown)
  [IID_IUSRCSet]
    function WSetMediaType(Pmt   : PAMMEDIATYPE): HRESULT; stdcall;
    function WSetBufSize  (Size  : Dword): HRESULT; stdcall;
    function WSendSample  (Buffer: PChar; BufferLength: Dword): HRESULT; stdcall;
  end;


{------------------------------------------------------------------------------
  Descript: The DirectShow object
 ------------------------------------------------------------------------------}
  TDirectShowGraph = class(TObject)
    FOwner              : HWND;                  // Owner of 'video'
    FLogFile            : THandle;               // Log file
    FOptions            : Dword;                 // Options
    FFilterGraphManager : IUnknown;              // Filter graph manager (IFilterGraph, IGraphBuilder, IFilterGraph2)
    FGraphBuilder       : IUnknown;              // Additional builder   (ICaptureGraphBuilder, ICaptureGraphBuilder2, IDvdGraphBuilder)
    FVmrFilter          : IBaseFilter;           // VMR filter
    FVmrFilterConfig7   : IVMRFilterConfig;
    FVmrFilterConfig9   : IVMRFilterConfig9;
    FVmrMixerControl7   : IVMRMixerControl;
    FVmrMixerControl9   : IVMRMixerControl9;
    FAlphaBitmap7       : VMRAlphaBitmap;
    FAlphaBitmap9       : VMR9AlphaBitmap;
    FVideoWindow        : IVideoWindow;          // Video window (non VMR mode)
    FBasicVideo         : IBasicVideo2;          // Used for retrieval of (required) window size
    FWindowlessControl  : IUnknown;
    FMPEG2Demux         : IBaseFilter;
    FUniversalSource    : IBaseFilter;
    FVideoPin           : IPin;
    FAudioPin           : IPin;
    FActiveAudioPid     : Integer;
    FActiveVideoPid     : Integer;
    FRotEntry           : LongInt;
    constructor Create(Owner: HWND);
    destructor  Destroy; override;
  private
    { Private declarations }
    function BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
  public
    { Public declarations }
  end;

var
  DirectShowGraph: TDirectShowGraph;


{------------------------------------------------------------------------------
  Params  : <ResultCode>     Result code to check
  Returns : <Result>         True if no error
            <ResultMessage>  Error message

  Descript: Chekc result code and generate an error message if in error.
  Notes   :
 ------------------------------------------------------------------------------}
function DirectShowCheck(ResultCode: HRESULT; var ResultMessage: string): Boolean;
var
//  RString     : PChar;
//  RDescription: PChar;
  RError: string;
  Size  : Dword;
begin
  Result := Succeeded(ResultCode);
  ResultMessage := '';
  if Failed(ResultCode) then
  begin
    SetLength(RError, 128);
    Size := AmGetErrorText(ResultCode, @RError[1], 128);
    if Size <> 0 then
    begin
      SetLength(RError, Size);
      ResultMessage := RError;
    end;
    // Next need DLL which is not always present ....
//    RString      := DxGetErrorString9(Code);
//    RDescription := DxGetErrorDescription9(Code);
//    ShowMessage('DirectShow error:'#13#13 +
//                 RString + #13 +
//                 RDescription);
  end;
end;


{------------------------------------------------------------------------------
  Params  : <UnknownGraph>  Graph to add to ROT
  Returns : <Result>        Result (S_OK if success)
            <RotEntry>      Assigned number in ROT

  Descript: Add graph to Running Object Table so we can connect to it when
            it is running.
  Notes   :
 ------------------------------------------------------------------------------}
function AddToRot(UnknownGraph: IUnknown; var RotEntry: LongInt): HRESULT;
var
  Moniker: IMoniker;
  Rot    : IRunningObjectTable;
  Wsz    : WideString;
  Wsz2   : WideString;
  Hr     : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, ROT);
    if Failed(Hr) then
    begin
      Result := E_FAIL;
      Exit;
    end;
    Wsz  := format('FilterGraph %p pid %8.8x', [Pointer(UnknownGraph), GetCurrentProcessId]);
    Wsz2 := '!';
    Hr := CreateItemMoniker(@Wsz2[1], @Wsz[1], Moniker);
    if Succeeded(Hr) then
    begin
      Hr := ROT.Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, UnknownGraph,
              Moniker, RotEntry);
    end;
    Result := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <RotEntry>      Assigned number in ROT
  Returns : -

  Descript: Remove graph from Running Object Table
  Notes   :
 ------------------------------------------------------------------------------}
procedure RemoveFromRot(RotEntry: LongInt);
var
  Rot: IRunningObjectTable ;
  Hr : HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, Rot);
    if Succeeded(Hr) then
      Rot.Revoke(RotEntry);
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileName>  Name of graph file
  Returns : -

  Descript: Load a DirectShow graph (.GRF) file
  Notes   :
 ------------------------------------------------------------------------------}
function LoadGraphFile(FileName: string): HRESULT;
var
  PStorage      : IStorage;
  PPersistStream: IPersistStream;
  PStream       : IStream;
  WideName      : array[0..1023] of WideChar;
  Hr            : HRESULT;
begin
  try
    PStorage       := nil;
    PPersistStream := nil;
    PStream        := nil;
    StringToWideChar(FileName, WideName, SizeOf(WideName));
    hr := StgIsStorageFile(@WideName);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    Hr := StgOpenStorage(@WideName, nil, STGM_TRANSACTED or STGM_READ or STGM_SHARE_DENY_WRITE,
            nil, 0, PStorage);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IPersistStream, PPersistStream);
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IPersistStream, PPersistStream);
      CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IPersistStream, PPersistStream);
    end;
    if Succeeded(Hr) then
    begin
      Hr := PStorage.OpenStream('ActiveMovieGraph', nil,
              STGM_READ or STGM_SHARE_EXCLUSIVE, 0, PStream);
      if Succeeded(Hr) then
      begin
        Hr := PPersistStream.Load(PStream);
        PStream := nil;
      end;
      PPersistStream := nil;
    end;
    PStorage := nil;
    Result  := Hr;
  except
    Result := S_FALSE;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Owner>           Handle to window for video
            <GraphicsFile>    .GRF file to use
                              If empty the graph is manually created
            <Options>         Options to use (see constants)
  Returns : <Result>          TRUE if success, FALSE if failure
            <ErrorMessage>    Error message (if any)

  Descript: Setup direct show graph
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowStart(Owner           : HWND;
                          GraphicsFile    : string;
                          Options         : Dword;
                          var ErrorMessage: string): Boolean;
var
  Hr                          : HRESULT;
  Pins                        : IPin;
  PinsIn                      : IPin;
  PinsOut                     : IPin;
  PinsEnum                    : IEnumPins;
  PinsInfo                    : TPinInfo;
//  PinDirection                : TPinDirection;
  MediaType                   : AM_MEDIA_TYPE;
  MediaControl                : IMediaControl;
  EnumFilters                 : IEnumFilters;
  BaseFilter                  : IBaseFilter;
  AllPins                     : array[0..99] of IPin;
  AllPinsCount                : Integer;
  Index1                      : Integer;
  MPEG2DemultiplexerInterface : IMPEG2Demultiplexer;
  EnumMediaTypes              : IEnumMediaTypes;
  PMediaType                  : PAMMediaType;
  APin                        : IPin;
  DestinationRect             : TRect;
  FlipRect7                   : TNormalizedRect;
  FlipRect9                   : TVMR9NormalizedRect;
  FilterState                 : TFilterState;
  ResultMessage               : string;
  {------------------------------------------------------------------------------
    Params  : <Name>    Name to find
    Returns : <Result>
              <Filter>  Filter found

    Descript: Find filter by name
    Notes   :
   ------------------------------------------------------------------------------}
  function FindFilterByName(Name: WideString; var Filter: IBaseFilter): HRESULT;
  begin
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).FindFilterByName(PWideChar(Name), Filter);
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Result := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).FindFilterByName(PWideChar(Name), Filter);
      CDirectShowMethodFilterGraph     : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).FindFilterByName(PWideChar(Name), Filter);
      else                               Result := E_NOTIMPL;
    end;
  end;


  {------------------------------------------------------------------------------
    Params  : <FilterName>      Filter which has the pin
              <PinName>         Name of the pin to render
    Returns : <Result>          TRUE if success, FALSE if failure
              <ResultMessage>   Error message

    Descript: Render a pin
    Notes   :
   ------------------------------------------------------------------------------}
  function RenderPin(FilterName: WideString; PinName: WideString; var ResultMessage: string): Boolean;
  var
    Hr       : HRESULT;
    Filter   : IBaseFilter;
    EnumPins : IEnumPins;
    Pin      : IPin;
    pInfo    : TPinInfo;
  begin
    Result := False;
    ResultMessage := '';
    Hr := FindFilterByName(FilterName, Filter);
    if Succeeded(Hr) then
    begin
      Filter.EnumPins(EnumPins);
      // Go through all pins until we find it
      EnumPins.Reset;
      while Succeeded(EnumPins.Next(1, Pin, nil)) do
      begin
        Pin.QueryPinInfo(pInfo);
        pInfo.pFilter := nil;
        if (pInfo.achName = PinName) then
        begin
          // Only render if not connected
          case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2    : begin
                                                 if (DirectShowGraph.FOptions and CDirectShowMethodRenderEx) <> 0 then
                                                 begin
                                                   Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).RenderEx(Pin, AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                                                   if Failed(Hr) then
                                                     Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(Pin);
                                                 end
                                                 else
                                                   Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(Pin);
                                               end;
            CDirectShowMethodGraphBuilder,
            CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).Render(Pin);
            CDirectShowMethodFilterGraph     : Hr := E_NOTIMPL;
            else                               Hr := E_NOTIMPL;
          end;
          // Failing of rendering can be normal since it might be already connected ...
          // Reconnecting it should succeed though
          if Failed(Hr) then
          begin
            case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
              CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).ReConnectEx(Pin, nil);
              CDirectShowMethodGraphBuilder,
              CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).ReConnect(Pin);
              CDirectShowMethodFilterGraph     : Hr := E_NOTIMPL;
              else                               Hr := E_NOTIMPL;
            end;
          end;
          if not DirectShowCheck(Hr, ResultMessage) then
            Result := False
          else
            Result := True;
          Exit;
        end;
      end;
    end
    else
      DirectShowCheck(Hr, ResultMessage);
  end;


  {------------------------------------------------------------------------------
    Params  : <Filter>   Base filter
              <Name>     Name of filter to add
    Returns : <Result>

    Descript: Add filter by name
    Notes   :
   ------------------------------------------------------------------------------}
  function AddFilter(Filter: IBaseFilter; Name: WideString): HRESULT;
  begin
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).AddFilter(Filter, PWideChar(Name));
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Result := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).AddFilter(Filter, PWideChar(Name));
      CDirectShowMethodFilterGraph     : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).AddFilter(Filter, PWideChar(Name));
      else                               Result := E_NOTIMPL;
    end;
  end;

  {------------------------------------------------------------------------------
    Params  : <PinOut>   Output pin
              <PinIn>    Input pin
    Returns : <Result>

    Descript: Connect pins
    Notes   :
   ------------------------------------------------------------------------------}
  function Connect(PinOut: IPin; PinIn: IPin): HRESULT;
  begin
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Connect(PinOut, PinIn);
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Result := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).Connect(PinOut, PinIn);
      CDirectShowMethodFilterGraph     : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).ConnectDirect(PinOut, PinIn, nil);
      else                               Result := E_NOTIMPL;
    end;
  end;

  {------------------------------------------------------------------------------
    Params  : <PinOut>   Output pin
              <PinIn>    Input pin
    Returns : <Result>

    Descript: Connect pins
    Notes   :
   ------------------------------------------------------------------------------}
  function SetDefaultSyncSource: HRESULT;
  begin
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).SetDefaultSyncSource;
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Result := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).SetDefaultSyncSource;
      CDirectShowMethodFilterGraph     : Result := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).SetDefaultSyncSource;
      else                               Result := E_NOTIMPL;
    end;
  end;

begin
  try
    // Make sure a previously started DirectShow interface is stopped and released
    DirectShowStop;
    ErrorMessage := '';
    Result := False;
    DirectShowGraph := TDirectShowGraph.Create(Owner);
    try
      DirectShowGraph.FOptions := Options;
      if (DirectShowGraph.FOptions and CDirectShowMethodLogToFile) <> 0 then
      begin
        DirectShowGraph.FLogFile := CreateFile('DirectShow.log'+#0,
          GENERIC_WRITE,
          0,
          nil,
          CREATE_ALWAYS,
          FILE_ATTRIBUTE_NORMAL,
          0 );
      end;

      {------------------------------------------------------------------------------
        Descript: Create the filter manager
       ------------------------------------------------------------------------------}
      case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2     : Hr := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER, IID_IFilterGraph2, DirectShowGraph.FFilterGraphManager);
        CDirectShowMethodGraphBuilder     : Hr := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, DirectShowGraph.FFilterGraphManager);
        CDirectShowMethodFilterGraph      : Hr := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER, IID_IFilterGraph,  DirectShowGraph.FFilterGraphManager);
        CDirectShowMethodDvdGraphBuilder  : begin
                                              // The DVD builder creates the filter manager
                                              Hr := CoCreateInstance(CLSID_DvdGraphBuilder,      nil, CLSCTX_INPROC_SERVER, IID_IDvdGraphBuilder,      DirectShowGraph.FGraphBuilder);
                                              if Succeeded(Hr) then
                                                Hr :=  (DirectShowGraph.FGraphBuilder as IDvdGraphBuilder).GetFilterGraph(IGraphBuilder(DirectShowGraph.FFilterGraphManager));
                                            end;
        else                                Hr := E_NOTIMPL;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := format('CoCreateInstance filter graph manager error $%8.8x.', [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      // Log file
      if (DirectShowGraph.FOptions and CDirectShowMethodLogToFile) <> 0 then
        if DirectShowGraph.FLogFile <> INVALID_HANDLE_VALUE then
          case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2    : (DirectShowGraph.FFilterGraphManager as IFilterGraph2).SetLogFile(DirectShowGraph.FLogFile);
            CDirectShowMethodGraphBuilder,
            CDirectShowMethodDvdGraphBuilder : (DirectShowGraph.FFilterGraphManager as IGraphBuilder).SetLogFile(DirectShowGraph.FLogFile);
          end;

      {------------------------------------------------------------------------------
        Descript: Additional builders
       ------------------------------------------------------------------------------}
      case (DirectShowGraph.FOptions and CDirectShowMethodBuilder) of
        CDirectShowMethodCaptureGraphBuilder   : begin
                                                   Hr := CoCreateInstance(CLSID_CaptureGraphBuilder,  nil, CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder,  DirectShowGraph.FGraphBuilder);
                                                   if Succeeded(Hr) then
                                                     case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
                                                       CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FGraphBuilder as ICaptureGraphBuilder).SetFilterGraph(IFilterGraph2(DirectShowGraph.FFilterGraphManager));
                                                       CDirectShowMethodGraphBuilder,
                                                       CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FGraphBuilder as ICaptureGraphBuilder).SetFilterGraph(IGraphBuilder(DirectShowGraph.FFilterGraphManager));
                                                       CDirectShowMethodFilterGraph     : Hr := E_NOTIMPL;
                                                     end;
                                                 end;
        CDirectShowMethodCaptureGraphBuilder2  : begin
                                                   Hr := CoCreateInstance(CLSID_CaptureGraphBuilder2, nil, CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder2, DirectShowGraph.FGraphBuilder);
                                                   if Succeeded(Hr) then
                                                     case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
                                                       CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FGraphBuilder as ICaptureGraphBuilder2).SetFilterGraph(IFilterGraph2(DirectShowGraph.FFilterGraphManager));
                                                       CDirectShowMethodGraphBuilder,
                                                       CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FGraphBuilder as ICaptureGraphBuilder2).SetFilterGraph(IGraphBuilder(DirectShowGraph.FFilterGraphManager));
                                                       CDirectShowMethodFilterGraph     : Hr := E_NOTIMPL;
                                                     end;
                                                 end;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := format('CoCreateInstance graph builder error $%8.8x.', [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: ROT
       ------------------------------------------------------------------------------}
      if (DirectShowGraph.FOptions and CDirectShowMethodDoNotAddToRot) = 0 then
      begin
        Hr := AddToRot(DirectShowGraph.FFilterGraphManager, DirectShowGraph.FRotEntry);
        if Failed(Hr) then
        begin
          DirectShowGraph.FRotEntry := -1;
        end;
      end;

      {------------------------------------------------------------------------------
        Descript: Load filter graph
       ------------------------------------------------------------------------------}
      if FileExists(GraphicsFile) then
      begin
        Hr := LoadGraphFile(GraphicsFile);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Loading graph file "%s" error $%8.8x.', [GraphicsFile, Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;

      {------------------------------------------------------------------------------
        Descript: VMR
       ------------------------------------------------------------------------------}
      case (DirectShowGraph.FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer  : Hr := NOERROR;
        CDirectShowMethodVideoRenderer7 : begin
                                            // Try locating if VMR already present
                                            Hr := FindFilterByName('Video Mixing Renderer 7', DirectShowGraph.FVmrFilter);
                                            if Failed(Hr) then
                                            begin
                                              Hr := FindFilterByName('VMR7', DirectShowGraph.FVmrFilter);
                                              if Failed(Hr) then
                                              begin
                                                // Not present, create it
                                                Hr := CoCreateInstance(CLSID_VideoMixingRenderer,  nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, DirectShowGraph.FVmrFilter);
                                                if Succeeded(Hr) then
                                                begin
                                                  Hr := AddFilter(DirectShowGraph.FVmrFilter, 'Video Mixing Renderer 7');
                                                  if Failed(Hr) then
                                                    Hr := AddFilter(DirectShowGraph.FVmrFilter, 'VMR7');
                                                end;
                                              end;
                                            end;
                                          end;
        CDirectShowMethodVideoRenderer9 : begin
                                            // Try locating if VMR already present (Note: will ALWAYS fail if nothing loaded)
                                            Hr := FindFilterByName('Video Mixing Renderer 9', DirectShowGraph.FVmrFilter);
                                            if Failed(Hr) then
                                            begin
                                              Hr := FindFilterByName('VMR9', DirectShowGraph.FVmrFilter);
                                              if Failed(Hr) then
                                              begin
                                                // Not present, create it
                                                Hr := CoCreateInstance(CLSID_VideoMixingRenderer9, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, DirectShowGraph.FVmrFilter);
                                                if Succeeded(Hr) then
                                                begin
                                                  Hr := AddFilter(DirectShowGraph.FVmrFilter, 'Video Mixing Renderer 9');
                                                  if Failed(Hr) then
                                                    Hr := AddFilter(DirectShowGraph.FVmrFilter, 'VMR9');
                                                end;
                                              end;
                                            end;
                                          end;
        else                              Hr := E_NOTIMPL;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := format('CoCreateInstance video mixing renderer error $%8.8x.', [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: Set VMR mode
       ------------------------------------------------------------------------------}
      case (DirectShowGraph.FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer  : ;
        CDirectShowMethodVideoRenderer7 : begin
                                            Hr := DirectShowGraph.FVmrFilter.QueryInterface(IID_IVMRFilterConfig, DirectShowGraph.FVmrFilterConfig7);
                                            if not DirectShowCheck(Hr, ResultMessage) then
                                            begin
                                              ErrorMessage := format('Could not get interface to filter configuration, error $%8.8x.', [Hr]);
                                              ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
                                              Exit;
                                            end;
                                            // We need to set the number of streams to be able to use the mixer and this must before setting window mode
                                            DirectShowGraph.FVmrFilterConfig7.SetNumberOfStreams(16);
                                            // Flip
                                            Hr := DirectShowGraph.FVmrFilterConfig7.QueryInterface(IID_IVMRMixerControl, DirectShowGraph.FVmrMixerControl7);
                                            if Succeeded(Hr) then
                                            begin
                                              FlipRect7.Left   := 0;
                                              FlipRect7.Top    := 0;
                                              FlipRect7.Right  := 1;
                                              FlipRect7.Bottom := 1;
                                              // Note: Using -1 instead of swapping does not work correctly
                                              if (DirectShowGraph.FOptions and CDirectShowMethodVideoRendererFlipX) <> 0 then
                                              begin
                                                FlipRect7.Left   := 1;
                                                FlipRect7.Right  := 0;
                                              end;
                                              if (DirectShowGraph.FOptions and CDirectShowMethodVideoRendererFlipY) <> 0 then
                                              begin
                                                FlipRect7.Top    := 1;
                                                FlipRect7.Bottom := 0;
                                              end;
                                              DirectShowGraph.FVmrMixerControl7.SetOutputRect(0, FlipRect7);
                                            end;
                                            // Window(less) mode
                                            if (DirectShowGraph.FOptions and CDirectShowMethodVmrSpecials) = CDirectShowMethodVideoRendererWindowless then
                                            begin
                                              DirectShowGraph.FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowless);
                                              Hr := DirectShowGraph.FVmrFilter.QueryInterface(IID_IVMRWindowlessControl, DirectShowGraph.FWindowlessControl);
                                              if Succeeded(Hr) then
                                                IVMRWindowlessControl(DirectShowGraph.FWindowlessControl).SetVideoClippingWindow(DirectShowGraph.FOwner)
                                              else
                                                DirectShowGraph.FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowed);
                                            end
                                            else
                                              DirectShowGraph.FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowed);
                                          end;
        CDirectShowMethodVideoRenderer9 : begin
                                            Hr := DirectShowGraph.FVmrFilter.QueryInterface(IID_IVMRFilterConfig9, DirectShowGraph.FVmrFilterConfig9);
                                            if not DirectShowCheck(Hr, ResultMessage) then
                                            begin
                                              ErrorMessage := format('Could not get interface to filter configuration, error $%8.8x.', [Hr]);
                                              ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
                                              Exit;
                                            end;
                                            // We need to set the number of streams to be able to use the mixer and this must before setting window mode
                                            DirectShowGraph.FVmrFilterConfig9.SetNumberOfStreams(16);
                                            // Flip
                                            Hr := DirectShowGraph.FVmrFilterConfig9.QueryInterface(IID_IVMRMixerControl9, DirectShowGraph.FVmrMixerControl9);
                                            if Succeeded(Hr) then
                                            begin
                                              FlipRect9.Left   := 0;
                                              FlipRect9.Top    := 0;
                                              FlipRect9.Right  := 1;
                                              FlipRect9.Bottom := 1;
                                              // Note: Using -1 instead of swapping does not work correctly
                                              if (DirectShowGraph.FOptions and CDirectShowMethodVideoRendererFlipX) <> 0 then
                                              begin
                                                FlipRect9.Left  := 1;
                                                FlipRect9.Right := 0;
                                              end;
                                              if (DirectShowGraph.FOptions and CDirectShowMethodVideoRendererFlipY) <> 0 then
                                              begin
                                                FlipRect9.Top    := 1;
                                                FlipRect9.Bottom := 0;
                                              end;
                                              DirectShowGraph.FVmrMixerControl9.SetOutputRect(0, @FlipRect9);
                                            end;
                                            // Window(less) mode
                                            if (DirectShowGraph.FOptions and CDirectShowMethodVmrSpecials) = CDirectShowMethodVideoRendererWindowless then
                                            begin
                                              DirectShowGraph.FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowless);
                                              Hr := DirectShowGraph.FVmrFilter.QueryInterface(IID_IVMRWindowlessControl9, DirectShowGraph.FWindowlessControl);
                                              if Succeeded(Hr) then
                                                IVMRWindowlessControl9(DirectShowGraph.FWindowlessControl).SetVideoClippingWindow(DirectShowGraph.FOwner)
                                              else
                                                DirectShowGraph.FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowed);
                                            end
                                            else
                                              DirectShowGraph.FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowed);
                                          end;
      end;

      {------------------------------------------------------------------------------
        Descript: Building the graph
       ------------------------------------------------------------------------------}
      if not FileExists(GraphicsFile) then
      begin
        {------------------------------------------------------------------------------
          Descript: Building a graph manually
         ------------------------------------------------------------------------------}
        // No graph filter file: create the graph manually (at least the major parts)
        // First the universal source
        Hr := CoCreateInstance(CLSID_IUSRCSet, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, DirectShowGraph.FUniversalSource);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not use the Universal Source filter.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := AddFilter(DirectShowGraph.FUniversalSource, 'Universal Source');
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Adding Universal Source filter error $%8.8x.', [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Then the MPEG-2 demultiplexer
        Hr := CoCreateInstance(CLSID_MPEG2Demultiplexer, nil, CLSCTX_INPROC_SERVER, IID_IBaseFilter, DirectShowGraph.FMPEG2Demux);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not use the MPEG-2 Demultiplexer filter.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := AddFilter(DirectShowGraph.FMPEG2Demux, 'MPEG-2 Demultiplexer');
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Adding MPEG-2 Demultiplexer filter error $%8.8x.', [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Now we must create the output pins on the MPEG-2 demultiplexer
        // The video pin
        Hr := DirectShowGraph.FMPEG2Demux.QueryInterface(IID_IMpeg2Demultiplexer, MPEG2DemultiplexerInterface);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter MPEG-2 Demultiplexer interface not found.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        MediaType.MajorType := MEDIATYPE_Video;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_VIDEO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Video', DirectShowGraph.FVideoPin);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the video output pin for the MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // The audio pin
        MediaType.MajorType := MEDIATYPE_Audio;
        MediaType.SubType   := MEDIASUBTYPE_MPEG2_AUDIO;
        Hr := MPEG2DemultiplexerInterface.CreateOutputPin(MediaType, 'Audio', DirectShowGraph.FAudioPin);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          MPEG2DemultiplexerInterface := nil;
          ErrorMessage := 'Could not create the audio output pin for the MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        MPEG2DemultiplexerInterface := nil;
        PinsIn  := nil;
        PinsOut := nil;

        // Enumerate the pins
        Hr := DirectShowGraph.FUniversalSource.EnumPins(PinsEnum);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not enumerate the Universal Source pins.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        PinsEnum.Reset;
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          PinsInfo.pFilter := nil;
          if UpperCase(PinsInfo.achName) = 'OUTPUT' then
            PinsOut := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        Hr := DirectShowGraph.FMPEG2Demux.EnumPins(PinsEnum);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not enumerate the MPEG-2 Demultiplexer pins.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        PinsEnum.Reset;
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          PinsInfo.pFilter := nil;
          if UpperCase(PinsInfo.achName) = 'MPEG-2 STREAM' then
            PinsIn := Pins;
        end;
        Pins     := nil;
        PinsEnum := nil;

        // Connect the pins
        if not Assigned(PinsIn) or not Assigned(PinsOut) then
        begin
          PinsIn  := nil;
          PinsOut := nil;
          ErrorMessage := 'Could not find correct pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
        Hr := Connect(PinsOut, PinsIn);
        PinsIn  := nil;
        PinsOut := nil;
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not connect pins Universal Source/MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        {------------------------------------------------------------------------------
          Descript: Building a graph manually end
         ------------------------------------------------------------------------------}
      end;

      {------------------------------------------------------------------------------
        Descript: Check for required components
       ------------------------------------------------------------------------------}
      if not Assigned(DirectShowGraph.FMPEG2Demux) then
      begin
        Hr := FindFilterByName('MPEG-2 Demultiplexer', DirectShowGraph.FMPEG2Demux);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer filter not found.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;
      Hr := DirectShowGraph.FMPEG2Demux.FindPin('Video', DirectShowGraph.FVideoPin);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer video pin not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      Hr := DirectShowGraph.FMPEG2Demux.FindPin('Audio', DirectShowGraph.FAudioPin);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer audio pin not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: Render
       ------------------------------------------------------------------------------}
      if (DirectShowGraph.FOptions and CDirectShowMethodDoNotRender) = 0 then
      begin
        if not RenderPin('MPEG-2 Demultiplexer', 'Audio', ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer audio pin rendering error.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        if not RenderPin('MPEG-2 Demultiplexer', 'Video', ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer video pin rendering error.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;

        {------------------------------------------------------------------------------
          Descript: Render all pins of all filters
         ------------------------------------------------------------------------------}
        if (DirectShowGraph.FOptions and CDirectShowMethodRenderAllPins) <> 0 then
        begin
          case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).EnumFilters(EnumFilters);
            CDirectShowMethodGraphBuilder,
            CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).EnumFilters(EnumFilters);
            CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).EnumFilters(EnumFilters);
          end;
          AllPinsCount := 0;
          // If we don't retrieve all filters/pins before rendering them, then we will not get all filters for some reason
          if Succeeded(Hr) then
          begin
            EnumFilters.Reset;
            while EnumFilters.Next(1, BaseFilter, nil) = S_OK do
            begin
              Hr := BaseFilter.EnumPins(PinsEnum);
              if Succeeded(Hr) then
              begin
                PinsEnum.Reset;
                while PinsEnum.Next(1, Pins, nil) = S_OK do
                begin
                  AllPins[AllPinsCount] := Pins;
                  Inc(AllPinsCount);
                end;
                Pins := nil;
              end;
            end;
            EnumFilters := nil;
          end;
          // We now have all the filters with their pins, so render then
          if AllPinsCount > 0 then
            for Index1 := 0 to AllPinsCount-1 do
              case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2    : begin
                                                     if (DirectShowGraph.FOptions and CDirectShowMethodRenderEx) <> 0 then
                                                     begin
                                                       Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).RenderEx(AllPins[Index1], AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                                                       if Failed(Hr) then
                                                         (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(AllPins[Index1]);
                                                     end
                                                     else
                                                       (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(AllPins[Index1]);
                                                   end;
                CDirectShowMethodGraphBuilder,
                CDirectShowMethodDvdGraphBuilder : (DirectShowGraph.FFilterGraphManager as IGraphBuilder).Render(AllPins[Index1]);
              end;
        end;
        {------------------------------------------------------------------------------
          Descript: Render all audio/video pins of all filters
         ------------------------------------------------------------------------------}
        if (DirectShowGraph.FOptions and CDirectShowMethodRenderAllAudioVideo) <> 0 then
        begin
          case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).EnumFilters(EnumFilters);
            CDirectShowMethodGraphBuilder,
            CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).EnumFilters(EnumFilters);
            CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).EnumFilters(EnumFilters);
          end;
          AllPinsCount := 0;
          // If we don't retrieve all filters/pins before rendering them, then we will not get all filters for some reason
          if Succeeded(Hr) then
          begin
            EnumFilters.Reset;
            while EnumFilters.Next(1, BaseFilter, nil) = S_OK do
            begin
              Hr := BaseFilter.EnumPins(PinsEnum);
              if Succeeded(Hr) then
              begin
                PinsEnum.Reset;
                while (PinsEnum.Next(1, Pins, nil) = S_OK) do
                begin
//                  Pins.QueryDirection(PinDirection);
//                  if PinDirection = PINDIR_OUTPUT then
                  Pins.QueryPinInfo(PinsInfo);
                  PinsInfo.pFilter := nil;
                  // If not connected
                  if PinsInfo.dir = PINDIR_OUTPUT then               // Must be output
                  begin
                    Hr := Pins.EnumMediaTypes(EnumMediaTypes);       // Check media types
                    if Succeeded(Hr) then
                    begin
                      EnumMediaTypes.Reset;
                      while EnumMediaTypes.Next(1, PMediaType, nil) = S_OK do
                      begin
                        if IsEqualGUID(PMediaType.majortype, MEDIATYPE_Video) or
                           IsEqualGUID(PMediaType.majortype, MEDIATYPE_Audio) then
                        begin
                          AllPins[AllPinsCount] := Pins;
                          Inc(AllPinsCount);
                        end;
                        PMediaType := nil;
                      end;
                    end;
                  end;
                end;
                APin := nil;
                Pins := nil;
              end;
            end;
            EnumFilters := nil;
          end;
          // We now have all the filters with their pins, so render then
          if AllPinsCount > 0 then
            for Index1 := 0 to AllPinsCount-1 do
              case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2    : begin
                                                     if (DirectShowGraph.FOptions and CDirectShowMethodRenderEx) <> 0 then
                                                     begin
                                                       Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).RenderEx(AllPins[Index1], AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                                                       if Failed(Hr) then
                                                         (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(AllPins[Index1]);
                                                     end
                                                     else
                                                       (DirectShowGraph.FFilterGraphManager as IFilterGraph2).Render(AllPins[Index1]);
                                                   end;
                CDirectShowMethodGraphBuilder,
                CDirectShowMethodDvdGraphBuilder : (DirectShowGraph.FFilterGraphManager as IGraphBuilder).Render(AllPins[Index1]);
              end;
        end;
      end;
      Hr := FindFilterByName('Universal Source', DirectShowGraph.FUniversalSource);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'Universal Source filter not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      DirectShowGraph.FMPEG2Demux.SetSyncSource(nil);
      SetDefaultSyncSource;

      // Get IBasicVideo2 interface
      case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IBasicVideo2, DirectShowGraph.FBasicVideo);
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IBasicVideo2, DirectShowGraph.FBasicVideo);
        CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IBasicVideo2, DirectShowGraph.FBasicVideo);
        else                               Hr := E_NOTIMPL;
      end;
      {------------------------------------------------------------------------------
        Descript: No VMR (default Video Renderer)
       ------------------------------------------------------------------------------}
      if ((DirectShowGraph.FOptions and CDirectShowMethodVmr) = CDirectShowMethodVideoRenderer) or
          (not Assigned(DirectShowGraph.FWindowlessControl)) then
      begin
        case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
          CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IVideoWindow, DirectShowGraph.FVideoWindow);
          CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IVideoWindow, DirectShowGraph.FVideoWindow);
          CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IVideoWindow, DirectShowGraph.FVideoWindow);
        end;
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not retrieve video window interface.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := DirectShowGraph.FVideoWindow.put_Owner(DirectShowGraph.FOwner);

        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Filter could not set video handle, error $%8.8x.', [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := DirectShowGraph.FVideoWindow.put_MessageDrain(DirectShowGraph.FOwner);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set messages handle.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := DirectShowGraph.FVideoWindow.put_WindowStyle(WS_CHILD or WS_CLIPSIBLINGS or WS_CLIPCHILDREN);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set window style.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;

        GetClientRect(DirectShowGraph.FOwner, DestinationRect);
         Hr := DirectShowGraph.FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set window size.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;

      // Media control
      case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IMediaControl, MediaControl);
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'Could not get media control interface.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      MediaControl.Stop;
      MediaControl.Run;
      MediaControl.GetState(1000, FilterState);
      MediaControl := nil;
      // Indicate success
      Result := True;
    finally
      // Make sure that at a failure all resources are freed
      if not Result then
        DirectShowStop;
    end;
  except
    Result := False;
    DirectShowStop;
    ShowMessage('DirectShow Start error');
  end;
end;


function  DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
begin
  Result := False;
  if not Assigned(DirectShowGraph) then
    Exit;
  if (DirectShowGraph.FOptions and CDirectShowMethodVmr) = CDirectShowMethodVideoRenderer then
    Exit;
  Result := DirectShowGraph.BlendApplicationImage(Bitmap, Src, Blend, ColorKey);
end;


{------------------------------------------------------------------------------
  Params  : <Owner>  Handle to window for video
  Returns : -

  Descript: Constructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
constructor TDirectShowGraph.Create(Owner: HWND);
begin
  inherited Create;
  FOwner              := Owner;
  FLogFile            := INVALID_HANDLE_VALUE;
  FRotEntry           := -1;
  FFilterGraphManager := nil;
  FGraphBuilder       := nil;
  FBasicVideo         := nil;
  FVmrFilter          := nil;
  FVmrFilterConfig7   := nil;
  FVmrFilterConfig9   := nil;
  FVmrMixerControl7   := nil;
  FVmrMixerControl9   := nil;
  FMPEG2Demux         := nil;
  FUniversalSource    := nil;
  FVideoPin           := nil;
  FAudioPin           := nil;
  FVideoWindow        := nil;
  FWindowlessControl  := nil;
  FActiveAudioPid     := -1;
  FActiveVideoPid     := -1;
  ZeroMemory(@FAlphaBitmap7, sizeof(FAlphaBitmap7));
  ZeroMemory(@FAlphaBitmap7, sizeof(FAlphaBitmap9));
  FAlphaBitmap7.Hdc := 0;
  FAlphaBitmap9.Hdc := 0;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor of object.
  Notes   :
 ------------------------------------------------------------------------------}
destructor TDirectShowGraph.Destroy;
var
  MediaControl : IMediaControl;
begin
  if Assigned(FFilterGraphManager) then
  begin
    try
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : (FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : (FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodFilterGraph     : (FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IMediaControl, MediaControl);
      end;
      MediaControl.Stop;
      MediaControl := nil;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : (FFilterGraphManager as IFilterGraph2).Abort;
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : (FFilterGraphManager as IGraphBuilder).Abort;
      end;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : (FFilterGraphManager as IFilterGraph2).SetLogFile(0);
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : (FFilterGraphManager as IGraphBuilder).SetLogFile(0);
      end;
      if FRotEntry <> -1 then
        RemoveFromRot(FRotEntry);
      // Important to stop the message drain, otherwise errors occur
      FRotEntry           := -1;
      FVideoWindow        := nil;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2    : IFilterGraph2(FFilterGraphManager) := nil;
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder : IGraphBuilder(FFilterGraphManager) := nil;
      end;
      FGraphBuilder       := nil;
      FVmrFilter          := nil;
      FVmrFilterConfig7   := nil;
      FVmrFilterConfig9   := nil;
      FVmrMixerControl7   := nil;
      FVmrMixerControl9   := nil;
      FBasicVideo         := nil;
      FMPEG2Demux         := nil;
      FUniversalSource    := nil;
      FVideoPin           := nil;
      FAudioPin           := nil;
      FActiveAudioPid     := -1;
      FActiveVideoPid     := -1;
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer7 : IVMRWindowlessControl (FWindowlessControl) := nil;
        CDirectShowMethodVideoRenderer9 : IVMRWindowlessControl9(FWindowlessControl) := nil;
      end;
      if FLogFile <> INVALID_HANDLE_VALUE then
        CloseHandle(FLogFile);
      FLogFile := INVALID_HANDLE_VALUE;
    except
    end;
  end;
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : <Bitmap>    Bitmap to blend in
            <Dest>      Source position/size
            <Blend>     Blending value (0.0...1.0 == transparent..opaque)
            <ColorKey>  Transparancy color
  Returns : <Result>  True if success

  Descript: Blend in application image
  Notes   :
 ------------------------------------------------------------------------------}
function TDirectShowGraph.BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend: Single; ColorKey: COLORREF): Boolean;
begin
  Result := True;
  case (FOptions and CDirectShowMethodVmr) of
    CDirectShowMethodVideoRenderer  : ;
    CDirectShowMethodVideoRenderer7 : begin
                                        with FAlphaBitmap7 do
                                        begin
                                          Hdc          := Bitmap;
                                          rSrc         := Src;
                                          rDest.Left   := 0;
                                          rDest.Top    := 0;
                                          rDest.Right  := 1;
                                          rDest.Bottom := 1;
                                          dwFlags   := VMRBITMAP_SRCCOLORKEY or VMRBITMAP_HDC;
                                          clrSrcKey := ColorKey;
                                          fAlpha    := Blend;
                                        end;
                                        (FVmrFilter as IVMRMixerBitmap).SetAlphaBitmap(FAlphaBitmap7);
                                      end;
    CDirectShowMethodVideoRenderer9 : begin
                                        with FAlphaBitmap9 do
                                        begin
                                          Hdc          := Bitmap;
                                          rSrc         := Src;
                                          rDest.Left   := 0;
                                          rDest.Top    := 0;
                                          rDest.Right  := 1;
                                          rDest.Bottom := 1;
                                          dwFlags   := VMRBITMAP_SRCCOLORKEY or VMRBITMAP_HDC;
                                          clrSrcKey := ColorKey;
                                          fAlpha    := Blend;
                                        end;
                                        (FVmrFilter as IVMRMixerBitmap9).SetAlphaBitmap(@FAlphaBitmap9);
                                      end;
  end;                                      
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if 'windowless' mode

  Descript: Get windowless mode (VMR)
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowWindowless: Boolean;
begin
  if Assigned(DirectShowGraph) and Assigned(DirectShowGraph.FWindowlessControl) then
    Result := True
  else
    Result := False;
end;


{------------------------------------------------------------------------------
  Params  : <Index>  Filter index (0 = first)
  Returns : <Result> Name of filter (empty if no filter at index)

  Descript: Get filter name
  Notes   :
 ------------------------------------------------------------------------------}
function  DirectShowGetFilterName(Index: Integer): string;
var
  EnumFilters: IEnumFilters;
  BaseFilter : IBaseFilter;
  FilterInfo : TFilterInfo;
  FilterIndex: Integer;
begin
  Result := '';
  if not Assigned(DirectShowGraph) then
    Exit;
  try
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : (DirectShowGraph.FFilterGraphManager as IFilterGraph2).EnumFilters(EnumFilters);
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : (DirectShowGraph.FFilterGraphManager as IGraphBuilder).EnumFilters(EnumFilters);
      CDirectShowMethodFilterGraph     : (DirectShowGraph.FFilterGraphManager as IFilterGraph ).EnumFilters(EnumFilters);
    end;
    FilterIndex := -1;
    EnumFilters.Reset;
    while (EnumFilters.Next(1, BaseFilter, nil) = S_OK) and
          (FilterIndex < Index) do
    begin
      BaseFilter.QueryFilterInfo(FilterInfo);
      Result := FilterInfo.achName;
      Inc(FilterIndex);
    end;
    // Make sure an incorrect index returns nothing
    if FilterIndex <> Index then
      Result := '';
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Parent>      Handle of parent to create property page in
            <FilterName>  Name of filter
            <QueryOnly>   True if only to check for property page (not to show)
  Returns : <Result>      True if property page available

  Descript: Show property page of filter
  Notes   :
 ------------------------------------------------------------------------------}
function DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString; QueryOnly: Boolean): Boolean;
var
  BaseFilter          : IBaseFilter;
  SpecifyPropertyPages: ISpecifyPropertyPages;
  CAGUID              : TCAGUID;
  FilterInfo          : TFilterInfo;
  Hr                  : HRESULT;
begin
  Result := False;
  if not Assigned(DirectShowGraph) then
    Exit;
  try
    // Get filter using the name
    case (DirectShowGraph.FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2    : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph2).FindFilterByName(PWideChar(FilterName), BaseFilter);
      CDirectShowMethodGraphBuilder,
      CDirectShowMethodDvdGraphBuilder : Hr := (DirectShowGraph.FFilterGraphManager as IGraphBuilder).FindFilterByName(PWideChar(FilterName), BaseFilter);
      CDirectShowMethodFilterGraph     : Hr := (DirectShowGraph.FFilterGraphManager as IFilterGraph ).FindFilterByName(PWideChar(FilterName), BaseFilter);
      else                               Hr := E_NOTIMPL;
    end;
    if Succeeded(Hr) then
    begin
  //    ZeroMemory(@FilterInfo, SizeOf(TFilterInfo));
      // Get property pages
      Hr := BaseFilter.QueryInterface(IID_ISpecifyPropertyPages, SpecifyPropertyPages);
      if Succeeded(Hr) then
      begin
        // Get GUID
        Hr := SpecifyPropertyPages.GetPages(CAGUID);
        if Succeeded(Hr) then
        begin
          // Check number of pages available
          if CAGUID.cElems < 1 then
            Exit;
          if not QueryOnly then
          begin
            // Get info
            Hr := BaseFilter.QueryFilterInfo(FilterInfo);
            if Succeeded(Hr) then
            begin
              Hr := OleCreatePropertyFrame(Parent, 0, 0, FilterInfo.achName, 1, @BaseFilter, CAGUID.cElems, CAGUID.pElems, 0, 0, nil);
              if Succeeded(Hr) then
                Result := True;
            end;
          end
          else
            Result := True;    
        end;
      end;
    end;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Stop DirectShow interface
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowStop;
begin
  if Assigned(DirectShowGraph) then
    FreeAndNil(DirectShowGraph);
end;


{------------------------------------------------------------------------------
  Params  : <FullScreen>  True to set full screen mode
                          False to restore original mode
  Returns : -

  Descript: Set full screen mode
  Notes   : Only for non-VMR9 modes
 ------------------------------------------------------------------------------}
procedure DirectShowFullScreenMode(FullScreen: Boolean);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    if Assigned(DirectShowGraph.FVideoWindow) then
      DirectShowGraph.FVideoWindow.put_FullScreenMode(FullScreen);
  except
    DirectShowStop;
    ShowMessage('DirectShow FullScreenMode error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Video>  Video window
  Returns : -

  Descript: Set video window
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowSetVideo(Video: HWND);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    if (DirectShowGraph.FOptions and CDirectShowMethodVmrSpecials) <> CDirectShowMethodVideoRendererWindowless  then
      Exit;
    if not Assigned(DirectShowGraph.FWindowlessControl) then
      Exit;
    DirectShowGraph.FOwner := Video;
    case (DirectShowGraph.FOptions and CDirectShowMethodVmr) of
      CDirectShowMethodVideoRenderer7 : IVMRWindowlessControl (DirectShowGraph.FWindowlessControl).SetVideoClippingWindow(DirectShowGraph.FOwner);
      CDirectShowMethodVideoRenderer9 : IVMRWindowlessControl9(DirectShowGraph.FWindowlessControl).SetVideoClippingWindow(DirectShowGraph.FOwner);
    end;
  except
    DirectShowStop;
    ShowMessage('DirectShow SetVideo error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Resize to full client area
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowResize;
var
  DestinationRect: TRect;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    GetWindowRect(DirectShowGraph.FOwner, DestinationRect);
    DestinationRect.Right  := DestinationRect.Right  - DestinationRect.Left;
    DestinationRect.Bottom := DestinationRect.Bottom - DestinationRect.Top;
    DestinationRect.Top := 0;
    DestinationRect.Left := 0;
    if ((DirectShowGraph.FOptions and CDirectShowMethodVmrSpecials) = CDirectShowMethodVideoRendererWindowless) and
        (Assigned(DirectShowGraph.FWindowlessControl))  then
    begin
      case (DirectShowGraph.FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer  : if Assigned(DirectShowGraph.FVideoWindow) then
                                            DirectShowGraph.FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        CDirectShowMethodVideoRenderer7 : IVMRWindowlessControl (DirectShowGraph.FWindowlessControl).SetVideoPosition(nil, @DestinationRect);

        CDirectShowMethodVideoRenderer9 : IVMRWindowlessControl9(DirectShowGraph.FWindowlessControl).SetVideoPosition(nil, @DestinationRect);
      end;
    end
    else
      if Assigned(DirectShowGraph.FVideoWindow) then
        DirectShowGraph.FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right - DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
  except
    DirectShowStop;
    ShowMessage('DirectShow Resize error');
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Width>  Width of native video
            <Height> Height of native video

  Descript: Get native video size
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowGetSize(var Width: Integer; var Height: Integer);
var
  AWidth     : Integer;
  AHeight    : Integer;
//  MediaType  : AM_MEDIA_TYPE;
//  VideoHeader: PVideoInfoHeader;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
//    // Next could be used instead, but needs the correct PIN of the video decoder
//    DirectShowGraph.XXXXXXPin.ConnectionMediaType(MediaType);
//    if IsEqualGUID(MediaType.majortype, MEDIATYPE_Video) then
//    begin
//      // Get pointer to header info
//      VideoHeader := PVideoInfoHeader(MediaType.pbFormat);
//      Width  := VideoHeader.bmiHeader.biWidth;
//      Height := VideoHeader.bmiHeader.biHeight;
//    end;
    if ((DirectShowGraph.FOptions and CDirectShowMethodVmrSpecials) = CDirectShowMethodVideoRendererWindowless)  and
        (Assigned(DirectShowGraph.FWindowlessControl))  then
    begin
      case (DirectShowGraph.FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer  : if Assigned(DirectShowGraph.FBasicVideo) then
                                            DirectShowGraph.FBasicVideo.GetVideoSize(Width, Height);
        CDirectShowMethodVideoRenderer7 : IVMRWindowlessControl (DirectShowGraph.FWindowlessControl).GetNativeVideoSize(Width, Height, AWidth, AHeight);
        CDirectShowMethodVideoRenderer9 : IVMRWindowlessControl9(DirectShowGraph.FWindowlessControl).GetNativeVideoSize(Width, Height, AWidth, AHeight);
      end;
    end
    else
      if Assigned(DirectShowGraph.FBasicVideo) then
        DirectShowGraph.FBasicVideo.GetVideoSize(Width, Height)
  except
    DirectShowStop;
    ShowMessage('DirectShow GetSize error');
  end;
end;


{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <VideoPid>  Video PID
            <AudioPid>  Audio PID
  Returns : -

  Descript: Set new PIDs to use for MPEG2 demultiplexer
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer; AudioPid: Integer);
var
  MapPid: Cardinal;
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;

    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      (DirectShowGraph.FVideoPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      (DirectShowGraph.FAudioPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;

    // Set new active PIDs
    DirectShowGraph.FActiveVideoPid := VideoPid;
    DirectShowGraph.FActiveAudioPid := AudioPid;

    if DirectShowGraph.FActiveVideoPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveVideoPid;
      (DirectShowGraph.FVideoPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;
    if DirectShowGraph.FActiveAudioPid >= 0 then
    begin
      MapPid := DirectShowGraph.FActiveAudioPid;
      (DirectShowGraph.FAudioPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;

  except
    DirectShowStop;
    ShowMessage('DirectShow Mpeg2DemultiplexerSetNewPids error');
  end;
end;


{------------------------------------------------------------------------------
  Descript: DirectShow Universal Source filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <Buffer>        Pointer to data
            <BufferLength>  Length of valid data in buffer
  Returns : -

  Descript: Send data to Universal Source filter
  Notes   :
 ------------------------------------------------------------------------------}
procedure DirectShowUniversalSourceSendData(Buffer: PChar; BufferLength: Integer);
begin
  try
    if not Assigned(DirectShowGraph) then
      Exit;
    (DirectShowGraph.FUniversalSource as IUSRCSet).WSendSample(Buffer, BufferLength);
  except
    DirectShowStop;
    ShowMessage('DirectShow UniversalSourceSendData');
  end;
end;


{------------------------------------------------------------------------------
  Descript: <DvbFilter> functions: Either interanl function called or
            <DvbMajorPsi> DirectShow filter is called
  Note    : For descriptions of functions/procedures see <DvbFilter>
 ------------------------------------------------------------------------------}
function DvbGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
begin
  Result := DvbFilter.DvbFilterGetPid(StreamPacketData, Pid);
end;


function  DvbSetSignallingCallback     (Pid: Word;  hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallback(Pid, hWnd, Msg);
end;


function  DvbSetSignallingCallbackPat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackCat  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackCat  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackPmt  ( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPmt  (hWnd, Msg);
end;


function  DvbSetSignallingCallbackEvent( hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackEvent(hWnd, Msg);
end;


function  DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPidFilter(Pid, Filter);
end;


function  DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := DvbFilter.DvbFilterGetPidFilter(Pid);
end;


function  DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPsiFilter(Psi, Filter);
end;


function  DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := DvbFilter.DvbFilterGetPsiFilter(Psi);
end;


function  DvbGetProgramInfo(ProgramNumber: Byte;
                            var ServiceId: Word;
                            var PmtPid: Word;
                            var PcrPid: Word;
                            var VideoPids: TDvbPids;
                            var AudioPids: TDvbPids;
                            var TeletextPids: TDvbPids;
                            var SubtitlePids: TDvbPids;
                            var AudioLanguages: TDvbLanguages;
                            var SubtitleLanguages: TDvbLanguages;
                            var EcmPids: TDvbPids;
                            var CaIds: TDvbPids;
                            var ProgramName: string): Boolean;
begin
  Result := DvbFilter.DvbFilterGetProgramInfo(ProgramNumber, ServiceId,
              PmtPid, PcrPid, VideoPids, AudioPids,
              TeletextPids, SubtitlePids, AudioLanguages,
              SubtitleLanguages, EcmPids, CaIds, ProgramName);
end;


function  DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
begin
  Result := DvbFilter.DvbFilterGetEventInfo(ProgramNumber, ServiceId, Present, Events);
end;


function  DvbGetNumberOfPrograms: Byte;
begin
  Result := DvbFilter.DvbFilterGetNumberOfPrograms;
end;


function  DvbGetErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetErrors;
end;


function  DvbGetPacketSyncErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetPacketSyncErrors;
end;


procedure DvbResetErrors;
begin
  DvbFilter.DvbFilterResetErrors;
end;


procedure DvbCreateTables;
begin
  DvbFilter.DvbFilterCreateTables;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}
procedure InitializeUnit;
begin
  DirectShowGraph := nil;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization
  Notes   :
 ------------------------------------------------------------------------------}
procedure FinalizeUnit;
begin
  DirectShowStop;
end;


initialization
  InitializeUnit;

finalization
  FinalizeUnit;
end.
