{******************************************************************************}
{ FileName............: CommonInterfaceSessionLayer                            }
{ Project.............: DVB-S                                                  }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Common interface support (session layer)                                    }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ The session layer (on the command interface) manages session protocol objects}
{ which are between host and module or between modules.                        }
{                                                                              }
{ The layers involved:                                                         }
{                  APPLICATION                                                 }
{               Application layer                                              }
{          ****   Session layer                                                }
{                Transport layer                                               }
{                   Link layer                                                 }
{                 Physical layer                                               }
{                    HARDWARE                                                  }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040926 - Initial release                                           }
{******************************************************************************}
unit CommonInterfaceSessionLayer;

interface
uses
  Windows;

type
  TStates = (stUnassigned,             //
             stAssigned,               //
             stIdle,                   //
             stInCreation,             //
             stActive,                 //
             stInDeletion);            // 

  function CommonInterfaceSessionLayerCheckModule
             (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
  function CommonInterfaceSessionLayerResetModule
             (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;

implementation
uses
  Classes,
  CommonInterfaceTransportLayer,
  Saa7146aDebi,
  SyncObjs,
  SysUtils;

const
  // Session Layer objects
  CTagSessionNumber         = #$90;
  CTagOpenSessionRequest    = #$91;
  CTagOpenSessionResponse   = #$92;
  CTagCreateSession         = #$93;
  CTagCreateSessionResponse = #$94;
  CTagCloseSessionRequest   = #$95;
  CTagCloseSessionResponse  = #$96;

type

  TSessionThread = class(TThread)
  private
    FParent: TObject;                                      // Reference to parent
  protected
    procedure Execute; override;
  end;

  TSession = class(TObject)
  private
    { Private declarations }
    FState              : TStates;                         // Current state
    FModule             : Byte;                            // Module socket
    FSessionIdentifier  : Byte;                            // Transport identification
    FHandle             : THandle;                         // Handle to driver
    FThread             : TSessionThread;                  // Thread handling the sessions
    FReceived           : Word;                            // (Debug) counter for received answers
    function  GetState: TStates;
    procedure SetState(State: TStates);
    function  GetReceiveCounter: Word;
    procedure SetReceiveCounter(ReceiveCounter: Word);
  protected
  public
    constructor Create(SessionIdentification: Byte);
    destructor  Destroy; override;
  published
    property State: TStates read GetState write SetState;
    property ReceiveCounter: Word read GetReceiveCounter write SetReceiveCounter;
  end;


var
  CommonInterfaceLock : TCriticalSection;        // Prevents multiple operations which interfere
  Sessions            : array[1..128] of TSession;
  Loop                : Integer;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result> Current state

  Descript: Get current state of session
  Notes   :
------------------------------------------------------------------------------}
function TSession.GetState: TStates;
begin
  Result := FState;
end;


{------------------------------------------------------------------------------
  Params  : <State>  State to set
  Returns : -

  Descript: Set current state of session
  Notes   :
------------------------------------------------------------------------------}
procedure TSession.SetState(State: TStates);
begin
  FState := State;
  if FThread.Suspended then
    FThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Receive counter

  Descript: Get receive counter of session
  Notes   :
------------------------------------------------------------------------------}
function TSession.GetReceiveCounter: Word;
begin
  Result := FReceived;
end;


{------------------------------------------------------------------------------
  Params  : <ReceiveCounter>  Value to set
  Returns : -

  Descript: Set receive counter of session
  Notes   :
------------------------------------------------------------------------------}
procedure TSession.SetReceiveCounter(ReceiveCounter: Word);
begin
  FReceived := ReceiveCounter;
end;


{------------------------------------------------------------------------------
  Params  : <TransportIdentifcation>  Identifier for transport object
  Returns : -

  Descript: Creation for session protocol object
  Notes   :
------------------------------------------------------------------------------}
constructor TSession.Create(SessionIdentification: Byte);
begin
  inherited Create;
  FState := stUnassigned;                                  // Not is use
  FReceived := 0;
  FModule := 0;
  FSessionIdentifier := SessionIdentification;
  FHandle := INVALID_HANDLE_VALUE;
  FThread := TSessionThread.Create(True);
  FThread.FParent := Self;
  FThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destruction for session object
  Notes   :
------------------------------------------------------------------------------}
destructor TSession.Destroy;
begin
  if FState <> stUnassigned then
  begin
  end;
  if Assigned(FThread) then
  begin
    FThread.Terminate;
    FThread.Resume;
    FThread.WaitFor;
  end;
  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>               Handle of driver
  Returns : <Result>               True if connection identifier available
            <SessionIdentifier>    Allocated identifier
                                   0 = invalid

  Descript: Allocate a new session
  Notes   : Does not 'creates' the session. This is done
            by changing the 'state'.
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerAllocateSession(Handle: THandle; var SessionIdentifier: Byte): Boolean;
var
  Loop : Byte;
  Found: Boolean;
begin
  Found := False;
  SessionIdentifier := 0;
  Loop := Low(Sessions);
  repeat
    if Sessions[Loop].FState <> stUnassigned then
      Inc(Loop)
    else
    begin
      Found := True;
      SessionIdentifier := Loop;
    end;
  until Found or (Loop > High(Sessions));
  if Found then
  begin
    Sessions[Loop].FHandle := Handle;
    Sessions[Loop].State   := stAssigned;
    Sessions[Loop].FThread.Resume;
  end;
  Result := Found;
end;


{------------------------------------------------------------------------------
  Params  : <SessionIdentifier>     Connection to check
  Returns : <Result>                True if valid connection
            <State>                 Current state
            <ReceiveCounter>        Number of received replies from module
                                    (in Active state updated only)
  Descript: Get state of session
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerGetStatus(
           SessionIdentifier: Byte;
           var State: TStates;
           var ReceiveCounter: Word): Boolean;
begin
  Result := False;
  ReceiveCounter := 0;
  State := stUnassigned;
  if SessionIdentifier < Low(Sessions) then
    Exit;
  if SessionIdentifier > High(Sessions) then
    Exit;
  State := Sessions[SessionIdentifier].State;
  ReceiveCounter := Sessions[SessionIdentifier].ReceiveCounter;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <SessionIdentifier>     Session to set
            <State>                 State to set session to
                                    stInCreation or stInDeletion allowed
  Returns : <Result>                State of session

  Descript: Set state of session
  Notes   :
------------------------------------------------------------------------------}
procedure CommonInterfaceSessionLayerSetState(SessionIdentifier: Byte; State: TStates);
begin
  if SessionIdentifier < Low(Sessions) then
    Exit;
  if SessionIdentifier > High(Sessions) then
    Exit;
  case State of
    stInCreation,
    stInDeletion: Sessions[SessionIdentifier].State := State;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <SessionIdentifier>    Allocated identifier
  Returns : <Result>               True if success

  Descript: Release a session.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerReleaseSession(SessionIdentifier: Byte): Boolean;
begin
  CommonInterfaceSessionLayerSetState(SessionIdentifier, stInDeletion);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 Not used
            <TransportConnectionId>  Identifier to find data for
            <Module>                 Module to 'access'
            <Buffer>                 Pointer to buffer
            <BufferSize>             Available buffer size
  Returns : <Result>                 True if success
                                     False if buffersize smaller than needed
                                     (required buffer size in <BufferSize>
                                     False if module id error
            <BufferSize>             Returned buffer size (or required when
                                     buffer size error)

  Descript: Get data which correspondents with transport connection (and module).
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerReadData
           (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceTransportLayerReadData(Handle, TransportConnectionId, Module, Buffer, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                 <Saa7146aCreateFile> handle
            <TransportConnectionId>  Identifier
            <Module>                 Module to write to
            <Buffer>                 Pointer to buffer to write
            <BufferSize>             Size of buffer
  Returns : <Result>                 True if success
                                     False if module not ready

  Descript: Write data to module. Adds header information and splits it up
            in several packages if necessary.
  Notes   : The SESSION data is contained in the data field of the transport data
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerWriteData
  (Handle: THandle; TransportConnectionId: Byte; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceTransportLayerWriteData(Handle, TransportConnectionId, Module, Buffer, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Buffer>               Pointer to target data buffer
            <Tag>                  Tag
            <ConnectionIdentifier> Connection identifier
            <DataBytes>            Points to the optional data bytes (nil if none)
            <DataBytesLength>      Number of bytes in data bytes
  Returns : <Result>               True if success
            <BufferIndex>          Points to after the last valid data

  Descript: Assemble a Session Protocol Data Unit
  Notes   :
------------------------------------------------------------------------------}
...
function CommonInterfaceSessionLayerAssembleSPDU(Buffer: PChar; Tag: Byte; SessionObject: Byte; DataBytes: PChar; DataBytesLength: Word): Boolean;
var
  BufferLength: Dword;
  Loop        : Word;
begin
  Result := False;
  try
    Result := True;
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <ModuleIsPresent> True if module is available
            <ModuleHasData>   True if module has data available

  Descript: Check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerCheckModule
  (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result := CommonInterfaceTransportLayerCheckModule(Handle, Module, ModuleIsPresent, ModuleHasData);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <BufferSize>      Negotiated buffer size

  Descript: Reset and check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfaceSessionLayerResetModule
  (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;
begin
  CommonInterfaceLock.Acquire;
  try
    Result :=CommonInterfaceTransportLayerResetModule(Handle, Module, BufferSize);
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Remove all sessions.
  Notes   :
------------------------------------------------------------------------------}
procedure RemoveSessionConnections;
var
  Loop: Byte;
begin
  for Loop := Low(Sessions) to High(Sessions) do
    Sessions[Loop].Free;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Session thread
  Notes   :
------------------------------------------------------------------------------}
procedure TSessionThread.Execute;
const
  CBufferSize    = 2048;
  CTimeOut       = 300;                // Max allowed timeout is 300 ms
  CPollingPeriod = 10;                 // Max allowed polling period is 100 ms
var
  PreviousState  : TStates;
  NewState       : Boolean;
  Buffer         : PChar;
  DataBytes      : PChar;
  BufferSize     : Word;
  BufferIndex    : Word;
  Tag            : Byte;
  ConnectionId   : Byte;
  DataBytesLength: Word;
  Status         : Byte;
  DataAvailable  : Boolean;            // True when module has data available
  DataAwaiting   : Boolean;            // True when waiting for reply from module
  TimeOut        : Dword;
  PollTimeOut    : Dword;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'idle'
    Notes   : Reset variables and go into suspension
  ------------------------------------------------------------------------------}
  procedure StateIdle;
  begin
    DataAvailable := False;
    DataAwaiting  := False;
    Suspend;
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'in creation'
    Notes   : When this state is entered for the first time a 'Create_T_C' is
              send. We then wait for a correct reply from the module. If we don't
              get a correct reply within a certain time we revert to the idle
              state.
              We expect C_T_C_Reply which includes T_SB
  ------------------------------------------------------------------------------}
  procedure StateInCreation;
  begin
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'active'
    Notes   : We get here the first time after receiving a correct reply from
              the 'Create_T_C' (InCreation). From that point on we have different
              types of data we can expect. Since these are only send to us by
              the module when specifically requested, we need to poll the
              module regularly. We do this for each polling period if we are not already
              expecting data from the module.
              We expect T_Data_More, T_Data_Last, Request_T_C, Delete_T_C, T_SB
  ------------------------------------------------------------------------------}
  procedure StateActive;
  begin
  end;
  {------------------------------------------------------------------------------
    Params  : -
    Returns : -

    Descript: State 'in deletion'
    Notes   : We expect D_T_C_Reply which includes T_SB
  ------------------------------------------------------------------------------}
  procedure StateInDeletion;
  begin
  end;

begin
  GetMem(Buffer, CBufferSize);
  GetMem(DataBytes, CBufferSize);
  try
    PreviousState := stUnassigned;
    repeat
      // Update state change
      NewState := (TSession(FParent).FState <> PreviousState);
      PreviousState := TSession(FParent).FState;
      // State machine
      case TSession(FParent).FState of
        stAssigned  : TSession(FParent).FState := stIdle;
        stIdle      : StateIdle;
        stInCreation: StateInCreation;
        stActive    : StateActive;
        stInDeletion: StateInDeletion;
        stUnassigned: Suspend;
      end;
      Sleep(1);
    until Terminated;
  finally
    FreeMem(Buffer, CBufferSize);
    FreeMem(DataBytes, CBufferSize);
  end;
end;


initialization
  CommonInterfaceLock   := TCriticalSection.Create;
  for Loop := Low(Sessions) to High(Sessions) do
    Sessions[Loop] := TSession.Create(Loop);


finalization
  RemoveSessionConnections;
  CommonInterfaceLock.Free;
end.

