{******************************************************************************}
{ FileName............: CommonInterfacePhysicalLayer                           }
{ Project.............: DVB-S                                                  }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Common interface support (physical layer)                                   }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ The physical layer is the only part of the common interface which is 'aware' }
{ of the hardware involved.                                                    }
{ The physical layer handles the following:                                    }
{ . the presence of a module                                                   }
{ . the initialization of a module                                             }
{ . allocation/releasing of buffer memory                                      }
{                                                                              }
{ Note: Detecting modules is probably the incorrect way, but since in all cases}
{       only one module is to be present the method works. The extended method }
{       is for future expansion only.                                          }
{       Also the reset of the module resets all modules (if this were possible)}
{                                                                              }
{ The layers involved:                                                         }
{                  APPLICATION                                                 }
{               Application layer                                              }
{                 Session layer                                                }
{                Transport layer                                               }
{                   Link layer                                                 }
{         ****    Physical layer                                               }
{         ****       HARDWARE                                                  }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030711 - Initial release                                           }
{******************************************************************************}
unit CommonInterfacePhysicalLayer;

interface
uses
  Windows;

  function CommonInterfacePhysicalLayerWriteData
             (Handle: THandle; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
  function CommonInterfacePhysicalLayerReadData
             (Handle: THandle; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
  function CommonInterfacePhysicalLayerCheckModule
             (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
  function CommonInterfacePhysicalLayerResetModule
             (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;

implementation
uses
  Saa7146aDebi,
  SyncObjs,
  SysUtils;

const
  // Common interface
  CCiStatus        = $0000;                      // Status register
  CCiControl       = $0000;                      // Control register
                                                 // Note: type 1 is used to access the
                                                 //       module hardware registers
  CCiBaseAddressType1 : array[0..3] of Word =    // Base addresses for modules (type 1)
                     ($1000, $5000, $9000, $D000);
  CCiBaseAddressType2 : array[0..3] of Word =    // Base addresses for modules (type 2)
                     ($2000, $6000, $A000, $E000);
  CCiBaseAddressType0 : array[0..3] of Word =    // Base addresses for modules (type 0)
                     ($3000, $7000, $B000, $F000);

  // Status of common interface
  CCiModuleEnable     = $01;                     // Indicates module(s) is/are enabled
  CCiTransportEnabled = $02;                     // Indicates transport stream is enabled
  CCiModulePresent    : array[0..3] of Byte =    // Indicates if a module is present (inserted)
                     ($08, $10, $20, $40);       // Note: not sure $10/$20/$40 exist

  CCiEnableModule     = $01;                     // Enable modules
  CCiEnableTransport  = $02;                     // Enable transport stream
  CCiEnableInterrupt1 = $40;                     // Enable interrupts module free to accept data (?)
  CCiEnableInterrupt2 = $80;                     // Enable interrupts data available (module -> host) (?)

  // Module
  // Hardware register base (base address for modules type 1)
  // There are only four registers
  CModDataRegister    = $0;
  CModCommandRegister = $1;
  CModStatusRegister  = $1;
  CModSizeRegisterLs  = $2;
  CModSizeRegisterMs  = $3;

  // Command register bits
  CModCommandReset       = $08;                  // Resets the module
  CModCommandSizeRead    = $04;                  // Asks module to provide max buffer size
  CModCommandSizeWrite   = $02;                  // Tells the module what buffer size to use
  CModCommandHostControl = $01;                  // Indicates host -> module transfer

  // Status register bits
  CModStatusDataAvailable = $80;                 // Indicates module has some data
  CModStatusFree          = $40;                 // Indicates module is free to accept data
  CModStatusWriteError    = $02;                 // Error in write length
  CModStatusReadError     = $01;                 // Error in read length

type
  TModule = class(TObject)
  private
    { Private declarations }
    FBufferIn        : PChar;                    // Transfer buffer in  (note: a single buffer is also allowed)
    FBufferOut       : PChar;                    // Transfer buffer out
    FBufferSize      : Word;                     // Size of transfer buffers
    FBaseAddressType0: Word;                     // Base address of module
    FBaseAddressType1: Word;                     // Base address of module (hardware register interface)
    FBaseAddressType2: Word;                     // Base address of module
  protected
  public
  published
  end;


var
  CommonInterfaceLock: TCriticalSection;         // Prevents multiple operations which interfere
  Modules            : array[0..3] of TModule;   // Modules supported
  Loop               : Integer;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Reset common interface.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerReset(Handle: THandle): Boolean;
begin
  Result := False;
  CommonInterfaceLock.Acquire;
  try
    if not Saa7146aWriteByteToDebi(Handle, CCiControl, 0) then
      Exit;
    Sleep(1);
    if not Saa7146aWriteByteToDebi(Handle, CCiControl, CCiEnableModule) then
      Exit;
    // For now also enable the transport stream
    if not Saa7146aWriteByteToDebi(Handle, CCiControl, CCiEnableModule or CCiEnableTransport) then
      Exit;
    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>      <Saa7146aCreateFile> handle
            <BaseAddress> Base address of module to access
            <Buffer>      Buffer to store data to
            <BufferSize>  Buffer size
  Returns : <Result>      True if success
            <BufferSize>  Data bytes read

  Descript: Read data from the hardware interface register (if data available).
  Notes   : The module is checked if it has data to be read. If so it is read.
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerReadCommandInterface(Handle: THandle; BaseAddress: Word; Buffer: PChar; var BufferSize: Word): Boolean;
var
  Data     : Byte;
  DataLs   : Byte;
  DataMs   : Byte;
  DataIndex: Word;
begin
  Result := False;                                         // Assume we fail

  CommonInterfaceLock.Acquire;
  try
    // Check if data available
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
      Exit;
    // If no data available, exit without an error
    if (Data and CModStatusDataAvailable) = 0 then
    begin
      BufferSize := 0;
      Result := True;
      Exit;
    end;

    // Read the size the module reports back
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModSizeRegisterLs, DataLs) then
      Exit;
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModSizeRegisterMs, DataMs) then
      Exit;
    // Check size  
    if ((DataMs shl 8) or DataLs) > BufferSize then
      Exit;
    // Combine LS and MS
    BufferSize := (DataMs shl 8) or DataLs;

    // Now read the data from the module
    for DataIndex := 0 to BufferSize - 1 do
    begin
      // Get data
      if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModDataRegister, Data) then
        Exit;
      Buffer[DataIndex] := Chr(Data);
    end;

    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>      <Saa7146aCreateFile> handle
            <BaseAddress> Base address of module to access
            <Buffer>      Buffer to write
            <BufferSize>  Data bytes to write
  Returns : <Result>      True if success (written)

  Descript: Write data to the hardware interface register (if allowed).
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerWriteCommandInterface(Handle: THandle; BaseAddress: Word; Buffer: PChar; BufferSize: Word): Boolean;
var
  Data     : Byte;
  DataIndex: Word;
begin
  Result := False;                                         // Assume we fail

  CommonInterfaceLock.Acquire;
  try
    // Check if allowed: we are not allowed to write when the module has
    // some pending data to be send
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
      Exit;
    // If module itself has data pending, exit with an error
    if (Data and CModStatusDataAvailable) <> 0 then
      Exit;

    // Indicate host to module transfer
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, CModCommandHostControl) then
      Exit;
    // Check status
    if not Saa7146aReadByteFromDebi(Handle, baseAddress + CModStatusRegister, Data) then
      Exit;
    // If not allowed, exit with an error
    if (Data and CModStatusFree) = 0 then
      Exit;

    // Write the size the module reports back
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModSizeRegisterLs, Lo(BufferSize)) then
      Exit;
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModSizeRegisterMs, Hi(BufferSize)) then
      Exit;

    // Now write the data from the module
    for DataIndex := 0 to BufferSize - 1 do
    begin
      // Get data
      Data := Ord(Buffer[DataIndex]);
      if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModDataRegister, Data) then
        Exit;
    end;
    // Indicate end of host to module transfer
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, 0) then
      Exit;

    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>      <Saa7146aCreateFile> handle
            <BaseAddress> Base address of module to access
  Returns : <Result>      True if success
            <DataSize     Negociated data size

  Descript: Initialize common interface. Includes negotiation of buffer size.
  Notes   : The process includes both resetting of the common interface
            and the initialization process of the module (as per EN 50221)
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerInitialize(Handle: THandle; BaseAddress: Word; var DataSize: Word): Boolean;
var
  StartTime: Dword;
  Data     : Byte;
  DataBytes: array[0..255] of Byte;
  DataIndex: Byte;
  DataLs   : Byte;
  DataMs   : Byte;
begin
  Result := False;                                         // Assume we fail

  DataSize := 0;
  CommonInterfaceLock.Acquire;
  try
    // First we have to put the interface in a defined state using the reset
    // First clear the command register so the reset bit will always change
    // from it's inactive state into the active state.

    // First reset the common interface itself
    if not CommonInterfacePhysicalLayerReset(Handle) then
      Exit;

    // Next write the reset to the (DVB-S) hardware interface register
    // First make sure a zero state is set
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, 0) then
      Exit;
    // Now issue the reset
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, CModCommandReset) then
      Exit;
    StartTime := GetTickCount;
    // Wait for acknowledge (free) with timeout
    repeat
      if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
        Exit;
    until ((Data and CModStatusFree) <> 0) or
          (Abs(GetTickCount - StartTime) > 20);
    if ((Data and CModStatusFree) = 0) then
      Exit;

    // After this reset we must find out the maximum allowed buffer size
    // First ask the module for it's size
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, CModCommandSizeRead) then
      Exit;
    StartTime := GetTickCount;
    // Wait for acknowledge (data available) with timoeut
    repeat
      if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
        Exit;
    until ((Data and CModStatusDataAvailable) <> 0) or
          (Abs(GetTickCount - StartTime) > 20);
    if ((Data and CModStatusDataAvailable) = 0) then
      Exit;

    // Read the size the module reports back
    // We expect only two bytes here ....
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModSizeRegisterLs, DataLs) then
      Exit;
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModSizeRegisterMs, DataMs) then
      Exit;
    // Combine LS and MS
    DataSize := (DataMs shl 8) or DataLs;
    // Sanity check: we should have had 2 bytes
    if DataSize <> 2 then
      Exit;
    // Now read the 2 bytes of data from the module
    for DataIndex := 0 to DataSize - 1 do
    begin
      // Get data
      if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModDataRegister, DataBytes[DataIndex]) then
        Exit;
    end;
    // Reset size read request
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, 0) then
      Exit;
    // Now get the modules reported size which is the size we will be using
    DataSize := (DataBytes[0] shl 8) or DataBytes[1];

    // Now report this back to the module
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, CModCommandSizeWrite) then
      Exit;
    StartTime := GetTickCount;
    // Wait for free with timoeut
    repeat
      if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
        Exit;
    until ((Data and CModStatusFree) <> 0) or
          (Abs(GetTickCount - StartTime) > 20);
    if ((Data and CModStatusFree) = 0) then
      Exit;
    // Indicate host to module transfer too
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister,
             CModCommandHostControl or CModCommandSizeWrite) then
      Exit;
    // Recheck free status
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
      Exit;
    if (Data and CModStatusFree) <> 0 then
    begin
      // Initiate transfer writing the size of the transfer
      // Size is 2 bytes
      if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModSizeRegisterLs, 2) then
        Exit;
      if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModSizeRegisterMs, 0) then
        Exit;
      // Now transmit the 2 bytes
      if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModDataRegister, Hi(DataSize)) then
        Exit;
      if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModDataRegister, Lo(DataSize)) then
        Exit;
    end;
    // Indicate end of host to module transfer
    if not Saa7146aWriteByteToDebi(Handle, BaseAddress + CModCommandRegister, 0) then
      Exit;
    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Module>  Module to check
            <Present> True when a module is present
  Returns : <Result>  True if success

  Descript: Returns the presence of the module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerModulePresent(Handle: THandle; Module: Byte; var Present: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  if not Module in [Low(CCiModulePresent)..High(CCiModulePresent)] then
    Exit;
  CommonInterfaceLock.Acquire;
  try
    if not Saa7146aReadByteFromDebi(Handle, CCiStatus, Data) then
      Exit;
    Present := (Data and CCiModulePresent[Module]) <> 0;
    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>       <Saa7146aCreateFile> handle
            <BaseAddress>  Base address of moudle
            <HasData>      True when the module has data available
  Returns : <Result>       True if success

  Descript: Returns the data availability of the module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerModuleHasData(Handle: THandle; BaseAddress: Word; var HasData: Boolean): Boolean;
var
  Data: Byte;
begin
  Result := False;

  CommonInterfaceLock.Acquire;
  try
    // Check if data available
    if not Saa7146aReadByteFromDebi(Handle, BaseAddress + CModStatusRegister, Data) then
      Exit;
    HasData := (Data and CModStatusDataAvailable) <> 0;
    Result := True;
  finally
    CommonInterfaceLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Moduel> Module to remove
  Returns : -

  Descript: Remove the module (from memory).
  Notes   :
------------------------------------------------------------------------------}
procedure CommonInterfacePhysicalLayerRemoveModule(Module: Byte);
begin
  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  if Assigned(Modules[Module]) then
  begin
    try
      if Assigned(Modules[Module].FBufferIn) then
        FreeMem(Modules[Module].FBufferIn, Modules[Module].FBufferSize);
      if Assigned(Modules[Module].FBufferOut) then
        FreeMem(Modules[Module].FBufferOut, Modules[Module].FBufferSize);
    finally
      FreeAndNil(Modules[Module]);
    end;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if success

  Descript: Remove all modules.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerRemoveModules: Boolean;
var
  Loop: Byte;
begin
  for Loop := Low(Modules) to High(Modules) do
    CommonInterfacePhysicalLayerRemoveModule(Loop);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Module>  Module to add
  Returns : <Result>  True if success (module has been added)

  Descript: Add the module (memory and initialization) if present.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerAddModule(Handle: THandle; Module: Byte): Boolean;
var
  ModuleIsPresent: Boolean;
begin
  Result := False;

  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  // Remove the module if already present
  CommonInterfacePhysicalLayerRemoveModule(Module);
  // Create the module
  Modules[Module] := TModule.Create;
  Modules[Module].FBufferSize       := 0;
  Modules[Module].FBufferIn         := nil;
  Modules[Module].FBufferOut        := nil;
  Modules[Module].FBaseAddressType0 := CCiBaseAddressType0[Module];
  Modules[Module].FBaseAddressType1 := CCiBaseAddressType1[Module];
  Modules[Module].FBaseAddressType2 := CCiBaseAddressType2[Module];

  // Check if module is physically present
  CommonInterfacePhysicalLayerModulePresent(Handle, Module, ModuleIsPresent);
  // If physically present: initialize it
  if ModuleIsPresent then
  begin
    if CommonInterfacePhysicalLayerInitialize(Handle, Modules[Module].FBaseAddressType1, Modules[Module].FBufferSize) then
    begin
      if Modules[Module].FBufferSize = 0 then
        Exit;
      // We succeeded in resetting the module and acquired the buffer size
      // We now have to allocate the memory
      // Note that we could just use a single buffer since data flow is
      // in a defined direction
      GetMem(Modules[Module].FBufferIn,  Modules[Module].FBufferSize);
      GetMem(Modules[Module].FBufferOut, Modules[Module].FBufferSize);
    end;
  end
  else
    CommonInterfacePhysicalLayerRemoveModule(Module);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <ModuleIsPresent> True if module is available
            <ModuleHasData>   True if module has data available

  Descript: Check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerCheckModule
  (Handle: THandle; Module: Byte; var ModuleIsPresent: Boolean; var ModuleHasData: Boolean): Boolean;
begin
  Result := False;

  if not Module in [Low(Modules)..High(Modules)] then
    Exit;

  if Assigned(Modules[Module]) then
  begin
    // The module exists (memory allocated) and was present, now check if it is present
    if not CommonInterfacePhysicalLayerModulePresent(Handle, Module, ModuleIsPresent) then
      Exit;
    // Check if the card status has changed (removed)
    if not ModuleIsPresent then
    begin
      // If removed, release memory and report not present
      CommonInterfacePhysicalLayerRemoveModule(Module);
      ModuleHasData := False;
      Result := True;
      Exit;
    end;
  end
  else
    begin
      // Add module and initialize it
      if not CommonInterfacePhysicalLayerAddModule(Handle, Module) then
        Exit;
      if not CommonInterfacePhysicalLayerModulePresent(Handle, Module, ModuleIsPresent) then
        Exit;
    end;
  // At this point we know if the module is present or not
  // Check if data available
  if ModuleIsPresent then
    if not CommonInterfacePhysicalLayerModuleHasData(Handle, Modules[Module].FBaseAddressType1, ModuleHasData) then
      Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
  Returns : <Result>          True if success
            <BufferSize>      Negotiated buffer size

  Descript: Reset and check status of module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerResetModule
  (Handle: THandle; Module: Byte; var BufferSize: Word): Boolean;
var
  ModuleIsPresent: Boolean;
  ModuleHasData  : Boolean;
begin
  Result := False;

  BufferSize := 0;

  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  // For a reset we simply remove the module from memory.
  CommonInterfacePhysicalLayerRemoveModule(Module);
  // The check will initialize the module
  if CommonInterfacePhysicalLayerCheckModule(Handle, Module, ModuleIsPresent, ModuleHasData) then
  begin
    if ModuleIsPresent then
      BufferSize := Modules[Module].FBufferSize;
    Result := True;
  end;

end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to write to
            <Buffer>          Pointer to buffer to write
            <BufferSize>      Size of buffer
  Returns : <Result>          True if success
                              False if buffersize larger than negotiated size
                              False if module not ready

  Descript: Write data to module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerWriteData
  (Handle: THandle; Module: Byte; Buffer: PChar; BufferSize: Word): Boolean;
begin
  Result := False;

  // Check valid module
  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  if not Assigned(Modules[Module]) then
    Exit;
  // Check buffer size
  if BufferSize > Modules[Module].FBufferSize then
    Exit;

  Result := CommonInterfacePhysicalLayerWriteCommandInterface(
              Handle, Modules[Module].FBaseAddressType1, Buffer, BufferSize);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Module>          Module to check
            <Buffer>          Pointer to buffer to read to
            <BufferSize>      Size of buffer
  Returns : <Result>          True if success
                              False if buffersize smaller than needed (data is lost)
                              False if module not ready
            <BufferSize>      Data read

  Descript: Read data from module.
  Notes   :
------------------------------------------------------------------------------}
function CommonInterfacePhysicalLayerReadData
  (Handle: THandle; Module: Byte; Buffer: PChar; var BufferSize: Word): Boolean;
begin
  Result := False;

  // Check valid module
  if not Module in [Low(Modules)..High(Modules)] then
    Exit;
  if not Assigned(Modules[Module]) then
    Exit;
  Result := CommonInterfacePhysicalLayerReadCommandInterface(
              Handle, Modules[Module].FBaseAddressType1, Buffer, BufferSize);
end;


initialization
  CommonInterfaceLock   := TCriticalSection.Create;
  for Loop := Low(Modules) to High(Modules) do
    Modules[Loop] := nil;

    
finalization
  CommonInterfacePhysicalLayerRemoveModules;
  CommonInterfaceLock.Free;
end.

