{******************************************************************************}
{ FileName............: UsingRPSUnit001                                        }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Infrared basic test with actual key code detection.                         }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ The infrared interface uses a dedicated uP and is connected to the GPIO3     }
{ input. When an infrared command is detected by this uP it activates this     }
{ line after which the code can be read through the DEBI interface.            }
{ In this example we use a RPS program which runs continuously. This RPS waits }
{ for a GPIO3 activation and then reads the infrared data.                     }
{ Here no direct GPIO interrupts are involved because of the continuous        }
{ 'polling' by means of the RPS program.                                       }
{ Note that the infrared uP activates GPIO3 three times for the total          }
{ information of the infrared data.                                            }
{ Two of the data words (low byte) contain two key codes itself, but with the  }
{ high nibbles changed: eg. $C4 and $44                                        }
{                           $Cx <> $4x                                         }
{                           $Dx <> $5x                                         }
{                           $Ex <> $6x                                         }
{                           $Fx <> $7x                                         }
{ These two key codes are the last detected keys. Typically these are the same }
{ except when a key is pressed very shortly.                                   }
{ The key code starts at $40 (command 0) and ends at $7F (command 63)          }
{ Extended commands (64..127) are not supported, meaning that command 64 will  }
{ be interpreted as command 0, and command 127 as 63.                          }
{ The third contains the toggle code and the device (low nibble)               }
{ eg. $x3 indicates device '3'                                                 }
{ The toggle adds $20 to the device code.                                      }
{ Depending on the address used to read out the uP the data received can be    }
{ different and the auto-repeat can be either set on or off.                   }
{ In short:                                                                    }
{   Data < $40 == device/repeat                                                }
{   Data >=$40 == key code                                                     }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030524 - Initial WDM release                                       }
{******************************************************************************}
unit UsingRPSUnit001;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Saa7146AInterface, Saa7146aGpio, Saa7146AI2c, Stv0299bRegisters,
  Tsa5059Registers, ExtCtrls,
  Saa7146aIoControl,
  Saa7146aRegisters, Mask;

const
  CInfraredAddress = $4000;
type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    tmrUpdate: TTimer;
    txtInterrupts: TStaticText;
    btnActivate: TButton;
    StaticText1: TStaticText;
    mskAddress: TMaskEdit;
    StaticText2: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure btnActivateClick(Sender: TObject);
  private
    { Private declarations }
    FCardHandle: THandle;
    FRPS0      : TSaa7146aDmaBuffer;             // DMA buffer for RPS0 program
    FRPSData   : TSaa7146aDmaBuffer;             // DMA buffer for RPS data
    FKey1      : Byte;
    FKey2      : Byte;
    FRepeat    : Boolean;
    FDevice    : Byte;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.DFM}


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  MajorVersion : Word;
  MinorVersion : Word;
  Build        : Dword;
  DeviceName   : PChar;
begin
  FCardHandle := INVALID_HANDLE_VALUE;
  FKey1 := 0;
  FKey2 := 1;
  FRepeat := True;
  FDevice := 99;
  // Check if the driver is present and/or multiple cards are present
  case Saa7146aGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No SAA7146A driver and/or card detected.');
    1:   mmoDriver.Lines.Add('One SAA7146A card detected.');
    else mmoDriver.Lines.Add(format('%d SAA7146A cards detected.', [Saa7146aGetNumberOfCards]));
  end;
  // If any card detected then display additional information
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    // At this time we don't have a link to the driver itself. We have to create this.
    // We use the first available card (-1==auto) there is.
    FCardHandle := Saa7146aCreateFile(-1);
    try
      // Now with the handle we can actually access the driver.
      // Start with getting the version information of the driver.
      GetMem(DeviceName, 128);
      if Saa7146aGetDriverVersion(FCardHandle, @MajorVersion, @MinorVersion, @Build, DeviceName) then
        mmoDriver.Lines.Add(format('Driver "%s", Version %d.%d, Build %8.8x.', [DeviceName, MajorVersion, MinorVersion, Build]));
      FreeMem(DeviceName, 128);
      // The RESET of the tuner is connected to a GPIO line of the SAA7146A, so we have to configure this
      if not Saa7146aInitializeGpio(FCardHandle, True) then
      begin
        mmoDriver.Lines.Add('Could not initialize GPIO.');
        Exit;
      end;
      // Now issue a reset signal
      if not Saa7146aIssueReset(FCardHandle, 2) then
      begin
        mmoDriver.Lines.Add('Could not issue a reset.');
        Exit;
      end;
      mmoDriver.Lines.Add('Tuner has been reset.');
    finally
//      Saa7146aCloseHandle(FCardHandle);
    end;
  end;
  if FCardHandle <> INVALID_HANDLE_VALUE then
    tmrUpdate.Enabled := True;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
var
  Channel: Dword;
begin
  tmrUpdate.Enabled := False;
  // Release all DMA memory
  Channel := 0;
  while Saa7146aReleaseDma(FCardHandle, Channel) do
    Inc(Channel);
  Saa7146aCloseHandle(FCardHandle);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Timer update
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  TransferBuffer : array[0..2] of Dword;
  InfraredData   : array[0..2] of Byte;
  Transfer       : TSaa7146aTransferBuffer;
  Key1           : Byte;
  Key2           : Byte;
  RepeatCode     : Boolean;
  Device         : Byte;
  Loop           : Integer;
  FirstKey       : Boolean;
  SecondKey      : Boolean;
  HasDevice      : Boolean;
  Valid          : Boolean;
begin
  // Setup transfer data and transfer it
  Transfer.Identifier := FRPSData.Identifier;
  Transfer.SourceIndex  := 0;
  Transfer.TargetIndex  := 0;
  Transfer.TransferAddress := @TransferBuffer[0];
  Transfer.TransferLength := Sizeof(TransferBuffer);      // Transfer data
  Saa7146aReadFromDma(FCardHandle, Transfer);
  // We have received three data items from the remote. Now seperate them.
  RepeatCode := False;
  Key1 := 0;
  Key2 := 0;
  Device := 0;
  FirstKey  := False;
  SecondKey := False;
  HasDevice := False;
  Valid     := True;
  InfraredData[0] := Swap(TransferBuffer[0]) and $FF;
  InfraredData[1] := Swap(TransferBuffer[1]) and $FF;
  InfraredData[2] := Swap(TransferBuffer[2]) and $FF;
  for Loop := 0 to 2 do
  begin
    if InfraredData[Loop] < $40 then
    begin
      if HasDevice then                                    // Check for more than 1 entry == invalid
        Valid := False;
      HasDevice := True;
      RepeatCode := (InfraredData[Loop] and $20) <> 0;
      if RepeatCode then
        Device := InfraredData[Loop] - $20
      else
        Device := InfraredData[Loop];
    end
    else
      if InfraredData[Loop] < $80 then
      begin
        if FirstKey then                                   // Check for more than 1 entry == invalid
          Valid := False;
        FirstKey := True;
        Key1 := InfraredData[Loop] - $40;
        Key1 := Key1 and $7F;
      end
      else
        begin
          if SecondKey then                                // Check for more than 1 entry == invalid
            Valid := False;
          SecondKey := True;
          Key2 := InfraredData[Loop] - $40;
          Key2 := Key2 and $7F;
        end;
  end;
  if Valid then
    txtInterrupts.Caption := format('%4.4x %4.4x %4.4x', [TransferBuffer[0] and $FFFF, TransferBuffer[1] and $FFFF, TransferBuffer[2] and $FFFF])
  else
    txtInterrupts.Caption := 'Invalid data';
  if (Key1 = Key2) and
      Valid then
  begin
    if (FRepeat <> RepeatCode) or
       (FDevice <> Device) then
    mmoDriver.Lines.Add(format('Key pressed -> RC5 %d %d', [Device, Key1]));
    FKey1 := Key1;
    FKey2 := Key2;
    FDevice := Device;
    FRepeat := RepeatCode;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: (De)activate infrared
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnActivateClick(Sender: TObject);
var
  Channel       : Dword;
  Status        : TSaa7146aDmaBuffer;
  TransferBuffer: array[0..50] of Dword;
  Index         : Integer;
  Transfer      : TSaa7146aTransferBuffer;
  Address       : Dword;
  LAddress      : Word;
begin
  if btnActivate.Tag = 0 then
  begin
    LAddress := CInfraredAddress;
    // Assume we are the ONLY one allocating DMA using the driver
    // Release any left-overs from (incomplete) runnings of the application
    // If we DON'T do this we eventually have all buffers used (and allocated memory for this),
    // so we won't be able to allocate any DMA (driver needs to restart in such a case)
    Channel := 0;
    while Saa7146aReleaseDma(FCardHandle, Channel) do
      Inc(Channel);
    // Disable all interrupts
    Saa7146aWriteToSaa7146aRegister(FCardHandle, CSaa7146aIer, 0);

    // Allocate some DMA buffers which will hold the RPS program
    if not (Saa7146aAllocateDma(FCardHandle, DMASIZE * 1, Status)) then
      Exit;
    // Save info of allocated memory
    FRPS0.Identifier      := Status.Identifier;
    FRPS0.VirtualAddress  := Status.VirtualAddress;
    FRPS0.PhysicalAddress := Status.PhysicalAddress;
    FRPS0.Size            := Status.Size;
    // Allocate one DMA buffer for returned data
    if not (Saa7146aAllocateDma(FCardHandle, DMASIZE * 1, Status)) then
      Exit;
    // Save info of allocated memory
    FRPSData.Identifier      := Status.Identifier;
    FRPSData.VirtualAddress  := Status.VirtualAddress;
    FRPSData.PhysicalAddress := Status.PhysicalAddress;
    FRPSData.Size            := Status.Size;

    // Now setup the RPS program. We first construct it in local memory and
    // then transport it to the SAA7146
    Index    := 0;
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 0;                                                                // Target index
    TransferBuffer[Index] := DWord(FRPSData.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 4;                                                                // Target index
    TransferBuffer[Index] := DWord(FRPSData.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsGpio3;             // Wait for GPIO3 being activated
    Inc(Index);
    // Do a DEBI transfer
    TransferBuffer[Index] := CSaa7146aRpsLdReg or (CSaa7146aDebiCommand shr 2);  // Write DEBI command and address to shadow RAM
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aDebiCmdRd or (2 shl 17) or LAddress;       // Read, Address, blocklength=2
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsClrSignal or CSaa7146aRpsDebi;          // Reset "shadow uploaded" flag
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsUpload or CSaa7146aRpsDebi;             // Invoke shadow upload
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsNop;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsDebi;              // Wait for shadow upload to finish
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsStReg or ((CSaa7146aDebiAd-4) shr 2);   // Get DEBI data
    Inc(Index);
    Address := 8;                                                                // Target index
    TransferBuffer[Index] := DWord(FRPSData.PhysicalAddress.LowPart) + Address;
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsPause or CSaa7146aRpsInvert or CSaa7146aRpsGpio3;             // Wait for GPIO3 being deactivated
    Inc(Index);
    TransferBuffer[Index] := CSaa7146aRpsJump;                                   // Restart program
    Inc(Index);
    TransferBuffer[Index] := DWord(FRPS0.PhysicalAddress.LowPart);               // Address of start program
    Inc(Index);

    // Setup transfer data and transfer it
    Transfer.Identifier := FRPS0.Identifier;
    Transfer.SourceIndex  := 0;
    Transfer.TargetIndex  := 0;
    Transfer.TransferAddress := @TransferBuffer[0];
    Transfer.TransferLength := Index * Sizeof(DWord);
    Saa7146aWriteToDma(FCardHandle, Transfer);             // Transfer data
    // Now make sure the RPS program is started
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aRpsAddr0,
      DWord(DWord(FRPS0.PhysicalAddress.LowPart)));        // Set physical start of RPS program
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aRpsPage0,
      0);                                                  // RPS program performs no explicit mem writes
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aRpsTov0 ,
      0);                                                  // Disable RPS timeouts
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps0On);                                 // Start RPS0 program


    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aDebiConfig,
//      CSaa7146aDebiCfgIntel or
      CSaa7146aDebiCfg16);                                 // Setup DEBI transfer type (MOTOROLA/INTEL does not care)
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aDebiPage,
      CSaa7146aDebiPageDisable);                           // No page check
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1DebiEnable);                             // Enable DEBI transfers
      
    btnActivate.Tag := 1;
    btnActivate.Caption := 'Disable infrared';
  end
  else
  begin
    Saa7146aWriteToSaa7146aRegister(FCardHandle,
      CSaa7146aMc1,
      CSaa7146aMc1Rps0Off);                                // Stop RPS0 program
    btnActivate.Tag := 0;
    btnActivate.Caption := 'Enable infrared';
  end;
end;

end.
