{******************************************************************************}
{ FileName............: CheckingBasicInterruptUnit001                          }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Infrared basic test.                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Here we use the recording capability of the driver to check if the infrared  }
{ interface of the interface card is operating.                                }
{ The infrared interface uses a dedicated uP and is connected to the GPIO3     }
{ input. When an infrared command is detected by this uP it activates this     }
{ line after which the code can be read through the DEBI interface.            }
{ In this example we only activate interrupts for GPIO3 and we use the counting}
{ capability of the driver to check if we receive anything (indicated by the   }
{ the GPIO3 interrupts).                                                       }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030523 - Initial WDM release                                       }
{******************************************************************************}
unit CheckingBasicInterruptUnit001;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Saa7146AInterface, Saa7146aGpio, Saa7146AI2c, Stv0299bRegisters,
  Tsa5059Registers, ExtCtrls,
  Saa7146aIoControl,
  Saa7146aRegisters;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    tmrUpdate: TTimer;
    txtInterrupts: TStaticText;
    btnActivate: TButton;
    StaticText1: TStaticText;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure btnActivateClick(Sender: TObject);
  private
    { Private declarations }
    FCardHandle: THandle;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Remove interrupt for infrared (GPIO3)
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aDisableInfrared(Handle: THandle): Boolean;
var
  Loop       : Integer;
  IrqHandling: TSaa7146aIrqTransferBuffer;
  Gpio       : Dword;
  Ie         : Dword;
begin
  Result := False;
  // Set GPIO as 'normal' input
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aGpioCtrl, Gpio) then
    Exit;
  Gpio := Gpio and not CSaa7146aGpio3Mask;
  Gpio := Gpio or CSaa7146aGpioBase;                       // Input
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aGpioCtrl, Gpio) then
    Exit;
  // Disable GPIO3 interrupt
  Result := False;
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aIer, Ie) then
    Exit;
  Ie := Ie and not (1 shl CSaa7146aIrqPin3);
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aIer, Ie) then
    Exit;

  IrqHandling.Information.IrqBufferingIsActive := False;
  for Loop := 0 to 31 do
  begin
    IrqHandling.Identifier := Loop;
    Saa7146aWriteToIrqHandling(Handle, IrqHandling);
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Setup interrupt for infrared (GPIO3)
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aEnableInfrared(Handle: THandle): Boolean;
var
  IrqHandling: TSaa7146aIrqTransferBuffer;
  Gpio       : Dword;
  Ie         : Dword;
begin
  Result := False;
  // Setup interrupt behaviour driver. For now just record interrupts, nothing more.
  with IrqHandling.Information do
  begin
    Irqs                 := 0;
    IrqsWhenActive       := 0;
    IrqBufferingIsActive := True;
    UseNotificationEvent := False;
    UseSignaling         := False;
    UseFifo              := False;
    IrqAutoDisable       := False;
  end;
  IrqHandling.Identifier := CSaa7146aIrqPin3;              // GPIO3 interrupt status
  Saa7146aWriteToIrqHandling(Handle, IrqHandling);

  // Make GPIO3 an interrupt
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aGpioCtrl, Gpio) then
    Exit;
  Gpio := Gpio and not CSaa7146aGpio3Mask;
  Gpio := Gpio or CSaa7146aGpio3Rising;                    // IRQ at rising edge
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aGpioCtrl, Gpio) then
    Exit;
  // Enable GPIO3 interrupt
  Result := False;
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aIer, Ie) then
    Exit;
  Ie := Ie or (1 shl CSaa7146aIrqPin3);
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aIer, Ie) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Get status of infrared (GPIO3)
  Notes   :
------------------------------------------------------------------------------}
function Saa7146aStatusInfrared(Handle: THandle): Dword;
var
  IrqHandling: TSaa7146aIrqTransferBuffer;
begin
  IrqHandling.Identifier := CSaa7146aIrqPin3;              // GPIO3 interrupt status
  if Saa7146aReadFromIrqHandling(Handle, IrqHandling) then
    Result := IrqHandling.Information.Irqs
  else
    Result := $FFFF;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  MajorVersion : Word;
  MinorVersion : Word;
  Build        : Dword;
  DeviceName   : PChar;
begin
  FCardHandle := INVALID_HANDLE_VALUE;
  // Check if the driver is present and/or multiple cards are present
  case Saa7146aGetNumberOfCards of
    0:   mmoDriver.Lines.Add('No SAA7146A driver and/or card detected.');
    1:   mmoDriver.Lines.Add('One SAA7146A card detected.');
    else mmoDriver.Lines.Add(format('%d SAA7146A cards detected.', [Saa7146aGetNumberOfCards]));
  end;
  // If any card detected then display additional information
  if Saa7146aGetNumberOfCards <> 0 then
  begin
    // At this time we don't have a link to the driver itself. We have to create this.
    // We use the first available card (-1==auto) there is.
    FCardHandle := Saa7146aCreateFile(-1);
    try
      // Now with the handle we can actually access the driver.
      // Start with getting the version information of the driver.
      GetMem(DeviceName, 128);
      if Saa7146aGetDriverVersion(FCardHandle, @MajorVersion, @MinorVersion, @Build, DeviceName) then
        mmoDriver.Lines.Add(format('Driver "%s", Version %d.%d, Build %8.8x.', [DeviceName, MajorVersion, MinorVersion, Build]));
      FreeMem(DeviceName, 128);
      // The RESET of the tuner is connected to a GPIO line of the SAA7146A, so we have to configure this
      if not Saa7146aInitializeGpio(FCardHandle, True) then
      begin
        mmoDriver.Lines.Add('Could not initialize GPIO.');
        Exit;
      end;
      // Now issue a reset signal
      if not Saa7146aIssueReset(FCardHandle, 2) then
      begin
        mmoDriver.Lines.Add('Could not issue a reset.');
        Exit;
      end;
      mmoDriver.Lines.Add('Tuner has been reset.');
    finally
//      Saa7146aCloseHandle(FCardHandle);
    end;
  end;
  if FCardHandle <> INVALID_HANDLE_VALUE then
    tmrUpdate.Enabled := True;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  tmrUpdate.Enabled := False;
  Saa7146aDisableInfrared(FCardHandle);
  Saa7146aCloseHandle(FCardHandle);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Timer update
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
//  Ie  : Dword;
  Irqs: Dword;
begin
//  Saa7146aReadFromSaa7146aRegister(FCardHandle, CSaa7146aIer, Ie);
  Irqs := Saa7146aStatusInfrared(FCardHandle);
  txtInterrupts.Caption := format('%d', [Irqs]);
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Activate infrared
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnActivateClick(Sender: TObject);
begin
  if btnActivate.Tag = 0 then
  begin
    Saa7146aEnableInfrared(FCardHandle);
    btnActivate.Tag := 1;
    btnActivate.Caption := 'Disable infrared';
  end
  else
  begin
    Saa7146aDisableInfrared(FCardHandle);
    btnActivate.Tag := 0;
    btnActivate.Caption := 'Enable infrared';
  end;
end;

end.
