{******************************************************************************}
{ FileName............: DumpRegistersTda10021                                  }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Dump TDA10021 registers to file                                             }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{  Usage:                                                                      }
{    DumpRegistersTda10021 Filename Index                                      }
{                                                                              }
{    Filename   Filename to use for output                                     }
{    Index      Demodulator index address 0..1                                 }
{                                                                              }
{ Result is also in exit code:                                                 }
{    $0000 = No error                                                          }
{    $01xx = Invalid number of parameters                                      }
{    $02xx = First parameter incorrect                                         }
{    $03xx = 2nd parameter incorrect                                           }
{    $04xx = 3rd parameter incorrect                                           }
{    $05xx = etc                                                               }
{    $10xx = Driver/card not available                                         }
{    $20xx = Operation failed                                                  }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050109 - Initial release                                           }
{******************************************************************************}
{ Note: Since only a READOUT is performed it is assumed that the setup of the  }
{       SAA7146A (I2C) is correct.                                             }
{******************************************************************************}
program DumpRegistersTda10021;

uses
  SysUtils,
  Windows,
  Saa7146aI2c,
  Saa7146aInterface,
  Tda10021Registers;

var
  CardHandle: THandle;
  FileName: string;
  OutputFile: Text;
  Loop: Byte;
  ReturnedByte: Byte;
  Error: Integer;
  DemodIndex: Integer;
  VError: Integer;
begin
  Error := $2000;
  if ParamCount <> 2 then
  begin
    Error := $0100;
    Halt(Error);
  end;
  // Check for index
  Val(ParamStr(2), DemodIndex, VError);
  if ((VError <> 0) or (DemodIndex < Low(CTda10021DemodulatorWriteAddress)) or
    (DemodIndex > High(CTda10021DemodulatorWriteAddress))) then
  begin
    Error := $0300;
    Halt(Error);
  end;
  FileName := ParamStr(1);
  AssignFile(OutputFile, FileName);
  try
    Rewrite(OutputFile);
    if Saa7146aGetNumberOfCards = 0 then
    begin
      WriteLn(OutputFile, 'No SAA7146A driver/card detected');
      Error := $1000;
      Exit;
    end;
    CardHandle := Saa7146aCreateFile(-1);
    try
      WriteLn(OutputFile, 'Contents of TDA10021 (' + FormatDateTime('', Now) +
        ')');
      WriteLn(OutputFile);
      if not Saa7146aSetTunerType(CTunerCU1216,
        CTda10021DemodulatorWriteAddress[DemodIndex], 0) then
        Exit;
      // Initialize I2C (in case an incorrect address was used prior to this)  
      if not Saa7146aInitializeI2c(CardHandle) then
        Exit;
      if not Saa7146aReadFromDemodulator(CardHandle, CTda10021Identity,
        ReturnedByte) then
      begin
        WriteLn(OutputFile,
          format('Error reading TDA10021 identification register (%s)',
          [Saa7146aGetLastI2cError]));
        WriteLn(OutputFile,
          'Reset migth be active (SAA7146A) or address incorrect.');
        Exit;
      end;
      if ReturnedByte = CTda10021IdentityValue then
        WriteLn(OutputFile,
          format('TDA10021 identification correct for TDA10021 revision 1 ($%2.2x)', [ReturnedByte]))
      else
        WriteLn(OutputFile,
          format('TDA10021 identification NOT correct for TDA10021 revision 1 ($%2.2x)', [ReturnedByte]));

      WriteLn(OutputFile);
      for Loop := Low(CTda10021RegisterTexts) to High(CTda10021RegisterTexts) do
      begin
        if not Saa7146aReadFromDemodulator(CardHandle, Loop, ReturnedByte) then
        begin
          WriteLn(OutputFile, format('Error reading TDA10021 register (%s)',
            [Saa7146aGetLastI2cError]));
          Exit;
        end;
        WriteLn(OutputFile, format('Register $%2.2x == $%2.2x (%s)', [Loop,
          ReturnedByte, CTda10021RegisterTexts[Loop]]));
      end;
      Error := $0000;
    finally
      Saa7146aCloseHandle(CardHandle);
    end;
  finally
    CloseFile(OutputFile);
    Halt(Error);
  end;
end.

