{******************************************************************************}
{ FileName............: DumpRegistersI2c                                       }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Dump registers of some I2C device to file                                   }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{  Usage:                                                                      }
{    DumpRegistersI2c Filename BaseAddress Register BytesToRead                }
{                                                                              }
{    Filename      Filename to use for output                                  }
{    BaseAddress   Base address I2C device                                     }
{    Register      Register to read                                            }
{    BytesToRead   Number of bytes to read                                     }
{                                                                              }
{ Result is also in exit code:                                                 }
{    $0000 = No error                                                          }
{    $01xx = Invalid number of parameters                                      }
{    $02xx = First parameter incorrect                                         }
{    $03xx = 2nd parameter incorrect                                           }
{    $04xx = 3rd parameter incorrect                                           }
{    $05xx = etc                                                               }
{    $10xx = Driver/card not available                                         }
{    $20xx = Operation failed                                                  }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050109 - Initial release                                           }
{******************************************************************************}
{ Note: Since only a READOUT is performed it is assumed that the setup of the  }
{       SAA7146A (I2C) is correct.                                             }
{******************************************************************************}
program DumpRegistersI2c;

uses
  SysUtils, Windows,
  Saa7146aI2c,
  Saa7146aInterface,
  Stv0299bRegisters,
  Tsa5059Registers;

var
  CardHandle: THandle;
  FileName: string;
  OutputFile: Text;
  ReturnedByte: Byte;
  Error: Integer;
  Address: Integer;
  AccessRegister: Integer;
  BytesToRead: Integer;
begin
  Error := $2000;
  if ParamCount <> 4 then
  begin
    Error := $0100;
    Halt(Error);
  end;
  Val(ParamStr(2), Address, Error);
  if Error <> 0 then
    Halt($0300);
  if Odd(Address) then
    Halt($0300);
  if Address > $FF then
    Halt($0300);
  Val(ParamStr(3), AccessRegister, Error);
  if Error <> 0 then
    Halt($0400);
  if AccessRegister > $FF then
    Halt($0400);
  Val(ParamStr(4), BytesToRead, Error);
  if Error <> 0 then
    Halt($0500);

  FileName := ParamStr(1);
  AssignFile(OutputFile, FileName);
  try
    Rewrite(OutputFile);
    if Saa7146aGetNumberOfCards = 0 then
    begin
      WriteLn(OutputFile, 'No SAA7146A driver/card detected');
      Error := $1000;
      Exit;
    end;
    CardHandle := Saa7146aCreateFile(-1);
    try
      WriteLn(OutputFile, 'Contents of I2C device (' + FormatDateTime('', Now) +
        ')');
      WriteLn(OutputFile);
      if not Saa7146aInitializeI2c(CardHandle) then
        Exit;

      repeat
        if Saa7146aReadFromI2c(CardHandle, Lo(Address), Lo(AccessRegister), ReturnedByte) then
          WriteLn(OutputFile, format('Register $%2.2x == $%2.2x',
            [AccessRegister, ReturnedByte]))
        else
          Exit;
        Inc(AccessRegister);
        Dec(BytesToRead);
      until BytesToRead <= 0;
      Error := $0000;
    finally
      Saa7146aCloseHandle(CardHandle);
    end;
  finally
    CloseFile(OutputFile);
    Halt(Error);
  end;
end.

