{******************************************************************************}
{ FileName............: FindingCarrierUnit001                                  }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  'Finding' a carrier.                                                        }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040714 - Initial DLL release                                       }
{******************************************************************************}
unit FindingCarrierUnit001;

interface

uses
  SyncObjs,
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Mask, Gauges;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    mmoDriver: TMemo;
    rgPolarity: TRadioGroup;
    mskSymbolRate: TMaskEdit;
    StaticText1: TStaticText;
    tmrUpdate: TTimer;
    StaticText3: TStaticText;
    mskFrequency: TMaskEdit;
    txtSignalPower: TStaticText;
    txtSignalState: TStaticText;
    StaticText2: TStaticText;
    StaticText6: TStaticText;
    StaticText5: TStaticText;
    txtNoiseIndicator: TStaticText;
    Label1: TLabel;
    GroupBox1: TGroupBox;
    cmbRepeats: TComboBox;
    btnOptionASatA: TButton;
    btnOptionASatB: TButton;
    btnOptionBSatA: TButton;
    btnOptionBSatB: TButton;
    Label2: TLabel;
    ggPower: TGauge;
    ggNoise: TGauge;
    mskAddress: TMaskEdit;
    mskCommand: TMaskEdit;
    mskData: TMaskEdit;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Button1: TButton;
    chkDataEnabled: TCheckBox;
    procedure btnExitClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure rgPolarityClick(Sender: TObject);
    procedure mskSymbolRateChange(Sender: TObject);
    procedure tmrUpdateTimer(Sender: TObject);
    procedure mskFrequencyChange(Sender: TObject);
    procedure mskFrequencyKeyPress(Sender: TObject; var Key: Char);
    procedure mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
    procedure FrontEndChange(Sender: TObject);
    procedure SatSelect(Sender: TObject; Satellite: Integer);
    procedure btnSat1Click(Sender: TObject);
    procedure btnSat2Click(Sender: TObject);
    procedure btnSat3Click(Sender: TObject);
    procedure btnSat4Click(Sender: TObject);
    procedure DiSEqCSend(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;


implementation

{$R *.DFM}

uses
  Saa7146aFrontEndDefines;

var
  FrontEndOpen     : TSaa7146aOpen;
  FrontEndClose    : TSaa7146aClose;
  FrontEndIoCtl    : TSaa7146aIoCtl;
  FrontEndDllHandle: THandle;
  FrontEndHandle   : THandle;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Creation of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
begin
  cmbRepeats.ItemIndex := 0;
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Destruction of form.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  //
end;


{------------------------------------------------------------------------------
  Params  : <Sender>  Sender
  Returns : -

  Descript: Exit application.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


procedure TfrmMain.rgPolarityClick(Sender: TObject);
var
  Result: Integer;
begin
  case rgPolarity.ItemIndex of
    0:   Result := FrontEndIoCtl(FrontEndHandle, FE_SET_VOLTAGE, Pointer(SEC_VOLTAGE_18));
    1:   Result := FrontEndIoCtl(FrontEndHandle, FE_SET_VOLTAGE, Pointer(SEC_VOLTAGE_13));
    else Result := 0;
  end;
  if Result < 0 then
    mmoDriver.Lines.Add('Unable to set voltage.');
end;


procedure TfrmMain.FrontEndChange(Sender: TObject);
const
  LNBLOFLowBand  = 9750;
  LNBLOFHighBand = 10600;
  LowestTunerFrequency  =  950000;                // Lowest frequency tuner: CAN ALSO BE RETRIEVED USING THE CAPABILITIES CALL
  HighestTunerFrequency = 2150000;                // Highest frequency tuner: CAN ALSO BE RETRIEVED USING THE CAPABILITIES CALL
var
  SymbolRate       : Integer;
  Frequency        : Integer;
  ActualFrequency  : Integer;
  Error            : Integer;
  Result           : Integer;
  FrontEndParameter: TDvbFrontEndParameters;
  LowBand          : Boolean;
begin
  Val(mskSymbolRate.EditText, SymbolRate, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Could not convert symbolrate.');
    Exit;
  end;
  Val(mskFrequency.EditText, Frequency, Error);
  if Error <> 0 then
  begin
    mmoDriver.Lines.Add('Conversion error frequency.');
    Exit;
  end;
  // The frequency we have now is NOT the frequency for the tuner.
  // The frequency should be corrected with the LOF (local oscillator frequency)
  // of the LNB used. Also a typical LNB has two of these LOF's. The
  // selection of the LOF the LNB uses is through the 22 kHz signal:
  // low band == no 22 kHz signal; high band == 22 kHz signal

  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  LowBand := True;
  // Note: The <Abs> allows for negative results (eg. C-band)
  ActualFrequency := Frequency;
  ActualFrequency := Abs(ActualFrequency - LNBLOFLowBand);
  ActualFrequency := ActualFrequency * 1000;
  if ActualFrequency > HighestTunerFrequency then
  begin
    LowBand := False;
    // Note: The <Abs> allows for negative results (eg. C-band)
    ActualFrequency := Frequency;
    ActualFrequency := Abs(ActualFrequency - LNBLOFHighBand);
    ActualFrequency := ActualFrequency * 1000;
  end;
  if (ActualFrequency > HighestTunerFrequency)  or
     (ActualFrequency < LowestTunerFrequency) then
    Exit;
  // Now the band is know we have to set this.
  if LowBand then
    FrontEndIoCtl(FrontEndHandle, FE_SET_TONE, Pointer(SEC_TONE_OFF))
  else
    FrontEndIoCtl(FrontEndHandle, FE_SET_TONE, Pointer(SEC_TONE_ON));

  FrontEndParameter.Frequency       := ActualFrequency;
  FrontEndParameter.Inversion       := INVERSION_AUTO;
  FrontEndParameter.Qpsk.FecInner   := FEC_AUTO;
  FrontEndParameter.Qpsk.SymbolRate := SymbolRate * 1000;  // Symbols per second
  Result := FrontEndIoCtl(FrontEndHandle, FE_SET_FRONTEND, @FrontEndParameter);
  if Result < 0 then
    mmoDriver.Lines.Add(format('Unable to set frequency / symbolrate (%d).', [Result]));
end;


procedure TfrmMain.mskSymbolRateChange(Sender: TObject);
begin
  FrontEndChange(Sender);
end;


procedure TfrmMain.tmrUpdateTimer(Sender: TObject);
var
  Result    : Integer;
  Status    : Integer;
  Noise     : Word;
  StatusStr : string;
  Strength  : Word;
begin
  Result := FrontEndIoCtl(FrontEndHandle, FE_READ_SIGNAL_STRENGTH, @Strength);
  if Result >= 0 then
  begin
    txtSignalPower.Caption := format('%d', [Strength]);
    ggPower.Progress := Round((Strength * 100)/65536);
  end;

  Result := FrontEndIoCtl(FrontEndHandle, FE_READ_STATUS, @Status);
  if Result >= 0 then
  begin
    if (Status and FE_HAS_SIGNAL) <> 0 then
      StatusStr := 'SIGNAL'
    else
      StatusStr := 'NO SIGNAL';
    if (Status and FE_HAS_CARRIER) <> 0 then
      StatusStr := StatusStr + ' - CARRIER'
    else
      StatusStr := StatusStr + ' - NO CARRIER';
    if (Status and FE_HAS_VITERBI) <> 0 then
      StatusStr := StatusStr + ' - FEC STABLE'
    else
      StatusStr := StatusStr + ' - FEC INSTABLE';
    if (Status and FE_HAS_SYNC) <> 0 then
      StatusStr := StatusStr + ' - SYNC'
    else
      StatusStr := StatusStr + ' - NO SYNC';
    txtSignalState.Caption := StatusStr;
    if (Status and (FE_HAS_SIGNAL or FE_HAS_CARRIER or FE_HAS_VITERBI or FE_HAS_SYNC)) =
                   (FE_HAS_SIGNAL or FE_HAS_CARRIER or FE_HAS_VITERBI or FE_HAS_SYNC) then
    begin
      ggPower.ForeColor := clLime;
      ggNoise.ForeColor := clLime;
    end
    else
    begin
      ggPower.ForeColor := clRed;
      ggNoise.ForeColor := clRed;
    end;
  end;
  Result := FrontEndIoCtl(FrontEndHandle, FE_READ_SNR, @Noise);
  if Result >= 0 then
  begin
    txtNoiseIndicator.Caption := format('%d', [Noise]);
    ggNoise.Progress := Round((Noise * 100)/65536);
  end;
end;


procedure TfrmMain.mskFrequencyChange(Sender: TObject);
begin
  FrontEndChange(Sender);
end;

procedure TfrmMain.mskFrequencyKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskFrequencyChange(nil);
end;

procedure TfrmMain.mskSymbolRateKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    mskSymbolrateChange(nil);
end;


procedure TfrmMain.SatSelect(Sender: TObject; Satellite: Integer);
var
  Result : Integer;
  DiSEqC : TDvbDiSEqCMasterCmd;
  Repeats: Integer;
begin
  DiSEqC.Msg[0] := $E0;                                    // Framing: Command from master, no reply required, first transmission
  DiSEqC.Msg[1] := $10;                                    // Address: Any LNB/switcher
  DiSEqC.Msg[2] := $38;                                    // Command: Committed sitches (DiSEqc level 1.0)
  case Satellite of
    0:   DiSEqC.Msg[3] := $F0;                             // Data   : Switch ID in low nibble 1
    1:   DiSEqC.Msg[3] := $F8;                             // Data   : Switch ID in low nibble 9
    2:   DiSEqC.Msg[3] := $F4;                             // Data   : Switch ID in low nibble 5
    3:   DiSEqC.Msg[3] := $FC;                             // Data   : Switch ID in low nibble 13
    else DiSEqC.Msg[3] := $F0;                             // Data   : Switch ID in low nibble
  end;
  DiSEqC.MsgLen := 4;
  Repeats := CmbRepeats.ItemIndex;
  repeat
    Result := FrontEndIoCtl(FrontEndHandle, FE_DISEQC_SEND_MASTER_CMD, @DiSEqC);
    if Result < 0 then
      mmoDriver.Lines.Add(format('Unable to send DiSEqC master command. Error code %d.', [Result]));
    DiSEqC.Msg[0] := $E1;                                  // Framing: Command from master, no reply required, repeated transmission
    if Repeats >= 0 then
      Sleep(100);                                          // When there is a repeated command we need a 100 ms delay in between
    Dec(Repeats);
  until Repeats < 0;
end;


procedure TfrmMain.btnSat1Click(Sender: TObject);
begin
  SatSelect(Sender, 0);
end;

procedure TfrmMain.btnSat2Click(Sender: TObject);
begin
  SatSelect(Sender, 1);
end;

procedure TfrmMain.btnSat3Click(Sender: TObject);
begin
  SatSelect(Sender, 2);
end;

procedure TfrmMain.btnSat4Click(Sender: TObject);
begin
  SatSelect(Sender, 3);
end;


procedure TfrmMain.DiSEqCSend(Sender: TObject);
var
  Result : Integer;
  DiSEqC : TDvbDiSEqCMasterCmd;
  Repeats: Integer;
  Address: Integer;
  Command: Integer;
  Data   : Integer;
  Error  : Integer;
begin
  Val(mskAddress.EditText, Address, Error);
  if Error <> 0 then
    Exit;
  if (Address < 0) or (Address > 255) then
    Exit;
  Val(mskCommand.EditText, Command, Error);
  if Error <> 0 then
    Exit;
  if (Command < 0) or (Command > 255) then
    Exit;
  Val(mskData.EditText,    Data, Error);
  if Error <> 0 then
    Exit;
  if (Data < 0) or (Data > 255) then
    Exit;
  DiSEqC.Msg[0] := $E0;                                    // Framing: Command from master, no reply required, first transmission
  DiSEqC.Msg[1] := Address;                                // Address
  DiSEqC.Msg[2] := Command;                                // Command
  DiSEqC.Msg[3] := Data;                                   // Data
  if chkDataEnabled.Checked then
    DiSEqC.MsgLen := 4
  else
    DiSEqC.MsgLen := 3;
  Repeats := CmbRepeats.ItemIndex;
  repeat
    Result := FrontEndIoCtl(FrontEndHandle, FE_DISEQC_SEND_MASTER_CMD, @DiSEqC);
    if Result < 0 then
      mmoDriver.Lines.Add(format('Unable to send DiSEqC master command. Error code %d.', [Result]));
    DiSEqC.Msg[0] := $E1;                                  // Framing: Command from master, no reply required, repeated transmission
    if Repeats >= 0 then
      Sleep(100);                                          // When there is a repeated command we need a 100 ms delay in between
    Dec(Repeats);
  until Repeats < 0;
  mmoDriver.Lines.Add(format('Address %d, command %d, data %d send to DiSEqC', [Address, Command, Data]));
end;




{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  Result    : Integer;
begin
  FrontEndDllHandle := 0;
  FrontEndHandle    := INVALID_HANDLE_VALUE;

  FrontEndDllHandle := LoadLibrary(PChar(ExtractFilepath(Application.ExeName)+'Saa7146aFrontEnd.dll'));
  if FrontEndDllHandle = 0 then
    ShowMessage('LoadLibrary Saa7146aFrontEnd.dll: failed');
//    raise Exception.CreateFmt('LoadLibrary(%s): failed',['Saa7146aFrontEnd.dll']);

  FrontEndOpen := GetProcAddress(FrontEndDllHandle,'Open');
  if not Assigned(@FrontEndOpen) then
  begin
    ShowMessage('GetProcAddress(Open): failed');
//    raise Exception.CreateFmt('GetProcAddress(%s): failed',['Open']);
    Application.Terminate;
  end;

  FrontEndClose := GetProcAddress(FrontEndDllHandle,'Close');
  if not Assigned(@FrontEndClose) then
  begin
    ShowMessage('GetProcAddress(Close): failed');
//    raise Exception.CreateFmt('GetProcAddress(%s): failed',['Close']);
    Application.Terminate;
  end;

  FrontEndIoCtl := GetProcAddress(FrontEndDllHandle,'IoCtl');
  if not Assigned(@FrontEndIoCtl) then
  begin
    ShowMessage('GetProcAddress(IoCtl): failed');
//    raise Exception.CreateFmt('GetProcAddress(%s): failed',['IoCtl']);
    Application.Terminate;
  end;

  Result := FrontEndOpen('frontend0'#0, 0);

  if Result <= 0 then
  begin
    ShowMessage('Getting handle for frontend failed [Saa7146aFrontEnd.dll].');
//    raise Exception.CreateFmt('Getting handle for frontend failed [Saa7146aFrontEnd.dll]', []);
    Application.Terminate;
  end
  else
    FrontEndHandle := Result;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  if FrontEndDllHandle <> 0 then
  begin
    if FrontEndHandle <> INVALID_HANDLE_VALUE then
      FrontEndClose(FrontEndHandle);
    FreeLibrary(FrontEndDllHandle);
  end;
end;


initialization
  Initialize;


finalization
  Finalize;
end.
