{******************************************************************************}
{ FileName............: Stv0299bControl                                        }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Stv0299b support                                                            }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030619 - Initial release                                           }
{******************************************************************************}
unit Stv0299bControl;

interface
uses
  Stv0299bRegisters,
  Windows;

function Stv0299bSetSymbolRate(Handle: THandle; SymbolRateKS: Dword): Boolean;
function Stv0299bSetInversion(Handle: THandle; Inversion: Boolean): Boolean;
function Stv0299bGetLockDetectorPercentage(Handle: THandle; var Percentage:
  Byte): Boolean;
function Stv0299bGetCarrierDeviation(Handle: THandle; var DeviationKHz:
  Integer): Boolean;
function Stv0299bGetSymbolRateDeviation(Handle: THandle; var DeviationKHz:
  Integer): Boolean;
function Stv0299bGetSignalPower(Handle: THandle; var Percentage: Byte): Boolean;
function Stv0299bGetNoiseIndicator(Handle: THandle; var Noise: Word): Boolean;

// LNB control
function Stv0299bControlLNB(Handle: THandle; SetOn: Boolean;
  SetPolarityHorizontal): Boolean;

//  function Saa7146aWriteToSynthesizer     (Handle: THandle; Data: Dword): Boolean;
function Saa7146aReadFromSynthesizer(Handle: THandle; var Data: Byte): Boolean;
function Saa7146aFrequencyLNB(Handle: THandle; FrequencyKHz: Dword;
  LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword): Boolean;

implementation
uses
  SyncObjs;

const
  CI2cClock = CSaa7146aI2cClkSel275kHz;

var
  LastI2cError: Dword; // Last I2c error (debugging, I2C status register)
  HighPerformanceAvailable: Boolean;
    // Indicates available high performance counter
  HighPerformanceFrequency: TLargeInteger;
    // Frequency of high performance counter
  I2cLock: TCriticalSection;
    // Prevents multiple I2C operations which interfere

  {------------------------------------------------------------------------------
    Params  : <UsDelay> Delay in us
    Returns : <Result>  True is success
                        False if no high performance counter (should not be possible)

    Descript: Delay for a defined time
    Notes   : If no high performace counter is available then the delay is
              still implemented but with a worse resolution.
              During the delay no processes are handled...
  ------------------------------------------------------------------------------}

function Saa7146aUsDelay(UsDelay: Dword): Boolean;
var
  StartTiming: TLargeInteger;
  Timing: TLargeInteger;
  DiffTiming: TLargeInteger;
  DeltaTiming: TLargeInteger;
  StartTime: Dword;
  MsDelay: DWord;
begin
  if HighPerformanceAvailable then
  begin
    Result := True;
    QueryPerformanceCounter(StartTiming);
    DeltaTiming := UsDelay * HighPerformanceFrequency;
    DeltaTiming := DeltaTiming div 1000000;
    repeat
      QueryPerformanceCounter(Timing);
      if Timing > StartTiming then
        DiffTiming := Timing - StartTiming
      else
        DiffTiming := StartTiming - Timing;
    until DiffTiming >= DeltaTiming;
  end
  else
  begin
    Result := False;
    MsDelay := UsDelay div 1000;
    StartTime := GetTickCount;
    // Now wait for the delay
    repeat
      repeat
      until StartTime <> GetTickCount;
    until Dword(Abs(GetTickCount - StartTime)) > MsDelay;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Explanatory string

  Descript: Get last I2C error (as string)
  Notes   : For debugging only. The error is cleared afterwards.
------------------------------------------------------------------------------}

function Saa7146aGetLastI2CError: string;
begin
  if (LastI2cError and CSaa7146aI2cInvalidStartStop) <> 0 then
    Result := 'I2C error: Bus error due to invalid start/stop condition';
  if (LastI2cError and CSaa7146aI2cNackAddressPhase) <> 0 then
    Result := 'I2C NACK: Error in address phase';
  if (LastI2cError and CSaa7146aI2cNackTransmission) <> 0 then
    Result := 'I2C NACK: Error in data transmission';
  if (LastI2cError and CSaa7146aI2cNackReception) <> 0 then
    Result := 'I2C NACK: Error when receiving data';
  if (LastI2cError and CSaa7146aI2cArbitrationLost) <> 0 then
    Result := 'I2C error: Arbitration lost';
  if LastI2cError = $FFFF then
    Result := 'I2C error: Timeout';
  if (LastI2cError and CSaa7146aI2cErr) = 0 then
    Result := 'No I2C error';
  LastI2cError := $0000;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Initialize I2C interface.
  Notes   : Send an abort command and sets the clock used for I2C transfers
------------------------------------------------------------------------------}

function Saa7146aInitializeI2c(Handle: THandle): Boolean;
var
  Loop: Word;
  Mc2: Dword;
  StartTime: Dword;
begin
  Result := False; // Assume we fail

  // Note that the I2C registers <CSaa7146aI2cSta> and <CSaa7146aI2cTrf> need
  // to be 'uploaded'
  // First setup the abort command (which clears the busy flag) and while we
  // at at set the communication speed
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cSta,
    CSaa7146aI2cAbort) then
    Exit;
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2,
    CSaa7146aMc2UpldI2c) then
    Exit;
  // The data is being uploaded and we need to wait for it to be uploaded
  // To prevent an endless wait we have a simple check on an excessive timeout
  StartTime := GetTickCount;
  repeat
    if GetTickCount <> StartTime then
    begin
      if Abs(GetTickCount - StartTime) > 1 then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
      Exit;
  until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);

  // Per SAA7146 data sheet, write to STATUS reg twice to reset all I2C error flags
  for Loop := 0 to 1 do
  begin
    // Reset errors, set clock and upload it
    if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cSta, CI2cClock)
      then
      Exit;
    if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2,
      CSaa7146aMc2UpldI2c) then
      Exit;
    // The data is being uploaded and we need to wait for it to be uploaded
    // To prevent an endless wait we have a simple check on an excessive timeout
    StartTime := GetTickCount;
    repeat
      if GetTickCount <> StartTime then
      begin
        if Abs(GetTickCount - StartTime) > 1 then
          Exit;
      end;
      if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
        Exit;
    until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Clear errors and BUSY flag of I2C interface (if any).
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aClearErrorsI2c(Handle: THandle): Boolean;
var
  I2cSta: Dword;
begin
  Result := False; // Assume we fail

  // Determine if error or busy
  if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cSta, I2cSta) then
    Exit;
  // If no errors/busy then no need to reset them
  if (I2cSta and (CSaa7146aI2cBusy or CSaa7146aI2cErr) = 0) then
  begin
    Result := True;
    Exit;
  end;
  Result := Saa7146aInitializeI2c(Handle);
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Value>   Value to transfer
            <TimeOut> Allowed timeout in ms
  Returns : <Result>  True for success

  Descript: Write I2C transfer control register and wait for transfer ready
  Notes   : -
------------------------------------------------------------------------------}

function Saa7146aI2cHandshake(Handle: THandle; Value: Dword; TimeOut: Dword):
  Boolean;
var
  Mc2: Dword;
  StartTime: Dword;
begin
  Result := False;
  LastI2cError := $FFFF; // Assume timeout error

  // Set up the value to write
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aI2cTrf, Value) then
    Exit;
  // Then we need to upload it
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aMc2,
    CSaa7146aMc2UpldI2c) then
    Exit;
  // And wait for the upload to finish
  StartTime := GetTickCount;
  repeat
    if GetTickCount <> StartTime then
    begin
      if Abs(GetTickCount - StartTime) > 1 then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, Mc2) then
      Exit;
  until ((Mc2 and CSaa7146aMc2UpldI2cMask) <> 0);

  // Now wait until I2C bus transfer is finished or an error occurs
  StartTime := GetTickCount;
  repeat
    LastI2cError := $FFFF; // Assume timeout error
    if GetTickCount <> StartTime then
    begin
      if Dword(Abs(GetTickCount - StartTime)) > TimeOut then
        Exit;
    end;
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cSta,
      LastI2cError) then
      Exit;
    if (LastI2cError and CSaa7146aI2cErr) <> 0 then
      Exit;
  until ((LastI2cError and CSaa7146aI2cBusy) = 0);

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessRegister>  Register to read from
  Returns : <Result>          True for success
            <Data>            Data read from register

  Descript: Read data from register of QPSK demodulator
  Notes   : Reading from the demodulator requires a dummy write (without data)
            to the register we want to read from. Then a read cycle will read
            the targetted data from the register.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}

function Saa7146aReadFromQpsk(Handle: THandle; AccessRegister: Byte; var Data:
  Byte): Boolean;
var
  I2cTrf: Dword;
  Retry: Byte;
  RetryWhole: Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // For a read we first have to write the register we like to read afterwards
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart shl 6) or // First byte send
          (CStv0299bDemodulatorWriteAddress shl 24) or
            // START - device address WRITE
          (CSaa7146aI2cAttrStop shl 4) or // Second byte send
          (AccessRegister shl 16) or // register - STOP
          (CSaa7146aI2cAttrNop shl 2) or // Third byte not sent
          (0 shl 8),
          20); // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if NeedNoRetry then
      begin
        // Execute the read
        Retry := 1;
        repeat
          NeedNoRetry := Saa7146aI2cHandshake(Handle,
            (CSaa7146aI2cAttrStart shl 6) or // First byte send
            (CStv0299bDemodulatorReadAddress shl 24) or
              // START - device address READ
            (CSaa7146aI2cAttrStop shl 4) or // Second byte send
            (0 shl 16) or // dummy (filled in by targetted device) - STOP
            (CSaa7146aI2cAttrNop shl 2) or // Third byte not sent
            (0 shl 8),
            20); // Timeout essential otherwise sometimes error
          if not NeedNoRetry then
            Dec(Retry);
        until NeedNoRetry or (Retry = 0);
      end;
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;
    // Return copy of read value
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cTrf, I2cTrf)
      then
      Exit;
    Data := (I2cTrf shr 16) and $FF;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessRegister>  Register to write to
            <Data>            Data to write to register
  Returns : <Result>          True for success

  Descript: Write data to register of QPSK demodulator
  Notes   : We always write a single register only, although it is possible
            to write consequtive registers.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}

function Saa7146aWriteToQpsk(Handle: THandle; AccessRegister: Byte; Data: Byte):
  Boolean;
var
  RetryWhole: Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      NeedNoRetry := Saa7146aI2cHandshake(Handle,
        (CSaa7146aI2cAttrStart shl 6) or // First byte is
        (CStv0299bDemodulatorWriteAddress shl 24) or
          //   START - device address WRITE
        (CSaa7146aI2cAttrCont shl 4) or // Second byte is
        (AccessRegister shl 16) or //   register
        (CSaa7146aI2cAttrStop shl 2) or // Third byte is
        (Data shl 8), //   data - STOP
        20); // Timeout essential otherwise sometimes error
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True if success

  Descript: Write all default values of the QSPK demodulator
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aWriteDefaultsToQpsk(Handle: THandle): Boolean;
var
  Loop: Byte;
begin
  Result := False;

  for Loop := Low(CStv0299bDefaults) to High(CStv0299bDefaults) do
  begin
    // Write value to register
    if CStv0299bDefaults[Loop, 0] <> $FF then
      if not Saa7146aWriteToQpsk(Handle, CStv0299bDefaults[Loop, 0],
        CStv0299bDefaults[Loop, 1]) then
        Exit;
  end;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True if values comply with default values

  Descript: Check values of the QSPK demodulator with there default values
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aCheckQpskDefaultValues(Handle: THandle): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;

  for Loop := Low(CStv0299bDefaults) to High(CStv0299bDefaults) do
  begin
    // Read value from register
    if CStv0299bDefaults[Loop, 0] <> $FF then
    begin
      if not Saa7146aReadFromQpsk(Handle, CStv0299bDefaults[Loop, 0], Data) then
        Exit;
      // Check it with expected value
      if Data <> CStv0299bDefaults[Loop, 1] then
        Exit;
    end;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True if values comply with reset values

  Descript: Check values of the QSPK demodulator with there reset values
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aCheckQpskResetValues(Handle: THandle): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;

  for Loop := Low(CStv0299bResets) to High(CStv0299bResets) do
  begin
    // Read value from register
    if not Saa7146aReadFromQpsk(Handle, CStv0299bResets[Loop, 0], Data) then
      Exit;
    // Check it with expected value
    if Data <> CStv0299bResets[Loop, 1] then
      Exit;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <SymbolRateKS>    Symbolrate to set (KSymbols/sec)
  Returns : <Result>          True for success

  Descript: Set symbol rate
  Notes   : Alpha_car and Beta_car are recommended values by the tuner
            manufacturer.
------------------------------------------------------------------------------}

function Saa7146aQpskSetSymbolRate(Handle: THandle; SymbolRateKS: Dword):
  Boolean;
var
  Data: Dword;
  AClc: Byte;
  BClc: Byte;
begin
  Result := False;

  // Check for excessive values
  if (SymbolRateKS * 1000) > CStv0299bMasterClock then
    Exit;

  // The symbolrate also requires additional settings
  AClc := $04; // Alpha_car
  BClc := $11; // Beta_car
  if SymbolRateKS < 30000 then
  begin
    AClc := $06; // Alpha_car
    BClc := $13; // Beta_car
  end;
  if SymbolRateKS < 14000 then
  begin
    AClc := $07; // Alpha_car
    BClc := $13; // Beta_car
  end;
  if SymbolRateKS < 7000 then
  begin
    AClc := $07; // Alpha_car
    BClc := $0F; // Beta_car
  end;
  if SymbolRateKS < 3000 then
  begin
    AClc := $07; // Alpha_car
    BClc := $0B; // Beta_car
  end;
  if SymbolRateKS < 1500 then
  begin
    AClc := $07; // Alpha_car
    BClc := $07; // Beta_car
  end;
  AClc := AClc or $B0; // Derotator on, 256 k symbols
  BClc := BClc or $40; // QPSK algorithm 1
  if not Saa7146aWriteToQpsk(Handle, CStv0299bAclc, AClc) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bBclc, BClc) then
    Exit;

  // Note: we need to write the MSB register before we write the Middle bit register
  Data := Round((SymbolRateKS * 1000) / CStv0299bGetSymbolrateUnits);
  // Write the values -> there are spread out over 5 nibbles (lowest byte has LSnibble in high part)
  Data := Data shl 4;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrH, (Data shr 16) and $FF) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrM, (Data shr 8) and $FF) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bSfrL, Data and $F0) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <Percentage>      Lock percentage (0-100)

  Descript: Get lock percentage
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskGetLockDetectorPercentage(Handle: THandle; var Percentage:
  Byte): Boolean;
var
  Data: Byte;
  Temp: Integer;
begin
  Result := False;

  // Read lock value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCldi, Data) then
    Exit;
  // Since the lock value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  // And to percentage
  Temp := Temp + 128;
  Temp := (Temp * 100) div 255;
  Percentage := Temp;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <SignalPower>     Signal power (AGC2 integrator) in percentage of
                              16-bit full range

  Descript: Get signal power percentage
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskGetSignalPower(Handle: THandle; var Percentage: Byte):
  Boolean;
var
  DataMSB: Byte;
  DataLSB: Byte;
  Data: Integer;
begin
  Result := False;

  // Read values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc2I1, DataMSB) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bAgc2I2, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;
  // To percentage of full scale
  Data := (Data * 100) div 65535;
  Percentage := Lo(Data);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <Noise>           Noise indicator

  Descript: Get noise indicator
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskGetNoiseIndicator(Handle: THandle; var Noise: Word):
  Boolean;
var
  DataMSB: Byte;
  DataLSB: Byte;
  Data: Integer;
begin
  Result := False;

  // Read values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bNirH, DataMSB) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bNirL, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;
  // To percentage of full scale
  Noise := Data;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Inversion>       True when inversion requested
  Returns : <Result>          True for success

  Descript: Set inversion of I/Q inputs
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskSetInversion(Handle: THandle; Inversion: Boolean): Boolean;
var
  OriginalData: Byte;
  Data: Byte;
begin
  Result := False;

  // Read original data
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, OriginalData) then
    Exit;
  Data := OriginalData and not CStv0299bInversionMask;
  if Inversion then
    Data := Data or CStv0299bInversionMask;
  if Data <> OriginalData then
    if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
      Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Frequency deviation (in kHz)

  Descript: Get deviation from carrier
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskGetCarrierDeviation(Handle: THandle; var DeviationKHz:
  Integer): Boolean;
var
  DataM: Byte;
  DataL: Byte;
  Temp: Int64;
begin
  Result := False;

  // Read derotator values
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrM, DataM) then
    Exit;
  if not Saa7146aReadFromQpsk(Handle, CStv0299bCfrL, DataL) then
    Exit;
  Temp := (DataM shl 8) or DataL;
  // Since its a signed number we convert it
  if (DataM and $80) <> 0 then
    Temp := Temp - 65536;
  Temp := Temp * (CStv0299bMasterClock div 1000);
  // And to percentage
  DeviationKHz := (Temp div 65536);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Symbol rate deviation in kHz

  Descript: Get deviation
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aQpskGetSymbolRateDeviation(Handle: THandle; var DeviationKHz:
  Integer): Boolean;
var
  Data: Byte;
  Temp: Integer;
  Calc: Extended;
begin
  Result := False;

  // Read timing frequency (deviation)
  if not Saa7146aReadFromQpsk(Handle, CStv0299bRtf, Data) then
    Exit;

  // Since the value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  Calc := Temp;
  Calc := Calc * CStv0299bGetSymbolrateUnits;
  Calc := Calc / 1000;
  DeviationKHz := Round(Calc);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Enable the LNB
  Notes   : The LNB (power) is controlled throug the OP1 output of the QPSK
            demodulator.
            Typical time for the enable is 1.5 ms
------------------------------------------------------------------------------}

function Saa7146aEnableLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  Data := Data and not CStv0299bOP1Mask;
  Data := Data or CStv0299bOP1Value;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Disable the LNB
  Notes   : The LNB (power) is controlled throug the OP1 output of the QPSK
            demodulator.
            Disabling is almost directly.
------------------------------------------------------------------------------}

function Saa7146aDisableLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  Data := Data and not CStv0299bOP1Mask;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set low band of LNB
  Notes   : The LNB band is controlled by the absence of a 22kHz signal.
------------------------------------------------------------------------------}

function Saa7146aLowBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulation0;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set high band of LNB
  Notes   : The LNB band is controlled by the presence of a 22kHz signal.
------------------------------------------------------------------------------}

function Saa7146aHighBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulationContinuous;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set vertical polarity of LNB
  Notes   : The polarity is controlled by the 13V signal on the LNB, which
            is controlled by OP0.
------------------------------------------------------------------------------}

function Saa7146aVerticalPolarityLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  Data := Data and not CStv0299bOP0Mask;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set horizontal polarity of LNB
  Notes   : The polarity is controlled by the 18V signal on the LNB, which
            is controlled by OP0.
------------------------------------------------------------------------------}

function Saa7146aHorizontalPolarityLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not Saa7146aReadFromQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;
  Data := Data and not CStv0299bOP0Mask;
  Data := Data or CStv0299bOP0Value;
  // Write the new value
  if not Saa7146aWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <Data>            Data to write (control2, control1, divider2, divider1)
  Returns : <Result>          True for success

  Descript: Write data to synthesizer
  Notes   : Since the synthesizer is NOT connected to the SAA7146A, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}

function Saa7146aWriteToSynthesizer(Handle: THandle; Data: Dword): Boolean;
var
  Retry: Byte;
  RetryWhole: Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // First we need to enable the repeater of the QSPK demodulator
      if not Saa7146aWriteToQpsk(Handle, CStv0299bI2cRpt,
        CStv0299bDefaults[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn) then
        Exit;
      // Execute the write (must be 2 stages because we can only send 3 bytes at a time
      // and we need to send 5 bytes in total
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart shl 6) or // First byte
          (CTsa5059SynthesizerWriteAddress shl 24) or
            //   START - device address WRITE
          (CSaa7146aI2cAttrCont shl 4) or // Second byte
          (Data and $FF shl 16) or //   programmable divider 1
          (CSaa7146aI2cAttrCont shl 2) or // Third byte
          (((Data shr 8) and $FF) shl 8), //   programmable divider 2
          20); // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if NeedNoRetry then
      begin
        Retry := 1;
        repeat
          NeedNoRetry := Saa7146aI2cHandshake(Handle,
            (CSaa7146aI2cAttrCont shl 6) or // Fourth byte
            (((Data shr 16) and $FF) shl 24) or //   control 1
            (CSaa7146aI2cAttrStop shl 4) or // Fifth byte
            (((Data shr 24) and $FF) shl 16) or //   control 2 - STOP
            (CSaa7146aI2cAttrNop shl 2) or // Sixth byte not sent
            (0 shl 8),
            20); // Timeout essential otherwise sometimes error
          if not NeedNoRetry then
            Dec(Retry);
        until NeedNoRetry or (Retry = 0);
      end;
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
  Returns : <Result>          True for success
            <Data>            Data read (status register)

  Descript: Read data from synthesizer
  Notes   : Since the synthesizer is NOT connected to the SAA7146A, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole>/<Retry> set to '1' will do NO retry at all!
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}

function Saa7146aReadFromSynthesizer(Handle: THandle; var Data: Byte): Boolean;
var
  I2cTrf: Dword;
  Retry: Byte;
  RetryWhole: Byte;
  NeedNoRetry: Boolean;
begin
  Result := False;

  I2cLock.Acquire;
  try
    // Since it is possible that we might fail we might need to retry the attempt
    RetryWhole := 1;
    repeat
      if not Saa7146aClearErrorsI2c(Handle) then
        Exit;
      // First we need to enable the repeater of the QSPK demodulator
      if not Saa7146aWriteToQpsk(Handle, CStv0299bI2cRpt,
        CStv0299bDefaults[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn) then
        Exit;
      // Execute the read
      Retry := 1;
      repeat
        NeedNoRetry := Saa7146aI2cHandshake(Handle,
          (CSaa7146aI2cAttrStart shl 6) or // First byte send
          (CTsa5059SynthesizerReadAddress shl 24) or
            // START - device address READ
          (CSaa7146aI2cAttrStop shl 4) or // Second byte send
          (0 shl 16) or // dummy (filled in by targetted device) - STOP
          (CSaa7146aI2cAttrNop shl 2) or // Third byte not sent
          (0 shl 8),
          20); // Timeout essential otherwise sometimes error
        if not NeedNoRetry then
          Dec(Retry);
      until NeedNoRetry or (Retry = 0);
      if not NeedNoRetry then
        Dec(RetryWhole);
    until NeedNoRetry or (RetryWhole = 0);
    if not NeedNoRetry then
      Exit;
    // Return copy of read value
    if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aI2cTrf, I2cTrf)
      then
      Exit;
    Data := (I2cTrf shr 16) and $FF;

    Result := True;
  finally
    I2cLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>                     <Saa7146aCreateFile> handle
            <Frequency>                  Frequency to set (demodulated, in kHz)
            <LowBandLocalOscillatorKHz>  Local oscillator frequency of low band
            <HighBandLocalOscillatorKHz> Local oscillator frequency of high band
  Returns : <Result>                     True for success

  Descript: Set frequency of LNB
  Notes   : The frequency is controlled by the TSA5059 synthesizer which
            is controlled by the I2C (on the repeater of the QPSK demodulator!).
            The <LocalOscillator> is used to calculate the actual frequency for
            the synthesizer.
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}

function Saa7146aFrequencyLNB(Handle: THandle; FrequencyKHz: Dword;
  LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword): Boolean;
const
  Reference = 4000000;
  Ratios: array[0..15] of Dword =
  (Reference div 2, Reference div 4, Reference div 8, Reference div 16,
    Reference div 32, Reference div 64, Reference div 128, Reference div 256,
    0,
    Reference div 5, Reference div 10, Reference div 20, Reference div 40,
    Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency = 950000000; // Lowest frequency tuner
  HighestTunerFrequency = 2150000000; // Highest frequency tuner
var
  Loop: Byte;
  ActualFrequency: Int64;
  SmallestError: Dword;
  CalcError: Int64;
  RatioForSmallestError: Word;
  CalcDivider: Int64;
  ProgrammableDivider1: Byte;
  ProgrammableDivider2: Byte;
  ControlData1: Byte;
  ControlData2: Byte;
  SynthesizerData: Dword;
  LowBand: Boolean;
begin
  Result := False;

  if FrequencyKHz < LowBandLocalOscillatorKHz then
    Exit;
  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  LowBand := True;
  ActualFrequency := FrequencyKHz - LowBandLocalOscillatorKHz;
  ActualFrequency := ActualFrequency * 1000;
  if ActualFrequency > HighestTunerFrequency then
  begin
    LowBand := False;
    ActualFrequency := FrequencyKHz - HighBandLocalOscillatorKHz;
    ActualFrequency := Int64(ActualFrequency) * 1000;
  end;
  if (ActualFrequency > HighestTunerFrequency) or
    (ActualFrequency < LowestTunerFrequency) then
    Exit;
  // Now the band is know we have to set this.
  if LowBand then
  begin
    if not Saa7146aLowBandLNB(Handle) then
      Exit;
  end
  else if not Saa7146aHighBandLNB(Handle) then
    Exit;
  // To set the correct frequency we have to set two different values: the
  // 'divider' and the 'ratio'. The actual calculations involved are:
  // FREQUENCY = DIVIDER * (4 MHz / RATIO)
  // Since we have to calculate the DIVIDER and RATIO from the FREQUENCY we have to
  // do some calculation, mostly to get an as accurate result.
  // RATIO    = 2,       4,       8,      16,      32,     64,    128,   256,
  // 4M/RATIO = 2000000, 1000000, 500000, 250000,  125000, 62500, 31250, 15625
  // RATIO    = 5,       10,      20,     40,      80,     160,   320
  // 4M/RATIO = 800000,  400000,  200000, 1000000, 50000,  25000, 12500
  // DIVIDER = 0..$1FFFF (0..131071)
  // At this point we could do some calculations and such to get the best result,
  // but the simplest method is just to check all possibilities and choose the
  // one with the smallest error. Alternatively we could choose a fixed ratio which
  // will limit the step size. For now we will use the one with the smallest error.

  RatioForSmallestError := High(Ratios);
  SmallestError := ActualFrequency;
  for Loop := Low(Ratios) to High(Ratios) do
  begin
    if Ratios[Loop] <> 0 then
    begin
      CalcDivider := ActualFrequency div Ratios[Loop];
      CalcError := ActualFrequency - (CalcDivider * Ratios[Loop]);
      if CalcError < SmallestError then
      begin
        SmallestError := CalcError;
        RatioForSmallestError := Loop;
      end;
    end;
    // If the error is zero then there is no need to continu checking....
    if SmallestError = 0 then
      Break;
  end;

  // With the divider information available we can construct the 4 bytes we
  // need to send to the synthesizer. The DIVIDER is divided over 3 bytes.
  CalcDivider := ActualFrequency div Ratios[RatioForSmallestError];
  ProgrammableDivider2 := CalcDivider and $FF;
  CalcDivider := CalcDivider shr 8;
  ProgrammableDivider1 := CalcDivider and $7F;
  CalcDivider := (CalcDivider shr 2) and $60;
  ControlData1 := $80 or CalcDivider or RatioForSmallestError;
  ControlData2 := CTsa5059DefaultControlRegister2;
  SynthesizerData := ProgrammableDivider1 or
    (ProgrammableDivider2 shl 8) or
    (ControlData1 shl 16) or
    (ControlData2 shl 24);
  // Now we can send the data to the synthesizer
  if not Saa7146aWriteToSynthesizer(Handle, SynthesizerData) then
    Exit;
  // Because the STV0299 might have to lock to this (new) frequency we have
  // to reset the de-rotator loop
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrM, $00) then
    Exit;
  if not Saa7146aWriteToQpsk(Handle, CStv0299bCfrL, $00) then
    Exit;

  Result := True;
end;

initialization
  LastI2cError := $0000;
  HighPerformanceAvailable :=
    QueryPerformanceFrequency(HighPerformanceFrequency);
  I2cLock := TCriticalSection.Create;

finalization
  I2cLock.Free;
end.

