{******************************************************************************}
{ FileName............: Saa7146aDebi                                           }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  Debi support                                                                }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030623 - Initial release                                           }
{******************************************************************************}
unit Saa7146aDebi;

interface
uses
  Saa7146aInterface,
  Saa7146aRegisters,
  Stv0299bRegisters,
  Tsa5059Registers,
  Windows;

function Saa7146aInitializeDebi(Handle: THandle): Boolean;
function Saa7146aReadByteFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Byte): Boolean;
function Saa7146aReadWordFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Word): Boolean;
function Saa7146aReadDwordFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Dword): Boolean;
function Saa7146aWriteByteToDebi(Handle: THandle; AccessAddress: Word; Data:
  Byte): Boolean;

implementation
uses
  SyncObjs;

var
  HighPerformanceAvailable: Boolean;
    // Indicates available high performance counter
  HighPerformanceFrequency: TLargeInteger;
    // Frequency of high performance counter
  DebiLock: TCriticalSection;
    // Prevents multiple DEBI operations which interfere

  {------------------------------------------------------------------------------
    Params  : <UsDelay> Delay in us
    Returns : <Result>  True is success
                        False if no high performance counter (should not be possible)

    Descript: Delay for a defined time
    Notes   : If no high performace counter is available then the delay is
              still implemented but with a worse resolution.
              During the delay no processes are handled...
  ------------------------------------------------------------------------------}

function Saa7146aUsDelay(UsDelay: Dword): Boolean;
var
  StartTiming: TLargeInteger;
  Timing: TLargeInteger;
  DiffTiming: TLargeInteger;
  DeltaTiming: TLargeInteger;
  StartTime: Dword;
  MsDelay: DWord;
begin
  if HighPerformanceAvailable then
  begin
    Result := True;
    QueryPerformanceCounter(StartTiming);
    DeltaTiming := UsDelay * HighPerformanceFrequency;
    DeltaTiming := DeltaTiming div 1000000;
    repeat
      QueryPerformanceCounter(Timing);
      if Timing > StartTiming then
        DiffTiming := Timing - StartTiming
      else
        DiffTiming := StartTiming - Timing;
    until DiffTiming >= DeltaTiming;
  end
  else
  begin
    Result := False;
    MsDelay := UsDelay div 1000;
    StartTime := GetTickCount;
    // Now wait for the delay
    repeat
      repeat
      until StartTime <> GetTickCount;
    until Dword(Abs(GetTickCount - StartTime)) > MsDelay;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
  Returns : <Result>  True if success

  Descript: Initialize DEBI interface.
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aInitializeDebi(Handle: THandle): Boolean;
begin
  Result := False; // Assume we fail

  if not Saa7146aWriteToSaa7146aRegister(Handle,
    CSaa7146aDebiPage,
    CSaa7146aDebiPageDisable) then // No page check
    Exit;
  if not Saa7146aWriteToSaa7146aRegister(Handle,
    CSaa7146aMc1,
    CSaa7146aMc1DebiEnable) then // Enable DEBI transfers
    Exit;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessAddress>   Address to read from
            <Data>            Data to write
            <Bytes>           Bytes to transfer
            <WriteToDebi>     True if write access
  Returns : <Result>          True for success
            <Data>            Data read from address

  Descript: Read data from Debi interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aAccessDebi(Handle: THandle; AccessAddress: Word; var Data:
  Dword; Bytes: Byte; WriteToDebi: Boolean): Boolean;
var
  StartTime: Dword;
  RegData: Dword;
begin
  Result := False;

  if (Bytes = 0) or
    (Bytes > 4) then
    Exit;

  DebiLock.Acquire;
  try
    // Setup Debi mode
    // Since the infrared data is accessed in a different way we must use this
    // method ...
    // We should also check if we don't interfere with the infrared communication...
    if not Saa7146aWriteToSaa7146aRegister(Handle,
      CSaa7146aDebiConfig,
      (2 shl 22) or // Timeout PCI clocks
      CSaa7146aDebiCfgIntel or // Intel mode
      CSaa7146aDebiCfgInc) then
        // Auto increment address (needed if more than 1 byte read)
      Exit;

    // Setup Debi control register value in shadow RAM, x bytes to transfer
    // Depending on the data direction we have to setup differently
    if WriteToDebi then
    begin
      if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aDebiAd, Data) then
        Exit;
      if not Saa7146aWriteToSaa7146aRegister(Handle,
        CSaa7146aDebiCommand,
        CSaa7146aDebiCmdWr or AccessAddress or (Bytes shl 17)) then
        Exit;
    end
    else if not Saa7146aWriteToSaa7146aRegister(Handle,
      CSaa7146aDebiCommand,
      CSaa7146aDebiCmdRd or AccessAddress or (Bytes shl 17)) then
      Exit;
    // Upload shadow RAM to Debi control register
    if not Saa7146aWriteToSaa7146aRegister(Handle,
      CSaa7146aMc2,
      CSaa7146aMc2UpldDebi) then
      Exit;

    StartTime := GetTickCount;
    // Wait for upload complete
    repeat
      if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aMc2, RegData)
        then
        Exit;
    until ((RegData and CSaa7146aMc2UpldDebiMask) <> 0) or
      (Abs(GetTickCount - StartTime) > 20);
    // Check failure
    if ((RegData and CSaa7146aMc2UpldDebiMask) = 0) then
      Exit;

    StartTime := GetTickCount;
    // Wait for Debi transfer complete
    repeat
      if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aPsr, RegData)
        then
        Exit;
    until ((RegData and CSaa7146aPsrDebiS) = 0) or
      (Abs(GetTickCount - StartTime) > 20);

    if ((RegData and CSaa7146aPsrDebiS) <> 0) then
      Exit;

    if not WriteToDebi then
    begin
      if not Saa7146aReadFromSaa7146aRegister(Handle, CSaa7146aDebiAd, RegData)
        then
        Exit;
      case Bytes of
        1: Data := RegData and $FF;
        2: Data := RegData and $FFFF;
        3: Data := RegData and $FFFFFF;
        4: Data := RegData;
      end;
    end;

    Result := True;
  finally
    DebiLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessAddress>   Address to read from
  Returns : <Result>          True for success
            <Data>            Data read from address

  Descript: Read data from Debi interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aReadByteFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Byte): Boolean;
var
  DataRead: Dword;
begin
  Result := Saa7146aAccessDebi(Handle, AccessAddress, DataRead, 1, False);
  Data := DataRead;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessAddress>   Address to read from
  Returns : <Result>          True for success
            <Data>            Data read from address

  Descript: Read data from Debi interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aReadWordFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Word): Boolean;
var
  DataRead: Dword;
begin
  Result := Saa7146aAccessDebi(Handle, AccessAddress, DataRead, 2, False);
  Data := DataRead;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessAddress>   Address to read from
  Returns : <Result>          True for success
            <Data>            Data read from address

  Descript: Read data from Debi interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aReadDwordFromDebi(Handle: THandle; AccessAddress: Word; var
  Data: Dword): Boolean;
var
  DataRead: Dword;
begin
  Result := Saa7146aAccessDebi(Handle, AccessAddress, DataRead, 4, False);
  Data := DataRead;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <Saa7146aCreateFile> handle
            <AccessAddress>   Address to read from
            <Data>            Data to write to address
  Returns : <Result>          True for success

  Descript: Write data to Debi interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aWriteByteToDebi(Handle: THandle; AccessAddress: Word; Data:
  Byte): Boolean;
var
  DataWrite: Dword;
begin
  DataWrite := Data;
  Result := Saa7146aAccessDebi(Handle, AccessAddress, DataWrite, 1, True);
end;

initialization
  HighPerformanceAvailable :=
    QueryPerformanceFrequency(HighPerformanceFrequency);
  DebiLock := TCriticalSection.Create;

finalization
  DebiLock.Free;
end.

