{******************************************************************************}
{ FileName............: USaa7146aDemux                                         }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.02                                                   }
{------------------------------------------------------------------------------}
{  Demux device budget card interface.                                         }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{  Demux device function calls                                                 }
{  . IoCtl()                                                                   }
{    - DMX_BACKDOOR_START_FEED                                                 }
{    - DMX_BACKDOOR_STOP_FEED                                                  }
{    - DMX_REGISTER_CALLBACK                                                   }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040711 - Initial release                                           }
{  1.01   20040801 - Fixes 100% load when no signal is being received          }
{  1.02   20041025 - Supports RPS0Program setting and uses tuner settings      }
{******************************************************************************}
unit USaa7146aDemux;

interface
  function IoCtl(const FileDescriptor: Integer; const Request: Integer; Parameter: Pointer) : Integer; stdcall;


implementation

uses
  Saa7146aDemuxDefines,
  DvbFilter,
  DvbStreamBuffering,
  Saa7146aInterface,
  Saa7146aIoControl,
  Saa7146aRegisters,
  Classes,
  IniFiles, SysUtils,  Windows;


type
  IoCtlCalls = (IoCtl_DMX_BACKDOOR_START_FEED   , IoCtl_DMX_BACKDOOR_STOP_FEED,
                IoCtl_DMX_REGISTER_CALLBACK
               );

  // The data thread receives the DVB-S data
  TDataThread = class(TThread)
  private
    { Private declarations }
    FHandle               : THandle;             // Handle to driver in thread
    FBufferId             : Dword;               // Buffer identifier for DMA data
    FPacketBuffers        : Word;                // Number of buffers used
    FHasStopped           : Boolean;             // Flag indicating thread not running
    FProgrammedStop       : Boolean;             // Flag indicating thread should stop
    procedure ProcessTransportStreamData(StreamData: PDvbTransportPackets);
  protected
    procedure Execute; override;
  end;

var
  StreamDataBuffer    : TDvbTransportPackets;    // DVB-S data

const
  CTunerBSRU6          = $00;                    // Copied from Saa7146aI2c
  CTunerSU1278         = $01;                    // Copied from Saa7146aI2c
  CTunerBSBE1          = $02;                    // Copied from Saa7146aI2c
  StreamDataBufferSize = SizeOf(StreamDataBuffer);

var
  IoCtlIdentifier      : array[Ord(IoCtl_DMX_BACKDOOR_START_FEED)..Ord(IoCtl_DMX_REGISTER_CALLBACK)] of Integer;  // IoCtl's to act upon

  IniFile              : TMemIniFile;            // Ini file

{$IFDEF USELOG}
  LogStream            : TFileStream;            // Filestream for log
  LogLevel             : Byte;                   // LogLevel (00 = no log)
{$ENDIF}
  // Non standard implementation (simplistic: only one target)
  PacketThread         : TDataThread;                      // Thread handling packets
  CardToUse            : Integer;                          // Card to use (0=1st)
  DriverHandle         : THandle;                          // Opened driver handle
  ActivePids           : array[0..$FFFF] of Boolean;       // Packets with PIDs which are send to callback
  CallBack             : TCallbackProc;                    // Callback function


{$IFDEF USELOG}
{------------------------------------------------------------------------------
  Params  : <LogString>  Data to log
            <Level>      Log level: if this number if <= <LogLevel> then
                         it is logged
                         $8x indicates that the string is intended for
                             displaying on form too.
  Returns : -

  Descript: Write data to log. A timestamp is added.
  Notes   :
 ------------------------------------------------------------------------------}
procedure ToLog(LogString: string; Level: Byte);
var
  NewLog: string;
begin
  try
    // Only log if the level is high enough
    if (Level and $0F) > LogLevel then
      Exit;
    // The check on <LogStream> is necessary because the application might
    // call this prior to completing initialization
    if not Assigned(LogStream) then
      Exit;
    NewLog := FormatDateTime('YYYYMMDD"T"HHMMSS"  "', Now) + LogString + #13#10;
    LogStream.Write(NewLog[1], Length(NewLog));
  except
  end;  
end;
{$ENDIF}


{------------------------------------------------------------------------------
  Params  : <Section>    Section to access in INI file
            <Paremeter>  Paremeter in INI file
            <Default>    Default value (eg. if not present in INI file)
  Returns : <Result>     String value for parameter

  Descript: Get parameter from INI file.
  Notes   :
 ------------------------------------------------------------------------------}
function GetParameter(Section: string; Parameter: string; Default: string): string;
var
  FromIni : string;
  Position: Integer;
  TempStr : string;
begin
  try
{$IFDEF USELOG}
    ToLog('Saa7146aDemux GetParameter (Section: [' + Section + '], Parameter: [' + Parameter + '].', $03);
{$ENDIF}
    if not Assigned(IniFile) then
      Exit;
    FromIni := IniFile.ReadString(Section, Parameter, Default);
    // Strip out leading/trailing spaces
    FromIni := Trim(FromIni);
    // Check for starting string parameter
    Position := Pos('''', FromIni);
    if Position = 1 then
    begin
      TempStr  := Copy(FromIni, Position+1, 255);
      // Find ending of string
      Position := Pos('''', TempStr);
      if Position <> 0 then
      begin
        // End of string found, which will be the result
        Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
        ToLog('Saa7146aDemux Result: [' + Result + '].', $05);
{$ENDIF}
        Exit;
      end;
    end
    else
    begin
      Position := Pos('"', FromIni);
      if Position = 1 then
      begin
        TempStr := Copy(FromIni, Position+1, 255);
        // Find ending of string
        Position := Pos('"', TempStr);
        if Position <> 0 then
        begin
          // End of string found, which will be the result
          Result := Copy(TempStr, 1, Position-1);
{$IFDEF USELOG}
          ToLog('Saa7146aDemux Result: [' + Result + '].', $05);
{$ENDIF}
          Exit;
        end;
      end;
    end;
    // We know we don't have a string type so handle the whole string as
    // normal text. We could have a comment in it so check this too
    Position := Pos(';', FromIni);
    if Position <> 0 then
    begin
      TempStr := Copy(FromIni, 1, Position-1);
      // Strip out leading/trailing spaces
      Result := Trim(TempStr);
    end
    else
      Result := FromIni;
{$IFDEF USELOG}
    ToLog('Saa7146aDemux Result: [' + Result + '].', $05);
{$ENDIF}
  except
  end;
end;


{******************************************************************************}
{                                 PACKET THREAD                                }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <StreamData>  Stream packets (multiple packets of 188 bytes)
  Returns : -

  Descript: Process transport stream.
  Notes   :
 ------------------------------------------------------------------------------}
procedure TDataThread.ProcessTransportStreamData(StreamData: PDvbTransportPackets);
var
  StreamPacket           : Word;
  Pid                    : Word;
begin
  try
    for StreamPacket := 0 to CDvbPacketVSync-1 do
      // The Pid is the identifier of what type of information we are dealing with
      if DvbFilterGetPid(@StreamData[StreamPacket], Pid) then
        if Assigned(@CallBack) and ActivePids[Pid] then
          CallBack(@StreamData[StreamPacket], 188);
  except
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Execution of thread. This thread handles reception of data.
  Notes   : We get an interrupt at both edges of the field indicator,
            meaning that we get notified when a switch is made from
            'odd' -> 'even' and from 'even' -> 'odd'
            The rate is about 25 interrupts a second
 ------------------------------------------------------------------------------}
procedure TDataThread.Execute;
var
  Data      : Dword;
  LastBuffer: Dword;
  Buffer    : TSaa7146aTransferBuffer;
begin
  try
    FHasStopped     := False;
    FProgrammedStop := False;
    // The driver must be running and must be in use
    // We MUST use a new file handle. If we don't then only one call per handle is
    // allowed and when this happens to be the 'wait for notification'
    // we can not send any other calls eg. 'generate manual notification'
    // which might be needed in a deadlock situation.
    FHandle := Saa7146aCreateFile(CardToUse);
    Buffer.Identifier      := FBufferId;
    Buffer.TransferAddress := @StreamDataBuffer;
    Buffer.TargetIndex     := 0;
    Buffer.TransferLength  := StreamDataBufferSize;

    // Wait for application to have started and such
    LastBuffer := 0;
    Data       := $FFFF;
    try
      repeat
        // First check if there are already buffers filled
        Saa7146aReadFromSaa7146aRegister(FHandle, CSaa7146aEcT1R, Data);
        // In case no data is being received the returned buffer can
        // be outside of range so correct for this
        if Data > (FPacketBuffers*2) then
          Data := 0;
        // If no new data, wait (or if first time)
        if Data = LastBuffer then
        begin
          Saa7146aWaitForNotification(FHandle);
          if FProgrammedStop then
            Exit;
        end;
        Buffer.SourceIndex := CDvbPacketBufferSize * LastBuffer;
        // Next buffer to wait for
        Inc(LastBuffer);
        // Circular buffer
        if LastBuffer = (FPacketBuffers*2) then
          LastBuffer := 0;
        // Get data in local buffer
        Saa7146aReadFromDma(FHandle, Buffer);
        // Process packet data
        ProcessTransportStreamData(@StreamDataBuffer);
      until FProgrammedStop or Terminated;
    finally
      Saa7146aCloseHandle(FHandle);
      FHasStopped := True;
{$IFDEF USELOG}
      ToLog('Data thread stopped.', $81);
{$ENDIF}
    end;
  except
    FHasStopped := True;
{$IFDEF USELOG}
    ToLog('Data thread exception occured.', $81);
{$ENDIF}
  end;
end;


{******************************************************************************}
{                               FUNCTIONS START                                }
{                                                                              }
{ Note: Minimal checks are done here because most of the checks are already    }
{       handled by the caller function <IoCtl>.                                }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to parameters
  Returns : <Result>          Result code

  Descript: IoCtl function: Start backdoor feed
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlDmxBackDoorStartFeed(FileDescriptor: Integer; Request: Integer; Parameter: PDmxBackDoorFeed): Integer;
var
  Loop: Integer;
begin
  try
{$IFDEF USELOG}
    ToLog(format('Saa7146aDemux DMX_BACKDOOR_START_FEED (Pid=%d).', [Parameter^.Pid]), $04);
{$ENDIF}
    if (Parameter^.Pid >= Low(ActivePids))  and
       (Parameter^.Pid <= High(ActivePids)) then
      ActivePids[Parameter^.Pid] := True;
    // Special PIDs: $FFFF means ALL PIDS
    if Parameter^.Pid = $FFFF then
      for Loop := Low(ActivePids) to High(ActivePids) do
        ActivePids[Loop] := True;
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to parameters
  Returns : <Result>          Result code

  Descript: IoCtl function: Stop backdoor feed
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlDmxBackDoorStopFeed(FileDescriptor: Integer; Request: Integer; Parameter: PDmxBackDoorFeed): Integer;
var
  Loop: Integer;
begin
  try
{$IFDEF USELOG}
    ToLog(format('Saa7146aDemux DMX_BACKDOOR_STOP_FEED (Pid=%d).', [Parameter^.Pid]), $04);
{$ENDIF}
    if (Parameter^.Pid >= Low(ActivePids))  and
       (Parameter^.Pid <= High(ActivePids)) then
      ActivePids[Parameter^.Pid] := False;
    // Special PIDs: $FFFF means ALL PIDS
    if Parameter^.Pid = $FFFF then
      for Loop := Low(ActivePids) to High(ActivePids) do
        ActivePids[Loop] := False;
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  File descriptor (handle) of device to access
            <Request>         Request to make (command)
            <Parameter>       Pointer to callback
  Returns : <Result>          Result code

  Descript: IoCtl function: Register callback function
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtlDmxRegisterCallback(FileDescriptor: Integer; Request: Integer; Parameter: Pointer): Integer;
begin
  try
{$IFDEF USELOG}
    ToLog(format('Saa7146aDemux DMX_REGISTER_CALLBACK (Address=%p).', [Parameter]), $04);
{$ENDIF}
    @Callback := Parameter;
    Result := ENOERROR;
  except
    Result := -EFAULT;
  end;
end;
{******************************************************************************}
{                                FUNCTIONS END                                 }
{******************************************************************************}


{******************************************************************************}
{                          EXPORTED FUNCTIONS START                            }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : <FileDescriptor>  NOT USED
            <Request>         Request to make (command)
            <Parameter>       Pointer to parameter
  Returns : <Result>          Result code

  Descript: IoCtl function entry
  Notes   :
 ------------------------------------------------------------------------------}
function IoCtl(const FileDescriptor: Integer; const Request: Integer; Parameter: Pointer): Integer; stdcall;
var
  IoCtlIndex: Integer;
begin
  try
{$IFDEF USELOG}
    ToLog(format('Saa7146aDemux ''IoCtl'', File descriptor %d, Request %d.', [FileDescriptor, Request]), $03);
{$ENDIF}
    // Find the function belonging to the <Request>
    IoCtlIndex := Low(IoCtlIdentifier);
    while (IoCtlIdentifier[IoCtlIndex] <> Request) and (IoCtlIndex <= High(IoCtlIdentifier)) do
      Inc(IoCtlIndex);
    // No identifier found means we don't support it
    if IoCtlIndex > High(IoCtlIdentifier) then
    begin
{$IFDEF USELOG}
      ToLog('Saa7146aDemux ''IoCtl'', Error: ''Function not supported''.', $02);
{$ENDIF}
      Result := -EOPNOTSUPP;
      Exit;
    end;
    // Check for correct <Parameter>
    case IoCtlIndex of
      Ord(IoCtl_DMX_REGISTER_CALLBACK):
      else
        begin
          if not Assigned(Parameter) then
          begin
{$IFDEF USELOG}
            ToLog('Saa7146aDemux ''IoCtl'', Error: ''Bad address''.', $02);
{$ENDIF}
            Result := -EFAULT;
            Exit;
          end;
        end;
    end;
    case IoCtlIndex of
      Ord(IoCtl_DMX_BACKDOOR_START_FEED): Result := IoCtlDmxBackdoorStartFeed(FileDescriptor, Request, PDmxBackDoorFeed(Parameter));
      Ord(IoCtl_DMX_BACKDOOR_STOP_FEED) : Result := IoCtlDmxBackdoorStopFeed (FileDescriptor, Request, PDmxBackDoorFeed(Parameter));
      Ord(IoCtl_DMX_REGISTER_CALLBACK)  : Result := IoCtlDmxRegisterCallback (FileDescriptor, Request, Parameter);
      else
        begin
          // We should NEVER get here ....
{$IFDEF USELOG}
          ToLog('Saa7146aDemux ''IoCtl'', Error: ''Function not supported''.', $02);
{$ENDIF}
          Result := -EOPNOTSUPP;
        end;
    end;
  except
    Result := -EFAULT;
  end;
end;
{******************************************************************************}
{                             EXPORTED FUNCTIONS END                           }
{******************************************************************************}


{******************************************************************************}
{                              INITIALIZE AND FINALIZE                         }
{******************************************************************************}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
var
  Parameter    : string;
  Error        : Integer;
{$IFDEF USELOG}
  LogClear     : Boolean;
{$ENDIF}
  Loop         : Integer;
  Value        : Integer;
  IsSlave      : Boolean;
  BufferSize   : Word;
  BufferId     : Dword;
  Rps0Program  : Integer;                        // RPS0 program running
  TunerType    : Byte;                           // Tuner type
  TunerReset   : Byte;                           // GPIO line for reset ($FF if not to be used)
  DvbCard      : string;                         // DVB card identifier used
  CheckString  : string;
begin
  // Set initial values
  IniFile      := nil;
{$IFDEF USELOG}
  LogStream    := nil;
  LogLevel     := 0;
  LogClear     := True;
{$ENDIF}

  IsSlave      := False;

  for Loop := Low(ActivePids) to High(ActivePids) do
    ActivePids[Loop] := False;
  @CallBack    := nil;
  DriverHandle := INVALID_HANDLE_VALUE;
  CardToUse    := 0;
  BufferSize   := 500;

  for Loop := Low(IoCtlIdentifier) to High(IoCtlIdentifier) do
    IoCtlIdentifier[Loop] := -1;

  if FileExists('Saa7146a.ini') then
  begin
    IniFile := TMemIniFile.Create('Saa7146a.ini');
{$IFDEF USELOG}
    // Only with a local INI file we allow logging
    Parameter := GetParameter('Demux', 'LogLevel', '0');
    Val(Parameter, LogLevel, Error);
    if Error <> 0 then
      LogLevel := 0;
    Parameter := GetParameter('Demux', 'LogClear', 'Yes');
    if LowerCase(Parameter) = 'no' then
      LogClear := False
    else
      LogClear := True;
{$ENDIF}
  end;
{$IFDEF USELOG}
  if LogLevel in [1..5] then
  begin
    try
      if (not LogClear) and
         FileExists('Saa7146aDemux.log') then
      begin
        LogStream := TFileStream.Create('Saa7146aDemux.log', fmOpenReadWrite);
        // Go to end of file (we will be appending)
        LogStream.Seek(0, soFromEnd);
      end
      else
        LogStream := TFileStream.Create('Saa7146aDemux.log', fmCreate);
      ToLog('-------------------------------------------------------------', $00);
      ToLog(format('Log level: %d', [LogLevel]), $00);
    except
      FreeAndNil(LogStream);
    end;
  end;
{$ENDIF}

  // Get IoCtl identifiers
  Parameter := LowerCase(GetParameter('IoCtl', 'DMX_BACKDOOR_START_FEED', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_DMX_BACKDOOR_START_FEED)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'DMX_BACKDOOR_STOP_FEED', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_DMX_BACKDOOR_STOP_FEED)] := Value;
  Parameter := LowerCase(GetParameter('IoCtl', 'DMX_REGISTER_CALLBACK', 'X'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    IoCtlIdentifier[Ord(IoCtl_DMX_REGISTER_CALLBACK)] := Value;

  // Driver settings
  Parameter := LowerCase(GetParameter('Demux', 'Device', '0'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    CardToUse := Value;
  Parameter := LowerCase(GetParameter('Demux', 'TSBufferSize', '500'));
  Val(Parameter, Value, Error);
  if Error = 0 then
    BufferSize := Value;

  // RPS0Program setting
  Parameter := GetParameter('Demux', 'RPS0Program', '-1');
  Val(Parameter, Rps0Program, Error);
  if Error <> 0 then
    Rps0Program := -1;

  // *** Next are taken from the frontend settings ****
  // Get tuner type
  CheckString := LowerCase(GetParameter('FrontEnd', 'Tuner', 'BSRU6'));
  if LowerCase(CheckString) = 'su1278' then
    TunerType := CTunerSU1278
  else
    if LowerCase(CheckString) = 'bsbe1' then
      TunerType := CTunerBSBE1
    else
      TunerType := CTunerBSRU6;
  // Get tuner reset line
  Parameter := LowerCase(GetParameter('FrontEnd', 'TunerReset', '2'));
  Val(Parameter, TunerReset, Error);
  if Error <> 0 then
    TunerReset := 2;
  if TunerReset > 3 then
    TunerReset := 255;
  // Get card type (overrules other settings)
  // Get card type (overrules other settings)
  DvbCard := GetParameter('FrontEnd', 'Card', '');
  if (UpperCase(DvbCard) = 'TTPCLINEBUDGET') or
     (UpperCase(DvbCard) = 'NOVA') or
     (UpperCase(DvbCard) = 'SATELCO') then
  begin
    TunerType          := CTunerBSRU6;
    TunerReset         := 2;
  end;
  if (UpperCase(DvbCard) = 'TTPCLINEBUDGET2') or
     (UpperCase(DvbCard) = 'NOVA2') then
  begin
    TunerType          := CTunerSU1278;
    TunerReset         := 2;
  end;
  if (UpperCase(DvbCard) = 'TERRATEC') or
     (UpperCase(DvbCard) = 'KNC') or
     (UpperCase(DvbCard) = 'TYPHOON') or
     (UpperCase(DvbCard) = 'ANUBIS') then
  begin
    TunerType          := CTunerSU1278;
    TunerReset         := 255;
  end;


  if Saa7146aGetNumberOfCards <> 0 then
    DriverHandle := Saa7146aCreateFile(CardToUse);

  if DriverHandle <> INVALID_HANDLE_VALUE then
  begin
    // Setup data transfer from tuner to memory and initialize tuner
    if not DvbSetupBuffering(DriverHandle, True, TunerType, TunerReset, @IsSlave, @BufferSize, @BufferId, Rps0Program) then
    begin
{$IFDEF USELOG}
      ToLog('Saa7146aDemux Initializing: COULD NOT SETUP BUFFERING MECHANISM.', $03);
{$ENDIF}
      Exit;
    end;
  end
  else
  begin
{$IFDEF USELOG}
    ToLog('Saa7146aDemux Initializing: NO CARD FOUND.', $03);
{$ENDIF}
    Exit;
  end;
{$IFDEF USELOG}
  ToLog(format('Saa7146aDemux Initializing: %d buffers.', [BufferSize]), $03);
{$ENDIF}

  // Start the thread which receives notifications
  PacketThread := TDataThread.Create(True);
  PacketThread.FBufferId      := BufferId;
  PacketThread.FPacketBuffers := BufferSize;
  PacketThread.Resume;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  try
{$IFDEF USELOG}
    ToLog('Saa7146aDemux Finalizing.', $03);
{$ENDIF}
    // Stop data receiving thread
    try
      if Assigned(PacketThread) and
        (not PacketThread.FHasStopped) then
      begin
        // Note: The behaviour of the thread (during finalization) is different
        //       if called from the DLL finalizepart. It seems the thread is
        //       then already 'stalled'. Waiting with 'sleeps' is also slow
        //       because the finalize here seems to get very low priority.
        //       Therefore no waiting in a loop or simular.
        PacketThread.FProgrammedStop := True;
        PacketThread.Terminate;                                // Mark it for termination
        if PacketThread.Suspended then
          PacketThread.Resume;
        // If we are waiting for a notification then generate (end) it manually.
        // Note the use of <DriverHandle> and NOT the handle from the thread. This is because
        // the other handle might be 'busy' with a <Saa7146aWaitForNotification>!
        // Because the thread might not react directly or requires multiple
        // notification we allow for that.
        Saa7146aGenerateManualNotification(DriverHandle);
        Sleep(20);
        Saa7146aGenerateManualNotification(DriverHandle);
        Sleep(100);
//        PacketThread.WaitFor;  For some reason never expires when DLL is dynamically used
        PacketThread.Free;
      end
      else
      begin
{$IFDEF USELOG}
        ToLog('Saa7146aDemux Finalizing: DATA THREAD WAS ALREADY TERMINATED.', $81);
{$ENDIF}
      end;
    except
{$IFDEF USELOG}
      ToLog('Saa7146aDemux Finalizing: DATA THREAD EXCEPTION DURING TERMINATION.', $81);
{$ENDIF}
    end;

{$IFDEF USELOG}
    ToLog('Saa7146aDemux Finalizing: Closing log.', $03);
    if Assigned(LogStream) then
      FreeAndNil(LogStream);
{$ENDIF}
    if Assigned(IniFile) then
      FreeAndNil(IniFile);
  except
  end;
end;


initialization
  Initialize;

finalization
  Finalize;

end.

