//{******************************************************************************}
//{ FileName............: Saa7146aDemuxTest                                      }
//{ Project.............: SAA7146A                                               }
//{ Author(s)...........: MM                                                     }
//{ Version.............: 1.00                                                   }
//{------------------------------------------------------------------------------}
//{  Demux (+Frontend) DLL sample                                                }
//{                                                                              }
//{  Copyright (C) 2003-2004  M.Majoor                                           }
//{                                                                              }
//{  This program is free software; you can redistribute it and/or               }
//{  modify it under the terms of the GNU General Public License                 }
//{  as published by the Free Software Foundation; either version 2              }
//{  of the License, or (at your option) any later version.                      }
//{                                                                              }
//{  This program is distributed in the hope that it will be useful,             }
//{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
//{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
//{  GNU General Public License for more details.                                }
//{                                                                              }
//{  You should have received a copy of the GNU General Public License           }
//{  along with this program; if not, write to the Free Software                 }
//{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
//{                                                                              }
//{------------------------------------------------------------------------------}
//{                                                                              }
//{ Version   Date   Comment                                                     }
//{  1.00   20040923 - Initial DLL release                                       }
//{******************************************************************************}


#include "stdafx.h"
#include <stdio.h>

// DLL defines
#include "Saa7146aFrontEndDefines.h"
#include "Saa7146aDemuxDefines.h"

WORD  PacketCount   = 0;
WORD  PacketSync    = 0;
WORD  PacketNoSync  = 0;
typedef CHAR PacketBuffer [188];


/* ------------------------------------------------------------------------------
  Params  : <Buffer>  Pointer to packet data
            <Length>  Length of data (always 188)
  Returns : <Result>  Not used

  Descript: Called for every packet with a PID which was set by _START_FEED
  Notes   :
------------------------------------------------------------------------------ */
DWORD __stdcall PacketCallback(PacketBuffer Buffer, DWORD DataLength)
{
  // Increase counter
  if (PacketCount == 0xFFFF)
    PacketCount = 0;
  else
    PacketCount += 1;
  // For demonstration purposes just check the first byte (which should be $47)
  if (Buffer[0] != 0x47)
  {
    if (PacketNoSync == 0xFFFF)
      PacketNoSync = 0;
    else
      PacketNoSync += 1;
  }
  else
  {
    if (PacketSync == 0xFFFF)
      PacketSync = 0;
    else
      PacketSync += 1;
  }

  return 0;
}


int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
  CHAR             ResultString[2000];
  INT              ResultLength = 0;
  INT              FResult;

  // Get the library functions
  HINSTANCE                Saa7146aFrontEndDLL;            // Handle to DLL
  HINSTANCE                Saa7146aDemuxDLL;               // Handle to DLL
  pSaa7146aFrontEndOpen    Saa7146aFrontEndOpen;           // Function pointers
  pSaa7146aFrontEndClose   Saa7146aFrontEndClose;
  pSaa7146aFrontEndIoCtl   Saa7146aFrontEndIoCtl;
  pSaa7146aDemuxIoCtl      Saa7146aDemuxIoCtl;

  //{******************************************************************************}
  //                                  DLL'S
  //{******************************************************************************}
  // We use the DLL's dynamically
  Saa7146aFrontEndDLL = LoadLibrary(Saa7146aFrontEndName);
  if (Saa7146aFrontEndDLL != NULL)
  {
    // Get functions dynamically
    Saa7146aFrontEndOpen   = (pSaa7146aFrontEndOpen) GetProcAddress(Saa7146aFrontEndDLL, "Open");
    Saa7146aFrontEndClose  = (pSaa7146aFrontEndClose)GetProcAddress(Saa7146aFrontEndDLL, "Close");
    Saa7146aFrontEndIoCtl  = (pSaa7146aFrontEndIoCtl)GetProcAddress(Saa7146aFrontEndDLL, "IoCtl");
    if (!Saa7146aFrontEndOpen  ||
        !Saa7146aFrontEndClose ||
        !Saa7146aFrontEndIoCtl)
    {
       FreeLibrary(Saa7146aFrontEndDLL);
       sprintf(ResultString, "Some functions in '%s.dll' not found.\n", Saa7146aFrontEndName);
       MessageBox(NULL, ResultString, NULL, MB_OK | MB_ICONHAND);
       return -1;
    }
  }
  else
  {
    // No DLL found ...
    sprintf(ResultString, "'%s.dll' could not be found.\n", Saa7146aFrontEndName);
    MessageBox(NULL, ResultString, NULL, MB_OK | MB_ICONHAND);
    return -1;
  }

  Saa7146aDemuxDLL = LoadLibrary(Saa7146aDemuxName);
  if (Saa7146aDemuxDLL != NULL)
  {
    // Get functions dynamically
    Saa7146aDemuxIoCtl  = (pSaa7146aDemuxIoCtl)GetProcAddress(Saa7146aDemuxDLL, "IoCtl");
    if (!Saa7146aDemuxIoCtl)
    {
       FreeLibrary(Saa7146aFrontEndDLL);
       FreeLibrary(Saa7146aDemuxDLL);
       sprintf(ResultString, "Some functions in '%s.dll' not found.\n", Saa7146aDemuxName);
       MessageBox(NULL, ResultString, NULL, MB_OK | MB_ICONHAND);
       return -1;
    }
  }
  else
  {
    FreeLibrary(Saa7146aFrontEndDLL);
    // No DLL found ...
    sprintf(ResultString, "'%s.dll' could not be found.\n", Saa7146aDemuxName);
    MessageBox(NULL, ResultString, NULL, MB_OK | MB_ICONHAND);
    return -1;
  }


  //{******************************************************************************}
  //                                  FRONTEND
  //{******************************************************************************}
  // To use any of the IOCTL functions we must open the device.
  // Lets open the device. Name is always 'frontend' and the number identifies the card.
  // Note: For all IOCTL's we assume the INI file is in its default state, meaning that
  //       the IOCTL's defined there are also default.
  CHAR  DeviceName[128];
  INT   FCardHandle;
  strcpy(DeviceName, "frontend0\0");
  FCardHandle = Saa7146aFrontEndOpen(DeviceName, 0);

  ResultLength += sprintf(ResultString + ResultLength, "Frontend handle for '%s': %d.\n", DeviceName, FCardHandle);


  // Get the tuner definition
  PTDvbFrontEndInfo  Stv0299bDvbFrontEndInfo;
  Stv0299bDvbFrontEndInfo = new TDvbFrontEndInfo;
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_GET_INFO, Stv0299bDvbFrontEndInfo);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error FE_GET_INFO: %d.\n", FResult);
  else
  {
    ResultLength += sprintf(ResultString + ResultLength, "Frontend     %s.\n", Stv0299bDvbFrontEndInfo->Name);
    ResultLength += sprintf(ResultString + ResultLength, "Frequency    %d - %d.\n", Stv0299bDvbFrontEndInfo->FrequencyMin, Stv0299bDvbFrontEndInfo->FrequencyMax);
    ResultLength += sprintf(ResultString + ResultLength, "Symbolrates  %d - %d.\n", Stv0299bDvbFrontEndInfo->SymbolRateMin, Stv0299bDvbFrontEndInfo->SymbolRateMax);
    ResultLength += sprintf(ResultString + ResultLength, "Capabilities $%8.8x.\n", Stv0299bDvbFrontEndInfo->Capabilities);
  }
  delete Stv0299bDvbFrontEndInfo;


  // Let's tune to a frequency
  // We use the following parameters (Astra 192.E):
  //  Frequency:  11836 (GHz)
  //  Polarity:   H
  //  Symbolrate: 27500
  //  FEC       : Auto  (FYI: transponder uses 3/4)

  // Note: We have to convert the '11836' (modulated) frequency into a TUNER
  //       (demodulated) frequency.
  //       This implies we have to take the LOFs of the LNB into account,
  //       which are typically 9750 and 10600.
  //       The selection of the two is done by means of the 22 kHz signal
  //       Example:  Possibilities we could use
  //                     11836 -  9750  = 2086
  //                 or  11836 - 10600  = 1236
  // From the tuner information we get the allowed range (eg. 950000 - 2150000 kHz,
  // which is 950 - 2150 GHz.
  // Since both our values (2086/1236) are within that range we can use either the low
  // (9750) or high band (10600) setting. In other cases we are typically forced to use
  // one of tho available bands.
  // In this case we will be using the high band. This means we will be using
  // 1236 as our tuning frequency and we must selct thenhigh band (22 kHz signal active).

  PTDvbFrontEndParameters   FrontEndParameter;
  FrontEndParameter = new  TDvbFrontEndParameters;
  FrontEndParameter->Frequency       = 11836000 - 10600000;
  FrontEndParameter->Inversion       = FALSE;
  FrontEndParameter->Qpsk.FecInner   = FEC_AUTO;
  FrontEndParameter->Qpsk.SymbolRate = 27500;

  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_SET_FRONTEND, FrontEndParameter);
  delete FrontEndParameter;

  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error FE_SET_FRONTEND: %d.\n", FResult);
  else
    ResultLength += sprintf(ResultString + ResultLength, "Frontend set.\n");

  // Tone off (low band) or tone on (high band selection)
//  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_SET_TONE, (PVOID)SEC_TONE_OFF);
//  if (FResult != ENOERROR)
//    ResultLength += sprintf(ResultString + ResultLength, "Error FE_SET_TONE.\n");
//  else
//    ResultLength += sprintf(ResultString + ResultLength, "Tone OFF.\n");
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_SET_TONE, (PVOID)SEC_TONE_ON);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error FE_SET_TONE.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "Tone ON.\n");

  // Polarity
//  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_SET_VOLTAGE, (PVOID)SEC_VOLTAGE_18);
//  if (FResult != ENOERROR)
//    ResultLength += sprintf(ResultString + ResultLength, "Error setting polarity.\n");
//  else
//    ResultLength += sprintf(ResultString + ResultLength, "Vertical polarity.\n");
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_SET_VOLTAGE, (PVOID)SEC_VOLTAGE_18);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error setting polarity.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "Horizontal polarity .\n");

  // Other things which might have to be setup are:
  //  . Satellite -> by means of burst and/or DiSEqC
  // Lets select a satellite by means of a DiSEqC command
  PTDvbDiSEqCMasterCmd DiSEqC;
  int                  Repeats   = 0;            // If DiSEqC devices are places in series then they need to be repeated
                                                 // Repeats = devices in series - 1
  int                  Satellite = 0;            // Satellite to select
  DiSEqC = new TDvbDiSEqCMasterCmd;
  DiSEqC->Msg[0] = (CHAR)0xE0;                   // Framing: Command from master, no reply required, first transmission
  DiSEqC->Msg[1] = (CHAR)0x10;                   // Address: Any LNB/switcher
  DiSEqC->Msg[2] = (CHAR)0x38;                   // Command: Committed sitches (DiSEqc level 1.0)
  switch (Satellite)
  {
    case 0:  DiSEqC->Msg[3] = (CHAR)0xF0;        // Data   : Switch ID in low nibble 1
    case 1:  DiSEqC->Msg[3] = (CHAR)0xF8;        // Data   : Switch ID in low nibble 9
    case 2:  DiSEqC->Msg[3] = (CHAR)0xF4;        // Data   : Switch ID in low nibble 5
    case 3:  DiSEqC->Msg[3] = (CHAR)0xFC;        // Data   : Switch ID in low nibble 13
    default: DiSEqC->Msg[3] = (CHAR)0xF0;        // Data   : Switch ID in low nibble 1
  }
  DiSEqC->MsgLen = 4;
  while (Repeats >= 0)
  {
    FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_DISEQC_SEND_MASTER_CMD, DiSEqC);
    if (FResult != ENOERROR)
      ResultLength += sprintf(ResultString + ResultLength, "Unable to send DiSEqC master command.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "DiSEqC command send.\n");
    DiSEqC->Msg[0] = (CHAR)0xE1;                 // Framing: Command from master, no reply required, repeated transmission
    if (Repeats >= 0)
      Sleep(100);                                // When there is a repeated command we need a 100 ms delay in between
    Repeats = Repeats - 1;
  }
  delete DiSEqC;

  // Alternative (older method) for selecting a satellite is using a burst
  // Note: It typically can co-exists with a DiSEqC command, but some switches are
  //       very typicali - they might only work correctly when only a burst is used
  //       and no DiSEqC command...
  if (Satellite == 0)
  {
    FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_DISEQC_SEND_BURST, (PVOID)SEC_MINI_A);
    if (FResult != ENOERROR)
      ResultLength += sprintf(ResultString + ResultLength, "Error sending burst A.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "Burst A send.\n");
  }
  if (Satellite == 1)
  {
    FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_DISEQC_SEND_BURST, (PVOID)SEC_MINI_B);
    if (FResult != ENOERROR)
      ResultLength += sprintf(ResultString + ResultLength, "Error sending burst B.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "Burst B send.\n");
  }



  // We are now tuned to a channel (assuming the first satellite is Astra)
  // Get the status of the tuner to check if we are tune
  int  Status;
  Sleep(100);                                    // Make sure time enough for stabilization
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_READ_STATUS, &Status);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error reading status.\n");
  else
  {
    ResultLength += sprintf(ResultString + ResultLength, "\nStatus information\n");
    ResultLength += sprintf(ResultString + ResultLength, "==================\n");
    if ((Status & FE_HAS_SIGNAL) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "Found something above the noise level.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "No signal above the noise level.\n");
    if ((Status & FE_HAS_CARRIER) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "Found a DVB signal.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "No DVB signal found.\n");
    if ((Status & FE_HAS_VITERBI) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "FEC is stable.\n");
    else
      ResultLength += sprintf(ResultString + ResultLength, "FEC is not stable.\n");
    if ((Status & FE_HAS_SYNC) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "Everything''s working...\n");
    if ((Status & FE_TIMEDOUT) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "No lock within the last ~2 seconds.\n");
    if ((Status & FE_REINIT) != 0)
      ResultLength += sprintf(ResultString + ResultLength, "Frontend was reinitialized.\n");
  }

  // And some other information
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_READ_BER, &Status);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error reading BER status.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "BER: %d.\n", Status);

  // And some other information
  WORD  Signal;
  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_READ_SIGNAL_STRENGTH, &Signal);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error reading signal strength.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "Signal strength: %d.\n", (Signal * 100) / 65535);
  // And some other information

  FResult = Saa7146aFrontEndIoCtl(FCardHandle, FE_READ_SNR, &Signal);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Error reading signal/noise.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "S/N: %d.\n", (Signal * 100) / 65535);




  //{******************************************************************************}
  //                                  DEMUX 
  //{******************************************************************************}
  ResultLength += sprintf(ResultString + ResultLength, "\nPacket data\n");
  ResultLength += sprintf(ResultString + ResultLength, "===========\n");
  // When data is being received it has to go to somewhere. For that we have to
  // set a callback procedure which will receive the data.
  PacketCount   = 0;
  PacketSync    = 0;
  PacketNoSync  = 0;
  FResult = Saa7146aDemuxIoCtl(0, DMX_REGISTER_CALLBACK, &PacketCallback);


  // Set the PIDs we like to receive. If we define 0xFFFF then all PIDs will be
  // received.
  PTDmxBackDoorFeed  DmxBackDoorFeed;
  DmxBackDoorFeed = new TDmxBackDoorFeed;
  // Special PID for enabling all
  DmxBackDoorFeed->Pid = 0xFFFF;
  FResult = Saa7146aDemuxIoCtl(0, DMX_BACKDOOR_START_FEED, DmxBackDoorFeed);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Could not start feed.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "Setting reception for all packets.\n");

  // Now delay for some time so the callback can receive it
  ResultLength += sprintf(ResultString + ResultLength, "Delaying 1 second.\n");
  Sleep(1000);
  

  // Stop all PIDs
  DmxBackDoorFeed->Pid = 0xFFFF;
  FResult = Saa7146aDemuxIoCtl(0, DMX_BACKDOOR_STOP_FEED, DmxBackDoorFeed);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Could not stop feed.\n");

  ResultLength += sprintf(ResultString + ResultLength, "Total packets received: %d of which correct %d.\n", PacketCount, PacketSync);

  DmxBackDoorFeed->Pid = 0;
  FResult = Saa7146aDemuxIoCtl(0, DMX_BACKDOOR_START_FEED, DmxBackDoorFeed);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Could not start feed for PID 0.\n");
  else
    ResultLength += sprintf(ResultString + ResultLength, "Setting reception for only PID 0 packets.\n");

  // Now delay for some time so the callback can receive it
  ResultLength += sprintf(ResultString + ResultLength, "Delaying 1 second.\n");
  Sleep(1000);

  DmxBackDoorFeed->Pid = 0xFFFF;
  FResult = Saa7146aDemuxIoCtl(0, DMX_BACKDOOR_STOP_FEED, DmxBackDoorFeed);
  if (FResult != ENOERROR)
    ResultLength += sprintf(ResultString + ResultLength, "Could not stop feed.\n");

  ResultLength += sprintf(ResultString + ResultLength, "Total packets received: %d of which correct %d.\n", PacketCount, PacketSync);

  delete DmxBackDoorFeed;
  

  // Disable the callback
  FResult = Saa7146aDemuxIoCtl(0, DMX_REGISTER_CALLBACK, NULL);


  // Close the device
  Saa7146aFrontEndClose(FCardHandle);

  FreeLibrary(Saa7146aFrontEndDLL);
  FreeLibrary(Saa7146aDemuxDLL);
  // Display results
  MessageBox(NULL, ResultString, "Information", MB_OK | MB_ICONINFORMATION);
  return 0;
}
