{ **************************************************************************** }
{ FileName............: MAIN.PAS                                               }
{ Project.............: RC5 Pronto                                             }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{ ---------------------------------------------------------------------------- }
{ Learned code generator RC5 IrDA device on parallel port.                     }
{ Supports both the internal format and the Pronto format.                     }
{                                                                              }
{ ---------------------------------------------------------------------------- }
{                                                                              }
{ The IrDA device on the parallel port is controlled by other software which   }
{ uses the PC timer as timing device. For the file format and additional       }
{ you should have a look on that particular software.                          }
{                                                                              }
{ Version    Date     Comment                                                  }
{   1.00    20011207  - Initial release                                        }
{                       Used RC5/Pronto code as base                           }
{                                                                              }
{ **************************************************************************** }
unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Grids, Mask, ExtCtrls;

type
  TfrmMain = class(TForm)
    btnExit: TButton;
    Label1: TLabel;
    Label2: TLabel;
    chkToggle1: TCheckBox;
    mskDelay1: TMaskEdit;
    Label3: TLabel;
    mmCode: TMemo;
    cbSystem2: TComboBox;
    cbCommand2: TComboBox;
    chkToggle2: TCheckBox;
    mskDelay2: TMaskEdit;
    cbSystem3: TComboBox;
    cbCommand3: TComboBox;
    chkToggle3: TCheckBox;
    mskDelay3: TMaskEdit;
    cbSystem4: TComboBox;
    cbCommand4: TComboBox;
    chkToggle4: TCheckBox;
    mskDelay4: TMaskEdit;
    cbSystem1: TComboBox;
    cbCommand1: TComboBox;
    chkToggle5: TCheckBox;
    mskDelay5: TMaskEdit;
    cbSystem6: TComboBox;
    cbCommand6: TComboBox;
    chkToggle6: TCheckBox;
    mskDelay6: TMaskEdit;
    cbSystem7: TComboBox;
    cbCommand7: TComboBox;
    chkToggle7: TCheckBox;
    mskDelay7: TMaskEdit;
    cbSystem8: TComboBox;
    cbCommand8: TComboBox;
    chkToggle8: TCheckBox;
    mskDelay8: TMaskEdit;
    cbSystem5: TComboBox;
    cbCommand5: TComboBox;
    cbSystem9: TComboBox;
    cbCommand9: TComboBox;
    chkToggle9: TCheckBox;
    mskDelay9: TMaskEdit;
    cbSystem10: TComboBox;
    cbCommand10: TComboBox;
    chkToggle10: TCheckBox;
    mskDelay10: TMaskEdit;
    Label4: TLabel;
    Label5: TLabel;
    chkEnabled1: TCheckBox;
    chkEnabled2: TCheckBox;
    chkEnabled3: TCheckBox;
    chkEnabled4: TCheckBox;
    chkEnabled5: TCheckBox;
    chkEnabled6: TCheckBox;
    chkEnabled7: TCheckBox;
    chkEnabled8: TCheckBox;
    chkEnabled9: TCheckBox;
    chkEnabled10: TCheckBox;
    Label6: TLabel;
    btnSaveBinary: TButton;
    SaveDialog1: TSaveDialog;
    btnSaveASCII: TButton;
    grpFormat: TRadioGroup;
    procedure btnExitClick(Sender: TObject);
    procedure ParameterChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure btnSaveBinaryClick(Sender: TObject);
    procedure btnSaveASCIIClick(Sender: TObject);
  private
    { Private declarations }
    function  GenerateCode(Toggle: boolean; SystemAddress, Command, Delay: word;
                           var Items: integer): string;
  public
    { Public declarations }
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.DFM}
uses
  IniFiles;

const
  Frequency   = 37000;                                       { Requested carrier frequency }

  { Pronto format specifics }
  HalfbitTime = 889E-06;                                     { 889 us half bit time }
  CarrierUnit = round(1E6/(Frequency  * 0.241246));          { Actual timing unit }
  Frequency2  = trunc(1E6/(CarrierUnit * 0.241246));         { Actual frequency }
  TimingUnitPronto  : word = round(HalfBitTime * Frequency2);      { -> 889 us / (1/Frequency) }

  { Internal format specifics }
  TimeUnit           = 838;                                  { 1 cycle == 0.838 us }
  TimeUnits          = 16;                                   { 16 cycles resolution }
  TimingUnit1        = 64;                                   { Base for 'frequency'  on/off signal }
  TimingUnitInternal = 66;                                   { Base for 'no signal' }



{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Exit button pressed.                                               }
{------------------------------------------------------------------------------}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
  Application.Terminate;
end;


{------------------------------------------------------------------------------}
{ Params  : <Toggle>         Toggle bit on or off                              }
{           <SystemAddress>  System address (0-31)                             }
{           <Command>        Command (0-127)                                   }
{           <Delay>          Delay in ms                                       }
{ Returns : <Result>         Result string                                     }
{                            Empty if invalid                                  }
{           <Items>          Number of sequences in result                     }
{                            i.e. a XXXX XXXX = 1 sequence                     }
{                                                                              }
{ Descript: Create a single RC5 code into Pronto/Internal format.              }
{------------------------------------------------------------------------------}
function TfrmMain.GenerateCode(Toggle: boolean; SystemAddress, Command, Delay: word;
                               var Items: integer): string;
var
  RC5Code           : word;
  RC5Result         : string;
  RC5String         : string;
  RC5Check          : string;
  Loop              : integer;
  Count1,
  Count0            : integer;
  DelayUnitPronto   : integer;
  DelayUnitInternal : integer;
begin
  Result    := '';
  RC5Result := '';
  Items := 0;
  if SystemAddress > 31   then Exit;
  if Command       > 127  then Exit;
  if Delay         > 5000 then Exit;

  { First we translate the system/command/toggle and such in straight binary }
  { Format of a RC5 sequence                                                 }
  {                                                                          }
  {  | S | S | T | A4 | A3 | A2 | A1 | A0 | C5 | C4 | C3 | C2 | C1 | C0 |    }
  {    xxSS TAAA AACC CCCC                                                   }
  {                                                                          }
  {  S = startbits                                                           }
  {    = '11' for command < 64                                               }
  {    = '10' for command >=64 (extended RC5)                                }
  {  T = toggle bit (inverted every repeat)                                  }
  {  A = address (system)                                                    }
  {  C = command                                                             }
  RC5Code := Command and $3F;
  if (Command and $40)<>0 then RC5Code := RC5Code or $2000
                          else RC5Code := RC5Code or $3000;
  RC5Code := RC5Code or (SystemAddress shl 6);
  if Toggle then RC5Code := RC5Code or $800;
  { Then we convert the bits into half bits (RC5 is bi-phase encoded)        }
  { '0' -> 10                                                                }
  { '1' -> 01                                                                }
  RC5String := '';
  RC5Check  := '';                                         { Just for debugging }
  for Loop := 1 to 14 do
  begin
    if (RC5Code and $2000)=0 then
    begin
      RC5Check  := RC5Check + '0';
      RC5String := RC5String + '10';
    end
    else
    begin
      RC5Check  := RC5Check + '1';
      RC5String := RC5String + '01';
    end;
    RC5Code := RC5Code shl 1;
  end;
  { We now want to know the 1->0 sequences.                                  }
  { The following sequences can occur:                                       }
  {  10    -> 1 unit  high, 1 unit  low                                      }
  {  100   -> 1 unit  high, 2 units low                                      }
  {  110   -> 2 units high, 1 unit  low                                      }
  {  1100  -> 2 units high, 2 units low                                      }

  { First we have some minor cleaning up to do. We must make sure we start   }
  { with a '1' and end with a '1'. A leading '0' can be removed and a        }
  { trailing '0' might be added.                                             }
  { Because we do this our string is always correct for the next stages.     }
  if (length(RC5STring)<>0) and (RC5String[1]='0') then
    Delete(RC5String, 1, 1);
  if (length(RC5String)<>0) and (RC5String[length(RC5String)]='1') then
    RC5String := RC5String + '0';

  DelayUnitPronto := round((Delay / 1000) * Frequency2);  { Delay in ms }
  if DelayUnitPronto < TimingUnitPronto then
    DelayUnitPronto := TimingUnitPronto;
  DelayUnitInternal := round((Delay * 1000) / ((TimeUnits * TimeUnit)/1000));         { Delay in ms }
  if DelayUnitInternal < TimingUnitInternal then
    DelayUnitInternal := TimingUnitInternal;

  { Now count them and while we are at it translate it into the result:      }
  { Each 1->0 sequence is translated into two numbers.                       }
  { These two numbers indicate the ON time (1) and the OFF time (0) of the   }
  { sequence. It is in units of the carrier frequency. We already calculated/}
  { defined the TimingUnits required for each bit in the string.             }
  { For the delay time we need to calculate it seperately.                   }
  while length(RC5String)<>0 do
  begin
    inc(Items);
    Count0 := 0;
    Count1 := 0;
    while (length(RC5String)<>0) and (RC5String[1]='1') do           { The added security (length check) is not really necessary }
    begin
      inc(Count1);
      delete(RC5String, 1, 1);
    end;
    while (length(RC5String)<>0) and (RC5String[1]='0') do           { The added security (length check) is not really necessary }
    begin
      inc(Count0);
      delete(RC5String, 1, 1);
    end;
    { At this point we know how long a '0'/'1' sequence should take }
    { Now all we have to do is convert this in the appropiate format }
    case grpFormat.ItemIndex of
      0: begin                                   { Pronto }
           if length(RC5String)<>0 then
             RC5Result := RC5Result + format(' %.4x %.4x', [Count1 * TimingUnitPronto, Count0 * TimingUnitPronto])
           else
             RC5Result := RC5Result + format(' %.4x %.4x', [Count1 * TimingUnitPronto, DelayUnitPronto]);
         end;
      1: begin                                   { Internal format }
           for Loop := 0 to (Count1*TimingUnit1)-2 do
             RC5Result := RC5Result + ' 01 00';
           if length(RC5String)<>0 then
             RC5Result := RC5Result + format(' %.2x %.2x', [(Count0 * TimingUnitInternal) and $FF, ((Count0 * TimingUnitInternal) shr 8) and $FF])
           else
             RC5Result := RC5Result + format(' %.2x %.2x', [(Count0 * DelayUnitInternal) and $FF, ((Count0 * DelayUnitInternal) shr 8) and $FF])
         end;
    end;
  end;
  { Now all we have to do is remove the single leading space }
  if (length(RC5Result)<>0) and (RC5Result[1]=' ') then
    Delete(RC5Result, 1, 1);

  Result := RC5Result;
end;


{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Change of a parameter.                                             }
{           If ANY of the parameters is changed the new code is generated.     }
{------------------------------------------------------------------------------}
procedure TfrmMain.ParameterChange(Sender: TObject);
var
  NrItems    : integer;
  TotalItems : integer;
  TotalString: AnsiString;
  PartString : string;
begin
  case grpFormat.ItemIndex of
    0: Caption := 'Learned code generator RC5 IrDA device on parallel port (Pronto format)';
    1: Caption := 'Learned code generator RC5 IrDA device on parallel port (internal format)';
  end;
  mmCode.Clear;                                  { Clear old code }

  TotalItems  := 0;
  PartString  := '';
  TotalString := '';

  { Extremely straightforward. Can be much, much more intelligent... but .. }
  if chkEnabled1.Checked then
  begin
    PartString := GenerateCode(chkToggle1.Checked, StrToInt(cbSystem1.Text), StrToInt(cbCommand1.Text), StrToInt(mskDelay1.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled2.Checked then
  begin
    PartString := GenerateCode(chkToggle2.Checked, StrToInt(cbSystem2.Text), StrToInt(cbCommand2.Text), StrToInt(mskDelay2.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled3.Checked then
  begin
    PartString := GenerateCode(chkToggle3.Checked, StrToInt(cbSystem3.Text), StrToInt(cbCommand3.Text), StrToInt(mskDelay3.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled4.Checked then
  begin
    PartString := GenerateCode(chkToggle4.Checked, StrToInt(cbSystem4.Text), StrToInt(cbCommand4.Text), StrToInt(mskDelay4.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled5.Checked then
  begin
    PartString := GenerateCode(chkToggle5.Checked, StrToInt(cbSystem5.Text), StrToInt(cbCommand5.Text), StrToInt(mskDelay5.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled6.Checked then
  begin
    PartString := GenerateCode(chkToggle6.Checked, StrToInt(cbSystem6.Text), StrToInt(cbCommand6.Text), StrToInt(mskDelay6.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled7.Checked then
  begin
    PartString := GenerateCode(chkToggle7.Checked, StrToInt(cbSystem7.Text), StrToInt(cbCommand7.Text), StrToInt(mskDelay7.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled8.Checked then
  begin
    PartString := GenerateCode(chkToggle8.Checked, StrToInt(cbSystem8.Text), StrToInt(cbCommand8.Text), StrToInt(mskDelay8.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled9.Checked then
  begin
    PartString := GenerateCode(chkToggle9.Checked, StrToInt(cbSystem9.Text), StrToInt(cbCommand9.Text), StrToInt(mskDelay9.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;
  if chkEnabled10.Checked then
  begin
    PartString := GenerateCode(chkToggle10.Checked, StrToInt(cbSystem10.Text), StrToInt(cbCommand10.Text), StrToInt(mskDelay10.EditText), NrItems);
    TotalString := TotalString + ' ' + PartString;
    inc(TotalItems, NrItems);
  end;

  { Now we have all our codes in a single string we only need to add the }
  { timing header (internal format only):                                }
  { TT TT 00 00 SS SS 00 00 CC CC 00 00 UU UU 00  00                     }
  PartString := '02 00 00 00';
  PartString := PartString + format(' %.2x %.2x 00 00', [(TimeUnits * 1000) and $FF, ((TimeUnits * 1000) shr 8) and $FF]);
  PartString := PartString + format(' %.2x %.2x 00 00', [1000 and $FF, (1000 shr 8) and $FF]);
  PartString := PartString + format(' %.2x %.2x 00 00', [TimeUnit and $FF, (TimeUnit shr 8) and $FF]);

  case grpFormat.ItemIndex of
    0: TotalString := '0000 ' + format('%.4x ', [CarrierUnit]) +
                      format('%.4x 0000', [TotalItems]) + TotalString;
    1: TotalString := PartString + TotalString;
  end;

  mmCode.Lines.Add(TotalString);
  mmCode.SelectAll;
  mmCode.CopyToClipboard;
end;


{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Destruction of form.                                               }
{           Save all current settings                                          }
{------------------------------------------------------------------------------}
procedure TfrmMain.FormDestroy(Sender: TObject);
var
  IniFile : TIniFile;
  I       : integer;
begin
  IniFile := TIniFile.Create(ExtractFilePath(ParamStr(0)) + 'RC5_IRDA.INI');
  try
    for I := 0 to ComponentCount -1 do
    begin
      if Components[I] is TCheckBox then
        with Components[I] as TCheckBox do IniFile.WriteBool  ('CheckBoxes'   , Name, Checked);
      if Components[I] is TComboBox then
        with Components[I] as TComboBox do IniFile.WriteString('ComboBoxes'   , Name, Text);
      if Components[I] is TMaskEdit then
        with Components[I] as TMaskEdit do IniFile.WriteString('MaskEditBoxes', Name, EditText);
      if Components[I] is TRadioGroup then
        with Components[I] as TRadioGroup do IniFile.WriteInteger('RadioGroups', Name, ItemIndex);
    end;
  finally
    IniFile.Free;
  end;
end;


{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Destruction of form.                                               }
{           Save all current settings                                          }
{------------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
  IniFile : TIniFile;
  Keys    : TStringList;
  I       : integer;
  ABool   : boolean;
  AnInt   : integer;
  AString : string;
begin
  Keys := TStringList.Create;
  try
    IniFile := TIniFile.Create(ExtractFilePath(ParamStr(0)) + 'RC5_IRDA.INI');
    try
      IniFile.ReadSection('CheckBoxes', Keys);
      for I := 0 to Keys.Count-1 do
      begin
        ABool := IniFile.ReadBool('CheckBoxes'   , Keys.Strings[I], false);
        with TCheckBox(FindComponent(Keys.Strings[I])) do Checked := ABool;
      end;
      IniFile.ReadSection('ComboBoxes', Keys);
      for I := 0 to Keys.Count-1 do
      begin
        AString := IniFile.ReadString('ComboBoxes'   , Keys.Strings[I], '0');
        with TComboBox(FindComponent(Keys.Strings[I])) do Text := AString;
      end;
      IniFile.ReadSection('MaskEditBoxes', Keys);
      for I := 0 to Keys.Count-1 do
      begin
        AString := IniFile.ReadString('MaskEditBoxes'   , Keys.Strings[I], '0000');
        with TMaskEdit(FindComponent(Keys.Strings[I])) do EditText := AString;
      end;
      IniFile.ReadSection('RadioGroups', Keys);
      for I := 0 to Keys.Count-1 do
      begin
        AnInt := IniFile.ReadInteger('RadioGroups'   , Keys.Strings[I], 0);
        with TRadioGroup(FindComponent(Keys.Strings[I])) do ItemIndex := AnInt;
      end;
    finally
      IniFile.Free;
    end;
  finally
    Keys.Free;
  end;
end;


{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Save data as binary file.                                          }
{------------------------------------------------------------------------------}
procedure TfrmMain.btnSaveBinaryClick(Sender: TObject);
var
  TheFile: file of byte;
  AString: string;
  Part   : string;
  Value  : word;
  AByte  : byte;
  Error  : integer;
  Loop   : integer;
begin
  if mmCode.Lines.Count=0 then Exit;
  if SaveDialog1.Execute then
  begin
    AssignFile(TheFile, SaveDialog1.FileName);
    Rewrite(TheFile);
    try
      for Loop := 0 to mmCode.Lines.Count-1 do
      begin
        AString := mmCode.Lines.Strings[Loop];
        while (length(AString)<>0) do
        begin
          Part := '';
          while (length(AString)<>0) and (AString[1]=' ') do
            Delete(AString, 1, 1);
          while (length(AString)<>0) and (AString[1]<>' ') do
          begin
            Part := Part + AString[1];
            Delete(AString, 1, 1);
          end;
          if Part<>'' then
          begin
            Part := '$' + Part;
            Val(Part, Value, Error);
            if Error = 0 then
              case grpFormat.ItemIndex of
                0: begin
                     { We save high byte first } 
                     AByte := (Value shr 8) and $FF;
                     Write(TheFile, AByte);
                     AByte := Value and $FF;
                     Write(TheFile, AByte);
                   end;
                1: Write(TheFile, Value);
              end;
          end;
        end;
      end;
    finally
      CloseFile(TheFile);
    end;  
  end;
end;

{------------------------------------------------------------------------------}
{ Params  : <Sender>  Sender object                                            }
{ Returns : -                                                                  }
{                                                                              }
{ Descript: Save data as ASCII file.                                           }
{------------------------------------------------------------------------------}
procedure TfrmMain.btnSaveASCIIClick(Sender: TObject);
var
  TheFile : file of byte;
  AByte   : byte;
  AString : string;
  Loop    : integer;
begin
  if mmCode.Lines.Count=0 then Exit;
  if SaveDialog1.Execute then
  begin
    AssignFile(TheFile, SaveDialog1.FileName);
    Rewrite(TheFile);
    try
      for Loop := 0 to mmCode.Lines.Count-1 do
      begin
        AString := mmCode.Lines.Strings[Loop];
        while (length(AString)<>0) do
        begin
          AByte := ord(AString[1]);
          Delete(AString, 1, 1);
          Write(TheFile, AByte);
        end;
      end;
    finally
      CloseFile(TheFile);
    end;
  end;
end;


end.
