--{******************************************************************************}
--{ FileName............: main_i2c.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2026 - 2026
--{------------------------------------------------------------------------------}
--{
--{ Implements:
--{ . Main clock
--{ . Blinky LED
--{ . Debug data on P2 PMOD port
--{ . Waits for keyboard input
--{ . Reads, via I2C, the EDID (first 128 bytes) from the connected display
--{ . Displays the EDID data while the data is being read from the display
--{ . Reads, via I2C, the real-time clock 'seconds' register
--{ . Displays the real-time clock 'seconds' data
--{
--{ Version       Comment
--{ 2026.01.03.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity main_i2c is
  port (
    clk_in      : in  std_logic;
    -- Buttons
    reset_button: in  std_logic;
    -- LED indications
    led         : out std_logic;
    led_g       : out std_logic;
    led_r       : out std_logic;
    -- UART
    uart_txd    : out std_logic;
    rsrx        : in  std_logic;
    -- To/From HDMI I2C
    hdmi_scl    : out   std_logic;
    hdmi_sda    : inout std_logic;
    -- To/From onboard I2C
    fpga_scl    : out   std_logic;
    fpga_sda    : inout std_logic;
    -- Debugging
    p2lo        : out std_logic_vector(3 downto 0);     
    p2hi        : out std_logic_vector(3 downto 0)    
  );
end main_i2c;


architecture Behavioral of main_i2c is
  constant  STRINGSIZE: integer := 40;
  constant  OUTPUTBITS: integer := 8;
  -- Our clock signal we derive our LED signals from
  signal Clock200                  : std_logic;
  -- Serial input
  signal SerialInBitStreamIn       : std_logic;
  signal SerialInDataOutByte       : std_logic_vector(7 downto 0);
  signal SerialInDataOutToggle     : std_logic;
  signal SerialInDataOutTogglePrev : std_logic := '0';
  -- Serial output
  signal SerialOutDataOutByte      : std_logic_vector(7 downto 0);
  signal SerialOutDataEnable       : std_logic;
  signal SerialOutBitStreamOut     : std_logic;
  signal SerialOutBusy             : std_logic;  
  -- Counters
  signal Led_Count                 : natural range 0 to 200000000;  
  -- LED toggle status
  signal Led_Toggle                : std_logic;  

  -- I2C (HDMI)
  signal HdmiI2cScl        : std_logic;
  signal HdmiI2cSda        : std_logic;
  signal HdmiI2cOperations : string(1 to STRINGSIZE);
  signal HdmiI2cStart      : std_logic;
  signal HdmiI2cData       : std_logic_vector(OUTPUTBITS-1 downto 0);      
  signal HdmiI2cError      : std_logic;
  signal HdmiI2cReady      : std_logic;
  signal HdmiI2cDebug      : std_logic_vector(7 downto 0);
  signal HdmiI2cRamIndex   : integer;
  
  -- I2C (onboard)
  signal MegaI2cScl        : std_logic;
  signal MegaI2cSda        : std_logic;
  signal MegaI2cOperations : string(1 to STRINGSIZE);
  signal MegaI2cStart      : std_logic;
  signal MegaI2cData       : std_logic_vector(OUTPUTBITS-1 downto 0);      
  signal MegaI2cError      : std_logic;
  signal MegaI2cReady      : std_logic;
  signal MegaI2cDebug      : std_logic_vector(7 downto 0);
  
  constant ASCIIHEX           : string := "0123456789ABCDEF";
  signal   AsciiMessageIndex1 : integer range 1 to 20;
  signal   AsciiMessageIndex2 : integer range 1 to 16;
  
  signal   I2cState           : integer range 0 to 63 := 0;
  -- Special attribute forcing the state machine in a specific state for invalid states
  attribute fsm_safe_state    : string;
  attribute fsm_safe_state of I2cState : signal is "power_on_state";
  --
  signal   I2cOperation       : integer;

 
begin
  -- ** From 'mega65r6.vhdl':
  -- New clocking setup, using more optimised selection of multipliers
  -- and dividers, as well as the ability of some clock outputs to provide an
  -- inverted clock for free.
  -- Also, the 50 and 100MHz ethernet clocks are now independent of the other
  -- clocks, so that Vivado shouldn't try to meet timing closure in the (already
  -- protected) domain crossings used for those.
  clocks1: entity work.clocking
    port map ( 
      clk_in     => clk_in,
      clock27    => open,      -- clock27,    --   27     MHz
      clock41    => open,      -- cpuclock,   --   40.5   MHz
      clock50    => open,      -- ethclock,   --   50     MHz
      clock74p22 => open,      -- clock74p22,
      clock81p   => open,      -- pixelclock, --   81     MHz
      clock163   => open,      -- clock162,   --  162     MHz
      clock163m  => open,      -- clock162m,  --  162     MHz, phase shifted by -207 degrees for SDRAM read timing
      clock200   => Clock200,  -- clock200,   --  200     MHz
      clock270   => open,      -- clock270,   --  270     MHz
      clock325   => open       -- clock325    --  325     MHz
    );


  -- Instantiation of the serial input
  SerialIn_I: entity work.serialin
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk           => Clock200,
      bitstream     => SerialInBitStreamIn,
      dataout       => SerialInDataOutByte,
      dataouttoggle => SerialInDataOutToggle
    );

  -- Instantiation of the serial output
  SerialOut_I: entity work.serialout
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk        => Clock200,
      datain     => SerialOutDataOutByte,
      dataenable => SerialOutDataEnable,
      bitstream  => SerialOutBitStreamOut,
      busy       => SerialOutBusy
    );

  -- Instantiation of the I2C (HDMI)
  HdmiI2c_I: entity work.i2c
    generic map (
      CLKRATE    => 200000000,
      I2CRATE    => 100000,
      STRINGSIZE => STRINGSIZE,
      OUTPUTBITS => OUTPUTBITS
    )
    port map (
      clk        => Clock200,
      scl        => HdmiI2cScl,
      sda        => HdmiI2cSda,
      operations => HdmiI2cOperations,
      start      => HdmiI2cStart,
      data       => HdmiI2cData,
      error      => HdmiI2cError,
      ready      => HdmiI2cReady,
      debug      => HdmiI2cDebug
    );


  -- Connect the I2C to the display adapter
  hdmi_scl <= HdmiI2cScl;
  hdmi_sda <= HdmiI2cSda;
  
  -- Instantiation of the I2C (onboard)
  MegaI2c_I: entity work.i2c
    generic map (
      CLKRATE    => 200000000,
      I2CRATE    => 100000,
      STRINGSIZE => STRINGSIZE,
      OUTPUTBITS => OUTPUTBITS
    )
    port map (
      clk        => Clock200,
      scl        => MegaI2cScl,
      sda        => MegaI2cSda,
      operations => MegaI2cOperations,
      start      => MegaI2cStart,
      data       => MegaI2cData,
      error      => MegaI2cError,
      ready      => MegaI2cReady,
      debug      => MegaI2cDebug
    );


  -- Connect the I2C to onboard peripherals
  fpga_scl <= MegaI2cScl;
  fpga_sda <= MegaI2cSda;
  
  --{------------------------------------------------------------------------------}
  --{ Descript: I2c handler
  --{           Waits for serial input and then starts reading and displaying EDID data
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  variable LChar : character;
  variable LOrd  : integer;
  variable LVec  : std_logic_vector(7 downto 0);
  begin
    if rising_edge(Clock200) then
      case I2cState is
        when 0      => -- If all instances are ready
                       if (HdmiI2cReady = '1') and (MegaI2cReady = '1') and (SerialOutBusy = '0') then
                         -- Next state waits for user input
                         I2cState             <= 1;
                         SerialOutDataOutByte <= x"00";
                         SerialOutDataEnable  <= '0';
                       end if;
        -- Start wait for user input                 
        when 1      => SerialInDataOutTogglePrev <= SerialInDataOutToggle;
                       I2cState                  <= 2;
        -- Wait for any user input               
        when 2      => if SerialInDataOutTogglePrev /= SerialInDataOutToggle then
                         -- Assume we need to wait for a next command next
                         I2cState <= 20;
                       end if;

        -- Output as hex with space               
        -- Output high nibble               
        when 10     => LChar := ASCIIHEX(AsciiMessageIndex1);                              
                       LOrd  := character'pos(LChar);
                       SerialOutDataOutByte  <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable   <= '1';
                       I2cState              <= 11;
        -- Wait for character accepted               
        when 11     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 12;
                       end if;               
        -- Wait for serial character send               
        when 12      => if SerialOutBusy = '0' then
                         I2cState <= 13;
                       end if;                      
        -- Output low nibble               
        when 13     => LChar := ASCIIHEX(AsciiMessageIndex2);                              
                       LOrd  := character'pos(LChar);
                       SerialOutDataOutByte  <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable   <= '1';
                       I2cState              <= 14;
        -- Wait for character accepted               
        when 14     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 15;
                       end if;               
        -- Wait for serial character send               
        when 15     => if SerialOutBusy = '0' then
                         I2cState <= 16;
                       end if;                                      
        -- Output space or LF (assume terminal is set to add CR for a LF)              
        when 16     => case HdmiI2cRamIndex is
                         when 16 | 32 | 
                              48 | 64 | 
                              80 | 96 | 
                              112|128 
                                     => LOrd  := 10;
                         when others => LOrd  := 32;
                       end case;  
                       SerialOutDataOutByte <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable  <= '1';
                       I2cState             <= 17;
        -- Wait for character accepted               
        when 17     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 18;
                       end if;               
        -- Wait for serial character send               
        when 18     => if SerialOutBusy = '0' then
                         -- Next data
                         if HdmiI2cRamIndex = 128 then
                           -- Continue with reading RTC
                           I2cState <= 30;
                         else
                           -- Continue with next HDMI DDE data  
                           I2cState <= 25;
                         end if;
                       end if;
               
        -- Load I2C operations string and start it
        -- Note that there is no STOP at this point
        -- After this 'initialization' 128 read operations will take place, which will be displayed,
        -- and then finally a STOP is used
        when 20     => I2cState          <= 21;
                       HdmiI2cRamIndex   <= 0;
                       HdmiI2cOperations <= "SHLHLLLLLaLLLLLLLLaSHLHLLLLHa???????????"; -- Read  $A0 primary DDC (first  128 bytes) (no STOP!)
        -- Start I2C
        when 21     => HdmiI2cStart <= '1';
                       I2cState     <= 22;
        -- Wait for I2C ready feedback
        when 22     => if HdmiI2cReady = '0' then
                         -- Started, release start
                         HdmiI2cStart <= '0';
                         I2cState     <= 23;
                       end if;               
        -- Wait for I2c transfer complete
        when 23      => if HdmiI2cReady = '1' then
                         I2cState <= 25;
                       end if;
              
        -- Read next/last byte data               
        when 25     => I2cState <= 26;
                       if HdmiI2cRamIndex = 127 then
                         HdmiI2cOperations <= "bbbbbbbbHP??????????????????????????????"; -- Read last byte (with STOP)
                       else
                         HdmiI2cOperations <= "bbbbbbbbL???????????????????????????????"; -- Read next byte (no STOP!)
                       end if;  
                       HdmiI2cRamIndex <= HdmiI2cRamIndex + 1;
        -- Start I2C
        when 26     => HdmiI2cStart <= '1';
                       I2cState     <= 27;
        -- Wait for I2C ready feedback
        when 27     => if HdmiI2cReady = '0' then
                         -- Started, release start
                         HdmiI2cStart <= '0';
                         I2cState     <= 28;
                       end if;               
        -- Wait for I2c transfer complete
        when 28      => if HdmiI2cReady = '1' then
                         I2cState <= 29;
                       end if;
        -- Display data just read
        when 29      => -- Convert data just read by I2C into hexadecimal data to display
                        AsciiMessageIndex1 <= to_integer(unsigned(HdmiI2cData(7 downto 4))) + 1;
                        AsciiMessageIndex2 <= to_integer(unsigned(HdmiI2cData(3 downto 0))) + 1;
                        -- Display data
                        I2cState <= 10;
        --                
        -- Real time clock I2C access (@ $51/$A2)
        -- We just read one address, the 'second' register
        when 30     => I2cState          <= 31;
                       MegaI2cOperations <= "SHLHLLLHLaLLLLLLLHaSHLHLLLHHabbbbbbbbHP?"; -- Read  $A2 RTC register 1 (seconds)
                       --                    \ write  /\ reg 1/ \ read reg 1     /\ end        
        -- Start I2C
        when 31     => MegaI2cStart <= '1';
                       I2cState     <= 32;
        -- Wait for I2C ready feedback
        when 32     => if MegaI2cReady = '0' then
                         -- Started, release start
                         MegaI2cStart <= '0';
                         I2cState     <= 33;
                       end if;               
        -- Wait for I2c transfer complete
        when 33     => if MegaI2cReady = '1' then
                         I2cState <= 34;
                       end if;
        -- Display contents (seconds)
        when 34      => -- Convert seconds to nibbles
                        AsciiMessageIndex1 <= to_integer(unsigned(MegaI2cData(7 downto 4))) + 1;
                        AsciiMessageIndex2 <= to_integer(unsigned(MegaI2cData(3 downto 0))) + 1;
                        -- Display data
                        I2cState  <= 35;
        -- Output as hex with space               
        -- Output high nibble               
        when 35     => LChar := ASCIIHEX(AsciiMessageIndex1);                              
                       LOrd  := character'pos(LChar);
                       SerialOutDataOutByte <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable  <= '1';
                       I2cState             <= 36;
        -- Wait for character accepted               
        when 36     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 37;
                       end if;               
        -- Wait for serial character send               
        when 37     => if SerialOutBusy = '0' then
                         I2cState <= 40;
                       end if;                      
        -- Output low nibble               
        when 40     => LChar := ASCIIHEX(AsciiMessageIndex2);                              
                       LOrd  := character'pos(LChar);
                       SerialOutDataOutByte <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable  <= '1';
                       I2cState             <= 41;
        -- Wait for character accepted               
        when 41     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 42;
                       end if;               
        -- Wait for serial character send               
        when 42     => if SerialOutBusy = '0' then
                         I2cState <= 43;
                       end if;                                      
        -- Output 's'
        when 43     => LOrd := 115;
                       SerialOutDataOutByte <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable  <= '1';
                       I2cState             <= 44;
        -- Wait for character accepted               
        when 44     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 45;
                       end if;               
        -- Wait for serial character send               
        when 45     => if SerialOutBusy = '0' then
                         I2cState <= 46;
                       end if;
        -- Output LF
        when 46     => LOrd := 10;
                       SerialOutDataOutByte <= std_logic_vector(to_unsigned(LOrd, 8));
                       SerialOutDataEnable  <= '1';
                       I2cState             <= 47;
        -- Wait for character accepted               
        when 47     => if SerialOutBusy = '1' then
                         SerialOutDataEnable <= '0';
                         I2cState            <= 48;
                       end if;               
        -- Wait for serial character send               
        when 48     => if SerialOutBusy = '0' then
                         I2cState <= 1;
                       end if;
        --
        when others => I2cState <= 0;
      end case;  
    end if;
  end process;
  
  -- Debug (for checking with an oscilloscope for example)
  -- PMOD P2
  --  1: p2lo(0)   7: p2hi(0)
  --  2: p2lo(1)   8: p2hi(1)
  --  3: p2lo(2)   9: p2hi(2)
  --  4: p2lo(3)  10: p2hi(3)
  --  5: GND      11: GND
  --  6: VCC (*)  12: VCC (*)
  -- (*) Only available if switched on
  p2lo(0) <= HdmiI2cDebug(0);
  p2lo(1) <= HdmiI2cDebug(1);
  p2lo(2) <= HdmiI2cDebug(2);
  p2lo(3) <= HdmiI2cDebug(3);
  p2hi(0) <= MegaI2cDebug(0);
  p2hi(1) <= MegaI2cDebug(1);
  p2hi(2) <= MegaI2cDebug(2);
  p2hi(3) <= MegaI2cDebug(3);

    
  -- UART input and output
  SerialInBitStreamIn <= rsrx;
  uart_txd            <= SerialOutBitStreamOut;
  -- Green LED D10 via 240E to 3.3V (LED_G)
  led_g <= SerialOutBusy;
  -- Red   LED D12 via 240E to 3.3V (LED_R)
  led_r <= SerialInDataOutToggle;


  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to derive our LED data from
  --{           Note that at startup the registers are at an undetermined state,
  --{           until the 'reset_button' is activated 
  --{ In      : <Clock200>      Clock
  --{           <reset_button>  Synchronous reset
  --{           <Led_Toggle>    Current toggle state
  --{           <Led_Count>     Current counter value
  --{ Out     : <Led_Toggle>    Next toggle state
  --{           <Led_Count>     Next counter value
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then
      if reset_button = '1' then
        -- When reset, set presets
        Led_Toggle <= '0';
        Led_Count  <=  0;
      else
        -- When we counted up to 2000000000 then it is time to toggle the LEDs
        if Led_Count = 200000000-1 then
          Led_Toggle <= not Led_Toggle;
          Led_Count  <= 0;
        else
          Led_Count  <= Led_Count + 1;
        end if;
      end if;
    end if;  
  end process;

  -- Red   LED D9  via 240E to GND (ULED)  
  led <= Led_Toggle;
  
end Behavioral;
