--{******************************************************************************}
--{ FileName............: i2c.vhdl 
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2026 - 2026
--{------------------------------------------------------------------------------}
--{
--{ I2C implementation that processes an operation 'string' with I2C instructions, 
--{ like "SLHHLLLLLabbbbbbbbaP????????????"
--{
--{ I2C operations:
--{ . Uppercase indicates master -> slave  (SDA write)
--{ . Lowercase indicates slave  -> master (SDA read )
--{   S       Start
--{   P       Stop
--{   A/a     Acknowledge  (master/slave)
--{   H/L/b   Bit transfer
--{   ?       End of operations/placeholder
--{                
--{ Version       Comment
--{ 2026.01.03.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity i2c is
  generic (
    CLKRATE    : natural := 200000000;                    -- Main clock frequency
    I2CRATE    : natural := 100000;                       -- I2C clock speed 100 kHz
    STRINGSIZE : natural := 32;                           -- Size of definition string
    OUTPUTBITS : natural := 8                             -- Number of (max) output bits
  );  
  port (
    clk        : in    std_logic;                                 -- Main clock
    scl        : inout std_logic;                                 -- SCL
    sda        : inout std_logic;                                 -- SDA
    operations : string(1 to STRINGSIZE);                         -- Operations to execute
    start      : in    std_logic;                                 -- Asserted for starting new operations
                                                                  -- ready = '0' if validated -> de-assert when ready becomes '1'
    --
    data       : out   std_logic_vector(OUTPUTBITS-1 downto 0);   -- Data received from slave
    error      : out   std_logic;                                 -- Slave acknowledge status (ORed, should result in '0')
    ready      : out   std_logic;                                 -- Asserted when done
    --
    debug      : inout std_logic_vector(7 downto 0)               -- Debug
  );
end i2c;


architecture Behavioral of i2c is
  signal   I2c_OperationIndex : integer range 1 to STRINGSIZE := STRINGSIZE;   -- Index into operations string sequence (note that strings start at index 1)
  --
  signal   I2c_State        : integer range 0 to 3 := 0;                     -- Our I2C states (for each operation)
  --
  signal   I2c_Operation    : character := '?';                              -- I2C operation to perform:  I S L H E a A b ?
                                                                             --   Lower case: Input  (Read)
                                                                             --   Upper case: Output (Write)
                                                                             --   A/a: Acknowledge
                                                                             --   S  : Start
                                                                             --   P  : Stop
                                                                             --   b  : Bit data
                                                                             --   I  : Idle
                                                                             --   H  : '1'
                                                                             --   L  : '0'
                                                                             --   ?  : End                           (no change of SCL/SDA)
                                                                             --   X  : Internal use (initialization) (no change of SCL/SDA)

  signal   I2c_Scl          : std_logic;                                     -- SCL clock (HIGH means -not- driving the output)
  signal   I2c_Sda          : std_logic;                                     -- SDA data  (HIGH means -not- driving the output)
  signal   I2c_Data         : std_logic_vector(OUTPUTBITS-1 downto 0);       -- 'data'
  signal   I2c_Error        : std_logic;                                     -- 'error'
  signal   I2c_Ready        : std_logic := '1';                              -- 'ready'
  --
  signal   I2c_DebugRead    : std_logic;                                     -- Asseeted when reading data from slave

  -- The I2C clock is derived from the main clock
  constant I2C_COUNTER_LOAD : natural := CLKRATE / I2CRATE / 4;              -- Reload counter for our derived I2C cycle
  signal   I2c_Counter      : integer range 0 to I2C_COUNTER_LOAD;           -- Counter (clock division) 
  signal   I2c_Clk          : std_logic;                                     -- Clock enable for deriving I2C clock

begin


  --{------------------------------------------------------------------------------}
  --{ Descript: Generate I2C pacing clock (@clk)
  --{ Params :  <clk>          Main clock
  --{           <I2c_Counter>  Current counter
  --{ Returns   <I2c_Clk>      I2C pacing clock
  --{                          Asserted one cycle at reload/restart
  --{           <I2c_Counter>  Next counter
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if (I2c_Counter = 0) then
        I2c_Counter <= I2C_COUNTER_LOAD;
        I2c_Clk     <= '1';
      else
        I2c_Counter <= I2c_Counter - 1;
        I2c_Clk     <= '0';
      end if;
    end if;
  end process;  


  --{------------------------------------------------------------------------------}
  --{ Descript: Generate I2c_State (@I2c_Clk)
  --{ Params :  <I2c_Clk>    I2C pace clock
  --{           <I2c_State>  Current state
  --{ Returns:  <I2c_State>  Next state
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if (I2c_Clk = '1') then
        case I2c_State is
          -- 0 -> 1
          when 0      => I2c_State <= 1;
          -- 1 -> 2
          when 1      => I2c_State <= 2;
          -- 2 -> 3
          when 2      => I2c_State <= 3;
          -- 3 -> 0               
          when others => I2c_State <= 0;
        end case;
      end if;
    end if;
  end process;

  
  --{------------------------------------------------------------------------------}
  --{ Descript: Get operation data
  --{ Params  : <I2c_Clk>             I2C pacing clock
  --{           <start>               Start (start next operations)
  --{           <I2c_State>           State
  --{           <I2c_OperationIndex>  Index in operations string
  --{ Returns:  <I2c_Operation>       Operation for next cycle
  --{           <I2c_OperationIndex>  Index in operations string
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if I2c_Clk = '1' then
        -- A 'start' resets things
        if start = '1' then
          I2c_OperationIndex <=  1;
          -- Initialization
          I2c_Operation      <= 'X';
        else
          case I2c_State is
            -- 2 -> 3 (ending cycle)
            -- Get next operation data            
            when 2      => I2c_Operation    <= operations(I2c_OperationIndex);
                           -- Keep in range
                           if I2c_OperationIndex = STRINGSIZE then
                             -- Force 'end'
                             I2c_Operation      <= '?';
                           else
                             I2c_OperationIndex <= I2c_OperationIndex + 1;
                           end if;
            when others => null;          
          end case;
        end if;
      end if;  
    end if;  
  end process;


  --{------------------------------------------------------------------------------}
  --{ Descript: <I2c_Scl> (@I2c_Clk)
  --{           . SDA/SCL
  --{             Main operations:
  --{           . I        : Idle            SDA/SCL both HIGH
  --{           . A/a/H/L/b: Bit transfer    SDA data must remain stable during SCL HIGH period
  --{           . S        : START condition SDA HIGH-to-LOW while SCL is HIGH
  --{           . P        : STOP condition  SDA LOW-to-HIGH while SCL is HIGH
  --{           . SDA is only allowed to change while SCL is LOW
  --{
  --{           State  3 0 1 2         3 0 1 2         3 0 1 2         3 0 1 2         3 0 1 2
  --{           change 0 1 2 3         0 1 2 3         0 1 2 3         0 1 2 3         0 1 2 3
  --{                  IDLE            START (*)       STOP (*)        DATA OUT        DATA IN
  --{                  ________        ______             _____           ___             ___  
  --{           SCL                          |           |               |   |           |   |
  --{                  . . . .               |_        __|             __|   |_        __|   |_
  --{                  ________        ____                 ___         _______        ________
  --{           SDA                        |               |           | DATA              R     R = read data
  --{                  . . . .             |___        ____|           |_______            R     Note: 'H' == 'Z"
  --{           (*) Assume IDLE situations before/after
  --{
  --{ Params  : <I2c_Clk>        I2C pacing clock
  --{           <I2c_Operation>  Operation to perform
  --{ Returns : <I2c_Scl>        SCL data
  --{           <I2c_Sda>        SDA data
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if I2c_Clk = '1' then
        -- Note that an 'end of operation' should not change the current outputs (we could be in the middle of
        -- a sequence of operations)
        -- SCL
        case I2c_Operation is
          -- Init/end do not change SCL/SDA
          when 'X'|'?'
                      => I2c_Scl <= I2c_Scl;
          -- Start
          when 'S'    => case I2c_State is
                           when 3 => I2c_Scl <= '1';
                           when 0 => I2c_Scl <= '1';
                           when 1 => I2c_Scl <= '1';
                           when 2 => I2c_Scl <= '0';
                         end case;
          -- Stop
          when 'P'    => case I2c_State is
                           when 3 => I2c_Scl <= '0';
                           when 0 => I2c_Scl <= '1';
                           when 1 => I2c_Scl <= '1';
                           when 2 => I2c_Scl <= '1';
                         end case;
          -- Bit transfers
          when 'A'|'a'|'b'|'L'|'H'              
                      => case I2c_State is
                           when 3 => I2c_Scl <= '0';
                           when 0 => I2c_Scl <= '1';
                           when 1 => I2c_Scl <= '1';
                           when 2 => I2c_Scl <= '0';
                         end case;
          when others => I2c_Scl <= '1';
        end case;
        --
        -- SDA
        case I2c_Operation is
          -- Init/end do not change SCL/SDA
          when 'X'|'?'
                      => I2c_Sda <= I2c_Sda;
          -- Start
          when 'S'    => case I2c_State is
                           when 3 => I2c_Sda <= '1';
                           when 0 => I2c_Sda <= '1';
                           when 1 => I2c_Sda <= '0';
                           when 2 => I2c_Sda <= '0';   
                         end case;
          -- Stop               
          when 'P'    => case I2c_State is
                           when 3 => I2c_Sda <= '0';
                           when 0 => I2c_Sda <= '0';
                           when 1 => I2c_Sda <= '1';
                           when 2 => I2c_Sda <= '1';   
                         end case;
          -- Bit transfers (whole cycle same data)               
          when 'L'    => I2c_Sda <= '0';
          when others => I2c_Sda <= '1';  -- Equals to not driving the output
        end case;        
      end if;
    end if;
  end process;

  
  --{------------------------------------------------------------------------------}
  --{ Descript: Slave data in (@I2c_Clk)
  --{             3 0 1 2
  --{             0 1 2 3
  --{             DATA IN
  --{             ________
  --{                R     R = read data
  --{                R     Note: 'H' == 'Z"
  --{ Params  : <I2c_Clk>        I2C pacing clock
  --{           <sda>            I2C data input
  --{           <start>          Start new operations string
  --{           <I2c_Operation>  Operation to perform
  --{           <I2c_State>      I2c state
  --{ Returns : <I2c_Error>      I2C error
  --{           <I2c_Ready>      I2C ready  
  --{           <I2c_Data>       I2C accumulated data
  --{           <I2c_DebugRead>  Indicate read of slave data  
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if I2c_Clk = '1' then
        I2c_DebugRead <= '0';
        -- Start resets ready/result
        if start = '1' then
          I2c_Error <= '0';
          I2c_Ready <= '0';
          -- Not really necessary, but is cleaner
          I2c_Data(OUTPUTBITS-1 downto 0) <= (others => '0');          
        end if;
        --
        -- Data in/out
        case I2c_State is
          -- 3 -> 0          
          when 3      => -- If end of operation data detected, indicate ready
                         if I2c_Operation = '?' then
                           I2c_Ready <= '1';
                         end if;  
          -- 1 -> 2
          when 1      => -- Shift out next bit data to write (this cycle)                  
                         case I2c_Operation is                    
                           -- Shift in bit data just read
                           -- Slave acknowledge must be zero, otherwise error
                           when 'a'    => I2c_Error        <= I2c_Error or sda;
                                          I2c_DebugRead    <= '1';
                           -- Shift in data and put in result
                           when 'b'    => I2c_Data(0)                     <= sda;
                                          I2c_Data(OUTPUTBITS-1 downto 1) <= I2c_Data(OUTPUTBITS-2 downto 0); 
                                          I2c_DebugRead                   <= '1';
                           when others => null;
                         end case;
          when others => null;                    
        end case;
      end if;
    end if;
  end process;


  -- For I2C we only drive the output LOW. We do not drive a HIGH output
  -- to comply with the open collector nature of I2C.
  scl                         <= 'Z' when I2c_Scl = '1' else '0';
  sda                         <= 'Z' when I2c_Sda = '1' else '0';
  data(OUTPUTBITS-1 downto 0) <= I2c_Data(OUTPUTBITS-1 downto 0);
  error                       <= I2c_Error;
  ready                       <= I2c_Ready;

  -- Debug  
  debug(0) <= I2c_Scl;
  debug(1) <= sda when (I2c_Operation = 'a' or I2c_Operation = 'b') else 
              I2c_Sda;
  debug(2) <= I2c_Ready;
  debug(3) <= I2c_Error;
  debug(4) <= I2c_DebugRead;
  -- Emulate '0' acknowledge/data input
  debug(5) <= '0' when (I2c_Sda = '0' or I2c_Operation = 'a' or I2c_Operation = 'b') else 
              '1';
  debug(6) <= '1' when (I2c_State = 0) else
              '0';
  debug(7) <= '1' when (I2c_State = 3) else
              '0';   

end Behavioral;
