--{******************************************************************************}
--{ FileName............: serialout.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2024 - 2024
--{------------------------------------------------------------------------------}
--{
--{ Serial output
--{ Based on RS232 'protocol', using 1 startbit, 8 databits, 1 stopbit
--{   ___     ___ ___ ___ ___ ___ ___ ___ ___ ___ ___
--{      | s | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | S |      bitstream
--{      |___|---|---|---|---|---|---|---|---|___|
--{ 
--{       ---------------------------------------
--{   ___|                                       |___   busy
--{
--{  s   : start bit ('0')
--{  0..7: data bits
--{  S   : stop bit  ('1')
--{
--{ Version       Comment
--{ 2024.12.24.0  - First release
--{------------------------------------------------------------------------------}

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity serialout is
  generic (
    CLKRATE   : natural := 200000000;                       -- Main clock frequency
    BAUDRATE  : natural := 115200                           -- Baudrate
  );
  port (
    clk       : in  std_logic;                              -- Main clock
    datain    : in  std_logic_vector(7 downto 0);           -- Data to output
    dataenable: in  std_logic;                              -- Enable signal for starting bit stream output using <datain>
                                                            -- Only accepted when not busy
    bitstream : out std_logic;                              -- Data stream output
    busy      : out std_logic                               -- Flag indicating data is being output
  );
end serialout;


architecture Behavioral of serialout is
  -- The baud clock is derived from the main clock
  constant COUNTER_LOAD    : natural := (CLKRATE / BAUDRATE) - 2;
  signal   BaudCounter     : integer range 0 to COUNTER_LOAD;   -- Counter (clock division) 
  signal   BaudClk         : std_logic;                         -- Clock enable for deriving baudrate  
  signal   BaudCounterReset: std_logic;                         -- Counter reset/reload
  
  signal   DataToOutput    : std_logic_vector(10 downto 0);     -- Data to output (includes start/stopbit)
  signal   DataReady       : std_logic_vector(10 downto 0);     -- Used for check for end of data
                                                                -- One bit more than the data since we must wait for the last bit to complete firect
                                                                -- Used instead of a counter
                                                                -- . Reset to '11111111111' at start
                                                                -- . Shifts right for each bit being output
                                                                -- . If bit 0 becomes 0 then we are done
  
begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Generate the baud clock
  --{ In      : <clk>               Clock
  --{           <BaudCounterReset>  Reset/reload clock counter
  --{           <BaudCounter>       Current counter
  --{ Out     : <BaudCounter>       Next counter
  --{           <BaudClk>           Asserted when <BaudCounter> is releaded/restarted
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if (BaudCounterReset = '1') or (BaudCounter = 0) then
        BaudCounter <= COUNTER_LOAD;
        BaudClk     <= '1';
      else
        BaudCounter <= BaudCounter - 1;
        BaudClk     <= '0';
      end if;
    end if;
  end process;   


  --{------------------------------------------------------------------------------}
  --{ Descript: Latch and output to bit stream
  --{ In      : <clk>               Clock
  --{           <dataenable>        Load signal for new data to output as a bit stream
  --{           <DataToOutput>      Current data being output
  --{           <DataReady>         Current indicator if all bits have been output
  --{ Out     : <DataToOutput>      Next data being output
  --{           <DataReady>         Next indicator if all bits have been output
  --{           <BaudCounterReset>  Restart/reload of baudrate clock
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      -- Defaults/Resets
      BaudCounterReset <= '0';                              -- No reload the baudrate clock
      -- Only accept new input when we are not outputting the bit stream
      if DataReady(0) = '0' then
        -- Not outputting a bit stream
        if dataenable = '1' then                            -- If latching indicated
          DataToOutput     <= '1' & datain & '0' & '1';     -- Set data to output (stop/data/start/current)
          DataReady        <= "11111111111";                -- Reset bit 'counter'
          BaudCounterReset <= '1';                          -- Reload the clock
        end if;  
      else  
        -- We are outputting the bit stream, synchronize to the baudrate clock
        -- The first <BaudClk> after <BaudCounterReset> is delayed by 2 <clk> cycles:
        -- . first  cycle will assert <BaudClk>
        -- . second cycle we detect this assertion here
        if BaudClk = '1' then                               -- Clocking out is done on the baud clock
          DataToOutput <= '1' & DataToOutput(10 downto 1);  -- Shift right for next output (shift '1' for stop level)
          DataReady    <= '0' & DataReady   (10 downto 1);  -- Shift right for end of data detection
        end if;
      end if;  
    end if;
  end process;   

  -- Derive signals
  busy      <= DataReady(0);
  bitstream <= DataToOutput(0);
  
end Behavioral;
