--{******************************************************************************}
--{ FileName............: serialin.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2024 - 2024
--{------------------------------------------------------------------------------}
--{
--{ Serial input
--{ Based on RS232 'protocol', using 1 startbit, 8 databits, 1 stopbit
--{   ___     ___ ___ ___ ___ ___ ___ ___ ___ ___ ___
--{      | s | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | S |      bitstream
--{      |___|---|---|---|---|---|---|---|---|___|
--{ 
--{  s   : start bit ('0')
--{  0..7: data bits
--{  S   : stop bit  ('1')
--{
--{ A typical approach to receiving data is using a 16* higher clock than
--{ the baudrate and sample at each 'half' bit for the data
--{ This means a state machine is used to keep track of what is being handled.
--{ In our implementation we don't use a state machine. Instead we also use
--{ a higher clock (albeit no 16*). With this higher clock we take samples and
--{ place them in a serial buffer. We then look at the whole serial buffer
--{ and decide if we have valid data or not.
--{ The baudrate setting is the bitrate. If we were to sample at the same
--{ rate as the bitrate then there is a change we will be sampling at the
--{ point that the data changes, which results in incorrect data.
--{ If we were to sample 3 times the bitrate, then only one of these 3 samples
--{ is possibly incorrect. By checking for 2 identical samples we know what
--{ the actual bit was.
--{
--{ Version       Comment
--{ 2024.12.24.0  - First release
--{------------------------------------------------------------------------------}

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity serialin is
  generic (
    CLKRATE   : natural := 200000000;                       -- Main clock frequency
    BAUDRATE  : natural := 115200                           -- Baudrate
  );
  port (
    clk          : in  std_logic;                           -- Main clock
    bitstream    : in  std_logic;                           -- Data stream input
    dataout      : out std_logic_vector(7 downto 0);        -- Data detected
    dataouttoggle: out std_logic                            -- Toggles for every new <dataout>
  );
end serialin;


architecture Behavioral of serialin is
  -- Do not change OVERSAMPLING here. Some code is hardcoded for 3 times oversampling.
  -- This could all be made variable but that is for later ...
  constant OVERSAMPLING    : natural := 3;
  constant COUNTER_LOAD    : natural := (CLKRATE / (BAUDRATE * OVERSAMPLING)) - 2;
  signal   BaudCounter     : integer range 0 to COUNTER_LOAD;   -- Counter (clock division) 
  signal   BaudClk         : std_logic;                         -- Clock enable for deriving baudrate  
  
  signal   DataIn          : std_logic_vector((OVERSAMPLING * 10)-1 downto 0);
  signal   StartOk         : std_logic;                         -- Indicates valid start bit
  signal   StopOk          : std_logic;                         -- Indicates valid stop  bit
  signal   DataOk          : std_logic_vector(7 downto 0);      -- Indicates valid data  bit
  signal   Data            : std_logic_vector(7 downto 0);      -- Data bits
  signal   DataToggle      : std_logic := '0';                  -- Data toggle
  signal   AllOk           : std_logic;                         -- Indicates all valid bits
  
begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Sample the input data at the baudrate (oversampling)
  --{ In      : <clk>               Clock
  --{           <BaudCounter>       Current counter
  --{           <bitstream>         Input data
  --{           <AllOk>             True when all data is valid
  --{           <DataToggle>        Current toggle status
  --{ Out     : <BaudCounter>       Next counter
  --{           <DataIn>            Serialized data
  --{           <dataout>           Latched from <DataIn> when all data is valid
  --{           <DataToggle>        Toggled for every new data      
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if AllOk = '1' then
        -- If all received bits are valid
        dataout    <= Data;
        DataToggle <= not DataToggle;
        -- Invalidate all samples
        DataIn  <= "010101010101010101010101010101";
      else
        if (BaudCounter = 0) then
          BaudCounter <= COUNTER_LOAD;
          DataIn <= DataIn((OVERSAMPLING * 10)-2 downto 0) & bitstream;  -- Shift left new data in        
        else
          BaudCounter <= BaudCounter - 1;
        end if;      
      end if;
    end if;  
  end process;   

  -- We take the center sample for the actual detected data
  -- The two left most -or- the two right bits must be equal to be valid data
  -- Check for correct stop bits ('1')
  StopOk    <= '1' when DataIn(1)  = '1' and DataIn(0)  = '1' else
               '1' when DataIn(1)  = '1' and DataIn(2)  = '1' else
               '0';
  -- Data bits
  -- Get the data and check for if valid
  Data  (7) <= DataIn(4);
  DataOk(7) <= '1' when DataIn(4)  = '1' and DataIn(3)  = '1' else
               '1' when DataIn(4)  = '0' and DataIn(3)  = '0' else
               '1' when DataIn(4)  = '1' and DataIn(5)  = '1' else
               '1' when DataIn(4)  = '0' and DataIn(5)  = '0' else
               '0';    
  Data  (6) <= DataIn(7);    
  DataOk(6) <= '1' when DataIn(7)  = '1' and DataIn(6)  = '1' else
               '1' when DataIn(7)  = '0' and DataIn(6)  = '0' else
               '1' when DataIn(7)  = '1' and DataIn(8)  = '1' else
               '1' when DataIn(7)  = '0' and DataIn(8)  = '0' else
               '0';    
  Data  (5) <= DataIn(10);    
  DataOk(5) <= '1' when DataIn(10) = '1' and DataIn(9)  = '1' else
               '1' when DataIn(10) = '0' and DataIn(9)  = '0' else
               '1' when DataIn(10) = '1' and DataIn(11) = '1' else
               '1' when DataIn(10) = '0' and DataIn(11) = '0' else
               '0';    
  Data  (4) <= DataIn(13);    
  DataOk(4) <= '1' when DataIn(13) = '1' and DataIn(12) = '1' else
               '1' when DataIn(13) = '0' and DataIn(12) = '0' else
               '1' when DataIn(13) = '1' and DataIn(14) = '1' else
               '1' when DataIn(13) = '0' and DataIn(14) = '0' else
               '0';    
  Data  (3) <= DataIn(16);    
  DataOk(3) <= '1' when DataIn(16) = '1' and DataIn(15) = '1' else
               '1' when DataIn(16) = '0' and DataIn(15) = '0' else
               '1' when DataIn(16) = '1' and DataIn(17) = '1' else
               '1' when DataIn(16) = '0' and DataIn(17) = '0' else
               '0';    
  Data  (2) <= DataIn(19);    
  DataOk(2) <= '1' when DataIn(19) = '1' and DataIn(18) = '1' else
               '1' when DataIn(19) = '0' and DataIn(18) = '0' else
               '1' when DataIn(19) = '1' and DataIn(20) = '1' else
               '1' when DataIn(19) = '0' and DataIn(20) = '0' else
               '0';    
  Data  (1) <= DataIn(22);    
  DataOk(1) <= '1' when DataIn(22) = '1' and DataIn(21) = '1' else
               '1' when DataIn(22) = '0' and DataIn(21) = '0' else
               '1' when DataIn(22) = '1' and DataIn(23) = '1' else
               '1' when DataIn(22) = '0' and DataIn(23) = '0' else
               '0';    
  Data  (0) <= DataIn(25);    
  DataOk(0) <= '1' when DataIn(25) = '1' and DataIn(24) = '1' else
               '1' when DataIn(25) = '0' and DataIn(24) = '0' else
               '1' when DataIn(25) = '1' and DataIn(26) = '1' else
               '1' when DataIn(25) = '0' and DataIn(26) = '0' else
               '0';    
  -- Check for correct start bits ('0')
  StartOk   <= '1' when DataIn(28)  = '0' and DataIn(27)  = '0' else
               '1' when DataIn(28)  = '0' and DataIn(29)  = '0' else
               '0';
  -- Combine all OK signals
  AllOk     <= '1' when StartOk   = '1' and 
                        DataOk(0) = '1' and
                        DataOk(1) = '1' and
                        DataOk(2) = '1' and
                        DataOk(3) = '1' and
                        DataOk(4) = '1' and
                        DataOk(5) = '1' and
                        DataOk(6) = '1' and
                        DataOk(7) = '1' and
                        StopOk    = '1' else
               '0';    
                
  dataouttoggle <= DataToggle;
  
end Behavioral;
