--{******************************************************************************}
--{ FileName............: main_keyboard.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Implements:
--{ . Main clock
--{ . Blinky LED
--{ . Keyboard input that is put on the serial output as readable ASCII text
--{ . Debug data on P2 PMOD port
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use work.mega_package.all;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity main_keyboard is
  port (
    clk_in      : in  std_logic;
    -- Buttons
    reset_button: in  std_logic;
    -- LED indications
    led         : out std_logic;
    led_g       : out std_logic;
    led_r       : out std_logic;
    -- UART
    uart_txd    : out std_logic;
    rsrx        : in  std_logic;
    -- To/From keyboard controller
    kb_io0      : out std_logic;
    kb_io1      : out std_logic;
    kb_io2      : in  std_logic;
    -- Debugging
    p2lo        : out std_logic_vector(3 downto 0);     
    p2hi        : out std_logic_vector(3 downto 0)    
  );
end main_keyboard;

architecture Behavioral of main_keyboard is
  -- Our clock signal we derive our LED signals from
  signal Clock200                  : std_logic;
  -- Serial input
  signal SerialInBitStreamIn       : std_logic;
  signal SerialInDataOutByte       : std_logic_vector(7 downto 0);
  signal SerialInDataOutToggle     : std_logic;
  signal SerialInDataOutTogglePrev : std_logic := '0';
  -- Serial output
  signal SerialOutDataOutByte      : std_logic_vector(7 downto 0);
  signal SerialOutDataEnable       : std_logic;
  signal SerialOutBitStreamOut     : std_logic;
  signal SerialOutBusy             : std_logic;  
  -- Counters
  signal Led_Count                 : natural range 0 to 200000000;  
  -- LED toggle status
  signal Led_Toggle                : std_logic;  
  -- Keyboard
  signal KeyboardDataFrom          : std_logic_vector(127 downto 0);      -- Data read from   keyboard controller
  signal KeyboardDataFromLatched   : std_logic_vector(127 downto 0);      -- Data read from   keyboard controller (latch at toggle)
  signal KeyboardDataTo            : std_logic_vector(127 downto 0);      -- Data to write to keyboard controller
  signal KeyboardDataFromToggle    : std_logic;                           -- Toggles for every completed keyboard read/write cycle
  signal KeyboardDataFromTogglePrev: std_logic := '0';                    -- Previous toggle state
  signal KeyboardDebug             : std_logic_vector(7 downto 0);        -- Debug data
  -- Keyboard to ASCII
  signal AsciiBitNumber            : std_logic_vector(6 downto 0);        -- Bit number to 'translate
  signal AsciiAsciiData            : TByteArray8;                         -- ASCII string
  signal AsciiDebug                : std_logic_vector(7 downto 0);
  -- Led to keyboard
  signal LedPowerLeftR             : std_logic_vector(  7 downto 0);
  signal LedPowerLeftG             : std_logic_vector(  7 downto 0);
  signal LedPowerLeftB             : std_logic_vector(  7 downto 0);
  signal LedPowerRightR            : std_logic_vector(  7 downto 0);
  signal LedPowerRightG            : std_logic_vector(  7 downto 0);
  signal LedPowerRightB            : std_logic_vector(  7 downto 0);
  signal LedDriveLeftR             : std_logic_vector(  7 downto 0);
  signal LedDriveLeftG             : std_logic_vector(  7 downto 0);
  signal LedDriveLeftB             : std_logic_vector(  7 downto 0);
  signal LedDriveRightR            : std_logic_vector(  7 downto 0);
  signal LedDriveRightG            : std_logic_vector(  7 downto 0);
  signal LedDriveRightB            : std_logic_vector(  7 downto 0);
  signal LedData                   : std_logic_vector(127 downto 0);
  signal LedDebug                  : std_logic_vector(  7 downto 0);


begin
  -- ** From 'mega65r6.vhdl':
  -- New clocking setup, using more optimised selection of multipliers
  -- and dividers, as well as the ability of some clock outputs to provide an
  -- inverted clock for free.
  -- Also, the 50 and 100MHz ethernet clocks are now independent of the other
  -- clocks, so that Vivado shouldn't try to meet timing closure in the (already
  -- protected) domain crossings used for those.
  clocks1: entity work.clocking
    port map ( 
      clk_in     => clk_in,
      clock27    => open,      -- clock27,    --   27     MHz
      clock41    => open,      -- cpuclock,   --   40.5   MHz
      clock50    => open,      -- ethclock,   --   50     MHz
      clock74p22 => open,      -- clock74p22,
      clock81p   => open,      -- pixelclock, --   81     MHz
      clock163   => open,      -- clock162,   --  162     MHz
      clock163m  => open,      -- clock162m,  --  162     MHz, phase shifted by -207 degrees for SDRAM read timing
      clock200   => Clock200,  -- clock200,   --  200     MHz
      clock270   => open,      -- clock270,   --  270     MHz
      clock325   => open       -- clock325    --  325     MHz
    );


  -- Instantiation of the serial input
  SerialIn_I: entity work.serialin
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk           => Clock200,
      bitstream     => SerialInBitStreamIn,
      dataout       => SerialInDataOutByte,
      dataouttoggle => SerialInDataOutToggle
    );

  -- Instantiation of the serial output
  SerialOut_I: entity work.serialout
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk        => Clock200,
      datain     => SerialOutDataOutByte,
      dataenable => SerialOutDataEnable,
      bitstream  => SerialOutBitStreamOut,
      busy       => SerialOutBusy
    );


  -- Instantiation of the keyboard
  Keyboard_I: entity work.keyboard
    generic map (
      CLKRATE     => 200000000,
      UPDATERATE  => 1000,
      ONLYCHANGES => True
    )
    port map (
      clk           => Clock200,
      kb_clkout     => kb_io0,
      kb_dataout    => kb_io1,
      kb_datain     => kb_io2,
      dataout       => KeyboardDataFrom,
      datain        => KeyboardDataTo,
      dataouttoggle => KeyboardDataFromToggle,
      debug         => KeyboardDebug
    );

  -- Instantiation for the keyboard to ASCII text conversion
  KeyboardToAscii_I: entity work.keyboard_to_ascii
    port map (
      clk           => Clock200,
      bitnumber     => AsciiBitNumber,
      asciidata     => AsciiAsciiData
    );

  -- Instantiation for the keyboard to LED conversion output
  KeyboardToAsciiLedOut_I: entity work.keyboard_to_ascii_ledout
    port map (
      clk                          => Clock200,
      -- Keyboard
      key_data                     => KeyboardDataFrom,
      key_datatoggle               => KeyboardDataFromToggle,
      -- For LED to keyboard conversion
      power_led_left_brightness_R  => LedPowerLeftR,
      power_led_left_brightness_G  => LedPowerLeftG,
      power_led_left_brightness_B  => LedPowerLeftB,
      power_led_right_brightness_R => LedPowerRightR,
      power_led_right_brightness_G => LedPowerRightG,
      power_led_right_brightness_B => LedPowerRightB,
      drive_led_left_brightness_R  => LedDriveLeftR,
      drive_led_left_brightness_G  => LedDriveLeftG,
      drive_led_left_brightness_B  => LedDriveLeftB,
      drive_led_right_brightness_R => LedDriveRightR,
      drive_led_right_brightness_G => LedDriveRightG,
      drive_led_right_brightness_B => LedDriveRightB,
      debug                        => LedDebug
    );  
      
      
  -- Instantiation for the LED to keyboard conversion
  LedToKeyboard_I: entity work.led_to_keyboard
    port map (
      clk                          => Clock200,
      power_led_left_brightness_R  => LedPowerLeftR,
      power_led_left_brightness_G  => LedPowerLeftG,
      power_led_left_brightness_B  => LedPowerLeftB,
      power_led_right_brightness_R => LedPowerRightR,
      power_led_right_brightness_G => LedPowerRightG,
      power_led_right_brightness_B => LedPowerRightB,
      drive_led_left_brightness_R  => LedDriveLeftR,
      drive_led_left_brightness_G  => LedDriveLeftG,
      drive_led_left_brightness_B  => LedDriveLeftB,
      drive_led_right_brightness_R => LedDriveRightR,
      drive_led_right_brightness_G => LedDriveRightG,
      drive_led_right_brightness_B => LedDriveRightB,
      led_data                     => LedData
    );

  KeyboardDataTo <= LedData;

  -- Instantiation for the keyboard to ASCII text conversion serial output
  KeyboardToAsciiSerialOut_I: entity work.keyboard_to_ascii_serialout
    port map (
      clk                    => Clock200,
      -- Keyboard
      key_data               => KeyboardDataFrom,
      key_datatoggle         => KeyboardDataFromToggle,
      -- ASCII conversion
      ascii_bitnumber        => AsciiBitNumber,
      ascii_data             => AsciiAsciiData,
      -- Serial out
      serial_data            => SerialOutDataOutByte,
      serial_enable          => SerialOutDataEnable, 
      serial_busy            => SerialOutBusy,
      -- Debug
      debug                  => AsciiDebug  
    );

  
  -- Debug (for checking with an oscilloscope for example)
  -- PMOD P2
  --  1: p2lo(0)   7: p2hi(0)
  --  2: p2lo(1)   8: p2hi(1)
  --  3: p2lo(2)   9: p2hi(2)
  --  4: p2lo(3)  10: p2hi(3)
  --  5: GND      11: GND
  --  6: VCC (*)  12: VCC (*)
  -- (*) Only available if switched on
  p2lo(0) <= LedDebug(0);
  p2lo(1) <= LedDebug(1);
  p2lo(2) <= LedDebug(2);
  p2lo(3) <= LedDebug(3);
  p2hi(0) <= LedDebug(4);
  p2hi(1) <= LedDebug(5);
  p2hi(2) <= LedDebug(6);
  p2hi(3) <= LedDebug(7);
  
  -- UART input and output
  SerialInBitStreamIn <= rsrx;
  uart_txd            <= SerialOutBitStreamOut;
  -- Green LED D10 via 240E to 3.3V (LED_G)
  led_g <= SerialOutBusy;
  -- Red   LED D12 via 240E to 3.3V (LED_R)
  led_r <= SerialInDataOutToggle;


  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to derive our LED data from
  --{           Note that at startup the registers are at an undetermined state,
  --{           until the 'reset_button' is activated 
  --{ In      : <Clock200>      Clock
  --{           <reset_button>  Synchronous reset
  --{           <Led_Toggle>    Current toggle state
  --{           <Led_Count>     Current counter value
  --{ Out     : <Led_Toggle>    Next toggle state
  --{           <Led_Count>     Next counter value
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then
      if reset_button = '1' then
        -- When reset, set presets
        Led_Toggle <= '0';
        Led_Count  <=  0;
      else
        -- When we counted up to 2000000000 then it is time to toggle the LEDs
        if Led_Count = 200000000-1 then
          Led_Toggle <= not Led_Toggle;
          Led_Count  <= 0;
        else
          Led_Count  <= Led_Count + 1;
        end if;
      end if;
    end if;  
  end process;

  -- Red   LED D9  via 240E to GND (ULED)  
  led <= Led_Toggle;
  
end Behavioral;
