--{******************************************************************************}
--{ FileName............: keyboard_to_ascii_serialout_simulation.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Simulation for sending a converted keyboard bit number to ASCII
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use work.mega_package.all;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity keyboard_to_ascii_serialout_simulation is
end keyboard_to_ascii_serialout_simulation;

architecture Behavioral of keyboard_to_ascii_serialout_simulation is
  signal ClkIn            : std_logic;
  signal KeyData          : std_logic_vector(127 downto 0);
  signal KeyDataToggle    : std_logic := '0';
  signal AsciiBitNumber   : std_logic_vector(  6 downto 0);
  signal AsciiData        : TByteArray8;
  signal SerialData       : std_logic_vector(  7 downto 0);
  signal SerialDataLatched: std_logic_vector(  7 downto 0);
  signal SerialEnable     : std_logic;
  signal SerialBusy       : std_logic;
  signal Debug            : std_logic_vector(  7 downto 0);
  
begin
  KeyboardToAsciiSerialOut: entity work.keyboard_to_ascii_serialout
    port map (
      clk             => ClkIn,
      key_data        => KeyData,
      key_datatoggle  => KeyDataToggle,
      ascii_bitnumber => AsciiBitNumber,
      ascii_data      => AsciiData,
      serial_data     => SerialData,
      serial_enable   => SerialEnable,
      serial_busy     => SerialBusy,
      debug           => Debug
  );

  -- Instantiation for the keyboard to ASCII text conversion
  KeyboardToAscii: entity work.keyboard_to_ascii
    port map (
      clk           => ClkIn,
      bitnumber     => AsciiBitNumber,
      asciidata     => AsciiData
    );


  --{------------------------------------------------------------------------------}
  --{ Descript: Generate the main clock
  --{------------------------------------------------------------------------------}
  process
  begin
    ClkIn <= '0';
    wait for 5 ns;
    ClkIn <= '1';
    wait for 5 ns;
  end process;

  --{------------------------------------------------------------------------------}
  --{ Descript: Emulate serial output data
  --{           We latch serial_data when serial_enable and keep serial_busy
  --{           active for some time 
  --{------------------------------------------------------------------------------}
  process
  begin
    wait for 1.25 ns;
    SerialBusy <= '0';
    wait for 1.25 ns;
    wait until (ClkIn = '1') and (SerialEnable = '1');
    SerialDataLatched <= SerialData;
    SerialBusy <= '1';
    -- Emulate transmitting the data
		wait for 50 ns;
  end process;


  --{------------------------------------------------------------------------------}
  --{ Descript: Emulate keyboard data
  --{           We generate KeyData and indicate the change with KeyDataToggle
  --{------------------------------------------------------------------------------}
  process
  begin
    for LLoop in KeyData'range loop
      KeyData(LLoop) <= '1';
    end loop;
    KeyDataToggle <= not KeyDataToggle;
    wait for 100 ns;
    -- Change one bit
    for LLoop in KeyData'range loop
      KeyData(LLoop) <= '1';
      if LLoop = 63 then
        KeyData(LLoop) <= '0';
      end if;
    end loop;
    KeyDataToggle <= not KeyDataToggle;
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4e" report "Error 63-0"; -- N
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4f" report "Error 63-1"; -- O
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"53" report "Error 63-2"; -- S
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"43" report "Error 63-3"; -- C
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"52" report "Error 63-4"; -- R
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4f" report "Error 63-5"; -- O
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4c" report "Error 63-6"; -- L
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4c" report "Error 63-7"; -- L          
    wait for 50 us;
    -- Change one bit
    for LLoop in KeyData'range loop
      KeyData(LLoop) <= '1';
      if LLoop = 126 then
        KeyData(LLoop) <= '0';
      end if;
    end loop;
    KeyDataToggle <= not KeyDataToggle;
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"52" report "Error 126-0"; -- R
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"45" report "Error 126-1"; -- E
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"54" report "Error 126-2"; -- T
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"55" report "Error 126-3"; -- U
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"52" report "Error 126-4"; -- R
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"4e" report "Error 126-5"; -- N
    wait for 100 ns;
    assert SerialDataLatched = x"4e" report "Error 126-5-2"; -- N  No change due to pipe symbol
    wait for 50 us;
    -- Change one bit
    for LLoop in KeyData'range loop
      KeyData(LLoop) <= '1';
      if LLoop = 84 then
        KeyData(LLoop) <= '0';
      end if;
    end loop;
    KeyDataToggle <= not KeyDataToggle;
    wait until SerialEnable = '1';
    wait until SerialEnable = '0';
    assert SerialDataLatched = x"2d" report "Error 84-0"; -- -
    wait for 100 ns;
    assert SerialDataLatched = x"2d" report "Error 84-0-2"; -- No change due to pipe symbol
    wait for 50 us;
  end process;


end Behavioral;
