--{******************************************************************************}
--{ FileName............: keyboard_to_ascii_serialout.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ 'Sending' a converted to ASCII string keyboard bitnumber to a byte-for-byte output stream
--{
--{ Version       Comment
--{ 2025.03.22.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use work.mega_package.all;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity keyboard_to_ascii_serialout is
  port (
    clk                   : in  std_logic;
    -- Keyboard
    key_data              : in  std_logic_vector(127 downto 0);  -- Keyboard data detected
    key_datatoggle        : in  std_logic;                       -- Toggled for every new <key_data>
    -- ASCII conversion
    ascii_bitnumber       : out std_logic_vector(6 downto 0);    -- Bit number to convert to ASCII
    ascii_data            : in  TByteArray8;                     -- Converted ASCII to output
    -- Serial out
    serial_data           : out std_logic_vector(7 downto 0);    -- Data to output
    serial_enable         : out std_logic;                       -- Enable for <serial_data>
    serial_busy           : in  std_logic;                       -- Busy indication (sending data in progress)
    --
    debug                 : out std_logic_vector(7 downto 0)     -- Debug
  );
end keyboard_to_ascii_serialout;

architecture Behavioral of keyboard_to_ascii_serialout is

  signal StateDelay          : std_logic;                           -- Delay for one clock cycle
  -- Keyboard
  signal KeyDataTogglePrev   : std_logic;                           -- Previous toggle state
  signal KeyData             : std_logic_vector(127 downto 0);      -- Data read from   keyboard controller (latch at toggle)
  -- Keyboard to ASCII
  signal AsciiBitNumber      : std_logic_vector(6 downto 0);        -- Bit number to 'translate
  signal AsciiCharIndex      : natural range 0 to 7;                -- One of the 8 bytes being output of the converted data
  signal AsciiDataOutputting : std_logic := '0';                    -- True when busy outputtting converted data
  signal AsciiEndString      : std_logic;                           -- True if end of string marker found (|)
  -- Serial out
  signal SerialData          : std_logic_vector(7 downto 0);        -- Data to output
  signal SerialDataEnable    : std_logic;                           -- Serial data latch
                                                                    -- Activated until we get a busy feedback


begin
  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to encode received keyboard data
  --{           into ASCII equivalents
  --{------------------------------------------------------------------------------}
  process (clk) is
  variable nibble: std_logic_vector(3 downto 0);
  begin
    if rising_edge(clk) then
      -- We use an external signal as our primary (p)reset
      if (key_datatoggle /= KeyDataTogglePrev) then
        -- New keyboard input data received
        KeyData             <= key_data;          
        -- Set for first bit to check for conversion
        AsciiCharIndex      <= 0;
        AsciiEndString      <= '0';
        AsciiBitNumber      <= b"0000000";
        -- Indicate we need to output converted data
        AsciiDataOutputting <= '1';
        SerialDataEnable    <= '0';
        StateDelay          <= '1';
        -- New status to check for new keyboard input data
        KeyDataTogglePrev <= key_datatoggle;
      else
        -- After changing the bit number, delay one cycle to wait for the
        -- conversion bneing update
        if StateDelay = '1' then
          StateDelay <= '0';
        else    
          -- If we are starting to output data, wait for it to be acknowledged
          if (SerialDataEnable = '1') then
            -- We are waiting for the serial data to be acknowledged  
            if (serial_busy = '1') then
              -- If sending indicates busy then we can remove our latch signal
              SerialDataEnable <= '0';
            end if;
          else
            -- Wait for serial data to be able to output again
            if (serial_busy = '0') then
              -- If already outputting converted data, continue with that
              if AsciiDataOutputting = '1' then
                -- We only want to output 'active' bits           
                if KeyData(0) = '0' then
                  -- Output next of the converted byte data
                  -- Check for special pipe symbol | that can indicate end of string 
                  if (ascii_data(AsciiCharIndex) /= x"7C") and (AsciiEndString = '0') then
                    SerialData       <= ascii_data(AsciiCharIndex);
                    SerialDataEnable <= '1';
                  else  
                    AsciiEndString <= '1';
                  end if;                
                end if;                            
                -- Select next byte and/or bit to convert
                if AsciiCharIndex = 7 then
                  -- End of converted ASCII bytes, select next bit or let it be the end
                  KeyData(126 downto 0) <= KeyData(127 downto 1);
                  AsciiCharIndex <=  0 ;
                  AsciiEndString <= '0';
                  if AsciiBitNumber = b"1111111" then
                    -- We don't reset AsciiBitNumber here
                    -- since a new conversion needs a change of AsciiBitNumber
                    -- so let this be done when new keyboard data is detected
                    AsciiDataOutputting  <= '0';
                  else
                    -- Next bit conversion to ASCII data
                    AsciiBitNumber <= AsciiBitNumber + 1;
                  end if;
                  StateDelay <= '1';
                else
                  -- Next converted byte
                  AsciiCharIndex <= AsciiCharIndex + 1;
                end if;
              end if;
            end if;  
          end if;
        end if;  
      end if;  
    end if;  
  end process;

  -- Output data
  ascii_bitnumber        <= AsciiBitNumber;
  serial_data            <= SerialData;
  serial_enable          <= SerialDataEnable;

  -- Debug (for checking with an oscilloscope for example)
  debug(0) <= SerialDataEnable;
  debug(1) <= serial_busy;
  debug(2) <= AsciiDataOutputting;
  debug(3) <= AsciiEndString;
  debug(4) <= '1' when AsciiCharIndex = 0 else '0';
  debug(5) <= '1' when AsciiCharIndex = 1 else '0';
  debug(6) <= '1' when AsciiCharIndex = 2 else '0';
  debug(7) <= '1' when AsciiCharIndex = 3 else '0';

end Behavioral;
