--{******************************************************************************}
--{ FileName............: keyboard_to_ascii_ledout.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Like keyboard_to_ascii_serialout, but outputting LED data instead, based
--{ on pressed key:
--{ Keys
--{  +  +25% brightness
--{  -  -25% brightness
--{  0  0%   brightness
--{  9  100% brightness
--{  1  Select left R
--{  2  Select left G
--{  3  Select left B
--{  4  Select right R
--{  5  Select right G
--{  6  Select right B
--{  7  Select all
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity keyboard_to_ascii_ledout is
  port (
    clk                         : in  std_logic;                       -- Main clock
    -- Keyboard
    key_data                    : in  std_logic_vector(127 downto 0);  -- Keyboard data detected
    key_datatoggle              : in  std_logic;                       -- Toggled for every new <key_data>
    -- For LED to keyboard conversion
    power_led_left_brightness_R : out std_logic_vector(7 downto 0);
    power_led_left_brightness_G : out std_logic_vector(7 downto 0);
    power_led_left_brightness_B : out std_logic_vector(7 downto 0);
    power_led_right_brightness_R: out std_logic_vector(7 downto 0);
    power_led_right_brightness_G: out std_logic_vector(7 downto 0);
    power_led_right_brightness_B: out std_logic_vector(7 downto 0);
    drive_led_left_brightness_R : out std_logic_vector(7 downto 0);
    drive_led_left_brightness_G : out std_logic_vector(7 downto 0);
    drive_led_left_brightness_B : out std_logic_vector(7 downto 0);
    drive_led_right_brightness_R: out std_logic_vector(7 downto 0);
    drive_led_right_brightness_G: out std_logic_vector(7 downto 0);
    drive_led_right_brightness_B: out std_logic_vector(7 downto 0);
    debug                       : out std_logic_vector(7 downto 0)        -- Debug
  );
end keyboard_to_ascii_ledout;

architecture Behavioral of keyboard_to_ascii_ledout is
  -- Bit number keys conversion:
  --   48- 55              CAPSLOCK RETUR INST/DEL RESTORE  LEFT UP     CPSLKLED
  --   56- 63     ESC      F13      F11   F9|      HELP     ALT  TAB    NOSCROLL
  --   64- 71     RUN/STOP Q        MEGA  SPACE    2        CTRL <-     1      
  --   72- 79     /        ^        =     RSHIFT|  CLR/HOME ;    *      GBP
  --   80- 87     <        @        :     >       -         L    P      +      
  --   88- 95     N        O        K     M       0         J    I      9      
  --   96-103     V        U        H     B       8         G    Y      7      
  --  104-111     X        T        F     C       6         D    R      5      
  --  112-119     LSHFTLCK E        S     Z       4         A    W      3      
  --  120-127     DOWN     F5       F3    F1      F7       RIGHT RETURN INST/DEL

  constant BITNUMBER_PLUS   : integer :=  87; -- + 25%
  constant BITNUMBER_MINUS  : integer :=  84; -- - 25%
  constant BITNUMBER_ZERO   : integer :=  92; -- 0 0%
  constant BITNUMBER_HUNDRED: integer :=  95; -- 9 100%
  constant BITNUMBER_LEFT_R : integer :=  71; -- 1
  constant BITNUMBER_LEFT_G : integer :=  68; -- 2
  constant BITNUMBER_LEFT_B : integer := 119; -- 3
  constant BITNUMBER_RIGHT_R: integer := 116; -- 4
  constant BITNUMBER_RIGHT_G: integer := 111; -- 5
  constant BITNUMBER_RIGHT_B: integer := 108; -- 6
  constant BITNUMBER_ALL    : integer := 103; -- 7

  signal KeyData   : std_logic_vector(127 downto 0);  -- Keyboard data
  -- Special attribute to prevent truncation/unconnected warnings
  attribute dont_touch : string;
  attribute dont_touch of KeyData : signal is "true";
  -- 0 1 2 3 4 5 == 0% 0% 25% 50% 75% 100% 100%   0 and 5 will be corrected to 1 and 4
  signal LedLeftR  : natural range 0 to 5;
  signal LedLeftG  : natural range 0 to 5;
  signal LedLeftB  : natural range 0 to 5;
  signal LedRightR : natural range 0 to 5;
  signal LedRightG : natural range 0 to 5;
  signal LedRightB : natural range 0 to 5;
  signal Selection : natural range 0 to 6;                     -- Aelection one of above or all (6)
  signal LedLeftRV : std_logic_vector(7 downto 0);
  signal LedLeftGV : std_logic_vector(7 downto 0);
  signal LedLeftBV : std_logic_vector(7 downto 0);
  signal LedRightRV: std_logic_vector(7 downto 0);
  signal LedRightGV: std_logic_vector(7 downto 0);
  signal LedRightBV: std_logic_vector(7 downto 0);
  
  signal Adjust           : natural range 0 to 5;              -- Adjust and state machine
  signal KeyDataTogglePrev: std_logic;                         -- To detect changes
  
begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Generate changed LED data based on bitnumber
  --{------------------------------------------------------------------------------}
  process (clk)
  variable LChar : character;
  variable LIndex: integer;
  variable LOrd  : integer;
  begin
    if rising_edge(clk) then
      if (key_datatoggle /= KeyDataTogglePrev) then
        KeyData           <= key_data;
        Adjust            <= 1;
        KeyDataTogglePrev <= key_datatoggle;
      else
        Adjust <= 0;
        case Adjust is
          when 0 => null;
          when 1 => -- New keyboard input data received
                    if KeyData(BITNUMBER_PLUS   ) = '0' then
                      Adjust <= 2;
                    end if;  
                    if KeyData(BITNUMBER_MINUS  ) = '0' then
                      Adjust <= 3;
                    end if;  
                    if KeyData(BITNUMBER_ZERO   ) = '0' then
                      Adjust <= 4;
                    end if;  
                    if KeyData(BITNUMBER_HUNDRED) = '0' then
                      Adjust <= 5;
                    end if;  
                    if KeyData(BITNUMBER_LEFT_R ) = '0' then
                      Selection <= 0;
                    end if;  
                    if KeyData(BITNUMBER_LEFT_G ) = '0' then
                      Selection <= 1;
                    end if;  
                    if KeyData(BITNUMBER_LEFT_B ) = '0' then
                      Selection <= 2;
                    end if;  
                    if KeyData(BITNUMBER_RIGHT_R) = '0' then
                      Selection <= 3;
                    end if;  
                    if KeyData(BITNUMBER_RIGHT_G) = '0' then
                      Selection <= 4;
                    end if;  
                    if KeyData(BITNUMBER_RIGHT_B) = '0' then
                      Selection <= 5;
                    end if;  
                    if KeyData(BITNUMBER_ALL    ) = '0' then
                      Selection <= 6;
                    end if;                      
          -- +25%
          when 2 => case Selection is
                      when 0 => LedLeftR  <= LedLeftR  + 1;
                      when 1 => LedLeftG  <= LedLeftG  + 1;
                      when 2 => LedLeftB  <= LedLeftB  + 1;
                      when 3 => LedRightR <= LedRightR + 1;
                      when 4 => LedRightG <= LedRightG + 1;
                      when 5 => LedRightB <= LedRightB + 1;
                      -- All
                      when 6 => LedLeftR  <= LedLeftR  + 1;
                                LedLeftG  <= LedLeftG  + 1;
                                LedLeftB  <= LedLeftB  + 1;
                                LedRightR <= LedRightR + 1;
                                LedRightG <= LedRightG + 1;
                                LedRightB <= LedRightB + 1;
                    end case;
          -- -25%          
          when 3 => case Selection is
                      when 0 => LedLeftR  <= LedLeftR  - 1;
                      when 1 => LedLeftG  <= LedLeftG  - 1;
                      when 2 => LedLeftB  <= LedLeftB  - 1;
                      when 3 => LedRightR <= LedRightR - 1;
                      when 4 => LedRightG <= LedRightG - 1;
                      when 5 => LedRightB <= LedRightB - 1;
                      -- All
                      when 6 => LedLeftR  <= LedLeftR  - 1;
                                LedLeftG  <= LedLeftG  - 1;
                                LedLeftB  <= LedLeftB  - 1;
                                LedRightR <= LedRightR - 1;
                                LedRightG <= LedRightG - 1;
                                LedRightB <= LedRightB - 1;
                    end case;
          -- 0%          
          when 4 => case Selection is
                      when 0 => LedLeftR  <= 1;
                      when 1 => LedLeftG  <= 1;
                      when 2 => LedLeftB  <= 1;
                      when 3 => LedRightR <= 1;
                      when 4 => LedRightG <= 1;
                      when 5 => LedRightB <= 1;
                      -- All
                      when 6 => LedLeftR  <= 1;
                                LedLeftG  <= 1;
                                LedLeftB  <= 1;
                                LedRightR <= 1;
                                LedRightG <= 1;
                                LedRightB <= 1;
                    end case;
          -- 100%          
          when 5 => case Selection is
                      when 0 => LedLeftR  <= 4;
                      when 1 => LedLeftG  <= 4;
                      when 2 => LedLeftB  <= 4;
                      when 3 => LedRightR <= 4;
                      when 4 => LedRightG <= 4;
                      when 5 => LedRightB <= 4;
                      -- All
                      when 6 => LedLeftR  <= 4;
                                LedLeftG  <= 4;
                                LedLeftB  <= 4;
                                LedRightR <= 4;
                                LedRightG <= 4;
                                LedRightB <= 4;
                    end case;
        end case;
        -- Under/overflow correction
        if LedLeftR  = 0 then
          LedLeftR <= 1;
        end if;  
        if LedLeftR  = 5 then
          LedLeftR <= 4;
        end if;  
        if LedLeftG  = 0 then
          LedLeftG <= 1;
        end if;  
        if LedLeftG  = 5 then
          LedLeftG <= 4;
        end if;  
        if LedLeftB  = 0 then
          LedLeftB <= 1;
        end if;  
        if LedLeftB  = 5 then
          LedLeftB <= 4;
        end if;  
        if LedRightR  = 0 then
          LedRightR <= 1;
        end if;  
        if LedRightR  = 5 then
          LedRightR <= 4;
        end if;  
        if LedRightG  = 0 then
          LedRightG <= 1;
        end if;  
        if LedRightG  = 5 then
          LedRightG <= 4;
        end if;  
        if LedRightB  = 0 then
          LedRightB <= 1;
        end if;  
        if LedRightB  = 5 then
          LedRightB <= 4;
        end if;  
      end if;      
    end if;
  end process;        

  -- Convert setting to brightness level
  LedLeftRV  <= x"00" when LedLeftR = 0 else
                x"00" when LedLeftR = 1 else  
                x"40" when LedLeftR = 2 else
                x"80" when LedLeftR = 3 else
                x"C0" when LedLeftR = 4 else
                x"FF" when LedLeftR = 5 else
                x"FF";
  LedLeftGV  <= x"00" when LedLeftG = 0 else
                x"00" when LedLeftG = 1 else  
                x"40" when LedLeftG = 2 else
                x"80" when LedLeftG = 3 else
                x"C0" when LedLeftG = 4 else
                x"FF" when LedLeftG = 5 else
                x"FF";
  LedLeftBV  <= x"00" when LedLeftB = 0 else
                x"00" when LedLeftB = 1 else  
                x"40" when LedLeftB = 2 else
                x"80" when LedLeftB = 3 else
                x"C0" when LedLeftB = 4 else
                x"FF" when LedLeftB = 5 else
                x"FF";
  LedRightRV <= x"00" when LedRightR = 0 else
                x"00" when LedRightR = 1 else  
                x"40" when LedRightR = 2 else
                x"80" when LedRightR = 3 else
                x"C0" when LedRightR = 4 else
                x"FF" when LedRightR = 5 else
                x"FF";
  LedRightGV <= x"00" when LedRightG = 0 else
                x"00" when LedRightG = 1 else  
                x"40" when LedRightG = 2 else
                x"80" when LedRightG = 3 else
                x"C0" when LedRightG = 4 else
                x"FF" when LedRightG = 5 else
                x"FF";
  LedRightBV <= x"00" when LedRightB = 0 else
                x"00" when LedRightB = 1 else  
                x"40" when LedRightB = 2 else
                x"80" when LedRightB = 3 else
                x"C0" when LedRightB = 4 else
                x"FF" when LedRightB = 5 else
                x"FF";

  power_led_left_brightness_R  <= LedLeftRV;
  power_led_left_brightness_G  <= LedLeftGV;
  power_led_left_brightness_B  <= LedLeftBV;
  power_led_right_brightness_R <= LedLeftRV;
  power_led_right_brightness_G <= LedLeftGV;
  power_led_right_brightness_B <= LedLeftBV;
  drive_led_left_brightness_R  <= LedRightRV;
  drive_led_left_brightness_G  <= LedRightGV;
  drive_led_left_brightness_B  <= LedRightBV;
  drive_led_right_brightness_R <= LedRightRV;
  drive_led_right_brightness_G <= LedRightGV;
  drive_led_right_brightness_B <= LedRightBV;

  debug(0) <= '1' when (Adjust = 0) else
              '0'; 
  debug(1) <= '1' when (Selection = 0) else
              '0';
  debug(2) <= '1' when (Selection = 1) else
              '0';
  debug(3) <= '1' when (Selection = 2) else
              '0';
  debug(4) <= '1' when (Selection = 3) else
              '0';
  debug(5) <= '1' when (Selection = 4) else
              '0';    
  debug(6) <= '1' when (Selection = 5) else
              '0';  
  debug(7) <= '1' when (Selection = 6) else
              '0';
  
end Behavioral;
