--{******************************************************************************}
--{ FileName............: keyboard_to_ascii.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Translate a bit number (0.. 127) into an ASCII string representation of the key pressed
--{
--{ 127..046  Key data
--{             127-120   INST/DEL   RETURN  CURS_RIGHT        F7           F1      F3         F5  CURS_DOWN
--{             119-112          3        W           A         4            Z       S          E      (**1)
--{             111-104          5        R           D         6            C       F          T          X
--{             103- 96          7        Y           G         8            B       H          U          V
--{              95- 88          9        I           J         0            M       K          O          N
--{              87- 80          +        P           L         -            >       :          @          <
--{              79- 72        GBP        *           ;  CLR/HOME  RIGHT_SHIFT       =          ^          /
--{              71- 64          1       <-        CTRL         2        SPACE    MEGA          Q   RUN/STOP
--{              63- 56  NO_SCROLL      TAB         ALT      HELP           F9     F11        F13        ESC
--{              55- 49      (**2)  CURS_UP   CURS_LEFT   RESTORE     INST/DEL  RETURN  CAPS_LOCK
--{              48- 46
--{                     (**1) LEFT_SHIFT and SHIFTLOCK
--{                     (**2) CAPS_LOCK (with timed re-invert, basically the LED status)
--{                
--{ Version       Comment
--{ 2025.02.22.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;
use work.mega_package.all;

entity keyboard_to_ascii is
  port (
    clk      : in  std_logic;                           -- Main clock
    bitnumber: in  std_logic_vector(6 downto 0);        -- Bit number to 'translate' (when it changes)
    asciidata: out TByteArray8                          -- Translated bit number to ASCII string
   );
end keyboard_to_ascii;


architecture Behavioral of keyboard_to_ascii is
  -- Note that 126/127 are skipped by the keyboard controller
  constant NUMBER000_127  : string := "|       |       |       |       |       |       |       |       " &
                                      "|       |       |       |       |       |       |       |       " &
                                      "|       |       |       |       |       |       |       |       " &
                                      "|       |       |       |       |       |       |       |       " &
                                      "|       |       |       |       |       |       |       |       " &
                                      "|       |       |       |       |       |       |       |       " &                                      
                                      "|       CAPSLOCKRETURN| INST/DELRESTORE|LEFT|   UP|     CPSLKLED" &
                                      "ESC|    F13|    F11|    F9|     HELP|   ALT|    TAB|    NOSCROLL" &
                                      "RUN/STOPQ|      MEGA|    |      2|      CTRL|   <-|     1|      " &
                                      "/|      ^|      =|      RSHIFT| CLR/HOME;|      *|      GBP|    " &
                                      "<|      @|      :|      >|      -|      L|      P|      +|      " &
                                      "N|      O|      K|      M|      0|      J|      I|      9|      " &
                                      "V|      U|      H|      B|      8|      G|      Y|      7|      " &
                                      "X|      T|      F|      C|      6|      D|      R|      5|      " &
                                      "LSHFTLCKE|      S|      Z|      4|      A|      W|      3|      " &
                                      "DOWN|   F5|     F3|     F1|     F7|     RIGHT|  RETURN| INST/DEL";


  signal   ACharacter     : character; 
  signal   BitNumberPrev  : std_logic_vector(6 downto 0);        -- Previous converted bit number
  
begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Generate 8 character ASCII text based on bitnumber
  --{------------------------------------------------------------------------------}
  process (clk)
  variable LChar : character;
  variable LIndex: integer;
  variable LOrd  : integer;
  begin
    if rising_edge(clk) then
      if BitNumberPrev /= bitnumber then
        BitNumberPrev <= bitnumber;
        for LLoop in asciidata'range loop
          -- One bit number results in 8 individual results
          -- Note that string indexes start at index 1
          LIndex := (to_integer(unsigned(bitnumber)) * 8) + LLoop + 1;
          LChar  := NUMBER000_127(LIndex);
          LOrd   := character'pos(LChar);
          asciidata(LLoop) <= std_logic_vector(to_unsigned(LOrd, 8));
        end loop;
      end if;
    end if;           
  end process;
  
end Behavioral;
