--{******************************************************************************}
--{ FileName............: keyboard.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Keyboard (MkI)
--{ Connected through J5
--{   Designation        Connection      U1 FPGA                   Description
--{   J5:                                                          Keyboard, 2x5 pin
--{        1  GND        GND                                       Power ground
--{        2  3V3        3.3V                                      Power 3V3
--{        3  KB_JTAGEN  KB_JTAGEN       B13   kb_jtagen           Keyboard Jtag enable
--{        4  KB_TMS     KB_TMS          D14   kb_tms              Keyboard Jtag TMS
--{        5  KB_TCK     KB_TCK          E13   kb_tck              Keyboard Jtag TCK
--{        6  KB_TDI     KB_TDI          D15   kb_tdi              Keyboard Jtag TDI
--{        7  KB_TDO     KB_TDO          E14   kb_tdo              Keyboard Jtag TDO
--{        8  KB_IO1     KB_IO1          A14   kb_io0              Keyboard I/O 0
--{        9  KB_IO2     KB_IO2          A13   kb_io1              Keyboard I/O 1
--{       10  KB_IO3     KB_IO3          C13   kb_io2              Keyboard I/O 2
--{
--{ The signals of interest:
--{   kb_io0  Keyboard I/O 0   Clock out (to   MkI keyboard)
--{   kb_io1  Keyboard I/O 1   Data out  (to   MkI keyboard)
--{   kb_io2  Keyboard I/O 2   Data in   (from MkI keyboard)
--{
--{ Keyboard controller notes (MkI):
--{ . Runs at 12 MHz
--{ . Has 4 RGB LEDs with 8-bit brightness
--{ . Samples our keyboard clock at 12MHz and does some deglitching
--{ . Latches data at the rising edge of our clock
--{ . The deglitching needs 5 of its cycles to accept a low or high (about 0.5 us)
--{   A 'sync' of the clock being high for 32 of its cycles is used to restart reading/writing key data
--{   Because 128 bits are being shifted in/out, this means the following:
--{   ------    ---     ---          ---     ---------//-----------
--{   Bit  |   |127|   |126|        | 0 |   |
--{         ---     ---     -- // --     ---
--{         <5> <5> <5>          <5> <5> <32>     <minimum cycles of the 12MHz clock>
--{   We will be using a 1 us high, 1 us low clock and use a delay after the 128 bits of data to get
--{   the requested updates/second.
--{   When 128 bits have been received then these bits are latched. 
--{ . Assumes 128 bit data streams. Data is shifted in/out starting from the most significant bit
--{   (thus bit 127 is output first and bit 0 last).
--{ . Data bits output (from keyboard controller):
--{     127..046  Key data
--{                 127-120   INST/DEL   RETURN  CURS_RIGHT        F7           F1      F3         F5  CURS_DOWN
--{                 119-112          3        W           A         4            Z       S          E      (**1)
--{                 111-104          5        R           D         6            C       F          T          X
--{                 103- 96          7        Y           G         8            B       H          U          V
--{                  95- 88          9        I           J         0            M       K          O          N
--{                  87- 80          +        P           L         -            >       :          @          <
--{                  79- 72        GBP        *           ;  CLR/HOME  RIGHT_SHIFT       =          ^          /
--{                  71- 64          1       <-        CTRL         2        SPACE    MEGA          Q   RUN/STOP
--{                  63- 56  NO_SCROLL      TAB         ALT      HELP           F9     F11        F13        ESC
--{                  55- 49      (**2)  CURS_UP   CURS_LEFT   RESTORE     INST/DEL  RETURN  CAPS_LOCK
--{                  48- 46
--{                     (**1) LEFT_SHIFT and SHIFTLOCK
--{                     (**2) CAPS_LOCK (with timed re-invert, basically the LED status)
--{     045..032  Date                             (not available for MKI (*))
--{     031..000  Version (x"4d4b4949" for MKII),  (not available for MKI (*))
--{   Data will be repeated after 128 clock cycles (*)
--{   (*) At least that is what the available source code was telling us, but observation
--{       indicates that after 81 bits the same key data is repeated. We might not
--{       have used the correct source code ....
--{   Key data:
--{     --{ . Data bits inputs (to keyboard controller):
--{     127..048  0
--{     095..004  Data          
--{               095..092  Green 0 / Blue  3 / Power LED right half
--{               087..084  Red   0 / Green 3 / Power LED right half
--{               079..076  Blue  0 / Red   3 / Power LED right half
--{               071..068  Green 1 / Blue  2 / Power LED left  half
--{               063..060  Red   1 / Green 2 / Power LED left  half
--{               055..052  Blue  1 / Red   2 / Power LED left  half
--{               047..044  Green 2 / Blue  1 / Drive LED right half
--{               039..036  Blue  2 / Green 1 / Drive LED right half
--{               031..024  Red   2 / Red   1 / Drive LED right half
--{               023..020  Green 3 / Blue  0 / Drive LED left  half
--{               015..012  Blue  3 / Green 0 / Drive LED left  half
--{               007..004  Red   3 / Red   0 / Drive LED left  half
--{     003..000  -
--{                
--{ Version       Comment
--{ 2025.02.16.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity keyboard is
  generic (
    CLKRATE      : natural := 200000000;                    -- Main clock frequency
    UPDATERATE   : natural := 1000;                         -- Updates/sec
    ONLYCHANGES  : boolean := True                          -- Report through DataOutToggle only when changed data or always
    );  
  port (
    clk          : in  std_logic;                           -- Main clock
    kb_clkout    : out std_logic;                           -- Keyboard clock output (to   keyboard controller)
    kb_dataout   : out std_logic;                           -- Keyboard data output  (to   keyboard controller)
    kb_datain    : in  std_logic;                           -- Keyboard data input   (from keyboard controller)
    dataout      : out std_logic_vector(127 downto 0);      -- Data read from   keyboard controller
    datain       : in  std_logic_vector(127 downto 0);      -- Data to write to keyboard controller
    dataouttoggle: out std_logic;                           -- Toggles for every completed keyboard read/write cycle
    debug        : out std_logic_vector(  7 downto 0)       -- Debug
   );
end keyboard;


architecture Behavioral of keyboard is
  -- The keyboard clock is derived from the main clock
  constant KEYBOARDRATE    : natural := 500000;                             -- 1 us high / 1us low
  constant COUNTER_LOAD    : natural := (CLKRATE / KEYBOARDRATE / 2) - 1;   -- Reload counter for our derived keyboard cycle
  constant SYNC_NUMBER     : integer := (KEYBOARDRATE / UPDATERATE) - 1;    -- Result in specified updates/sec
  signal   KeyboardCounter : integer range 0 to COUNTER_LOAD;               -- Counter (clock division) 
  signal   KeyboardClk     : std_logic;                                     -- Clock enable for deriving keyboard clock
  signal   ClkToggle       : std_logic;                                     -- Potential keyboard clock
  signal   BitNumberState  : integer range 0 to SYNC_NUMBER;                -- Bit number of data, also used as state machine
  signal   KbClkOut        : std_logic;                                     -- Keyboard clock
  signal   KbDataOut       : std_logic;                                     -- Keyboard data out
  signal   KbDataIn        : std_logic;                                     -- Keyboard data in
  signal   CtrlDataOut     : std_logic_vector(127 downto 0);                -- Data from keyboard controller (previous cycle)
  signal   CtrlDataIn      : std_logic_vector(127 downto 0);                -- Data for  keyboard controller (this cycle)
  signal   CycleDataOut    : std_logic_vector(127 downto 0);                -- Latched data input
  signal   CycleDataOutPrev: std_logic_vector(127 downto 0);                -- Latched data input (previous)
  signal   CycleToggle     : std_logic;                                     -- Toggled for every new CycleDataOut (if changed)
  signal   CycleToggleNext : std_logic;                                     -- Toggled for every new CycleDataOut (always)
  
begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Generate the keyboard clock
  --{ In      : <clk>               Clock
  --{           <KeyboardCounter>   Current counter
  --{ Out     : <KeyboardCounter>   Next counter
  --{           <KeyboardClk>       Asserted when <KeyboardCounter> is releaded/restarted
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if (KeyboardCounter = 0) then
        KeyboardCounter <= COUNTER_LOAD;
        KeyboardClk     <= '1';
      else
        KeyboardCounter <= KeyboardCounter - 1;
        KeyboardClk     <= '0';
      end if;
    end if;
  end process;  
  
  
  --{------------------------------------------------------------------------------}
  --{ Descript: For each keyboard clock
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if KeyboardClk = '1' then
        ClkToggle <= not ClkToggle;
        -- Check transition
        if (ClkToggle = '1'  ) then
          -- If 1->0 transition
          -- In the comments at the top of the page the order is depeicted as used
          -- bu the keyboard controller. However the keyboard controller does some additional
          -- things, that  
          -- Note that we can not use a CASE statement when using SYNC_NUMBER and such because
          -- it is then not 'detected' as a constant (which it is!)
          -- Therefore at some place if .. then .. else conditions had to be used 
          --
          -- The following strategy is used:
          --   BitNumberState        Data     Clock
          --      0 .. 127           Output    YES
          --      1 .. 81            Input     YES
          --     82 .. 127           Input '1' YES
          --    128                   /0             Latch input/output data
          --    129                   /0        NO   Indicate ready
          --    130 .. SYNC_NUMBER    /0        NO
          --    SYNC_NUMBER           /0             BitNumberState = 0
          --
          -- The very first 1->0 transistion is the start of the bitstream
          -- which means no data has yet been received from the keyboard controller,
          -- since the controller latches at a 1->0 transition.
          -- Also, the keyboard controller does not output the highest bit,
          -- bit 127, but outputs bit 125 instead. 
          -- It was observed that after 81 bits the same key data is repeated,
          -- thus -no- date/version bits
          --{   ------    ---     ---          ---     ---------//-----------
          --{   Bit  |   |127|   |126|        | 0 |   |
          --{         ---     ---     -- // --     ---
          --{         <5> <5> <5>          <5> <5> <32>     <minimum cycles of the 12MHz clock>
          -- ______    __    __    __    ___  _    __    __    __    __
          --       |__|  |__|  |__|  |__|  //  |__|  |__|  |__|  |__|  |__  KbClkOut    
          --  _____
          --_|  0  |_______________________//_____________________________ BitNumber 0 
          --        _____
          -- ______|  1  |_________________//_____________________________ BitNumber 1 == bit 125
          --                                    _____
          -- ______________________________//__| 81  |____________________ BitNumber 81
          -- ___________       ____________//_____________________________
          --            |_____|                                            Cursor right    
          -- _________________       ______//_____________________________
          --                  |_____|                                      F7    
          -- ________________________      //_____________________________
          --                         |____|                                F1    
          -- ______________________________//_______       _______________
          --                                        |_____|                INST/DEL    
          -- ______________________________//_____________       ________
          --                                              |_____|          RETURN    
          -- ______________________________//___________________       __
          --                                                    |_____|    Cursor right
          case BitNumberState is
            when   1 to 80   => -- Start
                                CtrlDataIn(127 downto 1) <= CtrlDataIn(126 downto 0);
                                CtrlDataIn(0)            <= KbDataIn;                                
            when  81 to 126  => -- Shift in '1' data (not active) until bit 125 is at position 125
                                CtrlDataIn(127 downto 1) <= CtrlDataIn(126 downto 0);
                                CtrlDataIn(0)            <= '1';
            when others      => null;
          end case;  
          
          -- Data out
          case BitNumberState is
            when   0 to 127  => KbDataOut                 <= CtrlDataOut(127);
                                CtrlDataOut(127 downto 1) <= CtrlDataOut(126 downto 0);
                                CtrlDataOut(0)            <= '0';
            when others      => KbDataOut                 <= '0';
          end case;
          
          -- Latching/Complete cycle toggle
          case BitNumberState is
            when 128         => -- Latching
                                CycleDataOut <= CtrlDataIn;
                                CtrlDataOut  <= datain;
                                CtrlDataIn   <= (others => '1');
            when 129         => -- Indicate data ready
                                CycleToggleNext  <= not CycleToggleNext;
                                if CycleDataOut /= CycleDataOutPrev then
                                  -- Indicate new data ready
                                  CycleToggle  <= not CycleToggle;
                                end if;
                                CycleDataOutPrev <= CycleDataOut;                 
            when others      => null;
          end case;  
       
          -- Next BitNumberState
          if BitNumberState = SYNC_NUMBER then
            BitNumberState <= 0;
          else
            BitNumberState <= BitNumberState + 1;
          end if;  
          
          -- Keyboard clock (1->0 transition or no clock)
          case BitNumberState is
            when   0 to 127  => KbClkOut <= '0';
            when others      => KbClkOut <= '1';
          end case;  
        else  
          -- Keyboard clock (0->1 transition)
          KbClkOut <= '1'; 
        end if;            
      end if;
    end if;
  end process;  

  dataouttoggle <= CycleToggle when ONLYCHANGES else CycleToggleNext;
  dataout       <= CycleDataOut;
  kb_clkout     <= KbClkOut;
  kb_dataOut    <= KbDataOut;
  KbDataIn      <= Kb_DataIn;

  debug(0) <= KbClkOut; 
  debug(1) <= KbDataIn;
  debug(2) <= KbDataOut;
  debug(3) <= CycleToggle;
  debug(4) <= CycleToggleNext;
  debug(5) <= '1' when (BitNumberState = 0) else
              '0';    
  debug(6) <= '1' when (BitNumberState = 82) else
              '0';  
  debug(7) <= ClkToggle;
  
  
end Behavioral;
