--{******************************************************************************}
--{ FileName............: bitstream_to_bitnumber_simulation.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Simulation for sending a converted keyboard bit number
--{
--{ Version       Comment
--{ 2025.04.05.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity bitstream_to_bitnumber_simulation is
end bitstream_to_bitnumber_simulation;

architecture Behavioral of bitstream_to_bitnumber_simulation is
  signal ClkIn                   : std_logic;
  signal BitstreamData           : std_logic_vector(127 downto 0);
  signal BitstreamDone           : std_logic;
  signal BitnumberValue          : std_logic_vector(  7 downto 0);
  signal BitnumberValidToggle    : std_logic;
  signal BitnumberValidTogglePrev: std_logic;
  signal BitnumberRequestToggle  : std_logic;
  
begin
  -- Instantiation of the device under test
  BitstreamToBitnunber: entity work.bitstream_to_bitnumber
    generic map (
      BITWIDTH       =>  128,
      BITNUMBERWIDTH =>    8,
      BITACTIVESTATE =>   '0'
    )  
    port map (
      clk                      => ClkIn,
      bitstream_data           => BitstreamData,
      bitstream_done           => BitstreamDone,
      bitnumber_value          => BitnumberValue,
      bitnumber_valid_toggle   => BitnumberValidToggle,
      bitnumber_request_toggle => BitnumberRequestToggle
    );
    

  --{------------------------------------------------------------------------------}
  --{ Descript: Generate the main clock
  --{------------------------------------------------------------------------------}
  process
  begin
    ClkIn <= '0';
		wait for 5 ns;
		ClkIn <= '1';
		wait for 5 ns;
  end process;

  --{------------------------------------------------------------------------------}
  --{ Descript: Do some conversions check
  --{           We generate KeyData and indicate the change with KeyDataToggle
  --{------------------------------------------------------------------------------}
  process
  begin
    BitnumberRequestToggle <= '0';
    -- All bits in the bittsream 'inactive'
    for LLoop in BitstreamData'range loop
      BitstreamData(LLoop) <= '1';
    end loop;
    -- Wait for ready for next start processing indication
    wait until BitstreamDone = '1';
    -- Take over 'next value' indication
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    -- Indicate and wait for next bitstream to convert
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until BitstreamDone = '0';
    -- We should not see a change of the bitnumber 
    for LLoop in 0 to 255 loop
      assert BitnumberValidToggle = BitnumberValidTogglePrev report "Bitnumber detected invalid activity 1" severity failure;
      wait until ClkIn = '1';
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '1' report "Bitstream not fully processed 00" severity failure;
    --
    --
    -- Change a  bit in the bitstream
    BitstreamData(64)        <= '0';
    -- Take over 'next value' indication
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    -- Indicate and wait for next bitstream to convert
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until BitstreamDone = '0';
    -- We should see the bitnumber '64' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 64
        assert BitnumberValue = std_logic_vector(to_unsigned(64, 8)) report "Bitnumber detected invalid activity 11" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '0' report "Bitstream fully processed 10" severity failure;
    -- After the first bit activity don't expect another one    
    -- Request next bitnumber
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until ClkIn = '1';
    -- Expect no other bitnumber
    for LLoop in 0 to 255 loop
      assert BitnumberValidToggle = BitnumberValidTogglePrev report "Bitnumber unexpectedly changed 11" severity failure;
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '1' report "Bitstream not fully processed 11" severity failure;
    BitstreamData(64 )       <= '1';
    --
    --

    -- Change two bits in the bitstream
    BitstreamData(32 )       <= '0';
    BitstreamData(127)       <= '0';
    -- Take over 'next value' indication
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    -- Indicate and wait for next bitstream to convert
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until BitstreamDone = '0';
    -- We should see the bitnumber '32' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 32
        assert BitnumberValue = std_logic_vector(to_unsigned(32, 8)) report "Bitnumber detected invalid activity 21" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '0' report "Bitstream fully processed 10" severity failure;
    -- Request next bitnumber
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    -- We should see the bitnumber '127' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 127
        assert BitnumberValue = std_logic_vector(to_unsigned(127, 8)) report "Bitnumber detected invalid activity 22" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    -- After this last bit bit activity don't expect another one
    -- And since bit 127 is the last bit, expect a 'done' too    
    assert BitstreamDone = '1' report "Bitstream not fully processed 23" severity failure;
    BitstreamData(32 )       <= '1';
    BitstreamData(127)       <= '1';


    --
    -- Change two bits in the bitstream
    BitstreamData(  0)       <= '0';
    BitstreamData(127)       <= '0';
    -- Take over 'next value' indication
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    -- Indicate and wait for next bitstream to convert
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until BitstreamDone = '0';
    -- We should see the bitnumber '0' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 0
        assert BitnumberValue = std_logic_vector(to_unsigned(0, 8)) report "Bitnumber detected invalid activity 31" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '0' report "Bitstream fully processed 30" severity failure;
    -- Request next bitnumber
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    -- We should see the bitnumber '127' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 127
        assert BitnumberValue = std_logic_vector(to_unsigned(127, 8)) report "Bitnumber detected invalid activity 32" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    -- After this last bit bit activity don't expect another one
    -- And since bit 127 is the last bit, expect a 'done' too    
    assert BitstreamDone = '1' report "Bitstream not fully processed 33" severity failure;
    BitstreamData(0  )       <= '1';
    BitstreamData(127)       <= '1';
    

    --
    -- Change two bits in the bitstream
    BitstreamData(0)       <= '0';
    BitstreamData(1)       <= '0';
    -- Take over 'next value' indication
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    -- Indicate and wait for next bitstream to convert
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until BitstreamDone = '0';
    -- We should see the bitnumber
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 0
        assert BitnumberValue = std_logic_vector(to_unsigned(0, 8)) report "Bitnumber detected invalid activity 41" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '0' report "Bitstream fully processed 40" severity failure;
    -- Request next bitnumber
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    wait until ClkIn = '1';
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    -- We should see the bitnumber '127' being indicated
    for LLoop in 0 to 255 loop
      -- Check for bitnumber indciation
      if BitnumberValidToggle /= BitnumberValidTogglePrev then
        -- Must be bit 1
        assert BitnumberValue = std_logic_vector(to_unsigned(1, 8)) report "Bitnumber detected invalid activity 42" severity failure;
        Exit;        
      end if;  
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    -- After the 2nd bit activity don't expect another one    
    -- Request next bitnumber
    BitnumberValidTogglePrev <= BitnumberValidToggle;
    BitnumberRequestToggle   <= not BitnumberRequestToggle;
    wait until ClkIn = '1';
    -- Expect no other bitnumber
    for LLoop in 0 to 255 loop
      assert BitnumberValidToggle = BitnumberValidTogglePrev report "Bitnumber unexpectedly changed 43" severity failure;
      wait until ClkIn = '1';
      -- Check for whole bitstream processed
      if BitstreamDone = '1' then
        exit;
      end if;  
    end loop;
    assert BitstreamDone = '1' report "Bitstream not fully processed 43" severity failure;
    BitstreamData(0)       <= '1';
    BitstreamData(1)       <= '1';


    assert BitstreamDone = not BitstreamDone report "This is NOT an error, simulation ended";
    wait;

  end process;


end Behavioral;
