--{******************************************************************************}
--{ FileName............: bitstream_to_bitnumber.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Translate active bits in a bitstream into individual bit numbers
--{ Note: Not used at the moment
--(
--{ Version       Comment
--{ 2025.04.05.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity bitstream_to_bitnumber is
  generic (
    BITWIDTH      : natural   := 128;                        -- Bitstream number of bits
    BITNUMBERWIDTH: natural   :=   7;                        -- Bitnumber number of bits
    BITACTIVESTATE: std_logic := '0'                         -- Level that indicates the active state of a bit in the stream
  );  
  port (
    clk                     : in  std_logic;
    bitstream_data          : in  std_logic_vector(BITWIDTH-1       downto 0);    -- Bitstream to process (latched when first conversion requested)
    bitstream_done          : out std_logic;                                      -- Indicates all bits in stream handled
    bitnumber_value         : out std_logic_vector(BITNUMBERWIDTH-1 downto 0);    -- Next active bitnumber
    bitnumber_valid_toggle  : out std_logic;                                      -- Toggle when next <bitnumber> available
    bitnumber_request_toggle: in  std_logic                                       -- Toggle for request next <bitnumber>
   );
end bitstream_to_bitnumber;


architecture Behavioral of bitstream_to_bitnumber is
  signal BitstreamData         : std_logic_vector(BITWIDTH-1       downto 0);    -- Bitstream to process (latched when first conversion requested)
  signal BitstreamDone         : std_logic := '0';                               -- Indicates all bits in stream handled
  signal BitnumberValue        : std_logic_vector(BITNUMBERWIDTH-1 downto 0) := (others => '0');    -- Next active bitnumber
  signal BitnumberCheckIndex   : integer range 0 to BITWIDTH-1;                  -- Active bitnumber to check
  signal BitnumberFindNext     : std_logic;                                      -- Indicates next bitnumber to generate
  signal BitnumberValidToggle  : std_logic := '0';                               -- Toggle when next <bitnumber> available
  signal BitnumberRequestToggle: std_logic;                                      -- Toggle for request next <bitnumber>



begin
  --{------------------------------------------------------------------------------}
  --{ Descript: Handle bitstream to bitnumber conversion
  --{------------------------------------------------------------------------------}
  process (clk)
  begin
    if rising_edge(clk) then
      if (bitnumber_request_toggle /= BitnumberRequestToggle) then
        -- If new bitstream, latch it and start at begin
        if (BitstreamDone = '1') then
          BitstreamDone       <= '0';
          BitnumberCheckIndex <= 0;
          BitstreamData       <= bitstream_data;
        end if;
        BitnumberRequestToggle <= bitnumber_request_toggle;
        BitnumberFindNext      <= '1';
      else
        -- No new request, check if we are searching the next bitnumber
        if (BitnumberFindNext = '1') then
          -- We are searching
          if (BitstreamData(BitnumberCheckIndex) = BITACTIVESTATE) then
            BitnumberValue       <= std_logic_vector(to_unsigned(BitnumberCheckIndex, BITNUMBERWIDTH));
            BitnumberValidToggle <= not BitnumberValidToggle;
            BitnumberFindNext    <= '0';            
          end if;
          -- Select next bit to search for
          if (BitnumberCheckIndex = (BITWIDTH-1)) then
            BitstreamDone <= '1';
          else  
            BitnumberCheckIndex <= BitnumberCheckIndex + 1;
          end if;
        end if;  
      end if;
    end if;           
  end process;

  bitstream_done         <= BitstreamDone;
  bitnumber_value        <= BitnumberValue;
  bitnumber_valid_toggle <= BitnumberValidToggle;
  
end Behavioral;
