--{******************************************************************************}
--{ FileName............: main_serialinout.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2025 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Implements:
--{ . Main clock
--{ . Blinky LED
--{ . Serial input that is echoed back to serial output
--{ . Debug data on P2 PMOD port
--{   . Data enable out   
--{   . Busy out        (also on green LED)
--{   . Bit stream out
--{   . Bit stream in
--{   . Toggle data in  (also on red LED)
--{   . Blinky          (also on main red LED)
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity main_serialinout is
  port (
    clk_in      : in  std_logic;
    reset_button: in  std_logic;
    led         : out std_logic;
    led_g       : out std_logic;
    led_r       : out std_logic;
    uart_txd    : out std_logic;
    rsrx        : in  std_logic;
    --
    p2lo        : out std_logic_vector(3 downto 0);     
    p2hi        : out std_logic_vector(3 downto 0)    
  );
end main_serialinout;

architecture Behavioral of main_serialinout is
  -- Our clock signal we derive our LED signals from
  signal   Clock200                 : std_logic;
  -- Serial data in
  signal   SerialInBitStreamIn      : std_logic;
  signal   SerialInDataOutByte      : std_logic_vector(7 downto 0);
  signal   SerialInDataOutToggle    : std_logic;
  signal   SerialInDataOutTogglePrev: std_logic := '0';
  -- Serial data out
  signal   SerialOutDataOutByte     : std_logic_vector(7 downto 0);
  signal   SerialOutDataEnable      : std_logic;
  signal   SerialOutBitStreamOut    : std_logic;
  signal   SerialOutBusy            : std_logic;
  -- Counters
  signal led_count                  : natural range 0 to 200000000;  
  -- LED toggle status
  signal led_toggle                 : std_logic;  

begin
  -- ** From 'mega65r6.vhdl':
  -- New clocking setup, using more optimised selection of multipliers
  -- and dividers, as well as the ability of some clock outputs to provide an
  -- inverted clock for free.
  -- Also, the 50 and 100MHz ethernet clocks are now independent of the other
  -- clocks, so that Vivado shouldn't try to meet timing closure in the (already
  -- protected) domain crossings used for those.
  clocks1: entity work.clocking
    port map ( 
      clk_in     => clk_in,
      clock27    => open,      -- clock27,    --   27     MHz
      clock41    => open,      -- cpuclock,   --   40.5   MHz
      clock50    => open,      -- ethclock,   --   50     MHz
      clock74p22 => open,      -- clock74p22,
      clock81p   => open,      -- pixelclock, --   81     MHz
      clock163   => open,      -- clock162,   --  162     MHz
      clock163m  => open,      -- clock162m,  --  162     MHz, phase shifted by -207 degrees for SDRAM read timing
      clock200   => Clock200,  -- clock200,   --  200     MHz
      clock270   => open,      -- clock270,   --  270     MHz
      clock325   => open       -- clock325    --  325     MHz
    );


  -- Instantiation of the serial input
  SerialIn_I: entity work.serialin
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk           => Clock200,
      bitstream     => SerialInBitStreamIn,
      dataout       => SerialInDataOutByte,
      dataouttoggle => SerialInDataOutToggle
    );

  -- Instantiation of the serial output
  SerialOut_I: entity work.serialout
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk        => Clock200,
      datain     => SerialOutDataOutByte,
      dataenable => SerialOutDataEnable,
      bitstream  => SerialOutBitStreamOut,
      busy       => SerialOutBusy
    );


  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to send echo back what is
  --{           received on the serial input. We assume we can always send data
  --{           (not busy)
  --{ Note    : No reset is used so startup is undefined (which character is being
  --{           being output)
  --{ In      : <Clock200>                   Clock
  --{           <SerialInDataOutTogglePrev>  Last detected new data status
  --{           <SerialInDataOutToggle>      Current new data status
  --{           <SerialOutBusy>              Indicates data is being transmitted
  --{ Out     : <SerialInDataOutTogglePrev>  Last detected new data status
  --{           <SerialOutDataEnable>        Data transmit latch
  --{           <SerialOutDataOutByte>       Data to transmit
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then    
      if (SerialInDataOutTogglePrev /= SerialInDataOutToggle) then       
        -- New serial input data received
        -- Echo the new data back
        SerialOutDataOutByte <= SerialInDataOutByte;
        SerialOutDataEnable  <= '1';
        -- New status to check for new serial input data
        SerialInDataOutTogglePrev <= SerialInDataOutToggle;
      end if;
      if (SerialOutDataEnable = '1') then
        -- We are waiting for the serial data to be taken over (acknowledged)  
        if (SerialOutBusy = '1') then
          -- If sending indicates busy then we can remove our latch signal
          SerialOutDataEnable <= '0';
        end if;
      end if;            
    end if;
  end process;


  -- UART input and output
  SerialInBitStreamIn <= rsrx;
  uart_txd            <= SerialOutBitStreamOut;
  -- Green LED D10 via 240E to 3.3V (LED_G)
  led_g <= SerialOutBusy;
  -- Red   LED D12 via 240E to 3.3V (LED_R)
  led_r <= SerialInDataOutToggle;

  -- Debug (for checking with an oscilloscope for example)
  -- PMOD P2
  --  1: p2lo(0)   7: p2hi(0)
  --  2: p2lo(1)   8: p2hi(1)
  --  3: p2lo(2)   9: p2hi(2)
  --  4: p2lo(3)  10: p2hi(3)
  --  5: GND      11: GND
  --  6: VCC (*)  12: VCC (*)
  -- (*) VCC needs to be switched on! 
  p2lo(0) <= SerialOutDataEnable;
  p2lo(1) <= SerialOutBusy;
  p2lo(2) <= SerialOutBitStreamOut;
  p2lo(3) <= led_toggle;
  p2hi(0) <= rsrx;
  p2hi(1) <= SerialInDataOutToggle;
  p2hi(2) <= led_toggle;
  p2hi(3) <= led_toggle;


  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to derive our LED data from
  --{           Note that at startup the registers are at an undetermined state,
  --{           until the 'reset_button' is activated 
  --{ In      : <Clock200>      Clock
  --{           <reset_button>
  --{           <led_toggle>    Current toggle state
  --{           <led_count>     Current counter value
  --{ Out     : <led_toggle>    Next toggle state
  --{           <led_count>     Next counter value
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then
      if reset_button = '1' then
        -- When reset, set presets
        led_toggle <= '0';
        led_count  <=  0;
      else
        -- When we counted up to 2000000000 then it is time to toggle the LEDs
        if led_count = 200000000-1 then
          led_toggle <= not led_toggle;
          led_count  <= 0;
        else
          led_count  <= led_count + 1;
        end if;
      end if;
    end if;  
  end process;

  -- Red   LED D9  via 240E to GND (ULED)  
  led <= led_toggle;
end Behavioral;
