--{******************************************************************************}
--{ FileName............: main_serialout.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 205 - 2025
--{------------------------------------------------------------------------------}
--{
--{ Implements:
--{ . Main clock
--{ . Blinky LED
--{ . Serial out (sending "Hello world ...." continuously with inter message delay))
--{ . Debug data on P2 PMOD port
--{   . Data enable  (also on green LED)   
--{   . Busy
--{   . Bit stream   (also on red LED)
--{   . Blinky       (alsp on main red LED)
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity main_serialout is
  port (
    clk_in      : in  std_logic;
    reset_button: in  std_logic;
    led         : out std_logic;
    led_g       : out std_logic;
    led_r       : out std_logic;
    uart_txd    : out std_logic;
    p2lo        : out std_logic_vector(3 downto 0);     
    p2hi        : out std_logic_vector(3 downto 0)    
  );
end main_serialout;

architecture Behavioral of main_serialout is
  -- Our clock signal we derive our LED signals from
  signal Clock200               : std_logic;
  -- Serial output
  type SerialMessage is array(0 to 31) of std_logic_vector(7 downto 0);
  constant SerialOutMessage     : SerialMessage := (x"48", x"65", x"6C", x"6C", x"6F", x"20", x"77", x"6F", x"72", x"6C", x"64", x"20",
                                                    x"31", x"32", x"33", x"34", x"35", x"36", x"37", x"38", x"39",
                                                    x"31", x"32", x"33", x"34", x"35", x"36", x"37", x"38", x"39",
                                                    x"0D", x"0A");
  signal   SerialOutDataOutByte : std_logic_vector(7 downto 0);
  signal   SerialOutDataEnable  : std_logic;
  signal   SerialOutBitStreamOut: std_logic;
  signal   SerialOutBusy        : std_logic;
  signal   SerialOutStringIndex : natural range 0 to SerialMessage'HIGH;
  signal   SerialOutDelay       : natural range 0 to 200000000;  
  
  -- Counters
  signal led_count  : natural range 0 to 200000000;  
  -- LED toggle status
  signal led_toggle : std_logic;  

begin
  -- ** From 'mega65r6.vhdl':
  -- New clocking setup, using more optimised selection of multipliers
  -- and dividers, as well as the ability of some clock outputs to provide an
  -- inverted clock for free.
  -- Also, the 50 and 100MHz ethernet clocks are now independent of the other
  -- clocks, so that Vivado shouldn't try to meet timing closure in the (already
  -- protected) domain crossings used for those.
  clocks1: entity work.clocking
    port map ( 
      clk_in     => clk_in,
      clock27    => open,      -- clock27,    --   27     MHz
      clock41    => open,      -- cpuclock,   --   40.5   MHz
      clock50    => open,      -- ethclock,   --   50     MHz
      clock74p22 => open,      -- clock74p22,
      clock81p   => open,      -- pixelclock, --   81     MHz
      clock163   => open,      -- clock162,   --  162     MHz
      clock163m  => open,      -- clock162m,  --  162     MHz, phase shifted by -207 degrees for SDRAM read timing
      clock200   => Clock200,  -- clock200,   --  200     MHz
      clock270   => open,      -- clock270,   --  270     MHz
      clock325   => open       -- clock325    --  325     MHz
    );


  -- Instantiation of the serial output
  SerialOut_I: entity work.serialout
    generic map (
      CLKRATE  => 200000000,
      BAUDRATE => 115200
    )
    port map (
      clk        => Clock200,
      datain     => SerialOutDataOutByte,
      dataenable => SerialOutDataEnable,
      bitstream  => SerialOutBitStreamOut,
      busy       => SerialOutBusy
    );


  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to send 'Hello world' continuously
  --{ Note    : No reset is used so startup is undefined (which character is being
  --{           being output)
  --{ In      : <Clock200>              Clock
  --{           <SerialOutDelay>        Current delay (if <> 0)
  --{           <SerialOutBusy>         Asserted when no new data is accepted
  --{           <SerialOutStringIndex>  Current string index
  --{ Out     : <SerialOutDataOutByte>  Next data to output
  --{           <SerialOutDataEnable>   Asserted when new string data is to be send
  --{           <SerialOutStringIndex>  Next string index
  --{           <SerialOutDelay>        Next delay
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then    
      if (SerialOutDataEnable = '1') then
        -- We are waiting for the serial data to be taken over (acknowledged)  
        if (SerialOutBusy = '1') then
          -- If sending indicates busy then we can remove our latch signal
          SerialOutDataEnable <= '0';
        end if;
      else
        -- Not waiting for acknowledge
        if (SerialOutDelay = 0) then
          -- Not delaying
          if (SerialOutBusy = '0') then
            -- We can send new data
            SerialOutDataOutByte <= SerialOutMessage(SerialOutStringIndex);
            SerialOutDataEnable  <= '1';
            -- Select next character for when we can output the next byte
            if (SerialOutStringIndex = SerialMessage'HIGH) then
              -- Complete message delay of somewhat less than 1 second
              -- The actual delay is less than a second because the delay starts
              -- when the last byte is starting to be transmitted
              SerialOutDelay       <= 200000000;
              SerialOutStringIndex <= 0;
            else  
              -- Inter character delay of somewhat less than 1 ms
              -- SerialOutDelay       <= 200000;
              SerialOutStringIndex <= SerialOutStringIndex + 1;
            end if;
          end if;
        else
          -- While delaying do not do anything special 
          SerialOutDelay <= SerialOutDelay - 1;
        end if;                    
      end if;      
    end if;
  end process;

  uart_txd <= SerialOutBitStreamOut;
  -- Green LED D10 via 240E to 3.3V (LED_G)
  led_g <= SerialOutBusy;
  -- Red   LED D12 via 240E to 3.3V (LED_R)
  led_r <= SerialOutBitStreamOut;

  -- Debug (for checking with an oscilloscope for example)
  -- PMOD P2
  --  1: p2lo(0)   7: p2hi(0)
  --  2: p2lo(1)   8: p2hi(1)
  --  3: p2lo(2)   9: p2hi(2)
  --  4: p2lo(3)  10: p2hi(3)
  --  5: GND      11: GND
  --  6: VCC (*)  12: VCC (*)
  -- (*) VCC needs to be switched on! 
  p2lo(0) <= SerialOutDataEnable;
  p2lo(1) <= SerialOutBusy;
  p2lo(2) <= SerialOutBitStreamOut;
  p2lo(3) <= led_toggle;
  p2hi(0) <= led_toggle;
  p2hi(1) <= led_toggle;
  p2hi(2) <= led_toggle;
  p2hi(3) <= led_toggle;

  --{------------------------------------------------------------------------------}
  --{ Descript: The clocked process that we use to derive our LED data from
  --{           Note that at startup the registers are at an undetermined state,
  --{           until the 'reset_button' is activated 
  --{ In      : <Clock200>      Clock
  --{           <reset_button>
  --{           <led_toggle>    Current toggle state
  --{           <led_count>     Current counter value
  --{ Out     : <led_toggle>    Next toggle state
  --{           <led_count>     Next counter value
  --{------------------------------------------------------------------------------}
  process (Clock200) is
  begin
    if rising_edge(Clock200) then
      if reset_button = '1' then
        -- When reset, set presets
        led_toggle <= '0';
        led_count  <=  0;
      else
        -- When we counted up to 2000000000 then it is time to toggle the LEDs
        if led_count = 200000000-1 then
          led_toggle <= not led_toggle;
          led_count  <= 0;
        else
          led_count  <= led_count + 1;
        end if;
      end if;
    end if;  
  end process;

  -- Red   LED D9  via 240E to GND (ULED)  
  led <= led_toggle;
end Behavioral;
