--{******************************************************************************}
--{ FileName............: clockonly_blinky.vhdl
--{ Author(s)...........: Marcel Majoor
--{ Copyright...........: 2024 - 2024
--{------------------------------------------------------------------------------}
--{
--{ Blinky example
--{
--{ . The 'reset_button' is a manually operated button, which means that at
--{   startup reset is not necessarily active, and blinking registers 
--{   are at an undetermided state
--{ . Expect -only- a single warning about 'parallel synthesis criteria not met', 
--{   since this is such a simple design that no parallelism is needed 
--{
--{ Version       Comment
--{ 2025.03.29.0  - First release
--{------------------------------------------------------------------------------}
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

entity clockonly_blinky is
  port ( 
    clk_in      : in STD_LOGIC;
    reset_button: in STD_LOGIC;
    led         : out STD_LOGIC;
    led_g       : out STD_LOGIC;
    led_r       : out STD_LOGIC);
end clockonly_blinky;

architecture Behavioral of clockonly_blinky is
  -- Our clock signal we derive our LED signals from
  signal Clock200   : std_logic;
  -- Counters
  signal Led_Count  : natural range 0 to 200000000;  
  -- LED toggle status
  signal Led_Toggle : std_logic;  

begin

  -- ** From 'mega65r6.vhdl':
  -- New clocking setup, using more optimised selection of multipliers
  -- and dividers, as well as the ability of some clock outputs to provide an
  -- inverted clock for free.
  -- Also, the 50 and 100MHz ethernet clocks are now independent of the other
  -- clocks, so that Vivado shouldn't try to meet timing closure in the (already
  -- protected) domain crossings used for those.
  clocks1: entity work.clocking
    port map ( 
      clk_in     => clk_in,
      clock27    => open,      -- clock27,    --   27     MHz
      clock41    => open,      -- cpuclock,   --   40.5   MHz
      clock50    => open,      -- ethclock,   --   50     MHz
      clock74p22 => open,      -- clock74p22,
      clock81p   => open,      -- pixelclock, --   81     MHz
      clock163   => open,      -- clock162,   --  162     MHz
      clock163m  => open,      -- clock162m,  --  162     MHz, phase shifted by -207 degrees for SDRAM read timing
      clock200   => Clock200,  -- clock200,   --  200     MHz
      clock270   => open,      -- clock270,   --  270     MHz
      clock325   => open       -- clock325    --  325     MHz
    );


  -- The clocked process that we use to derive our LED data from
  -- Note that at startup the registers are at an undetermined state,
  -- until the 'reset_button' is activated 
  process (Clock200, reset_button) is
  begin
    if rising_edge(Clock200) then
      if reset_button = '1' then
        -- When reset, set presets
        Led_Toggle <= '0';
        Led_Count  <=  0;
      else
        -- When we counted up to 2000000000 then it is time to toggle the LEDs
        if Led_Count = 200000000-1 then
          Led_Toggle <= not Led_Toggle;
          Led_Count  <= 0;
        else
          Led_Count  <= Led_Count + 1;
        end if;
      end if;
    end if;  
  end process;

  -- Red   LED D9  via 240E to GND (ULED)  
  led   <= Led_Toggle;
  -- Green LED D10 via 240E to 3.3V (LED_G)
  led_g <= not Led_Toggle;
  -- Red   LED D12 via 240E to 3.3V (LED_R)
  led_r <= not Led_Toggle;
  
end Behavioral;
