{******************************************************************************}
{ FileName............: FlexCopI2c                                             }
{ Project.............: FLEXCOP                                                }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{  I2C support                                                                 }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050409 - Initial release                                           }
{******************************************************************************}
unit FlexCopI2c;

interface
uses
  Saa7146aI2c, // Just for type
  FlexCopInterface,
  FlexCopRegisters,
  FlexCopGeneral,
  Sl1935Registers,
  Stv0299bRegisters,
  Windows;

const
  CDiSEqCBurstA  = $01;
  CDiSEqCBurstB  = $02;
  CDiSEqCCommand = $04;

//type
//  TDiSEqCData = array[0..5] of Byte;

  function FlexCopUsDelay                          (UsDelay : Dword): Boolean;
  function FlexCopReadEeprom                       (Handle  : THandle;
                                                    Address : Word;
                                                    var Data: array of Byte): Boolean;
  function FlexCopWriteEeprom                      (Handle  : THandle;
                                                    Address : Word;
                                                    Data    : array of Byte): Boolean;
  function FlexCopEepromChecksum                   (Data    : array of Byte): Byte;

  function FlexCopReadFromQpsk                     (Handle: THandle; AccessRegister: Byte; var Data: Byte): Boolean;
  function FlexCopWriteToQpsk                      (Handle: THandle; AccessRegister: Byte; Data: Byte): Boolean;
  function FlexCopWriteDefaultsToQpsk              (Handle: THandle; TunerType: Byte): Boolean;
  function FlexCopCheckQpskResetValues             (Handle: THandle): Boolean;
  function FlexCopCheckQpskDefaultValues           (Handle: THandle; TunerType: Byte): Boolean;
  function FlexCopWriteToSynthesizer               (Handle: THandle; AddressIndex: Byte; Data: array of Byte): Boolean;
  function FlexCopReadFromSynthesizer              (Handle: THandle; AddressIndex: Byte; var Data: Byte): Boolean;
  function FlexCopDetectSynthesizerAddress         (Handle: THandle): Byte;

  function FlexCopDiSEqCCommandVdr                 (Handle: THandle; CommandData: ShortString): Boolean;
  function FlexCopDiSEqCCommand2                   (Handle: THandle; CommandType: Byte;
                                                    CommandLength: Byte; CommandData: TDiSEqCData): Boolean;

  function FlexCopQpskSetSymbolRate                (Handle: THandle; SymbolRateKS: Dword): Boolean;
  function FlexCopQpskSetAGCInversion              (Handle: THandle; Inversion: Boolean): Boolean;
  function FlexCopQpskSetInversion                 (Handle: THandle; Inversion: Boolean): Boolean;
  function FlexCopQpskGetLockDetectorPercentage    (Handle: THandle; var Percentage: Byte): Boolean;
  function FlexCopQpskGetCarrierDeviation          (Handle: THandle; var DeviationKHz: Integer): Boolean;
  function FlexCopQpskGetSymbolRateDeviation       (Handle: THandle; var DeviationKHz: Integer): Boolean;
  function FlexCopQpskGetSignalPower               (Handle: THandle;
                                                    var StrengthPercentage: Double;
                                                    var StrengthDecibel   : Double): Boolean;
  function FlexCopQpskGetNoiseIndicator            (Handle: THandle; var Noise: Double): Boolean;
  function FlexCopQpskGetInputLevel                (Handle: THandle; var InputLevelDbm: Integer;
                                                    var InputLevelPercentage: Byte): Boolean;

  // LNB control
  function FlexCopLowBandLNB                       (Handle: THandle): Boolean;
  function FlexCopHighBandLNB                      (Handle: THandle): Boolean;
  function FlexCopDiSEqCCommand                    (Handle: THandle; Repeats: Byte; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;

  function FlexCopFrequencyLNB                     (Handle: THandle; FrequencyKHz: Dword;
                                                    LowBandLocalOscillatorKHz: Dword;
                                                    HighBandLocalOscillatorKHz: Dword;
                                                    TunerType: Byte;
                                                    AddressIndex: Byte): Boolean;

  function FlexCopFrequencyTunerLNB                (Handle: THandle; FrequencyKHz: Dword;
                                                    TunerType: Byte; AddressIndex: Byte): Boolean;

  function FlexCopUseLowBand                       (Handle: THandle; FrequencyKHz: Dword;
                                                    LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword;
                                                    TunerType: Byte): Boolean;


implementation
uses
  Math,
  SyncObjs,
  SysUtils;

var
  HighPerformanceAvailable: Boolean;                       // Indicates available high performance counter
  HighPerformanceFrequency: TLargeInteger;                 // Frequency of high performance counter
  I2cLock                 : TCriticalSection;              // Prevents multiple I2C operations which interfere


{------------------------------------------------------------------------------
  Params  : <UsDelay> Delay in us
  Returns : <Result>  True is success
                      False if no high performance counter (should not be possible)

  Descript: Delay for a defined time
  Notes   : If no high performace counter is available then the delay is
            still implemented but with a worse resolution.
            During the delay no processes are handled...
------------------------------------------------------------------------------}
function FlexCopUsDelay(UsDelay: Dword): Boolean;
var
  StartTiming: TLargeInteger;
  Timing     : TLargeInteger;
  DiffTiming : TLargeInteger;
  DeltaTiming: TLargeInteger;
  StartTime  : Dword;
  MsDelay    : DWord;
begin
  if HighPerformanceAvailable then
  begin
    Result := True;
    QueryPerformanceCounter(StartTiming);
    DeltaTiming := UsDelay * HighPerformanceFrequency;
    DeltaTiming := DeltaTiming div 1000000;
    repeat
      QueryPerformanceCounter(Timing);
      if Timing > StartTiming then
        DiffTiming := Timing - StartTiming
      else
        DiffTiming := StartTiming - Timing;
    until DiffTiming >= DeltaTiming;
  end
  else
  begin
    Result := False;
    MsDelay   := UsDelay div 1000;
    StartTime := GetTickCount;
    // Now wait for the delay
    repeat
      repeat
      until StartTime <> GetTickCount;
    until Dword(Abs(GetTickCount - StartTime)) > MsDelay;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>       <FlexCopCreateFile> handle
            <Command>      Command
            <SecondByte>   Second I2C byte to write
            <MaxTime>      Maximum time allowed for result (in us)
  Returns : <Result>       True if success
            <SecondByte>   Second I2C byte read

  Descript: Write control command for I2C setup
  Notes   : Typical timeout is about 400 us for a read. For a write operation
            to EEPROM this time is about 4000 us.
------------------------------------------------------------------------------}
function FlexCopI2cWriteCommand(Handle: THandle; Command: Dword; var SecondByte: Byte; MaxTime: Integer): Boolean;
var
  Retry  : Integer;
  Value  : Dword;
  Send   : Boolean;
  Success: Boolean;
  Retries: Integer;
begin
  Result := False;

  Send    := True;
  Retry   := 0;
  Retries := MaxTime div 50;
  Success := False;

  // Now write the command/data until send
  repeat
    if Send then
    begin
      if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopI2cControl, CFlexCopI2cReset) then
        Exit;
      if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopI2cControl, Command) then
        Exit;
    end;
    // 'Interval'
    FlexCopUsDelay(50);

    // Read the status
    if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopI2cControl, Value) then
      Exit;

    // Check for error and ready
    if (Value and CFlexCopI2cNoAcknowledge) <> 0 then
      // No new status requires resending the command
      Send := True
    else
    begin
      // Check for ready
      Send := False;
      if (Value and (CFlexCopI2cReady or CFlexCopI2cBusy)) = CFlexCopI2cReady then
      begin
        SecondByte := (Value shr CFlexCopI2c2ndByteShift) and CFlexCopI2c2ndByteAnd;
        Success := True;
      end;
    end;
    Inc(Retry);
  until Success or (Retry >= Retries);

  Result := Success;
end;


{------------------------------------------------------------------------------
  Params  : <Channel>        I2C channel to address (hardware line output)
            <I2cAddress>     I2C address of device to read/write to/from
                             The first bit defines the read (1) or write (0)
            <I2cFirstByte>   First I2C data byte
            <I2cSecondByte>  Second I2C data byte (if any)
            <TrasnferLength> Number of bytes to transfer (0=1, 1=2, 2=3, 3=4)
  Returns : <Result>         Constructed command from parameters

  Descript: Construct the command for the I2C operation
  Notes   : From the I2C address the operation (read or write) is derived
------------------------------------------------------------------------------}
function FlexCopI2cConstructCommand(Channel       : Dword;
                                    I2cAddress    : Byte;
                                    I2cFirstByte  : Byte;
                                    I2cSecondByte : Byte;
                                    TransferLength: Byte): Dword;
var
  Write: Dword;
begin
  if (I2cAddress and $01) = 0 then
    Write := CFlexCopI2cWriteOperation or CFlexCopI2cStart
  else
    Write := CFlexCopI2cReadOperation  or CFlexCopI2cStart;
  Result := ((Channel        and CFlexCopI2cChannelAnd    ) shl CFlexCopI2cChannelShift    ) or
            ((TransferLength and CFlexCopI2cWriteLengthAnd) shl CFlexCopI2cWriteLengthShift) or
            ((I2cSecondByte  and CFlexCopI2c2ndByteAnd    ) shl CFlexCopI2c2ndByteShift    ) or
            ((I2cFirstByte   and CFlexCopI2c1stByteAnd    ) shl CFlexCopI2c1stByteShift    ) or
            ((I2cAddress     and CFlexCopI2cAddressAnd    ) shr CFlexCopI2cAddressShift    ) or
            Write;

end;


{------------------------------------------------------------------------------
  Params  : <Handle>         <FlexCopCreateFile> handle
            <Channel>        I2C channel to address
            <I2cAddress>     I2C address of device to read/write to/from
                             Bit 0 should be '1' since it is a read operation
            <I2cRegister     Register of I2C device to read/write to/from
            <Data>           Data bytes to read (size of dynamic array indicates size)
                             Must be 1..4
  Returns : <Result>         True if success
            <Data>           Data bytes read

  Descript: Read bytes from I2C device
  Notes   :
------------------------------------------------------------------------------}
function FlexCopI2cReadBytes(Handle     : THandle;
                             Channel    : Dword;
                             I2cAddress : Byte;
                             I2cRegister: Byte;
                             var Data   : array of Byte): Boolean;
var
  Command: Dword;
  Loop   : Integer;
  RData  : Dword;
begin
  Result := False;
  // Write operation address is not allowed
  if (I2cAddress and 1) = 0 then
    Exit;
  if High(Data) > 3 then
    Exit;
  // Create command
  Command := FlexCopI2cConstructCommand(Channel, I2cAddress, I2cRegister, 0, High(Data));
  // Execute I2C operation (also reads first byte)
  if not FlexCopI2cWriteCommand(Handle, Command, Data[0], 1000) then
    Exit;
  // If more than 1 byte to read
  if High(Data) > 0 then
  begin
    // Read the data. We get 4 bytes in one go at the maximum
    if not FlexCopReadFromFlexCopRegister(Handle, CFlexCopI2cData, RData) then
      Exit;
    for Loop := 1 to High(Data) do
    begin
      Data[Loop] := RData and $FF;
      RData := RData shr 8;
    end;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>         <FlexCopCreateFile> handle
            <Channel>        I2C channel to address
                             Bit 0 should be '0' since it is a write operation
            <I2cAddress>     I2C address of device to read/write to/from
            <I2cRegister     Register of I2C device to read/write to/from
                             This is actually the 1st data byte send
            <Data>           Data bytes to write (size of dynamic array indicates size)
                             These data bytes are the 2nd, 3rd, ... data bytes
  Returns : <Result>         True if success

  Descript: Write bytes to I2C device
  Notes   : Some devices do not specificly use the first data byte as an
            register byte. In those situations the caller should re-arrange
            the bytes such that <I2CRegister> is the 1st data byte to be transmitted
            and <Data> the 2nd, 3rd etc.
------------------------------------------------------------------------------}
function FlexCopI2cWriteBytes(Handle     : THandle;
                              Channel    : Dword;
                              I2cAddress : Byte;
                              I2cRegister: Byte;
                              Data       : array of Byte): Boolean;
var
  Command: Dword;
  Loop   : Integer;
  RData  : Dword;
begin
  Result := False;
  // Read operation address is not allowed
  if (I2cAddress and 1) <> 0 then
    Exit;
  if High(Data) > 3 then
    Exit;
  // If more than 1 byte to write we first have to setup the I2C data register
  // with the remaining bytes (the first byte is through the initial command)
  if High(Data) > 0 then
  begin
    RData := 0;
    for Loop := High(Data) downto 1 do
    begin
      RData := RData shl 8;
      RData := RData or Data[Loop];
    end;
    // Write the data
    if not FlexCopWriteToFlexCopRegister(Handle, CFlexCopI2cData, RData) then
      Exit;
  end;

  // Create command
  Command := FlexCopI2cConstructCommand(Channel, I2cAddress, I2cRegister, Data[0], High(Data));
  // Execute I2C operation (also writes first byte) and automatically will use
  // I2C data register if more bytes were requested to be written
  // Note that the first byte written is by means of the <Command> and not <Data[0]>,
  // which is used as dummy
  if not FlexCopI2cWriteCommand(Handle, Command, Data[0], 10000) then
    Exit;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>         <FlexCopCreateFile> handle
            <Address>        Address of memory location
            <Data>           Size sets bytes to read
  Returns : <Result>         True if success
            <Data>           Data byte read

  Descript: Read byte(s) from EEPROM (on I2C bus)
  Notes   : The high bits of the requested address to read from have to be combined
            into the I2C address.
            Since only 4 bytes can be transferred at a time, more bytes are
            by means of multiple transfers.
------------------------------------------------------------------------------}
function FlexCopReadEeprom(Handle  : THandle;
                           Address : Word;
                           var Data: array of Byte): Boolean;
var
  I2cAddress  : Byte;
  ToTransfer  : Byte;
  Index       : Byte;
  Copy        : Byte;
  NrBytes     : Byte;
  TransferData: array of Byte;
begin
  ToTransfer := High(Data)+1;
  Index      := 0;
  repeat
    NrBytes := ToTransfer;
    if NrBytes > 4 then
      NrBytes := 4;
    SetLength(TransferData, NrBytes);
    I2cAddress := CFlexCopI2cAddressEepromRead or
                  ((Hi(Address+Index) and CFlexCopI2cAddressEepromHighAnd) shl CFlexCopI2cAddressEepromHighShift);
    Result := FlexCopI2cReadBytes(Handle, CFlexCopI2cChannelEeprom,
                I2cAddress, Lo(Address+Index), TransferData);
    for Copy := 0 to NrBytes-1 do
      Data[Index+Copy] := TransferData[Copy];
    Inc(Index, NrBytes);
    Dec(ToTransfer, NrBytes);
  until (ToTransfer = 0);
end;


{------------------------------------------------------------------------------
  Params  : <Handle>         <FlexCopCreateFile> handle
            <Address>        Address of memory location
            <Data>           Data to write
  Returns : <Result>         True if success

  Descript: Write byte(s) to EEPROM (on I2C bus)
  Notes   : The high bits of the requested address to read from have to be combined
            into the I2C address.
            Since only 4 bytes can be transferred at a time, more bytes are
            by means of multiple transfers.
------------------------------------------------------------------------------}
function FlexCopWriteEeprom(Handle : THandle;
                            Address: Word;
                            Data   : array of Byte): Boolean;
var
  I2cAddress  : Byte;
  ToTransfer  : Byte;
  Index       : Byte;
  Copy        : Byte;
  NrBytes     : Byte;
  TransferData: array of Byte;
begin
  ToTransfer := High(Data)+1;
  Index      := 0;
  repeat
    NrBytes := ToTransfer;
    if NrBytes > 4 then
      NrBytes := 4;
    SetLength(TransferData, NrBytes);
    for Copy := 0 to NrBytes-1 do
      TransferData[Copy] := Data[Index+Copy];
    I2cAddress := CFlexCopI2cAddressEepromWrite or
                  ((Hi(Address+Index) and CFlexCopI2cAddressEepromHighAnd) shl CFlexCopI2cAddressEepromHighShift);
    Result := FlexCopI2cWriteBytes(Handle, CFlexCopI2cChannelEeprom,
                I2cAddress, Lo(Address+Index), TransferData);
    Inc(Index, NrBytes);
    Dec(ToTransfer, NrBytes);
  until (ToTransfer = 0);
end;


{------------------------------------------------------------------------------
  Params  : <Data>           Data to calculate checksum for
  Returns : <Result>         Checksum

  Descript: Calculate checksum from bytes
  Notes   : Some data in the EEPROM has a simple checksum mechanism.
------------------------------------------------------------------------------}
function FlexCopEepromChecksum(Data: array of Byte): Byte;
var
  Loop: Integer;
begin
  Result := 0;
  for Loop := 0 to High(Data) do
    Result := Result xor Data[Loop];
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <AccessRegister>  Register to read from
  Returns : <Result>          True for success
            <Data>            Data read from register

  Descript: Read data from register of QPSK demodulator
  Notes   : Reading from the demodulator requires a dummy write (without data)
            to the register we want to read from. Then a read cycle will read
            the targetted data from the register.
------------------------------------------------------------------------------}
function FlexCopReadFromQpsk(Handle: THandle; AccessRegister: Byte; var Data: Byte): Boolean;
begin
  I2cLock.Acquire;
  try
    Result := FlexCopI2cReadBytes(Handle, CFlexCopI2cChannelFrontEnd,
              CStv0299bDemodulatorReadAddress, AccessRegister, Data);
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <AccessRegister>  Register to write to
            <Data>            Data to write to register
  Returns : <Result>          True for success

  Descript: Write data to register of QPSK demodulator
  Notes   : We always write a single register only, although it is possible
            to write consequtive registers.
            Although a retry is implemented this is essentially disabled because
            <RetryWhole> set to '1' will do NO retry at all!
------------------------------------------------------------------------------}
function FlexCopWriteToQpsk(Handle: THandle; AccessRegister: Byte; Data: Byte): Boolean;
begin
  I2cLock.Acquire;
  try
    Result := FlexCopI2cWriteBytes(Handle, CFlexCopI2cChannelFrontEnd,
                CStv0299bDemodulatorWriteAddress, AccessRegister, Data);
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <TunerType>       Type of tuner
  Returns : <Result>          True if success

  Descript: Write all default values of the QSPK demodulator
  Notes   :
------------------------------------------------------------------------------}
function FlexCopWriteDefaultsToQpsk(Handle: THandle; TunerType: Byte): Boolean;
var
  Loop: Byte;
begin
  Result := False;
  for Loop := Low(CStv0299bDefaultsTBMU24112) to High(CStv0299bDefaultsTBMU24112) do
  begin
    // Write value to register
    if CStv0299bDefaultsTBMU24112[Loop, 0] <> $FF then
      if not FlexCopWriteToQpsk(Handle, CStv0299bDefaultsTBMU24112[Loop, 0], CStv0299bDefaultsTBMU24112[Loop, 1]) then
        Exit;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <TunerType>       Type of tuner
  Returns : <Result>          True if values comply with default values

  Descript: Check values of the QSPK demodulator with there default values
  Notes   :
------------------------------------------------------------------------------}
function FlexCopCheckQpskDefaultValues(Handle: THandle; TunerType: Byte): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;
  for Loop := Low(CStv0299bDefaultsTBMU24112) to High(CStv0299bDefaultsTBMU24112) do
  begin
    if CStv0299bDefaultsTBMU24112[Loop, 0] <> $FF then
    begin
      // Read value from register
      if not FlexCopReadFromQpsk(Handle, CStv0299bDefaultsTBMU24112[Loop, 0], Data) then
        Exit;
      // Check it with expected value
      if Data <> CStv0299bDefaultsTBMU24112[Loop, 1] then
        Exit;
    end;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True if values comply with reset values

  Descript: Check values of the QSPK demodulator with there reset values
  Notes   :
------------------------------------------------------------------------------}
function FlexCopCheckQpskResetValues(Handle: THandle): Boolean;
var
  Loop: Byte;
  Data: Byte;
begin
  Result := False;

  for Loop := Low(CStv0299bResets) to High(CStv0299bResets) do
  begin
    // Read value from register
    if CStv0299bResets[Loop, 0] <> $FF then
    begin
      if not FlexCopReadFromQpsk(Handle, CStv0299bResets[Loop, 0], Data) then
        Exit;
      // Check it with expected value
      if Data <> CStv0299bResets[Loop, 1] then
        Exit;
    end;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Wait for empty Fifo
  Notes   :
------------------------------------------------------------------------------}
function FlexCopDiSEqCWaitIdle(Handle: THandle): Boolean;
var
  Data : Byte;
  Empty: Boolean;
  Retry: Integer;
begin
  Result := False;
  // Data is send as 9-bits; each bit is 33 periods of the 22 kHz signal
  // 33 * 9 == 300 periods == 14 ms per byte to transfer
  // A transfer is typically 54 ms long and will have a total length of 6 bytes
  // although 3 bytes is more typical
  Retry  := 100;
  repeat
    if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqCStatus, Data) then
      Exit;
    Empty := ((Data and CStv0299bFifoMask) = CStv0299bFifoEmpty);
    if not Empty then
    begin
      FlexCopUsDelay(1000);
      Dec(Retry);
    end;
  until Empty or (Retry = 0);
  if Retry = 0 then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Bit>             Bit to transmit
  Returns : <Result>          True for success

  Descript: Wait for fifo not full (can be written to)
  Notes   :
------------------------------------------------------------------------------}
function FlexCopDiSEqCWaitFree(Handle: THandle): Boolean;
var
  Data : Byte;
  Full : Boolean;
  Retry: Integer;
begin
  Result := False;
  // Data is send as 9-bits; each bit is 33 periods of the 22 kHz signal
  // 33 * 9 == 300 periods == 14 ms per byte to transfer
  // A transfer is typically 54 ms long and will have a total length of 6 bytes
  // although 3 bytes is more typical
  Retry  := 20;
  repeat
    if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqCStatus, Data) then
      Exit;
    Full := ((Data and CStv0299bFifoFull) = CStv0299bFifoFull);
    if Full then
    begin
      FlexCopUsDelay(1000);
      Dec(Retry);
    end;
  until (not Full) or (Retry = 0);
  if Retry = 0 then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <data>            Byte to transmit
  Returns : <Result>          True for success

  Descript: Send DiSEqC byte
  Notes   :
------------------------------------------------------------------------------}
function FlexCopDiSEqCSendByte(Handle: THandle; Data: Byte): Boolean;
begin
  Result := False;

  // Wait for space in fifo
  if not FlexCopDiSEqCWaitFree(Handle) then
    Exit;
  // Write data to fifo
  if not FlexCopWriteToQpsk(Handle, CStv0299bDiSEqCFifo, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <CommandType>     Type of command
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              Note: Only one bit should be activated here!
            <CommandLength>   Number of commands
            <Commands>        Pointer to commands
            <Delay>           Use implicit delay
  Returns : <Result>          True for success

  Descript: Send single DiSEqC message or burst (no repeat feature,
            no reset of original signals). The data is send as is.
  Notes   : This function is typically called from <FlexCopDiSEqCCommand>.
------------------------------------------------------------------------------}
function FlexCopSendDiSEqCMsg(Handle: THandle; CommandType: Byte;
  CommandLength: Byte; CommandData: TDiSEqCData; Delay: Boolean): Boolean;
var
  Loop: Byte;
begin
  Result := False;

  // Check for double bits (not allowed since we do either a burst or DiSEqc command)
  // or 'idle' commands (nothing to do).
  // Double bits return an error.
  case (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB or CDiSEqCCommand)) of
    $00:  begin
            Result := True;
            Exit;
          end;
    CDiSEqCBurstA: ;
    CDiSEqCBurstB: ;
    CDiSEqCCommand: if CommandLength = 0 then
         begin
           Result := True;
           Exit;
         end;
    else Exit;
  end;

  // Wait for empty DiSEqc fifo
  if not FlexCopDiSEqCWaitIdle(Handle) then
    Exit;
  // Setup for DiSEqC
  if not FlexCopWriteToQpsk(Handle, CStv0299bDiSEqC, CStv0299bModulationDiSEqC) then
    Exit;
  // After removing the continuous tone or change of voltage a delay of >15ms
  // is required
  if Delay then
    FlexCopUsDelay(16000);

  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) then
  begin
    for Loop := 0 to CommandLength-1 do
    begin
      if not FlexCopDiSEqCSendByte(Handle, CommandData[Loop]) then
        Exit;
    end;
    // Wait for empty fifo
    if not FlexCopDiSEqCWaitIdle(Handle) then
      Exit;
  end;
  // Do burst
  if ((CommandType and (CDiSEqCBurstA or CDiSEqcBurstB)) <> 0) then
  begin
    if not FlexCopWriteToQpsk(Handle, CStv0299bDiSEqC, CStv0299bModulationUnmodulated) then
      Exit;
    if (CommandType and CDiSEqCBurstA) <> 0 then
      FlexCopDiseqcSendByte(Handle, $00)
    else
      FlexCopDiseqcSendByte(Handle, $FF);
    // Wait for empty fifo
    if not FlexCopDiSEqCWaitIdle(Handle) then
      Exit;
  end;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Repeats>         Repeats for DiSEqC command
                              Typically the same as the number of cascaded DiSEqc
                              devices
            <CommandType>     Type of command (bit-wise selection)
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              It is allowed to enable both burst and
                              DiSEqc, although only one type of burst can be selected
                              (A has priority)
            <CommandLength>   Number of <CommandData> bytes
            <CommandData>     Data bytes to transmit through DiSEqC
                              First byte is not used but instead replaced by
                              locally generated framing byte.
  Returns : <Result>          Error number

  Descript: Send DiSEqc command and/or simple burst mechanism (or both)
            Can send it repeatedly. The data is expanded with the
            framing byte.
  Notes   : This call is used to send a sequence of DiSEqC and/or V-SEC
            commands.
            The 'burst' is used for simple switchers and can be used together
            with a 'normal' DiSEqC command.
            DiSEqC basically consists of the following:
            . framing byte
              $E0 == Command from master, no reply required, first transmission
              $E1 == Command from master, no reply required, repeated transmission
              $E2..$E7 require a reply from a slave which is not supported
            . address byte
              $X_ == family
              $_X == sub type
              $00 == Any device
              $10 == Any LNB, switcher or SMATV
              $11 == LNB
              $12 == LNB with loop through switching
              $14 == Switcher (DC blocking)
              $15 == Switcher with DC loop through
              $18 == SMATV
              $20 == Any polariser
              $21 == Linear polariser (skwe) controller
              $30 == Any positioner
              $31 == Polar/azimuth positioner
              $32 == Elevation positioner
              $40 == Any installer aid
              $41 == Signal strength analog value
              $6x == Reserved for address re-allocations
              $70 == Any intelligent slave interface
              $71 == Interface for subscriber controlled headends
              $Fx == Reserved for OEM extensions
            . command byte (only mandatory commands are shown)
              $00 == Reset DiSEqC microcontroller
              $38 == Write to port group 0 (committed switches), 1 additional byte for switch id
                     Additional data byte:
                     %1111Oxxx = Option (A/B)
                     %1111xPxx = Satellite Position (A/B)
                     %1111xxHx = Polarization (Vertical/Horizontal)
                     %1111xxxB = Band (Low/High)
              $39 == Write to port group 1 (uncommitted switches), 1 additional byte for switch id
                     Additional data byte:
                     %11114xxx = Switch 4 (A/B)
                     %1111x3xx = Switch 3 (A/B)
                     %1111xx2x = Switch 2 (A/B)
                     %1111xxx1 = Switch 1 (A/B)
              $58 == Write channel frequency (BCD), 2/3 additional bytes for frequency
              $60 == Stop positioner movement
              $63 == Disable limits
              $66 == Set east limit
              $67 == Set west limit
              $68 == Drive motor east, 1 additional byte with timeout/steps
              $69 == Drive motor west, 1 additional byte with timeout/steps
              $6A == Store satellite position, 1 additional byte for satellite id
              $6B == Drive motor to satellite position,  1 additional byte for satellite id
            . additional data bytes
------------------------------------------------------------------------------}
function FlexCopDiSEqCCommand(Handle: THandle; Repeats: Byte; CommandType: Byte; CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
var
  DiSEqCOriginal: Byte;
  RepeatCommands: Byte;
begin
  Result := False;

  // Because DiSEqC distorts the 22 kHz signal we need to restore it afterwards
  // For this we need the current status
  if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqC, DiSEqCOriginal) then
    Exit;

  // First we have to handle any DiSEqc command
  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) and (CommandLength > 0) then
  begin
    CommandData[0] := $E0;
    if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength, CommandData, True) then
      Exit;
    RepeatCommands := Repeats;
    while RepeatCommands > 0 do
    begin
      // Before a repeated transmission is send a pause of 100 ms is
      // required to have the device being initialized.
      FlexCopUsDelay(100000);
      // Command from master, no reply required, repeated transmission
      CommandData[0] := $E1;
      if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength, CommandData, True) then
        Exit;
      Dec(RepeatCommands);
    end;
  end;
  // If only a burst is requested
  if (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB)) <> 0 then
  begin
    if (CommandType and CDiSEqCBurstA) <> 0 then
    begin
      if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, CommandData, True) then
        Exit;
    end
    else
      if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, CommandData, True) then
        Exit;
  end;

  // Restore 22kHz tone, we need a minimum of 15 ms delay first
  FlexCopUsDelay(16000);
  if (DiSEqCOriginal and not CStv0299bModulationModeMask) = CStv0299bModulationContinuous then
    FlexCopHighBandLNB(Handle)
  else
    FlexCopLowBandLNB(Handle);

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          Handle
            <CommandType>     Type of command (bit-wise selection)
                              $01 = Burst satellite position A
                              $02 = Burst satellite position B
                              $04 = DiSEqC command in <CommandData>
                              It is allowed to enable both burst and
                              DiSEqc, although only one type of burst can be selected
                              (A has priority)
            <CommandLength>   Number of <CommandData> bytes
            <CommandData>     Data bytes to transmit through DiSEqC
                              First byte is not used but instead replaced by
                              locally generated framing byte.
  Returns : <Result>          Error number

  Descript: Send DiSEqc command and/or simple burst mechanism (or both)
            Data is used -as is-.
  Notes   : This call is used to send a sequence of DiSEqC and/or V-SEC
            commands.
            The 'burst' is used for simple switchers and can be used together
            with a 'normal' DiSEqC command.
            DiSEqC basically consists of the following:
            . framing byte
              $E0 == Command from master, no reply required, first transmission
              $E1 == Command from master, no reply required, repeated transmission
              $E2..$E7 require a reply from a slave which is not supported
            . address byte
              $X_ == family
              $_X == sub type
              $00 == Any device
              $10 == Any LNB, switcher or SMATV
              $11 == LNB
              $12 == LNB with loop through switching
              $14 == Switcher (DC blocking)
              $15 == Switcher with DC loop through
              $18 == SMATV
              $20 == Any polariser
              $21 == Linear polariser (skwe) controller
              $30 == Any positioner
              $31 == Polar/azimuth positioner
              $32 == Elevation positioner
              $40 == Any installer aid
              $41 == Signal strength analog value
              $6x == Reserved for address re-allocations
              $70 == Any intelligent slave interface
              $71 == Interface for subscriber controlled headends
              $Fx == Reserved for OEM extensions
            . command byte (only mandatory commands are shown)
              $00 == Reset DiSEqC microcontroller
              $38 == Write to port group 0 (committed switches), 1 additional byte for switch id
                     Additional data byte:
                     %1111Oxxx = Option (A/B)
                     %1111xPxx = Satellite Position (A/B)
                     %1111xxHx = Polarization (Vertical/Horizontal)
                     %1111xxxB = Band (Low/High)
              $39 == Write to port group 1 (uncommitted switches), 1 additional byte for switch id
                     Additional data byte:
                     %11114xxx = Switch 4 (A/B)
                     %1111x3xx = Switch 3 (A/B)
                     %1111xx2x = Switch 2 (A/B)
                     %1111xxx1 = Switch 1 (A/B)
              $58 == Write channel frequency (BCD), 2/3 additional bytes for frequency
              $60 == Stop positioner movement
              $63 == Disable limits
              $66 == Set east limit
              $67 == Set west limit
              $68 == Drive motor east, 1 additional byte with timeout/steps
              $69 == Drive motor west, 1 additional byte with timeout/steps
              $6A == Store satellite position, 1 additional byte for satellite id
              $6B == Drive motor to satellite position,  1 additional byte for satellite id
            . additional data bytes
------------------------------------------------------------------------------}

function FlexCopDiSEqCCommand2(Handle: THandle; CommandType: Byte;
  CommandLength: Byte; CommandData: TDiSEqCData): Boolean;
var
  DiSEqCOriginal: Byte;
begin
  Result := False;

  // Because DiSEqC distorts the 22 kHz signal we need to restore it afterwards
  // For this we need the current status
  if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqC, DiSEqCOriginal) then
    Exit;

  // First we have to handle any DiSEqc command
  if ((CommandType and CDiSEqCCommand) = CDiSEqCCommand) and (CommandLength > 0)
    then
    if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCCommand, CommandLength,
      CommandData, True) then
      Exit;
  // If only a burst is requested
  if (CommandType and (CDiSEqCBurstA or CDiSEqCBurstB)) <> 0 then
  begin
    if (CommandType and CDiSEqCBurstA) <> 0 then
    begin
      if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, CommandData, True)
        then
        Exit;
    end
    else if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, CommandData,
      True) then
      Exit;
  end;

  // Restore 22kHz tone, we need a minimum of 15 ms delay first
  FlexCopUsDelay(16000);
  if (DiSEqCOriginal and CStv0299bModulationModeMask) =
    CStv0299bModulationContinuous then
    FlexCopHighBandLNB(Handle)
  else
    FlexCopLowBandLNB(Handle);

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Extracted>  String to process with as result the extracted part
  Returns : <Result>     Original string without extracted part

  Descript: Extract part from string (space separated)
  Notes   :
------------------------------------------------------------------------------}

function ExtractSpacedPart(var Extracted: ShortString): ShortString;
var
  SpacePos: Integer;
begin
  Result := '';
  Extracted := Trim(Extracted);
  SpacePos := Pos(' ', Extracted);
  if SpacePos <> 0 then
  begin
    Result := Copy(Extracted, 1, SpacePos - 1);
    Delete(Extracted, 1, SpacePos);
  end
  else
  begin
    Result := Extracted;
    Extracted := '';
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Setting>         Linux VDR type of string
  Returns : <Result>          True for success
                              False if error detected (DiSEqC part is not checked)
            <Satellite>       Satellite
            <SLOF>            Switch LOF
                              0 if error
            <Polarity>        Polarity
                              ' ' if error
            <LOF>             LOF
                              0 if error
            <DiSEqC>          DiSEqC command

  Descript: Extract individual information from setting string
  Notes   : DiSEqC configuration for VDR
              Format:   satellite slof polarization lof command
                            |       |        |       |      |
                            |       |        |       |     'DiSEqC' command (optional)
                            |       |        |      Local Oscillator Frequency
                            |       |       H/V polarization
                            |      Switch frequency
                           Satellite identifier
            S19.2E  11700 V  9750  t v W15 [E0 10 38 F0] W15 A W15 t
------------------------------------------------------------------------------}

function ExtractFromDiSEqCSetting(
  Setting: ShortString;
  var Satellite: ShortString;
  var SLOF: Integer;
  var Polarity: Char;
  var LOF: Integer;
  var DiSEqC: ShortString): Boolean;
var
  ProcessString: ShortString;
  Error: Integer;
begin
  Result := True;
  Satellite := '';
  SLOF := 0;
  Polarity := ' ';
  LOF := 0;
  DiSEqC := '';

  Satellite := ExtractSpacedPart(Setting); // Satellite
  ProcessString := ExtractSpacedPart(Setting); // SLOF
  Val(ProcessString, SLOF, Error);
  if (ProcessString = '') or (Error <> 0) then
  begin
    SLOF := 0;
    Result := False;
  end;
  ProcessString := ExtractSpacedPart(Setting); // Polarity
  if Length(ProcessString) <> 1 then
  begin
    Polarity := ' ';
    Result := False;
  end
  else
  begin
    Polarity := ProcessString[1];
    if not ((Polarity) in ['V', 'v', 'H', 'h']) then
      Result := False;
  end;
  ProcessString := ExtractSpacedPart(Setting); // LOF
  Val(ProcessString, LOF, Error);
  if (ProcessString = '') or (Error <> 0) then
  begin
    LOF := 0;
    Result := False;
  end;
  DiSEqC := Setting; // DiSEqC
end;

{------------------------------------------------------------------------------
  Params  : <Setting>         Linux VDR type of string
  Returns : <Result>          True for success
                              False if error detected (DiSEqC part is not checked)
            <Satellite>       Satellite
            <SLOF>            Switch LOF
                              0 if error
            <Polarity>        Polarity
                              ' ' if error
            <LOF>             LOF
                              0 if error
            <DiSEqC>          DiSEqC command

  Descript: Create setting string froms ettings
  Notes   :
------------------------------------------------------------------------------}

function CreateDiSEqCSetting(
  Satellite: ShortString;
  SLOF: Integer;
  Polarity: Char;
  LOF: Integer;
  DiSEqC: ShortString): ShortString;
begin
  Result := format('%s %5.5d %s %5.5d %s', [Satellite, SLOF, Polarity, LOF,
    DiSEqC]);
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          handle
            <CommandData>     Linux VDR type of string
  Returns : <Result>          True for success

  Descript: Send DiSEqC sequence. Equivalent with Linux (VDR) definition.
  Notes   : Commands are processed until no more commands or an error is detected,
            which means that incorrect command sequences are carried out until
            the incorrect part is processed.
            DiSEqC configuration for VDR
              Format:   satellite slof polarization lof [command]
                            |       |        |       |      |
                            |       |        |       |     'DiSEqC' command (optional)
                            |       |        |      Local Oscillator Frequency
                            |       |       H/V polarization
                            |      Switch frequency
                           Satellite identifier
              The optional [command] is made up of the following
                t         Tone off
                T         Tone on
                v         Voltage low (13V)
                V         Voltage high (18V)
                A         Mini A (burst A)
                B         Mini B (burst B)
                Wxx       Wait xx milliseconds
                [xx ..]   Hex code sequence (max 6), eg [E0 10 38 F0]
            Example:
            S19.2E  11700 V  9750  t v W15 [E0 10 38 F0] W15 A W15 t
------------------------------------------------------------------------------}

function FlexCopDiSEqCCommandVdr(Handle: THandle; CommandData: ShortString):
  Boolean;
var
  Satellite: ShortString;
  Slof: Integer;
  Polarity: Char;
  Lof: Integer;
  DiSEqCCommand: ShortString;
  ProcessString: ShortString;
  Error: Integer;
  Value: Integer;
  DiSEqCData: TDiSEqCData;
  DiSEqCIndex: Integer;
  EndDetected: Boolean;
begin
  Result := True;

  if ExtractFromDiSEqCSetting(CommandData, Satellite, Slof, Polarity, Lof,
    DiSEqCCommand) then
    while (DiSEqCCommand <> '') and (Result = True) do
    begin
      ProcessString := ExtractSpacedPart(DiSEqCCommand);
      if ProcessString <> '' then
      begin
        case Ord(ProcessString[1]) of
          Ord('t'):
            begin
              // Tone off (22 kHz)
              if not FlexCopLowBandLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('T'):
            begin
              // Tone on (22 kHz)
              if not FlexCopHighBandLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('v'):
            begin
              // 13V
              if not FlexCopVerticalPolarityLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('V'):
            begin
              // 18V
              if not FlexCopHorizontalPolarityLNB(Handle) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('A'), Ord('a'):
            begin
              // Burst A
              if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstA, 0, DiSEqCData,
                False) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('B'), Ord('b'):
            begin
              // Burst B
              if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCBurstB, 0, DiSEqCData,
                False) then
              begin
                Result := False;
                Exit;
              end;
            end;
          Ord('W'), Ord('w'):
            begin
              // Wait
              Delete(ProcessString, 1, 1);
              Val(ProcessString, Value, Error);
              if (Error = 0) and (Value > 0) then
                FlexCopUsDelay(Value * 1000);
            end;
          Ord('['):
            begin
              // DiSEqC command sequence
              Delete(ProcessString, 1, 1);
              // In case a '[' without trailing data was used read next data
              if ProcessString = '' then
                ProcessString := ExtractSpacedPart(DiSEqCCommand);
              DiSEqCIndex := Low(DiSEqCData);
              EndDetected := False;
              repeat
                // Check on exceeding length
                if DiSEqCIndex > High(DiSEqCData) then
                begin
                  Result := False;
                  Exit;
                end;
                // Check on two byte value
                // Since a single byte command just don't exists it does not
                // matter if the command was something like '[E0]' which
                // should be allowed but is detected as en error here (which
                // is good BTW).
                if Length(ProcessString) <> 2 then
                begin
                  Result := False;
                  Exit;
                end;
                // Convert as hexadecimal
                Val('$' + ProcessString, DiSEqCData[DiSEqCIndex], Error);
                // Check on conversion error
                if Error <> 0 then
                begin
                  Result := False;
                  Exit;
                end;
                Inc(DiSEqCIndex);
                if not EndDetected then
                begin
                  // Get next data 'byte'. If we detect the ']' marker then
                  // indicate this is only a final pass is done
                  ProcessString := ExtractSpacedPart(DiSEqCCommand);
                  if (Length(ProcessString) = 1) and (ProcessString[1] = ']')
                    then
                  begin
                    // If we are dealing with a separate ']' we are done too
                    ProcessString := '';
                  end;
                  if Length(ProcessString) > 2 then
                  begin
                    if ProcessString[3] <> ']' then
                    begin
                      Result := False;
                      Exit;
                    end;
                    Delete(ProcessString, 3, 1);
                    EndDetected := True;
                  end;
                end
                else
                  ProcessString := '';
              until (ProcessString = '');
              // Send DiSEqCData
              if not FlexCopSendDiSEqCMsg(Handle, CDiSEqCCommand, DiSEqCIndex,
                DiSEqCData, False) then
              begin
                Result := False;
                Exit;
              end;
            end;
        else
          Result := False;
        end;
      end;
    end
  else
    Result := False;
end;



{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <AddressIndex>    Address of SL1935 chip (0..3)
            <Data>            Data to write (diver1, divider2, control1, control2)
  Returns : <Result>          True for success

  Descript: Write data to synthesizer
  Notes   : Since the synthesizer is NOT connected to the FLEXCOP, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
------------------------------------------------------------------------------}
function FlexCopWriteToSynthesizer(Handle: THandle; AddressIndex: Byte; Data: array of Byte): Boolean;
var
  AByte    : Byte;
  SData    : array[0..2] of Byte;
begin
  Result := False;
  if AddressIndex > High(CSl1935SynthesizerWriteAddress) then
    Exit;
  if High(Data) <> 3 then
    Exit;
  I2cLock.Acquire;
  try
    // First we need to enable the repeater of the QSPK demodulator
    AByte := CStv0299bDefaultsTBMU24112[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn;
    if not FlexCopI2cWriteBytes(Handle, CFlexCopI2cChannelFrontEnd,
             CStv0299bDemodulatorWriteAddress, CStv0299bI2cRpt, AByte) then
      Exit;
    // Now write to the synthesizer
    // Since the first byte send is the 'register' we need to do some rearranging
    SData[0] := Data[1];
    SData[1] := Data[2];
    SData[2] := Data[3];
    Result := FlexCopI2cWriteBytes(Handle, CFlexCopI2cChannelFrontEnd,
               CSl1935SynthesizerWriteAddress[AddressIndex], Data[0], SData);
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <AddressIndex>    Address of SL1935 chip (0..3)
  Returns : <Result>          True for success
            <Data>            Data read (status register)

  Descript: Read data from synthesizer
  Notes   : Since the synthesizer is NOT connected to the FLEXCOP, but to the
            I2C repeater of the QSPK demodulator we need to enable the repeater
            too prior to this. This repeater is automatically disabled after
            a STOP on the I2C bus.
------------------------------------------------------------------------------}
function FlexCopReadFromSynthesizer(Handle: THandle; AddressIndex: Byte; var Data: Byte): Boolean;
var
  AByte    : Byte;
begin
  Result := False;
  if AddressIndex > High(CSl1935SynthesizerWriteAddress) then
    Exit;

  I2cLock.Acquire;
  try
    // First we need to enable the repeater of the QSPK demodulator
    AByte := CStv0299bDefaultsTBMU24112[CStv0299bI2cRpt, 1] or CStv0299bI2cRepeaterOn;
    if not FlexCopI2cWriteBytes(Handle, CFlexCopI2cChannelFrontEnd,
             CStv0299bDemodulatorWriteAddress, CStv0299bI2cRpt, AByte) then
      Exit;
    // Now read from the synthesizer
    Result := FlexCopI2cReadBytes(Handle, CFlexCopI2cChannelFrontEnd,
               CSl1935SynthesizerReadAddress[AddressIndex], 0, Data);
  finally
    I2cLock.Release;
  end;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          Address of synthesizer (0..3) or 255 if failed

  Descript: Detect address to be used for synthesizer
  Notes   : Must be used AFTER I2C initialization ....
------------------------------------------------------------------------------}
function FlexCopDetectSynthesizerAddress(Handle: THandle): Byte;
var
  Address     : Byte;
  Success     : Boolean;
  ReturnedByte: Byte;
begin
  // Scan for correct synthesizer address
  // This is needed because different cards can set this address differently
  // (by hardware)
  Address := 0;
  repeat
    Success := FlexCopReadFromSynthesizer(Handle, Address, ReturnedByte);
    if not Success then
      Inc(Address);
  until Success or (Address > 3);
  if Success then
    Result := Address
  else
    Result := 255;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <SymbolRateKS>    Symbolrate to set (KSymbols/sec)
  Returns : <Result>          True for success

  Descript: Set symbol rate
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskSetSymbolRate(Handle: THandle; SymbolRateKS: Dword): Boolean;
var
  Data: Dword;
  AClc: Byte;
  BClc: Byte;
begin
  Result := False;

  // Check for excessive values
  if (SymbolRateKS * 1000) > CStv0299bMasterClock then
    Exit;

  // The symbolrate also requires additional settings
  AClc := $04;                                             // Alpha_car
  BClc := $11;                                             // Beta_car
  if SymbolRateKS < 30000 then
  begin
    AClc := $06;                                           // Alpha_car
    BClc := $13;                                           // Beta_car
  end;
  if SymbolRateKS < 14000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $13;                                           // Beta_car
  end;
  if SymbolRateKS < 7000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $0F;                                           // Beta_car
  end;
  if SymbolRateKS < 3000 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $0B;                                           // Beta_car
  end;
  if SymbolRateKS < 1500 then
  begin
    AClc := $07;                                           // Alpha_car
    BClc := $07;                                           // Beta_car
  end;
  AClc := AClc or $B0;                                     // Derotator on, 256 k symbols
  BClc := BClc or $40;                                     // QPSK algorithm 1
  if not FlexCopWriteToQpsk(Handle, CStv0299bAclc, AClc) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bBclc, BClc) then
    Exit;

  // Note: we need to write the MSB register before we write the Middle bit register
  Data := Round((SymbolRateKS * 1000) / CStv0299bSetSymbolrateUnits);
  // Write the values -> they are spread out over 5 nibbles (lowest byte has LSnibble in high part)
  Data := Data shl 4;
  if not FlexCopWriteToQpsk(Handle, CStv0299bSfrH, (Data shr 16) and $FF) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bSfrM, (Data shr 8) and $FF) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bSfrL, Data and $F0) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success
            <Percentage>      Lock percentage (0-100)

  Descript: Get lock percentage
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskGetLockDetectorPercentage(Handle: THandle; var Percentage: Byte): Boolean;
var
  Data: Byte;
  Temp: Integer;
begin
  Result := False;

  // Read lock value
  if not FlexCopReadFromQpsk(Handle, CStv0299bCldi, Data) then
    Exit;
  // Since the lock value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  // And to percentage
  Temp := Temp + 128;
  Temp := (Temp * 100) div 255;
  Percentage := Temp;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>             <FlexCopCreateFile> handle
  Returns : <Result>             True for success
            <StrengthPercentage> Signal strength in %
            <StrengthDecibel>    Signal strength in dB

  Descript: Get signal power
  Notes   : We read the value of the signal AGC. A low AGC setting indicates
            a low amplification and therefore a high input signal. A small
            input signal has to be amplifier more than a high input signal,
            resulting in a higher gain (= higher AGC value).
            Typically an amplification is given in dB which is based on a
            logarithmic scale. 'dB' itself indicates the relation of the
            output with reference to it's input. 'dB' by itself has no 'fixed'
            reference, as does for example 'dBmV' has ('dBmV' indicate that
            0 dB equals 1 mV).
            We have to determine our own reference, meaning what the input
            signal (range) is at the minimum AGC and at the maximum AGC.
            It turns out this is about 4. This means that when the AGC goes
            from it's minimum setting to it's maximum, it amplifies the
            input signal by 4.
            In dB this factor 4 is equivalent to 12 dB (a factor 1 equals 0 dB).
            So, if the AGC is at it's minimum value it means that the signal is
            12 dB stronger than when the AGC is at it's maximum
            (smaller AGC == less amplification == stronger input signal).
------------------------------------------------------------------------------}
function FlexCopQpskGetSignalPower(Handle: THandle;
  var StrengthPercentage: Double;
  var StrengthDecibel   : Double): Boolean;
var
  DataMSB    : Byte;
  DataLSB    : Byte;
  Data       : Integer;
begin
  Result := False;

  // Read values
  if not FlexCopReadFromQpsk(Handle, CStv0299bAgc2I1, DataMSB) then
    Exit;
  if not FlexCopReadFromQpsk(Handle, CStv0299bAgc2I2, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;

  // The lower the AGC, the stronger the signal, so '100% - AGC%'
  StrengthPercentage := 100 - ((Data * 100) / $FFFF);
  if StrengthPercentage <> 0 then
    StrengthDecibel := (Log10(4) * 20) + Log10(StrengthPercentage/100) * 20
  else
    StrengthDecibel := 0;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>               <Saa7146aCreateFile> handle
  Returns : <Result>               True if valid
            <InputLevelDbm>        Dbm level
            <InputLevelPercentage> Level in percentage

  Descript: Get signal level.
  Notes   : Approximation only. Could correct for different tuners but this
            is not done. Eg. BSRU6 tuner is more or less linearly running
            from -70 dBm (-100 AGC1I) to -20 dBm (+50 AGC1I). A BSBE1
            is not so linear and runs from -70 dBm (-100 AGC1I) to
            -20 dBm (+65 AGC1I).
------------------------------------------------------------------------------}

function FlexCopQpskGetInputLevel(Handle: THandle; var InputLevelDbm: Integer;
  var InputLevelPercentage: Byte): Boolean;
var
  Data: Byte;
  NewData: Integer;
  Calc: Integer;
begin
  Result := False;

  // Read values
  if not FlexCopReadFromQpsk(Handle, CStv0299bAgc1I, Data) then
    Exit;
  // Data is actually signed
  if Data > 128 then
    NewData := Data - 256
  else
    NewData := Data;
  // Convert to dBm range
  // -128..+128 -> -178..78
  // -100..50 to -70..-20 -> 150 range to 50 range
  // First -150..0
  NewData := NewData - 50;
  // -178..0
  if NewData > 0 then
    NewData := 0;
  Calc := 100 - Abs((NewData * 2) div 3);
  if Calc < 0 then
    Calc := 0;
  InputLevelPercentage := Calc;
  // Then factor 3: -50..0
  NewData := NewData div 3;
  // Then the -20
  NewData := NewData - 20;
  InputLevelDbm := NewData;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success
            <Noise>           Noise indicator

  Descript: Get noise indicator
  Notes   : The C/N is calculated with the following formula:
              C/N (dB) = -0.0017 * Nir + 19.02
            To check out:
            The sensitivity is dependent on the averaging period (SN) of the
            rate error detection and the pucture rate.
            It is also dependent on the AGC1 reference setting.
            Note that the result is only valid with a carrier
            and puncture rate found and known.
------------------------------------------------------------------------------}
function FlexCopQpskGetNoiseIndicator(Handle: THandle; var Noise: Double): Boolean;
var
  DataMSB: Byte;
  DataLSB: Byte;
  Data   : Word;
begin
  Result := False;

  // Read values
  if not FlexCopReadFromQpsk(Handle, CStv0299bNirH, DataMSB) then
    Exit;
  if not FlexCopReadFromQpsk(Handle, CStv0299bNirL, DataLSB) then
    Exit;
  Data := (DataMSB shl 8) or DataLSB;

  Noise := (-0.0017 * Data) + 19.02;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Inversion>       True when inversion requested
  Returns : <Result>          True for success

  Descript: Set inversion of AGC output
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskSetAGCInversion(Handle: THandle; Inversion: Boolean): Boolean;
var
  OriginalData: Byte;
  Data        : Byte;
begin
  Result := False;

  // Read original data
  if not FlexCopReadFromQpsk(Handle, CStv0299bAgc1R, OriginalData) then
    Exit;
  Data := OriginalData and not CStv0299bAGCInversion;
  if Inversion then
    Data := Data or CStv0299bAGCInversion;
  if Data <> OriginalData then
    if not FlexCopWriteToQpsk(Handle, CStv0299bAgc1R, Data) then
      Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
            <Inversion>       True when inversion requested
  Returns : <Result>          True for success

  Descript: Set inversion of I/Q inputs
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskSetInversion(Handle: THandle; Inversion: Boolean): Boolean;
var
  OriginalData: Byte;
  Data        : Byte;
begin
  Result := False;

  // Read original data
  if not FlexCopReadFromQpsk(Handle, CStv0299bIoCfg, OriginalData) then
    Exit;
  Data := OriginalData and not CStv0299bInversion;
  if Inversion then
    Data := Data or CStv0299bInversion;
  if Data <> OriginalData then
    if not FlexCopWriteToQpsk(Handle, CStv0299bIoCfg, Data) then
      Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Frequency deviation (in kHz)

  Descript: Get deviation from carrier
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskGetCarrierDeviation(Handle: THandle; var DeviationKHz: Integer): Boolean;
var
  DataM: Byte;
  DataL: Byte;
  Temp: Int64;
begin
  Result := False;

  // Read derotator values
  if not FlexCopReadFromQpsk(Handle, CStv0299bCfrM, DataM) then
    Exit;
  if not FlexCopReadFromQpsk(Handle, CStv0299bCfrL, DataL) then
    Exit;
  Temp := (DataM shl 8) or DataL;
  // Since its a signed number we convert it
  if (DataM and $80) <> 0 then
    Temp := Temp - 65536;
  Temp := Temp * (CStv0299bMasterClock div 1000);
  // And to percentage
  DeviationKHz := (Temp div 65536);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success
            <DeviationKHz>    Symbol rate deviation in kHz

  Descript: Get deviation
  Notes   :
------------------------------------------------------------------------------}
function FlexCopQpskGetSymbolRateDeviation(Handle: THandle; var DeviationKHz: Integer): Boolean;
var
  Data: Byte;
  Temp: Integer;
  Calc: Extended;
begin
  Result := False;

  // Read timing frequency (deviation)
  if not FlexCopReadFromQpsk(Handle, CStv0299bRtf, Data) then
    Exit;

  // Since the value is a signed number we convert it
  if (Data and $80) <> 0 then
    Temp := Data - 256
  else
    Temp := Data;
  Calc := Temp;
  Calc := Calc * CStv0299bGetSymbolrateUnits;
  Calc := Calc / 1000;
  // Additional factor so what we return can be used directly for the symbol rate
  Calc := Calc / 2;
  DeviationKHz := Round(Calc);
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set low band of LNB
  Notes   : The LNB band is controlled by the absence of a 22kHz signal.
------------------------------------------------------------------------------}
function FlexCopLowBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulation0;
  // Write the new value
  if not FlexCopWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>          <FlexCopCreateFile> handle
  Returns : <Result>          True for success

  Descript: Set high band of LNB
  Notes   : The LNB band is controlled by the presence of a 22kHz signal.
------------------------------------------------------------------------------}
function FlexCopHighBandLNB(Handle: THandle): Boolean;
var
  Data: Byte;
begin
  Result := False;

  // First read original value
  if not FlexCopReadFromQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;
  Data := Data and not CStv0299bModulationMask;
  Data := Data or CStv0299bModulationContinuous;
  // Write the new value
  if not FlexCopWriteToQpsk(Handle, CStv0299bDiSEqC, Data) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>                     <FlexCopCreateFile> handle
            <Frequency>                  Frequency to set (demodulated, in kHz)
            <LowBandLocalOscillatorKHz>  Local oscillator frequency of low band
            <HighBandLocalOscillatorKHz> Local oscillator frequency of high band
            <TunerType>                  Type of tuner
  Returns : <Result>                     True  for low  band
                                         False for high band

  Descript: Check if low band to use
------------------------------------------------------------------------------}

function FlexCopUseLowBand(Handle: THandle; FrequencyKHz: Dword;
  LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword;
    TunerType: Byte): Boolean;
const
  LowestTunerFrequency = 950000;
    // Lowest  frequency tuner kHz (for all tuners sofar)
  HighestTunerFrequency = 2150000; // Highest frequency tuner kHz
var
  ActualFrequency: Integer;
begin
  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  Result := True;
  // Note: The <Abs> allows for negative results (eg. C-band)
  ActualFrequency := FrequencyKHz;
  ActualFrequency := Abs(Integer(ActualFrequency) -
    Integer(LowBandLocalOscillatorKHz));
  if ActualFrequency > HighestTunerFrequency then
    Result := False;
end;


{------------------------------------------------------------------------------
  Params  : <Handle>                     <FlexCopCreateFile> handle
            <Frequency>                  Frequency to set (demodulated, in kHz)
            <LowBandLocalOscillatorKHz>  Local oscillator frequency of low band
            <HighBandLocalOscillatorKHz> Local oscillator frequency of high band
            <TunerType>                  Type of tuner
            <AddressIndex>               Address index of synthesizer (0..3)
                                         TBMU24112 (SkyStar2) : 1
  Returns : <Result>                     True for success

  Descript: Set frequency of LNB
  Notes   : The frequency is controlled by the SL1935 synthesizer which
            is controlled by the I2C (on the repeater of the QPSK demodulator!).
            The <LocalOscillator> is used to calculate the actual frequency for
            the synthesizer.
            Typical 'lock' time is less than ? ms but may be ?? ms.
------------------------------------------------------------------------------}
function FlexCopFrequencyLNB(Handle: THandle; FrequencyKHz: Dword;
  LowBandLocalOscillatorKHz: Dword; HighBandLocalOscillatorKHz: Dword;
  TunerType: Byte; AddressIndex: Byte): Boolean;
const
  Reference = 4000000;
  // Note: The SL1935 supports more ratio's...
  Ratios : array[0..15] of Dword =
    (Reference div 2,  Reference div 4,   Reference div 8,   Reference div 16,
     Reference div 32, Reference div 64,  Reference div 128, Reference div 256,
     0,
     Reference div 5,  Reference div 10,  Reference div 20,  Reference div 40,
     Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency  =  950000000;                      // Lowest frequency tuner
  HighestTunerFrequency = 2150000000;                      // Highest frequency tuner
var
  Loop                 : Byte;
  ActualFrequency      : Int64;
//  SmallestError        : Dword;
//  CalcError            : Int64;
  RatioForSmallestError: Word;
  CalcDivider          : Int64;
  ProgrammableDivider1 : Byte;
  ProgrammableDivider2 : Byte;
  ControlData1         : Byte;
  ControlData2         : Byte;
  SynthesizerData      : array[0..3] of Byte;
  LowBand              : Boolean;
begin
  Result := False;

  // The actual frequency to set is the requested frequency (demodulated!)
  // with the local oscillator (modulation frequency) of the LNB which is
  // either the low band or the high band.
  LowBand := True;
  // Note: The <Abs> allows for negative results (eg. C-band)
  ActualFrequency := FrequencyKHz;
  ActualFrequency := Abs(ActualFrequency - LowBandLocalOscillatorKHz);
  ActualFrequency := ActualFrequency * 1000;
  if ActualFrequency > HighestTunerFrequency then
  begin
    LowBand := False;
    // Note: The <Abs> allows for negative results (eg. C-band)
    ActualFrequency := FrequencyKHz;
    ActualFrequency := Abs(ActualFrequency - HighBandLocalOscillatorKHz);
    ActualFrequency := Int64(ActualFrequency) * 1000;
  end;
  if (ActualFrequency > HighestTunerFrequency)  or
     (ActualFrequency < LowestTunerFrequency) then
    Exit;
  // Now the band is know we have to set this.
  if LowBand then
  begin
    if not FlexCopLowBandLNB(Handle) then
      Exit;
  end
  else
    if not FlexCopHighBandLNB(Handle) then
      Exit;
  // Fixed setting
  RatioForSmallestError := 4;

  // With the divider information available we can construct the 4 bytes we
  // need to send to the synthesizer. The DIVIDER is divided over 2 bytes.
  CalcDivider := ActualFrequency div Ratios[RatioForSmallestError];
  ProgrammableDivider2 := CalcDivider and $FF;
  CalcDivider := CalcDivider shr 8;
  ProgrammableDivider1 := CalcDivider and $7F;

  ControlData1 := CSl1935DefaultControlRegister1 or RatioForSmallestError;
  ControlData2 := CSl1935DefaultControlRegister2;
  // Depending on the frequency range we need to set the oscillator
  // 950-1500  or 1500-2150
  if ActualFrequency >= 1500000000 then
    ControlData2 := ControlData2 or CSl1935SetOsc1500_2150
  else
    ControlData2 := ControlData2 or CSl1935SetOsc950_1500;

  SynthesizerData[0] := ProgrammableDivider1;
  SynthesizerData[1] := ProgrammableDivider2;
  SynthesizerData[2] := ControlData1;
  SynthesizerData[3] := ControlData2;
  // Now we can send the data to the synthesizer
  if not FlexCopWriteToSynthesizer(Handle, AddressIndex, SynthesizerData) then
    Exit;
  // Because the STV0299 might have to lock to this (new) frequency we have
  // to reset the de-rotator loop
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfrM, $00) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfrL, $00) then
    Exit;
  if not FlexCopReadFromQpsk(Handle, CStv0299bCfrL, Loop) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfd, $B9) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>                     <FlexCopCreateFile> handle
            <Frequency>                  Frequency to set (modulated, in kHz)
            <TunerType>                  Type of tuner
            <AddressIndex>               Address index of synthesizer (0..3)
                                         TBMU24112 (SkyStar2) : 1
  Returns : <Result>                     True for success

  Descript: Set frequency of LNB. Frequency is already in tuner format.
            Assumes low/high band is handled elsewhere.
  Notes   : The frequency is controlled by the SL1935 synthesizer which
            is controlled by the I2C (on the repeater of the QPSK demodulator!).
            The <LocalOscillator> is used to calculate the actual frequency for
            the synthesizer.
            Typical 'lock' time is less than 3 ms but may be 10 ms.
------------------------------------------------------------------------------}

function FlexCopFrequencyTunerLNB(Handle: THandle; FrequencyKHz: Dword;
  TunerType: Byte; AddressIndex: Byte): Boolean;
const
  Reference = 4000000;
  // Note: The SL1935 supports more ratio's...
  Ratios : array[0..15] of Dword =
    (Reference div 2,  Reference div 4,   Reference div 8,   Reference div 16,
     Reference div 32, Reference div 64,  Reference div 128, Reference div 256,
     0,
     Reference div 5,  Reference div 10,  Reference div 20,  Reference div 40,
     Reference div 80, Reference div 160, Reference div 320);
  LowestTunerFrequency  =  950000000;                      // Lowest frequency tuner
  HighestTunerFrequency = 2150000000;                      // Highest frequency tuner
var
  Loop: Byte;
  ActualFrequency: Int64;
  SmallestError: Dword;
  CalcError: Int64;
  RatioForSmallestError: Word;
  CalcDivider: Int64;
  ProgrammableDivider1: Byte;
  ProgrammableDivider2: Byte;
  ControlData1: Byte;
  ControlData2: Byte;
  SynthesizerData      : array[0..3] of Byte;
begin
  Result := False;

  ActualFrequency := FrequencyKHz;
  ActualFrequency := ActualFrequency * 1000;
  // To set the correct frequency we have to set two different values: the
  // 'divider' and the 'ratio'. The actual calculations involved are:
  // FREQUENCY = DIVIDER * (4 MHz / RATIO)
  // Since we have to calculate the DIVIDER and RATIO from the FREQUENCY we have to
  // do some calculation, mostly to get an as accurate result.
  // RATIO    = 2,       4,       8,      16,      32,     64,    128,   256,
  // 4M/RATIO = 2000000, 1000000, 500000, 250000,  125000, 62500, 31250, 15625
  // RATIO    = 5,       10,      20,     40,      80,     160,   320
  // 4M/RATIO = 800000,  400000,  200000, 1000000, 50000,  25000, 12500
  // DIVIDER = 0..$1FFFF (0..131071)
  // At this point we could do some calculations and such to get the best result,
  // but the simplest method is just to check all possibilities and choose the
  // one with the smallest error. Alternatively we could choose a fixed ratio which
  // will limit the step size. For now we will use the one with the smallest error.

//  RatioForSmallestError := High(Ratios);
  SmallestError := ActualFrequency;
  for Loop := Low(Ratios) to High(Ratios) do
  begin
    if Ratios[Loop] <> 0 then
    begin
      CalcDivider := ActualFrequency div Ratios[Loop];
      // Must fit in 17 bits
      if CalcDivider <= $1FFFF then
      begin
        CalcError := ActualFrequency - (CalcDivider * Ratios[Loop]);
        if CalcError < SmallestError then
        begin
          SmallestError := CalcError;
//          RatioForSmallestError := Loop;
        end;
      end;
    end;
    // If the error is zero then there is no need to continu checking....
    if SmallestError = 0 then
      Break;
  end;

  // Fixed setting
  RatioForSmallestError := 4;

  // With the divider information available we can construct the 4 bytes we
  // need to send to the synthesizer. The DIVIDER is divided over 2 bytes.
  CalcDivider := ActualFrequency div Ratios[RatioForSmallestError];
  ProgrammableDivider2 := CalcDivider and $FF;
  CalcDivider := CalcDivider shr 8;
  ProgrammableDivider1 := CalcDivider and $7F;

  ControlData1 := CSl1935DefaultControlRegister1 or RatioForSmallestError;
  ControlData2 := CSl1935DefaultControlRegister2;
  // Depending on the frequency range we need to set the oscillator
  // 950-1500  or 1500-2150
  if ActualFrequency >= 1500000000 then
    ControlData2 := ControlData2 or CSl1935SetOsc1500_2150
  else
    ControlData2 := ControlData2 or CSl1935SetOsc950_1500;

  SynthesizerData[0] := ProgrammableDivider1;
  SynthesizerData[1] := ProgrammableDivider2;
  SynthesizerData[2] := ControlData1;
  SynthesizerData[3] := ControlData2;
  // Now we can send the data to the synthesizer
  if not FlexCopWriteToSynthesizer(Handle, AddressIndex, SynthesizerData) then
    Exit;
  // Because the STV0299 might have to lock to this (new) frequency we have
  // to reset the de-rotator loop
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfrM, $00) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfrL, $00) then
    Exit;
  if not FlexCopReadFromQpsk(Handle, CStv0299bCfrL, Loop) then
    Exit;
  if not FlexCopWriteToQpsk(Handle, CStv0299bCfd, $B9) then
    Exit;

  Result := True;
end;


initialization
  HighPerformanceAvailable := QueryPerformanceFrequency(HighPerformanceFrequency);
  I2cLock := TCriticalSection.Create;

finalization
  I2cLock.Free;
end.

