{******************************************************************************}
{ FileName............: FlexCopDvbStreamBuffering                              }
{ Project.............: FLEXCOP                                                }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.00                                                   }
{------------------------------------------------------------------------------}
{   Buffering of stream data.                                                  }
{                                                                              }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Includes master / slave principle for multiple instances which will use the  }
{ same packet data.                                                            }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20050413 - Initial release                                           }
{******************************************************************************}
unit FlexCopDvbStreamBuffering;

interface
uses
  Windows;

type
  PBoolean = ^Boolean;

function FlexCopDvbSetupBuffering(const CardHandle: THandle; const Initialize: Boolean;
  const TunerType: Byte; const Reserved1: Byte;
  IsSlave: PBoolean; BufferTime: PWord; IrqId: PDword; Reserved2: Integer):
    Boolean; stdcall;

implementation
uses
  FlexCopI2c,
  FlexCopInterface,
  FlexCopIoControl,
  FlexCopGeneral,
  FlexCopRegisters,
  Stv0299bRegisters;

const
  CBufferPackets   = 512;
  CPacketSize      = 188;
  CBufferSize      = CBufferPackets * CPacketSize;
  MaxPacketBuffers = 250; // Absolute maximum number of buffers
  // Larger values are obtainable but time needed
  // by operating system becomes longer (needs to be
  // consecutive piece of memory!)
  // A value of 340 is about 64 Mbytes!!

  {------------------------------------------------------------------------------
    Params  : <CardHandle>  Handle to driver
              <Reserved>
    Returns : <Result>      True if success

    Descript: Do a (re)initialization of the card
    Notes   :
   ------------------------------------------------------------------------------}

function InitializeCard(CardHandle: THandle; TunerType: Byte; Reserved: Byte):
  Boolean;
var
  ReturnedByte: Byte;
  Dummy       : Dword;
begin
  Result := False;

  // Clear pending interrupt(s)
  if not FlexCopReadFromFlexCopRegister(CardHandle, CFlexCopEnable, Dummy) then
    Exit;
  // Disable all interrupts
  if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0) then
    Exit;

  // Pseudo reset
  if not FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopSpecials, CFlexCopSpecialReset) then
    Exit;
  Sleep(1);
  if not FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, 0) then
    Exit;
  Sleep(1);
  if not FlexCopWriteToFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, CFlexCopBoardReset) then
    Exit;
  Sleep(1);

  // Wake up
  if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopPowerAndRevision, CFlexCopTunerPowerOn or
    CFlexCopLNBPolarityHorizontal or CFlexCopSystemPowerOn or CFlexCopWakeUpOn) then
    Exit;
  if not FlexCopReadFromFlexCopRegister(Cardhandle, CFlexCopPowerAndRevision, Dummy) then
    Exit;

  // Check QPSK
  if not FlexCopReadFromQpsk(CardHandle, CStv0299bId, ReturnedByte) then
    Exit;
  // Check identifier of QPSK
  if not ReturnedByte = $A1 then
    Exit;
  if not FlexCopWriteDefaultsToQpsk(CardHandle, 0) then
    Exit;
  FlexCopRamDetectSize(CardHandle);
  if not FlexCopSelect(CardHandle, CFlexCopMemorySelectMedia, CFlexCopMemorySelect1) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
  Returns : <Result>      $FFFF if no master
                          Otherwise Interrupt identifier

  Descript: Detect a POSSIBLE master (interrupts for DMA, subbuffer 0 are
            then enabled)
  Notes   :
 ------------------------------------------------------------------------------}

function GetMasterChannel(CardHandle: THandle): Dword;
var
  IrqHandling: TFlexCopIrqTransferBuffer;
begin
  IrqHandling.Identifier := CFlexCopIrqDma10Irq;
  FlexCopReadFromIrqHandling(CardHandle, IrqHandling);
  if not IrqHandling.Information.IrqBufferingIsActive then
    Result := $FFFF
  else
    Result := CFlexCopIrqDma10Irq;
end;

{------------------------------------------------------------------------------
  Params  : <CardHandle>    Handle to driver
            <DmaChannel2>   False for DMA1
                            True  for DMA2
            <ActivateDma>   True if DMA also to be activated (in address)
            <Packets>       Number of packets to use for size
            <SubBuffer0>    Pointer to physical address DMA subbuffer 0
            <SubBuffer1>    Pointer to physical address DMA subbuffer 0
            <PacketIrqs>    True if packet interrupts are to be enabled
  Returns : <Result>        True if success

  Descript: Setup FlexCop registers
  Notes   :
 ------------------------------------------------------------------------------}
function SetupFlexCopAddress(CardHandle: THandle; DmaChannel2: Boolean; ActivateDma: Boolean;
           Packets: Word; SubBuffer0: Cardinal; SubBuffer1: Cardinal; PacketIrqs: Boolean): Boolean;
var
  Dma        : Dword;
  Address    : Dword;
  PacketSized: Integer;
  SetSize    : Dword;
begin
  Result      := False;
  PacketSized := Packets * CPacketSize;

  if ActivateDma then
    Dma := CFlexCopSubBufferEnableIrq
  else
    Dma := 0;
  // Fortunately our allocated memory is always dword aligned ....
  // Buffer 1 address
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer0Address
  else
    Address := CFlexCopDma1SubBuffer0Address;
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer0 or Dma) then
    Exit;

  // Sub buffer 1 is either enabled or disabled (nothing to do with actual DMA enabling)
  if ActivateDma then
    Dma := CFlexCopSubBufferEnableSubBuffer1
  else
    Dma := 0;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBuffer1Address
  else
    Address := CFlexCopDma1SubBuffer1Address;
  // Buffer 2 address
  if not FlexCopWriteToFlexCopRegister(CardHandle,
           Address, SubBuffer1 or Dma) then
    Exit;

  // Buffer size
  // We have allocated memory which is a multiple of the DMA block size.
  // For convenience we adjust this to the even packet size.
  SetSize := PacketSized;
  SetSize := SetSize shr 2;                              // Size if 4-byte chunks
  SetSize := SetSize shl 8;                              // Shift size
  if not PacketIrqs then
    SetSize := SetSize or CFlexCopSubBufferDisablePacketIrq;
  if DmaChannel2 then
    Address := CFlexCopDma2SubBufferSize
  else
    Address := CFlexCopDma1SubBufferSize;
  if not FlexCopWriteToFlexCopRegister(CardHandle, Address, SetSize) then
    Exit;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <CardHandle>     Handle to driver
            <IsSlave>        True if setup as slave
            <PacketBuffers>  Number of buffers to allocate
            <Reserved>
  Returns : <Result>         True if success
            <PacketBuffers>  Actual buffers allocated
            <IrqId>          IRQ identifier

  Descript: Setup buffering of DMA3 data.
  Notes   :
 ------------------------------------------------------------------------------}

function BufferingSetup(CardHandle: THandle; IsSlave: Boolean; var
  PacketBuffers: Word; var IrqId: Dword; Reserved: Integer): Boolean;
var
  Channel    : Dword;
  Buffer1    : TFlexCopDmaBuffer;
  Buffer2    : TFlexCopDmaBuffer;
  Buffer3    : TFlexCopDmaBuffer;
  Buffer4    : TFlexCopDmaBuffer;
  FifoBuffer : TFlexCopFifoTransferBuffer;
  IrqHandling: TFlexCopIrqTransferBuffer;
  Succeeded  : Boolean;
begin
  Result := False;
  // Note: If we are acting as a slave then we don't setup anything
  if CardHandle = INVALID_HANDLE_VALUE then
    Exit;

  if IsSlave then
  begin
    // Get the 'channel' (=identifier) of the 'master'.
    Channel := GetMasterChannel(CardHandle);
    IrqId := Channel;
    if Channel = $FFFF then
      Exit;     // The driver is used as slave but could not detect an active master

    // Get fifo buffer index for number of buffers
    IrqHandling.Identifier := Channel;
    if not FlexCopReadFromIrqHandling(CardHandle, IrqHandling) then
      Exit;
    PacketBuffers := IrqHandling.Information.FifoBufferLastIndex -
      IrqHandling.Information.FifoBufferFirstIndex + 1;
  end
  else
  begin
    // Setup DMA
    //
    // We need to setup several buffers.
    // Buffer 1:  FlexCop target buffer for satellite data (subbuffer 0) first DMA
    // Buffer 2:  FlexCop target buffer for satellite data (subbuffer 1) first DMA
    // Buffer 3:  Dummy buffer for second DMA
    // Buffer 4:  Dummy buffer for second DMA
    //
    // The FlexCop issues an interrupt when one of it's buffers is filled (in our
    // case this buffer is buffer 1 or 2). The driver (if properly set up) then
    // copies this data to somewere in buffer 3.
    // Typically the FlexCop issues two interrupts. One when it's first subbuffer
    // has been filled, and another when it's second subbuffer has been filled.
    // Using the 'sharing' mechanism of the FIFO we can make sure that all data
    // is placed on our common buffer 3.
    //
    // We are dealing with DMA 1 only here (DMA2 is not used). To recap, the
    // following registers/settings are to be set up:
    // DMA1 - generates two distinguishes interrupts: One for subbuffer 0, and
    //        another for subbuffer 1
    //        CFlexCopIrqDma10                : DMA1, subbuffer 0 interrupt
    //        CFlexCopIrqDma11                : DMA1, subbuffer 1 interrupt
    //        CFlexCopEnableDma10             : DMA1, subbuffer 0 interrupt enable
    //        CFlexCopEnableDma11             : DMA1, subbuffer 1 interrupt enable
    //        CFlexCopDma1SubBuffer0Address   : DMA1, subbuffer 0 target address
    //        CFlexCopDma1SubBufferSize       : DMA1, subbuffer size
    //        CFlexCopDma1SubBuffer1Address   : DMA1, subbuffer 1 target address

    // First disable all possible interrupts and such
    if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable, 0) then
      Exit;

    // Release any memory already allocated
    Channel := 0;
    while FlexCopReleaseDma(CardHandle, Channel) do
      Inc(Channel);
    // Release all FIFO's
    FifoBuffer.Identifier := -1;
    FlexCopReleaseFifo(CardHandle, FifoBuffer);
    with IrqHandling.Information do
    begin
      Irqs                 := 0;
      IrqsWhenActive       := 0;
      IrqBufferingIsActive := False;
      UseNotificationEvent := False;
      UseSignaling         := False;
      UseFifo              := False;
      FifoBufferPreviousIndex := 0;
      FifoBufferFirstIndex := 0;
      FifoBufferLastIndex  := 0;
      FifoOverflows        := 0;
    end;
    IrqHandling.Identifier := CFlexCopIrqDma10Irq;
    if not FlexCopWriteToIrqHandling(CardHandle, IrqHandling) then
      Exit;
    IrqHandling.Identifier := CFlexCopIrqDma11Irq;
    if not FlexCopWriteToIrqHandling(CardHandle, IrqHandling) then
      Exit;

    // Try allocating the required buffer size
    // Allocate DMA buffers which will hold the FIFO data
    repeat
      // Allocate buffers
      Succeeded := False;
      if FlexCopAllocateDma(CardHandle, (((CBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer1) then
        Succeeded := FlexCopAllocateDma(CardHandle, (((CBufferSize * PacketBuffers) div DMASIZE) + 1) * DMASIZE, Buffer2);
      // If we did not succeeded, so try using less memory. Strip down by 0.5/0.1 second at a time.
      if not (Succeeded) then
      begin
        Channel := 0;
        while FlexCopReleaseDma(CardHandle, Channel) do
          Inc(Channel);
        if PacketBuffers > 49 then
          Dec(PacketBuffers, 25)
        else if PacketBuffers > 19 then
          Dec(PacketBuffers, 5);
      end;
    until Succeeded or (PacketBuffers < 20);
    if not Succeeded then
      Exit;
    // Dummy buffers so we have at least a valid buffer, which we use for DMA2
    // Allocate buffer 3
    if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer3) then
      Exit;
    // Allocate buffer 4
    if not FlexCopAllocateDma(CardHandle, DMASIZE, Buffer4) then
      Exit;

    // Setup initial addresses/size
    if not SetupFlexCopAddress(CardHandle, False, False, CBufferPackets,
      Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, False) then
      Exit;
    // Also do this for DMA2 although we don't use DMA2
    // (just to make sure DMA addresses are valid)
    if not SetupFlexCopAddress(CardHandle, True, False, 1,
      Buffer3.PhysicalAddress.LowPart, Buffer4.PhysicalAddress.LowPart, False) then
      Exit;

    // Setup Fifo. This allocates the number of buffers we want.
    with FifoBuffer do
    begin
      NumberOfBuffers := PacketBuffers;
      // Make sure all unused items are reset
      for Channel := Low(TransferAddress) to High(TransferAddress) do
        TransferAddress[Channel] := nil;
      // Note: The VIRTUAL address !
      // Fill in for subbuffer 0 and 1
      TransferAddress[CFlexCopIrqDma10Irq] := Pointer(Buffer1.VirtualAddress);
      TransferAddress[CFlexCopIrqDma11Irq] := Pointer(Buffer2.VirtualAddress);
      TransferLength := CBufferSize;
    end;
    if not FlexCopAllocateFifo(CardHandle, FifoBuffer) then
      Exit;

    // At this point all buffers has been setup (allocated):
    //  Buffer 1 and 2 (DMA mechanism, so contiguous memory)

    // Enable interrupt mechanism of the driver
    // When this is set the actual buffering of data (to the FIFO) is activated
    // Setup for subbuffer 0 (DMA1 - subbuffer 0)
    with IrqHandling.Information do
    begin
      Irqs                 := 0;
      IrqsWhenActive       := 0;
      IrqBufferingIsActive := True;
      UseNotificationEvent := True;
      UseSignaling         := False;
      UseFifo              := True;
      FifoBufferPreviousIndex := 0;
      FifoBufferFirstIndex := FifoBuffer.Identifier;
      FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
      FifoOverflows        := 0;
    end;
    IrqHandling.Identifier := CFlexCopIrqDma10Irq;
    IrqId := IrqHandling.Identifier;
    if not FlexCopWriteToIrqHandling(CardHandle, IrqHandling) then
      Exit;

    // Setup for subbuffer 1 (DMA1 - subbuffer 1)
    // Subbuffer 1 uses the same fifo buffers as subbuffer 0
    with IrqHandling.Information do
    begin
      Irqs                 := 0;
      IrqsWhenActive       := 0;
      IrqBufferingIsActive := True;
      UseNotificationEvent := True;
      UseSignaling         := False;
      UseFifo              := True;
      FifoBufferPreviousIndex := 0;
      FifoBufferFirstIndex := FifoBuffer.Identifier;
      FifoBufferLastIndex  := FifoBufferFirstIndex + FifoBuffer.NumberOfBuffers-1;
      FifoOverflows        := 0;
    end;
    IrqHandling.Identifier := CFlexCopIrqDma11Irq;
    if not FlexCopWriteToIrqHandling(CardHandle, IrqHandling) then
      Exit;

    // Now we have to enable DMA of the FlexCop. We already have setup the address
    // and such.

    // Enable subbuffer 0/1 DMA
    // Now we have to enable the reception of data
    // First the DMA bits in the address of the subbuffers
    if not SetupFlexCopAddress(CardHandle, False, True, CBufferPackets,
      Buffer1.PhysicalAddress.LowPart, Buffer2.PhysicalAddress.LowPart, False) then
      Exit;

    // Enable reception of data
    if not FlexCopWriteToFlexCopRegister(CardHandle, CFlexCopEnable,
      CFlexCopEnableDma10 or CFlexCopEnableDma11 or CFlexCopEnableReceive) then
    Exit;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <CardHandle>  Handle to driver
            <Initialize>  True if initialization allowed
            <TunerType>   Type of tuner to address
            <Reserved1>
            <IsSlave>     False if forcing master mode
                          True for auto detection of master/slave
            <BufferTime>  Time (in ms) for buffer
            <Reserved2>
  Returns : <Result>      True if success
                          False if not: most likely incorrect tuner type
            <IsSlave>     True indicates slave mode has been set
                          False indicates master mode has been set
            <BufferTime>  Number of buffers actually used
            <IrqId>       IRQ identifier used for storing data

  Descript: Setup data throughput of driver (packet data)
  Notes   : This will setup the SAA7146A so data from the tuner is passed through
            the DD1 port. The DD1 port has the unprocessed data.
            Includes detection of master/slave.
 ------------------------------------------------------------------------------}

function FlexCopDvbSetupBuffering(const CardHandle: THandle; const Initialize: Boolean;
  const TunerType: Byte; const Reserved1: Byte;
  IsSlave: PBoolean; BufferTime: PWord; IrqId: PDword; Reserved2: Integer):
    Boolean; stdcall;
begin
  try
    Result := False;
    if not Assigned(IsSlave) then
      Exit;
    if not Assigned(BufferTime) then
      Exit;
    if not Assigned(IrqId) then
      Exit;

    // Force master mode if slave requested but no master present
    if IsSlave^ then
      if GetMasterChannel(CardHandle) = $FFFF then
        IsSlave^ := False
      else
        IsSlave^ := True;

    if not IsSlave^ then
      if Initialize then
        if not InitializeCard(CardHandle, TunerType, Reserved1) then
          Exit;

    // Number of buffers is approximated using 20 ms per buffer as reference
    BufferTime^ := BufferTime^ div 20;
    if BufferTime^ > MaxPacketBuffers then
      BufferTime^ := MaxPacketBuffers;
    if BufferTime^ < 2 then
      BufferTime^ := 2;

    // Setup buffering/addresses
    if not BufferingSetup(CardHandle, IsSlave^, BufferTime^, IrqId^,
      Reserved2) then
      Exit;

    Result := True;
  except
    Result := False;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}

procedure Initialize;
begin
  //
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}

procedure Finalize;
begin
  //
end;

initialization
  Initialize;

finalization
  Finalize;
end.

