{******************************************************************************}
{ FileName............: DvbFilterXml                                           }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.01                                                   }
{------------------------------------------------------------------------------}
{  Filtering of data streams (XML supporting functions) of stored XML data     }
{  most of which is specific to the used XML parser                            }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20051008 - Initial release                                           }
{  1.01   20051126 - name changes of called DvbFilter functions                }
{******************************************************************************}
unit DvbFilterXmlJan;

interface
uses
  ComCtrls,
  DvbFilterXml,
  DvbFilter,   // Only for types
  Classes,
  JanXmlParser2,
  SysUtils,
  Forms,
  Windows;


const
  CDvbEventTypeUndefined        = $00;
  CDvbEventTypeShortName        = $01;
  CDvbEventTypeShortText        = $02;
  CDvbEventTypeExtendedItemName = $03;
  CDvbEventTypeExtendedItemText = $04;
  CDvbEventTypeExtendedText     = $05;


procedure DvbXmlSaveData(Service: Integer; TransponderName: AnsiString);
procedure DvbXmlGetActiveSectionNames(SectionList: TStringList);
procedure DvbXmlCreateSectionTree(TreeView: TTreeView; Range: LongInt; TransponderName: AnsiString);

function DvbXmlGetUtcTime(Section: Byte): SYSTEMTIME;
function DvbXmlGetIndexedProgramInfo(Index: Integer; var ServiceId: Word; var PmtPid: Word; var ServiceName: AnsiString): Boolean;
function DvbXmlGetServiceName(ServiceId: Integer): AnsiString;
function DvbXmlGetEmmPids(var EmmPids: TDvbPids; var CaIds: TDvbPids): Boolean;
function DvbXmlGetNumberOfprograms: Integer;
function DvbXmlGetProgramInfo(ServiceId: Word; var PmtPid: Word; var PcrPid: Word;
  var VideoPids        : TDvbPids;
  var AudioPids        : TDvbPids;
  var TeletextPids     : TDvbPids;
  var SubtitlePids     : TDvbPids;
  var AudioLanguages   : TDvbLanguages;
  var TeletextLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids          : TDvbPids;
  var CaIds            : TDvbPids;
  var ServiceName      : AnsiString): Boolean;
function DvbXmlGetEventInfo(ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;


implementation


{------------------------------------------------------------------------------
  Params  : <ServiceId>     Service identifier
            <Present>       True if present information only
                            False if all information is requested
            <Events>        Event information storage object
  Returns : <Result>        True if valid
            <Events>        Event information

  Descript: Get event information of program
  Notes   : We assume we have enough room for each single event
 ------------------------------------------------------------------------------}

function DvbXmlGetEventInfo(ServiceId: Word; Present: Boolean; Events: TDvbEventSections): Boolean;
var
  Year              : Word;
  Month             : Word;
  Day               : Word;
  Hours             : Word;
  Minutes           : Word;
  Seconds           : Word;
  MSeconds          : Word;
  XmlParser         : TJanXmlParser2;
  ANode             : TjanXMLNode2;
  NodeListStart     : TList;
  NodeListDuration  : TList;
  NodeListDescriptor: TList;
  NodeListEvent     : TList;
  NodeListItem      : TList;
  SectionIndex      : Byte;
  LoopMain          : Integer;
  LoopDescriptor    : Integer;
  LoopEvent         : Integer;
  LoopItem          : Integer;
  Valid             : Boolean;         // True if valid section index
  Last              : Boolean;         // True if last section index processing
  SectionNumbers    : Boolean;         // True if individual section numbers to retrieve
  SectionNumber     : Integer;         // Section number to process
  NumberFrom        : Integer;         // First section number
  NumberTo          : Integer;         // Last section number
  EventIndex        : Integer;
  PString           : PAnsiString;
  TheTime           : Int64;
  Error             : Integer;
  XmlString         : AnsiString;
begin
  Result := False;
  if not Assigned(Events) then
    Exit;
  SectionIndex := CDvbSectionActualTransportStreamPresentFollowingEventInformation;
  Last         := False;
  EventIndex   := 0;

  XmlParser          := TJanXmlParser2.Create;
  NodelistStart      := TList.Create;
  NodelistDuration   := TList.Create;
  NodelistDescriptor := TList.Create;
  NodelistEvent      := TList.Create;
  NodelistItem       := TList.Create;
  try
    while not Last do
    begin
      case SectionIndex of
        CDvbSectionActualTransportStreamPresentFollowingEventInformation:
          begin
            SectionNumbers := False;
            Valid := True;
            if Present then
              Last := True
            else
              Last := False;
          end;
        CDvbSectionActualTransportStreamScheduleEventInformation0..CDvbSectionActualTransportStreamScheduleEventInformationE:
          begin
            // Since scheduled events can contain a lot of data, we process it
            // section number by section number
            SectionNumbers := True;
            Valid := True;
            Last  := False;
          end;
        CDvbSectionActualTransportStreamScheduleEventInformationF:
          begin
            SectionNumbers := False;
            Valid := True;
            Last  := True;
          end;
        else
          begin
            SectionNumbers := False;
            Valid := False;
            Last  := False;
          end;
      end;
      if Valid then
      begin
        if SectionNumbers then
        begin
          NumberFrom := 0;
          NumberTo   := 255;
        end
        else
        begin
          NumberFrom := -1;
          NumberTo   := -1;
        end;
        for SectionNumber := NumberFrom to NumberTo do
        begin
          // Get section data
          XmlString :=  DvbFilterXmlGetXmlInfoFromSection(-ServiceId, SectionIndex, SectionNumber, '  ');
          if Length(XmlString) > 0 then
          begin
            XmlParser.Xml := Xmlstring;
            // Get event items
            NodeListStart.Clear;
            XmlParser.SelectNodes(XmlParser, NodeListStart, 'name=''start_time''');
            NodeListDuration.Clear;
            XmlParser.SelectNodes(XmlParser, NodeListDuration, 'name=''duration''');
            // Sanity check
            if NodeListStart.Count <> NodeListDuration.Count then
              Exit;
            if NodeListStart.Count > 0 then
              for LoopMain := 0 to NodeListStart.Count-1 do
              begin
                // Set time/duration
                Val(TjanXMLNode2(NodeListStart.Items[LoopMain]).Text, TheTime, Error);
                if Error = 0 then
                  DvbFilterXmlDecodeUtcCode(TheTime, Year, Month, Day, Hours, Minutes, Seconds)
                else
                begin
                  DecodeDate(Now, Year, Month, Day);
                  DecodeTime(Now, Hours, Minutes, Seconds, MSeconds);
                end;
                Events.StartTime[EventIndex] := EncodeDate(Year, Month, Day) + EncodeTime(Hours, Minutes, Seconds, 0);

                Val(TjanXMLNode2(NodeListDuration.Items[LoopMain]).Text, TheTime, Error);
                if Error = 0 then
                  DvbFilterXmlDecodeUtcTime(TheTime, Hours, Minutes, Seconds)
                else
                begin
                  Hours   := 0;
                  Minutes := 0;
                  Seconds := 0;
                end;
                Events.Duration[EventIndex] := EncodeTime(Hours, Minutes, Seconds, 0);

                // Process the 'short event'
                // Note: Although a loop is used for multiple short events, there can
                //       actually be only one short event ....
                ANode := TjanXMLNode2(NodeListStart.Items[LoopMain]);
                ANode := ANode.ParentNode;
                NodeListDescriptor.Clear;
                ANode.SelectNodes(NodeListDescriptor, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorShortEvent]));
                if NodeListDescriptor.Count > 0 then
                begin
                  for LoopDescriptor := 0 to NodeListDescriptor.Count-1 do
                  begin
                    ANode := TjanXMLNode2(NodeListDescriptor.Items[LoopDescriptor]);
                    ANode := ANode.ParentNode;
                    NodeListEvent.Clear;
                    ANode.SelectNodes(NodeListEvent, 'name=''event_name_char''');
                    if NodeListEvent.Count > 0 then
                      for LoopEvent := 0 to NodeListEvent.Count-1 do
                      begin
                        New(PString);
                        PString^ := TjanXMLNode2(NodeListEvent.Items[LoopEvent]).Text;
                        PString^ := Copy(PString^, 2, Length(PString^)-2);      // Strip delimiters
                        PString^ := DvbFilterXmlDecodeBinaryStringToString(PString^, True);
                        PString^ := Chr(CDvbEventTypeShortName) + PString^;
                        Events.EventText[EventIndex].Add(PString);
                      end;
                    ANode := TjanXMLNode2(NodeListDescriptor.Items[LoopDescriptor]);
                    ANode := ANode.ParentNode;
                    NodeListEvent.Clear;
                    ANode.SelectNodes(NodeListEvent, 'name=''text_char''');
                    if NodeListEvent.Count > 0 then
                      for LoopEvent := 0 to NodeListEvent.Count-1 do
                      begin
                        New(PString);
                        PString^ := TjanXMLNode2(NodeListEvent.Items[LoopEvent]).Text;
                        PString^ := Copy(PString^, 2, Length(PString^)-2);      // Strip delimiters
                        PString^ := DvbFilterXmlDecodeBinaryStringToString(PString^, True);
                        PString^ := Chr(CDvbEventTypeShortText) + PString^;
                        Events.EventText[EventIndex].Add(PString);
                      end;
                  end;
                end;
                // Process the 'extended event'
                ANode := TjanXMLNode2(NodeListStart.Items[LoopMain]);
                ANode := ANode.ParentNode;
                NodeListDescriptor.Clear;
                ANode.SelectNodes(NodeListDescriptor, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorExtendedEvent]));
                if NodeListDescriptor.Count > 0 then
                begin
                  for LoopDescriptor := 0 to NodeListDescriptor.Count-1 do
                  begin
                    ANode := TjanXMLNode2(NodeListDescriptor.Items[LoopDescriptor]);
                    ANode := ANode.ParentNode;
                    NodeListItem.Clear;
                    ANode.SelectNodes(NodeListItem, 'name=''item''');
                    if NodeListItem.Count > 0 then
                      for LoopItem := 0 to NodeListItem.Count-1 do
                      begin
                        ANode := TjanXMLNode2(NodeListItem.Items[LoopItem]);
                        ANode := ANode.ParentNode;
                        NodeListEvent.Clear;
                        ANode.SelectNodes(NodeListEvent, 'name=''item_description_char''');
                        if NodeListEvent.Count > 0 then
                          for LoopEvent := 0 to NodeListEvent.Count-1 do
                          begin
                            New(PString);
                            PString^ := TjanXMLNode2(NodeListEvent.Items[LoopEvent]).Text;
                            PString^ := Copy(PString^, 2, Length(PString^)-2);      // Strip delimiters
                            PString^ := DvbFilterXmlDecodeBinaryStringToString(PString^, True);
                            PString^ := Chr(CDvbEventTypeExtendedItemName) + PString^;
                            Events.EventText[EventIndex].Add(PString);
                          end;
                        ANode := TjanXMLNode2(NodeListItem.Items[LoopItem]);
                        ANode := ANode.ParentNode;
                        NodeListEvent.Clear;
                        ANode.SelectNodes(NodeListEvent, 'name=''item_char''');
                        if NodeListEvent.Count > 0 then
                          for LoopEvent := 0 to NodeListEvent.Count-1 do
                          begin
                            New(PString);
                            PString^ := TjanXMLNode2(NodeListEvent.Items[LoopEvent]).Text;
                            PString^ := Copy(PString^, 2, Length(PString^)-2);      // Strip delimiters
                            PString^ := DvbFilterXmlDecodeBinaryStringToString(PString^, True);
                            PString^ := Chr(CDvbEventTypeExtendedItemText) + PString^;
                            Events.EventText[EventIndex].Add(PString);
                          end;
                      end;
                    ANode := TjanXMLNode2(NodeListDescriptor.Items[LoopDescriptor]);
                    ANode := ANode.ParentNode;
                    NodeListEvent.Clear;
                    ANode.SelectNodes(NodeListEvent, 'name=''text_char''');
                    if NodeListEvent.Count > 0 then
                      for LoopEvent := 0 to NodeListEvent.Count-1 do
                      begin
                        New(PString);
                        PString^ := TjanXMLNode2(NodeListEvent.Items[LoopEvent]).Text;
                        PString^ := Copy(PString^, 2, Length(PString^)-2);      // Strip delimiters
                        PString^ := DvbFilterXmlDecodeBinaryStringToString(PString^, True);
                        PString^ := Chr(CDvbEventTypeExtendedText) + PString^;
                        Events.EventText[EventIndex].Add(PString);
                      end;
                  end;
                end;
                Inc(EventIndex);
              end;
            end;  
          end;
      end;
      Inc(SectionIndex);
    end;
  finally
    XmlParser.Free;
    NodeListStart.Free;
    NodeListDuration.Free;
    NodeListDescriptor.Free;
    NodeListEvent.Free;
    NodeListItem.Free;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <TreeView>         Treeview object to place data into
            <Range>            Section range (High part = 'from', low part 'to')
            <TransponderName>  Informative name of transponder
  Returns : -

  Descript: Handle XML showing
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbXmlCreateSectionTree(TreeView: TTreeView; Range: LongInt; TransponderName: AnsiString);
const
  Img_Tag          = 0;
  Img_TagWithAttr  = 1;
  Img_UndefinedTag = 2;
  Img_AttrDef      = 3;
  Img_EntityDef    = 4;
  Img_ParEntityDef = 5;
  Img_Text         = 6;
  Img_Comment      = 7;
  Img_PI           = 8;
  Img_DTD          = 9;
  Img_Notation     = 10;
  Img_Prolog       = 11;

var
  ParserText  : TStringList;
  XmlParser   : TJanXmlParser2;
  Section     : Byte;

  {------------------------------------------------------------------------------
    Params  : <Node>   Node in treeview to add to
              <DNode>  XML node to process
    Returns : <Result> False if no more nodes 'upstream'

    Descript: Handle XML node
    Notes   : The data is displayed somewhat 'compressed', so the user does not
              have to expand too deep to see the actual value of an element.
   ------------------------------------------------------------------------------}
  function ParseTreeNode(Node: TTreeNode; DNode:TjanXMLNode2): Boolean;
  var
    N2      : TTreeNode;
    Dn2     : TJanXMLNode2;
    I, C,
    A, J    : Integer;
    NodeName: string;
    First   : Boolean;
  begin
    Result := False;
    C := DNode.Nodes.Count;
    if C = 0 then
      Exit;
    Result := True;
    for I := 0 to C-1 do
    begin
      Dn2 := TjanXMLNode2(DNode.Nodes[I]);
      NodeName := Dn2.Name;
      A := Dn2.Attributes.Count;
      if A > 0 then
      begin
        NodeName := NodeName + ' [';
        First := True;
        for J := 0 to A-1 do
        begin
          if not First then
            NodeName := NodeName + ' ';
          NodeName := NodeName + format('%s="%s"', [Dn2.AttributeName[J], Dn2.Attribute[J]]);
          First := False;
        end;
        NodeName := NodeName + ']';
      end;
      if Dn2.Text <> '' then
        NodeName := NodeName + ' --> ' + Dn2.Text;
      N2 := TreeView.Items.AddChild(Node, NodeName);
      N2.Data := Dn2;
      if A = 0 then
        N2.ImageIndex := Img_Tag
      else
        N2.ImageIndex := Img_TagWithAttr;
      ParseTreeNode(N2, Dn2);                    // Recursive call
    end;
  end;

  procedure ParseTree;
  var
    N       : TTreeNode;
    Dn      : TjanXMLNode2;
    NodeName: string;
  begin
    Dn := XmlParser;
    NodeName := Dn.Name;
    N := TreeView.Items.AddObject(nil, NodeName, Dn);
    N.ImageIndex := Img_UndefinedTag;
    ParseTreeNode(N, Dn);
  end;

begin
  XmlParser := TJanXmlParser2.Create;
  try
    ParserText := TStringList.Create;
    try
      ParserText.Add('<?xml version="1.0" encoding="ISO-8859-1" ?>'#10);
      ParserText.Add(format('<dvb transponder="%s">'#10, [TransponderName]));
      // A negative value indicates the service, otherwise, otherwise a section
      // range is indicated (high part = 'from', low part = 'to')
      if Range < 0 then
      begin
        for Section := $00 to $FF do
        begin
          Application.ProcessMessages;
          ParserText.Add(DvbFilterXmlGetXmlInfoFromSection(Range, Section, -1, '  '));
        end;
      end
      else
      begin
        for Section := ((Range and $FF00) shr 8) to (Range and $FF) do
        begin
          Application.ProcessMessages;
          ParserText.Add(DvbFilterXmlGetXmlInfoFromSection($FFFF, Section, -1, '  '));
        end;
      end;
      ParserText.Add('</dvb>'#10);
      XmlParser.Xml := ParserText.Text;
      TreeView.Items.BeginUpdate;
      TreeView.Items.Clear;
      ParseTree;
      TreeView.Items.EndUpdate;
    finally
      ParserText.Free;
    end;
  finally
    XmlParser.Free
  end;
end;

{------------------------------------------------------------------------------
  Params  : <SectionList>  List of strings to fill with names
  Returns : <SectionList>  .Strings contains the names, were the first
                                    character is the section number applicable
                                    for that name

  Descript: Get section names of active data.
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbXmlGetActiveSectionNames(SectionList: TStringList);
var
  Section    : Byte;
  XmlParser  : TJanXmlParser2;
  NodeList   : TList;
  Name       : AnsiString;
  LastName   : AnsiString;
begin
  SectionList.Clear;
  XmlParser   := TJanXmlParser2.Create;;
  Nodelist    := TList.Create;
  try
    for Section := $FF downto $00 do
    begin
      Application.ProcessMessages;
      Name := DvbFilterXmlGetXmlInfoFromSection($FFFF, Section, -1, '');
      if Name <> '' then
      begin
        XmlParser.Xml := Name;
        // Find section name and id
        NodeList.Clear;
        XmlParser.SelectNodes(XmlParser, NodeList, 'name=''section_name''');
        if NodeList.Count > 0 then
        begin
          Name := TjanXMLNode2(NodeList.Items[0]).Text;
          if Name <> LastName then
          begin
            LastName := Name;
            while Pos('"', Name) <> 0 do
              Delete(Name, Pos('"', Name), 1);
            while Pos('''', Name) <> 0 do
              Delete(Name, Pos('''', Name), 1);
            while Pos('_', Name) <> 0 do
              Name[Pos('_', Name)] := ' ';
            SectionList.Add(Chr(Section) + Name);
          end;
        end;
      end;
    end;
  finally
    XmlParser.Free;
    NodeList.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Service>          Service to save (-1 for all)
            <TransponderName>  Name of transponder
  Returns : -

  Descript: Save XML data for a service, or all services, to a file
  Notes   :
 ------------------------------------------------------------------------------}
procedure DvbXmlSaveData(Service: Integer; TransponderName: AnsiString);
var
  ParserText: TStringList;
  TempText  : Ansistring;
  Id        : Integer;
  Section   : Byte;
begin
  ParserText := TStringList.Create;
  try
    ParserText.Add('<?xml version="1.0" encoding="ISO-8859-1" ?>'#10);
    ParserText.Add(format('<dvb transponder="%s">'#10, [TransponderName]));
    if Service >= 0 then
      Id := -Service
    else
      Id := $FFFF;
    for Section := $00 to $FF do
    begin
      TempText := (DvbFilterXmlGetXmlInfoFromSection(Id, Section, -1, '  '));
      if TempText <> '' then
        ParserText.Add(TempText);
    end;
    ParserText.Add('</dvb>'#10);
    if Service >= 0 then
      ParserText.SaveToFile(format('xml_service_%5.5d.xml', [Service]))
    else
    begin
      while Pos(', ', TransponderName) <> 0 do
        TransponderName[Pos(', ', TransponderName)] := '_';
      while Pos(' ', TransponderName) <> 0 do
        Delete(TransponderName, Pos(' ', TransponderName), 1);
      ParserText.SaveToFile('xml_stream_' + TransponderName + '.xml');
    end;
  finally
    ParserText.Free;
  end;
end;



{------------------------------------------------------------------------------
  Params  : <Section>  Section to get time from
                       CDvbSectionTimeDate or CDvbSectionTimeOffset
  Returns : <Result>   UTC time (or current UTC time if error)

  Descript: Get utc time from time section
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetUtcTime(Section: Byte): SYSTEMTIME;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  TheTime  : Int64;
  Error    : Integer;
begin
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  try
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, Section, -1, '  ');
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''utc_time''');
    GetSystemTime(Result);
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      Val(TjanXMLNode2(NodeList.Items[0]).Text, TheTime, Error);
      if Error = 0 then
      begin
        DvbFilterXmlDecodeUtcCode(TheTime, Result.wYear, Result.wMonth, Result.wDay, Result.wHour, Result.wMinute, Result.wSecond);
      end;
    end;
  finally
    XmlParser.Free;
    NodeList.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>   False if error
            <EmmPids>  EMM pids
            <CaIds>    Ca identifeirs of EM Pids

  Descript: Get EMM Pids (CDvbDescriptorCa in the CDvbSectionCa section)
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetEmmPids(var EmmPids: TDvbPids; var CaIds: TDvbPids): Boolean;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  NodeList2: TList;
  NodeList3: TList;
  ANode    : TjanXMLNode2;
  Loop     : Integer;
  Loop2    : Integer;
  Value    : Word;
  Value2   : Word;
  Error    : Integer;
  EcmIndex : Integer;
begin
//  Result     := False;
  EcmIndex   := 0;
  EmmPids[0] := 0;
  CaIds[0]   := 0;
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  Nodelist2 := TList.Create;
  Nodelist3 := TList.Create;
  try
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, CDvbSectionCa, -1, '  ');
    NodeList.Clear;
    XmlParser.SelectNodes(NodeList, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorCa]));
    if NodeList.Count > 0 then
      for Loop := 0 to NodeList.Count-1 do
      begin
        ANode := TjanXMLNode2(NodeList.Items[Loop]);
        ANode := ANode.ParentNode;
        NodeList2.Clear;
        ANode.SelectNodes(NodeList2, 'name=''ca_system_id''');
        if NodeList2.Count > 0 then
        begin
          NodeList3.Clear;
          ANode.SelectNodes(NodeList3, 'name=''ca_pid''');
          if NodeList3.Count > 0 then
          begin
            Val(TjanXMLNode2(NodeList2.Items[0]).Text, Value, Error);
            if Error = 0 then
            begin
              Val(TjanXMLNode2(NodeList3.Items[0]).Text, Value2, Error);
              if Error = 0 then
              begin
                // Remove duplicates (a value can be used twice: audi + video...)
                Loop2 := 0;
                if EcmIndex > 0 then
                  while (Loop2 < High(EmmPids)) and (EmmPids[Loop2] <> 0) and (EmmPids[Loop2] <> Value2) do
                    Inc(Loop2);
                if (Loop2 < High(EmmPids)) and (EmmPids[Loop2] <> Value2) then
                begin
                  EmmPids[EcmIndex] := Value2;
                  CaIds[EcmIndex] := Value;
                  Inc(EcmIndex);
                  EmmPids[EcmIndex] := 0;
                  CaIds[EcmIndex] := 0;
                end;
              end;
            end;
          end;
        end;
      end;
  finally
    NodeList.Free;
    NodeList2.Free;
    NodeList3.Free;
    XmlParser.Free;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <ServiceId>  Service id to look for
  Returns : <Result>     $FFFF if error, otherwise PMT PID

  Descript: Get PMT PID from XML node using service id
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetPmtPid(ServiceId: Integer): Word;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  NodeList2: TList;
  Value    : Word;
  Error    : Integer;
begin
  Result := $FFFF;
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  Nodelist2 := TList.Create;
  try
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, CDvbSectionProgramAssociation, -1, '  ');
    XmlParser.SelectNodes(XmlParser, NodeList, format('name=''program_number'' and value=''%d''', [ServiceId]));
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      TjanXMLNode2(NodeList.Items[0]).ParentNode.SelectNodes(NodeList2, 'name=''program_map_pid''');
      if NodeList2.Count > 0 then
      begin
        Val(TjanXMLNode2(NodeList2.Items[0]).Text, Value, Error);
        if Error = 0 then
          Result := Value;
      end;
    end;
  finally
    NodeList.Free;
    NodeList2.Free;
    XmlParser.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <ServiceId>  Service id (program_number) to look for
  Returns : <Result>     $FFFF if error, otherwise PCR PID

  Descript: Get PCR PID from XML node using PID
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetPcrPid(ServiceId: Integer): Word;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  NodeList2: TList;
  Value    : Word;
  Error    : Integer;
begin
  Result := $FFFF;
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  Nodelist2 := TList.Create;
  try
    // Get the correct data directly (no need to look any further)
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection(-ServiceId, CDvbSectionTsProgramMap, -1, '  ');
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''pcr_pid''');
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      Val(TjanXMLNode2(NodeList.Items[0]).Text, Value, Error);
      if Error = 0 then
        Result := Value;
    end;
  finally
    NodeList.Free;
    NodeList2.Free;
    XmlParser.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <ServiceId>  Service id to look for
  Returns : <Result>     Empty if not found, otherwise service name

  Descript: Get service name from XML node using service id
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetServiceName(ServiceId: Integer): AnsiString;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  NodeList2: TList;
begin
  Result := '';
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  Nodelist2 := TList.Create;
  try
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, CDvbSectionActualTransportStreamServiceDescription, -1, '  ');
    XmlParser.SelectNodes(XmlParser, NodeList, format('name=''service_id'' and value=''%d''', [ServiceId]));
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      TjanXMLNode2(NodeList.Items[0]).ParentNode.SelectNodes(NodeList2, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorService]));
      if NodeList2.Count > 0 then
      begin
        NodeList.Clear;
        TjanXMLNode2(NodeList2.Items[0]).ParentNode.SelectNodes(NodeList, 'name=''service_name''');
        if NodeList.Count > 0 then
        begin
          Result := TjanXMLNode2(NodeList.Items[0]).Text;
          Result := Copy(Result, 2, Length(Result)-2);     // Strip delimiters
          Result := DvbFilterXmlDecodeBinaryStringToString(Result, True);
        end;
      end;
    end;
  finally
    NodeList.Free;
    NodeList2.Free;
    XmlParser.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Index>        Index of program to get information for (0=first)
  Returns : <Result>       True if valid index was used
            <ServiceId>    Service identifier/Program number
            <PmtPid>       PMT PID        
            <ServiceName>  Name of service

  Descript: Get basic program information of indexed based program
  Notes   : Typicaly used to get the service_id's of the transponder. All
            other functions are typically service_id based.
------------------------------------------------------------------------------}

function DvbXmlGetIndexedProgramInfo(Index: Integer; var ServiceId: Word; var PmtPid: Word; var ServiceName: AnsiString): Boolean;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
  Error    : Integer;
begin
  Result      := False;
  XmlParser := TJanXmlParser2.Create;
  ServiceName := '';
  ServiceId   := $FFFF;
  Nodelist  := TList.Create;
  try
    // From the PAT we get the indexed service id
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, CDvbSectionProgramAssociation, -1, '  ');
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''program_number''');
    // Get indexed Service
    if Index >= NodeList.Count then
      Exit;
    Val(TjanXMLNode2(NodeList.Items[Index]).Text, ServiceId, Error);
    if Error <> 0 then
      Exit;
    PmtPid      := DvbXmlGetPmtPid(ServiceId);
    ServiceName := DvbXmlGetServiceName(ServiceId);
  finally
    NodeList.Free;
    XmlParser.Free;
  end;
  Result := ServiceId <> $FFFF;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Number of programs (services)

  Descript: Get number of programs
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetNumberOfPrograms: Integer;
var
  XmlParser: TJanXmlParser2;
  NodeList : TList;
begin
  XmlParser := TJanXmlParser2.Create;
  Nodelist  := TList.Create;
  try
    // From the PAT we get the indexed service id
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection($FFFF, CDvbSectionProgramAssociation, -1, '  ');
    // Do not include network id
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''program_number'' and value!=''0''');
    Result := NodeList.Count;
  finally
    NodeList.Free;
    XmlParser.Free;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <ServiceId>      Program number (service identifier)
  Returns : <Result>         True if valid
            <PmtPid>         PMT PID
            <PcrPid>         PCR PID
            <VideoPids>      Video PIDs
            <AudioPids>      Audio PIDs
            <TeletextPids>   Teletext PIDs
            <AudioLanguages> Language identifiers audio (capitals if AC3)
            <EcmPids>        ECM PIDs
            <CaIds>          CA System Ids
            <ServiceName>    Name of program/service

  Descript: Get all kind of program information base on service_id (primarily
            PIDs)
  Notes   :
------------------------------------------------------------------------------}

function DvbXmlGetProgramInfo(ServiceId: Word; var PmtPid: Word; var PcrPid: Word;
  var VideoPids        : TDvbPids;
  var AudioPids        : TDvbPids;
  var TeletextPids     : TDvbPids;
  var SubtitlePids     : TDvbPids;
  var AudioLanguages   : TDvbLanguages;
  var TeletextLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids          : TDvbPids;
  var CaIds            : TDvbPids;
  var ServiceName      : AnsiString): Boolean;
var
  XmlParser    : TJanXmlParser2;
  ANode        : TjanXMLNode2;
  NodeList     : TList;
  NodeList2    : TList;
  NodeList3    : TList;
  NodeList4    : TList;
  Value        : Integer;
  Value2       : Integer;
  Error        : Integer;
  Loop         : Integer;
  Loop2        : Integer;
  VideoIndex   : Integer;
  AudioIndex   : Integer;
  TeletextIndex: Integer;
  SubtitleIndex: Integer;
  EcmIndex     : Integer;
  Pid          : Integer;
begin
  Result := False;

  VideoIndex    := 0;
  AudioIndex    := 0;
  TeletextIndex := 0;
  SubtitleIndex := 0;
  EcmIndex      := 0;

  PmtPid               := DvbXmlGetPmtPid(ServiceId);
  PcrPid               := 0;
  VideoPids[0]         := 0;
  AudioPids[0]         := 0;
  TeletextPids[0]      := 0;
  SubtitlePids[0]      := 0;
  AudioLanguages[0]    := '   ';
  TeletextLanguages[0] := '   ';
  SubtitleLanguages[0] := '   ';
  EcmPids[0]           := 0;
  CaIds[0]             := 0;
  ServiceName          := '';

  if PmtPid = $FFFF then
    Exit;
  XmlParser := TJanXmlParser2.Create;
  NodeList  := TList.Create;
  NodeList2 := TList.Create;
  NodeList3 := TList.Create;
  NodeList4 := TList.Create;
  try
    // Get all data referring to service_id
    XmlParser.Xml := DvbFilterXmlGetXmlInfoFromSection(-ServiceId,CDvbSectionTsProgramMap, -1, '  ');
    NodeList.Clear;
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''pcr_pid''');
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      Val(TjanXMLNode2(NodeList.Items[0]).Text, Value, Error);
      if Error = 0 then
        PcrPid := Value;
    end;
    NodeList.Clear;
    XmlParser.SelectNodes(NodeList, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorCa]));
    if NodeList.Count > 0 then
      for Loop := 0 to NodeList.Count-1 do
      begin
        ANode := TjanXMLNode2(NodeList.Items[Loop]);
        ANode := ANode.ParentNode;
        NodeList2.Clear;
        ANode.SelectNodes(NodeList2, 'name=''ca_system_id''');
        if NodeList2.Count > 0 then
        begin
          NodeList3.Clear;
          ANode.SelectNodes(NodeList3, 'name=''ca_pid''');
          if NodeList3.Count > 0 then
          begin
            Val(TjanXMLNode2(NodeList2.Items[0]).Text, Value, Error);
            if Error = 0 then
            begin
              Val(TjanXMLNode2(NodeList3.Items[0]).Text, Value2, Error);
              if Error = 0 then
              begin
                // Remove duplicates (a value can be used twice: audio + video...)
                Loop2 := 0;
                if EcmIndex > 0 then
                  while (Loop2 < High(EcmPids)) and (EcmPids[Loop2] <> 0) and (EcmPids[Loop2] <> Value2) do
                    Inc(Loop2);
                if (Loop2 < High(EcmPids)) and (EcmPids[Loop2] <> Value2) then
                begin
                  EcmPids[EcmIndex] := Value2;
                  CaIds[EcmIndex] := Value;
                  Inc(EcmIndex);
                  EcmPids[EcmIndex] := 0;
                  CaIds[EcmIndex] := 0;
                end;  
              end;
            end;
          end;  
        end;
      end;

    NodeList.Clear;
    XmlParser.SelectNodes(XmlParser, NodeList, 'name=''stream_type''');
    // If we found one (there should only be one)
    if NodeList.Count > 0 then
    begin
      for Loop := 0 to NodeList.Count-1 do
      begin
        // Find video by means of the attribute text
        if Pos('video', TjanXMLNode2(NodeList.Items[Loop]).Attribute[0]) <> 0 then
        begin
          ANode := TjanXMLNode2(NodeList.Items[Loop]);
          ANode := ANode.ParentNode;
          NodeList2.Clear;
          ANode.SelectNodes(NodeList2, 'name=''elementary_pid''');
          if NodeList2.Count > 0 then
          begin
            Val(TjanXMLNode2(NodeList2.Items[0]).Text, Value, Error);
            If Error = 0 then
            begin
              VideoPids[VideoIndex] := Value;
              Inc(VideoIndex);
              VideoPids[VideoIndex] := 0;
            end;
          end;
        end;
        // Find audio by means of the attribute text
        if Pos('audio', TjanXMLNode2(NodeList.Items[Loop]).Attribute[0]) <> 0 then
        begin
          ANode := TjanXMLNode2(NodeList.Items[Loop]);
          ANode := ANode.ParentNode;
          NodeList2.Clear;
          ANode.SelectNodes(NodeList2, 'name=''elementary_pid''');
          if NodeList2.Count > 0 then
          begin
            Val(TjanXMLNode2(NodeList2.Items[0]).Text, Value, Error);
            If Error = 0 then
            begin
              AudioPids[AudioIndex] := Value;
              AudioLanguages[AudioIndex] := '???';
              // Find audio language
              ANode := TjanXMLNode2(NodeList2.Items[0]);
              ANode := ANode.ParentNode;
              NodeList3.Clear;
              ANode.SelectNodes(NodeList3, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorIso639Language]));
              if NodeList3.Count > 0 then
              begin
                ANode := TjanXMLNode2(NodeList3.Items[0]);
                ANode := ANode.ParentNode;
                NodeList4.Clear;
                ANode.SelectNodes(NodeList4, 'name=''iso_639_language_code''');
                if NodeList4.Count > 0 then
                  AudioLanguages[AudioIndex] := TjanXMLNode2(NodeList4.Items[0]).Attribute[0];
              end;
              Inc(AudioIndex);
              AudioPids[AudioIndex] := 0;
              AudioLanguages[AudioIndex] := '   ';
            end;
          end;
        end;
        // Find other data, which is in the private section)  by means of the attribute text
        if Pos('private', TjanXMLNode2(NodeList.Items[Loop]).Attribute[0]) <> 0 then
        begin
          ANode := TjanXMLNode2(NodeList.Items[Loop]);
          ANode := ANode.ParentNode;
          NodeList2.Clear;
          ANode.SelectNodes(NodeList2, 'name=''elementary_pid''');
          if NodeList2.Count > 0 then
          begin
            Val(TjanXMLNode2(NodeList2.Items[0]).Text, Pid, Error);
            If Error = 0 then
            begin
              ANode := TjanXMLNode2(NodeList2.Items[0]);
              ANode := ANode.ParentNode;
              NodeList3.Clear;
              ANode.SelectNodes(NodeList3, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorTeletext]));
              if NodeList3.Count > 0 then
              begin
                TeletextPids[TeletextIndex] := Pid;
                ANode := TjanXMLNode2(NodeList3.Items[0]);
                ANode := ANode.ParentNode;
                NodeList4.Clear;
                ANode.SelectNodes(NodeList4, 'name=''iso_639_language_code''');
                if NodeList4.Count > 0 then
                  TeletextLanguages[TeletextIndex] := TjanXMLNode2(NodeList4.Items[0]).Attribute[0];
                Inc(TeletextIndex);
                TeletextPids[TeletextIndex] := 0;
                TeletextLanguages[TeletextIndex] := '   ';
              end
              else
              begin
                NodeList3.Clear;
                ANode.SelectNodes(NodeList3, format('name=''descriptor_tag'' and value=''%d''', [CDvbDescriptorSubtitling]));
                if NodeList3.Count > 0 then
                begin
                  SubtitlePids[SubtitleIndex] := Pid;
                  ANode := TjanXMLNode2(NodeList3.Items[0]);
                  ANode := ANode.ParentNode;
                  NodeList4.Clear;
                  ANode.SelectNodes(NodeList4, 'name=''iso_639_language_code''');
                  if NodeList4.Count > 0 then
                    SubtitleLanguages[TeletextIndex] := TjanXMLNode2(NodeList4.Items[0]).Attribute[0];
                  Inc(SubtitleIndex);
                  SubtitlePids[SubtitleIndex] := 0;
                  SubtitleLanguages[TeletextIndex] := '   ';
                end
              end;
            end;
          end;
        end;
      end;
    end;

    ServiceName := DvbXmlGetServiceName(ServiceId);
  finally
    NodeList4.Free;
    NodeList3.Free;
    NodeList2.Free;
    NodeList.Free;
    XmlParser.Free;
  end;
  Result := True;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   :
------------------------------------------------------------------------------}

procedure Initialize;
begin
  //
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   :
------------------------------------------------------------------------------}

procedure Finalize;
begin
  //
end;


initialization
  Initialize;

finalization
  Finalize;

end.

