{******************************************************************************}
{ FileName............: DvbFilterXml                                           }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 2.01                                                   }
{------------------------------------------------------------------------------}
{  Filtering of data streams                                                   }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ There are different PID's. PID's which indicate information (PSI) or data    }
{ (PES). We first have to receive PSI data before we can identify the PES data.}
{ The global structure is as follows:                                          }
{   Program Association Table (PAT) = PID $0000                                }
{      -> PAT indicates other PID's for the Program Map Table (PMT)            }
{          -> PMT indicates other PID's for the individual streams (PES)       }
{ Thus at first we have no information at all. Then, after receiving the PAT   }
{ we can identify the PID's which carry PMT data. Then we can identify         }
{ PID's used for stream data (PES).                                            }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20040222 - Initial release                                           }
{  1.01   20040507 - AC3 audio is indicated by capital language (or '***'      }
{                    instead of '---' if no language specified)                }
{  1.02   20040801 - Real program number (service identification) added        }
{  1.03   20040811 - CaDescriptors made available                              }
{  1.04   20040828 - Additional info send with PostMessage                     }
{                  - CA has now version number and callback                    }
{                  - PAT has callback                                          }
{                  - <DvbFilterGetEventInfo> also using service id             }
{  1.05   20041229 - <DvbGetPayloadOffset> detected offset now checks on       }
{                    value                                                     }
{  1.06   20050426 - <DvbGetProgramPids> returns all PIDs of program           }
{  1.07   20050814 - Time update from transponder added                        }
{  2.00   20051105 - Completely revised: based on sections (table id) and      }
{                    descriptors (descriptor_tag)                              }
{                  - Data is returned as XML data                              }
{                  - All 'arrays' (binary or a string) are returned as         }
{                    'binary' ASCII data (2-digit hexadecimal characters) and  }
{                    a 'text' attribute with the string representation.        }
{  2.01   20051128 - Name changes external accessible functions                }
{                  - Problem with GetReference - sometimes fails - unsure why  }
{                    because the 'if Assigned(..)' fails (also when dynamic    }
{                    array part removed)                                       }
{                  - UTC date conversion catches illegal dates                 }  
{******************************************************************************}
unit DvbFilterXml;

interface

uses
  Windows;


type
  // For single packet filtering. StreamPacketData must point to 188 byte packet
  TDvbPacketFilter = function(StreamPacketData: Pointer): Boolean; stdcall;


function  DvbFilterXmlGetPid(StreamPacketData: Pointer; var Pid: Word): Boolean; stdcall;
procedure DvbFilterXmlResetErrors; stdcall;
procedure DvbFilterXmlResetData(Preset: Boolean); stdcall;
function  DvbFilterXmlGetErrors: Word; stdcall;
function  DvbFilterXmlGetPacketSyncErrors: Word; stdcall;
function  DvbFilterXmlGetSectionCount: Word; stdcall;
function  DvbFilterXmlGetDebug: Word; stdcall;
function  DvbFilterXmlGetPacketFilter(Pid: Word): TDvbPacketFilter; stdcall;
procedure DvbFilterXmlActivatePacketFilter(Pid: Word); stdcall;
procedure DvbFilterXmlDeActivatePacketFilter(Pid: Word); stdcall;
function  DvbFilterXmlSetSignallingCallback(Section: Byte; hWnd: HWND; Msg: UINT): Boolean; stdcall;
function  DvbFilterXmlGetXmlInfoFromSection(What: Integer; Section: Byte; SectionNumber: Integer; Indent: string): AnsiString; stdcall;

function  DvbFilterXmlDecodeUtcDate(ToDecode: Word; var Year, Month, Day: Word): Boolean; stdcall;
function  DvbFilterXmlDecodeUtcTime(ToDecode: Dword; var Hours, Minutes, Seconds: Word): Boolean; stdcall;
function  DvbFilterXmlDecodeUtcCode(ToDecode: Int64; var Year, Month, Day, Hours, Minutes, Seconds: Word): Boolean; stdcall;
function  DvbFilterXmlDecodeBinaryStringToString(ToDecode: AnsiString; XmlConvert: Boolean): AnsiString;
function  DvbFilterXmlDecodeBinaryStringToXmlString(ToDecode: AnsiString): AnsiString;
function  DvbFilterXmlXmlCorrectString(StringIn: AnsiString): AnsiString;

procedure DvbFilterXmlInitialize;
procedure DvbFilterXmlFinalize;


const
  CXmlLf: string[2] = #10;

  CDvbPacketSyncWord = $47;                      // DVB sync word
  CDvbPacketSize     = 188;

  // PIDs which can carry Service Information sections
  CDvbPidPAT  = $0000;                           // Program Association Table           etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidCAT  = $0001;                           // Conditional Access Table            etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidTSDT = $0002;                           // Transport Stream Description Table  etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidNIT  = $0010;                           // Network Information Table           etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidST1  = $0010;                           // Stuffing Table                      etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidSDT  = $0011;                           // Service Description Table           etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidST2  = $0011;                           // Stuffing Table                      etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidBAT  = $0011;                           // Bouquet Association Table           etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidEIT  = $0012;                           // Event Information Table             etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidSTCIT= $0012;                           // Stuffing Table                      etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidRST  = $0013;                           // Running Status Table                etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidST3  = $0013;                           // Stuffing Table                      etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidTDT  = $0014;                           // Time and Date Table                 etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidTOT  = $0014;                           // Time Offset Table                   etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidST4  = $0014;                           // Stuffing Table                      etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidNS   = $0015;                           // Network synchronization             etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidRNT  = $0016;                           //                                     etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidIS   = $001C;                           // Inband signalling                   etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidM    = $001D;                           // Measurement                         etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidDIT  = $001E;                           // Discontinuity Information Table     etsi en 300 468 v1.6.1 (2004-11)
  CDvbPidSIT  = $001F;                           // Selection Information Table         etsi en 300 468 v1.6.1 (2004-11)

  // Service Information sections (table_id)
  CDvbSectionProgramAssociation                                    = $00;      // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbSectionCa                                                    = $01;      // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbSectionTsProgramMap                                          = $02;      // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbSectionTransportStreamDescription                            = $03;
  CDvbSectionMetadata                                              = $06;      // iso/iec 13818-1:2000/fdam 1:2003

  CDvbSectionMeasurement                                           = $10;      // testdata (pid $1d)
  CDvbSectionTestSignal                                            = $11;      // testdata (pid $1d)
  CDvbSectionRemoteControl                                         = $12;      // testdata (pid $1d)
  CDvbSectionReceptionStatus                                       = $13;      // testdata (pid $1d)
  CDvbSectionNetworkStatus                                         = $14;      // testdata (pid $1d)

  CDvbSectionDsmcc0                                                = $3B;      // atsc data broadcast standard (2000-07)
  CDvbSectionDsmcc1                                                = $3C;      // atsc data broadcast standard (2000-07)
  CDvbSectionDsmccAddressable                                      = $3F;      // atsc data broadcast standard (2000-07)

  CDvbSectionActualNetworkNetworkInformation                       = $40;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherNetworkNetworkInformation                        = $41;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamServiceDescription               = $42;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamServiceDescription                = $46;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionBouquetAssociation                                    = $4A;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamPresentFollowingEventInformation = $4E;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamPresentFollowingEventInformation  = $4F;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation0        = $50;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation1        = $51;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation2        = $52;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation3        = $53;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation4        = $54;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation5        = $55;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation6        = $56;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation7        = $57;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation8        = $58;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformation9        = $59;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationA        = $5A;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationB        = $5B;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationC        = $5C;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationD        = $5D;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationE        = $5E;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionActualTransportStreamScheduleEventInformationF        = $5F;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation0         = $60;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation1         = $61;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation2         = $62;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation3         = $63;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation4         = $64;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation5         = $65;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation6         = $66;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation7         = $67;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation8         = $68;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformation9         = $69;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationA         = $6A;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationB         = $6B;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationC         = $6C;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationD         = $6D;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationE         = $6E;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionOtherTransportStreamScheduleEventInformationF         = $6F;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionTimeDate                                              = $70;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionRunningStatus                                         = $71;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionStuffing                                              = $72;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionTimeOffset                                            = $73;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionResolutionNotification                                = $74;
  CDvbSectionContainer                                             = $75;
  CDvbSectionRelatedContent                                        = $76;
  CDvbSectionContentIdentifier                                     = $77;
  CDvbSectionMpeFec                                                = $78;
  CDvbSectionDiscontinuityInformation                              = $7E;      // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbSectionSelectionInformation                                  = $7F;      // * etsi en 300 468 v1.6.1 (2004-11)

  CDvbSectionCaMessageEcm0                                         = $80;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEcm1                                         = $81;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate0                       = $82;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate1                       = $83;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate2                       = $84;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate3                       = $85;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate4                       = $86;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate5                       = $87;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate6                       = $88;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate7                       = $89;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate8                       = $8A;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivate9                       = $8B;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivateA                       = $8C;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivateB                       = $8D;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivateC                       = $8E;      // * etr 289 (1996-10)
  CDvbSectionCaMessageEmmAndCaSystemPrivateD                       = $8F;      // * etr 289 (1996-10)

  CDvbSectionDataEventTable                                        = $CE;      // atsc data broadcast standard (2000-07)
  CDvbSectionDataServiceTable                                      = $CF;      // atsc data broadcast standard (2000-07)
  CDvbSectionNetworkResourcesTable                                 = $D1;      // atsc data broadcast standard (2000-07)
  CDvbSectionLongTermServiceTable                                  = $D2;      // atsc data broadcast standard (2000-07)

// CDvbSectionPrivate      // itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)

  // Descriptors (descriptor_tag) within sections
  CDvbDescriptorVideoStream                  = $02;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorAudioStream                  = $03;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorHierarchy                    = $04;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorRegistration                 = $05;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorDataStreamAlignment          = $06;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorTargetBackgroundGrid         = $07;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorVideoWindow                  = $08;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorCa                           = $09;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorIso639Language               = $0A;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorSystemClock                  = $0B;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorMultiplexBufferUtilization   = $0C;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorCopyright                    = $0D;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorMaximumBitrate               = $0E;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorPrivateDataIndicator         = $0F;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorSmoothingBuffer              = $10;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorStd                          = $11;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
  CDvbDescriptorIbp                          = $12;        // * itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)

  CDvbDescriptorCarouselIdentifier           = $13;
  CDvbDescriptorAssociationTag               = $14;        // atsc data broadcast standard (2000-07)
  CDvbDescriptorDeferredAssociationTags      = $15;
  CDvbDescriptorNptReference                 = $17;
  CDvbDescriptorNptEndpoint                  = $18;
  CDvbDescriptorStreamMode                   = $19;
  CDvbDescriptorStreamEvent                  = $1A;

  CDvbDescriptorMpeg4Video                   = $1B;
  CDvbDescriptorMpeg4Audio                   = $1C;
  CDvbDescriptorIod                          = $1D;
  CDvbDescriptorSl                           = $1E;
  CDvbDescriptorFmc                          = $1F;
  CDvbDescriptorExternalEsId                 = $20;
  CDvbDescriptorMuxCode                      = $21;
  CDvbDescriptorFmxBufferSize                = $22;
  CDvbDescriptorMultiplexBuffer              = $23;
  CDvbDescriptorContentLabeling              = $24;        // iso/iec 13818-1:2000/fdam 1:2003
  CDvbDescriptorMetadataPointer              = $25;        // iso/iec 13818-1:2000/fdam 1:2003
  CDvbDescriptorMetadata                     = $26;        // iso/iec 13818-1:2000/fdam 1:2003
  CDvbDescriptorMetadataStd                  = $27;        // iso/iec 13818-1:2000/fdam 1:2003

  CDvbDescriptorAvcVideo                     = $28;
  CDvbDescriptorIpmp                         = $29;
  CDvbDescriptorAvcTimingAndHrd              = $2A;
  CDvbDescriptorMpeg2AacAudio                = $2B;
  CDvbDescriptorMpeg4Text                    = $2C;

  CDvbDescriptorElementaryId                 = $40;        // testdata (pid $1d)
  CDvbDescriptorContentInformation           = $41;        // testdata (pid $1d)
  CDvbDescriptorSourceIdentifier             = $42;        // testdata (pid $1d)
  CDvbDescriptorTestSignal                   = $43;        // testdata (pid $1d)
  CDvbDescriptorTimeReference                = $44;        // testdata (pid $1d)
  CDvbDescriptorGps                          = $45;        // testdata (pid $1d)
  CDvbDescriptorReducedPcr                   = $46;        // testdata (pid $1d)
  CDvbDescriptorProgram                      = $47;        // testdata (pid $1d)

  CDvbDescriptorNetworkName                  = $40;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorServiceList                  = $41;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorStuffing                     = $42;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorSatelliteDeliverySystem      = $43;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorCableDeliverySystem          = $44;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorVbiData                      = $45;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorVbiTeletext                  = $46;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorBouquetName                  = $47;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorService                      = $48;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorCountryAvailability          = $49;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorLinkage                      = $4A;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorNvodReference                = $4B;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTimeShiftedService           = $4C;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorShortEvent                   = $4D;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorExtendedEvent                = $4E;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTimeShiftedEvent             = $4F;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorComponent                    = $50;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorMosaic                       = $51;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorStreamIdentifier             = $52;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorCaIdentifier                 = $53;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorContent                      = $54;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorParentalRating               = $55;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTeletext                     = $56;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTelephone                    = $57;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorLocalTimeOffset              = $58;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorSubtitling                   = $59;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTerrestrialDeliverySystem    = $5A;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorMultilingualNetworkName      = $5B;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorMultilingualBouquetName      = $5C;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorMultilingualServiceName      = $5D;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorMultilingualComponent        = $5E;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorPrivateDataSpecifier         = $5F;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorServiceMove                  = $60;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorShortSmoothingBuffer         = $61;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorFrequencyList                = $62;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorPartialTransportStream       = $63;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorDataBroadcast                = $64;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorScrambling                   = $65;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorDataBroadcastId              = $66;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorTransportStream              = $67;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorDsng                         = $68;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorPdc                          = $69;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorAc3                          = $6A;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorAncillaryData                = $6B;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorCellList                     = $6C;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorCellFrequencyLink            = $6D;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorAnnouncementSupport          = $6E;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorApplicationSignalling        = $6F;
  CDvbDescriptorAdaptationFieldData          = $70;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorServiceIdentifier            = $71;
  CDvbDescriptorServiceAvailability          = $72;        // * etsi en 300 468 v1.6.1 (2004-11)
  CDvbDescriptorDefaultAuthority             = $73;
  CDvbDescriptorRelatedContent               = $74;
  CDvbDescriptorTvaId                        = $75;
  CDvbDescriptorContentIdentifier            = $76;
  CDvbDescriptorTimeSliceFecIdentifier       = $77;
  CDvbDescriptorEcmRepetitionRate            = $78;
  CDvbDescriptorEnhancedAc3                  = $7A;        // * etsi ts 101 154 V1.7.1 (2005-06)

  CDvbDescriptorAtscCa                       = $88;        // * atsc standard (2000-05)
  CDvbDescriptorDataService                  = $A4;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorPidCount                     = $A5;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorDownload                     = $A6;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorMultiprotocolEncapsulation   = $A7;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorModuleLink                   = $B4;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorCrc32                        = $B5;        // * atsc data broadcast standard (2000-07)
  CDvbDescriptorGroupLink                    = $B7;        // * atsc data broadcast standard (2000-07)

  CDvbDescriptorForbidden                    = $FF;

implementation

uses
  SyncObjs,
  SysUtils;

const
  // 16 Bit CCITT standard polynomial x^16 + x^12 + x^5 + x^1 + x^0
  CDvbPolynomial16 = $1021;
  CDvbCrcInitial16 = $FFFF;
  // 32 Bit standard polynomial x^32 + x^26 + x^23 + x^22 + x^16 +
  //   x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x^1 + x^0
  CDvbPolynomial32 = $04C11DB7;
  CDvbCrcInitial32 = $FFFFFFFF;

  // Next table is for reference only. This is the generated data and should
  // be equal to <Crc32Table>.
  Crc32Table2: array[Byte] of Dword =
    ($00000000, $04C11DB7, $09823B6E, $0D4326D9, $130476DC, $17C56B6B,
     $1A864DB2, $1E475005, $2608EDB8, $22C9F00F, $2F8AD6D6, $2B4BCB61,
     $350C9B64, $31CD86D3, $3C8EA00A, $384FBDBD, $4C11DB70, $48D0C6C7,
     $4593E01E, $4152FDA9, $5F15ADAC, $5BD4B01B, $569796C2, $52568B75,
     $6A1936C8, $6ED82B7F, $639B0DA6, $675A1011, $791D4014, $7DDC5DA3,
     $709F7B7A, $745E66CD, $9823B6E0, $9CE2AB57, $91A18D8E, $95609039,
     $8B27C03C, $8FE6DD8B, $82A5FB52, $8664E6E5, $BE2B5B58, $BAEA46EF,
     $B7A96036, $B3687D81, $AD2F2D84, $A9EE3033, $A4AD16EA, $A06C0B5D,
     $D4326D90, $D0F37027, $DDB056FE, $D9714B49, $C7361B4C, $C3F706FB,
     $CEB42022, $CA753D95, $F23A8028, $F6FB9D9F, $FBB8BB46, $FF79A6F1,
     $E13EF6F4, $E5FFEB43, $E8BCCD9A, $EC7DD02D, $34867077, $30476DC0,
     $3D044B19, $39C556AE, $278206AB, $23431B1C, $2E003DC5, $2AC12072,
     $128E9DCF, $164F8078, $1B0CA6A1, $1FCDBB16, $018AEB13, $054BF6A4,
     $0808D07D, $0CC9CDCA, $7897AB07, $7C56B6B0, $71159069, $75D48DDE,
     $6B93DDDB, $6F52C06C, $6211E6B5, $66D0FB02, $5E9F46BF, $5A5E5B08,
     $571D7DD1, $53DC6066, $4D9B3063, $495A2DD4, $44190B0D, $40D816BA,
     $ACA5C697, $A864DB20, $A527FDF9, $A1E6E04E, $BFA1B04B, $BB60ADFC,
     $B6238B25, $B2E29692, $8AAD2B2F, $8E6C3698, $832F1041, $87EE0DF6,
     $99A95DF3, $9D684044, $902B669D, $94EA7B2A, $E0B41DE7, $E4750050,
     $E9362689, $EDF73B3E, $F3B06B3B, $F771768C, $FA325055, $FEF34DE2,
     $C6BCF05F, $C27DEDE8, $CF3ECB31, $CBFFD686, $D5B88683, $D1799B34,
     $DC3ABDED, $D8FBA05A, $690CE0EE, $6DCDFD59, $608EDB80, $644FC637,
     $7A089632, $7EC98B85, $738AAD5C, $774BB0EB, $4F040D56, $4BC510E1,
     $46863638, $42472B8F, $5C007B8A, $58C1663D, $558240E4, $51435D53,
     $251D3B9E, $21DC2629, $2C9F00F0, $285E1D47, $36194D42, $32D850F5,
     $3F9B762C, $3B5A6B9B, $0315D626, $07D4CB91, $0A97ED48, $0E56F0FF,
     $1011A0FA, $14D0BD4D, $19939B94, $1D528623, $F12F560E, $F5EE4BB9,
     $F8AD6D60, $FC6C70D7, $E22B20D2, $E6EA3D65, $EBA91BBC, $EF68060B,
     $D727BBB6, $D3E6A601, $DEA580D8, $DA649D6F, $C423CD6A, $C0E2D0DD,
     $CDA1F604, $C960EBB3, $BD3E8D7E, $B9FF90C9, $B4BCB610, $B07DABA7,
     $AE3AFBA2, $AAFBE615, $A7B8C0CC, $A379DD7B, $9B3660C6, $9FF77D71,
     $92B45BA8, $9675461F, $8832161A, $8CF30BAD, $81B02D74, $857130C3,
     $5D8A9099, $594B8D2E, $5408ABF7, $50C9B640, $4E8EE645, $4A4FFBF2,
     $470CDD2B, $43CDC09C, $7B827D21, $7F436096, $7200464F, $76C15BF8,
     $68860BFD, $6C47164A, $61043093, $65C52D24, $119B4BE9, $155A565E,
     $18197087, $1CD86D30, $029F3D35, $065E2082, $0B1D065B, $0FDC1BEC,
     $3793A651, $3352BBE6, $3E119D3F, $3AD08088, $2497D08D, $2056CD3A,
     $2D15EBE3, $29D4F654, $C5A92679, $C1683BCE, $CC2B1D17, $C8EA00A0,
     $D6AD50A5, $D26C4D12, $DF2F6BCB, $DBEE767C, $E3A1CBC1, $E760D676,
     $EA23F0AF, $EEE2ED18, $F0A5BD1D, $F464A0AA, $F9278673, $FDE69BC4,
     $89B8FD09, $8D79E0BE, $803AC667, $84FBDBD0, $9ABC8BD5, $9E7D9662,
     $933EB0BB, $97FFAD0C, $AFB010B1, $AB710D06, $A6322BDF, $A2F33668,
     $BCB4666D, $B8757BDA, $B5365D03, $B1F740B4);

type
  // For section filtering
  TDvbSectionFilter    = function(SectionData: Pointer; Pid: Word): Boolean; stdcall;
  // For descriptor filtering
  TDvbDescriptorFilter = function(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
  // For decoding
  TDvbDecodeSection    = function(SectionData   : Pointer; Indent: AnsiString): AnsiString; stdcall;
  TDvbDecodeDescriptor = function(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;

  TDvbPacketFilters = array[0..$1FFF] of TDvbPacketFilter;

  // For message callbacks
  PDvbCallback = ^TDvbCallback;
  TDvbCallback = record
    FHandle       : HWND;                        // Handle to send message to
    FMsg          : UINT;                        // Message to send
  end;

  TDvbSectionsCommonCallbacks = array[Byte] of TDvbCallback; // Section, current/next indexed

  // Data being received is stored in the TDvbSections array, based
  // on the base identifier (network_id/bouquet_id/transport-stream_id),
  // on the section identifier (table_id), the current/next indication
  // (current_next_indicator) and the section number (section_number)
  // There is a 'double' buffering mechanism (ActiveIndex) which should give an
  // application enough time to get valid data.

  // Note: Data is SERVICE based stored.
  //       Data is -not- PID based because a single PID can carry information
  //       for different services. This is most noticable for the event
  //       information.

  // Section: Complete section (all section numbers)
  TDvbSection = class(TObject)
    FSection    : array[Boolean, Byte] of Pointer;         // Accumulated individual section numbers (double buffered)
    FActive     : Boolean;                                 // Active 'index'
    FLastVersion: Byte;                                    // Last version which generated a callback
    function  GetSection(Active: Boolean; SectionNumber: Byte): Pointer;
    procedure SetSection(Active: Boolean; SectionNumber: Byte; Value: Pointer);
  protected
  public
    constructor Create;
    destructor  Destroy; override;
    property    Section[Active: Boolean; SectionNumber: Byte]: Pointer read GetSection   write SetSection;
    property    ActiveIndex                                  : Boolean read FActive      write FActive;
    property    LastVersion                                  : Byte    read FLastVersion write FLastVersion;
  end;

  // Sections belonging to same identifier (current/next based)
  TDvbSections = class(TObject)
    FSections        : array[Word, Byte] of TDvbSection;   // Service, TableId
    FServiceLastIndex: Byte;                               // Last index used
    function  GetSection(Active: Boolean; Service: Word; TableId: Byte; SectionNumber: Byte): Pointer;
    procedure SetSection(Active: Boolean; Service: Word; TableId: Byte; SectionNumber: Byte; Value: Pointer);
    function  GetLastVersion(Service: Word; TableId: Byte): Byte;
    procedure SetLastVersion(Service: Word; TableId: Byte; Value: Byte);
    function  GetActiveIndex(Service: Word; TableId: Byte): Boolean;
    procedure SetActiveIndex(Service: Word; TableId: Byte; Value: Boolean);
  protected
  public
    constructor Create;
    destructor  Destroy; override;
    property    Section    [Active: Boolean; Service: Word; TableId: Byte; SectionNumber: Byte]: Pointer read GetSection     write SetSection;
    property    LastVersion[Service: Word; TableId: Byte]                                      : Byte    read GetLastVersion write SetLastVersion;
    property    ActiveIndex[Service: Word; TableId: Byte]                                      : Boolean read GetActiveIndex write SetActiveIndex;
  end;


  // Sections can never be larger than this
  PDvbAccumulateSection = ^TDvbAccumulateSection;
  TDvbAccumulateSection = array[0..4095] of Byte; // Max number of bytes in a section (EIT can be max 4096, others max 1024)

  // Section: Gathered over multiple packets
  TDvbReceivingSection = class(TObject)
    FPointerField: Word;                         // Bytes to discard (PointerField)
    FStreamData  : PDvbAccumulateSection;        // Data of multiple packets
    FDataIndex   : Word;                         // Index in data (set to zero when processed)
    function  GetStreamData: PDvbAccumulateSection;
    procedure SetStreamData(Value: PDvbAccumulateSection);
  protected
  public
    constructor Create;
    destructor  Destroy; override;
    property PointerField: Word                  read FPointerField write FPointerField;
    property StreamData  : PDvbAccumulateSection read GetStreamData write SetStreamData;
    property DataIndex   : Word                  read FDataIndex    write FDataIndex;
  end;

  // Section: Gathered over multiple packets
  TDvbReceivingSections = class(TObject)
    FDvbReceivingSections: array[0..$1FFF] of TDvbReceivingSection;
  protected
    function  GetPointerField(Index: Word): Word;
    procedure SetPointerField(Index: Word; Value: Word);
    function  GetStreamData  (Index: Word): PDvbAccumulateSection;
    procedure SetStreamData  (Index: Word; Value: PDvbAccumulateSection);
    function  GetDataIndex   (Index: Word): Word;
    procedure SetDataIndex   (Index: Word; Value: Word);
  public
    constructor Create;
    destructor  Destroy; override;
    property PointerField[Index: Word]: Word                  read GetPointerField write SetPointerField;
    property StreamData  [Index: Word]: PDvbAccumulateSection read GetStreamData   write SetStreamData;
    property DataIndex   [Index: Word]: Word                  read GetDataIndex    write SetDataIndex;
  end;

  // Pid to service references
  TDvbReference = class(TObject)
    FReferences: array[Word] of array of Word;
  protected
    function  GetReference(Index: Word; Index2: Word): Integer;
    procedure SetReference(Index: Word; Index2: Word; Value: Integer);
    function  GetReferenceExists(Index: Word; Value: Word): Boolean;
    property  ReferenceExists[Index: Word; Value: Word]: Boolean read GetReferenceExists;
  public
    constructor Create;
    destructor  Destroy; override;
    property Reference[Index: Word; Index2: Word]: Integer read GetReference write SetReference;
  end;


  function DvbResetValueCrc16: Word; forward;
  function DvbResetValueCrc32: Dword; forward;
  function DvbNewValueCrc16(Checksum: Word; Data: Byte): Word; forward;
  function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword; forward;
  function DvbGetXmlInfoFromDescriptor(DescriptorData: Pointer; Indent: string): AnsiString; forward;



var
  PacketFilters         : TDvbPacketFilters;               // Filters (function calls) for each PID

  PidToServicesReference    : TDvbReference;              // Cross reference PID      -> services
  PidToSectionsReference    : TDvbReference;              // Cross reference PID      -> sections
  ServiceToPidsReference    : TDvbReference;              // Cross reference service  -> PIDs
  ServiceToSectionsReference: TDvbReference;              // Cross reference service  -> sections
  SectionToPidsReference    : TDvbReference;              // Cross reference section  -> PIDs
  SectionToServicesReference: TDvbReference;              // Cross reference sections -> services

  SectionsReceiving     : TDvbReceivingSections;           // Sections build up from several packets
  Sections              : TDvbSections;                    // All data of sections being received
  SectionsCommonCallback: TDvbSectionsCommonCallbacks;     // Assigned callback if ALL pids are concerned

  SectionsDecode         : array[Byte] of TDvbDecodeSection;         // Decoding functions for sections
  SectionsDecodeStartName: array[Byte] of string;                    // Decoding function XML start name
  SectionsDecodeStopName : array[Byte] of string;                    // Decoding function XMl stop name
  DescriptorsDecode      : array[Byte] of TDvbDecodeDescriptor;      // Decoding functions for descriptors

  Debug              : Word;                               // Generic use for debug purposes
  PacketError        : Word;                               // Checksum error counter
  PacketSyncErrors   : Word;                               // Sync error counter
  SectionCount       : Word;                               // New sections counter
  ResetLock          : TCriticalSection;                   // Prevents processing during 'reset' condition
  ActiveIndexLock    : TCriticalSection;                   // Prevents changing active index during processing

  Crc16Table         : array[Byte] of Word;
  Crc32Table         : array[Byte] of Dword;


{------------------------------------------------------------------------------
                            TDvbReference
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbReference.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FReferences) to High(FReferences) do
    FReferences[Loop] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbReference.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FReferences) to High(FReferences) do
    FReferences[Loop] := nil;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Index1
            <Index2>  Index2
  Returns : <Result>  Value of indexed value (-1 if non available)

  Descript: Get indexed value.
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReference.GetReference(Index: Word; Index2: Word): Integer;
begin
  if Assigned(FReferences[Index]) and (Index2 <= High(FReferences[Index])) then
    Result := FReferences[Index, Index2]
  else
    Result := -1;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Index
            <Value>   Value to look for
  Returns : <Result>  True if value exists

  Descript: Check if value exists for index.
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReference.GetReferenceExists(Index: Word; Value: Word): Boolean;
var
  Check: Integer;
begin
  Result := False;
  if Assigned(FReferences[Index]) then
    for Check := Low(FReferences[Index]) to High(FReferences[Index]) do
      if FReferenceS[Index][Check] = Value then
      begin
        Result := True;
        Exit;
      end;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   Index1
            <Index2>  Not used
            <Value>   Value to set index to
  Returns : <Result>  Value of indexed value (-1 if non available)

  Descript: Set indexed value.
  Notes   : The value is added to the reference list if it is not yet in it.
            The actual index (Index2) is automatically determined.
 ------------------------------------------------------------------------------}

procedure TDvbReference.SetReference(Index: Word; Index2: Word; Value: Integer);
var
  SetValue: Word;
begin
  SetValue := Word(Value);
  if not ReferenceExists[Index, SetValue] then
  begin
    if Assigned(FReferences[Index]) then
      SetLength(FReferences[Index], Length(FReferences[Index]) + 1)
    else
      SetLength(FReferences[Index], 1);
    FReferences[Index][High(FReferences[Index])] := SetValue;
  end;
end;


{------------------------------------------------------------------------------
                            TDvbReference
 ------------------------------------------------------------------------------}


{------------------------------------------------------------------------------
                                 TDvbSection
 ------------------------------------------------------------------------------}

 {------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbSection.Create;
var
  Loop : Boolean;
  Loop2: Integer;
begin
  inherited Create;
  FActive      := False;
  FLastVersion := $FF;
  for Loop := Low(FSection) to High(FSection) do
    for Loop2 := Low(FSection[False]) to High(FSection[False]) do
      FSection[Loop, Loop2] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbSection.Destroy;
var
  Loop : Boolean;
  Loop2: Integer;
begin
  for Loop := Low(FSection) to High(FSection) do
    for Loop2 := Low(FSection[False]) to High(FSection[False]) do
      if Assigned(FSection[Loop, Loop2]) then
        FreeMem(FSection[Loop, Loop2]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Active>         Buffer to use
            <SectionNumber>  Section number
  Returns : <Result>         Pointer to section data

  Descript: Get pointer to section data
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbSection.GetSection(Active: Boolean; SectionNumber: Byte): Pointer;
begin
  Result := FSection[Active, SectionNumber];
end;

{------------------------------------------------------------------------------
  Params  : <Active>         Buffer to use
            <SectionNumber>  Section number
            <Value>          Pointer to stream data
  Returns : -

  Descript: Make a copy of section data
  Notes   : If current pointer is valid this memory is released first
            Typically used with a 'nil' assignment to release the memory.
 ------------------------------------------------------------------------------}

procedure TDvbSection.SetSection(Active: Boolean; SectionNumber: Byte; Value: Pointer);
var
  Size: Integer;
begin
  if Assigned(FSection[Active, SectionNumber]) then
  begin
    FreeMem(FSection[Active, SectionNumber]);
    FSection[Active, SectionNumber] := nil;
  end;
  if Value <> nil then
  begin
    Size := ((PByteArray(Value)[1] and $0F) shl 8) or PByteArray(Value)[2] + 3;
    GetMem(FSection[Active, SectionNumber], Size);
    try
      CopyMemory(FSection[Active, SectionNumber], Value, Size);
    except
      FreeMem(FSection[Active, SectionNumber]);
      FSection[Active, SectionNumber] := nil;
    end;
  end;
end;

{------------------------------------------------------------------------------
                                 TDvbSection
 ------------------------------------------------------------------------------}

 {------------------------------------------------------------------------------
                                 TDvbSections
 ------------------------------------------------------------------------------}

 {------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbSections.Create;
var
  Loop : Integer;
  Loop2: Integer;
begin
  inherited Create;
  for Loop := Low(FSections) to High(FSections) do
    for Loop2 := Low(FSections[0]) to High(FSections[0]) do
      FSections[Loop, Loop2] := nil;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbSections.Destroy;
var
  Loop : Integer;
  Loop2: Integer;
begin
  for Loop := Low(FSections) to High(FSections) do
    for Loop2 := Low(FSections[0]) to High(FSections[0]) do
      if Assigned(FSections[Loop, Loop2]) then
        FSections[Loop, Loop2].Free;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Active>         Buffer to use
            <Service>        Service
            <TableId>        Table id
            <SectionNumber>  Section number
  Returns : <Result>         Pointer to section data

  Descript: Get pointer to section data
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbSections.GetSection(Active: Boolean; Service: Word; TableId: Byte; SectionNumber: Byte): Pointer;
begin
  Result := nil;
//  if Service > High(FSections) then
//    Exit;
  if Assigned(FSections[Service, TableId]) then
    Result := FSections[Service, TableId].Section[Active, SectionNumber];
end;

{------------------------------------------------------------------------------
  Params  : <Active>         Buffer to use
            <Service>        Service
            <TableId>        Table id
            <SectionNumber>  Section number
            <Value>          Pointer to stream data
  Returns : -

  Descript: Set pointer to section data
  Notes   : If current pointer is valid this memory is released first
            Typically used with a 'nil' assignment to release the memory.
 ------------------------------------------------------------------------------}

procedure TDvbSections.SetSection(Active: Boolean; Service: Word; TableId: Byte; SectionNumber: Byte; Value: Pointer);
begin
//  if Service > High(FSections) then
//    Exit;
  if not Assigned(FSections[Service, TableId]) then
    FSections[Service, TableId] := TDvbSection.Create;
  FSections[Service, TableId].Section[Active, SectionNumber] := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Service>        Service
            <TableId>        Table id
  Returns : <Result>         Pointer to section data

  Descript: Get version number of data which generated a callback
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbSections.GetLastVersion(Service: Word; TableId: Byte): Byte;
begin
  Result := $FF;
//  if Service > High(FSections) then
//    Exit;
  if Assigned(FSections[Service, TableId]) then
    Result := FSections[Service, TableId].LastVersion;
end;

{------------------------------------------------------------------------------
  Params  : <Service>        Service
            <TableId>        Table id
            <Value>          Pointer to stream data
  Returns : -

  Descript: Set version number of data which generated a callback
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbSections.SetLastVersion(Service: Word; TableId: Byte; Value: Byte);
begin
//  if Service > High(FSections) then
//    Exit;
  if Assigned(FSections[Service, TableId]) then
    FSections[Service, TableId].LastVersion := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Service>        Service
            <TableId>        Table id
  Returns : <Result>         Pointer to section data

  Descript: Get active index
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbSections.GetActiveIndex(Service: Word; TableId: Byte): Boolean;
begin
  Result := False;
//  if Service > High(FSections) then
//    Exit;
  if Assigned(FSections[Service, TableId]) then
    Result := FSections[Service, TableId].ActiveIndex;
end;

{------------------------------------------------------------------------------
  Params  : <Service>        Service
            <TableId>        Table id
            <Value>          Pointer to stream data
  Returns : -

  Descript: Set active index.
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbSections.SetActiveIndex(Service: Word; TableId: Byte; Value: Boolean);
begin
//  if Service > High(FSections) then
//    Exit;
  if Assigned(FSections[Service, TableId]) then
    FSections[Service, TableId].ActiveIndex := Value;
end;

{------------------------------------------------------------------------------
                                 TDvbSections
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbReceivingSection
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbReceivingSection.Create;
begin
  inherited Create;
  FPointerField := 0;
  FStreamData   := nil;
  FDataIndex    := 0;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbReceivingSection.Destroy;
begin
  if Assigned(FStreamData) then
    Dispose(FStreamData);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  :
  Returns : <Result>  Pointer to stream data

  Descript: Get pointer to stream data
  Notes   : Memory is allocated if needed
 ------------------------------------------------------------------------------}

function TDvbReceivingSection.GetStreamData: PDvbAccumulateSection;
begin
  if not Assigned(FStreamData) then
  begin
    FPointerField := 0;
    try
      GetMem(FStreamData, sizeof(TDvbAccumulateSection));
    except
      FStreamData := nil;
    end;
    FDataIndex := 0;
  end;
  Result := FStreamData;
end;

{------------------------------------------------------------------------------
  Params  : <Value>   Pointer to stream data
  Returns : -

  Descript: Set pointer to stream data
  Notes   : If current pointer is valid this memory is disposed first
            Typically used with a 'nil' assignment to release the memory.
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSection.SetStreamData(Value: PDvbAccumulateSection);
begin
  if Assigned(FStreamData) then
    FreeMem(FStreamData);
  FStreamData := Value;
  if Value = nil then
    FDataIndex := 0;
end;

{------------------------------------------------------------------------------
                                 TDvbReceivingSection
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                                 TDvbReceivingSections
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDvbReceivingSections.Create;
var
  Loop: Integer;
begin
  inherited Create;
  for Loop := Low(FDvbReceivingSections) to High(FDvbReceivingSections) do
    FDvbReceivingSections[Loop] := TDvbReceivingSection.Create;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDvbReceivingSections.Destroy;
var
  Loop: Integer;
begin
  for Loop := Low(FDvbReceivingSections) to High(FDvbReceivingSections) do
    FreeAndNil(FDvbReceivingSections[Loop]);
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer field

  Descript: Get pointer field
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetPointerField(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbReceivingSections) then
    Exit;
  Result := FDvbReceivingSections[Index].FPointerField;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Type to set
  Returns : -

  Descript: Set pointer field
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetPointerField(Index: Word; Value: Word);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  FDvbReceivingSections[Index].FPointerField := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer to stream data
                      <nil> if invalid PID id

  Descript: Get stream data
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetStreamData(Index: Word): PDvbAccumulateSection;
begin
  Result := nil;
  if Index > High(FDvbReceivingSections) then
    Exit;
  if Assigned(FDvbReceivingSections[Index].StreamData) then
    Result := FDvbReceivingSections[Index].StreamData;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to stream data
  Returns : -

  Descript: Set stream data
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetStreamData(Index: Word; Value: PDvbAccumulateSection);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  if Assigned(FDvbReceivingSections[Index].StreamData) then
    FDvbReceivingSections[Index].StreamData := Value;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
  Returns : <Result>  Pointer to stream data

  Descript: Get data index
  Notes   :
 ------------------------------------------------------------------------------}

function TDvbReceivingSections.GetDataIndex(Index: Word): Word;
begin
  Result := 0;
  if Index > High(FDvbReceivingSections) then
    Exit;
  Result := FDvbReceivingSections[Index].FDataIndex;
end;

{------------------------------------------------------------------------------
  Params  : <Index>   PID id
            <Value>   Pointer to stream data
  Returns : -

  Descript: Set data index
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDvbReceivingSections.SetDataIndex(Index: Word; Value: Word);
begin
  if Index > High(FDvbReceivingSections) then
    Exit;
  FDvbReceivingSections[Index].FDataIndex := Value;
end;
{------------------------------------------------------------------------------
                                 TDvbReceivingSections
 ------------------------------------------------------------------------------}


{------------------------------------------------------------------------------
  Params  : <ToDecode> 16 bit UTC date
  Returns : <Result>   True if success
            <Year>     Year
            <Month>    Month
            <Day>      Day

  Descript: Convert UTC date
  Notes   :
 ------------------------------------------------------------------------------}
function DvbFilterXmlDecodeUtcDate(ToDecode: Word; var Year, Month, Day: Word): Boolean; stdcall;
var
  Calc : Real;
  YCalc: Real;
  MCalc: Real;
  YInt : Word;
  MInt : Word;
  Date : Word;
begin
  Date := ToDecode;
  Calc := (Date - 15078.2) / 365.25;
  // Catch illegal dates (e.g. if Calc becomes negative)
  try
    Year := Trunc(Calc);
  except
    DecodeDate(Now, Year, Month, Day);
    Exit;
  end;
  YCalc := Year * 365.25;
  YInt := Trunc(YCalc);
  Calc := Date - 14956.1 - YInt;
  Calc := Calc / 30.6001;
  Month := Trunc(Calc);
  MCalc := Month * 30.6001;
  MInt := Trunc(MCalc);
  Calc := Date - 14956 - YInt - MInt;
  Day := Trunc(Calc);
  if (Month = 14) or (Month = 15) then
  begin
    Inc(Year);
    Month := Month - 13;
  end
  else
    Dec(Month);
  Year := Year + 1900;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <ToDecode> 24 bit UTC time
  Returns : <Result>   True if success
            <Hours>    Hours
            <Minutes>  Minutes
            <Seconds>  Seconds

  Descript: Convert UTC time
  Notes   :
 ------------------------------------------------------------------------------}
function DvbFilterXmlDecodeUtcTime(ToDecode: Dword; var Hours, Minutes, Seconds: Word): Boolean; stdcall;
begin
  Hours   := (((ToDecode and $F00000) shr 20) * 10) + ((ToDecode and $0F0000) shr 16);
  Minutes := (((ToDecode and $00F000) shr 12) * 10) + ((ToDecode and $000F00) shr  8);
  Seconds := (((ToDecode and $0000F0) shr  4) * 10) + ((ToDecode and $00000F));
  if (Hours > 23) or (Minutes > 59) or (Seconds > 59) then
  begin
    Hours   := 0;
    Minutes := 0;
    Seconds := 0;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <ToDecode> 40 bit UTC code
  Returns : <Result>   True if success
            <Year>     Year
            <Month>    Month
            <Day>      Day
            <Hours>    Hours
            <Minutes>  Minutes
            <Seconds>  Seconds

  Descript: Convert UTC code
  Notes   :
 ------------------------------------------------------------------------------}
function DvbFilterXmlDecodeUtcCode(ToDecode: Int64; var Year, Month, Day, Hours, Minutes, Seconds: Word): Boolean; stdcall;
begin
  DvbFilterXmlDecodeUtcDate((ToDecode and $FFFF000000) shr 24, Year, Month, Day);
  DvbFilterXmlDecodeUtcTime( ToDecode and     $FFFFFF, Hours, Minutes, Seconds);
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Packet data buffer
  Returns : <Result>            True if valid Pid (no other errors detected)
            <Pid>               Pid value

  Descript: Get PID from transport stream.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetPid(StreamPacketData: Pointer; var Pid: Word): Boolean; stdcall;
begin
  Result := False;
  // One of the few checks at this point is the synchronization byte which must be correct
  if PByteArray(StreamPacketData)[0] = CDvbPacketSyncWord then
    Result := True
  else if PacketSyncErrors = $FFFF then
    PacketSyncErrors := 0
  else
    Inc(PacketSyncErrors);
  Pid := ((PByteArray(StreamPacketData)[1] and $1F) shl 8) or PByteArray(StreamPacketData)[2];
end;

{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Packet data buffer
  Returns : <Result>            True if valid offset
                                False if error or no payload
            <Offset>            Offset into first data byte of payload

  Descript: Get offset where payload starts.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbGetPayloadOffset(StreamPacketData: Pointer; var Offset:  Word): Boolean;
var
  AdaptationFieldControl: Byte;
begin
  Result := False;
  Offset := 0;
  // The start of the payload depends on the adaptation field. If there is
  // an adaptation field then the payload starts after this. If there is no
  // adaptation field then the payload start immediately.
  // Also there is the posibility that there is no payload at all.
  AdaptationFieldControl := PByteArray(StreamPacketData)[3] and $30;
  case AdaptationFieldControl of
    $00: Exit; // Reserved
    $10: Offset := 4; // Payload only
    $20: Exit; // Adaptation field only, no payload
    $30: begin
           Offset := PByteArray(StreamPacketData)[4] + 5; // Adaptation field with payload
           // Check for excessive value
           if Offset > CDvbPacketSize then
           begin
             Offset := 0;
             Exit;
           end;
         end;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
                             XML support start
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : <StringIn>  String to process
  Returns : <Result>    Processed string;

  Descript: Change special XML notation into normal characters ('&#...;')
  Notes   :
------------------------------------------------------------------------------}

function DvbFilterXmlXmlCorrectString(StringIn: AnsiString): AnsiString;
var
  ThePos   : Integer;
  Error    : Integer;
  Check    : AnsiString;
  ToProcess: AnsiString;
  Processed: AnsiString;
  Changed  : Integer;
  TheChar  : Integer;
begin
  ToProcess := StringIn;
  Processed := '';
  repeat
    ThePos := Pos('&#', ToProcess);
    if ThePos <> 0 then
    begin
      if ThePos > 1 then
      begin
        Processed := Processed + Copy(ToProcess, 1, ThePos-1);
        Delete(ToProcess, 1, ThePos-1);
      end;
      Check := Copy(ToProcess, 1, 10);
      Changed := Pos(';', Check);
      if Changed <> 0 then
      begin
        Check := Copy(ToProcess, 3, Changed-3);
        repeat
          ThePos := Pos('x', Check);
          if ThePos <> 0 then
            Check[ThePos] := '$';
        until ThePos = 0;
        Delete(ToProcess, 1, Changed);
        Val(Check, TheChar, Error);
        if Error <> 0 then
          TheChar := 0;
        Processed := Processed + Chr(Lo(TheChar));
      end
      else
      begin
        Processed := Processed + Copy(ToProcess, 1, 2);
        Delete(ToProcess, 1, 2);
      end;
    end
    else
    begin
      Processed := Processed + ToProcess;
      ToProcess := '';
    end;
  until ToProcess = '';
  Result := Processed;
end;

{------------------------------------------------------------------------------
  Params  : <Transform>  String to convert
  Returns : <Result>     Converted string

  Descript: Check and modify string for possible XML problems, meaning some
            characters are translated into 'coded' form (&#..;)
  Notes   : Invalid characters (from a XML point of view for strings) are -NOT-
            taken into account here!
 ------------------------------------------------------------------------------}
function DvbXmlTransformStringAsAscii(Transform: AnsiString): AnsiString;
var
  Loop: Integer;
begin
  Result := '';
  if Transform <> '' then
    for Loop := 1 to Length(Transform) do
      case Transform[Loop] of
        #$00..#$1F, '"', '''', '&', '<':
             Result := Result + format('&#x%x;', [Ord(Transform[Loop])]);
        else Result := Result + Transform[Loop];
      end;
end;

{------------------------------------------------------------------------------
  Params  : <Transform>  String to convert
  Returns : <Result>     Converted string

  Descript: Convert each character into a 2-byte hexadecimal ASCII
            equivalent.
  Notes   :
 ------------------------------------------------------------------------------}
function DvbXmlTransformStringAsBinary(Transform: AnsiString): AnsiString;
var
  Loop: Integer;
begin
  Result := '';
  if Transform <> '' then
    for Loop := 1 to Length(Transform) do
      Result := Result + format('%2.2x', [Ord(Transform[Loop])]);
end;

{------------------------------------------------------------------------------
  Params  : <ToDecode>    String to convert
            <XmlConvert>  True will also convert '&#..' notation to a single
                          character
  Returns : <Result>      Converted string

  Descript: Convert a 'binary' (2 hexadecimal ASCII character string) to a
            normal string.
  Notes   :
 ------------------------------------------------------------------------------}
function DvbFilterXmlDecodeBinaryStringToString(ToDecode: AnsiString; XmlConvert: Boolean): AnsiString;
var
  Index   : Integer;
  Loop    : Integer;
  Loop2   : Integer;
  Value   : Byte;
begin
  Result := '';
  Loop   := 1;
  for Index := 1 to (Length(ToDecode) div 2) do
  begin
    Value := 0;
    for Loop2 := 0 to 1 do
    begin
      Value := Value shl 4;
      if ToDecode[Loop+Loop2] in ['0'..'9'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('0'));
      if ToDecode[Loop+Loop2] in ['A'..'F'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('A')+10);
      if ToDecode[Loop+Loop2] in ['a'..'f'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('a')+10);
    end;
    Result := Result + Chr(Value);
    Inc(Loop, 2);
  end;
  if XmlConvert then
    Result := DvbFilterXmlXmlCorrectString(result);
end;

{------------------------------------------------------------------------------
  Params  : <Transform>  String to convert
  Returns : <Result>     Converted string

  Descript: Convert a 'binary' (2 hexadecimal ASCII character string) to a
            normal XML string. Non-conformative XML characters are excluded.
  Notes   : Does not include the text delimiters
 ------------------------------------------------------------------------------}
function DvbFilterXmlDecodeBinaryStringToXmlString(ToDecode: AnsiString): AnsiString;
var
  Index   : Integer;
  Loop    : Integer;
  Loop2   : Integer;
  Value   : Byte;
begin
  Result := '';
  Loop   := 1;
  for Index := 1 to (Length(ToDecode) div 2) do
  begin
    Value := 0;
    for Loop2 := 0 to 1 do
    begin
      Value := Value shl 4;
      if ToDecode[Loop+Loop2] in ['0'..'9'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('0'));
      if ToDecode[Loop+Loop2] in ['A'..'F'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('A')+10);
      if ToDecode[Loop+Loop2] in ['a'..'f'] then
        Value := Value or (Ord(ToDecode[Loop+Loop2])-Ord('a')+10);
    end;
    if not (Value in [$00..$08, $0B..$0C, $0E..$1F]) then
      Result := Result + Chr(Value);
    Inc(Loop, 2);
  end;
  Result := DvbXmlTransformStringAsAscii(Result);
end;

{------------------------------------------------------------------------------
  Params  : <ForceBinary>  True if forced binary conversion required
            <Transform>    String to convert
  Returns : <Result>       Converted string

  Descript: Check and modify string for possible XML problems. If there are
            illegal characters (from an XML point of view) in the string
            then each character is translated into it's ASCII 2-digit hex number
            equivalent. Each 2 characters then represent a single byte.
  Notes   : Valid XML characters are:
            $09, $0A, $0D, $20..$FF
 ------------------------------------------------------------------------------}
function DvbXmlTransformString(ForceBinary: Boolean; Transform: AnsiString): AnsiString;
var
  Loop     : Integer;
  Compliant: Boolean;
begin
  // Check for compliance
  Compliant := not ForceBinary;
  Loop := 1;
  while Compliant and (Loop <= Length(Transform)) do
  begin
    if Transform[Loop] in [#$00..#$08, #$0B..#$0C, #$0E..#$1F] then
      Compliant := False;
    Inc(Loop);
  end;
  if Compliant then
    Result := DvbXmlTransformStringAsAscii(Transform)
  else
    Result := DvbXmlTransformStringAsBinary(Transform);
end;

{------------------------------------------------------------------------------
                             XML support end
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                             Table decoders start
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
            <NoHeaders>   Suppresses XML headers (for decoding into existing XML header)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAc3Type(Identifier: Byte; Indent: AnsiString; NoHeaders: Boolean): AnsiString; stdcall;
begin
  Result := '';
  if not NoHeaders then
    Result := format('%s%s<ac3_type text="', [Result, Indent]);
  if (Identifier and $80) <> 0 then
    Result := Result + 'reserved'
  else
  begin
    if (Identifier and $40) <> 0 then
      Result := Result + 'decoded audio stream is a full service (suitable for decoding and presentation to the listener)'
    else
      Result := Result + 'decoded audio stream is intended to be combined with another decoded audio stream before presentation to the listener)';
    case (Identifier and $07) of
      $00: Result := Result + ', mono';
      $01: Result := Result + ', 1=1 mode';
      $02: Result := Result + ', 2 channel (stereo)';
      $03: Result := Result + ', 2 channel dolby surround encoded (stereo)';
      $04: Result := Result + ', multichannel audio (>2 channels)';
      $05..$07:
           Result := Result + ', reserved';
    end;
    case (Identifier and $78) of
      $40: Result := Result + ', complete main (cm)';
      $08: Result := Result + ', music and effects (me)';
      $10, $50:
           Result := Result + ', visually impaired (vi))';
      $18, $58:
           Result := Result + ', hearing impaired (hi))';
      $20: Result := Result + ', dialogue (d)';
      $78: Result := Result + ', karaoke (mono and 1+1 prohibited)';
    end;
    case (Identifier and $7F) of
      $28, $68:
           Result := Result + ', commentary (c)';
      $70: Result := Result + ', emergency (e)';
      $38: Result := Result + ', voiceover (vo)';
    end;
  end;
  if not NoHeaders then
    Result := format('%s">%d</ac3_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
            <NoHeaders>   Suppresses XML headers (for decoding into existing XML header)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAc3Substream(Identifier: Byte; Substream: Byte; Indent: AnsiString; NoHeaders: Boolean): AnsiString; stdcall;
begin
  Result := '';
  if not NoHeaders then
    Result := format('%s%s<substream%d text="', [Result, Indent, Substream]);
  if (Identifier and $80) <> 0 then
    Result := Result + 'Mixing metadata present in substream'
  else
    Result := Result + 'No mixing metadata present in substream';
  case (Identifier and $C0) of
    $40:      Result := Result + ', main service';
    $00, $80: Result := Result + ', associated service';
  end;
  case (Identifier and $07) of
    $00: Result := Result + ', mono';
    $01: Result := Result + ', 1=1 mode';
    $02: Result := Result + ', 2 channel (stereo)';
    $03: Result := Result + ', 2 channel dolby surround encoded (stereo)';
    $04: Result := Result + ', multichannel audio (>2 channels)';
    $05: Result := Result + ', multichannel audio (>5.1 channels)';
    $06..$07:
         Result := Result + ', reserved';
  end;
  case (Identifier and $78) of
    $40: if (Identifier and $80) = 0 then
           Result := Result + ', complete main (cm)';
    $08: Result := Result + ', music and effects (me)';
    $10, $50:
         Result := Result + ', visually impaired (vi))';
    $18, $58:
         Result := Result + ', hearing impaired (hi))';
    $20: Result := Result + ', dialogue (d)';
    $78: Result := Result + ', karaoke (mono and 1+1 prohibited)';
  end;
  case (Identifier and $7F) of
    $28, $68:
         Result := Result + ', commentary (c)';
    $70: if (Identifier and $80) = 0 then
           Result := Result + ', emergency (e)';
    $38: Result := Result + ', voiceover (vo)';
  end;
  if not NoHeaders then
    Result := format('%s">%d</substream%d>%s', [Result, Identifier, Substream, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAdaptationFieldDataIdentifier(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
var
  New  : Boolean;
  Shift: Byte;
  Loop : Integer;
begin
  Result := '';
  Result := format('%s%s<adaptation_field_data_identifier text="', [Result, Indent]);
  New   := True;
  Shift := $01;
  if Identifier = 0 then
    Result := Result + 'reserved for future use';
  for Loop := 0 to 0 do
  begin
    if (Identifier and Shift) <> 0 then
    begin
      if not New then
        Result := Result + ', ';
      case Loop of
        0: Result := Result + 'announcement switching data field';
      end;
      New := False;
    end;
    Shift := Shift shl 1;
  end;
  Result := format('%s">%d</adaptation_field_data_identifier>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAlignmentType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<alignment_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'reserved';
    $01: Result := Result + 'video stream: slice, or video access unit  -  audio stream: syncword';
    $02: Result := Result + 'video stream: video access unit  -  audio stream: reserved';
    $03: Result := Result + 'video stream: gop, or seq  -  audio stream: reserved';
    $04: Result := Result + 'video stream: seq  -  audio stream: reserved';
  end;
  Result := format('%s">%d</alignment_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAncillaryDataIdentifier(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
var
  New  : Boolean;
  Shift: Byte;
  Loop : Integer;
begin
  Result := '';
  Result := format('%s%s<ancillary_data_identifier text="', [Result, Indent]);
  New   := True;
  Shift := $01;
  if Identifier = 0 then
    Result := Result + 'reserved for future use';
  for Loop := 0 to 4 do
  begin
    if (Identifier and Shift) <> 0 then
    begin
      if not New then
        Result := Result + ', ';
      case Loop of
        0: Result := Result + 'dvd-video ancillary data';
        1: Result := Result + 'extended ancillary data';
        2: Result := Result + 'announcement switching data';
        3: Result := Result + 'dab ancillary data';
        4: Result := Result + 'scale factor error check (scf-crc)';
      end;
      New := False;
    end;
    Shift := Shift shl 1;
  end;
  Result := format('%s">%d</ancillary_data_identifier>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAnnouncementSupportIndicator(Identifier: Word; Indent: AnsiString): AnsiString; stdcall;
var
  New  : Boolean;
  Shift: Word;
  Loop : Integer;
begin
  Result := '';
  Result := format('%s%s<announcement_support_indicator text="', [Result, Indent]);
  New   := True;
  Shift := $01;
  if Identifier = 0 then
    Result := Result + 'reserved for future use';
  for Loop := 0 to 7 do
  begin
    if (Identifier and Shift) <> 0 then
    begin
      if not New then
        Result := Result + ', ';
      case Loop of
        0: Result := Result + 'emergency alarm';
        1: Result := Result + 'road traffic flash';
        2: Result := Result + 'public transport flash';
        3: Result := Result + 'warning message';
        4: Result := Result + 'news flash';
        5: Result := Result + 'weather flash';
        6: Result := Result + 'event announcement';
        7: Result := Result + 'personal call';
      end;
      New := False;
    end;
    Shift := Shift shl 1;
  end;
  Result := format('%s">%d</announcement_support_indicator>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAnnouncementType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<announcement_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'emergency traffic';
    $01: Result := Result + 'road traffic flash';
    $02: Result := Result + 'public transport flash';
    $03: Result := Result + 'warning message';
    $04: Result := Result + 'news flash';
    $05: Result := Result + 'weather flash';
    $06: Result := Result + 'event announcement';
    $07: Result := Result + 'personal call';
    $08..$0F:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</announcement_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableAudioType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<audio_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'undefined';
    $01: Result := Result + 'clean effects';
    $02: Result := Result + 'hearing impaired';
    $03: Result := Result + 'visual impaired commentary';
    $04..$FF:
         Result := Result + 'reserved';
  end;
  Result := format('%s">%d</audio_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableBandwidth(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<bandwidth text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + '8 mhz';
    $01: Result := Result + '7 mhz';
    $02: Result := Result + '6 mhz';
    $03: Result := Result + '5 mhz';
    $04..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</bandwidth>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCaSystemId(Identifier: Word; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<ca_system_id text="', [Result, Indent]);
  case Identifier of
    $0000:        Result := Result + 'reserved';
    $0001..$00FF: Result := Result + 'standardized systems';
    $0100..$01FF: Result := Result + 'canal plus';
    $0200..$02FF: Result := Result + 'ccett';
    $0300..$03FF: Result := Result + 'deutsche telecom';
    $0400..$04FF: Result := Result + 'eurodec';
    $0500..$05FF: Result := Result + 'france telecom';
    $0600..$06FF: Result := Result + 'irdeto';
    $0700..$07FF: Result := Result + 'jerrold/gi';
    $0800..$08FF: Result := Result + 'matra communication';
    $0900..$09FF: Result := Result + 'news datacom';
    $0A00..$0AFF: Result := Result + 'nokia';
    $0B00..$0BFF: Result := Result + 'norwegian telekom';
    $0C00..$0CFF: Result := Result + 'ntl';
    $0D00..$0DFF: Result := Result + 'philips';
    $0E00..$0EFF: Result := Result + 'scientific atlanta';
    $0F00..$0FFF: Result := Result + 'sony';
    $1000..$10FF: Result := Result + 'tandberg television';
    $1100..$11FF: Result := Result + 'thomson';
    $1200..$12FF: Result := Result + 'tv/com';
    $1300..$13FF: Result := Result + 'hpt - croatian post and telecommunications';
    $1400..$14FF: Result := Result + 'hrt - croatian radio and television';
    $1500..$15FF: Result := Result + 'ibm';
    $1600..$16FF: Result := Result + 'nera';
    $1700..$17FF: Result := Result + 'betatechnik';
    $1800..$18FF: Result := Result + 'kudelski sa';
    $1900..$19FF: Result := Result + 'titan information systems';
    $2000..$20FF: Result := Result + 'telefonica servicios audiovisuales';
    $2100..$21FF: Result := Result + 'stentor (france telecom, cnes and dga)';
    $2200..$22FF: Result := Result + 'tadiran scopus';
    $2300..$23FF: Result := Result + 'barcos as';
    $2400..$24FF: Result := Result + 'starguide digital networks';
    else          Result := Result + 'unknown';
  end;
  Result := format('%s">%d</ca_system_id>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCellLinkageInfo(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<cell_linkage_info text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'undefined';
    $01: Result := Result + 'bouquet related';
    $02: Result := Result + 'service related';
    $03: Result := Result + 'other mosaic related';
    $04: Result := Result + 'event related';
    $05..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</cell_linkage_info>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <AddText>     Additional text header
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCodeRate(Identifier: Byte; AddText: AnsiString; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<code_rate%s text="', [Result, AddText, Indent]);
  case Identifier of
    $00: Result := Result + '1/2';
    $01: Result := Result + '2/3';
    $02: Result := Result + '3/4';
    $03: Result := Result + '5/6';
    $04: Result := Result + '7/8';
    $05..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</code_rate>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCodingType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<coding_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'not defined';
    $01: Result := Result + 'satellite';
    $02: Result := Result + 'cable';
    $03: Result := Result + 'terrestrial';
  end;
  Result := format('%s">%d</coding_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Content>    Stream content identifier
            <Component>  Component type
            <NoHeader>   Suppresses headers (for other types using same data)
            <Indent>     'Header' text (typically indent)
  Returns : <Result>     Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableComponentType(Content: Byte; Component: Byte; NoHeader: Boolean; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not NoHeader then
    Result := format('%s%s<component_type text="', [Result, Indent]);
  case Content of
    $00: Result := Result + 'reserved for future use';
    $01: case Component of
           $00, $11..$AF, $FF:
                Result := Result + 'reserved for future use';
           $01: Result := Result + 'video, 4:3 aspect ratio, 25 hz';
           $02: Result := Result + 'video, 16:9 aspect ratio with pan vectors, 25 hz';
           $03: Result := Result + 'video, 16:9 aspect ratio without pan vectors, 25 hz';
           $04: Result := Result + 'video, > 16:9 aspect ratio, 25 hz';
           $05: Result := Result + 'video, 4:3 aspect ratio, 30 hz';
           $06: Result := Result + 'video, 16:9 aspect ratio with pan vectors, 30 hz';
           $07: Result := Result + 'video, 16:9 aspect ratio without pan vectors, 30 hz';
           $08: Result := Result + 'video, > 16:9 aspect ratio, 30 hz';
           $09: Result := Result + 'high definition video, 4:3 aspect ratio, 25 hz';
           $0A: Result := Result + 'high definition video, 16:9 aspect ratio with pan vectors, 25 hz';
           $0B: Result := Result + 'high definition video, 16:9 aspect ratio without pan vectors, 25 hz';
           $0C: Result := Result + 'high definition video, > 16:9 aspect ratio, 25 hz';
           $0D: Result := Result + 'high definition video, 4:3 aspect ratio, 30 hz';
           $0E: Result := Result + 'high definition video, 16:9 aspect ratio with pan vectors, 30 hz';
           $0F: Result := Result + 'high definition video, 16:9 aspect ratio without pan vectors, 30 hz';
           $10: Result := Result + 'high definition video, > 16:9 aspect ratio, 30 hz';
           $B0..$FE:
                Result := Result + 'user defined';
         end;
    $02: case Component of
           $00, $06..$3F, $43..$AF, $FF:
                Result := Result + 'reserved for future use';
           $01: Result := Result + 'audio, single mono channel';
           $02: Result := Result + 'audio, dual mono channel';
           $03: Result := Result + 'audio, stereo (2 channel)';
           $04: Result := Result + 'audio, multi-lingual, multi-channel';
           $05: Result := Result + 'audio, surround sound';
           $40: Result := Result + 'audio description for the visually impaired';
           $41: Result := Result + 'audio for the hard of hearing';
           $42: Result := Result + 'receiver-mixed supplementary audio as per annex g of tr 101 154';
           $B0..$FE:
                Result := Result + 'user defined';
         end;
    $03: case Component of
           $00, $04..$0F, $14..$1F, $24..$AF, $FF:
                Result := Result + 'reserved for future use';
           $01: Result := Result + 'ebu teletext subtitles';
           $02: Result := Result + 'associated ebu teletext';
           $03: Result := Result + 'vbi data';
           $10: Result := Result + 'dvb subtitles (normal) with no monitor aspect ratio critically';
           $11: Result := Result + 'dvb subtitles (normal) for display on 4:3 aspect ratio monitor';
           $12: Result := Result + 'dvb subtitles (normal) for display on 16:9 aspect ratio monitor';
           $13: Result := Result + 'dvb subtitles (normal) for display on 2.21:1 aspect ratio monitor';
           $20: Result := Result + 'dvb subtitles (for the hard of hearing) with no monitor aspect ratio critically';
           $21: Result := Result + 'dvb subtitles (for the hard of hearing) for display on 4:3 aspect ratio monitor';
           $22: Result := Result + 'dvb subtitles (for the hard of hearing) for display on 16:9 aspect ratio monitor';
           $23: Result := Result + 'dvb subtitles (for the hard of hearing) for display on 2.21:1 aspect ratio monitor';
           $B0..$FE:
                Result := Result + 'user defined';
         end;
    $04: case Component of
           $00..$7F:
                Result := format('%sac3 audio mode: %s', [Result, DvbDecodeTableAc3Type(Component, '', True)]);
           $80..$FF:
                Result := Result + 'reserved for future use';
         end;
    $05..$0B:
         Result := Result + 'reserved for future use';
    $0C..$0F:
         Result := Result + 'user defined';
  end;
  if not NoHeader then
    Result := format('%s">%d</component_type>%s', [Result, Component, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableConstellation(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<constellation text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'qpsk';
    $01: Result := Result + '16-qam';
    $02: Result := Result + '64-qam';
    $03: Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</constellation>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableContentNibbleLevel1(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<content_nibble_level_1 text="', [Result, Indent]);
  case Identifier of
    $0: Result := Result + 'undefined content';
    $1: Result := Result + 'movie/drama';
    $2: Result := Result + 'news/current affairs';
    $3: Result := Result + 'show/game show';
    $4: Result := Result + 'sports';
    $5: Result := Result + 'children/youth programmes';
    $6: Result := Result + 'music/ballet/dance';
    $7: Result := Result + 'arts/culture (without music)';
    $8: Result := Result + 'social/political issues/economics';
    $9: Result := Result + 'education/science/factual topics';
    $A: Result := Result + 'leisure hobbies';
    $B: Result := Result + 'special characteristics';
    $C..$E:
         Result := Result + 'reserved for future use';
    $F: Result := Result + 'user defined';
  end;
  Result := format('%s">%d</content_nibble_level_1>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>    Identifier status data
            <Level1Nibble>  Nibble for level 1
            <Indent>        'Header' text (typically indent)
  Returns : <Result>        Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableContentNibbleLevel2(Identifier: Byte; Level1Nibble: Integer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<content_nibble_level_2 text="', [Result, Indent]);
  case Level1Nibble of
    $0: Result := Result + 'undefined content';
    $1: case Identifier of
          $0: Result := Result + 'movie/drama (general)';
          $1: Result := Result + 'detective/thriller';
          $2: Result := Result + 'adventure/western/war';
          $3: Result := Result + 'science fiction/fantasy/horror';
          $4: Result := Result + 'comedy';
          $5: Result := Result + 'soap/melodrama/folkloric';
          $6: Result := Result + 'romance';
          $7: Result := Result + 'serious/classical/religious/historical movie/drama';
          $8: Result := Result + 'adult movie/drama';
          $9..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $2: case Identifier of
          $0: Result := Result + 'news/current affairs (general)';
          $1: Result := Result + 'news/weather report';
          $2: Result := Result + 'news magazine';
          $3: Result := Result + 'documentary';
          $4: Result := Result + 'discussion/interview/debate';
          $5..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $3: case Identifier of
          $0: Result := Result + 'show/game show (general)';
          $1: Result := Result + 'game show/quiz/contest';
          $2: Result := Result + 'variety show';
          $3: Result := Result + 'talk show';
          $4..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $4: case Identifier of
          $0: Result := Result + 'sports (general)';
          $1: Result := Result + 'special events (olympic games, world cup, etc.)';
          $2: Result := Result + 'sports magazines';
          $3: Result := Result + 'football/soccer';
          $4: Result := Result + 'tennis/squash';
          $5: Result := Result + 'team sports (excluding football)';
          $6: Result := Result + 'athletics';
          $7: Result := Result + 'motor sport';
          $8: Result := Result + 'water sport';
          $9: Result := Result + 'winter sports';
          $A: Result := Result + 'equestrian';
          $B: Result := Result + 'martial sports';
          $C..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $5: case Identifier of
          $0: Result := Result + 'childrens/youth programmes (general)';
          $1: Result := Result + 'pre-school childrens programmes';
          $2: Result := Result + 'entertainment programmes for 6 to14';
          $3: Result := Result + 'entertainment programmes for 10 to 16';
          $4: Result := Result + 'informational/educational/school programmes';
          $5: Result := Result + 'cartoons/puppets';
          $6..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $6: case Identifier of
          $0: Result := Result + 'music/ballet/dance (general)';
          $1: Result := Result + 'rock/pop';
          $2: Result := Result + 'serious music/classical music';
          $3: Result := Result + 'folk/traditional music';
          $4: Result := Result + 'jazz';
          $5: Result := Result + 'musical/opera';
          $6: Result := Result + 'ballet';
          $7..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $7: case Identifier of
          $0: Result := Result + 'arts/culture (without music, general)';
          $1: Result := Result + 'performing arts';
          $2: Result := Result + 'fine arts';
          $3: Result := Result + 'religion';
          $4: Result := Result + 'popular culture/traditional arts';
          $5: Result := Result + 'literature';
          $6: Result := Result + 'film/cinema';
          $7: Result := Result + 'experimental film/video';
          $8: Result := Result + 'broadcasting/press';
          $9: Result := Result + 'new media';
          $A: Result := Result + 'arts/culture magazines';
          $B: Result := Result + 'fashion';
          $C..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $8: case Identifier of
          $0: Result := Result + 'social/political issues/economics (general)';
          $1: Result := Result + 'magazines/reports/documentary';
          $2: Result := Result + 'economics/social advisory';
          $3: Result := Result + 'remarkable people';
          $4..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $9: case Identifier of
          $0: Result := Result + 'education/science/factual topics (general)';
          $1: Result := Result + 'nature/animals/environment';
          $2: Result := Result + 'technology/natural sciences';
          $3: Result := Result + 'medicine/physiology/psychology';
          $4: Result := Result + 'foreign countries/expeditions';
          $5: Result := Result + 'social/spiritual sciences';
          $6: Result := Result + 'further education';
          $7: Result := Result + 'languages';
          $8..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $A: case Identifier of
          $0: Result := Result + 'leisure hobbies (general)';
          $1: Result := Result + 'tourism/travel';
          $2: Result := Result + 'handicraft';
          $3: Result := Result + 'motoring';
          $4: Result := Result + 'fitness and health';
          $5: Result := Result + 'cooking';
          $6: Result := Result + 'advertisement/shopping';
          $7: Result := Result + 'gardening';
          $8..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $B: case Identifier of
          $0: Result := Result + 'original language';
          $1: Result := Result + 'black and white';
          $2: Result := Result + 'unpublished';
          $3: Result := Result + 'live broadcast';
          $4..$E:
              Result := Result + 'reserved for future use';
          $F: Result := Result + 'user defined';
        end;
    $C..$E:
        Result := Result + 'reserved for future use';
    $F: Result := Result + 'user defined';
  end;
  Result := format('%s">%d</content_nibble_level_2>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCountryRegionId(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<country_region_id text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'no time zone extension used';
    $01..$3C:
         Result := format('%stime zone %d', [Result, Identifier]);
    $3D..$3F:
         Result := Result + 'reserved';
  end;
  Result := format('%s">%d</country_region_id>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableCountryCode(Identifier: Integer; Indent: AnsiString): AnsiString; stdcall;
var
  Name: AnsiString;
begin
  Result := '';
  Result := format('%s%s<country_code text="', [Result, Indent]);
  Name := Chr((Identifier and $FF0000) shr 16) + Chr((Identifier and $FF00) shr 8) + Chr(Identifier and $FF);
  Result := format('%s%s', [Result, DvbXmlTransformString(False, Name)]);
  Result := format('%s">%d</country_code>%s',  [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableDataServiceId(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<data_service_id text="', [Result, Indent]);
  case Identifier of
    $00, $08..$FF:
         Result := Result + 'reserved for future use';
    $01: Result := Result + 'ebu teletext';
    $02: Result := Result + 'inverted teletext';
    $03: Result := Result + 'reserved';
    $04: Result := Result + 'vps';
    $05: Result := Result + 'wss';
    $06: Result := Result + 'closed captioning';
    $07: Result := Result + 'monochrome 4:2:2 samples';
  end;
  Result := format('%s">%d</data_service_id>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableFecInner(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<fec_inner text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'not defined';
    $01: Result := Result + '1/2 conv. code rate';
    $02: Result := Result + '2/3 conv. code rate';
    $03: Result := Result + '3/4 conv. code rate';
    $04: Result := Result + '5/6 conv. code rate';
    $05: Result := Result + '7/8 conv. code rate';
    $06: Result := Result + '8/9 conv. code rate';
    $0F: Result := Result + 'no conv. coding';
    $07..$0E:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</fec_inner>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableFecOuter(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<fec_outer text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'not defined';
    $01: Result := Result + 'no outer fec coding';
    $02: Result := Result + 'rs (204/188)';
    $03..$0F:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</fec_outer>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableFrameRateCode(Identifier: Byte; Identifier2: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<frame_rate_code text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'forbidden';
    $01: Result := Result + '23,976';
    $02: Result := Result + '24,0';
    $03: Result := Result + '25,0';
    $04: Result := Result + '29,970';
    $05: Result := Result + '30,0';
    $06: Result := Result + '50,0';
    $07: Result := Result + '59,940';
    $08: Result := Result + '60,0';
    $09..$0F:
         Result := Result + 'reserved';
  end;
  if Identifier2 <> 0 then
  case Identifier of
    $02: Result := Result + ' (or 23,976)';
    $04: Result := Result + ' (or 23,976)';
    $05: Result := Result + ' (or 23,976 24,0 29,970)';
    $06: Result := Result + ' (or 25,0)';
    $07: Result := Result + ' (or 23,976 29,970)';
    $08: Result := Result + ' (or 23,976 24,0 29,970 30,0 59,940)';
  end;

  Result := format('%s">%d</frame_rate_code>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableGuardInterval(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<guard_interval text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + '1/32';
    $01: Result := Result + '1/16';
    $02: Result := Result + '1/8';
    $03: Result := Result + '1/4';
  end;
  Result := format('%s">%d</guard_interval>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableHandOverType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<handover_type text="', [Result, Indent]);
  case Identifier of
    $00, $04..$0F:
         Result := Result + 'reserved for future use';
    $01: Result := Result + 'dvb hand-over to an identical service in neighbouring country';
    $02: Result := Result + 'dvb hand-over to a local variation of the same service';
    $03: Result := Result + 'dvb hand-over to an associated service';
  end;
  Result := format('%s">%d</handover_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableHierarchyInformation(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<hierarchy_information text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'non-hierarchical, native interleaver';
    $01: Result := Result + 'alpha = 1, native interleaver';
    $02: Result := Result + 'alpha = 2, native interleaver';
    $03: Result := Result + 'alpha = 4, native interleaver';
    $04: Result := Result + 'non-hierarchical, in-depth interleaver';
    $05: Result := Result + 'alpha = 1, in-depth interleaver';
    $06: Result := Result + 'alpha = 2, in-depth interleaver';
    $07: Result := Result + 'alpha = 4, in-depth interleaver';
  end;
  Result := format('%s">%d</hierarchy_information>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableHierarchyType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<hierarchy_type text="', [Result, Indent]);
  case Identifier of
    $00, $07..$0E:
         Result := Result + 'reserved';
    $01: Result := Result + 'itu-t rec. h.262  iso/iec 13818-2 spatial scalability';
    $02: Result := Result + 'itu-t rec. h.262  iso/iec 13818-2 snr scalability';
    $03: Result := Result + 'itu-t rec. h.262  iso/iec 13818-2 temporal scalability';
    $04: Result := Result + 'itu-t rec. h.262  iso/iec 13818-2 data partitioning';
    $05: Result := Result + 'iso/iec 13818-3 extension bitstream';
    $06: Result := Result + 'itu-t rec. h.222  iso/iec 13818-1 private stream';
    $0F: Result := Result + 'base layer';
  end;
  Result := format('%s">%d</hierarchy_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableIso639LanguageCode(Identifier: Integer; Indent: AnsiString): AnsiString; stdcall;
var
  Name: AnsiString;
begin
  Result := '';
  Result := format('%s%s<iso_639_language_code text="', [Result, Indent]);
  Name :=  Chr((Identifier and $FF0000) shr 16) + Chr((Identifier and $FF00) shr 8) + Chr(Identifier and $FF);
  Result := format('%s%s', [Result, DvbXmlTransformString(False, Name)]);
  Result := format('%s">%d</iso_639_language_code>%s',  [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableLinkageType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<linkage_type text="', [Result, Indent]);
  case Identifier of
    $00, $0D..$7F, $FF:
         Result := Result + 'reserved for future use';
    $01: Result := Result + 'information service';
    $02: Result := Result + 'epg service';
    $03: Result := Result + 'ca replacement service';
    $04: Result := Result + 'ts containing complete network/bouquet si';
    $05: Result := Result + 'service replacement service';
    $06: Result := Result + 'data broadcast service';
    $07: Result := Result + 'rcs map';
    $08: Result := Result + 'mobile hand-over';
    $09: Result := Result + 'system software update service';
    $0A: Result := Result + 'ts containing bat or nit';
    $0B: Result := Result + 'ip/mac notification service';
    $0C: Result := Result + 'ts containing int bat or nit';
    $80..$FE:
         Result := Result + 'user defined';
  end;
  Result := format('%s">%d</linkage_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableLogicalCellPresentationInfo(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<logical_cell_presentation_info text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'undefined';
    $01: Result := Result + 'video';
    $02: Result := Result + 'still picture';
    $03: Result := Result + 'graphics/text';
    $04..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</logical_cell_presentation_info>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableModulationCable(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<modulation text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'not defined';
    $01: Result := Result + '16-qam';
    $02: Result := Result + '32-qam';
    $03: Result := Result + '64-qam';
    $04: Result := Result + '128-qam';
    $05: Result := Result + '256-qam';
    $06..$FF:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</modulation>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableModulationSatellite(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<modulation text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'not defined';
    $01: Result := Result + 'qpsk';
    $02: Result := Result + '8psk';
    $03: Result := Result + '16-qam';
    $04..$1F:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</modulation>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableOriginType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<origin_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'nit';
    $01: Result := Result + 'sdt';
  end;
  Result := format('%s">%d</origin_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTablePolarization(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<polarization text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'linear - horizontal';
    $01: Result := Result + 'linear - vertical';
    $02: Result := Result + 'circular - left';
    $03: Result := Result + 'circular - right';
  end;
  Result := format('%s">%d</polarization>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTablePriority(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<priority text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'lp (low priority)';
    $01: Result := Result + 'hp (high priority)';
  end;
  Result := format('%s">%d</priority>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableRating(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<rating text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'undefined';
    $01..$0F:
         Result := format('%sminimum age = %d', [Result, Identifier + 3]);
    $10..$FF:
         Result := Result + 'defined by the broadcaster';
  end;
  Result := format('%s">%d</rating>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableReferenceType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<reference_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'announcement is broadcast in the usual audio stream of the service';
    $01: Result := Result + 'announcement is broadcast in a separate audio stream that is part of the service';
    $02: Result := Result + 'announcement is broadcast by means of a different service within the same transport stream';
    $03: Result := Result + 'announcement is broadcast by means of a different service within a different transport stream';
    $04..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</reference_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableRunningStatus(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<running_status text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'undefined';
    $01: Result := Result + 'not running';
    $02: Result := Result + 'starts in a few seconds';
    $03: Result := Result + 'pausing';
    $04: Result := Result + 'running';
    $05..$07:
         Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</running_status>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   : itu-t rec. h.222.0  iso/iec 13818-1 (1994-11)
 ------------------------------------------------------------------------------}

function DvbDecodeTableSbLeakRate(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<sb_leak_rate text="', [Result, Indent]);
  case Identifier of
    $00, $3D..$3F:
         Result := Result + 'dvb reserved';
    $01: Result := Result + '0,0009';
    $02: Result := Result + '0,0018';
    $03: Result := Result + '0,0036';
    $04: Result := Result + '0,0072';
    $05: Result := Result + '0,0108';
    $06: Result := Result + '0,0144';
    $07: Result := Result + '0,0216';
    $08: Result := Result + '0,0288';
    $09: Result := Result + '0,075';
    $0A: Result := Result + '0,5';
    $0B: Result := Result + '0,5625';
    $0C: Result := Result + '0,8437';
    $0D: Result := Result + '1,0';
    $0E: Result := Result + '1,1250';
    $0F: Result := Result + '1,5';
    $10: Result := Result + '1,6875';
    $11: Result := Result + '2,0';
    $12: Result := Result + '2,2500';
    $13: Result := Result + '2,5';
    $14: Result := Result + '3,0';
    $15: Result := Result + '3,3750';
    $16: Result := Result + '3,5';
    $17: Result := Result + '4,0';
    $18: Result := Result + '4,5';
    $19: Result := Result + '5,0';
    $1A: Result := Result + '5,5';
    $1B: Result := Result + '6,0';
    $1C: Result := Result + '6,5';
    $1D: Result := Result + '6,7500';
    $1E: Result := Result + '7,0';
    $1F: Result := Result + '7,5';
    $20: Result := Result + '8,0';
    $21: Result := Result + '9,0';
    $22: Result := Result + '10,0';
    $23: Result := Result + '11,0';
    $24: Result := Result + '12,0';
    $25: Result := Result + '13,0';
    $26: Result := Result + '13,5';
    $27: Result := Result + '14,0';
    $28: Result := Result + '15,0';
    $29: Result := Result + '16,0';
    $2A: Result := Result + '17,0';
    $2B: Result := Result + '18,0';
    $2C: Result := Result + '20,0';
    $2D: Result := Result + '22,0';
    $2E: Result := Result + '24,0';
    $2F: Result := Result + '26,0';
    $30: Result := Result + '27,0';
    $31: Result := Result + '28,0';
    $32: Result := Result + '30,0';
    $33: Result := Result + '32,0';
    $34: Result := Result + '34,0';
    $35: Result := Result + '36,0';
    $36: Result := Result + '38,0';
    $37: Result := Result + '40,0';
    $38: Result := Result + '44,0';
    $39: Result := Result + '48,0';
    $3A: Result := Result + '54,0';
    $3B: Result := Result + '72,0';
    $3C: Result := Result + '108,0';
  end;
  case Identifier of
    $00, $3D..$3F: ;
    else Result := Result + ' Mbit/s';
  end;
  Result := format('%s">%d</sb_leak_rate>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableSbSize(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<sb_size text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'dvb_reserved';
    $01: Result := Result + '1536';
    $02: Result := Result + 'dvb_reserved';
    $03: Result := Result + 'dvb_reserved';
  end;
  Result := format('%s">%d</sb_size>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableScramblingMode(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<scrambling_mode text="', [Result, Indent]);
  case Identifier of
    $00, $FF: Result := Result + 'reserved for future use';
    $01..$7F: Result := Result + 'dvb defined';
    $80..$FE: Result := Result + 'user defined';
  end;
  Result := format('%s">%d</scrambling_mode>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableServiceType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<service_type text="', [Result, Indent]);
  case Identifier of
    $00, $11..$7F, $FF:
         Result := Result + 'reserved for future use';
    $01: Result := Result + 'digital television service';
    $02: Result := Result + 'digital radio sound service';
    $03: Result := Result + 'teletext service';
    $04: Result := Result + 'nvod reference service';
    $05: Result := Result + 'nvod time-shifted service';
    $06: Result := Result + 'mosaic service';
    $07: Result := Result + 'pal coded signal';
    $08: Result := Result + 'secam coded signal';
    $09: Result := Result + 'd/d2-mac';
    $0A: Result := Result + 'fm radio';
    $0B: Result := Result + 'ntsc coded signal';
    $0C: Result := Result + 'data broadcast service';
    $0D: Result := Result + 'reserved for common interface usage';
    $0E: Result := Result + 'rcs map';
    $0F: Result := Result + 'rcs fls';
    $10: Result := Result + 'dvb mhp service';
// $84 download ?
// $83 IEPG data

// ?? $11: MPEG-2 HD digital television service
// ?? $19: AVC HD digital television service
    $80..$FE:
         Result := Result + 'user defined';
  end;
  Result := format('%s">%d</service_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableStreamContent(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<stream_content text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'reserved for future use';
    $01: Result := Result + 'video';
    $02: Result := Result + 'audio';
    $03: Result := Result + 'ebu';
    $04: Result := Result + 'ac3 audio';
    $05..$0B:
         Result := Result + 'reserved for future use';
    $0C..$0F:
         Result := Result + 'user defined';
  end;
  Result := format('%s">%d</stream_content>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableStreamId(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<stream_id text="', [Result, Indent]);
  case Identifier of
    $BC: Result := Result + 'program_stream_map';
    $BD: Result := Result + 'private_stream_1';
    $BE: Result := Result + 'padding_stream';
    $BF: Result := Result + 'private_stream_2';
    $C0..$DF:
         Result := format('%siso/iec 13818-3 or iso/iec 11172-3 or iso/iec 13818-7 or iso/iec 14496-3 audio stream number %d', [Result, Identifier and $1F]);
    $E0..$EF:
         Result := format('%situ-t rec. h.262  iso/iec 13818-2, iso/iec 11172-2, iso/iec 14496-2 or itu-t rec. h.264  iso/iec 14496-10 video stream number %d', [Result, Identifier and $1F]);
    $F0: Result := Result + 'ecm_stream';
    $F1: Result := Result + 'emm_stream';
    $F2: Result := Result + 'itu-t rec. h.222.0  iso/iec 13818-1 annex a or iso/iec 13818-6_dsmcc_stream';
    $F3: Result := Result + 'iso/iec_13522_stream';
    $F4: Result := Result + 'itu-t rec. h.222.1 type a';
    $F5: Result := Result + 'itu-t rec. h.222.1 type b';
    $F6: Result := Result + 'itu-t rec. h.222.1 type c';
    $F7: Result := Result + 'itu-t rec. h.222.1 type d';
    $F8: Result := Result + 'itu-t rec. h.222.1 type e';
    $F9: Result := Result + 'ancillary_stream';
    $FA: Result := Result + 'iso/iec14496-1_sl-packetized_stream';
    $FB: Result := Result + 'iso/iec14496-1_flexmux_stream';
    $FC: Result := Result + 'descriptive data stream';
    $FD: Result := Result + 'reserved data stream';
    $FE: Result := Result + 'reserved data stream';
    $FF: Result := Result + 'program_stream_directory';
  end;
  Result := format('%s">%d</stream_id>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableStreamType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<stream_type text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + 'itu-t  iso/eic reserved';
    $01: Result := Result + 'iso/iec 11172 video';
    $02: Result := Result + 'itu-t rec. h.262  iso/iec 13818-2 video or iso/iec 11172-2 constrained parameter video stream';
    $03: Result := Result + 'iso/iec 11172 audio';
    $04: Result := Result + 'iso/iec 13818-3 audio';
    $05: Result := Result + 'itu-t rec. h.222.0  iso/iec 13818-1 private sections';
    $06: Result := Result + 'itu-t rec. h.222.0  iso/iec 13818-1 pes packets containing private data';
    $07: Result := Result + 'iso/iec 13522 mheg';
    $08: Result := Result + 'itu-t rec. h.222.0  iso/iec 13818-1 annex a dsm cc';
    $09: Result := Result + 'itu-t rec. h.222.1';
    $0A: Result := Result + 'iso/iec 13818-6 type a';
    $0B: Result := Result + 'iso/iec 13818-6 type b  dsm-cc sections containing streaming, synchronized data';
    $0C: Result := Result + 'iso/iec 13818-6 type c';
    $0D: Result := Result + 'iso/iec 13818-6 type d  dsm-cc addressable sections';
    $0E: Result := Result + 'iso/iec 13818-1 auxiliary';
    $0F: Result := Result + 'iso/iec 13818-7 audio with adts transport syntax';
    $10: Result := Result + 'iso/iec 14496-2 visual';
    $11: Result := Result + 'iso/iec 14496-3 audio with the latm transport syntax as defined in iso/iec 14496-3 / amd 1';
    $12: Result := Result + 'iso/iec 14496-1 sl-packetized stream or flexmux stream carried in pes packets';
    $13: Result := Result + 'iso/iec 14496-1 sl-packetized stream or flexmux stream carried in iso/iec14496_sections';
    $14: Result := Result + 'iso/iec 13818-6 synchronized download protocol  dsm-cc sections containing non-streaming, synchronized data';
    $15: Result := Result + 'metadata carried in pes packets';
    $16: Result := Result + 'metadata carried in metadata_sections';
    $17: Result := Result + 'metadata carried in iso/iec 13818-6 data carousel';
    $18: Result := Result + 'metadata carried in iso/iec 13818-6 object carousel';
    $19: Result := Result + 'metadata carried in iso/iec 13818-6 synchronized download protocol';
//    $1A: Result := Result + 'itu-t rec. h.264  iso/eic 14496-10 video';
    $1A: Result := Result + 'ipmp stream (defined in iso/iec 13818-11, mpeg-2 ipmp)';
    $1B: Result := Result + 'avc video stream as defined in itu-t rec. h.264  iso/iec 14496-10 video';
    $1C: Result := Result + 'iso/iec 14496-3 audio, without using any additional transport syntax';
    $1D: Result := Result + 'iso/iec 14496-17 text';
    $1E: Result := Result + 'mpeg-21 svc video';
    $1F..$7E:
         Result := Result + 'itu-t rec. h.222.0  iso/iec 13818-1 reserved';
    $7F: Result := Result + 'ipmp stream';
    $95: Result := Result + 'sections coveying data service table, network resource table';
    $C2: Result := Result + 'pes packets containing streaming, synchroznied data';
    $80..$94, $96..$C1, $C3..$FF:
         Result := Result + 'user private';
  end;
  Result := format('%s">%d</stream_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableTableIdCaMessage(Identifier: Integer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<table_id text="', [Result, Indent]);
  case Identifier of
    $00..$02: Result := Result + 'mpeg specified';
    $03..$3F: Result := Result + 'mpeg_reserved';
    $40..$72: Result := Result + 'v2-si_specified';
    $73..$7F: Result := Result + 'dvb_reserved';
    $80..$81: Result := Result + 'ca_message_section, ecm';
    $82..$8F: Result := Result + 'ca_message_section, ca system private';
    $90..$FE: Result := Result + 'private';
    $FF:      Result := Result + 'iso_reserved';
  end;
  Result := format('%s">%d</table_id>%s',  [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableTeletextType(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<teletext_type text="', [Result, Indent]);
  case Identifier of
    $00, $06..$1F:
         Result := Result + 'reserved for future use';
    $01: Result := Result + 'initial teletext page';
    $02: Result := Result + 'teletext subtitle page';
    $03: Result := Result + 'additional information page';
    $04: Result := Result + 'programme schedule page';
    $05: Result := Result + 'teletext subtitle page for hearing impaired people';
  end;
  Result := format('%s">%d</teletext_type>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <Identifier>  Identifier status data
            <Indent>      'Header' text (typically indent)
  Returns : <Result>      Empty if failure otherwise XML data

  Descript: Decode table data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeTableTransmissionMode(Identifier: Byte; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  Result := format('%s%s<transmission_mode text="', [Result, Indent]);
  case Identifier of
    $00: Result := Result + '2k mode';
    $01: Result := Result + '8k mode';
    $02: Result := Result + '4k mode';
    $03: Result := Result + 'reserved for future use';
  end;
  Result := format('%s">%d</transmission_mode>%s', [Result, Identifier, CXmlLf]);
end;

{------------------------------------------------------------------------------
                             Table decoders end
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                             Section decoders start
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : <SectionData>   Pointer to section data
            <Indent>        'Header' text (typically indent)
  Returns : <Result>        Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionBouquetAssociation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength   : Integer;
  DescriptorLength: Integer;
  StreamLength    : Integer;
  Index           : Integer;
  Items           : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionBouquetAssociation then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'bouquet_association_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <bouquet_id>%d</bouquet_id>%s',                            [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);

  DescriptorLength := ((PByteArray(SectionData)[8] and $0F) shl 8) or PByteArray(SectionData)[9];
  Result := format('%s%s  <bouquet_descriptors_lenght>%d</bouquet_descriptors_lenght>%s', [Result, Indent, DescriptorLength, CXmlLf]);
  Index := 10;
  while DescriptorLength > 0 do
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase/decrease according to descriptor length
    Dec(DescriptorLength, PByteArray(SectionData)[Index+1] + 2);
    Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
  end;
  // Note: we don't use the streamlength further on, but the sectionlength (more generic)
  StreamLength := ((PByteArray(SectionData)[Index] and $0F) shl 8) or PByteArray(SectionData)[Index+1];
  Result := format('%s%s  <transport_stream_loop_length>%d</transport_stream_loop_length>%s',  [Result, Indent, StreamLength, CXmlLf]);
  Inc(Index, 2);

  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index]   shl 8) or PByteArray(SectionData)[Index+1], CXmlLf]);
    Result := format('%s%s    <original_network_id>%d</original_network_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index+2] shl 8) or PByteArray(SectionData)[Index+3], CXmlLf]);

    DescriptorLength := ((PByteArray(SectionData)[Index+4] and $0F) shl 8) or PByteArray(SectionData)[Index+5];
    Result := format('%s%s    <transport_descriptors_length>%d</transport_descriptors_length>%s', [Result, Indent, DescriptorLength, CXmlLf]);
    Inc(Index, 6);
    while DescriptorLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(Descriptorlength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionCa(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength: Integer;
  Index        : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionCa then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'ca_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);

  Index := 8;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase according to descriptor length
    Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionCaMessage(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength: Integer;
  Loop         : Integer;
  Name         : AnsiString;
  BinaryString : AnsiString;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] < CDvbSectionCaMessageEcm0) or
     (PByteArray(SectionData)[0] > CDvbSectionCaMessageEmmAndCaSystemPrivateD) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'ca_message_section', CXmlLf]);

  Result := Result + DvbDecodeTableTableIdCaMessage(PByteArray(SectionData)[0], Indent + '  ');
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <ca_section_length>%d</ca_section_length>%s',              [Result, Indent, SectionLength, CXmlLf]);

  if SectionLength > 0 then
  begin
    Name := '';
    for Loop := 0 to SectionLength-1 do
      Name := Name + PChar(SectionData)[Loop+3];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <ca_data_byte text="%s">"%s"</ca_data_byte>%s',          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionContainer(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionContainer then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'container_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionContentIdentifier(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionContentIdentifier then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'content__identifier_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionDataEventTable(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionDataEventTable then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'data_event_table_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionDataServiceTable(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionDataServiceTable then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'data_service_table_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionDiscontinuityInformation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength: Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionDiscontinuityInformation then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'discontinuity_information_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);

  Result := format('%s%s  <transition_flag>%d</transition_flag>%s',                  [Result, Indent, (PByteArray(SectionData)[3] and $80) shr 7, CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionDsmcc(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] < CDvbSectionDsmcc0) or
     (PByteArray(SectionData)[0] > CDvbSectionDsmcc1) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'dsmcc_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionDsmccAddressable(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionDsmccAddressable then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'dsmcc_addressable_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionEventInformation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength   : Integer;
  DescriptorLength: Integer;
  Index           : Integer;
  Loop            : Integer;
  Items           : Integer;
  UtcTime         : Int64;
  Year            : Word;
  Month           : Word;
  Day             : Word;
  Hours           : Word;
  Minutes         : Word;
  Seconds         : word;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] < CDvbSectionActualTransportStreamPresentFollowingEventInformation) or
     (PByteArray(SectionData)[0] > CDvbSectionOtherTransportStreamScheduleEventInformationF) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  case PByteArray(SectionData)[0] of
    CDvbSectionActualTransportStreamPresentFollowingEventInformation:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'event_information_section - actual_transport_stream, present/following', CXmlLf]);
    CDvbSectionOtherTransportStreamPresentFollowingEventInformation:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'event_information_section - other_transport_stream, present/following', CXmlLf]);
    CDvbSectionActualTransportStreamScheduleEventInformation0..CDvbSectionActualTransportStreamScheduleEventInformationF:
      Result := format('%s%s  <section_name id="%d">"%s"</section_name>%s',          [Result, Indent, PByteArray(SectionData)[0] - CDvbSectionActualTransportStreamScheduleEventInformation0, 'event_information_section - actual_transport_stream, schedule', CXmlLf]);
    CDvbSectionOtherTransportStreamScheduleEventInformation0..CDvbSectionOtherTransportStreamScheduleEventInformationF:
      Result := format('%s%s  <section_name id="%d">"%s"</section_name>%s',          [Result, Indent, PByteArray(SectionData)[0] - CDvbSectionOtherTransportStreamScheduleEventInformation0, 'event_information_section - other_transport_stream, schedule', CXmlLf]);
  end;

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <service_id>%d</service_id>%s',                            [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);
  Result := format('%s%s  <transport_stream_id>%d</transport_stream_id>%s',          [Result, Indent, (PByteArray(SectionData)[8] shl 8) or PByteArray(SectionData)[9], CXmlLf]);
  Result := format('%s%s  <original_network_id>%d</original_network_id>%s',          [Result, Indent, (PByteArray(SectionData)[10] shl 8) or PByteArray(SectionData)[11], CXmlLf]);
  Result := format('%s%s  <segment_last_section_number>%d</segment_last_section_number>%s', [Result, Indent, PByteArray(SectionData)[12], CXmlLf]);
  Result := format('%s%s  <last_table_id>%d</last_table_id>%s',                      [Result, Indent, PByteArray(SectionData)[13], CXmlLf]);

  Index := 14;
  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                        [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <event_id>%d</event_id>%s',                             [Result, Indent, (PByteArray(SectionData)[Index] shl 8) or PByteArray(SectionData)[Index+1], CXmlLf]);

    Result := format('%s%s    <start_time',                                           [Result, Indent]);
    UtcTime := 0;
    for Loop := 2 to 6 do
      UtcTime := (UtcTime shl 8) + PByteArray(SectionData)[Index+Loop];
    DvbFilterXmlDecodeUtcCode(UtcTime, Year, Month, Day, Hours, Minutes, Seconds);
    Result := format('%s text="%s">',                                                 [Result, format('%4.4d%2.2d%2.2dTZ%2.2d%2.2d%2.2d', [Year, Month, day, Hours, Minutes, Seconds])]);
    Result := format('%s%d</start_time>%s',                                           [Result, UtcTime, CXmlLf]);

    Result := format('%s%s    <duration',                                             [Result, Indent]);
    UtcTime := 0;
    for Loop := 7 to 9 do
      UtcTime := (UtcTime shl 8) + PByteArray(SectionData)[Index+Loop];
    DvbFilterXmlDecodeUtcTime(UtcTime, Hours, Minutes, Seconds);
    Result := format('%s text="%s">',                                                 [Result, format('PT%2.2dH%2.2dM%2.2dS', [Hours, Minutes, Seconds])]);
    Result := format('%s%d</duration>%s',                                             [Result, UtcTime, CXmlLf]);

    Result := Result + DvbDecodeTableRunningStatus(((PByteArray(SectionData)[Index+10] and $E0) shr 5), Indent + '    ');
    Result := format('%s%s    <free_ca_mode>%d</free_ca_mode>%s',                     [Result, Indent, ((PByteArray(SectionData)[Index+10] and $10) shr 4), CXmlLf]);

    DescriptorLength := ((PByteArray(SectionData)[Index+10] and $0F) shl 8) or PByteArray(SectionData)[Index+11];
    Result := format('%s%s    <descriptors_loop_length>%d</descriptors_loop_length>%s', [Result, Indent, DescriptorLength, CXmlLf]);
    Inc(Index, 12);
    while DescriptorLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(Descriptorlength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                               [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionLongTermServiceTable(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionLongTermServiceTable then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'long_term_service_table_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionMetadata(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionMetadata then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'metadata_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionMpeFec(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionMpeFec then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'mpe_fec_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionNetworkInformation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength   : Integer;
  DescriptorLength: Integer;
  StreamLength    : Integer;
  Index           : Integer;
  Items           : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] <> CDvbSectionActualNetworkNetworkInformation) and
     (PByteArray(SectionData)[0] <> CDvbSectionOtherNetworkNetworkInformation) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  case PByteArray(SectionData)[0] of
    CDvbSectionActualNetworkNetworkInformation:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'network_information_section - actual_network', CXmlLf]);
    CDvbSectionOtherNetworkNetworkInformation:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'network_information_section - other_network', CXmlLf]);
  end;

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <network_id>%d</network_id>%s',                            [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);

  DescriptorLength := ((PByteArray(SectionData)[8] and $0F) shl 8) or PByteArray(SectionData)[9];
  Result := format('%s%s  <network_descriptors_lenght>%d</network_descriptors_lenght>%s', [Result, Indent, DescriptorLength, CXmlLf]);
  Index := 10;
  while DescriptorLength > 0 do
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase/decrease according to descriptor length
    Dec(DescriptorLength, PByteArray(SectionData)[Index+1] + 2);
    Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
  end;
  // Note: we don't use the streamlength further on, but the sectionlength (more generic)
  StreamLength := ((PByteArray(SectionData)[Index] and $0F) shl 8) or PByteArray(SectionData)[Index+1];
  Result := format('%s%s  <transport_stream_loop_length>%d</transport_stream_loop_length>%s',  [Result, Indent, StreamLength, CXmlLf]);
  Inc(Index, 2);

  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index]   shl 8) or PByteArray(SectionData)[Index+1], CXmlLf]);
    Result := format('%s%s    <original_network_id>%d</original_network_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index+2] shl 8) or PByteArray(SectionData)[Index+3], CXmlLf]);

    DescriptorLength := ((PByteArray(SectionData)[Index+4] and $0F) shl 8) or PByteArray(SectionData)[Index+5];
    Result := format('%s%s    <transport_descriptors_length>%d</transport_descriptors_length>%s', [Result, Indent, DescriptorLength, CXmlLf]);
    Inc(Index, 6);
    while DescriptorLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(Descriptorlength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionNetworkResourcesTable(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionNetworkResourcesTable then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'network_resources_table_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionProgramAssociation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items: Integer;
  Loop : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionProgramAssociation then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'program_association_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2], CXmlLf]);
  Result := format('%s%s  <transport_stream_id>%d</transport_stream_id>%s',          [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);

  Items := ((((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2]) - 9) div 4;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);
      Result := format('%s%s    <program_number>%d</program_number>%s',              [Result, Indent, (PByteArray(SectionData)[Loop * 4 + 8] shl 8) + PByteArray(SectionData)[Loop * 4 + 9], CXmlLf]);
      if ((PByteArray(SectionData)[Loop * 4 + 8] shl 8) + PByteArray(SectionData)[Loop * 4 + 9]) = 0 then
        Result := format('%s%s    <network_pid>%d</network_pid>%s',                  [Result, Indent, ((PByteArray(SectionData)[Loop * 4 + 10] and $1F) shl 8) + PByteArray(SectionData)[Loop * 4 + 11], CXmlLf])
      else
        Result := format('%s%s    <program_map_pid>%d</program_map_pid>%s',          [Result, Indent, ((PByteArray(SectionData)[Loop * 4 + 10] and $1F) shl 8) + PByteArray(SectionData)[Loop * 4 + 11], CXmlLf]);
      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionRelatedContent(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionRelatedContent then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'related_content_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionResolutionNotification(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionResolutionNotification then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'resolution_notification_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionRunningStatus(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength: Integer;
  Index        : Integer;
  Items        : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionRunningStatus then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'running_status_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);

  Index := 3;
  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index  ] shr 8)  or PByteArray(SectionData)[Index+1], CXmlLf]);
    Result := format('%s%s    <original_network_id>%d</original_network_id>%s',      [Result, Indent, (PByteArray(SectionData)[Index+2] shr 8)  or PByteArray(SectionData)[Index+3], CXmlLf]);
    Result := format('%s%s    <service_id>%d</service_id>%s',                        [Result, Indent, (PByteArray(SectionData)[Index+4] shr 8)  or PByteArray(SectionData)[Index+5], CXmlLf]);
    Result := format('%s%s    <event_id>%d</event_id>%s',                            [Result, Indent, (PByteArray(SectionData)[Index+6] shr 8)  or PByteArray(SectionData)[Index+7], CXmlLf]);
    Result := Result + DvbDecodeTableRunningStatus(PByteArray(SectionData)[Index+8] and $07, Indent + '    ');
    Inc(Index, 9);
    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionSelectionInformation(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength   : Integer;
  DescriptorLength: Integer;
  Index           : Integer;
  Items           : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionSelectionInformation then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'selection_information_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);

  DescriptorLength := ((PByteArray(SectionData)[8] and $0F) shl 8) or PByteArray(SectionData)[9];
  Result := format('%s%s  <transmission_info_loop_lenght>%d</transmission_info_loop_lenght>%s', [Result, Indent, DescriptorLength, CXmlLf]);
  Index := 10;
  while DescriptorLength > 0 do
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase/decrease according to descriptor length
    Dec(DescriptorLength, PByteArray(SectionData)[Index+1] + 2);
    Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
  end;

  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <service_id>%d</service_id>%s',                        [Result, Indent, (PByteArray(SectionData)[Index] shl 8) or PByteArray(SectionData)[Index+1], CXmlLf]);
    Result := Result + DvbDecodeTableRunningStatus(((PByteArray(SectionData)[Index+2] and $70) shr 4), Indent + '    ');

    DescriptorLength := ((PByteArray(SectionData)[Index+2] and $0F) shl 8) or PByteArray(SectionData)[Index+3];
    Result := format('%s%s    <service_loop_length>%d</service_loop_length>%s',      [Result, Indent, DescriptorLength, CXmlLf]);
    Inc(Index, 4);
    while DescriptorLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(Descriptorlength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionServiceDescription(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength   : Integer;
  DescriptorLength: Integer;
  Index           : Integer;
  Items           : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] <> CDvbSectionActualTransportStreamServiceDescription) and
     (PByteArray(SectionData)[0] <> CDvbSectionOtherTransportStreamServiceDescription) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  case PByteArray(SectionData)[0] of
    CDvbSectionActualTransportStreamServiceDescription:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'service_description_section - actual_transport_stream', CXmlLf]);
    CDvbSectionOtherTransportStreamServiceDescription:
      Result := format('%s%s  <section_name>"%s"</section_name>%s',                  [Result, Indent, 'service_description_section - other_transport_stream', CXmlLf]);
  end;

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <transport_stream_id>%d</transport_stream_id>%s',          [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);
  Result := format('%s%s  <original_network_id>%d</original_network_id>%s',          [Result, Indent, (PByteArray(SectionData)[8] shl 8) or PByteArray(SectionData)[9], CXmlLf]);

  Index := 11;
  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                        [Result, Indent, Items, CXmlLf]);
    Result := format('%s%s    <service_id>%d</service_id>%s',                         [Result, Indent, (PByteArray(SectionData)[Index] shl 8) or PByteArray(SectionData)[Index+1], CXmlLf]);
    Result := format('%s%s    <eit_schedule_flag>%d</eit_schedule_flag>%s',           [Result, Indent, ((PByteArray(SectionData)[Index+2] and $02) shl 1), CXmlLf]);
    Result := format('%s%s    <eit_present_following_flag>%d</eit_present_following_flag>%s', [Result, Indent, (PByteArray(SectionData)[Index+2] and $01), CXmlLf]);
    Result := Result + DvbDecodeTableRunningStatus(((PByteArray(SectionData)[Index+3] and $E0) shr 5), Indent + '    ');
    Result := format('%s%s    <free_ca_mode>%d</free_ca_mode>%s',                     [Result, Indent, ((PByteArray(SectionData)[Index+3] and $10) shr 4), CXmlLf]);

    DescriptorLength := ((PByteArray(SectionData)[Index+3] and $0F) shl 8) or PByteArray(SectionData)[Index+4];
    Result := format('%s%s    <descriptors_loop_length>%d</descriptors_loop_length>%s', [Result, Indent, DescriptorLength, CXmlLf]);
    Inc(Index, 5);
    while DescriptorLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(Descriptorlength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                               [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionStuffing(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionStuffing then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'stuffing_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionTestdata(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if (PByteArray(SectionData)[0] < CDvbSectionMeasurement) or
     (PByteArray(SectionData)[0] > CDvbSectionNetworkStatus) then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'testdata_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionTimeDate(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Year   : Word;
  Month  : Word;
  Day    : Word;
  Hours  : Word;
  Minutes: Word;
  Seconds: word;
  UtcTime: Int64;
  Index  : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionTimeDate then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'time_date_section', CXmlLf]);

  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2], CXmlLf]);

  Result := format('%s%s  <utc_time',                                                [Result, Indent]);

  UtcTime := 0;
  for Index := 3 to 7 do
    UtcTime := (UtcTime shl 8) + PByteArray(SectionData)[Index];
  DvbFilterXmlDecodeUtcCode(UtcTime, Year, Month, Day, Hours, Minutes, Seconds);
  Result := format('%s text="%s">',                                                  [Result, format('%4.4d%2.2d%2.2dTZ%2.2d%2.2d%2.2d', [Year, Month, day, Hours, Minutes, Seconds])]);
  Result := format('%s%d</utc_time>%s',                                              [Result, UtcTime, CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionTimeOffset(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Year            : Word;
  Month           : Word;
  Day             : Word;
  Hours           : Word;
  Minutes         : Word;
  Seconds         : word;
  UtcTime         : Int64;
  Index           : Integer;
  DescriptorLength: Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionTimeOffset then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'time_offset_section', CXmlLf]);

  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2], CXmlLf]);

  Result := format('%s%s  <utc_time',                                                [Result, Indent]);

  UtcTime := 0;
  for Index := 3 to 7 do
    UtcTime := (UtcTime shl 8) + PByteArray(SectionData)[Index];
  DvbFilterXmlDecodeUtcCode(UtcTime, Year, Month, Day, Hours, Minutes, Seconds);
  Result := format('%s text="%s">',                                                  [Result, format('%4.4d%2.2d%2.2dTZ%2.2d%2.2d%2.2d', [Year, Month, day, Hours, Minutes, Seconds])]);
  Result := format('%s%d</utc_time>%s',                                              [Result, UtcTime, CXmlLf]);

  DescriptorLength := ((PByteArray(SectionData)[8] and $0F) shl 8) or PByteArray(SectionData)[9];
  Result := format('%s%s  <descriptors_loop_length>%d</descriptors_loop_length>%s',          [Result, Indent, DescriptorLength, CXmlLf]);
  Index := 10;
  while DescriptorLength > 0 do
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase/decrease according to descriptor length
    Dec(DescriptorLength, PByteArray(SectionData)[Index+1] + 2);
    Inc(Index,            PByteArray(SectionData)[Index+1] + 2);
  end;

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionTransportStreamDescription(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionTransportStreamDescription then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'transport_stream_description_section', CXmlLf]);

  Result := format('%s%s</section>%s',                                               [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <SectionData>  Pointer to section data
            <Indent>       'Header' text (typically indent)
  Returns : <Result>       Empty if failure otherwise XML data

  Descript: Decode section data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeSectionTsProgramMap(SectionData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SectionLength: Integer;
  Infolength   : Integer;
  EsLength     : Integer;
  Index        : Integer;
  Items        : Integer;
begin
  Result := '';
  if not Assigned(SectionData) then
    Exit;
  if PByteArray(SectionData)[0] <> CDvbSectionTsProgramMap then
    Exit;
  Result := format('%s%s<section id="%d">%s',                                        [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_name>"%s"</section_name>%s',                      [Result, Indent, 'ts_program_map_section', CXmlLf]);

  Result := format('%s%s  <table_id>%d</table_id>%s',                                [Result, Indent, PByteArray(SectionData)[0], CXmlLf]);
  Result := format('%s%s  <section_syntax_indicator>%d</section_syntax_indicator>%s', [Result, Indent, (PByteArray(SectionData)[1] and $80) shr 7, CXmlLf]);
  SectionLength := ((PByteArray(SectionData)[1] and $0F) shl 8) or PByteArray(SectionData)[2];
  Result := format('%s%s  <section_length>%d</section_length>%s',                    [Result, Indent, SectionLength, CXmlLf]);
  Result := format('%s%s  <program_number>%d</program_number>%s',                    [Result, Indent, (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4], CXmlLf]);
  Result := format('%s%s  <version_number>%d</version_number>%s',                    [Result, Indent, ((PByteArray(SectionData)[5] and $3E) shr 1), CXmlLf]);
  Result := format('%s%s  <current_next_indicator>%d</current_next_indicator>%s',    [Result, Indent, PByteArray(SectionData)[5] and $01, CXmlLf]);
  Result := format('%s%s  <section_number>%d</section_number>%s',                    [Result, Indent, PByteArray(SectionData)[6], CXmlLf]);
  Result := format('%s%s  <last_section_number>%d</last_section_number>%s',          [Result, Indent, PByteArray(SectionData)[7], CXmlLf]);
  Result := format('%s%s  <pcr_pid>%d</pcr_pid>%s',                                  [Result, Indent, ((PByteArray(SectionData)[8] and $1F) shl 8) or PByteArray(SectionData)[9], CXmlLf]);
  InfoLength := ((PByteArray(SectionData)[10] and $0F) shl 8) or PByteArray(SectionData)[11];
  Result := format('%s%s  <program_info_length>%d</program_info_length>%s',          [Result, Indent, InfoLength, CXmlLf]);
  Index := 12;
  while Infolength > 0 do
  begin
    Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '  ');
    // Increase/decrease according to descriptor length
    Dec(InfoLength, PByteArray(SectionData)[Index+1] + 2);
    Inc(Index,      PByteArray(SectionData)[Index+1] + 2);
  end;
  Items := 0;
  while Index < (SectionLength - 2) do           // +2 for where section_length started and -4 for CRC
  begin
    Result := format('%s%s  <item id="%d">%s',                                        [Result, Indent, Items, CXmlLf]);
    Result := result + DvbDecodeTableStreamType(PByteArray(SectionData)[Index], Indent + '    ');
    Result := format('%s%s    <elementary_pid>%d</elementary_pid>%s',                 [Result, Indent, ((PByteArray(SectionData)[Index+1] and $1F) shl 8) or PByteArray(SectionData)[Index+2], CXmlLf]);
    EsLength := ((PByteArray(SectionData)[Index+3] and $0F) shl 8) or PByteArray(SectionData)[Index+4];
    Result := format('%s%s    <es_info_length>%d</es_info_length>%s',                 [Result, Indent, EsLength, CXmlLf]);
    Inc(Index, 5);
    while EsLength > 0 do
    begin
      Result := Result + DvbGetXmlInfoFromDescriptor(@(PByteArray(SectionData)[Index]), Indent + '    ');
      // Increase/decrease according to descriptor length
      Dec(EsLength, PByteArray(SectionData)[Index+1] + 2);
      Inc(Index,    PByteArray(SectionData)[Index+1] + 2);
    end;
    Result := format('%s%s  </item>%s',                                               [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</section>%s',                                                [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
                             Section decoders end
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
                             Descriptor decoders start
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAc3(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAc3 then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ac3_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <component_type_flag>%d</component_type_flag>%s',          [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <bsid_flag>%d</bsid_flag>%s',                              [Result, Indent, (PByteArray(DescriptorData)[2] and $40) shr 6, CXmlLf]);
  Result := format('%s%s  <mainid_flag>%d</mainid_flag>%s',                          [Result, Indent, (PByteArray(DescriptorData)[2] and $20) shr 5, CXmlLf]);
  Result := format('%s%s  <asvc_flag>%d</asvc_flag>%s',                              [Result, Indent, (PByteArray(DescriptorData)[2] and $10) shr 4, CXmlLf]);

  Index := 3;
  if ((PByteArray(DescriptorData)[2] and $80) <> 0) then
  begin
    Result := Result + DvbDecodeTableAc3Type(PByteArray(DescriptorData)[Index], Indent + '  ', True);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $40) <> 0) then
  begin
    Result := format('%s%s  <bsid>%d</bsid>%s',                                      [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $20) <> 0) then
  begin
    Result := format('%s%s  <mainid>%d</mainid>%s',                                  [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $10) <> 0) then
  begin
    Result := format('%s%s  <asvc>%d</asvc>%s',                                      [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;

  if Index < PByteArray(DescriptorData)[1] then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-Index do
       Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <additional_info text="%s">"%s"</additional_info>%s',    [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAdaptationFieldData(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAdaptationFieldData then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'adaptation_field_data_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableAdaptationFieldDataIdentifier(PByteArray(DescriptorData)[2], Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAncillaryData(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAncillaryData then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ancillary_data_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableAncillaryDataIdentifier(PByteArray(DescriptorData)[2], Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAnnouncementSupport(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items        : Integer;
  Index        : Integer;
  ReferenceType: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAnnouncementSupport then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'announcement_support_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableAnnouncementSupportIndicator((PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], Indent + '  ');

  Index := 4;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    Result := Result + DvbDecodeTableAnnouncementType((PByteArray(DescriptorData)[Index] and $F0) shr 4, Indent + '    ');
    ReferenceType := PByteArray(DescriptorData)[Index] and $07;
    Result := Result + DvbDecodeTableReferenceType(ReferenceType, Indent + '    ');
    Inc(Index);
    if ReferenceType in [$01..$03] then
    begin
      Result := format('%s%s    <original_network_id>%d</original_network_id>%s',    [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
      Inc(Index, 2);
      Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s',    [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
      Inc(Index, 2);
      Result := format('%s%s    <service_id>%d</service_id>%s',                      [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
      Inc(Index, 2);
      Result := format('%s%s    <component_tag>%d</component_tag>%s',                [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
      Inc(Index);
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorApplicationSignalling(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorApplicationSignalling then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'application_signalling_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   : DSM CC
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAssociationTag(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAssociationTag then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'association_tag_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAtscCa(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAtscCa then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'atsc_ca_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableCaSystemId((PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], Indent + '  ');
  if PByteArray(DescriptorData)[1] > 2 then
  begin
    Name  := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-3 do
      Name := Name + PChar(DescriptorData)[Loop+6];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <private_data_byte text="%s">"%s"</private_data_byte>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAudioStream(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAudioStream then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'audio_stream_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <free_format_flag>%d</free_format_flag>%s',                [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <id>%d</id>%s',                                            [Result, Indent, (PByteArray(DescriptorData)[2] and $40) shr 6, CXmlLf]);
  Result := format('%s%s  <layer>%d</layer>%s',                                      [Result, Indent, (PByteArray(DescriptorData)[2] and $30) shr 4, CXmlLf]);
  Result := format('%s%s  <variable_rate_audio_indicator>%d</variable_rate_audio_indicator>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $08) shr 3, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAvcTimingAndHrd(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAvcTimingAndHrd then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'avc_timing_and_hrd_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorAvcVideo(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorAvcVideo then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'avc_video_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <avc_profile_and_level>%d</avc_profile_and_level>%s',      [Result, Indent, PByteArray(DescriptorData)[2], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorBouquetName(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorBouquetName then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'bouquet_name_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  if PByteArray(DescriptorData)[1] > 0 then
  begin
    Name  := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-1 do
      Name := Name + PChar(DescriptorData)[Loop+2];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <char text="%s">"%s"</char>%s',                          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCa(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : Ansistring;
  Binarystring: Ansistring;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCa then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ca_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableCaSystemId((PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], Indent + '  ');
  Result := format('%s%s  <ca_pid>%d</ca_pid>%s',                                    [Result, Indent, ((PByteArray(DescriptorData)[4] and $1F) shl 8) or PByteArray(DescriptorData)[5] , CXmlLf]);

  if PByteArray(DescriptorData)[1] > 4 then
  begin
    Name  := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-5 do
      Name := Name + PChar(DescriptorData)[Loop+6];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <private_data_byte text="%s">"%s"</private_data_byte>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCaIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items   : Integer;
  Loop    : Integer;
  CaSystem: Word;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCaIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ca_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 2;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      CaSystem := (PByteArray(DescriptorData)[Loop * 2 + 2] shl 8) or PByteArray(DescriptorData)[Loop * 2 + 3];
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);
      Result := Result + DvbDecodeTableCaSystemId(CaSystem, Indent + '    ');
      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCableDeliverySystem(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCableDeliverySystem then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'cable_delivery_system_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <frequency' ,                                              [Result, Indent]);
  Result := format('%s text="%s%s%s%s,%s%s%s%s MHz">',                               [Result,
              Chr(Ord('0') + ((PByteArray(DescriptorData)[2] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[2] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[3] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[3] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[4] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[4] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[5] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[5] and $0F)), CXmlLf]);
  Result := format('%s%d</frequency>%s',                                             [Result,
              ((PByteArray(DescriptorData)[2] and $F0) shr 4) * 1000000000 +
               (PByteArray(DescriptorData)[2] and $0F)        * 100000000  +
              ((PByteArray(DescriptorData)[3] and $F0) shr 4) * 10000000   +
               (PByteArray(DescriptorData)[3] and $0F)        * 1000000    +
              ((PByteArray(DescriptorData)[4] and $F0) shr 4) * 100000     +
               (PByteArray(DescriptorData)[4] and $0F)        * 10000      +
              ((PByteArray(DescriptorData)[5] and $F0) shr 4) * 1000       +
               (PByteArray(DescriptorData)[5] and $0F)        * 100, CXmlLf]);

  Result := Result + DvbDecodeTableFecOuter(PByteArray(DescriptorData)[7] and $0F, Indent + '  ');
  Result := Result + DvbDecodeTableModulationCable(PByteArray(DescriptorData)[8], Indent + '  ');

  Result := format('%s%s  <symbolrate',                                              [Result, Indent]);
  Result := format('%s text="%s%s%s,%s%s%s%s Msymbols/s">',                          [Result,
              Chr(Ord('0') + ((PByteArray(DescriptorData)[9]  and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[9]  and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[10] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[10] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[11] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[11] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[12] and $F0) shr 4))]);
  Result := format('%s%d</symbolrate>%s',                                            [Result,
              ((PByteArray(DescriptorData)[9]  and $F0) shr 4) * 100000000 +
               (PByteArray(DescriptorData)[9]  and $0F)        * 10000000  +
              ((PByteArray(DescriptorData)[10] and $F0) shr 4) * 1000000   +
               (PByteArray(DescriptorData)[10] and $0F)        * 100000    +
              ((PByteArray(DescriptorData)[11] and $F0) shr 4) * 10000     +
               (PByteArray(DescriptorData)[11] and $0F)        * 1000      +
              ((PByteArray(DescriptorData)[12] and $F0) shr 4) * 100, CXmlLf]);

  Result := Result + DvbDecodeTableFecInner(PByteArray(DescriptorData)[12] and $0F, Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   : DSM CC
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCarouselIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCarouselIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'carousel_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCellFrequencyLink(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items     : Integer;
  Index     : Integer;
  SubLength : Integer;
  SubItems  : Integer;
  Loop      : Integer;
  Loop2     : Integer;
  Frequency : Cardinal;
  Frequency2: Int64;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCellFrequencyLink then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'cell_frequency_link_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);


  Items := 0;
  Index := 2;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    Result := format('%s%s    <cell_id>%d</cell_id>%s',                              [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
    Inc(Index, 2);
    Frequency := 0;
    for Loop2 := 0 to 3 do
      Frequency := (Frequency shl 8) or PByteArray(DescriptorData)[Index+Loop2];
    Frequency2 := Frequency;
    Frequency2 := Frequency2 * 10;
    Result := format('%s%s    <frequency text="%d hz">%d</frequency>%s',             [Result, Indent, Frequency2, Frequency, CXmlLf]);
    Inc(Index, 4);
    Sublength := PByteArray(DescriptorData)[Index];
    SubItems := SubLength div 5;
    if SubItems > 0 then
      for Loop := 0 to SubItems-1 do
      begin
        Result := format('%s%s    <item id="%d">%s',                                 [Result, Indent, Loop, CXmlLf]);

        Result := format('%s%s      <cell_id_extension>%d</cell_id_extension>%s',    [Result, Indent, PByteArray(DescriptorData)[Loop*5+Index], CXmlLf]);
        Frequency := 0;
        for Loop2 := 1 to 4 do
          Frequency := (Frequency shl 8) or PByteArray(DescriptorData)[Loop*5+Index+Loop2];
        Frequency2 := Frequency;
        Frequency2 := Frequency2 * 10;
        Result := format('%s%s      <transposer_frequency text="%d hz">%d</transposer_frequency>%s', [Result, Indent, Frequency2, Frequency, CXmlLf]);

        Result := format('%s%s    </item>%s',                                        [Result, Indent, CXmlLf]);
      end;
    Inc(Index, SubLength);

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCellList(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items    : Integer;
  Index    : Integer;
  SubLength: Integer;
  SubItems : Integer;
  Loop     : Integer;
  Signed   : SmallInt;
  Number   : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCellList then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'cell_list_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := 0;
  Index := 2;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    Number := (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1];
    Result := format('%s%s    <cell_id>%d</cell_id>%s',                              [Result, Indent, Number , CXmlLf]);
    Inc(Index, 2);
    Number := (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1];
    Signed := SmallInt(Number);
    Result := format('%s%s    <cell_latitude text="%.1f">%d</cell_latitude>%s',       [Result, Indent, 90/Signed, Number, CXmlLf]);
    Inc(Index, 2);
    Number := (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1];
    Signed := SmallInt(Number);
    Result := format('%s%s    <cell_longitude text="%.1f">%d</cell_longitude>%s',     [Result, Indent, 180/Signed, Number, CXmlLf]);
    Inc(Index, 2);
    Number := (PByteArray(DescriptorData)[Index] shl 4) or ((PByteArray(DescriptorData)[Index+1] and $F0) shr 4);
    Result := format('%s%s    <cell_extend_of_latitude text="%.1f">%d</cell_extend_of_latitude>%s',  [Result, Indent, 90/Number, Number, CXmlLf]);
    Inc(Index, 1);
    Number := ((PByteArray(DescriptorData)[Index] and $0F) shl 4) or PByteArray(DescriptorData)[Index+1];
    Result := format('%s%s    <cell_extend_of_longitude text="%.1f">%d</cell_extend_of_longitude>%s',  [Result, Indent, 180/Number, Number, CXmlLf]);
    Inc(Index, 2);
    SubLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <subcell_info_loop_length>%d</subcell_info_loop_length>%s',  [Result, Indent, SubLength, CXmlLf]);
    Inc(Index);
    SubItems := SubLength div 8;
    if SubItems > 0 then
      for Loop := 0 to SubItems-1 do
      begin
        Result := format('%s%s    <item id="%d">%s',                                 [Result, Indent, Loop, CXmlLf]);

        Result := format('%s%s      <cell_id_extension>%d</cell_id_extension>%s',    [Result, Indent, PByteArray(DescriptorData)[Loop*8+Index], CXmlLf]);
        Number := (PByteArray(DescriptorData)[Loop*8+Index+1] shl 8) or PByteArray(DescriptorData)[Loop*8+Index+2];
        Signed := SmallInt(Number);
        Result := format('%s%s      <subcell_latitude text="%.1f">%d</subcell_latitude>%s', [Result, Indent, 90/Signed, Number, CXmlLf]);
        Number := (PByteArray(DescriptorData)[Loop*8+Index+3] shl 8) or PByteArray(DescriptorData)[Loop*8+Index+4];
        Signed := SmallInt(Number);
        Result := format('%s%s      <subcell_longitude text="%.1f">%d</subcell_longitude>%s', [Result, Indent, 180/Signed, Number, CXmlLf]);
        Number := (PByteArray(DescriptorData)[Loop*8+Index+5] shl 4) or ((PByteArray(DescriptorData)[Loop*8+Index+6] and $F0) shr 4);
        Result := format('%s%s      <subcell_extend_of_latitude text="%.1f">%d</subcell_extend_of_latitude>%s',  [Result, Indent, 90/Number, Number, CXmlLf]);
        Number := ((PByteArray(DescriptorData)[Loop*8+Index+6] and $0F) shl 4) or PByteArray(DescriptorData)[Loop*8+Index+7];
        Result := format('%s%s      <subcell_extend_of_longitude text="%.1f">%d</subcell_extend_of_longitude>%s',  [Result, Indent, 180/Number, Number, CXmlLf]);

        Result := format('%s%s    </item>%s',                                        [Result, Indent, CXmlLf]);
      end;
    Inc(Index, SubLength);

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorComponent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  LanguageCode: Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorComponent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'component_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableStreamContent(PByteArray(DescriptorData)[2] and $0F, Indent + '  ');
  Result := Result + DvbDecodeTableComponentType(PByteArray(DescriptorData)[2] and $0F, PByteArray(DescriptorData)[3], False, Indent + '  ');
  Result := format('%s%s  <component_tag>%d</component_tag>%s',                      [Result, Indent, PByteArray(DescriptorData)[4], CXmlLf]);

  LanguageCode := 0;
  for Loop := 5 to 7 do
    LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Loop];
  Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '  ');

  if PByteArray(DescriptorData)[1] > 6 then
  begin
    Name  := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-7 do
      Name := Name + PChar(DescriptorData)[Loop+8];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <text_char text="%s">"%s"</text_char>%s',                [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorContent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items: Integer;
  Loop : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorContent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'content_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 2;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      Result := Result + DvbDecodeTableContentNibbleLevel1((PByteArray(DescriptorData)[(Loop*2)+2] and $F0) shr 4, Indent + '    ');
      Result := Result + DvbDecodeTableContentNibbleLevel2( PByteArray(DescriptorData)[(Loop*2)+2] and $0F, (PByteArray(DescriptorData)[(Loop*2)+2] and $F0) shr 4, Indent + '    ');
      Result := format('%s%s    <user_nibble>%d</user_nibble>%s',                    [Result, Indent, (PByteArray(DescriptorData)[(Loop*2)+3] and $F0) shr 4, CXmlLf]);
      Result := format('%s%s    <user_nibble>%d</user_nibble>%s',                    [Result, Indent,  PByteArray(DescriptorData)[(Loop*2)+3] and $0F, CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorContentIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorContentIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'content_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorContentLabeling(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorContentLabeling then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'content_labeling_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCopyright(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Identifier  : Integer;
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCopyright then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'copyright_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Identifier := 0;
  for Loop := 0 to 3 do
    Identifier := (Identifier shl 8) or PByteArray(DescriptorData)[Loop + 2];
  Result := format('%s%s  <copyright_identifier>%d</copyright_identifier>%s',        [Result, Indent, Identifier, CXmlLf]);

  if PByteArray(DescriptorData)[1] > 4 then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-5 do
      Name := Name + PChar(DescriptorData)[Loop + 6];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <additional_copyright_info text="%s">"%s"</additional_copyright_info>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCountryAvailability(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items      : Integer;
  Loop       : Integer;
  Loop2      : Integer;
  CountryCode: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCountryAvailability then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'country_availability_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <country_availability_flag>%d</country_availability_flag>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Items := (PByteArray(DescriptorData)[1]-1) div 3;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      CountryCode := 0;
      for Loop2 := 3 to 5 do
        CountryCode := (CountryCode shl 8) or PByteArray(DescriptorData)[(Loop * 3) + Loop2];
      Result := Result + DvbDecodeTableCountryCode(CountryCode, Indent + '    ');

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorCrc32(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Number: Cardinal;
  Loop  : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorCrc32 then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'crc32_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Number := 0;
  for Loop := 2 to 5 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <crc_32>%d</crc_32>%s',                                    [Result, Indent, Number, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDataBroadcast(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  SelectorLength: Integer;
  Loop          : Integer;
  Name          : AnsiString;
  BinaryString  : AnsiString;
  Index         : Integer;
  LanguageCode  : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDataBroadcast then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'data_broadcast_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <data_broadcast_id>%d</data_broadcast_id>%s',              [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <component_tag>%d</component_tag>%s',                      [Result, Indent, PByteArray(DescriptorData)[4], CXmlLf]);
  SelectorLength := PByteArray(DescriptorData)[5];
  Result := format('%s%s  <selector_length>%d</selector_length>%s',                  [Result, Indent, SelectorLength, CXmlLf]);
  Index := 6;
  if SelectorLength > 0 then
  begin
    Name := '';
    for Loop := 0 to SelectorLength-1 do
      Name := Name + PChar(DescriptorData)[Loop+Index];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <selector_byte text="%s">"%s"</selector_byte>%s',        [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    Inc(Index, SelectorLength);
  end;
  LanguageCode := 0;
  for Loop := Index to Index+2 do
    LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
  Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '  ');
  Inc(Index, 3);
  SelectorLength := PByteArray(DescriptorData)[Index];
  Result := format('%s%s  <text_length>%d</text_length>%s',                          [Result, Indent, SelectorLength, CXmlLf]);
  Inc(Index);
  if SelectorLength > 0 then
  begin
    Name := '';
    for Loop := 0 to SelectorLength-1 do
      Name := Name + PChar(DescriptorData)[Loop+Index];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <text_char text="%s">"%s"</text_char>%s',                [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDataBroadcastId(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDataBroadcastId then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'data_broadcast_id_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <data_broadcast_id>%d</data_broadcast_id>%s',              [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  if PByteArray(DescriptorData)[1]>2 then
  begin
    Name := '';
    for Loop := 4 to PByteArray(DescriptorData)[1]+1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <id_selector_byte text="%s">"%s"</id_selector_byte>%s',  [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDataService(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDataService then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'data_service_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <data_service_profile>%d</data_service_profile>%s',        [Result, Indent, PByteArray(DescriptorData)[2], CXmlLf]);
  Result := format('%s%s  <data_service_level>%d</data_service_level>%s',            [Result, Indent, PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <private_data_length>%d</private_data_length>%s',          [Result, Indent, PByteArray(DescriptorData)[4], CXmlLf]);
  if PByteArray(DescriptorData)[4] > 0 then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[4] do
      Name := Name + PChar(DescriptorData)[Loop+5];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <private_data_byte text="%s">"%s"</private_data_byte>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDataStreamAlignment(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDataStreamAlignment then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'data_stream_alignment_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableAlignmentType(PByteArray(DescriptorData)[2], Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDefaultAuthority(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDefaultAuthority then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'default_authority_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDownload(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Number: Cardinal;
  Loop  : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDownload then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'download_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Number := 0;
  for Loop := 2 to 5 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <download_id>%d</download_id>%s',                          [Result, Indent, Number, CXmlLf]);
  Number := 0;
  for Loop := 6 to 9 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <carousel_period>%d</carousel_period>%s',                  [Result, Indent, Number, CXmlLf]);
  Number := 0;
  for Loop := 10 to 13 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <control_msg_time_out_value>%d</control_msg_time_out_value>%s', [Result, Indent, Number, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   : DSM CC
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDeferredAssociationTags(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDeferredAssociationTags then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'deferred_association_tags_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorDsng(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorDsng then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'dsng_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  if PByteArray(DescriptorData)[1] > 0 then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-1 do
       Name := Name + PChar(DescriptorData)[2 + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <byte text="%s">"%s"</byte>%s',                          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorEcmRepetitionRate(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorEcmRepetitionRate then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ecm_repetition_rate_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorEnhancedAc3(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorEnhancedAc3 then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'enhanced_ac3_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <component_type_flag>%d</component_type_flag>%s',          [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <bsid_flag>%d</bsid_flag>%s',                              [Result, Indent, (PByteArray(DescriptorData)[2] and $40) shr 6, CXmlLf]);
  Result := format('%s%s  <mainid_flag>%d</mainid_flag>%s',                          [Result, Indent, (PByteArray(DescriptorData)[2] and $20) shr 5, CXmlLf]);
  Result := format('%s%s  <asvc_flag>%d</asvc_flag>%s',                              [Result, Indent, (PByteArray(DescriptorData)[2] and $10) shr 4, CXmlLf]);
  Result := format('%s%s  <mixinfoexists>%d</mixinfoexists>%s',                      [Result, Indent, (PByteArray(DescriptorData)[2] and $08) shr 3, CXmlLf]);
  Result := format('%s%s  <substream1_flag>%d</substream1_flag>%s',                  [Result, Indent, (PByteArray(DescriptorData)[2] and $04) shr 2, CXmlLf]);
  Result := format('%s%s  <substream2_flag>%d</substream2_flag>%s',                  [Result, Indent, (PByteArray(DescriptorData)[2] and $02) shr 1, CXmlLf]);
  Result := format('%s%s  <substream3_flag>%d</substream3_flag>%s',                  [Result, Indent, PByteArray(DescriptorData)[2] and $01 , CXmlLf]);

  Index := 3;
  if ((PByteArray(DescriptorData)[2] and $80) <> 0) then
  begin
    Result := Result + DvbDecodeTableAc3Type(PByteArray(DescriptorData)[Index], Indent + '  ', True);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $40) <> 0) then
  begin
    Result := format('%s%s  <bsid>%d</bsid>%s',                                      [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $20) <> 0) then
  begin
    Result := format('%s%s  <mainid>%d</mainid>%s',                                  [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $10) <> 0) then
  begin
    Result := format('%s%s  <asvc>%d</asvc>%s',                                      [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $04) <> 0) then
  begin
    Result := Result + DvbDecodeTableAc3Substream(PByteArray(DescriptorData)[Index], 1, Indent + '  ', True);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $02) <> 0) then
  begin
    Result := Result + DvbDecodeTableAc3Substream(PByteArray(DescriptorData)[Index], 2, Indent + '  ', True);
    Inc(Index);
  end;
  if ((PByteArray(DescriptorData)[2] and $01) <> 0) then
  begin
    Result := Result + DvbDecodeTableAc3Substream(PByteArray(DescriptorData)[Index], 3, Indent + '  ', True);
    Inc(Index);
  end;

  if Index < PByteArray(DescriptorData)[1] then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-Index do
       Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <additional_info text="%s">"%s"</additional_info>%s',    [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorExternalEsId(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorExternalEsId then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'external_es_id_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorExtendedEvent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop            : Integer;
  Index           : Integer;
  Name            : AnsiString;
  BinaryString    : AnsiString;
  LanguageCode    : Integer;
  ItemsLength     : Integer;
  Items           : Integer;
  DescriptorLength: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorExtendedEvent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'extended_event_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);
  Result := format('%s%s  <descriptor_number>%d</descriptor_number>%s',              [Result, Indent, (PByteArray(DescriptorData)[2] and $F0) shr 4, CXmlLf]);
  Result := format('%s%s  <last_descriptor_number>%d</last_descriptor_number>%s',    [Result, Indent, PByteArray(DescriptorData)[2] and $0F, CXmlLf]);

  LanguageCode := 0;
  for Index := 3 to 5 do
    LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
  Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '  ');

  ItemsLength := PByteArray(DescriptorData)[6];
  Result := format('%s%s  <length_of_items>%d</length_of_items>%s',                  [Result, Indent, ItemsLength, CXmlLf]);

  Index := 7;
  Items := 0;
  while ItemsLength > 0 do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    DescriptorLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <item_description_length>%d</item_description_length>%s', [Result, Indent, DescriptorLength, CXmlLf]);
    if DescriptorLength > 0 then
    begin
      Name := '';
      for Loop := 1 to DescriptorLength do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <item_description_char text="%s">"%s"</item_description_char>%s',    [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    end;
    Inc(Index,       DescriptorLength+1);
    Dec(ItemsLength, DescriptorLength+1);
    DescriptorLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <item_length>%d</item_length>%s',                      [Result, Indent, DescriptorLength, CXmlLf]);
    if DescriptorLength > 0 then
    begin
      Name := '';
      for Loop := 1 to DescriptorLength do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <item_char text="%s">"%s"</item_char>%s',            [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    end;
    Inc(Index,       DescriptorLength+1);
    Dec(ItemsLength, DescriptorLength+1);

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  DescriptorLength := PByteArray(DescriptorData)[Index];
  Result := format('%s%s  <text_length>%d</text_length>%s',                          [Result, Indent, DescriptorLength, CXmlLf]);
  if DescriptorLength > 0 then
  begin
    Name := '';
    for Loop := 1 to DescriptorLength do
      Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <text_char text="%s">"%s"</text_char>%s',                [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorFmc(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorFmc then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'fmc_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorFmxBufferSize(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorFmxBufferSize then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'fmxbuffersize_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorForbidden(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorForbidden then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'forbidden_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorFrequencyList(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items          : Integer;
  CentreFrequency: Cardinal;
  Loop           : Integer;
  Loop2          : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorFrequencyList then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'frequency_list_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableCodingType(PByteArray(DescriptorData)[2] and $03, Indent + '  ');
  Items :=  (PByteArray(DescriptorData)[1]-1) div 4;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      CentreFrequency := 0;
      for Loop2 := 3 to 6 do
        CentreFrequency := (CentreFrequency shl 8) or PByteArray(DescriptorData)[Loop*4+Loop2];
      Result := format('%s%s  <centre_frequency>%d</descriptor_length>%s',           [Result, Indent, CentreFrequency, CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorGroupLink(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Number: Cardinal;
  Loop  : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorGroupLink then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'group_link_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <position>%d</position>%s',                                [Result, Indent, PByteArray(DescriptorData)[2], CXmlLf]);
  Number := 0;
  for Loop := 3 to 6 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <group_id>%d</group_id>%s',                                [Result, Indent, Number, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorHierarchy(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorHierarchy then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'hierarchy_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableHierarchyType(PByteArray(DescriptorData)[2] and $0F, Indent + '  ');
  Result := format('%s%s  <hierarchy_layer_index>%d</hierarchy_layer_index>%s',      [Result, Indent, PByteArray(DescriptorData)[3] and $3F, CXmlLf]);
  Result := format('%s%s  <hierarchy_embedded_layer_index>%d</hierarchy_embedded_layer_index>%s', [Result, Indent, PByteArray(DescriptorData)[4] and $3F, CXmlLf]);
  Result := format('%s%s  <hierarchy_channel>%d</hierarchy_channel>%s',              [Result, Indent, PByteArray(DescriptorData)[5] and $3F, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorIbp(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorIbp then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ibp_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <closed_gop_flag>%d</closed_gop_flag>%s',                  [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <identical_gop_flag>%d</identical_gop_flag>%s',            [Result, Indent, (PByteArray(DescriptorData)[2] and $40) shr 6, CXmlLf]);
  Result := format('%s%s  <max_gop_length>%d</max_gop_length>%s',                    [Result, Indent, ((PByteArray(DescriptorData)[2] and $3F) shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorIpmp(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorIpmp then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'ipmp_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorIso639Language(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items       : Integer;
  Loop        : Integer;
  Index       : Integer;
  LanguageCode: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorIso639Language then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'iso_639_language_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 4;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      LanguageCode := 0;
      for Index := 2 to 4 do
        LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Loop * 4 + Index];
      Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
      Result := Result + DvbDecodeTableAudioType(PByteArray(DescriptorData)[Loop * 4 + 5], Indent + '    ');

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorIod(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorIod then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'iod_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorLinkage(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
  Index       : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorLinkage then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'linkage_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <transport_stream_id>%d</transport_stream_id>%s',          [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <original_network_id>%d</original_network_id>%s',          [Result, Indent, (PByteArray(DescriptorData)[4] shl 8) or PByteArray(DescriptorData)[5], CXmlLf]);
  Result := format('%s%s  <service_id>%d</service_id>%s',                            [Result, Indent, (PByteArray(DescriptorData)[6] shl 8) or PByteArray(DescriptorData)[7], CXmlLf]);
  Result := Result + DvbDecodeTableLinkageType(PByteArray(DescriptorData)[8], Indent + '  ');
  if PByteArray(DescriptorData)[8] = $08 then
  begin
    Result := Result + DvbDecodeTableHandOverType((PByteArray(DescriptorData)[9] and $F0) shr 4, Indent + '  ');
    Result := Result + DvbDecodeTableOriginType(PByteArray(DescriptorData)[9] and $01, Indent + '  ');
    Index := 10;
    if ((PByteArray(DescriptorData)[9] and $F0) shr 4) in [$01..$03] then
    begin
      Result := format('%s%s  <network_id>%d</network_id>%s',                        [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
      Inc(Index, 2);
    end;
    if (PByteArray(DescriptorData)[9] and $01) = 0 then
    begin
      Result := format('%s%s  <initial_service_id>%d</initial_service_id>%s',        [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
      Inc(Index, 2);
    end;
    if Index < (PByteArray(DescriptorData)[1]+2) then
    begin
      Name := '';
      for Loop := Index to PByteArray(DescriptorData)[1]+1 do
        Name := Name + PChar(DescriptorData)[Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s  <private_data_byte text="%s">"%s"</private_data_byte>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    end;
  end
  else
  begin
    if PByteArray(DescriptorData)[1] > 7 then
    begin
      Name := '';
      for Loop := 1 to PByteArray(DescriptorData)[1]-7 do
        Name := Name + PChar(DescriptorData)[Loop+8];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s  <private_data_byte text="%s">"%s"</private_data_byte>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    end;
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorLocalTimeOffset(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items       : Integer;
  Index       : Integer;
  CountryCode : Integer;
  Loop        : Integer;
  TimeOffset  : Integer;
  Offset      : AnsiString;
  TimeOfChange: Int64;
  Year        : Word;
  Month       : Word;
  Day         : Word;
  Hours       : Word;
  Minutes     : Word;
  Seconds     : Word;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorLocalTimeOffset then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'local_time_offset_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Index := 2;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);
    CountryCode := 0;
    for Loop := 0 to 2 do
      CountryCode := (CountryCode shl 8) or PByteArray(DescriptorData)[Index+Loop];
    Result := Result + DvbDecodeTableCountryCode(CountryCode, Indent + '    ');
    Inc(Index, 3);
    Result := Result + DvbDecodeTableCountryRegionId((PByteArray(DescriptorData)[Index] and $FC) shr 2, Indent + '    ');
    Result := format('%s%s    <local_time_offset_polarity>%d</local_time_offset_polarity>%s', [Result, Indent, PByteArray(DescriptorData)[Index] and $01, CXmlLf]);
    Inc(Index);
    TimeOffset := (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1];
    Offset   := Chr(Ord('0') + (TimeOffset and $F000) shr 12) + Chr(Ord('0') + ((TimeOffset and $0F00) shr 8));
    Offset   := Offset + ':' + Chr(Ord('0') + (TimeOffset and $00F0) shr 4) + Chr(Ord('0') + (TimeOffset and $0F));
    Result := format('%s%s    <local_time_offset text="%s">%d</local_time_offset_polarity>%s', [Result, Indent, Offset, TimeOffset, CXmlLf]);
    Inc(Index, 2);

    Result := format('%s%s    <time_of_change',                                       [Result, Indent]);
    TimeOfChange := 0;
    for Loop := 0 to 4 do
      TimeOfChange := (TimeOfChange shl 8) or PByteArray(DescriptorData)[Index+Loop];
    DvbFilterXmlDecodeUtcCode(TimeOfChange, Year, Month, Day, Hours, Minutes, Seconds);
    Result := format('%s text="%s">',                                                 [Result, format('%4.4d%2.2d%2.2dTZ%2.2d%2.2d%2.2d', [Year, Month, day, Hours, Minutes, Seconds])]);
    Result := format('%s%d</time_of_change>%s',                                       [Result, TimeOfChange, CXmlLf]);
    Inc(Index, 5);
    TimeOffset := (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1];
    Offset   := Chr(Ord('0') + (TimeOffset and $F000) shr 12) + Chr(Ord('0') + ((TimeOffset and $0F00) shr 8));
    Offset   := Offset + ':' + Chr(Ord('0') + (TimeOffset and $00F0) shr 4) + Chr(Ord('0') + (TimeOffset and $0F));
    Result := format('%s%s    <next_time_offset text="%s">%d</next_time_offset>%s',   [Result, Indent, Offset, TimeOffset, CXmlLf]);
    Inc(Index, 2);

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMaximumBitrate(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Rate: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMaximumBitrate then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'maximum_bitrate_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Rate := ((PByteArray(DescriptorData)[2] and $3F) shl 16) or (PByteArray(DescriptorData)[3] shl 8) or PByteArray(DescriptorData)[4];
  Result := format('%s%s  <maximum_bitrate text="%d bytes/second">%d</maximum_bitrate>%s', [Result, Indent, Rate * 50, Rate, CXmlLf]);


  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMetaData(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMetadata then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'metadata_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMetaDataPointer(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMetadataPointer then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'metadata_pointer_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMetaDataStd(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMetadataStd then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'metadata_std_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorModuleLink(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorModuleLink then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'module_link_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <position>%d</position>%s',                                [Result, Indent, PByteArray(DescriptorData)[2], CXmlLf]);
  Result := format('%s%s  <module_id>%d</module_id>%s',                              [Result, Indent, (PByteArray(DescriptorData)[3] shl 8) or PByteArray(DescriptorData)[4], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMosaic(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items     : Integer;
  Index     : Integer;
  CellLength: Integer;
  Loop      : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMosaic then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'mosaic_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <mosaic_entry_point>%d</mosaic_entry_point>%s',            [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <number_of_horizontal_elementary_cells>%d</number_of_horizontal_elementary_cells>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $70) shr 4, CXmlLf]);
  Result := format('%s%s  <number_of_vertical_elementary_cells>%d</number_of_vertical_elementary_cells>%s', [Result, Indent, PByteArray(DescriptorData)[2] and $07, CXmlLf]);

  Index := 3;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    Result := format('%s%s    <logical_cell_id>%d</logical_cell_id>%s',              [Result, Indent, (PByteArray(DescriptorData)[Index] and $FC) shr 2, CXmlLf]);
    Inc(Index);
    Result := format('%s%s    <logical_cell_presentation_info>%d</logical_cell_presentation_info>%s', [Result, Indent, PByteArray(DescriptorData)[Index] and $07, CXmlLf]);
    Inc(Index);
    CellLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <elementary_cell_field_length>%d</elementary_cell_field_length>%s', [Result, Indent, CellLength, CXmlLf]);
    Inc(Index);
    if CellLength > 0 then
    begin
      for Loop := 0 to CellLength-1 do
      begin
        Result := format('%s%s    <item id="%d">%s',                                  [Result, Indent, Loop, CXmlLf]);
        Result := format('%s%s      <elementary_cell_id>%d</elementary_cell_id>%s',   [Result, Indent, PByteArray(DescriptorData)[Index+Loop] and $3F, CXmlLf]);
        Result := format('%s%s    </item>%s',                                         [Result, Indent, CXmlLf]);
      end;
      Inc(Index, CellLength);
    end;
    Result := format('%s%s    <cell_linkage_info>%d</cell_linkage_info>%s',           [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
    Inc(Index);
    case PByteArray(DescriptorData)[Index-1] of
      $01: begin
             Result := format('%s%s    <bouquet_id>%d</bouquet_id>%s',                [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
           end;
      $02, $03:
           begin
             Result := format('%s%s    <original_network_id>%d</original_network_id>%s', [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
             Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s', [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
             Result := format('%s%s    <service_id>%d</service_id>%s',                   [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
           end;
      $04: begin
             Result := format('%s%s    <original_network_id>%d</original_network_id>%s', [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
             Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s', [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
             Result := format('%s%s    <service_id>%d</service_id>%s',                   [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
             Result := format('%s%s    <event_id>%d</event_id>%s',                       [Result, Indent, (PByteArray(DescriptorData)[Index] shl 8) or PByteArray(DescriptorData)[Index+1], CXmlLf]);
             Inc(Index, 2);
           end;
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMpeg2AacAudio(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMpeg2AacAudio then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'mpeg2_aac_audio_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMpeg4Audio(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMpeg4Audio then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'mpeg4_audio_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMpeg4Text(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMpeg4Text then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'mpeg4_text_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMpeg4Video(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMpeg4Video then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'mpeg4_video_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultilingualBouquetName(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Items       : Integer;
  Loop        : Integer;
  LanguageCode: Integer;
  NameLength  : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultilingualBouquetName then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multilingual_bouquet_name_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Index := 2;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    LanguageCode := 0;
    for Loop := 0 to 2 do
      LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
    Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
    Inc(Index, 3);
    NameLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <bouquet_name_length>%d</bouquet_name_length>%s',      [Result, Indent, nameLength, CXmlLf]);
    Inc(Index);
    if NameLength > 0 then
    begin
      Name := '';
      for Loop := 0 to NameLength-1 do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <char text="%s">"%s"</char>%s',                      [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
      Inc(Index, NameLength);
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultilingualComponent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Items       : Integer;
  Loop        : Integer;
  LanguageCode: Integer;
  NameLength  : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultilingualComponent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multilingual_component_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Index := 2;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    LanguageCode := 0;
    for Loop := 0 to 2 do
      LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
    Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
    Inc(Index, 3);
    NameLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <text_description_length>%d</text_description_length>%s', [Result, Indent, NameLength, CXmlLf]);
    Inc(Index);
    if NameLength > 0 then
    begin
      Name := '';
      for Loop := 0 to NameLength-1 do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <text_char text="%s">"%s"</text_char>%s',            [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
      Inc(Index, NameLength);
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultilingualNetworkName(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Items       : Integer;
  Loop        : Integer;
  LanguageCode: Integer;
  NameLength  : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultilingualNetworkName then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multilingual_network_name_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Index := 2;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    LanguageCode := 0;
    for Loop := 0 to 2 do
      LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
    Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
    Inc(Index, 3);
    NameLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <network_name_length>%d</network_name_length>%s',      [Result, Indent, nameLength, CXmlLf]);
    Inc(Index);
    if NameLength > 0 then
    begin
      Name := '';
      for Loop := 0 to NameLength-1 do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <char text="%s">"%s"</char>%s',                      [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
      Inc(Index, NameLength);
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultilingualServiceName(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Index       : Integer;
  Items       : Integer;
  Loop        : Integer;
  LanguageCode: Integer;
  NameLength  : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultilingualServiceName then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multilingual_service_name_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Index := 2;
  Items := 0;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                       [Result, Indent, Items, CXmlLf]);

    LanguageCode := 0;
    for Loop := 0 to 2 do
      LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
    Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
    Inc(Index, 3);
    NameLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <service_provider_name_length>%d</service_provider_name_length>%s',  [Result, Indent, nameLength, CXmlLf]);
    Inc(Index);
    if NameLength > 0 then
    begin
      Name := '';
      for Loop := 0 to NameLength-1 do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <char text="%s">"%s"</char>%s',                      [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
      Inc(Index, NameLength);
    end;
    NameLength := PByteArray(DescriptorData)[Index];
    Result := format('%s%s    <service_name_length>%d</service_name_length>%s',      [Result, Indent, nameLength, CXmlLf]);
    Inc(Index);
    if NameLength > 0 then
    begin
      Name := '';
      for Loop := 0 to NameLength-1 do
        Name := Name + PChar(DescriptorData)[Index + Loop];
      BinaryString := DvbXmlTransformString(True, Name);
      Result := format('%s%s    <char text="%s">"%s"</char>%s',                      [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
      Inc(Index, NameLength);
    end;

    Result := format('%s%s  </item>%s',                                              [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultiplexBuffer(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultiplexBuffer then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multiplexbuffer_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultiplexBufferUtilization(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Offset: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultiplexBufferUtilization then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multiplex_buffer_utilization_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <bound_valid_flag>%d</bound_valid_flag>%s',                [Result, Indent, ((PByteArray(DescriptorData)[2] and $80) shr 7), CXmlLf]);
  Offset := ((PByteArray(DescriptorData)[2] and $7F) shl 8) or PByteArray(DescriptorData)[3];
  Result := format('%s%s  <ltw_offset_lower_bound>%d</ltw_offset_lower_bound>%s',    [Result, Indent, Offset, CXmlLf]);
  Offset := ((PByteArray(DescriptorData)[4] and $7F) shl 8) or PByteArray(DescriptorData)[5];
  Result := format('%s%s  <ltw_offset_upper_bound>%d</ltw_offset_upper_bound>%s',    [Result, Indent, Offset, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;
{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMultiprotocolEncapsulation(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMultiprotocolEncapsulation then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'multiprotocol_encapsulation_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <deviceid_address_range>%d</deviceid_address_range>%s',    [Result, Indent, (PByteArray(DescriptorData)[2] and $E0) shr 5, CXmlLf]);
  Result := format('%s%s  <deviceid_ip_mapping_flag>%d</deviceid_ip_mapping_flag>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $10) shr 4, CXmlLf]);
  Result := format('%s%s  <alignment_indicator>%d</alignment_indicator>%s',          [Result, Indent, (PByteArray(DescriptorData)[2] and $08) shr 3, CXmlLf]);
  Result := format('%s%s  <max_sections_per_datagram>%d</max_sections_per_datagram>%s', [Result, Indent, PByteArray(DescriptorData)[3], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorMuxCode(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorMuxCode then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'muxcode_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorNetworkName(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorNetworkName then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'network_name_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  if PByteArray(DescriptorData)[1] > 0 then
  begin
    Name := '';
    for Loop := 1 to PByteArray(DescriptorData)[1] do
      Name := Name + PChar(DescriptorData)[1 + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <delivery_system_name text="%s">"%s"</delivery_system_name>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorNptEndpoint(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorNptEndpoint then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'npt_endpoint_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorNptReference(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorNptReference then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'npt_reference_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorNvodReference(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items: Integer;
  Loop : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorNvodReference then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'nvod_reference_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 6;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      Result := format('%s%s    <transport_stream_id>%d</transport_stream_id>%s',    [Result, Indent, (PByteArray(DescriptorData)[(Loop*6)+2] shl 8) or PByteArray(DescriptorData)[(Loop*6)+3], CXmlLf]);
      Result := format('%s%s    <original_network_id>%d</original_network_id>%s',    [Result, Indent, (PByteArray(DescriptorData)[(Loop*6)+4] shl 8) or PByteArray(DescriptorData)[(Loop*6)+5], CXmlLf]);
      Result := format('%s%s    <service_id>%d</service_id>%s',                      [Result, Indent, (PByteArray(DescriptorData)[(Loop*6)+6] shl 8) or PByteArray(DescriptorData)[(Loop*6)+7], CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorParentalRating(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items      : Integer;
  Loop       : Integer;
  Loop2      : Integer;
  CountryCode: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorParentalRating then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'parental_rating_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 4;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      CountryCode := 0;
      for Loop2 := 0 to 2 do
        CountryCode := (CountryCode shl 8) or PByteArray(DescriptorData)[Loop * 4 + Loop2 + 2];
      Result := Result + DvbDecodeTableCountryCode(CountryCode, Indent + '    ');

      Result := Result + DvbDecodeTableRating(PByteArray(DescriptorData)[Loop * 4 + 5], Indent + '    ');
      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorPartialTransportStream(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Number: Integer;
  Loop  : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorPartialTransportStream then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'partial_transport_stream_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Number := PByteArray(DescriptorData)[2] and $3F;
  for Loop := 3 to 4 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <peak_rate text="%d bit/s">%d</peak_rate>%s',              [Result, Indent, Number * 400, Number, CXmlLf]);
  Number := PByteArray(DescriptorData)[5] and $3F;
  for Loop := 6 to 7 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  if Number = $3FFFFF then
    Result := format('%s%s  <minimum_overall_smoothing_rate text="undefined">%d</minimum_overall_smoothing_rate>%s', [Result, Indent, Number, CXmlLf])
  else
    Result := format('%s%s  <minimum_overall_smoothing_rate text="%d bit/s">%d</minimum_overall_smoothing_rate>%s', [Result, Indent, Number * 400, Number, CXmlLf]);
  Number := ((PByteArray(DescriptorData)[8] and $3F) shl 8) or PByteArray(DescriptorData)[9];
  if Number = $3FFF then
    Result := format('%s%s  <maximum_overall_smoothing_buffer text="undefined">%d</maximum_overall_smoothing_buffer>%s', [Result, Indent, Number, CXmlLf])
  else
    Result := format('%s%s  <maximum_overall_smoothing_buffer text="%d bytes">%d</maximum_overall_smoothing_buffer>%s', [Result, Indent, Number, Number, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorPdc(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Pdc: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorPdc then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'pdc_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Pdc := (PByteArray(DescriptorData)[2] and $0F);
  Pdc := (Pdc shl 8) or PByteArray(DescriptorData)[3];
  Pdc := (Pdc shl 8) or PByteArray(DescriptorData)[4];

  Result := format('%s%s  <programme_identification_label',                          [Result, Indent]);
  Result := format('%s text="%2.2d%2.2dT%2.2d%2.2d">',                               [Result, (Pdc and $7800) shr 11, (Pdc and $F8000) shr 15, (Pdc and $7D0) shr 6, Pdc and $3F]);
  Result := format('%s%d</programme_identification_label>%s',                        [Result, Pdc, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorPidCount(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorPidCount then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'pid_count_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <total_number_of_pids>%d</total_number_of_pids>%s',        [Result, Indent, ((PByteArray(DescriptorData)[2] and $1F) shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <min_number_of_pids>%d</min_number_of_pids>%s',            [Result, Indent, ((PByteArray(DescriptorData)[4] and $1F) shl 8) or PByteArray(DescriptorData)[5], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorPrivateDataIndicator(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorPrivateDataIndicator then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'private_data_indicator_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <private_data_indicator>%d</private_data_indicator>%s',    [Result, Indent,
                                                                                      (PByteArray(DescriptorData)[2] shl 24) or
                                                                                      (PByteArray(DescriptorData)[3] shl 16) or
                                                                                      (PByteArray(DescriptorData)[4] shl 8) or
                                                                                       PByteArray(DescriptorData)[5],
                                                                                      CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorPrivateDataSpecifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Number: Cardinal;
  Loop:   Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorPrivateDataSpecifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'private_data_specifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Number := 0;
  for Loop := 2 to 5 do
    Number := (Number shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <private_data_specifier>%d</private_data_specifier>%s',    [Result, Indent, Number, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorRegistration(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Identifier  : Integer;
  Loop        : Integer;
  Name        : AnsiString;
  Binarystring: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorRegistration then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'registration_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Identifier := 0;
  for Loop := 0 to 3 do
    Identifier := (Identifier shl 8) or PByteArray(DescriptorData)[Loop + 2];
  Result := format('%s%s  <format_identifier>%d</format_identifier>%s',              [Result, Indent, Identifier, CXmlLf]);

  if PByteArray(DescriptorData)[1] > 4 then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-5 do
      Name := Name + PChar(DescriptorData)[Loop + 6];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <additional_identification_info text="%s">"%s"</additional_identification_info>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorRelatedContent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorRelatedContent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'related_content_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorScrambling(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorScrambling then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'scrambling_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableScramblingMode(PByteArray(DescriptorData)[2], Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorSatelliteDeliverySystem(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorSatelliteDeliverySystem then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'satellite_delivery_system_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <frequency' ,                                              [Result, Indent]);
  Result := format('%s text="%s%s%s,%s%s%s%s%s GHz">',                               [Result,
              Chr(Ord('0') + ((PByteArray(DescriptorData)[2] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[2] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[3] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[3] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[4] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[4] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[5] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[5] and $0F)), CXmlLf]);
  Result := format('%s%d</frequency>%s',                                             [Result,
              ((PByteArray(DescriptorData)[2] and $F0) shr 4) * 100000000000 +
               (PByteArray(DescriptorData)[2] and $0F)        * 10000000000  +
              ((PByteArray(DescriptorData)[3] and $F0) shr 4) * 1000000000   +
               (PByteArray(DescriptorData)[3] and $0F)        * 100000000    +
              ((PByteArray(DescriptorData)[4] and $F0) shr 4) * 10000000     +
               (PByteArray(DescriptorData)[4] and $0F)        * 1000000      +
              ((PByteArray(DescriptorData)[5] and $F0) shr 4) * 100000       +
               (PByteArray(DescriptorData)[5] and $0F)        * 10000, CXmlLf]);

  Result := format('%s%s  <orbital_position',                                        [Result, Indent]);
  Result := format('%s text="%s%s%s,%s">',                                          [Result,
              Chr(Ord('0') + ((PByteArray(DescriptorData)[6] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[6] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[7] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[7] and $0F)), CXmlLf]);
  Result := format('%s%.1f</orbital_position>%s',                                    [Result,
              ((PByteArray(DescriptorData)[6] and $F0) shr 4) * 100 +
               (PByteArray(DescriptorData)[6] and $0F)        * 10  +
              ((PByteArray(DescriptorData)[7] and $F0) shr 4) * 1   +
               (PByteArray(DescriptorData)[7] and $0F)        * 0.1, CXmlLf]);

  Result := format('%s%s  <west_east_flag',                                          [Result, Indent]);
  case (PByteArray(DescriptorData)[8] and $80) of
    $00: Result := format('%s text="western">',                                      [Result]);
    $80: Result := format('%s text="eastern">',                                      [Result]);
  end;
  Result := format('%s%d</west_east_flag>%s',                                        [Result, (PByteArray(DescriptorData)[8] and $80) shr 8, CXmlLf]);

  Result := Result + DvbDecodeTablePolarization((PByteArray(DescriptorData)[8] and $60) shr 5, Indent + '  ');
  Result := Result + DvbDecodeTableModulationSatellite(PByteArray(DescriptorData)[8] and $1F, Indent + '  ');

  Result := format('%s%s  <symbolrate',                                              [Result, Indent]);
  Result := format('%s text="%s%s%s,%s%s%s%s Msymbols/s">',                          [Result,
              Chr(Ord('0') + ((PByteArray(DescriptorData)[9]  and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[9]  and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[10] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[10] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[11] and $F0) shr 4)),
              Chr(Ord('0') +  (PByteArray(DescriptorData)[11] and $0F)),
              Chr(Ord('0') + ((PByteArray(DescriptorData)[12] and $F0) shr 4))]);
  Result := format('%s%d</symbolrate>%s',                                            [Result,
              ((PByteArray(DescriptorData)[9]  and $F0) shr 4) * 100000000 +
               (PByteArray(DescriptorData)[9]  and $0F)        * 10000000  +
              ((PByteArray(DescriptorData)[10] and $F0) shr 4) * 1000000   +
               (PByteArray(DescriptorData)[10] and $0F)        * 100000    +
              ((PByteArray(DescriptorData)[11] and $F0) shr 4) * 10000     +
               (PByteArray(DescriptorData)[11] and $0F)        * 1000      +
              ((PByteArray(DescriptorData)[12] and $F0) shr 4) * 100, CXmlLf]);

  Result := Result + DvbDecodeTableFecInner(PByteArray(DescriptorData)[12] and $0F, Indent + '  ');

  Result := format('%s%s</descriptor>%s',                                           [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode service descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorService(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Index       : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorService then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'service_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableServiceType(PByteArray(DescriptorData)[2], Indent + '  ');

  Index := 3;
  Result := format('%s%s  <service_provider_name_length>%d</service_provider_name_length>%s', [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);

  if PByteArray(DescriptorData)[Index] > 0 then
  begin
    Name  := '';
    for Loop := 1 to PByteArray(DescriptorData)[Index] do
      Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <service_provider_name text="%s">"%s"</service_provider_name>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Index := 4 + PByteArray(DescriptorData)[3];
  Result := format('%s%s  <service_name_length>%d</service_name_length>%s',          [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
  if PByteArray(DescriptorData)[Index] > 0 then
  begin
    Name  := '';
    for Loop := 1 to PByteArray(DescriptorData)[Index] do
      Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <service_name text="%s">"%s"</service_name>%s',          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorServiceAvailability(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items: Integer;
  Loop : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorServiceAvailability then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'service_availability_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <availability_flag>%d</availability_flag>%s',              [Result, Indent, ((PByteArray(DescriptorData)[2] and $80) shr 7), CXmlLf]);

  Items := (PByteArray(DescriptorData)[1]-1) div 2;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);
      Result := format('%s%s    <cell_id>%d</cell_id>%s',                            [Result, Indent, (PByteArray(DescriptorData)[3 + (Loop * 2)] shl 8) or PByteArray(DescriptorData)[4 + (Loop * 2)], CXmlLf]);
      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorServiceIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorServiceIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'service_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorServiceList(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items: Integer;
  Loop : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorServiceList then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'service_list_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 3;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);
      Result := format('%s%s    <service_id>%d</service_id>%s',                      [Result, Indent, (PByteArray(DescriptorData)[Loop * 3 + 2] shl 8) + PByteArray(DescriptorData)[Loop * 3 + 3], CXmlLf]);
      Result := Result + DvbDecodeTableServiceType(PByteArray(DescriptorData)[Loop * 3 + 4], Indent + '    ');
      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorServiceMove(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorServiceMove then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'service_move_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <new_original_network_id>%d</new_original_network_id>%s',  [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <new_transport_stream_id>%d</new_transport_stream_id>%s',  [Result, Indent, (PByteArray(DescriptorData)[4] shl 8) or PByteArray(DescriptorData)[5], CXmlLf]);
  Result := format('%s%s  <new_service_id>%d</new_service_id>%s',                    [Result, Indent, (PByteArray(DescriptorData)[6] shl 8) or PByteArray(DescriptorData)[7], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode short event descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorShortEvent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Index       : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
  LanguageCode: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorShortEvent then
    Exit;
    Result := format('%s%s<descriptor id="%d">%s',                                   [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'short_event_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  LanguageCode := 0;
  for Index := 2 to 4 do
    LanguageCode := (LanguageCode shl 8) + PByteArray(DescriptorData)[Index];
  Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '  ');

  Index := 5;
  Result := format('%s%s  <event_name_length>%d</event_name_length>%s',              [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
  if PByteArray(DescriptorData)[Index] > 0 then
  begin
    Name  := '';
    for Loop := 1 to PByteArray(DescriptorData)[Index] do
      Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <event_name_char text="%s">"%s"</event_name_char>%s',    [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Index := 6 + PByteArray(DescriptorData)[5];
  Result := format('%s%s  <text_length>%d</text_length>%s',                          [Result, Indent, PByteArray(DescriptorData)[Index], CXmlLf]);
  if PByteArray(DescriptorData)[Index] > 0 then
  begin
    Name  := '';
    for Loop := 1 to PByteArray(DescriptorData)[Index] do
      Name := Name + PChar(DescriptorData)[Index + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <text_char text="%s">"%s"</text_char>%s',                [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorShortSmoothingBuffer(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorShortSmoothingBuffer then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'short_smoothing_buffer_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := Result + DvbDecodeTableSbSize((PByteArray(DescriptorData)[2] and $C0) shr 6, Indent + '  ');
  Result := Result + DvbDecodeTableSbLeakRate(PByteArray(DescriptorData)[2] and $3F, Indent + '  ');

  if PByteArray(DescriptorData)[1] > 1 then
  begin
    Name := '';
    for Loop := 3 to PByteArray(DescriptorData)[1]+1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <dvb_reserved text="%s">"%s"</dvb_reserved>%s',          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorSl(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorSl then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'sl_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorSmoothingBuffer(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Size: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorSmoothingBuffer then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'smoothing_buffer_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Size := ((PByteArray(DescriptorData)[2] and $3F) shl 16) or (PByteArray(DescriptorData)[3] shl 8) or PByteArray(DescriptorData)[4];
  Result := format('%s%s  <sb_leak_rate text="%d bits/second">%d</sb_leak_rate>%s',              [Result, Indent, Size * 400, Size, CXmlLf]);
  Size := ((PByteArray(DescriptorData)[5] and $3F) shl 16) or (PByteArray(DescriptorData)[6] shl 8) or PByteArray(DescriptorData)[7];
  Result := format('%s%s  <sb_size>%d</sb_size>%s',                                  [Result, Indent, Size, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorStd(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorStd then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'std_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <leak_valid_flag>%d</leak_valid_flag>%s',                  [Result, Indent, PByteArray(DescriptorData)[2] and $01, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorStreamEvent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorStreamEvent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'stream_event_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorStreamIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorStreamIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'stream_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <component_tag>%d</component_tag>%s',                      [Result, Indent, PByteArray(DescriptorData)[2], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorStreamMode(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorStreamMode then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'stream_mode_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorStuffing(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Loop        : Integer;
  Name        : AnsiString;
  BinaryString: AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorStuffing then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'stuffing_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  if PByteArray(DescriptorData)[1] > 0 then
  begin
    Name := '';
    for Loop := 1 to PByteArray(DescriptorData)[1] do
       Name := Name + PChar(DescriptorData)[1 + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <stuffing_byte text="%s">"%s"</stuffing_byte>%s',        [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorSubtitling(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items       : Integer;
  Loop        : Integer;
  Loop2       : Integer;
  LanguageCode: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorSubtitling then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'subtitling_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 8;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      LanguageCode := 0;
      for Loop2 := 2 to 4 do
        LanguageCode := (LanguageCode shl 8) or PByteArray(DescriptorData)[(Loop*8)+Loop2];
      Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
      Result := format('%s%s    <subtiting_type text="',                             [Result, Indent]);
      Result := Result + DvbDecodeTableComponentType(3, PByteArray(DescriptorData)[Loop*8+3], True, Indent + '  ');
      Result := format('%s">%d</subtiting_type>%s',                                  [Result, PByteArray(DescriptorData)[Loop*8+3], CXmlLf]);
      Result := format('%s%s    <composition_page_id>%d</composition_page_id>%s',    [Result, Indent, (PByteArray(DescriptorData)[(Loop*8)+4] shl 8) or PByteArray(DescriptorData)[(Loop*8)+5], CXmlLf]);
      Result := format('%s%s    <ancillary_page_id>%d</ancillary_page_id>%s',        [Result, Indent, (PByteArray(DescriptorData)[(Loop*8)+6] shl 8) or PByteArray(DescriptorData)[(Loop*8)+7], CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorSystemClock(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorSystemClock then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'system_clock_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <external_clock_reference_indicator>%d</external_clock_reference_indicator>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := format('%s%s  <clock_accuracy_integer>%d</clock_accuracy_integer>%s',    [Result, Indent, PByteArray(DescriptorData)[2] and $3F, CXmlLf]);
  Result := format('%s%s  <clock_accuracy_exponent>%d</clock_accuracy_exponent>%s',  [Result, Indent, (PByteArray(DescriptorData)[3] and $E0) shr 5, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTelephone(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  CountryPrefixLength: Integer;
  InternationalLength: Integer;
  OperatorCodeLength : Integer;
  NationalLength     : Integer;
  CoreLength         : Integer;
  Name               : AnsiString;
  BinaryString       : AnsiString;
  Index              : Integer;
  Loop               : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTelephone then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'telephone_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <foreign_availability>%d</foreign_availability>%s',        [Result, Indent, (PByteArray(DescriptorData)[2] and $20) shr 5, CXmlLf]);
  Result := format('%s%s  <connection_type>%d</connection_type>%s',                  [Result, Indent, PByteArray(DescriptorData)[2] and $1F, CXmlLf]);
  CountryPrefixLength := (PByteArray(DescriptorData)[3] and $60) shr 5;
  InternationalLength := (PByteArray(DescriptorData)[3] and $1C) shr 2;
  OperatorCodeLength  :=  PByteArray(DescriptorData)[3] and $03;
  NationalLength      := (PByteArray(DescriptorData)[4] and $70) shr 4;
  CoreLength          :=  PByteArray(DescriptorData)[4] and $0F;
  Result := format('%s%s  <country_prefix_length>%d</country_prefix_length>%s',      [Result, Indent, CountryPrefixLength, CXmlLf]);
  Result := format('%s%s  <international_area_code_length>%d</international_area_code_length>%s', [Result, Indent, InternationalLength, CXmlLf]);
  Result := format('%s%s  <operator_code_length>%d</operator_code_length>%s',        [Result, Indent, OperatorCodeLength, CXmlLf]);
  Result := format('%s%s  <national_area_code_length>%d</national_area_code_length>%s', [Result, Indent, NationalLength, CXmlLf]);
  Result := format('%s%s  <core_number_length>%d</core_number_length>%s',            [Result, Indent, CoreLength, CXmlLf]);

  Index := 5;
  if CountryPrefixLength > 0 then
  begin
    Name := '';
    for Loop := Index to Index+CountryPrefixLength-1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <country_prefix_char text="%s">"%s"</country_prefix_char>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    Inc(Index, CountryPrefixLength);
  end;
  if InternationalLength > 0 then
  begin
    Name := '';
    for Loop := Index to Index+InternationalLength-1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <international_area_code_char text="%s">"%s"</international_area_code_char>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    Inc(Index, InternationalLength);
  end;
  if OperatorCodeLength > 0 then
  begin
    Name := '';
    for Loop := Index to Index+OperatorCodeLength-1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <operator_code_char text="%s">"%s"</operator_code_char>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    Inc(Index, OperatorCodeLength);
  end;
  if NationalLength > 0 then
  begin
    Name := '';
    for Loop := Index to Index+NationalLength-1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <national_area_code_char text="%s">"%s"</national_area_code_char>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
    Inc(Index, NationalLength);
  end;
  if CoreLength > 0 then
  begin
    Name := '';
    for Loop := Index to Index+CoreLength-1 do
      Name := Name + PChar(DescriptorData)[Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <core_number_char text="%s">"%s"</core_number_char>%s', [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTeletext(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items       : Integer;
  Loop        : Integer;
  Loop2       : Integer;
  LanguageCode: Integer;
  Page        : AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTeletext then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'teletext_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 5;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      LanguageCode := 0;
      for Loop2 := 2 to 4 do
        LanguageCode := (LanguageCode shl 8) or PByteArray(DescriptorData)[(Loop*5)+Loop2];
      Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
      Result := Result + DvbDecodeTableTeletextType((PByteArray(DescriptorData)[(Loop*5)+5] and $F8) shr 3, Indent + '    ');
      Result := format('%s%s    <teletext_magazine_number>%d</teletext_magazine_number>%s', [Result, Indent, PByteArray(DescriptorData)[(Loop*5)+5] and $07, CXmlLf]);
      Page := Chr(Ord('0') + ((PByteArray(DescriptorData)[(Loop*5)+6] and $F0) shr 4)) + Chr(Ord('0') + (PByteArray(DescriptorData)[(Loop*5)+6] and $0F));
      Result := format('%s%s    <teletext_page_number text="%s">%d</teletext_page_number>%s',  [Result, Indent, Page, PByteArray(DescriptorData)[(Loop*5)+6], CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTerrestrialDeliverySystem(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  CentreFrequency: Cardinal;
  Loop           : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTerrestrialDeliverySystem then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'terrestrial_delivery_system_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  CentreFrequency := 0;
  for Loop := 2 to 5 do
    CentreFrequency := (CentreFrequency shl 8) or PByteArray(DescriptorData)[Loop];
  Result := format('%s%s  <centre_frequency text="%d hz">%d</descriptor_length>%s',  [Result, Indent, CentreFrequency * 10, CentreFrequency, CXmlLf]);
  Result := Result + DvbDecodeTableBandwidth((PByteArray(DescriptorData)[6] and $E0) shr 5, Indent + '  ');
  Result := Result + DvbDecodeTablePriority((PByteArray(DescriptorData)[6] and $10) shr 4, Indent + '  ');
  Result := format('%s%s  <time_slicing_indicator>%d</time_slicing_indicator>%s',    [Result, Indent, (PByteArray(DescriptorData)[6] and $08) shr 3, CXmlLf]);
  Result := format('%s%s  <mpe-fec_indicator>%d</fec_indicator>%s',                  [Result, Indent, (PByteArray(DescriptorData)[6] and $04) shr 2, CXmlLf]);
  Result := Result + DvbDecodeTableConstellation((PByteArray(DescriptorData)[7] and $C0) shr 6, Indent + '  ');
  Result := Result + DvbDecodeTableHierarchyInformation((PByteArray(DescriptorData)[7] and $38) shr 3, Indent + '  ');
  Result := Result + DvbDecodeTableCodeRate(PByteArray(DescriptorData)[7] and $07, '-hp_stream', Indent + '  ');
  Result := Result + DvbDecodeTableCodeRate((PByteArray(DescriptorData)[8] and $E0) shr 5, '-lp_stream', Indent + '  ');
  Result := Result + DvbDecodeTableGuardInterval((PByteArray(DescriptorData)[8] and $18) shr 3, Indent + '  ');
  Result := Result + DvbDecodeTableTransmissionMode((PByteArray(DescriptorData)[8] and $06) shr 1, Indent + '  ');
  Result := format('%s%s  <other_frequency_flag>%d</other_frequency_flag>%s',       [Result, Indent, PByteArray(DescriptorData)[8] and $01, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTargetBackgroundGrid(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Size: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTargetBackgroundGrid then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'target_background_grid_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Size := (PByteArray(DescriptorData)[2] shl 6) or ((PByteArray(DescriptorData)[3] and $FC) shr 2);
  Result := format('%s%s  <horizontal_size>%d</horizontal_size>%s',                  [Result, Indent, Size, CXmlLf]);
  Size := ((PByteArray(DescriptorData)[3] and $03) shl 8) or ((PByteArray(DescriptorData)[4]) shl 4) or ((PByteArray(DescriptorData)[5] and $F0) shr 4);
  Result := format('%s%s  <vertical_size>%d</vertical_size>%s',                      [Result, Indent, Size, CXmlLf]);
  Result := format('%s%s  <aspect_ratio_information>%d</aspect_ratio_information>%s', [Result, Indent, PByteArray(DescriptorData)[5] and $0F, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTimeShiftedEvent(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTimeShiftedEvent then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'time_shifted_event_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <reference_service_id>%d</reference_service_id>%s',        [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);
  Result := format('%s%s  <reference_event_id>%d</reference_event_id>%s',            [Result, Indent, (PByteArray(DescriptorData)[4] shl 8) or PByteArray(DescriptorData)[5], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTimeShiftedService(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTimeShiftedService then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'time_shifted_service_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <reference_service_id>%d</reference_service_id>%s',        [Result, Indent, (PByteArray(DescriptorData)[2] shl 8) or PByteArray(DescriptorData)[3], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTimeSliceFecIdentifier(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTimeSliceFecIdentifier then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'time_slice_fec_identifier_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTransportStream(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Name        : AnsiString;
  BinaryString: AnsiString;
  Loop        : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTransportStream then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'transport_stream_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  if PByteArray(DescriptorData)[1] > 0 then
  begin
    Name := '';
    for Loop := 0 to PByteArray(DescriptorData)[1]-1 do
       Name := Name + PChar(DescriptorData)[2 + Loop];
    BinaryString := DvbXmlTransformString(True, Name);
    Result := format('%s%s  <byte text="%s">"%s"</byte>%s',                          [Result, Indent, DvbFilterXmlDecodeBinaryStringToXmlString(BinaryString), BinaryString, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorTvaId(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorTvaId then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'tva_id_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorUnknown(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'unknown_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorVbiData(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items : Integer;
  Loop  : Integer;
  Index : Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorVbiData then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'vbi_data_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := 0;
  Index := 2;
  while Index < (PByteArray(DescriptorData)[1]+2) do
  begin
    Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Items, CXmlLf]);

    Result := Result + DvbDecodeTableDataServiceId(PByteArray(DescriptorData)[Index], Indent + '    ');
    if PByteArray(DescriptorData)[Index+1] > 0 then
      for Loop := 1 to PByteArray(DescriptorData)[Index+1] do
      begin
        Result := format('%s%s    <item id="%d">%s',                               [Result, Indent, Loop, CXmlLf]);

        if PByteArray(DescriptorData)[Index] in [$01..$07] then
        begin
          Result := format('%s%s      <field_parity>%d</field_parity>',            [Result, Indent, (PByteArray(DescriptorData)[Index+Loop+1] and $20) shr 5, CXmlLf]);
          Result := format('%s%s      <line_offset>%d</line_offset>',              [Result, Indent,  PByteArray(DescriptorData)[Index+Loop+1] and $1F, CXmlLf]);
        end;

        Result := format('%s%s    </item>%s',                                      [Result, Indent, CXmlLf]);
      end;
    Inc(Index, PByteArray(DescriptorData)[Index+1] + 2);

    Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    Inc(Items);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Data to convert
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorVbiTeletext(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Items       : Integer;
  Loop        : Integer;
  LanguageCode: Integer;
  Loop2       : Integer;
  Page        : AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorVbiTeletext then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'vbi_teletext_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Items := PByteArray(DescriptorData)[1] div 5;
  if Items > 0 then
    for Loop := 0 to Items-1 do
    begin
      Result := format('%s%s  <item id="%d">%s',                                     [Result, Indent, Loop, CXmlLf]);

      LanguageCode := 0;
      for Loop2 := 2 to 4 do
        LanguageCode := (LanguageCode shl 8) or PByteArray(DescriptorData)[(Loop * 5) + Loop2];
      Result := Result + DvbDecodeTableIso639LanguageCode(LanguageCode, Indent + '    ');
      Result := Result + DvbDecodeTableTeletextType((PByteArray(DescriptorData)[(Loop* 5)+5] and $F8) shr 3, Indent + '    ');
      Result := format('%s%s    <teletext_magazine_number>%d</teletext_magazine_number>%s', [Result, Indent, PByteArray(DescriptorData)[(Loop* 5)+5] and $07, CXmlLf]);
      Page := Chr(Ord('0') + ((PByteArray(DescriptorData)[(Loop*5)+6] and $F0) shr 4)) + Chr(Ord('0') + (PByteArray(DescriptorData)[(Loop*5)+6] and $0F));
      Result := format('%s%s    <teletext_page_number text="%s">%d</teletext_page_number>%s', [Result, Indent, Page, PByteArray(DescriptorData)[(Loop* 5)+6], CXmlLf]);

      Result := format('%s%s  </item>%s',                                            [Result, Indent, CXmlLf]);
    end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorVideoStream(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorVideoStream then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'video_stream_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Result := format('%s%s  <multiple_frame_rate_flag>%d</multiple_frame_rate_flag>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $80) shr 7, CXmlLf]);
  Result := Result + DvbDecodeTableFrameRateCode((PByteArray(DescriptorData)[2] and $38) shr 3, (PByteArray(DescriptorData)[2] and $80) shr 7, Indent + '  ');
  Result := format('%s%s  <mpeg_1_only_flag>%d</mpeg_1_only_flag>%s',                [Result, Indent, (PByteArray(DescriptorData)[2] and $04) shr 2, CXmlLf]);
  Result := format('%s%s  <constrained_parameter_flag>%d</constrained_parameter_flag>%s', [Result, Indent, (PByteArray(DescriptorData)[2] and $02) shr 1, CXmlLf]);
  Result := format('%s%s  <still_picture_flag>%d</still_picture_flag>%s',            [Result, Indent, PByteArray(DescriptorData)[2] and $01, CXmlLf]);

  if ((PByteArray(DescriptorData)[2] and $04) shr 2) <> 0 then
  begin
    Result := format('%s%s  <profile_and_level_indication>%d</profile_and_level_indication>%s', [Result, Indent, PByteArray(DescriptorData)[3], CXmlLf]);
    Result := format('%s%s  <chroma_format>%d</chroma_format>%s',                    [Result, Indent, (PByteArray(DescriptorData)[4] and $C0) shr 6, CXmlLf]);
    Result := format('%s%s  <frame_rate_extension_flag>%d</frame_rate_extension_flag>%s', [Result, Indent, (PByteArray(DescriptorData)[4] and $20) shr 5, CXmlLf]);
  end;

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          'Header' text (typically indent)
  Returns : <Result>          Empty if failure otherwise XML data

  Descript: Decode descriptor data
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeDescriptorVideoWindow(DescriptorData: Pointer; Indent: AnsiString): AnsiString; stdcall;
var
  Size: Integer;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  if PByteArray(DescriptorData)[0] <> CDvbDescriptorVideoWindow then
    Exit;
  Result := format('%s%s<descriptor id="%d">%s',                                     [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_name>"%s"</descriptor_name>%s',                [Result, Indent, 'video_window_descriptor', CXmlLf]);

  Result := format('%s%s  <descriptor_tag>%d</descriptor_tag>%s',                    [Result, Indent, PByteArray(DescriptorData)[0], CXmlLf]);
  Result := format('%s%s  <descriptor_length>%d</descriptor_length>%s',              [Result, Indent, PByteArray(DescriptorData)[1], CXmlLf]);

  Size := (PByteArray(DescriptorData)[2] shl 6) or ((PByteArray(DescriptorData)[3] and $FC) shr 2);
  Result := format('%s%s  <horizontal_offset>%d</horizontal_offset>%s',              [Result, Indent, Size, CXmlLf]);
  Size := ((PByteArray(DescriptorData)[3] and $03) shl 8) or ((PByteArray(DescriptorData)[4]) shl 4) or ((PByteArray(DescriptorData)[5] and $F0) shr 4);
  Result := format('%s%s  <vertical_offset>%d</vertical_offset>%s',                  [Result, Indent, Size, CXmlLf]);
  Result := format('%s%s  <window_priority>%d</window_priority>%s',                  [Result, Indent, PByteArray(DescriptorData)[5] and $0F, CXmlLf]);

  Result := format('%s%s</descriptor>%s',                                            [Result, Indent, CXmlLf]);
end;

{------------------------------------------------------------------------------
                             Descriptor decoders end
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : <Pid>          Source PID
            <SectionData>  Pointer to section data
  Returns : <Result>       True if no error

  Descript: Store data into section data
  Notes   : Data is only stored when the version number is different from the
            already stored data. When data is stored and all section numbers
            have the same version number (meaning all section numbers have been
            updated) then a callback is generated.
            Data is stored using the section identifier (table_id), the
            current/next indicator and the section number. Unfortunately
            not all sections have these properties, but this is easily detected
            by the section_syntax_indicator.
 ------------------------------------------------------------------------------}

function DvbStoreSection(Pid: Word; SectionData: Pointer): Boolean; stdcall;
var
  TableId             : Byte;
  VersionNumber       : Byte;
  CurrentNextIndicator: Boolean;
  SectionNumber       : Byte;
  LastSectionNumber   : Byte;
  Section             : Byte;
  AllSameVersion      : Boolean;
  PartialData         : Boolean;
  CheckVersion        : Byte;
  Id                  : Word;
  ActiveIndex         : Boolean;
begin
  Result := False;
  TableId := PByteArray(SectionData)[0];
  // Check section_syntax_indicator which indicates current/next data or
  // direct data
  if (PByteArray(SectionData)[1] and $80) = 0 then
  begin
    // Save new section data (to active data!)
    Sections.Section[Sections.ActiveIndex[0, TableId], 0, TableId, 0] := SectionData;
    if SectionCount= $FFFF then
      SectionCount := 0
    else
      Inc(SectionCount);
    Sections.LastVersion[0, TableId] := 0;
    // Switch to next buffers
    ActiveIndexLock.Acquire;
    try
      Sections.ActiveIndex[0, TableId] := not Sections.ActiveIndex[0, TableId];
    finally
      ActiveIndexLock.Release;
    end;
    // Send a message
    if SectionsCommonCallback[TableId].FHandle <> INVALID_HANDLE_VALUE then
      PostMessage(SectionsCommonCallback[TableId].FHandle, SectionsCommonCallback[TableId].FMsg, TableId, 0);
  end
  else
  begin
    // Section carrying current/next data
    Id                   := (PByteArray(SectionData)[3] shl 8) or PByteArray(SectionData)[4];
    VersionNumber        := (PByteArray(SectionData)[5] and $3E) shr 1;
    CurrentNextIndicator := (PByteArray(SectionData)[5] and $01) <> 0;
    SectionNumber        := PByteArray(SectionData)[6];
    LastSectionNumber    := PByteArray(SectionData)[7];

    // Only current information is being processed
    if not CurrentNextIndicator then
      Exit;
    ActiveIndexLock.Acquire;
    try
      ActiveIndex := Sections.ActiveIndex[Id, TableId];
      // Save new section data (to active data!)
      Sections.Section[ActiveIndex, Id, TableId, SectionNumber] := SectionData;
      if (LastSectionNumber = SectionNumber) and (VersionNumber <> Sections.LastVersion[Id, TableId]) then
      begin
        // We get here if we received the last section and if the version is different
        // from the last version which initiated a callback.
        // We now check if all data we have is the same version. If it is the same
        // version then activate the callback (if applicable).
        AllSameVersion := True;
        PartialData    := False;
        Section := 0;
        repeat
          SectionData := Sections.Section[ActiveIndex, Id, TableId, Section];
          // Check for received data
          if not Assigned(SectionData) then
          begin
//            AllSameVersion := False; // DISABLED - some scheduled events don't send all sections .... ??
            PartialData := True;
          end
          else
          begin
            // Check for same versions
            CheckVersion := (PByteArray(SectionData)[5] and $3E) shr 1;
            if CheckVersion <> VersionNumber then
              AllSameVersion := False;
          end;
          Inc(Section);
        until (Section > LastSectionNumber) or (not AllSameVersion);
        // Call callback (if applicable)
        if AllSameVersion then
        begin
          if not PartialData then
            Sections.LastVersion[Id, TableId] := VersionNumber;
          Sections.ActiveIndex[Id, TableId] := not ActiveIndex;
          if SectionCount= $FFFF then
            SectionCount := 0
          else
            Inc(SectionCount);
          // Update references
          PidToServicesReference.Reference[Pid, 0]         := Id;
          ServiceToPidsReference.Reference[Id, 0]          := Pid;
          PidToSectionsReference.Reference[Pid, 0]         := TableId;
          SectionToPidsReference.Reference[TableId, 0]     := Pid;
          ServiceToSectionsReference.Reference[Id, 0]      := TableId;
          SectionToServicesReference.Reference[TableId, 0] := Id;
          // Send a message
          if SectionsCommonCallback[TableId].FHandle <> INVALID_HANDLE_VALUE then
            PostMessage(SectionsCommonCallback[TableId].FHandle, SectionsCommonCallback[TableId].FMsg, TableId, Id);
        end;
      end;
    finally
      ActiveIndexLock.Release;
    end;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <StreamPacketData>  Stream packet data (188 bytes)
  Returns : <Result>            True if no error

  Descript: Decode program specific information data.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbDecodeProgramSpecificInformationData(StreamPacketData: Pointer): Boolean; stdcall;
var
  LoopStart    : Word;
  InitialStart : Word;
  Loop         : Word;
  Checksum32   : Dword;
  Pid          : Word;
  PayloadBytes : Word;
  SectionLength: Word;
begin
  Result := False;
  ResetLock.Acquire;
  try
    // Sanity checks
    if PByteArray(StreamPacketData)[0] <> CDvbPacketSyncWord then
      Exit;
    if not DvbFilterXmlGetPid(StreamPacketData, Pid) then
      Exit;
    try
      begin
        // Because the actual data can be spread out over multiple packets we have
        // to acquire them first. The payload unit start indicator gives us the
        // indication if we are dealing with a new block of data or that data is
        // being 'continued'.
        if (PByteArray(StreamPacketData)[1] and $40) <> 0 then
        begin
          // Start of a new packet resets the indexes
          SectionsReceiving.DataIndex[Pid] := 0;
          // All so-called PSI tables have a first byte which is the <PointerField>.
          // This indicates at what offset the actual data starts. We know that
          // the offset points into this very first packet...
          // First add the pointer field to the start of the payload data
          if not DvbGetPayloadOffset(StreamPacketData, InitialStart) then
          begin
            // No payload then exit without an error
            Result := True;
            Exit;
          end;
          // Add pointer field value to current offset
          SectionsReceiving.PointerField[Pid] := PByteArray(StreamPacketData)[InitialStart];
          LoopStart := InitialStart + SectionsReceiving.PointerField[Pid] + 1;
          // Some providers use an offset in the <PointerField> which is
          // larger than the packet size. This does not comply with the standard
          // since the payload unit start indicator should not be set in such
          // circumstances. However, to be as flexible as possible we have to
          // allow for that .... We need to throw away data in the next packet(s)
          // as well when the <PointerField> is more than this packet ...
          // CORRECTION HAS BEEN REMOVED (NOT TESTED SO JUST SEE IT AS AN ERROR)
          if LoopStart > CDvbPacketSize then
          begin
            // Adjust pointer field value with currently discarded values and
            // exit without an error
  //          Dec(PointerField, CDvbPacketSize - InitialStart - 1);
  //          Result := True;
            Exit;
          end;
          SectionsReceiving.PointerField[Pid] := 0;
          // Calculate additional number of bytes in payload
          PayloadBytes := CDvbPacketSize - LoopStart;
        end
        else
        begin
          // Must have received something (since we are continuing data)
          if SectionsReceiving.DataIndex[Pid] = 0 then
            Exit;
          // If this is the continuing data then we only have to add the
          // complete payload to our data collected sofar
          if not DvbGetPayloadOffset(StreamPacketData, LoopStart) then
          begin
            // No payload then exit without an error
            Result := True;
            Exit;
          end;
          // Calculate number of bytes in payload
          // CORRECTION HAS BEEN REMOVED (NOT TESTED SO JUST SEE IT AS AN ERROR)
  //        if PointerField > 0 then
  //        begin
  //          // If PointerField stretches over multiple packets (non-confirming to the
  //          // standard) then we have to discard additional data in the next
  //          // packet(s)
  //          if (LoopStart + PointerField) > CDvbPacketSize then
  //          begin
  //            // If still no data in this packet
  //            Dec(PointerField, CDvbPacketSize - LoopStart - 1);
  //            PayloadBytes := 0;
  //          end
  //          else
  //          begin
  //            // If data starts in this packet make the necessary correction
  //            PayLoadBytes := CDvbPacketSize - LoopStart - PointerField;
  //            PointerField := 0;
  //          end;
  //        end
  //        else
          PayloadBytes := CDvbPacketSize - LoopStart;
        end;
        if LoopStart < PayloadBytes then
        begin
          // Copy payload data to stream data buffer
          if (SectionsReceiving.DataIndex[Pid] + PayloadBytes) >= SizeOf(SectionsReceiving.StreamData[Pid]^) then
            PayloadBytes := SizeOf(SectionsReceiving.StreamData[Pid]^) - SectionsReceiving.DataIndex[Pid] - 1;
          CopyMemory(@SectionsReceiving.StreamData[Pid][SectionsReceiving.DataIndex[Pid]], @PByteArray(StreamPacketData)[LoopStart], PayloadBytes);
          SectionsReceiving.DataIndex[Pid] := SectionsReceiving.DataIndex[Pid] + PayloadBytes;
          SectionLength := ((PByteArray(SectionsReceiving.StreamData[Pid])[1] and $0F) shl 8) or PByteArray(SectionsReceiving.StreamData[Pid])[2];
          if SectionLength < 6 then
          begin
            Result := True;
            Exit;
          end;
          if (SectionsReceiving.DataIndex[Pid] > (SectionLength + 2)) then      // If we have it all
          begin
            // Before we do any further processing at all we check for a correct checksum
            Checksum32 := DvbResetValueCrc32;
            for Loop := 0 to SectionLength + 2 do
              Checksum32 := DvbNewValueCrc32(Checksum32,
                PByteArray(SectionsReceiving.StreamData[Pid])[Loop]);
            // Check for correct checksum
            if Checksum32 <> 0 then
            begin
              if PacketError = $FFFF then
                PacketError := 0
              else
                Inc(PacketError);
              Exit;
            end;
            // Store the data for each data here (a section). This will also
            // do some check on new section data being received. If so a
            // callback is called
            DvbStoreSection(Pid, SectionsReceiving.StreamData[Pid]);
            // Remove allocated memory of accumulated section
            SectionsReceiving.StreamData[Pid] := nil;
          end;
        end;
        Result := True;
      end;
    except
      if PacketError = $FFFF then
        PacketError := 0
      else
        Inc(PacketError);
    end;
  finally
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Preset>  True if preset all (otherwise only partial data
                      is reset). False if callbacks should be left alone
                      left alone.
  Returns : -

  Descript: Set all table references to unassigned status and create required
            data.
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbFilterXmlResetData(Preset: Boolean); stdcall;
var
  SectionId: Byte;
begin
  // Allow to be called without being initialized
  if not Assigned(ResetLock) then
    Exit;
  ResetLock.Acquire;
  ActiveIndexLock.Acquire;
  try
    if Preset then
      for SectionId := Low(SectionsCommonCallback) to High(SectionsCommonCallback) do
        SectionsCommonCallback[SectionId].FHandle := INVALID_HANDLE_VALUE;

    FreeAndNil(PidToServicesReference);
    FreeAndNil(ServiceToPidsReference);
    FreeAndNil(PidToSectionsReference);
    FreeAndNil(SectionToPidsReference);
    FreeAndNil(ServiceToSectionsReference);
    FreeAndNil(SectionToServicesReference);
    FreeAndNil(Sections);
    FreeAndNil(SectionsReceiving);
    Sections                   := TDvbSections.Create;
    SectionsReceiving          := TDvbReceivingSections.Create;
    PidToServicesReference     := TDvbReference.Create;
    ServiceToPidsReference     := TDvbReference.Create;
    PidToSectionsReference     := TDvbReference.Create;
    SectionToPidsReference     := TDvbReference.Create;
    ServiceToSectionsReference := TDvbReference.Create;
    SectionToServicesReference := TDvbReference.Create;
    // Set up default packet callbacks
    if Preset then
      for SectionId := 0 to $1F do
        PacketFilters[SectionId]  := @DvbDecodeProgramSpecificInformationData;
  finally
    ActiveIndexLock.Release;
    ResetLock.Release;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Generate CRC tables
  Notes   :
------------------------------------------------------------------------------}

procedure DvbGenerateCrcTables;
var
  ByteLoop: Byte;
  BitLoop : Byte;
  Crc16   : Word;
  Crc32   : Dword;
begin
  // Go through the whole array
  for ByteLoop := 0 to 255 do
  begin
    Crc16 := ByteLoop shl 8;
    Crc32 := ByteLoop shl 24;
    for BitLoop := 0 to 7 do
    begin
      if (Crc16 and $8000) <> 0 then
        Crc16 := ((Crc16 and $7FFF) shl 1) xor CDvbPolynomial16
      else
        Crc16 := (Crc16 shl 1);
      if (Crc32 and $80000000) <> 0 then
        Crc32 := ((Crc32 and $7FFFFFFF) shl 1) xor CDvbPolynomial32
      else
        Crc32 := (Crc32 shl 1);
      Crc16Table[ByteLoop] := Crc16;
      Crc32Table[ByteLoop] := Crc32;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc16: Word;
begin
  Result := CDvbCrcInitial16;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>    Initial checksum

  Descript: Return initial checksum
  Notes   :
------------------------------------------------------------------------------}

function DvbResetValueCrc32: Dword;
begin
  Result := CDvbCrcInitial32;
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC16 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial16> as
            starting value
------------------------------------------------------------------------------}

function DvbNewvalueCrc16(Checksum: Word; Data: Byte): Word;
begin
  Result := ((Checksum and $FF) shl 8) xor Crc16Table[((Checksum shr 8) xor Data)
    and $FF];
end;

{------------------------------------------------------------------------------
  Params  : <Checksum>  Original checksum
            <Data>      Data to add to checksum
  Returns : <Result>    New checksum

  Descript: Calculate new CRC32 checksum.
  Notes   : When starting a new checksum make sure to use <CCrcInitial32> as
            starting value
------------------------------------------------------------------------------}

function DvbNewValueCrc32(Checksum: Dword; Data: Byte): Dword;
begin
  Result := ((Checksum and $FFFFFF) shl 8) xor Crc32Table[((Checksum shr 24) xor
    Data) and $FF];
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Generate function calls for decoding sections.
  Notes   :
------------------------------------------------------------------------------}

procedure DvbGenerateDecodeSections;
var
  MatchSection: Byte;
begin
  for MatchSection := Low(SectionsDecode) to High(SectionsDecode) do
  begin
    case MatchSection of
      CDvbSectionProgramAssociation:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionProgramAssociation;
          SectionsDecodeStartName[MatchSection] := '<program_association_sections>';
        end;
      CDvbSectionCa:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionCa;
          SectionsDecodeStartName[MatchSection] := '<ca_sections>';
        end;
      CDvbSectionTsProgramMap:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionTsProgramMap;
          SectionsDecodeStartName[MatchSection] := '<ts_program_map_sections>';
        end;
      CDvbSectionTransportStreamDescription:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionTransportStreamDescription;
          SectionsDecodeStartName[MatchSection] := '<transport_stream_description_sections>';
        end;
      CDvbSectionMetadata:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionMetadata;
          SectionsDecodeStartName[MatchSection] := '<metadata_sections>';
        end;
      CDvbSectionMeasurement..CDvbSectionNetworkStatus:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionTestdata;
          SectionsDecodeStartName[MatchSection] := '<testdata_sections>';
        end;
      CDvbSectionDsmcc0..CDvbSectionDsmcc1:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionDsmcc;
          SectionsDecodeStartName[MatchSection] := '<dsmcc_sections>';
        end;
      CDvbSectionDsmccAddressable:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionDsmccAddressable;
          SectionsDecodeStartName[MatchSection] := '<dsmcc_addressable_sections>';
        end;
      CDvbSectionActualNetworkNetworkInformation,
      CDvbSectionOtherNetworkNetworkInformation:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionNetworkInformation;
          SectionsDecodeStartName[MatchSection] := '<network_information_sections>';
        end;
      CDvbSectionActualTransportStreamServiceDescription,
      CDvbSectionOtherTransportStreamServiceDescription:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionServiceDescription;
          SectionsDecodeStartName[MatchSection] := '<service_description_sections>';
        end;
      CDvbSectionBouquetAssociation:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionBouquetAssociation;
          SectionsDecodeStartName[MatchSection] := '<bouquet_association_sections>';
        end;
      CDvbSectionActualTransportStreamPresentFollowingEventInformation..
      CDvbSectionOtherTransportStreamScheduleEventInformationF:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionEventInformation;
          SectionsDecodeStartName[MatchSection] := '<event_information_sections>';
        end;
      CDvbSectionTimeDate:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionTimeDate;
          SectionsDecodeStartName[MatchSection] := '<time_date_sections>';
        end;
      CDvbSectionRunningStatus:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionRunningStatus;
          SectionsDecodeStartName[MatchSection] := '<running_status_sections>';
        end;
      CDvbSectionStuffing:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionStuffing;
          SectionsDecodeStartName[MatchSection] := '<stuffing_sections>';
        end;
      CDvbSectionTimeOffset:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionTimeOffset;
          SectionsDecodeStartName[MatchSection] := '<time_offset_sections>';
        end;
      CDvbSectionResolutionNotification:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionResolutionNotification;
          SectionsDecodeStartName[MatchSection] := '<resolution_notification_sections>';
        end;
      CDvbSectionContainer:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionContainer;
          SectionsDecodeStartName[MatchSection] := '<container_sections>';
        end;
      CDvbSectionRelatedContent:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionRelatedContent;
          SectionsDecodeStartName[MatchSection] := '<related_content_sections>';
        end;
      CDvbSectionContentIdentifier:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionContentIdentifier;
          SectionsDecodeStartName[MatchSection] := '<content_identifier_sections>';
        end;
      CDvbSectionMpeFec:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionMpeFec;
          SectionsDecodeStartName[MatchSection] := '<mpe_fec_sections>';
        end;
      CDvbSectionDiscontinuityInformation:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionDiscontinuityInformation;
          SectionsDecodeStartName[MatchSection] := '<discontinuity_information_sections>';
        end;
      CDvbSectionSelectionInformation:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionSelectionInformation;
          SectionsDecodeStartName[MatchSection] := '<selection_information_sections>';
        end;
      CDvbSectionCaMessageEcm0..
      CDvbSectionCaMessageEmmAndCaSystemPrivateD:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionCaMessage;
          SectionsDecodeStartName[MatchSection] := '<ca_message_sections>';
        end;
      CDvbSectionDataEventTable:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionDataEventTable;
          SectionsDecodeStartName[MatchSection] := '<data_event_table_sections>';
        end;
      CDvbSectionDataServiceTable:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionDataServiceTable;
          SectionsDecodeStartName[MatchSection] := '<data_service_table_sections>';
        end;
      CDvbSectionNetworkResourcesTable:
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionNetworkResourcesTable;
          SectionsDecodeStartName[MatchSection] := '<network_resource_table_sections>';
        end;
      CDvbSectionLongTermServiceTable :
        begin
          SectionsDecode[MatchSection]          := @DvbDecodeSectionLongTermServiceTable;
          SectionsDecodeStartName[MatchSection] := '<long_term_service_table_sections>';
        end;
      else
          SectionsDecode[MatchSection] := nil;
    end;
    if Assigned(@SectionsDecode[MatchSection]) then
    begin
      SectionsDecodeStopName[MatchSection]  := SectionsDecodeStartName[MatchSection];
      Insert('/', SectionsDecodeStopName[MatchSection], 2);
    end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Generate function calls for decoding descriptors.
  Notes   :
------------------------------------------------------------------------------}

procedure DvbGenerateDecodeDescriptors;
var
  Descriptor: Byte;
begin
  for Descriptor := Low(DescriptorsDecode) to High(DescriptorsDecode) do
  begin
    case Descriptor of
      CDvbDescriptorVideoStream                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorVideoStream;
      CDvbDescriptorAudioStream                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAudioStream;
      CDvbDescriptorHierarchy                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorHierarchy;
      CDvbDescriptorRegistration                : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorRegistration;
      CDvbDescriptorDataStreamAlignment         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDataStreamAlignment;
      CDvbDescriptorTargetBackgroundGrid        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTargetBackgroundGrid;
      CDvbDescriptorVideoWindow                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorVideoWindow;
      CDvbDescriptorCa                          : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCa;
      CDvbDescriptorIso639Language              : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorIso639Language;
      CDvbDescriptorSystemClock                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorSystemClock;
      CDvbDescriptorMultiplexBufferUtilization  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultiplexBufferUtilization;
      CDvbDescriptorCopyright                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCopyright;
      CDvbDescriptorMaximumBitrate              : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMaximumBitrate;
      CDvbDescriptorPrivateDataIndicator        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorPrivateDataIndicator;
      CDvbDescriptorSmoothingBuffer             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorSmoothingBuffer;
      CDvbDescriptorStd                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorStd;
      CDvbDescriptorIbp                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorIbp;

      CDvbDescriptorCarouselIdentifier          : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCarouselIdentifier;
      CDvbDescriptorAssociationTag              : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAssociationTag;
      CDvbDescriptorDeferredAssociationTags     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDeferredAssociationTags;
      CDvbDescriptorNptReference                : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorNptReference;
      CDvbDescriptorNptEndpoint                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorNptEndpoint;
      CDvbDescriptorStreamMode                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorStreamMode;
      CDvbDescriptorStreamEvent                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorStreamEvent;

      CDvbDescriptorMpeg4Video                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMpeg4Video;
      CDvbDescriptorMpeg4Audio                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMpeg4Audio;
      CDvbDescriptorIod                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorIod;
      CDvbDescriptorSl                          : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorSl;
      CDvbDescriptorFmc                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorFmc;
      CDvbDescriptorExternalEsId                : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorExternalEsId;
      CDvbDescriptorMuxCode                     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMuxCode;
      CDvbDescriptorFmxBufferSize               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorFmxBufferSize;
      CDvbDescriptorMultiplexBuffer             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultiplexBuffer;
      CDvbDescriptorContentLabeling             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorContentLabeling;
      CDvbDescriptorMetadataPointer             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMetadataPointer;
      CDvbDescriptorMetadata                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMetadata;
      CDvbDescriptorMetadataStd                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMetadataStd;

      CDvbDescriptorAvcVideo                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAvcVideo;
      CDvbDescriptorIpmp                        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorIpmp;
      CDvbDescriptorAvcTimingAndHrd             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAvcTimingAndHrd;
      CDvbDescriptorMpeg2AacAudio               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMpeg2AacAudio;
      CDvbDescriptorMpeg4Text                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMpeg4Text;

      CDvbDescriptorNetworkName                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorNetworkName;
      CDvbDescriptorServiceList                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorServiceList;
      CDvbDescriptorStuffing                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorStuffing;
      CDvbDescriptorSatelliteDeliverySystem     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorSatelliteDeliverySystem;
      CDvbDescriptorCableDeliverySystem         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCableDeliverySystem;
      CDvbDescriptorVbiData                     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorVbiData;
      CDvbDescriptorVbiTeletext                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorVbiTeletext;
      CDvbDescriptorBouquetName                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorBouquetName;
      CDvbDescriptorService                     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorService;
      CDvbDescriptorCountryAvailability         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCountryAvailability;
      CDvbDescriptorLinkage                     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorLinkage;
      CDvbDescriptorNvodReference               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorNvodReference;
      CDvbDescriptorTimeShiftedService          : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTimeShiftedService;
      CDvbDescriptorShortEvent                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorShortEvent;
      CDvbDescriptorExtendedEvent               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorExtendedEvent;
      CDvbDescriptorTimeShiftedEvent            : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTimeShiftedEvent;
      CDvbDescriptorComponent                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorComponent;
      CDvbDescriptorMosaic                      : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMosaic;
      CDvbDescriptorStreamIdentifier            : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorStreamIdentifier;
      CDvbDescriptorCaIdentifier                : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCaIdentifier;
      CDvbDescriptorContent                     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorContent;
      CDvbDescriptorParentalRating              : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorParentalRating;
      CDvbDescriptorTeletext                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTeletext;
      CDvbDescriptorTelephone                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTelephone;
      CDvbDescriptorLocalTimeOffset             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorLocalTimeOffset;
      CDvbDescriptorSubtitling                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorSubtitling;
      CDvbDescriptorTerrestrialDeliverySystem   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTerrestrialDeliverySystem;
      CDvbDescriptorMultilingualNetworkName     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultilingualNetworkName;
      CDvbDescriptorMultilingualBouquetName     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultilingualBouquetName;
      CDvbDescriptorMultilingualServiceName     : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultilingualServiceName;
      CDvbDescriptorMultilingualComponent       : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultilingualComponent;
      CDvbDescriptorPrivateDataSpecifier        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorPrivateDataSpecifier;
      CDvbDescriptorServiceMove                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorServiceMove;
      CDvbDescriptorShortSmoothingBuffer        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorShortSmoothingBuffer;
      CDvbDescriptorFrequencyList               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorFrequencyList;
      CDvbDescriptorPartialTransportStream      : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorPartialTransportStream;
      CDvbDescriptorDataBroadcast               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDataBroadcast;
      CDvbDescriptorScrambling                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorScrambling;
      CDvbDescriptorDataBroadcastId             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDataBroadcastId;
      CDvbDescriptorTransportStream             : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTransportStream;
      CDvbDescriptorDsng                        : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDsng;
      CDvbDescriptorPdc                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorPdc;
      CDvbDescriptorAc3                         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAc3;
      CDvbDescriptorAncillaryData               : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAncillaryData;
      CDvbDescriptorCellList                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCellList;
      CDvbDescriptorCellFrequencyLink           : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCellFrequencyLink;
      CDvbDescriptorAnnouncementSupport         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAnnouncementSupport;
      CDvbDescriptorApplicationSignalling       : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorApplicationSignalling;
      CDvbDescriptorAdaptationFieldData         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAdaptationFieldData;
      CDvbDescriptorServiceIdentifier           : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorServiceIdentifier;
      CDvbDescriptorServiceAvailability         : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorServiceAvailability;
      CDvbDescriptorDefaultAuthority            : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDefaultAuthority;
      CDvbDescriptorRelatedContent              : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorRelatedContent;
      CDvbDescriptorTvaId                       : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTvaId;
      CDvbDescriptorContentIdentifier           : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorContentIdentifier;
      CDvbDescriptorTimeSliceFecIdentifier      : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorTimeSliceFecIdentifier;
      CDvbDescriptorEcmRepetitionRate           : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorEcmRepetitionRate;
      CDvbDescriptorEnhancedAc3                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorEnhancedAc3;

      CDvbDescriptorAtscCa                      : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorAtscCa;
      CDvbDescriptorDataService                 : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDataService;
      CDvbDescriptorPidCount                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorPidCount;
      CDvbDescriptorDownload                    : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorDownload;
      CDvbDescriptorMultiprotocolEncapsulation  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorMultiprotocolEncapsulation;
      CDvbDescriptorModuleLink                  : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorModuleLink;
      CDvbDescriptorCrc32                       : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorCrc32;
      CDvbDescriptorGroupLink                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorGroupLink;

      CDvbDescriptorForbidden                   : DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorForbidden;
      else                                        DescriptorsDecode[Descriptor] := @DvbDecodeDescriptorUnknown;
    end;
  end;
end;

{------------------------------------------------------------------------------
                        External accessible functions start
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Reset error count.
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbFilterXmlResetErrors; stdcall;
begin
  PacketError      := 0;
  PacketSyncErrors := 0;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Errors counted

  Descript: Get error count.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetErrors: Word; stdcall;
begin
  Result := PacketError;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Synchronization errors counted

  Descript: Get synchronization error count.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetPacketSyncErrors: Word; stdcall;
begin
  Result := PacketSyncErrors;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  New sections counter

  Descript: Get new sections count.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetSectionCount: Word; stdcall;
begin
  Result := SectionCount;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  debug value

  Descript: Get debug value.
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetDebug: Word; stdcall;
begin
  Result := Debug;
end;

{------------------------------------------------------------------------------
  Params  : <Pid>        Pid identifier
  Returns : <Result>     Assigned if valid Pid
                         Nil otherwise

  Descript: Get filter for Pid (only if valid Pid)
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetPacketFilter(Pid: Word): TDvbPacketFilter; stdcall;
begin
  Result := nil;
  if Pid > High(PacketFilters) then
    Exit;
  Result := PacketFilters[Pid];
end;

{------------------------------------------------------------------------------
  Params  : <Pid>  Pid identifier
  Returns : -

  Descript: Avtivate (internal) filter filter for Pid (only if valid Pid)
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbFilterXmlActivatePacketFilter(Pid: Word); stdcall;
begin
  if Pid > High(PacketFilters) then
    Exit;
  PacketFilters[Pid] := @DvbDecodeProgramSpecificInformationData;
end;

{------------------------------------------------------------------------------
  Params  : <Pid>  Pid identifier
  Returns : -

  Descript: Avtivate (internal) filter filter for Pid (only if valid Pid)
  Notes   :
 ------------------------------------------------------------------------------}

procedure DvbFilterXmlDeActivatePacketFilter(Pid: Word); stdcall;
begin
  if Pid > High(PacketFilters) then
    Exit;
  PacketFilters[Pid] := nil;
end;

{------------------------------------------------------------------------------
  Params  : <Section>  Section number callback is for
            <hWnd>     Window handle to send message to
            <Msg>      Message to send
  Returns : <Result>   True if success

  Descript: Set message 'callback' for section number (table_id)
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlSetSignallingCallback(Section: Byte; hWnd: HWND; Msg: UINT): Boolean; stdcall;
begin
  SectionsCommonCallback[Section].FMsg    := Msg;
  SectionsCommonCallback[Section].FHandle := hWnd;
  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <DescriptorData>  Pointer to descriptor data
            <Indent>          Indent to add before each line
  Returns : <Result>          XML data

  Descript: Get XML information for descriptor
  Notes   :
 ------------------------------------------------------------------------------}

function DvbGetXmlInfoFromDescriptor(DescriptorData: Pointer; Indent: string): AnsiString;
begin
  Result := '';
  if not Assigned(DescriptorData) then
    Exit;
  Result := DescriptorsDecode[PByteArray(DescriptorData)[0]](DescriptorData, Indent);
end;


{------------------------------------------------------------------------------
  Params  : <What>           0..$1FFF = From indicated PID only
                                 < 0  = From indicated service only
                             >$1FFF   = From all available data
            <Section>        Section
            <SectionNumber>  Section number (<0 or >255 for all)
                             Can be used for sections which can contain a LOT
                             of data, for example scheduled events
            <Indent>         Indent to add before each line
  Returns : <Result>         XML data

  Descript: Get XML information for section
  Notes   :
 ------------------------------------------------------------------------------}

function DvbFilterXmlGetXmlInfoFromSection(What: Integer; Section: Byte; SectionNumber: Integer; Indent: string): AnsiString; stdcall;
var
  Last                : Byte;
  Loop                : Byte;
  ThereIsData         : Boolean;
  Active              : Boolean;
  MatchService        : Word;
  MatchPid            : Word;
  PidIndex            : Word;
  MatchSection        : Word;
  SectionIndex        : Word;
  Valid               : Boolean;
  UseService          : Boolean;
  UsePid              : Boolean;
  UseSection          : Boolean;
  NumberFrom          : Integer;
  NumberTo            : Integer;
  SectionStopValue    : AnsiString;              // 'Closing' text when new section detected
  SectionStartedNumber: Integer;                 // Last section with a starting text and needs a closing text
  IdStopvalue         : AnsiString;              // 'Closing' text when new id detected
  IdStartedNumber     : Integer;                 // Last id with a starting text and needs a closing text
  SectionData         : Pointer;                 // Pointer to section data
  SectionNumberStart  : Integer;                 // First section number
  SectionNumberEnd    : Integer;                 // Last section number
  ResultValue         : AnsiString;
begin
  Result       := '';
  ResultValue  := '';
  UsePid       := False;
  UseService   := False;
  UseSection   := False;
  PidIndex     := 0;
  MatchService := 0;
  MatchPid     := 0;
  MatchSection := section;
  SectionIndex := 0;
  if (SectionNumber < 0) or (SectionNumber > 255)  then
  begin
    SectionNumberStart := 0;
    SectionNumberEnd   := 255;
  end
  else
  begin
    SectionNumberStart := Lo(SectionNumber);
    SectionNumberEnd   := SectionNumberStart;
  end;

  if What > $1FFF then
  begin
    // All available data
    UseSection   := True;
    SectionIndex := 0;
  end
  else
  begin
    if What >= 0 then
    begin
      // Only for indicated PID
      UsePid   := True;
      PidIndex := 0;
      MatchPid := What;
    end
    else
    begin
      // Only for indicated service
      UseService   := True;
      MatchService := -What;
    end;
  end;

  IdStopValue          := '';
  SectionStopValue     := '';
  SectionStartedNumber := -1;
  IdStartedNumber      := -1;
  Valid                := True;
  ResetLock.Acquire;
  try
    while Valid do
    begin
      Valid := True;
      // Check if reference exist, and what service it is
      if UseService then
      begin
        // When referring to a specific service it is easy. No reference needed
        // since the data is stored using the service as base.
        // Just use one of the cross references to check it's availability
        Valid := (ServiceToSectionsReference.Reference[MatchService, 0] <> -1);
      end;
      if UsePid then
      begin
        Valid := (PidToServicesReference.Reference[MatchPid, PidIndex] <> -1);
        if Valid then
          MatchService := PidToServicesReference.Reference[MatchPid, PidIndex];
        Inc(PidIndex);
      end;
      if UseSection then
      begin
        Valid := (SectionToServicesReference.Reference[MatchSection, SectionIndex] <> -1);
        if Valid then
          MatchService := SectionToServicesReference.Reference[MatchSection, SectionIndex];
        Inc(SectionIndex);
      end;
      if Valid then
      begin
        ActiveIndexLock.Acquire;
        try
          Active := not Sections.ActiveIndex[MatchService, MatchSection];
          SectionData := Sections.Section[Active, MatchService, MatchSection, 0];
          Valid := Assigned(SectionData);
          if Valid then
          begin
            // Get highest section number
            Last        := PByteArray(SectionData)[7];
            ThereIsData := False;
            if Last <> $FF then
            begin
              // Set initial range
              NumberFrom := SectionNumberStart;
              NumberTo   := SectionNumberEnd;
              // Limit range according to highest section number
              if NumberTo > Last then
                NumberTo := Last;
              // If highest section number outside of selected range set invalid range
              if NumberFrom > Last then
                NumberTo := -1;
              // If valid range
              if NumberTo >= NumberFrom then
              begin
                for Loop := NumberFrom to NumberTo do
                  if Assigned(Sections.Section[Active, MatchService, MatchSection, Loop]) then
                  begin
                    ThereIsData := True;
                    Break;
                  end;
              end;
              if ThereIsData then
              begin
                // End/Start ID 'section'
                if MatchService <> IdStartedNumber then
                begin
                  ResultValue := ResultValue + SectionStopValue;
                  SectionStopValue := '';
                  SectionStartedNumber := -1;
                  ResultValue := ResultValue + IdStopValue;
                  IdStopValue := format('%s</id>%s', [Indent, CXmlLf]);
                  ResultValue := format('%s%s<id>%d%s', [ResultValue, Indent, MatchService, CXmlLf]);
                  IdStartedNumber := MatchService;
                end;

                if SectionStartedNumber <> MatchSection then
                begin
                  ResultValue := ResultValue + SectionStopValue;
                  SectionStopValue := format('%s%s%s', [Indent + '  ', SectionsDecodeStopName[MatchSection], CXmlLf]);
                  ResultValue := format('%s%s%s%s', [ResultValue, Indent + '  ', SectionsDecodeStartName[MatchSection], CXmlLf]);
                  SectionStartedNumber := MatchSection;
                end;
                for Loop := NumberFrom to NumberTo do
                  ResultValue := ResultValue + SectionsDecode[MatchSection](Sections.Section[Active, MatchService, MatchSection, Loop], Indent + '    ');
              end;
            end;
          end;
        finally
          ActiveIndexLock.Release;
        end;
      end;
      if UseService then
        Valid := False;
    end;
  finally
    ResetLock.Release;
  end;
  ResultValue := ResultValue + SectionStopValue;
  ResultValue := ResultValue + IdStopValue;
  Result := ResultValue;
end;

{------------------------------------------------------------------------------
                        External accessible functions end
 ------------------------------------------------------------------------------}

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization of unit
  Notes   : Create objects
------------------------------------------------------------------------------}

procedure DvbFilterXmlInitialize;
begin
  // The CRC tables are generated at startup
  DvbGenerateCrcTables;
  DvbGenerateDecodeSections;
  DvbGenerateDecodeDescriptors;

  ActiveIndexLock        := TCriticalSection.Create;
  ResetLock              := TCriticalSection.Create;
  SectionsReceiving      := TDvbReceivingSections.Create;
  Sections               := TDvbSections.Create;
  DvbFilterXmlResetData(True);
  Debug                  := 0;
  PacketError            := 0;
  PacketSyncErrors       := 0;
  SectionCount           := 0;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization of unit
  Notes   : Destroy objects
------------------------------------------------------------------------------}

procedure DvbFilterXmlFinalize;
begin
  DvbFilterXmlResetData(True);
  FreeAndNil(SectionsReceiving);
  FreeAndNil(Sections);
  FreeAndNil(PidToServicesReference);
  FreeAndNil(ServiceToPidsReference);
  FreeAndNil(PidToSectionsReference);
  FreeAndNil(SectionToPidsReference);
  FreeAndNil(ServiceToSectionsReference);
  FreeAndNil(SectionToServicesReference);
  FreeAndNil(ResetLock);
  FreeAndNil(ActiveIndexLock);
end;


initialization
//  DvbFilterXmlInitialize;  Called manually

finalization
  DvbFilterXmlFinalize;

end.

