{******************************************************************************}
{ FileName............: Saa7146aGpio                                           }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.01                                                   }
{------------------------------------------------------------------------------}
{  GPIO support                                                                }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  1.00   20030614 - Initial release                                           }
{  1.01   20040418 - Added <Saa7146aSetGpio>                                   }
{                  - added parameter to <Saa7146aInitializeGpio> to allow      }
{                    for other cards                                           }
{                  - Added parameter to <Saa7146aIssueReset> to allow for      }
{                    different cards                                           }
{******************************************************************************}
{                                                                              }
{ The GPIO signals of the SAA7146A are connected to the following hardware:    }
{ GPIO  I/O   Card  Description                                                }
{   0   IN    NOVA  Common Interface Interrupt                                 }
{                   Typically set low when a card is inserted/removed          }
{   1   OUT   NOVA  Count on MPEG in - start (1) or count on MPEG out - start  }
{                   (0)                                                        }
{   2   OUT   NOVA  RESET# of tuner                                            }
{   3   IN    NOVA  Signal from infrared device that new data is available.    }
{                   A 2-byte DEBI read (using 16-bit access) will read the     }
{                   data. Typically the infrared device generates 3 signals so }
{                   all it's data can be read when new data is available.      }
{ **************************************************************************** }

unit Saa7146aGpio;

interface
uses
  Saa7146aInterface,
  Saa7146aRegisters,
  Windows;

function Saa7146aInitializeGpio(Handle: THandle; DefaultCard: Boolean): Boolean;
function Saa7146aSetGpio(Handle: THandle; Gpio: Byte; OnOff: Boolean): Boolean;
function Saa7146aIssueReset(Handle: THandle; Gpio: Byte): Boolean;
function Saa7146aSelectCommonInterface(Handle: THandle; Select: Boolean):
  Boolean;

implementation

{------------------------------------------------------------------------------
  Params  : <Handle>       <Saa7146aCreateFile> handle
            <DefaultCard>  True when default card used (Nova / Budget)
  Returns : <Result>       True if success

  Descript: Initialize GPIO signals (all inputs)
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aInitializeGpio(Handle: THandle; DefaultCard: Boolean): Boolean;
begin
  Result := False; // Assume we fail

  // All input
  if not Saa7146aWriteToSaa7146aRegister(Handle, CSaa7146aGpioCtrl, 0) then
    Exit;

  if DefaultCard then
  begin
    // This handles GPIO1
    if not Saa7146aSelectCommonInterface(Handle, False) then
      Exit;
    // This handles GPIO2 (reset off)
    if not Saa7146aSetGpio(Handle, 2, True) then
      Exit;
  end;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <GPIO>    Identifier of GPIO (0..3)
            <OnOff>   GPIO to set (True) or reset (False)
  Returns : <Result>  True if success

  Descript: Set GPIO (de)active
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aSetGpio(Handle: THandle; GPIO: Byte; OnOff: Boolean): Boolean;
var
  Mask: Dword;
  Level: Dword;
begin
  Result := False;

  if OnOff then
    Level := CSaa7146aGpioOutputHigh
  else
    Level := CSaa7146aGpioOutputLow;

  case GPIO of
    0: Mask := CSaa7146aGpio0Mask;
    1: Mask := CSaa7146aGpio1Mask;
    2: Mask := CSaa7146aGpio2Mask;
    3: Mask := CSaa7146aGpio3Mask;
  else
    Exit;
  end;

  if not Saa7146aWriteWithMaskToSaa7146aRegister(Handle,
    CSaa7146aGpioCtrl,
    Mask,
    Level) then
    Exit;

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Gpio>    GPIO to issue reset on (eg. 2 for Nova/Budget cards)
  Returns : <Result>  True if success

  Descript: Issue a RESET signal (low - high toggle)
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aIssueReset(Handle: THandle; Gpio: Byte): Boolean;
begin
  Result := False;

  // Activate reset
  if not Saa7146aSetGpio(Handle, Gpio, False) then
    // Small delay
    Sleep(10);
  // Deactivate reset
  if not Saa7146aSetGpio(Handle, Gpio, True) then
    Exit;
  Sleep(50);

  Result := True;
end;

{------------------------------------------------------------------------------
  Params  : <Handle>  <Saa7146aCreateFile> handle
            <Select>  True  for data from common interface
                      False for data from tuner
  Returns : <Result>  True if success

  Descript: Set data (DD1 interface) from tuner or common interface
  Notes   :
------------------------------------------------------------------------------}

function Saa7146aSelectCommonInterface(Handle: THandle; Select: Boolean):
  Boolean;
var
  State: Dword;
begin
  Result := False;

  if Select then
    State := CSaa7146aGpioOutputLow
  else
    State := CSaa7146aGpioOutputHigh;
  if not Saa7146aWriteWithMaskToSaa7146aRegister(Handle,
    CSaa7146aGpioCtrl,
    CSaa7146aGpio1Mask,
    State) then
    Exit;

  Result := True;
end;

end.

