{******************************************************************************}
{ FileName............: DvbDirectShow                                          }
{ Project.............: DVB-SB                                                 }
{ Author(s)...........: MM                                                     }
{ Version.............: 1.15                                                   }
{------------------------------------------------------------------------------}
{  DirectShow interface                                                        }
{                                                                              }
{  Copyright (C) 2003-2005  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Graph is also added to the Running Object Table so it can be connected to    }
{ when running (typically for debugging purposes)                              }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  1.00   20031113 - Initial release                                           }
{  1.01   20040213 - Added 'NoRender' parameter                                }
{  1.02   20040218 - Cleanup: might have introduced spurious runtime error at  }
{                    application ending                                        }
{                  - External access to graph (AddToRot) made variable         }
{                  - LogFile handle reset prior to destruction                 }
{  1.05   20040320 - Added <MajorDvbPsi> DirectShow filter support             }
{                    (V2.01/V2.02)                                             }
{                  - All <DvbFilter> calls are through this unit               }
{                  - Added windowless mode                                     }
{                  - If no .GRF file given the graph is created manually       }
{  1.06   20040605 - Additional Try/Except added                               }
{  1.07   20040815 - ServiceId in <..GetProgramInfo> added                     }
{                  - Added <DirectShowResize>                                  }
{  1.08   20040828 - Added CAT callback                                        }
{                  - Added PAT callback                                        }
{                  - Removed support for PsiFilter (too much differences)      }
{  1.09   20040920 - VMR9 instead of windowless mode plus some additional VRM  }
{                    stuff (eg. 'OSD')                                         }
{  1.10   20041003 - Retrieval if used filters added                           }
{                  - Added Property page detection/showing                     }
{  1.11   20041021 - Switchable method (for debugging mainly)                  }
{                    Complete rebuild                                          }
{  1.12   20041212 - More object based so multiple instances are possible      }
{                  - Auto registration of usrc.ax                              }
{  1.13   20041225 - Added AC3 pin on MPEG2 demultiplexer plus options         }
{  1.14   20050808 - Added GetCurrentImage support for getting a picture dump  }
{                    IBasicVideo used instead of IBasicVideo2                  }
{                  - E_FALSE replaced by E_FAIL so it is recognized as error   }
{  1.15   20050815 - Function added SDT callback                               }
{******************************************************************************}
unit DvbDirectShow;

interface
uses
  Classes,
  Controls,
  DirectShow9,
  //  DxErr9,
  DvbFilter,
  ExtCtrls,
  Graphics,
  SysUtils,
  Windows;

const
  // Note that not all combinations are possible (eg. VRM with CaptureGraph)
  // Miscellaneous
  CDirectShowMethodSpecials = $FF000000; // Special options
  CDirectShowMethodDoNotRender = $80000000; // Do not render graph
  CDirectShowMethodDoNotAddToRot = $40000000; // Do not add to ROT
  CDirectShowMethodRenderAllPins = $20000000; // Render all pins
  CDirectShowMethodRenderAllAudioVideo = $10000000;
    // Render all audio/video pins
  CDirectShowMethodNativeVideoSize = $01000000; // Keep native video size
  CDirectShowMethodLogToFile = $02000000; // Use log file
  CDirectShowMethodRenderEx = $04000000; // Use RenderEx first
  CDirectShowMethodAlternative = $08000000;
  CDirectShowMethodNoAC3Pin = $00100000; // Do not create AC3 specific pin
    // Use alternative (= older) DirectShow methods
  // VMR
  CDirectShowMethodVmr = $0000F000; // VMR options
  CDirectShowMethodVideoRenderer = $00000000; // Do not use VMR
  CDirectShowMethodVideoRenderer7 = $00007000; // Use VMR7
  CDirectShowMethodVideoRenderer9 = $00009000; // Use VMR9
  // VMR specials
  CDirectShowMethodVmrSpecials = $00000F00; // VMR special options
  CDirectShowMethodVideoRendererWindowless = $00000100; // Windowless mode
  CDirectShowMethodVideoRendererFlipX = $00000200; // Swap/Flip/Mirror X
  CDirectShowMethodVideoRendererFlipY = $00000400; // Swap/Flip/Mirror Y

  // Graph (builder)
  CDirectShowMethodGraph = $000000F0; // Graph
  CDirectShowMethodFilterGraph2 = $00000000; // Use FilterGraph2
  CDirectShowMethodGraphBuilder = $00000010; // Use GraphBuilder
  CDirectShowMethodFilterGraph = $00000020; // Use FilterGraph
  CDirectShowMethodDvdGraphBuilder = $00000030; // Use DvdGraphBuilder
  // Additional builders
  CDirectShowMethodBuilder = $0000000F; // Builder
  CDirectShowMethodNoBuilders = $00000000; // No additional builders
  CDirectShowMethodCaptureGraphBuilder = $00000001; // Use CaptureGraphBuilder
  CDirectShowMethodCaptureGraphBuilder2 = $00000002; // Use CaptureGraphBuilder2


  {------------------------------------------------------------------------------
    Descript: Helper functions
   ------------------------------------------------------------------------------}
function GetDibHeaderSize(Dib: PBitmapInfo): Integer;
function GetDibDataSize(Dib: PBitmapInfo): Integer;
function PackedDibToDibSection(Dib: PBitmapInfo): HBITMAP;

  {------------------------------------------------------------------------------
    Descript: DvbFilter functions
    Note    : These functions are passed onto <DvbFilter>
   ------------------------------------------------------------------------------}
function DvbGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
function DvbSetSignallingCallback     (Pid: Word; hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackPat  (hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackCat  (hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackPmt  (hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackSdt  (hWnd: HWND; Msg: UINT): Boolean;
function DvbSetSignallingCallbackNit  (hWnd: HWND; Msg: UINT): Boolean;
function DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
function DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
function DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
function DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
function DvbGetProgramInfo(ProgramNumber: Byte;
  var ServiceId: Word;
  var PmtPid: Word;
  var PcrPid: Word;
  var VideoPids: TDvbPids;
  var AudioPids: TDvbPids;
  var TeletextPids: TDvbPids;
  var SubtitlePids: TDvbPids;
  var AudioLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids: TDvbPids;
  var CaIds: TDvbPids;
  var ProgramName: string): Boolean;
function DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean;
  Events: TDvbEventSections): Boolean;
function DvbGetNumberOfPrograms: Byte;
function DvbGetErrors: Word;
function DvbGetPacketSyncErrors: Word;
procedure DvbResetErrors;
procedure DvbCreateTables;

type
  {------------------------------------------------------------------------------
    Descript: The DirectShow object
   ------------------------------------------------------------------------------}
  TDirectShowGraph = class(TObject)
    FOwner: HWND; // Owner of 'video'
    FLogFile: THandle; // Log file
    FOptions: Dword; // Options
    FFilterGraphManager: IUnknown;
      // Filter graph manager (IFilterGraph, IGraphBuilder, IFilterGraph2)
    FGraphBuilder: IUnknown;
      // Additional builder   (ICaptureGraphBuilder, ICaptureGraphBuilder2, IDvdGraphBuilder)
    FVmrFilter: IBaseFilter; // VMR filter
    FVmrFilterConfig7: IVMRFilterConfig;
    FVmrFilterConfig9: IVMRFilterConfig9;
    FVmrMixerControl7: IVMRMixerControl;
    FVmrMixerControl9: IVMRMixerControl9;
    FAlphaBitmap7: VMRAlphaBitmap;
    FAlphaBitmap9: VMR9AlphaBitmap;
    FVideoWindow: IVideoWindow; // Video window (non VMR mode)
    FBasicVideo: IBasicVideo; // Used for retrieval of (required) window size
    FWindowlessControl: IUnknown;
    FMPEG2Demux: IBaseFilter;
    FMPEG2Demultiplexer: IMPEG2Demultiplexer;
    FUniversalSource: IBaseFilter;
    FVideoPin: IPin;
    FAudioPin: IPin;
    FAudioAc3Pin: IPin;
    FActiveAudioPid: Integer;
    FActiveVideoPid: Integer;
    FActiveAudioAc3: Boolean;
    FRotEntry: LongInt;
    constructor Create(Owner: HWND);
    destructor Destroy; override;
  private
    { Private declarations }
    function BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend: Single;
      ColorKey: COLORREF): Boolean;
    function DirectShowCheck(ResultCode: HRESULT; var ResultMessage: string):
      Boolean;
    function AddToRot(UnknownGraph: IUnknown; var RotEntry: LongInt): HRESULT;
    procedure RemoveFromRot(RotEntry: LongInt);
    function LoadGraphFile(FileName: string): HRESULT;
    procedure DirectShowAssign(Owner: HWND);
  public
    { Public declarations }
    function DirectShowStart(Owner: HWND;
      GraphicsFile: string;
      Options: Dword; // Bitwise options
      var ErrorMessage: string): Boolean;
    procedure DirectShowStop;
    procedure DirectShowFullScreenMode(FullScreen: Boolean);
    procedure DirectShowSetVideo(Video: HWND);
    procedure DirectShowResize;
    procedure DirectShowGetSize(var Width: Integer; var Height: Integer);
    function  DirectShowGetImage: HBITMAP;
    function DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend: Single;
      ColorKey: COLORREF): Boolean;
    function DirectShowGetFilterName(Index: Integer): string;
    function DirectShowWindowless: Boolean;
    function DirectShowShowPropertyPage(Parent: THandle; FilterName: WideString;
      QueryOnly: Boolean): Boolean;
    procedure DirectShowMpeg2DemultiplexerSetNewPids(VideoPid: Integer;
      AudioPid: Integer; AudioAc3: Boolean);
    procedure DirectShowUniversalSourceSendData(Buffer: PChar; BufferLength:
      Integer);
  end;

var
  DirectShowGraph: TDirectShowGraph; // The global created DirectShow object

implementation
uses
  ActiveX,
  Dialogs,
  Forms,
  ShellApi;

const
  {------------------------------------------------------------------------------
    Descript: DirectShow identifiers
   ------------------------------------------------------------------------------}
  IID_IUSRCSet = '{AE1A2884-540E-4077-B1AB-67A34A72299F}';
  T_IID_IUSRCSet: TGUID = IID_IUSRCSet;
  CLSID_IUSRCSet: TGUID = '{FD501041-8888-1111-9153-00AB00577DA2}';
  IID_IPersistStream: TGUID = '{00000109-0000-0000-C000-000000000046}';

  CLSID_MPEG2Demultiplexer: TGUID = (D1: $AFB6C280; D2: $2C41; D3: $11D3; D4:
    ($8A, $60, $00, $00, $F8, $1E, $0E, $4A));

  IID_ISpecifyPropertyPages: TGUID = '{B196B28B-BAB4-101A-B69C-00AA00341D07}';
    // See DsUtil

type
  {------------------------------------------------------------------------------
    Descript: Universal Source filter interface
   ------------------------------------------------------------------------------}
  IUSRCSet = interface(IUnknown)
    [IID_IUSRCSet]
    function WSetMediaType(Pmt: PAMMEDIATYPE): HRESULT; stdcall;
    function WSetBufSize(Size: Dword): HRESULT; stdcall;
    function WSendSample(Buffer: PChar; BufferLength: Dword): HRESULT; stdcall;
  end;

  {------------------------------------------------------------------------------
    Params  : <ResultCode>     Result code to check
    Returns : <Result>         True if no error
              <ResultMessage>  Error message

    Descript: Check result code and generate an error message if in error.
    Notes   :
   ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowCheck(ResultCode: HRESULT; var
  ResultMessage: string): Boolean;
var
  //  RString     : PChar;
  //  RDescription: PChar;
  RError: string;
  Size: Dword;
begin
  Result := Succeeded(ResultCode);
  ResultMessage := '';
  if Failed(ResultCode) then
  begin
    SetLength(RError, 128);
    Size := AmGetErrorText(ResultCode, @RError[1], 128);
    if Size <> 0 then
    begin
      SetLength(RError, Size);
      ResultMessage := RError;
    end;
    // Next need DLL which is not always present ....
//    RString      := DxGetErrorString9(Code);
//    RDescription := DxGetErrorDescription9(Code);
//    ShowMessage('DirectShow error:'#13#13 +
//                 RString + #13 +
//                 RDescription);
  end;
end;

{------------------------------------------------------------------------------
  Params  : <UnknownGraph>  Graph to add to ROT
  Returns : <Result>        Result (S_OK if success)
            <RotEntry>      Assigned number in ROT

  Descript: Add graph to Running Object Table so we can connect to it when
            it is running.
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.AddToRot(UnknownGraph: IUnknown; var RotEntry:
  LongInt): HRESULT;
var
  Moniker: IMoniker;
  Rot: IRunningObjectTable;
  Wsz: WideString;
  Wsz2: WideString;
  Hr: HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, ROT);
    if Failed(Hr) then
    begin
      Result := E_FAIL;
      Exit;
    end;
    Wsz := format('FilterGraph %p pid %8.8x', [Pointer(UnknownGraph),
      GetCurrentProcessId]);
    Wsz2 := '!';
    Hr := CreateItemMoniker(@Wsz2[1], @Wsz[1], Moniker);
    if Succeeded(Hr) then
    begin
      Hr := ROT.Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, UnknownGraph,
        Moniker, RotEntry);
    end;
    Result := Hr;
  except
    Result := E_FAIL;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <RotEntry>      Assigned number in ROT
  Returns : -

  Descript: Remove graph from Running Object Table
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.RemoveFromRot(RotEntry: LongInt);
var
  Rot: IRunningObjectTable;
  Hr: HRESULT;
begin
  try
    Hr := GetRunningObjectTable(0, Rot);
    if Succeeded(Hr) then
      Rot.Revoke(RotEntry);
  except
  end;
end;

{------------------------------------------------------------------------------
  Params  : <FileName>  Name of graph file
  Returns : -

  Descript: Load a DirectShow graph (.GRF) file
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.LoadGraphFile(FileName: string): HRESULT;
var
  PStorage: IStorage;
  PPersistStream: IPersistStream;
  PStream: IStream;
  WideName: array[0..1023] of WideChar;
  Hr: HRESULT;
begin
  try
    PStorage := nil;
    PPersistStream := nil;
    PStream := nil;
    StringToWideChar(FileName, WideName, SizeOf(WideName));
    hr := StgIsStorageFile(@WideName);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    Hr := StgOpenStorage(@WideName, nil, STGM_TRANSACTED or STGM_READ or
      STGM_SHARE_DENY_WRITE,
      nil, 0, PStorage);
    if Failed(Hr) then
    begin
      Result := Hr;
      Exit;
    end;
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
        IFilterGraph2).QueryInterface(IID_IPersistStream, PPersistStream);
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
          IGraphBuilder).QueryInterface(IID_IPersistStream, PPersistStream);
      CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
        IFilterGraph).QueryInterface(IID_IPersistStream, PPersistStream);
    end;
    if Succeeded(Hr) then
    begin
      Hr := PStorage.OpenStream('ActiveMovieGraph', nil,
        STGM_READ or STGM_SHARE_EXCLUSIVE, 0, PStream);
      if Succeeded(Hr) then
      begin
        Hr := PPersistStream.Load(PStream);
        PStream := nil;
      end;
      PPersistStream := nil;
    end;
    PStorage := nil;
    Result := Hr;
  except
    Result := E_FAIL;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Owner>           Handle to window for video
            <GraphicsFile>    .GRF file to use
                              If empty the graph is manually created
            <Options>         Options to use (see constants)
  Returns : <Result>          TRUE if success, FALSE if failure
            <ErrorMessage>    Error message (if any)

  Descript: Setup direct show graph
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowStart(Owner: HWND;
  GraphicsFile: string;
  Options: Dword;
  var ErrorMessage: string): Boolean;
var
  Hr: HRESULT;
  Pins: IPin;
  PinsIn: IPin;
  PinsOut: IPin;
  PinsEnum: IEnumPins;
  PinsInfo: TPinInfo;
  //  PinDirection                : TPinDirection;
  MediaType: AM_MEDIA_TYPE;
  MediaControl: IMediaControl;
  EnumFilters: IEnumFilters;
  BaseFilter: IBaseFilter;
  AllPins: array[0..99] of IPin;
  AllPinsCount: Integer;
  Index1: Integer;
  EnumMediaTypes: IEnumMediaTypes;
  PMediaType: PAMMediaType;
  APin: IPin;
  DestinationRect: TRect;
  FlipRect7: TNormalizedRect;
  FlipRect9: TVMR9NormalizedRect;
  FilterState: TFilterState;
  ResultMessage: string;
  {------------------------------------------------------------------------------
    Params  : <Name>    Name to find
    Returns : <Result>
              <Filter>  Filter found

    Descript: Find filter by name
    Notes   :
   ------------------------------------------------------------------------------}
  function FindFilterByName(Name: WideString; var Filter: IBaseFilter): HRESULT;
  begin
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Result := (FFilterGraphManager as
        IFilterGraph2).FindFilterByName(PWideChar(Name), Filter);
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Result := (FFilterGraphManager as
          IGraphBuilder).FindFilterByName(PWideChar(Name), Filter);
      CDirectShowMethodFilterGraph: Result := (FFilterGraphManager as
        IFilterGraph).FindFilterByName(PWideChar(Name), Filter);
    else
      Result := E_NOTIMPL;
    end;
  end;

  {------------------------------------------------------------------------------
    Params  : <FilterName>      Filter which has the pin
              <PinName>         Name of the pin to render
    Returns : <Result>          TRUE if success, FALSE if failure
              <ResultMessage>   Error message

    Descript: Render a pin
    Notes   :
   ------------------------------------------------------------------------------}
  function RenderPin(FilterName: WideString; PinName: WideString; var
    ResultMessage: string): Boolean;
  var
    Hr: HRESULT;
    Filter: IBaseFilter;
    EnumPins: IEnumPins;
    Pin: IPin;
    pInfo: TPinInfo;
  begin
    Result := False;
    try
      ResultMessage := '';
      Hr := FindFilterByName(FilterName, Filter);
      if Succeeded(Hr) then
      begin
        Filter.EnumPins(EnumPins);
        // Go through all pins until we find it
        EnumPins.Reset;
        while Succeeded(EnumPins.Next(1, Pin, nil)) do
        begin
          Pin.QueryPinInfo(pInfo);
          pInfo.pFilter := nil;
          if (pInfo.achName = PinName) then
          begin
            // Only render if not connected
            case (FOptions and CDirectShowMethodGraph) of
              CDirectShowMethodFilterGraph2:
                begin
                  if (FOptions and CDirectShowMethodRenderEx) <> 0 then
                  begin
                    Hr := (FFilterGraphManager as IFilterGraph2).RenderEx(Pin,
                      AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                    if Failed(Hr) then
                      Hr := (FFilterGraphManager as IFilterGraph2).Render(Pin);
                  end
                  else
                    Hr := (FFilterGraphManager as IFilterGraph2).Render(Pin);
                end;
              CDirectShowMethodGraphBuilder,
                CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
                  IGraphBuilder).Render(Pin);
              CDirectShowMethodFilterGraph: Hr := E_NOTIMPL;
            else
              Hr := E_NOTIMPL;
            end;
            // Failing of rendering can be normal since it might be already connected ...
            // Reconnecting it should succeed though
            if Failed(Hr) then
            begin
              case (FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
                  IFilterGraph2).ReConnectEx(Pin, nil);
                CDirectShowMethodGraphBuilder,
                  CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
                    IGraphBuilder).ReConnect(Pin);
                CDirectShowMethodFilterGraph: Hr := E_NOTIMPL;
              else
                Hr := E_NOTIMPL;
              end;
            end;
            if not DirectShowCheck(Hr, ResultMessage) then
              Result := False
            else
              Result := True;
            Exit;
          end;
        end;
      end
      else
        DirectShowCheck(Hr, ResultMessage);
    except
    end;    
  end;

  {------------------------------------------------------------------------------
    Params  : <Filter>   Base filter
              <Name>     Name of filter to add
    Returns : <Result>

    Descript: Add filter by name
    Notes   :
   ------------------------------------------------------------------------------}
  function AddFilter(Filter: IBaseFilter; Name: WideString): HRESULT;
  begin
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Result := (FFilterGraphManager as
        IFilterGraph2).AddFilter(Filter, PWideChar(Name));
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Result := (FFilterGraphManager as
          IGraphBuilder).AddFilter(Filter, PWideChar(Name));
      CDirectShowMethodFilterGraph: Result := (FFilterGraphManager as
        IFilterGraph).AddFilter(Filter, PWideChar(Name));
    else
      Result := E_NOTIMPL;
    end;
  end;

  {------------------------------------------------------------------------------
    Params  : <PinOut>   Output pin
              <PinIn>    Input pin
    Returns : <Result>

    Descript: Connect pins
    Notes   :
   ------------------------------------------------------------------------------}
  function Connect(PinOut: IPin; PinIn: IPin): HRESULT;
  begin
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Result := (FFilterGraphManager as
        IFilterGraph2).Connect(PinOut, PinIn);
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Result := (FFilterGraphManager as
          IGraphBuilder).Connect(PinOut, PinIn);
      CDirectShowMethodFilterGraph: Result := (FFilterGraphManager as
        IFilterGraph).ConnectDirect(PinOut, PinIn, nil);
    else
      Result := E_NOTIMPL;
    end;
  end;

  {------------------------------------------------------------------------------
    Params  : <PinOut>   Output pin
              <PinIn>    Input pin
    Returns : <Result>

    Descript: Connect pins
    Notes   :
   ------------------------------------------------------------------------------}
  function SetDefaultSyncSource: HRESULT;
  begin
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Result := (FFilterGraphManager as
        IFilterGraph2).SetDefaultSyncSource;
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Result := (FFilterGraphManager as
          IGraphBuilder).SetDefaultSyncSource;
      CDirectShowMethodFilterGraph: Result := (FFilterGraphManager as
        IFilterGraph).SetDefaultSyncSource;
    else
      Result := E_NOTIMPL;
    end;
  end;

begin
  try
    // Make sure a previously started DirectShow interface is stopped and released
    DirectShowStop;
    ErrorMessage := '';
    Result := False;
    DirectShowAssign(Owner);
    try
      FOptions := Options;
      if (FOptions and CDirectShowMethodLogToFile) <> 0 then
      begin
        FLogFile := CreateFile('DirectShow.log' + #0,
          GENERIC_WRITE,
          0,
          nil,
          CREATE_ALWAYS,
          FILE_ATTRIBUTE_NORMAL,
          0);
      end;

      {------------------------------------------------------------------------------
        Descript: Create the filter manager
       ------------------------------------------------------------------------------}
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: Hr := CoCreateInstance(CLSID_FilterGraph,
          nil, CLSCTX_INPROC_SERVER, IID_IFilterGraph2, FFilterGraphManager);
        CDirectShowMethodGraphBuilder: Hr := CoCreateInstance(CLSID_FilterGraph,
          nil, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, FFilterGraphManager);
        CDirectShowMethodFilterGraph: Hr := CoCreateInstance(CLSID_FilterGraph,
          nil, CLSCTX_INPROC_SERVER, IID_IFilterGraph, FFilterGraphManager);
        CDirectShowMethodDvdGraphBuilder:
          begin
            // The DVD builder creates the filter manager
            Hr := CoCreateInstance(CLSID_DvdGraphBuilder, nil,
              CLSCTX_INPROC_SERVER, IID_IDvdGraphBuilder, FGraphBuilder);
            if Succeeded(Hr) then
              Hr := (FGraphBuilder as
                IDvdGraphBuilder).GetFilterGraph(IGraphBuilder(FFilterGraphManager));
          end;
      else
        Hr := E_NOTIMPL;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage :=
          format('CoCreateInstance filter graph manager error $%8.8x.', [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      // Log file
      if (FOptions and CDirectShowMethodLogToFile) <> 0 then
        if FLogFile <> INVALID_HANDLE_VALUE then
          case (FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2: (FFilterGraphManager as
              IFilterGraph2).SetLogFile(FLogFile);
            CDirectShowMethodGraphBuilder,
              CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
                IGraphBuilder).SetLogFile(FLogFile);
          end;

      {------------------------------------------------------------------------------
        Descript: Additional builders
       ------------------------------------------------------------------------------}
      case (FOptions and CDirectShowMethodBuilder) of
        CDirectShowMethodCaptureGraphBuilder:
          begin
            Hr := CoCreateInstance(CLSID_CaptureGraphBuilder, nil,
              CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder, FGraphBuilder);
            if Succeeded(Hr) then
              case (FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2: Hr := (FGraphBuilder as
                  ICaptureGraphBuilder).SetFilterGraph(IFilterGraph2(FFilterGraphManager));
                CDirectShowMethodGraphBuilder,
                  CDirectShowMethodDvdGraphBuilder: Hr := (FGraphBuilder as
                    ICaptureGraphBuilder).SetFilterGraph(IGraphBuilder(FFilterGraphManager));
                CDirectShowMethodFilterGraph: Hr := E_NOTIMPL;
              end;
          end;
        CDirectShowMethodCaptureGraphBuilder2:
          begin
            Hr := CoCreateInstance(CLSID_CaptureGraphBuilder2, nil,
              CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder2, FGraphBuilder);
            if Succeeded(Hr) then
              case (FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2: Hr := (FGraphBuilder as
                  ICaptureGraphBuilder2).SetFilterGraph(IFilterGraph2(FFilterGraphManager));
                CDirectShowMethodGraphBuilder,
                  CDirectShowMethodDvdGraphBuilder: Hr := (FGraphBuilder as
                    ICaptureGraphBuilder2).SetFilterGraph(IGraphBuilder(FFilterGraphManager));
                CDirectShowMethodFilterGraph: Hr := E_NOTIMPL;
              end;
          end;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := format('CoCreateInstance graph builder error $%8.8x.',
          [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: ROT
       ------------------------------------------------------------------------------}
      if (FOptions and CDirectShowMethodDoNotAddToRot) = 0 then
      begin
        Hr := AddToRot(FFilterGraphManager, FRotEntry);
        if Failed(Hr) then
        begin
          FRotEntry := -1;
        end;
      end;

      {------------------------------------------------------------------------------
        Descript: Load filter graph
       ------------------------------------------------------------------------------}
      if FileExists(GraphicsFile) then
      begin
        Hr := LoadGraphFile(GraphicsFile);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Loading graph file "%s" error $%8.8x.',
            [GraphicsFile, Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;

      {------------------------------------------------------------------------------
        Descript: VMR
       ------------------------------------------------------------------------------}
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer: Hr := NOERROR;
        CDirectShowMethodVideoRenderer7:
          begin
            // Try locating if VMR already present
            Hr := FindFilterByName('Video Mixing Renderer 7', FVmrFilter);
            if Failed(Hr) then
            begin
              Hr := FindFilterByName('VMR7', FVmrFilter);
              if Failed(Hr) then
              begin
                // Not present, create it
                Hr := CoCreateInstance(CLSID_VideoMixingRenderer, nil,
                  CLSCTX_INPROC_SERVER, IID_IBaseFilter, FVmrFilter);
                if Succeeded(Hr) then
                begin
                  Hr := AddFilter(FVmrFilter, 'Video Mixing Renderer 7');
                  if Failed(Hr) then
                    Hr := AddFilter(FVmrFilter, 'VMR7');
                end;
              end;
            end;
          end;
        CDirectShowMethodVideoRenderer9:
          begin
            // Try locating if VMR already present (Note: will ALWAYS fail if nothing loaded)
            Hr := FindFilterByName('Video Mixing Renderer 9', FVmrFilter);
            if Failed(Hr) then
            begin
              Hr := FindFilterByName('VMR9', FVmrFilter);
              if Failed(Hr) then
              begin
                // Not present, create it
                Hr := CoCreateInstance(CLSID_VideoMixingRenderer9, nil,
                  CLSCTX_INPROC_SERVER, IID_IBaseFilter, FVmrFilter);
                if Succeeded(Hr) then
                begin
                  Hr := AddFilter(FVmrFilter, 'Video Mixing Renderer 9');
                  if Failed(Hr) then
                    Hr := AddFilter(FVmrFilter, 'VMR9');
                end;
              end;
            end;
          end;
      else
        Hr := E_NOTIMPL;
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage :=
          format('CoCreateInstance video mixing renderer error $%8.8x.', [Hr]);
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: Set VMR mode
       ------------------------------------------------------------------------------}
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer: ;
        CDirectShowMethodVideoRenderer7:
          begin
            Hr := FVmrFilter.QueryInterface(IID_IVMRFilterConfig,
              FVmrFilterConfig7);
            if not DirectShowCheck(Hr, ResultMessage) then
            begin
              ErrorMessage :=
                format('Could not get interface to filter configuration, error $%8.8x.', [Hr]);
              ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
              Exit;
            end;
            // We need to set the number of streams to be able to use the mixer and this must before setting window mode
            FVmrFilterConfig7.SetNumberOfStreams(16);
            // Flip
            Hr := FVmrFilterConfig7.QueryInterface(IID_IVMRMixerControl,
              FVmrMixerControl7);
            if Succeeded(Hr) then
            begin
              FlipRect7.Left := 0;
              FlipRect7.Top := 0;
              FlipRect7.Right := 1;
              FlipRect7.Bottom := 1;
              // Note: Using -1 instead of swapping does not work correctly
              if (FOptions and CDirectShowMethodVideoRendererFlipX) <> 0 then
              begin
                FlipRect7.Left := 1;
                FlipRect7.Right := 0;
              end;
              if (FOptions and CDirectShowMethodVideoRendererFlipY) <> 0 then
              begin
                FlipRect7.Top := 1;
                FlipRect7.Bottom := 0;
              end;
              FVmrMixerControl7.SetOutputRect(0, FlipRect7);
            end;
            // Window(less) mode
            if (FOptions and CDirectShowMethodVmrSpecials) =
              CDirectShowMethodVideoRendererWindowless then
            begin
              FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowless);
              Hr := FVmrFilter.QueryInterface(IID_IVMRWindowlessControl,
                FWindowlessControl);
              if Succeeded(Hr) then
                IVMRWindowlessControl(FWindowlessControl).SetVideoClippingWindow(FOwner)
              else
                FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowed);
            end
            else
              FVmrFilterConfig7.SetRenderingMode(VMRMode_Windowed);
          end;
        CDirectShowMethodVideoRenderer9:
          begin
            Hr := FVmrFilter.QueryInterface(IID_IVMRFilterConfig9,
              FVmrFilterConfig9);
            if not DirectShowCheck(Hr, ResultMessage) then
            begin
              ErrorMessage :=
                format('Could not get interface to filter configuration, error $%8.8x.', [Hr]);
              ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
              Exit;
            end;
            // We need to set the number of streams to be able to use the mixer and this must before setting window mode
            FVmrFilterConfig9.SetNumberOfStreams(16);
            // Flip
            Hr := FVmrFilterConfig9.QueryInterface(IID_IVMRMixerControl9,
              FVmrMixerControl9);
            if Succeeded(Hr) then
            begin
              FlipRect9.Left := 0;
              FlipRect9.Top := 0;
              FlipRect9.Right := 1;
              FlipRect9.Bottom := 1;
              // Note: Using -1 instead of swapping does not work correctly
              if (FOptions and CDirectShowMethodVideoRendererFlipX) <> 0 then
              begin
                FlipRect9.Left := 1;
                FlipRect9.Right := 0;
              end;
              if (FOptions and CDirectShowMethodVideoRendererFlipY) <> 0 then
              begin
                FlipRect9.Top := 1;
                FlipRect9.Bottom := 0;
              end;
              FVmrMixerControl9.SetOutputRect(0, @FlipRect9);
            end;
            // Window(less) mode
            if (FOptions and CDirectShowMethodVmrSpecials) =
              CDirectShowMethodVideoRendererWindowless then
            begin
              FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowless);
              Hr := FVmrFilter.QueryInterface(IID_IVMRWindowlessControl9,
                FWindowlessControl);
              if Succeeded(Hr) then
                IVMRWindowlessControl9(FWindowlessControl).SetVideoClippingWindow(FOwner)
              else
                FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowed);
            end
            else
              FVmrFilterConfig9.SetRenderingMode(VMRMode_Windowed);
          end;
      end;

      {------------------------------------------------------------------------------
        Descript: Building the graph
       ------------------------------------------------------------------------------}
      if not FileExists(GraphicsFile) then
      begin
        {------------------------------------------------------------------------------
          Descript: Building a graph manually
         ------------------------------------------------------------------------------}
        // No graph filter file: create the graph manually (at least the major parts)
        // First the universal source
        Hr := CoCreateInstance(CLSID_IUSRCSet, nil, CLSCTX_INPROC_SERVER,
          IID_IBaseFilter, FUniversalSource);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          // Try register it (it might not have registered)
          ShellExecute(Application.Handle, nil, 'regsvr32', '/s usrc.ax', nil, SW_SHOW);
          // Make sure register is done
          Sleep(100);
          // Retry
          Hr := CoCreateInstance(CLSID_IUSRCSet, nil, CLSCTX_INPROC_SERVER,
            IID_IBaseFilter, FUniversalSource);
          if not DirectShowCheck(Hr, ResultMessage) then
          begin
            ErrorMessage := 'Could not use the Universal Source filter.';
            ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
            Exit;
          end;  
        end;
        Hr := AddFilter(FUniversalSource, 'Universal Source');
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := format('Adding Universal Source filter error $%8.8x.',
            [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Then the MPEG-2 demultiplexer
        Hr := CoCreateInstance(CLSID_MPEG2Demultiplexer, nil,
          CLSCTX_INPROC_SERVER, IID_IBaseFilter, FMPEG2Demux);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not use the MPEG-2 Demultiplexer filter.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := AddFilter(FMPEG2Demux, 'MPEG-2 Demultiplexer');
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage :=
            format('Adding MPEG-2 Demultiplexer filter error $%8.8x.', [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Now we must create the output pins on the MPEG-2 demultiplexer
        // The video pin
        Hr := FMPEG2Demux.QueryInterface(IID_IMpeg2Demultiplexer,
          FMPEG2Demultiplexer);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter MPEG-2 Demultiplexer interface not found.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        MediaType.MajorType := MEDIATYPE_Video;
        MediaType.SubType := MEDIASUBTYPE_MPEG2_VIDEO;
        Hr := FMPEG2Demultiplexer.CreateOutputPin(MediaType, 'Video',
          FVideoPin);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage :=
            'Could not create the video output pin for the MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // The audio pin
        MediaType.MajorType := MEDIATYPE_Audio;
        MediaType.SubType := MEDIASUBTYPE_MPEG2_AUDIO;
        Hr := FMPEG2Demultiplexer.CreateOutputPin(MediaType, 'Audio',
          FAudioPin);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage :=
            'Could not create the audio output pin for the MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // The audio pin (AC3)
        if (FOptions and CDirectShowMethodNoAC3Pin) = 0 then
        begin
          MediaType.MajorType := MEDIATYPE_Audio;
          MediaType.SubType := MEDIASUBTYPE_DOLBY_AC3;
          Hr := FMPEG2Demultiplexer.CreateOutputPin(MediaType, 'AudioAC3',
            FAudioAc3Pin);
          if not DirectShowCheck(Hr, ResultMessage) then
          begin
            ErrorMessage :=
              'Could not create the AC3 audio output pin for the MPEG-2 Demultiplexer.';
            ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
            Exit;
          end;
        end;  
        PinsIn := nil;
        PinsOut := nil;

        // Enumerate the pins
        Hr := FUniversalSource.EnumPins(PinsEnum);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not enumerate the Universal Source pins.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        PinsEnum.Reset;
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          PinsInfo.pFilter := nil;
          if UpperCase(PinsInfo.achName) = 'OUTPUT' then
            PinsOut := Pins;
        end;
        Pins := nil;
        PinsEnum := nil;

        Hr := FMPEG2Demux.EnumPins(PinsEnum);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not enumerate the MPEG-2 Demultiplexer pins.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // Go through the pins and record the ones we are interested in
        PinsEnum.Reset;
        while PinsEnum.Next(1, Pins, nil) = S_OK do
        begin
          Pins.QueryPinInfo(PinsInfo);
          PinsInfo.pFilter := nil;
          if UpperCase(PinsInfo.achName) = 'MPEG-2 STREAM' then
            PinsIn := Pins;
        end;
        Pins := nil;
        PinsEnum := nil;

        // Connect the pins
        if not Assigned(PinsIn) or not Assigned(PinsOut) then
        begin
          PinsIn := nil;
          PinsOut := nil;
          ErrorMessage :=
            'Could not find correct pins Universal Source/MPEG-2 Demultiplexer.';
          Exit;
        end;
        Hr := Connect(PinsOut, PinsIn);
        PinsIn := nil;
        PinsOut := nil;
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage :=
            'Could not connect pins Universal Source/MPEG-2 Demultiplexer.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        {------------------------------------------------------------------------------
          Descript: Building a graph manually end
         ------------------------------------------------------------------------------}
      end;

      {------------------------------------------------------------------------------
        Descript: Check for required components
       ------------------------------------------------------------------------------}
      if not Assigned(FMPEG2Demux) then
      begin
        Hr := FindFilterByName('MPEG-2 Demultiplexer', FMPEG2Demux);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer filter not found.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // We need the interface as well for some functions
        Hr := FMPEG2Demux.QueryInterface(IID_IMpeg2Demultiplexer,
          FMPEG2Demultiplexer);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter MPEG-2 Demultiplexer interface not found.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;
      Hr := FMPEG2Demux.FindPin('Video', FVideoPin);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer video pin not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      Hr := FMPEG2Demux.FindPin('Audio', FAudioPin);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'MPEG-2 Demultiplexer audio pin not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      // We do make a link but we allow that it fails
      Hr := FMPEG2Demux.FindPin('AudioAC3', FAudioAc3Pin);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        FAudioAc3Pin := nil;
//        ErrorMessage := 'MPEG-2 Demultiplexer AC3 audio pin not found.';
//        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
//        Exit;
      end;

      {------------------------------------------------------------------------------
        Descript: Render
       ------------------------------------------------------------------------------}
      if (FOptions and CDirectShowMethodDoNotRender) = 0 then
      begin
        if not RenderPin('MPEG-2 Demultiplexer', 'Audio', ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer audio pin rendering error.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        // AC3 should not generate an error
        RenderPin('MPEG-2 Demultiplexer', 'AudioAC3', ResultMessage);
        if not RenderPin('MPEG-2 Demultiplexer', 'Video', ResultMessage) then
        begin
          ErrorMessage := 'MPEG-2 Demultiplexer video pin rendering error.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;

        {------------------------------------------------------------------------------
          Descript: Render all pins of all filters
         ------------------------------------------------------------------------------}
        if (FOptions and CDirectShowMethodRenderAllPins) <> 0 then
        begin
          case (FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
              IFilterGraph2).EnumFilters(EnumFilters);
            CDirectShowMethodGraphBuilder,
              CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
                IGraphBuilder).EnumFilters(EnumFilters);
            CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
              IFilterGraph).EnumFilters(EnumFilters);
          end;
          AllPinsCount := 0;
          // If we don't retrieve all filters/pins before rendering them, then we will not get all filters for some reason
          if Succeeded(Hr) then
          begin
            EnumFilters.Reset;
            while EnumFilters.Next(1, BaseFilter, nil) = S_OK do
            begin
              Hr := BaseFilter.EnumPins(PinsEnum);
              if Succeeded(Hr) then
              begin
                PinsEnum.Reset;
                while PinsEnum.Next(1, Pins, nil) = S_OK do
                begin
                  AllPins[AllPinsCount] := Pins;
                  Inc(AllPinsCount);
                end;
                Pins := nil;
              end;
            end;
            EnumFilters := nil;
          end;
          // We now have all the filters with their pins, so render then
          if AllPinsCount > 0 then
            for Index1 := 0 to AllPinsCount - 1 do
              case (FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2:
                  begin
                    if (FOptions and CDirectShowMethodRenderEx) <> 0 then
                    begin
                      Hr := (FFilterGraphManager as
                        IFilterGraph2).RenderEx(AllPins[Index1],
                        AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                      if Failed(Hr) then
                        (FFilterGraphManager as
                          IFilterGraph2).Render(AllPins[Index1]);
                    end
                    else
                      (FFilterGraphManager as
                        IFilterGraph2).Render(AllPins[Index1]);
                  end;
                CDirectShowMethodGraphBuilder,
                  CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
                    IGraphBuilder).Render(AllPins[Index1]);
              end;
        end;
        {------------------------------------------------------------------------------
          Descript: Render all audio/video pins of all filters
         ------------------------------------------------------------------------------}
        if (FOptions and CDirectShowMethodRenderAllAudioVideo) <> 0 then
        begin
          case (FOptions and CDirectShowMethodGraph) of
            CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
              IFilterGraph2).EnumFilters(EnumFilters);
            CDirectShowMethodGraphBuilder,
              CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
                IGraphBuilder).EnumFilters(EnumFilters);
            CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
              IFilterGraph).EnumFilters(EnumFilters);
          end;
          AllPinsCount := 0;
          // If we don't retrieve all filters/pins before rendering them, then we will not get all filters for some reason
          if Succeeded(Hr) then
          begin
            EnumFilters.Reset;
            while EnumFilters.Next(1, BaseFilter, nil) = S_OK do
            begin
              Hr := BaseFilter.EnumPins(PinsEnum);
              if Succeeded(Hr) then
              begin
                PinsEnum.Reset;
                while (PinsEnum.Next(1, Pins, nil) = S_OK) do
                begin
                  // Pins.QueryDirection(PinDirection);
                  //  if PinDirection = PINDIR_OUTPUT then
                  Pins.QueryPinInfo(PinsInfo);
                  PinsInfo.pFilter := nil;
                  // If not connected
                  if PinsInfo.dir = PINDIR_OUTPUT then // Must be output
                  begin
                    Hr := Pins.EnumMediaTypes(EnumMediaTypes);
                      // Check media types
                    if Succeeded(Hr) then
                    begin
                      EnumMediaTypes.Reset;
                      while EnumMediaTypes.Next(1, PMediaType, nil) = S_OK do
                      begin
                        if IsEqualGUID(PMediaType.majortype, MEDIATYPE_Video) or
                          IsEqualGUID(PMediaType.majortype, MEDIATYPE_Audio)
                            then
                        begin
                          AllPins[AllPinsCount] := Pins;
                          Inc(AllPinsCount);
                        end;
                        PMediaType := nil;
                      end;
                    end;
                  end;
                end;
                APin := nil;
                Pins := nil;
              end;
            end;
            EnumFilters := nil;
          end;
          // We now have all the filters with their pins, so render then
          if AllPinsCount > 0 then
            for Index1 := 0 to AllPinsCount - 1 do
              case (FOptions and CDirectShowMethodGraph) of
                CDirectShowMethodFilterGraph2:
                  begin
                    if (FOptions and CDirectShowMethodRenderEx) <> 0 then
                    begin
                      Hr := (FFilterGraphManager as
                        IFilterGraph2).RenderEx(AllPins[Index1],
                        AM_RENDEREX_RENDERTOEXISTINGRENDERERS, nil);
                      if Failed(Hr) then
                        (FFilterGraphManager as
                          IFilterGraph2).Render(AllPins[Index1]);
                    end
                    else
                      (FFilterGraphManager as
                        IFilterGraph2).Render(AllPins[Index1]);
                  end;
                CDirectShowMethodGraphBuilder,
                  CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
                    IGraphBuilder).Render(AllPins[Index1]);
              end;
        end;
      end;
      Hr := FindFilterByName('Universal Source', FUniversalSource);
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'Universal Source filter not found.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;

      FMPEG2Demux.SetSyncSource(nil);
      SetDefaultSyncSource;

      // Get IBasicVideo interface
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
          IFilterGraph2).QueryInterface(IID_IBasicVideo, FBasicVideo);
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
            IGraphBuilder).QueryInterface(IID_IBasicVideo, FBasicVideo);
        CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
          IFilterGraph).QueryInterface(IID_IBasicVideo, FBasicVideo);
      else
        Hr := E_NOTIMPL;
      end;
      {------------------------------------------------------------------------------
        Descript: No VMR (default Video Renderer)
       ------------------------------------------------------------------------------}
      if ((FOptions and CDirectShowMethodVmr) = CDirectShowMethodVideoRenderer)
        or
        (not Assigned(FWindowlessControl)) then
      begin
        case (FOptions and CDirectShowMethodGraph) of
          CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
            IFilterGraph2).QueryInterface(IID_IVideoWindow, FVideoWindow);
          CDirectShowMethodGraphBuilder,
            CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
              IGraphBuilder).QueryInterface(IID_IVideoWindow, FVideoWindow);
          CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
            IFilterGraph).QueryInterface(IID_IVideoWindow, FVideoWindow);
        end;
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Could not retrieve video window interface.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := FVideoWindow.put_Owner(FOwner);

        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage :=
            format('Filter could not set video handle, error $%8.8x.', [Hr]);
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := FVideoWindow.put_MessageDrain(FOwner);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set messages handle.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
        Hr := FVideoWindow.put_WindowStyle(WS_CHILD or WS_CLIPSIBLINGS or
          WS_CLIPCHILDREN);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set window style.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;

        GetClientRect(FOwner, DestinationRect);
        Hr := FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right -
          DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        if not DirectShowCheck(Hr, ResultMessage) then
        begin
          ErrorMessage := 'Filter could not set window size.';
          ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
          Exit;
        end;
      end;

      // Media control
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
          IFilterGraph2).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
            IGraphBuilder).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
          IFilterGraph).QueryInterface(IID_IMediaControl, MediaControl);
      end;
      if not DirectShowCheck(Hr, ResultMessage) then
      begin
        ErrorMessage := 'Could not get media control interface.';
        ErrorMessage := ErrorMessage + #13#13 + ResultMessage;
        Exit;
      end;
      MediaControl.Stop;
      MediaControl.Run;
      MediaControl.GetState(1000, FilterState);
      MediaControl := nil;
      // Indicate success
      Result := True;
    finally
      // Make sure that at a failure all resources are freed
      if not Result then
        DirectShowStop;
    end;
  except
    Result := False;
    DirectShowStop;
    ShowMessage('DirectShow Start error');
  end;
end;

function TDirectShowGraph.DirectShowBlendImage(Bitmap: HDC; Src: TRect; Blend:
  Single; ColorKey: COLORREF): Boolean;
begin
  Result := False;
  if (FOptions and CDirectShowMethodVmr) = CDirectShowMethodVideoRenderer then
    Exit;
  Result := BlendApplicationImage(Bitmap, Src, Blend, ColorKey);
end;

{------------------------------------------------------------------------------
  Params  : <Owner>  Handle to window for video
  Returns : -

  Descript: Constructor of object.
  Notes   :
 ------------------------------------------------------------------------------}

constructor TDirectShowGraph.Create(Owner: HWND);
begin
  inherited Create;
  DirectShowAssign(Owner);
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor of object.
  Notes   :
 ------------------------------------------------------------------------------}

destructor TDirectShowGraph.Destroy;
begin
  DirectShowStop;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Params  : <Owner>  Handle to window for video
  Returns : -

  Descript: Initialize of object.
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowAssign(Owner: HWND);
begin
  FOwner := Owner;
  FLogFile := INVALID_HANDLE_VALUE;
  FRotEntry := -1;
  FFilterGraphManager := nil;
  FGraphBuilder := nil;
  FBasicVideo := nil;
  FVmrFilter := nil;
  FVmrFilterConfig7 := nil;
  FVmrFilterConfig9 := nil;
  FVmrMixerControl7 := nil;
  FVmrMixerControl9 := nil;
  FMPEG2Demux := nil;
  FMPEG2Demultiplexer := nil;
  FUniversalSource := nil;
  FVideoPin := nil;
  FAudioPin := nil;
  FAudioAc3Pin := nil;
  FVideoWindow := nil;
  FWindowlessControl := nil;
  FActiveAudioPid := -1;
  FActiveVideoPid := -1;
  FActiveAudioAc3 := False;
  ZeroMemory(@FAlphaBitmap7, sizeof(FAlphaBitmap7));
  ZeroMemory(@FAlphaBitmap7, sizeof(FAlphaBitmap9));
  FAlphaBitmap7.Hdc := 0;
  FAlphaBitmap9.Hdc := 0;
end;

{------------------------------------------------------------------------------
  Params  : <Bitmap>    Bitmap to blend in
            <Dest>      Source position/size
            <Blend>     Blending value (0.0...1.0 == transparent..opaque)
            <ColorKey>  Transparancy color
  Returns : <Result>  True if success

  Descript: Blend in application image
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.BlendApplicationImage(Bitmap: HDC; Src: TRect; Blend:
  Single; ColorKey: COLORREF): Boolean;
begin
  Result := True;
  case (FOptions and CDirectShowMethodVmr) of
    CDirectShowMethodVideoRenderer: ;
    CDirectShowMethodVideoRenderer7:
      begin
        with FAlphaBitmap7 do
        begin
          Hdc := Bitmap;
          rSrc := Src;
          rDest.Left := 0;
          rDest.Top := 0;
          rDest.Right := 1;
          rDest.Bottom := 1;
          dwFlags := VMRBITMAP_SRCCOLORKEY or VMRBITMAP_HDC;
          clrSrcKey := ColorKey;
          fAlpha := Blend;
        end;
        (FVmrFilter as IVMRMixerBitmap).SetAlphaBitmap(FAlphaBitmap7);
      end;
    CDirectShowMethodVideoRenderer9:
      begin
        with FAlphaBitmap9 do
        begin
          Hdc := Bitmap;
          rSrc := Src;
          rDest.Left := 0;
          rDest.Top := 0;
          rDest.Right := 1;
          rDest.Bottom := 1;
          dwFlags := VMRBITMAP_SRCCOLORKEY or VMRBITMAP_HDC;
          clrSrcKey := ColorKey;
          fAlpha := Blend;
        end;
        (FVmrFilter as IVMRMixerBitmap9).SetAlphaBitmap(@FAlphaBitmap9);
      end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if 'windowless' mode

  Descript: Get windowless mode (VMR)
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowWindowless: Boolean;
begin
  if Assigned(DirectShowGraph) and Assigned(FWindowlessControl) then
    Result := True
  else
    Result := False;
end;

{------------------------------------------------------------------------------
  Params  : <Index>  Filter index (0 = first)
  Returns : <Result> Name of filter (empty if no filter at index)

  Descript: Get filter name
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowGetFilterName(Index: Integer): string;
var
  EnumFilters: IEnumFilters;
  BaseFilter: IBaseFilter;
  FilterInfo: TFilterInfo;
  FilterIndex: Integer;
begin
  Result := '';
  try
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: (FFilterGraphManager as
        IFilterGraph2).EnumFilters(EnumFilters);
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
          IGraphBuilder).EnumFilters(EnumFilters);
      CDirectShowMethodFilterGraph: (FFilterGraphManager as
        IFilterGraph).EnumFilters(EnumFilters);
    end;
    FilterIndex := -1;
    EnumFilters.Reset;
    while (EnumFilters.Next(1, BaseFilter, nil) = S_OK) and
      (FilterIndex < Index) do
    begin
      BaseFilter.QueryFilterInfo(FilterInfo);
      Result := FilterInfo.achName;
      Inc(FilterIndex);
    end;
    // Make sure an incorrect index returns nothing
    if FilterIndex <> Index then
      Result := '';
  except
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Parent>      Handle of parent to create property page in
            <FilterName>  Name of filter
            <QueryOnly>   True if only to check for property page (not to show)
  Returns : <Result>      True if property page available

  Descript: Show property page of filter
  Notes   :
 ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowShowPropertyPage(Parent: THandle;
  FilterName: WideString; QueryOnly: Boolean): Boolean;
var
  BaseFilter: IBaseFilter;
  SpecifyPropertyPages: ISpecifyPropertyPages;
  CAGUID: TCAGUID;
  FilterInfo: TFilterInfo;
  Hr: HRESULT;
begin
  Result := False;
  try
    // Get filter using the name
    case (FOptions and CDirectShowMethodGraph) of
      CDirectShowMethodFilterGraph2: Hr := (FFilterGraphManager as
        IFilterGraph2).FindFilterByName(PWideChar(FilterName), BaseFilter);
      CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: Hr := (FFilterGraphManager as
          IGraphBuilder).FindFilterByName(PWideChar(FilterName), BaseFilter);
      CDirectShowMethodFilterGraph: Hr := (FFilterGraphManager as
        IFilterGraph).FindFilterByName(PWideChar(FilterName), BaseFilter);
    else
      Hr := E_NOTIMPL;
    end;
    if Succeeded(Hr) then
    begin
      //    ZeroMemory(@FilterInfo, SizeOf(TFilterInfo));
          // Get property pages
      Hr := BaseFilter.QueryInterface(IID_ISpecifyPropertyPages,
        SpecifyPropertyPages);
      if Succeeded(Hr) then
      begin
        // Get GUID
        Hr := SpecifyPropertyPages.GetPages(CAGUID);
        if Succeeded(Hr) then
        begin
          // Check number of pages available
          if CAGUID.cElems < 1 then
            Exit;
          if not QueryOnly then
          begin
            // Get info
            Hr := BaseFilter.QueryFilterInfo(FilterInfo);
            if Succeeded(Hr) then
            begin
              Hr := OleCreatePropertyFrame(Parent, 0, 0, FilterInfo.achName, 1,
                @BaseFilter, CAGUID.cElems, CAGUID.pElems, 0, 0, nil);
              if Succeeded(Hr) then
                Result := True;
            end;
          end
          else
            Result := True;
        end;
      end;
    end;
  except
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Stop DirectShow interface by releasing all assigned sruff
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowStop;
var
  MediaControl: IMediaControl;
begin
  if Assigned(FFilterGraphManager) then
  begin
    try
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: (FFilterGraphManager as
          IFilterGraph2).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
            IGraphBuilder).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodFilterGraph: (FFilterGraphManager as
          IFilterGraph).QueryInterface(IID_IMediaControl, MediaControl);
      end;
      MediaControl.Stop;
      MediaControl := nil;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: (FFilterGraphManager as
          IFilterGraph2).Abort;
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
            IGraphBuilder).Abort;
      end;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: (FFilterGraphManager as
          IFilterGraph2).SetLogFile(0);
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as
            IGraphBuilder).SetLogFile(0);
      end;
      if FRotEntry <> -1 then
        RemoveFromRot(FRotEntry);
      // Important to stop the message drain, otherwise errors occur
      FRotEntry := -1;
      FVideoWindow := nil;
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2: IFilterGraph2(FFilterGraphManager) :=
          nil;
        CDirectShowMethodGraphBuilder,
          CDirectShowMethodDvdGraphBuilder: IGraphBuilder(FFilterGraphManager)
            := nil;
      end;
      FGraphBuilder := nil;
      FVmrFilter := nil;
      FVmrFilterConfig7 := nil;
      FVmrFilterConfig9 := nil;
      FVmrMixerControl7 := nil;
      FVmrMixerControl9 := nil;
      FBasicVideo := nil;
      FMPEG2Demux := nil;
      FMPEG2Demultiplexer := nil;
      FUniversalSource := nil;
      FVideoPin := nil;
      FAudioPin := nil;
      FAudioAc3Pin := nil;
      FActiveAudioPid := -1;
      FActiveVideoPid := -1;
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer7:
          IVMRWindowlessControl(FWindowlessControl) := nil;
        CDirectShowMethodVideoRenderer9:
          IVMRWindowlessControl9(FWindowlessControl) := nil;
      end;
      if FLogFile <> INVALID_HANDLE_VALUE then
        CloseHandle(FLogFile);
      FLogFile := INVALID_HANDLE_VALUE;
    except
    end;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <FullScreen>  True to set full screen mode
                          False to restore original mode
  Returns : -

  Descript: Set full screen mode
  Notes   : Only for non-VMR9 modes
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowFullScreenMode(FullScreen: Boolean);
begin
  try
    if Assigned(FVideoWindow) then
      FVideoWindow.put_FullScreenMode(FullScreen);
  except
    DirectShowStop;
    ShowMessage('DirectShow FullScreenMode error');
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Video>  Video window
  Returns : -

  Descript: Set video window
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowSetVideo(Video: HWND);
begin
  try
    if (FOptions and CDirectShowMethodVmrSpecials) <>
      CDirectShowMethodVideoRendererWindowless then
      Exit;
    if not Assigned(FWindowlessControl) then
      Exit;
    FOwner := Video;
    case (FOptions and CDirectShowMethodVmr) of
      CDirectShowMethodVideoRenderer7:
        IVMRWindowlessControl(FWindowlessControl).SetVideoClippingWindow(FOwner);
      CDirectShowMethodVideoRenderer9:
        IVMRWindowlessControl9(FWindowlessControl).SetVideoClippingWindow(FOwner);
    end;
  except
    DirectShowStop;
    ShowMessage('DirectShow SetVideo error');
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Resize to full client area
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowResize;
var
  DestinationRect: TRect;
begin
  try
    GetWindowRect(FOwner, DestinationRect);
    DestinationRect.Right := DestinationRect.Right - DestinationRect.Left;
    DestinationRect.Bottom := DestinationRect.Bottom - DestinationRect.Top;
    DestinationRect.Top := 0;
    DestinationRect.Left := 0;
    if ((FOptions and CDirectShowMethodVmrSpecials) =
      CDirectShowMethodVideoRendererWindowless) and
      (Assigned(FWindowlessControl)) then
    begin
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer: if Assigned(FVideoWindow) then
            FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right -
              DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
        CDirectShowMethodVideoRenderer7:
          IVMRWindowlessControl(FWindowlessControl).SetVideoPosition(nil,
          @DestinationRect);

        CDirectShowMethodVideoRenderer9:
          IVMRWindowlessControl9(FWindowlessControl).SetVideoPosition(nil,
          @DestinationRect);
      end;
    end
    else if Assigned(FVideoWindow) then
      FVideoWindow.SetWindowPosition(0, 0, DestinationRect.Right -
        DestinationRect.Left, DestinationRect.Bottom - DestinationRect.Top);
  except
    DirectShowStop;
    ShowMessage('DirectShow Resize error');
  end;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Width>  Width of native video
            <Height> Height of native video

  Descript: Get native video size
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowGetSize(var Width: Integer; var Height:
  Integer);
var
  AWidth: Integer;
  AHeight: Integer;
  //  MediaType  : AM_MEDIA_TYPE;
  //  VideoHeader: PVideoInfoHeader;
begin
  try
    //    // Next could be used instead, but needs the correct PIN of the video decoder
    //    XXXXXXPin.ConnectionMediaType(MediaType);
    //    if IsEqualGUID(MediaType.majortype, MEDIATYPE_Video) then
    //    begin
    //      // Get pointer to header info
    //      VideoHeader := PVideoInfoHeader(MediaType.pbFormat);
    //      Width  := VideoHeader.bmiHeader.biWidth;
    //      Height := VideoHeader.bmiHeader.biHeight;
    //    end;
    if ((FOptions and CDirectShowMethodVmrSpecials) =
      CDirectShowMethodVideoRendererWindowless) and
      (Assigned(FWindowlessControl)) then
    begin
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer: if Assigned(FBasicVideo) then
            FBasicVideo.GetVideoSize(Width, Height);
        CDirectShowMethodVideoRenderer7:
          IVMRWindowlessControl(FWindowlessControl).GetNativeVideoSize(Width,
          Height, AWidth, AHeight);
        CDirectShowMethodVideoRenderer9:
          IVMRWindowlessControl9(FWindowlessControl).GetNativeVideoSize(Width,
          Height, AWidth, AHeight);
      end;
    end
    else if Assigned(FBasicVideo) then
      FBasicVideo.GetVideoSize(Width, Height)
  except
    DirectShowStop;
    ShowMessage('DirectShow GetSize error');
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Dib>     Pointer to DIB
  Returns : <Result>  Size of header

  Descript: Get header size of DIB
  Notes   :
 ------------------------------------------------------------------------------}

function GetDibHeaderSize(Dib: PBitmapInfo): Integer;
begin
  // Defaults for colors used, so get the color table count from
  // the number of bits per pixel
  if (Dib.bmiHeader.biClrUsed = 0) then
  begin
    case Dib.bmiHeader.biBitCount of
      1: Dib.bmiHeader.biClrUsed := 2;
      2: Dib.bmiHeader.biClrUsed := 4;
      3: Dib.bmiHeader.biClrUsed := 8;
      4: Dib.bmiHeader.biClrUsed := 16;
      5: Dib.bmiHeader.biClrUsed := 32;
      6: Dib.bmiHeader.biClrUsed := 64;
      7: Dib.bmiHeader.biClrUsed := 128;
      8: Dib.bmiHeader.biClrUsed := 256;
    end;
  end;
  Result := Dib.bmiHeader.biSize + (Dib.bmiHeader.biClrUsed * 4);
  if ((Dib.bmiHeader.biCompression = BI_BITFIELDS) and (Dib.bmiHeader.biSize = SizeOf(TBitmapInfoHeader))) then
    Result := Result + 12;
end;

{------------------------------------------------------------------------------
  Params  : <Width>   Width in pixels
            <Depth>   Depth in pixels
  Returns : <Result>  Size of header

  Descript: Return 4 byte aligned data
  Notes   :
 ------------------------------------------------------------------------------}

function AlignScan(Width: Dword; Depth: Dword): Dword;
begin
  Result := (((Width * Depth) + $1F) and $FFFFFFE0) div 8;
end;

{------------------------------------------------------------------------------
  Params  : <Dib>     Pointer to DIB
  Returns : <Result>  Size of data

  Descript: Return DIB data size
  Notes   :
 ------------------------------------------------------------------------------}

function GetDibDataSize(Dib: PBitmapInfo): Integer;
begin
  case Dib.bmiHeader.biCompression of
    BI_RGB, BI_BITFIELDS:
      begin
        Result := Integer(AlignScan(Dword(Dib.bmiHeader.biWidth), Dword(Dib.bmiHeader.biBitCount))) * Abs(Dib.bmiHeader.biHeight);
        Dib.bmiHeader.biSizeImage := Result;
      end;
    else Result := Dib.bmiHeader.biSizeImage;
  end;
end;

{------------------------------------------------------------------------------
  Params  : <Dib>     Pointer to DIB
  Returns : <Result>  Handle to bitmap (0 if none)

  Descript: Convert packed DIB into DIB section (handle)
  Notes   : The caller should release the handle (e.g. with DeleteObject(x)!
 ------------------------------------------------------------------------------}

function PackedDibToDibSection(Dib: PBitmapInfo): HBITMAP;
var
  BmHead   : TBitmapInfoHeader;
  HeadSize : Integer;
  DataSize : Integer;
  DataBits : Pointer;
begin
  CopyMemory(@BmHead, Dib, SizeOf(Bmhead));
  // Get header and data size
  HeadSize := GetDibHeaderSize(@BmHead);
  DataSize := GetDibDataSize(@BmHead);
  // Create DibSection based on header
  Result := CreateDibSection(0, Dib^, DIB_RGB_COLORS, DataBits, 0, 0);
  // Copy bitmap to it
  if Result <> 0 then
    CopyMemory(DataBits, Pointer(@PByteArray(Dib)^[HeadSize]), DataSize);
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Handle to bitmap (0 if none)

  Descript: Get current image
  Notes   : The caller must release the handle with DeleteObject(x)!
 ------------------------------------------------------------------------------}

function TDirectShowGraph.DirectShowGetImage: HBITMAP;
var
  Local       : PBitmapInfo;
  MediaControl: IMediaControl;
  FilterState : TFilterState;
  Hr          : HRESULT;

  procedure BasicVideoGetImage;
  var
    ASize  : Integer;
  begin
    if Assigned(FBasicVideo) then
    begin
      case (FOptions and CDirectShowMethodGraph) of
        CDirectShowMethodFilterGraph2   : (FFilterGraphManager as IFilterGraph2).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodGraphBuilder,
        CDirectShowMethodDvdGraphBuilder: (FFilterGraphManager as IGraphBuilder).QueryInterface(IID_IMediaControl, MediaControl);
        CDirectShowMethodFilterGraph    : (FFilterGraphManager as IFilterGraph ).QueryInterface(IID_IMediaControl, MediaControl);
      end;
      if Assigned(MediaControl) then
      begin
        MediaControl.Pause;
        MediaControl.GetState(1, FilterState);
      end;
      // Check required memory
      ASize := 0;
      Local := nil;
      Hr := FBasicVideo.GetCurrentImage(ASize, Local^);
      if Succeeded(Hr) then
      begin
        GetMem(Local, ASize);
        try
          Hr := FBasicVideo.GetCurrentImage(ASize, Local^);
          Result := PackedDibToDibSection(Local);
        finally
          FreeMem(Local);
        end;
      end;  
      if Assigned(MediaControl) then
        MediaControl.Run;
      MediaControl := nil;
    end;
  end;
begin
  Result := 0;
  try
    if ((FOptions and CDirectShowMethodVmrSpecials) =
      CDirectShowMethodVideoRendererWindowless) and
      (Assigned(FWindowlessControl)) then
    begin
      case (FOptions and CDirectShowMethodVmr) of
        CDirectShowMethodVideoRenderer7:
          begin
            Hr := IVMRWindowlessControl(FWindowlessControl).GetCurrentImage(Local);
            if Succeeded(Hr) then
            begin
              Result := PackedDibToDibSection(Local);
              CoTaskMemFree(Local);
            end;
          end;
        CDirectShowMethodVideoRenderer9:
          begin
            Hr := IVMRWindowlessControl9(FWindowlessControl).GetCurrentImage(PByte(Local));
            if Succeeded(Hr) then
            begin
              Result := PackedDibToDibSection(Local);
              CoTaskMemFree(Local);
            end;
          end;
        else BasicVideoGetImage;
      end;
    end
    else
      BasicVideoGetImage;
  except
    ShowMessage('DirectShow GetCurrentImage error');
    Result := 0;
  end;
end;

{------------------------------------------------------------------------------
  Descript: DirectShow MPEG2 demultipexer filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <VideoPid>  Video PID
            <AudioPid>  Audio PID
            <AudioAC3>  True if AC3 pin to use
  Returns : -

  Descript: Set new PIDs to use for MPEG2 demultiplexer
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowMpeg2DemultiplexerSetNewPids(VideoPid:
  Integer; AudioPid: Integer; AudioAC3: Boolean);
var
  MapPid: Cardinal;
begin
  try
    if not Assigned(FVideoPin) then
      Exit;
    if not Assigned(FAudioPin) and not Assigned(FAudioAc3Pin) then
      Exit;
    if FActiveVideoPid >= 0 then
    begin
      MapPid := FActiveVideoPid;
      (FVideoPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;
    if FActiveAudioPid >= 0 then
    begin
      MapPid := FActiveAudioPid;
      if FActiveAudioAc3 then
        (FAudioAc3Pin as IMPEG2PIDMap).UnmapPID(1, @MapPid)
      else
        (FAudioPin as IMPEG2PIDMap).UnmapPID(1, @MapPid);
    end;

    // AC3 only if pin is available and AC3 output is required
    if Assigned(FAudioAc3Pin) and AudioAc3 then
      FActiveAudioAc3 := True
    else
      FActiveAudioAc3 := False;

    // Set new active PIDs
    FActiveVideoPid := VideoPid;
    FActiveAudioPid := AudioPid;

    if FActiveVideoPid >= 0 then
    begin
      MapPid := FActiveVideoPid;
      (FVideoPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;
    if FActiveAudioPid >= 0 then
    begin
      MapPid := FActiveAudioPid;
      if not FActiveAudioAc3 then
        (FAudioPin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM)
      else
        (FAudioAc3Pin as IMPEG2PIDMap).MapPID(1, @MapPid, MEDIA_ELEMENTARY_STREAM);
    end;
  except
    DirectShowStop;
    ShowMessage('DirectShow Mpeg2DemultiplexerSetNewPids error');
  end;
end;

{------------------------------------------------------------------------------
  Descript: DirectShow Universal Source filter
 ------------------------------------------------------------------------------}
{------------------------------------------------------------------------------
  Params  : <Buffer>        Pointer to data
            <BufferLength>  Length of valid data in buffer
  Returns : -

  Descript: Send data to Universal Source filter
  Notes   :
 ------------------------------------------------------------------------------}

procedure TDirectShowGraph.DirectShowUniversalSourceSendData(Buffer: PChar;
  BufferLength: Integer);
begin
  try
    if not Assigned(FUniversalSource) then
      Exit;
    (FUniversalSource as IUSRCSet).WSendSample(Buffer, BufferLength);
  except
    DirectShowStop;
    ShowMessage('DirectShow UniversalSourceSendData error');
  end;
end;

{------------------------------------------------------------------------------
  Descript: <DvbFilter> functions: Either internal function called or
            <DvbMajorPsi> DirectShow filter is called -> REMOVED
  Note    : For descriptions of functions/procedures see <DvbFilter>
 ------------------------------------------------------------------------------}

function DvbGetPid(StreamPacketData: PDvbTransportPacket; var Pid: Word): Boolean;
begin
  Result := DvbFilter.DvbFilterGetPid(StreamPacketData, Pid);
end;

function DvbSetSignallingCallback(Pid: Word; hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallback(Pid, hWnd, Msg);
end;

function DvbSetSignallingCallbackPat(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPat(hWnd, Msg);
end;

function DvbSetSignallingCallbackCat(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackCat(hWnd, Msg);
end;

function DvbSetSignallingCallbackPmt(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackPmt(hWnd, Msg);
end;

function DvbSetSignallingCallbackEvent(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackEvent(hWnd, Msg);
end;

function DvbSetSignallingCallbackSdt(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackSdt(hWnd, Msg);
end;

function DvbSetSignallingCallbackNit(hWnd: HWND; Msg: UINT): Boolean;
begin
  Result := DvbFilter.DvbFilterSetSignallingCallbackNit(hWnd, Msg);
end;

function DvbSetPidFilter(Pid: Word; Filter: TDvbPacketFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPidFilter(Pid, Filter);
end;

function DvbGetPidFilter(Pid: Word): TDvbPacketFilter;
begin
  Result := DvbFilter.DvbFilterGetPidFilter(Pid);
end;

function DvbSetPsiFilter(Psi: Byte; Filter: TDvbSectionFilter): Boolean;
begin
  Result := DvbFilter.DvbFilterSetPsiFilter(Psi, Filter);
end;

function DvbGetPsiFilter(Psi: Byte): TDvbSectionFilter;
begin
  Result := DvbFilter.DvbFilterGetPsiFilter(Psi);
end;

function DvbGetProgramInfo(ProgramNumber: Byte;
  var ServiceId: Word;
  var PmtPid: Word;
  var PcrPid: Word;
  var VideoPids: TDvbPids;
  var AudioPids: TDvbPids;
  var TeletextPids: TDvbPids;
  var SubtitlePids: TDvbPids;
  var AudioLanguages: TDvbLanguages;
  var SubtitleLanguages: TDvbLanguages;
  var EcmPids: TDvbPids;
  var CaIds: TDvbPids;
  var ProgramName: string): Boolean;
begin
  Result := DvbFilter.DvbFilterGetProgramInfo(ProgramNumber, ServiceId,
    PmtPid, PcrPid, VideoPids, AudioPids,
    TeletextPids, SubtitlePids, AudioLanguages,
    SubtitleLanguages, EcmPids, CaIds, ProgramName);
end;

function DvbGetEventInfo(ProgramNumber: Byte; ServiceId: Word; Present: Boolean;
  Events: TDvbEventSections): Boolean;
begin
  Result := DvbFilter.DvbFilterGetEventInfo(ProgramNumber, ServiceId, Present,
    Events);
end;

function DvbGetNumberOfPrograms: Byte;
begin
  Result := DvbFilter.DvbFilterGetNumberOfPrograms;
end;

function DvbGetErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetErrors;
end;

function DvbGetPacketSyncErrors: Word;
begin
  Result := DvbFilter.DvbFilterGetPacketSyncErrors;
end;

procedure DvbResetErrors;
begin
  DvbFilter.DvbFilterResetErrors;
end;

procedure DvbCreateTables;
begin
  DvbFilter.DvbFilterCreateTables;
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialization
  Notes   :
 ------------------------------------------------------------------------------}

procedure InitializeUnit;
begin
  DirectShowGraph := TDirectShowGraph.Create(0);
end;

{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalization
  Notes   :
 ------------------------------------------------------------------------------}

procedure FinalizeUnit;
begin
  if Assigned(DirectShowGraph) then
    FreeAndNil(DirectShowGraph);
end;

initialization
  InitializeUnit;

finalization
  FinalizeUnit;
end.

