{******************************************************************************}
{ FileName............: MajorDvbSetupDirectShow                                }
{ Project.............: SAA7146A                                               }
{ Author(s)...........: MM                                                     }
{ Version.............: 0.02                                                   }
{------------------------------------------------------------------------------}
{  DirectShow options                                                          }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{ Version   Date   Comment                                                     }
{  0.01   20041116 - Initial release                                           }
{  0.02   20041225 - Added options                                             }
{******************************************************************************}
unit MajorDvbSetupDirectShow;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls;

type
  TfrmSetupDirectShow = class(TForm)
    rgFilter: TRadioGroup;
    rgBuilders: TRadioGroup;
    btnOK: TButton;
    btnDiscard: TButton;
    grpMiscellaneous: TGroupBox;
    chkDoNotRenderGraph: TCheckBox;
    chkDoNotAddToRot: TCheckBox;
    chkRenderAllPins: TCheckBox;
    chkRenderAllAudioVideo: TCheckBox;
    chkNativeVideoSize: TCheckBox;
    Shape2: TShape;
    Shape1: TShape;
    gbVideoMixingRenderer: TGroupBox;
    rgVmr: TRadioGroup;
    rgVmrWindowed: TRadioGroup;
    chkFlipX: TCheckBox;
    chkFlipY: TCheckBox;
    chkLogToFile: TCheckBox;
    chkRenderEx: TCheckBox;
    cmbDirectShowGraph: TComboBox;
    Label1: TLabel;
    chkDirectShowOff: TCheckBox;
    chkDirectShowSendAll: TCheckBox;
    chkAlternativeDirectShow: TCheckBox;
    btnRevert: TButton;
    chkNoAC3Pin: TCheckBox;
    procedure SetOptions(Value: Dword);
    function  GetOptions: Dword;
    procedure SetDirectShowOff(Value: Boolean);
    function  GetDirectShowOff: Boolean;
    procedure SetDirectShowSendAll(Value: Boolean);
    function  GetDirectShowSendAll: Boolean;
    procedure btnOKClick(Sender: TObject);
    procedure btnDiscardClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure chkDoNotRenderGraphClick(Sender: TObject);
    procedure btnRevertClick(Sender: TObject);
  private
    { Private declarations }
    FOptions             : Dword;
    FOrgOptions          : Dword;
    FOrgDirectShowOff    : Boolean;
    FOrgDirectShowSendAll: Boolean;
    FOrgGraph            : string;
    FExitCallback : TNotifyEvent;
    FAccepted     : Boolean;
    procedure Revert;
    function  GetSettingChanged: Boolean;
    function  GetGraph         : string;
    procedure SetGraph(Value: string);
  public
    { Public declarations }
    property Options          : Dword        read GetOptions           write SetOptions;
    property DirectShowOff    : Boolean      read GetDirectShowOff     write SetDirectShowOff;
    property DirectShowSendAll: Boolean      read GetDirectShowSendAll write SetDirectShowSendAll;
    property Graph            : string       read GetGraph             write SetGraph;
    property ExitNotify       : TNotifyEvent read FExitCallback        write FExitCallback;
    property SettingChanged   : Boolean      read GetSettingChanged;
    property Accepted         : Boolean      read FAccepted;
  end;

implementation

uses
  DvbDirectShow;

{$R *.DFM}


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if a setting has changed

  Descript: Get flag if a setting has changed
  Notes   :
 ------------------------------------------------------------------------------}
function TfrmSetupDirectShow.GetSettingChanged: Boolean;
begin
  Result := False;
  Result := Result or (FOrgOptions           <> Options);
  Result := Result or (FOrgGraph             <> cmbDirectShowGraph.Text);
  Result := Result or (FOrgDirectShowOff     <> chkDirectShowOff.Checked);
  Result := Result or (FOrgDirectShowSendAll <> chkDirectShowSendAll.Checked);
end;


{------------------------------------------------------------------------------
  Params  : <Value>  Options to set
  Returns : -

  Descript: Set options
  Notes   :
 ------------------------------------------------------------------------------}
procedure TfrmSetupDirectShow.SetOptions(Value: Dword);
begin
  FOptions    := Value;
  FOrgOptions := FOptions;
  chkDoNotRenderGraph.Checked      := (FOptions and CDirectShowMethodDoNotRender)         <> 0;
  chkDoNotAddToRot.Checked         := (FOptions and CDirectShowMethodDoNotAddToRot)       <> 0;
  chkRenderAllPins.Checked         := (FOptions and CDirectShowMethodRenderAllPins)       <> 0;
  chkRenderAllAudioVideo.Checked   := (FOptions and CDirectShowMethodRenderAllAudioVideo) <> 0;
  chkRenderEx.Checked              := (FOptions and CDirectShowMethodRenderEx)            <> 0;
  chkNativeVideoSize.Checked       := (FOptions and CDirectShowMethodNativeVideoSize)     <> 0;
  chkLogToFile.Checked             := (FOptions and CDirectShowMethodLogToFile)           <> 0;
  chkAlternativeDirectShow.Checked := (FOptions and CDirectShowMethodAlternative)         <> 0;
  chkNoAC3Pin.Checked              := (FOptions and CDirectShowMethodNoAC3Pin)            <> 0;

  chkRenderAllPins.Enabled       := not chkDoNotRenderGraph.Checked;
  chkRenderAllAudioVideo.Enabled := not chkDoNotRenderGraph.Checked;
  chkRenderEx.Enabled            := not chkDoNotRenderGraph.Checked;

  case FOptions and CDirectShowMethodVmr of
    CDirectShowMethodVideoRenderer  : rgVmr.ItemIndex := 0;
    CDirectShowMethodVideoRenderer7 : rgVmr.ItemIndex := 1;
    CDirectShowMethodVideoRenderer9 : rgVmr.ItemIndex := 2;
    else                              rgVmr.ItemIndex := -1;
  end;

  if (FOptions and CDirectShowMethodVideoRendererWindowless) <> 0 then
    rgVmrWindowed.ItemIndex := 1
  else
    rgVmrWindowed.ItemIndex := 0;

  chkFlipX.Checked := (FOptions and CDirectShowMethodVideoRendererFlipX) <> 0;
  chkFlipY.Checked := (FOptions and CDirectShowMethodVideoRendererFlipY) <> 0;

  case FOptions and CDirectShowMethodGraph of
    CDirectShowMethodFilterGraph2    : rgFilter.ItemIndex := 0;
    CDirectShowMethodGraphBuilder    : rgFilter.ItemIndex := 1;
    CDirectShowMethodFilterGraph     : rgFilter.ItemIndex := 2;
    CDirectShowMethodDvdGraphBuilder : rgFilter.ItemIndex := 3;
    else                               rgFilter.ItemIndex := -1;
  end;

  case FOptions and CDirectShowMethodBuilder of
    CDirectShowMethodNoBuilders           : rgBuilders.ItemIndex := 0;
    CDirectShowMethodCaptureGraphBuilder  : rgBuilders.ItemIndex := 1;
    CDirectShowMethodCaptureGraphBuilder2 : rgBuilders.ItemIndex := 2;
    else                                    rgBuilders.ItemIndex := -1;
  end;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Options

  Descript: Get options (from components)
  Notes   :
 ------------------------------------------------------------------------------}
function TfrmSetupDirectShow.GetOptions: Dword;
begin
  FOptions := 0;
  if chkDoNotRenderGraph.Checked    then
    FOptions := FOptions or CDirectShowMethodDoNotRender;
  if chkDoNotAddToRot.Checked       then
    FOptions := FOptions or CDirectShowMethodDoNotAddToRot;
  if chkRenderAllPins.Checked       then
    FOptions := FOptions or CDirectShowMethodRenderAllPins;
  if chkRenderAllAudioVideo.Checked then
    FOptions := FOptions or CDirectShowMethodRenderAllAudioVideo;
  if chkRenderEx.Checked then
    FOptions := FOptions or CDirectShowMethodRenderEx;
  if chkNativeVideoSize.Checked     then
    FOptions := FOptions or CDirectShowMethodNativeVideoSize;
  if chkLogToFile.Checked     then
    FOptions := FOptions or CDirectShowMethodLogToFile;
  if chkAlternativeDirectShow.Checked     then
    FOptions := FOptions or CDirectShowMethodAlternative;
  if chkNoAc3Pin.Checked then
    FOptions := FOptions or CDirectShowMethodNoAC3Pin;

  case rgVmr.ItemIndex of
    0 : FOptions := FOptions or CDirectShowMethodVideoRenderer;
    1 : FOptions := FOptions or CDirectShowMethodVideoRenderer7;
    2 : FOptions := FOptions or CDirectShowMethodVideoRenderer9;
  end;

  case rgVmrWindowed.ItemIndex of
    1 : FOptions := FOptions or CDirectShowMethodVideoRendererWindowless;
  end;

  if chkFlipX.Checked then
    FOptions := FOptions or CDirectShowMethodVideoRendererFlipX;
  if chkFlipY.Checked then
    FOptions := FOptions or CDirectShowMethodVideoRendererFlipY;

  case rgFilter.ItemIndex of
    0 : FOptions := FOptions or CDirectShowMethodFilterGraph2;
    1 : FOptions := FOptions or CDirectShowMethodGraphBuilder;
    2 : FOptions := FOptions or CDirectShowMethodFilterGraph;
    3 : FOptions := FOptions or CDirectShowMethodDvdGraphBuilder;
  end;

  case rgBuilders.ItemIndex of
    0 : FOptions := FOptions or CDirectShowMethodNoBuilders;
    1 : FOptions := FOptions or CDirectShowMethodCaptureGraphBuilder;
    2 : FOptions := FOptions or CDirectShowMethodCaptureGraphBuilder2;
  end;
  Result := FOptions;
end;


procedure TfrmSetupDirectShow.chkDoNotRenderGraphClick(Sender: TObject);
begin
  chkRenderAllPins.Enabled       := not chkDoNotRenderGraph.Checked;
  chkRenderAllAudioVideo.Enabled := not chkDoNotRenderGraph.Checked;
  chkRenderEx.Enabled            := not chkDoNotRenderGraph.Checked;
end;


function TfrmSetupDirectShow.GetGraph: string;
begin
  // return empty string if internal generated graph to use)
  if cmbDirectShowGraph.Text = cmbDirectShowGraph.Items[0] then
    Result := ''
  else
    Result := cmbDirectShowGraph.Text;
end;


procedure TfrmSetupDirectShow.SetGraph(Value: string);
begin
  // An empty string refers to internal generated format (which is set as first
  // item in combo box)
  if Value = '' then
    Value := cmbDirectShowGraph.Items[0];
  cmbDirectShowGraph.Text := Value;
  FOrgGraph := Value;
end;


function TfrmSetupDirectShow.GetDirectShowOff: Boolean;
begin
  Result := chkDirectShowOff.Checked;
end;


procedure TfrmSetupDirectShow.SetDirectShowOff(Value: Boolean);
begin
  FOrgDirectShowOff        := Value;
  chkDirectShowOff.Checked := Value;
end;


function TfrmSetupDirectShow.GetDirectShowSendAll: Boolean;
begin
  Result := chkDirectShowSendAll.Checked;
end;


procedure TfrmSetupDirectShow.SetDirectShowSendAll(Value: Boolean);
begin
  FOrgDirectShowSendAll        := Value;
  chkDirectShowSendAll.Checked := Value;
end;


procedure TfrmSetupDirectShow.Revert;
begin
  Options                      := FOrgOptions;
  cmbDirectShowGraph.Text      := ForgGraph;
  chkDirectShowOff.Checked     := FOrgDirectShowOff;
  chkDirectShowSendAll.Checked := FOrgDirectShowSendAll;
end;


procedure TfrmSetupDirectShow.FormCreate(Sender: TObject);
var
  SearchRec : TSearchRec;
  Success   : Integer;
  NewName   : string;
begin
  FOptions       := 0;
  FOrgGraph      := '';
  @FExitCallback := nil;
  FAccepted      := False;
  Success := FindFirst('*.GRF', faAnyFile, SearchRec);
  if Success = 0 then
  try
    repeat
      NewName := SearchRec.Name;
      Delete(Newname, Pos('.GRF', UpperCase(NewName)), 255);
      cmbDirectShowGraph.Items.Add(ExtractFileName(NewName));
      Success := FindNext(SearchRec);
    until Success <> 0;
  finally
    SysUtils.FindClose(SearchRec);
  end;
  FOrgDirectShowOff := False;
  chkDirectShowOff.Checked := FOrgDirectShowOff;
  FOrgDirectShowSendAll := False;
  chkDirectShowSendAll.Checked := FOrgDirectShowSendAll;
end;


procedure TfrmSetupDirectShow.btnRevertClick(Sender: TObject);
begin
  Revert;
end;


procedure TfrmSetupDirectShow.btnOKClick(Sender: TObject);
begin
  FAccepted := True;
  Hide;
  Refresh;
  if Assigned(@FExitCallback) then
    FExitCallback(nil);
  // Never call it again
  @FExitCallback := nil;
  Close;
end;


procedure TfrmSetupDirectShow.btnDiscardClick(Sender: TObject);
begin
  Hide;
  Refresh;
  Revert;
  if Assigned(@FExitCallback) then
    FExitCallback(nil);
  // Never call it again
  @FExitCallback := nil;
  Close;
end;


procedure TfrmSetupDirectShow.FormDestroy(Sender: TObject);
begin
  btnDiscardClick(nil);
  @FExitCallback := nil;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
begin
  //
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Finalize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  //
end;


initialization
  Initialize;


finalization
  Finalize;

end.

