{******************************************************************************}
{ FileName............: MajorDvbUnitVolume                                     }
{ Project.............:                                                        }
{ Author(s)...........: MM                                                     }
{ Version.............: 0.00                                                   }
{------------------------------------------------------------------------------}
{  Volume control                                                              }
{                                                                              }
{  Copyright (C) 2003-2004  M.Majoor                                           }
{                                                                              }
{  This program is free software; you can redistribute it and/or               }
{  modify it under the terms of the GNU General Public License                 }
{  as published by the Free Software Foundation; either version 2              }
{  of the License, or (at your option) any later version.                      }
{                                                                              }
{  This program is distributed in the hope that it will be useful,             }
{  but WITHOUT ANY WARRANTY; without even the implied warranty of              }
{  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               }
{  GNU General Public License for more details.                                }
{                                                                              }
{  You should have received a copy of the GNU General Public License           }
{  along with this program; if not, write to the Free Software                 }
{  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. }
{                                                                              }
{------------------------------------------------------------------------------}
{                                                                              }
{ Version   Date   Comment                                                     }
{  0.00   20040818 - Initial release                                           }
{******************************************************************************}
unit MajorDvbUnitVolume;

interface

uses
  Classes,
  Controls,
  Forms,
  Messages,
  MMSystem,
  Windows;

type
  TMixer = class(TCustomForm)                    // Note: Only needed because of messaging ..
  private
    FMixerHandle       : HMIXER;                 // Handle to mixer device
    FMixerMuteControl  : Dword;                  // Control identifier for mute operation
    FMixerVolumeControl: Dword;                  // Control identifier for volume operation
    FMixerVolumeMin    : Integer;                // Minimum volume level
    FMixerVolumeMax    : Integer;                // Maxmum volume level
    FMixerVolumeSteps  : Integer;                // Stepsize for volume adjustments
    FOnMuteChange      : TNotifyEvent;           // Event to call at change of mute
    FOnVolumeChange    : TNotifyEvent;           // Event to call at change of volume
    FLastMute          : Boolean;                // Last detected mute
    FLastVolume        : Dword;                  // Last detected volume
    procedure MixerInitialize;
    function  GetVolume: Dword;
    procedure SetVolume(Value: Dword);
    function  GetVolumeAvailable: Boolean;
    function  GetVolumeMute: Boolean;
    procedure SetVolumeMute(Value: Boolean);
    procedure VolumeAdjust(Up: Boolean);
  protected
    procedure MMChangeMixerLine   (var Message: TMessage); message MM_MIXM_LINE_CHANGE;
    procedure MMChangeMixerControl(var Message: TMessage); message MM_MIXM_CONTROL_CHANGE;
  public
    constructor Create(Owner: TComponent); override;
    destructor  Destroy; override;
  published
    property  OnMuteChange: TNotifyEvent   read FOnMuteChange   write FOnMuteChange;
    property  OnVolumeChange: TNotifyEvent read FOnVolumeChange write FOnVolumeChange;
    property  Volume: Dword                read GetVolume       write SetVolume;
    property  VolumeMute: Boolean          read GetVolumeMute   write SetVolumeMute;
    property  VolumeMin: Integer           read FMixerVolumeMin;
    property  VolumeMax: Integer           read FMixerVolumeMax;
    procedure VolumeUp;
    procedure VolumeDown;
    property  VolumeAvailable: Boolean     read GetVolumeAvailable;
  end;

var
  Mixer: TMixer;


implementation


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Constructor
  Notes   :
 ------------------------------------------------------------------------------}
constructor TMixer.Create(Owner: TComponent);
begin
  inherited CreateNew(Owner);
  FMixerHandle  := Integer(INVALID_HANDLE_VALUE);
  FLastMute     := False;
  FOnMuteChange := nil;
  MixerInitialize;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Destructor
  Notes   :
 ------------------------------------------------------------------------------}
destructor TMixer.Destroy;
begin
  if FMixerHandle <> Integer(INVALID_HANDLE_VALUE) then
    MixerClose(FMixerHandle);

  inherited Destroy;
end;


{------------------------------------------------------------------------------
  Params  : <Message>  lParam = line identifier
                       wParam = mixer device handle
  Returns : -

  Descript: Change line message
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.MMChangeMixerLine(var Message: TMessage);
begin
  // Check if it is for us
  if (FMixerHandle = Integer(INVALID_HANDLE_VALUE)) or
     (Message.wParam <> FMixerHandle) then
  begin
    inherited;
    Exit;
  end;
  if FLastVolume <> GetVolume then
    if Assigned(@OnVolumeChange) then
      OnVolumeChange(Mixer);
  if FLastMute <> GetVolumeMute then
    if Assigned(@OnMuteChange) then
      OnMuteChange(Mixer);
end;


{------------------------------------------------------------------------------
  Params  : <Message>  lParam = control identifier
                       wParam = mixer device handle
  Returns : -

  Descript: Change control message
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.MMChangeMixerControl(var Message: TMessage);
var
  LVolume: Dword;
  LMute  : Boolean;
begin
  // Check if it is for us
  if (FMixerHandle = Integer(INVALID_HANDLE_VALUE)) or
     (Message.wParam <> FMixerHandle) then
  begin
    inherited;
    Exit;
  end;
  LVolume := FLastVolume;
  LMute   := FLastMute;
  if LVolume <> GetVolume then
    if Assigned(@OnVolumeChange) then
      OnVolumeChange(Mixer);
  if LMute <> GetVolumeMute then
    if Assigned(@OnMuteChange) then
      OnMuteChange(Mixer);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  Volume

  Descript: Get volume
  Notes   :
 ------------------------------------------------------------------------------}
function TMixer.GetVolume: Dword;
var
  Control: TMixerControlDetails;
  Details: TMixerControlDetails_Unsigned;
begin
  Result := 0;
  if FMixerHandle = Integer(INVALID_HANDLE_VALUE) then
    Exit;

  Control.cbStruct       := SizeOf(Control);
  Control.dwControlID    := FMixerVolumeControl;
  Control.cChannels      := 1;
  Control.cMultipleItems := 0;
  Control.cbDetails      := SizeOf(Details);
  Control.paDetails      := @Details;

  MixerGetControlDetails(FMixerHandle, @Control, MIXER_OBJECTF_HMIXER or MIXER_SETCONTROLDETAILSF_VALUE);
  Result :=  Details.dwValue;
  FLastVolume := Result;
end;


{------------------------------------------------------------------------------
  Params  : <Value>  Volume to set
  Returns : -

  Descript: Set volume
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.SetVolume(Value: Dword);
var
  Control: TMixerControlDetails;
  Details: TMixerControlDetails_Unsigned;
begin
  if FMixerHandle = Integer(INVALID_HANDLE_VALUE) then
    Exit;

  Details.dwValue        := Value;
  Control.cbStruct       := SizeOf(Control);
  Control.dwControlID    := FMixerVolumeControl;
  Control.cChannels      := 1;
  Control.cMultipleItems := 0;
  Control.cbDetails      := SizeOf(Details);
  Control.paDetails      := @Details;

  MixerSetControlDetails(FMixerHandle, @Control, MIXER_OBJECTF_HMIXER or MIXER_SETCONTROLDETAILSF_VALUE);
  FLastVolume := Value;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Volume control initialization
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.MixerInitialize;
var
  Result : MMRESULT;
  MixerL : MIXERLINE;
  MixerC : MIXERCONTROL;
  MixerLC: MIXERLINECONTROLS;
begin
  // Volume control is by means of the mixer device
  // For this we have to have a handle to this device (we always use the
  // first mixer we can find)
  FMixerHandle := Integer(INVALID_HANDLE_VALUE);
  if MixerGetNumDevs = 0 then
    Exit;
  if Handle <> INVALID_HANDLE_VALUE then
    Result := MixerOpen(@FMixerHandle, 0, Handle, 0, CALLBACK_WINDOW)
  else
    Result := MixerOpen(@FMixerHandle, 0, 0, 0, CALLBACK_WINDOW);
  if Result <> MMSYSERR_NOERROR then
    FMixerHandle := Integer(INVALID_HANDLE_VALUE);

  // We want to use the master controls so get information for this
  // Unfortunately we have to do this via-via
  // First get the information of the speaker destination
  MixerL.cbStruct := SizeOf(MixerL);
  MixerL.dwComponentType := MIXERLINE_COMPONENTTYPE_DST_SPEAKERS;
  Result := MixerGetLineInfo(FMixerHandle, @MixerL, MIXER_OBJECTF_HMIXER or MIXER_GETLINEINFOF_COMPONENTTYPE);
  if Result <> MMSYSERR_NOERROR then
  begin
    MixerClose(FMixerHandle);
    FMixerHandle := Integer(INVALID_HANDLE_VALUE);
  end;

  // Now we have the basic information, get specific information
  // Mute
  MixerLC.cbStruct := SizeOf(MixerLC);
  MixerLC.dwLineId := MixerL.dwLineId;
  MixerLC.dwControlType := MIXERCONTROL_CONTROLTYPE_MUTE;
  MixerLC.cControls := 1;
  MixerLC.cbmxctrl := SizeOf(MixerC);
  MixerLC.pamxctrl := @MixerC;
  Result :=  MixerGetLineControls(FMixerHandle, @MixerLC, MIXER_OBJECTF_HMIXER or MIXER_GETLINECONTROLSF_ONEBYTYPE);
  if Result <> MMSYSERR_NOERROR then
  begin
    MixerClose(FMixerHandle);
    FMixerHandle := Integer(INVALID_HANDLE_VALUE);
  end;
  FMixerMuteControl := MixerC.dwControlId;
  // Volume
  MixerLC.cbStruct := SizeOf(MixerLC);
  MixerLC.dwLineId := MixerL.dwLineId;
  MixerLC.dwControlType := MIXERCONTROL_CONTROLTYPE_VOLUME;
  MixerLC.cControls := 1;
  MixerLC.cbmxctrl := SizeOf(MixerC);
  MixerLC.pamxctrl := @MixerC;
  Result :=  MixerGetLineControls(FMixerHandle, @MixerLC, MIXER_OBJECTF_HMIXER or MIXER_GETLINECONTROLSF_ONEBYTYPE);
  if Result <> MMSYSERR_NOERROR then
  begin
    MixerClose(FMixerHandle);
    FMixerHandle := Integer(INVALID_HANDLE_VALUE);
  end;
  FMixerVolumeControl := MixerC.dwControlId;
  FMixerVolumeMin     := MixerC.Bounds.dwMinimum;
  FMixerVolumeMax     := MixerC.Bounds.dwMaximum;
  FMixerVolumeSteps   := (FMixerVolumeMax - FMixerVolumeMin) div 25;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if available

  Descript: Check if volume control available
  Notes   :
 ------------------------------------------------------------------------------}
function TMixer.GetVolumeAvailable: Boolean;
begin
  Result :=  (FMixerHandle <> Integer(INVALID_HANDLE_VALUE));
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : <Result>  True if muted

  Descript: Get volume mute
  Notes   :
 ------------------------------------------------------------------------------}
function TMixer.GetVolumeMute: Boolean;
var
  Control: TMixerControlDetails;
  Details: TMixerControlDetails_Boolean;
begin
  Result := False;
  if FMixerHandle = Integer(INVALID_HANDLE_VALUE) then
    Exit;

  Control.cbStruct       := SizeOf(Control);
  Control.dwControlID    := FMixerMuteControl;
  Control.cChannels      := 1;
  Control.cMultipleItems := 0;
  Control.cbDetails      := SizeOf(Details);
  Control.paDetails      := @Details;

  MixerGetControlDetails(FMixerHandle, @Control, MIXER_OBJECTF_HMIXER or MIXER_SETCONTROLDETAILSF_VALUE);
  Result :=  (Details.fValue <> 0);
  FLastMute := Result;
end;


{------------------------------------------------------------------------------
  Params  : <Mute>  True if to mute
  Returns : -

  Descript: Set volume mute
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.SetVolumeMute(Value: Boolean);
var
  Control: TMixerControlDetails;
  Details: TMixerControlDetails_Boolean;
begin
  if FMixerHandle = Integer(INVALID_HANDLE_VALUE) then
    Exit;

  if Value then
    Details.fValue       := 1
  else
    Details.fValue       := 0;
  Control.cbStruct       := SizeOf(Control);
  Control.dwControlID    := FMixerMuteControl;
  Control.cChannels      := 1;
  Control.cMultipleItems := 0;
  Control.cbDetails      := SizeOf(Details);
  Control.paDetails      := @Details;

  MixerSetControlDetails(FMixerHandle, @Control, MIXER_OBJECTF_HMIXER or MIXER_SETCONTROLDETAILSF_VALUE);
  FLastMute := Value;
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Volume adjust (steps)
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.VolumeAdjust(Up: Boolean);
var
  CurrentVolume: Dword;
  NewVolume    : Integer;
begin
  CurrentVolume := GetVolume;
  NewVolume := CurrentVolume;
  if Up then
    NewVolume := NewVolume + FMixerVolumeSteps
  else
    NewVolume := NewVolume - FMixerVolumeSteps;
  if NewVolume < FMixerVolumeMin then
    NewVolume := FMixerVolumeMin;
  if NewVolume > FMixerVolumeMax then
    NewVolume := FMixerVolumeMax;
  SetVolume(NewVolume);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Volume adjust (up)
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.VolumeUp;
begin
  VolumeAdjust(True);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Volume adjust (down)
  Notes   :
 ------------------------------------------------------------------------------}
procedure TMixer.VolumeDown;
begin
  VolumeAdjust(False);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Initialize
  Notes   :
 ------------------------------------------------------------------------------}
procedure Initialize;
begin
  Mixer := TMixer.Create(nil);
end;


{------------------------------------------------------------------------------
  Params  : -
  Returns : -

  Descript: Volume adjust (down)
  Notes   :
 ------------------------------------------------------------------------------}
procedure Finalize;
begin
  Mixer.Free;
end;


initialization
  Initialize;


finalization
  Finalize;

end.
